<?php
/**
 * @package     OSL
 * @subpackage  Controller
 *
 * @copyright   Copyright (C) 2016 Ossolution Team, Inc. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE
 */

namespace OSL\Model;

use Exception;
use JDatabaseDriver, JText, JTable, JCache;
use OSL\Container\Container;
use OSL\Input\Input;

class Model
{

	/**
	 * The model name
	 *
	 * @var string
	 */
	protected $name;

	/**
	 * Model state
	 *
	 * @var State
	 */
	protected $state;

	/**
	 * The database driver.
	 *
	 * @var JDatabaseDriver
	 */
	protected $db;

	/**
	 * The name of the database table
	 *
	 * @var string
	 */
	protected $table;

	/**
	 * Ignore request or not. If set to Yes, model states won't be set when it is created
	 *
	 * @var boolean
	 */
	protected $ignoreRequest = false;

	/**
	 * Remember model states value in session
	 *
	 * @var boolean
	 */
	protected $rememberStates = false;

	/**
	 * Constructor
	 *
	 * @param mixed $config Model configuration data, could be an array or an OSFConfig object
	 *
	 * @throws Exception
	 */
	public function __construct(Container $container, $config = [])
	{
		$this->container = $container;

		if (isset($config['name']))
		{
			$this->name = $config['name'];
		}
		else
		{
			$r = null;

			if (!preg_match('/(.*)\\\\Model\\\\(.*)/i', get_class($this), $r))
			{
				throw new Exception(JText::_('JLIB_APPLICATION_ERROR_MODEL_GET_NAME'), 500);
			}

			$this->name = $r[2];
		}

		if (isset($config['db']))
		{
			$this->db = $config['db'];
		}
		else
		{
			$this->db = $container->get('db');
		}

		if (isset($config['state']))
		{
			$this->state = $config['state'];
		}
		else
		{
			$this->state = new State();
		}

		if (isset($config['table']))
		{
			$this->table = $config['table'];
		}
		else
		{
			$this->table = $container->tablePrefix . strtolower($container->inflector->pluralize($this->name));
		}

		if (isset($config['ignore_request']))
		{
			$this->ignoreRequest = $config['ignore_request'];
		}

		if (isset($config['remember_states']))
		{
			$this->rememberStates = $config['remember_states'];
		}

		$this->initialize();
	}

	/**
	 * Populate model state from input
	 */
	public function populateState()
	{
		if ($this->ignoreRequest)
		{
			return;
		}

		$input = $this->container->input;

		$data = $input->getData();

		// Try to get the state properties data from user session
		if ($this->rememberStates)
		{
			$properties = $this->state->getProperties();

			if (count($properties))
			{
				$context = $this->container->option . '.' . $input->get('view', $this->container->defaultView) . '.';

				foreach ($properties as $property)
				{
					$newState = $this->getUserStateFromRequest($input, $context . $property, $property);

					if ($newState != null)
					{
						$data[$property] = $newState;
					}
				}
			}
		}

		$this->setState($data);
	}

	/**
	 * Get JTable object for the model
	 *
	 * @param string $name
	 *
	 * @return JTable
	 */
	public function getTable($name = '')
	{
		if (!$name)
		{
			$name = $this->container->inflector->singularize($this->name);
		}


		return $this->container->factory->createTable($name, $this->getDbo());
	}

	/**
	 * Set the model state properties
	 *
	 * @param string|array The    name of the property, an array
	 *
	 * @param mixed $value The value of the property
	 *
	 * @return Model
	 */
	public function setState($property, $value = null)
	{
		$changed = false;

		if (is_array($property))
		{
			foreach ($property as $key => $value)
			{
				if (isset($this->state->$key) && $this->state->$key != $value)
				{
					$changed = true;
					break;
				}
			}

			$this->state->setData($property);
		}
		else
		{
			if (isset($this->state->$property) && $this->state->$property != $value)
			{
				$changed = true;
			}

			$this->state->$property = $value;
		}

		if ($changed)
		{
			// Reset the data
			$this->data  = null;
			$this->total = null;
		}

		return $this;
	}

	/**
	 * Get the model state properties
	 *
	 * If no property name is given then the function will return an associative array of all properties.
	 *
	 * @param string $property The name of the property
	 *
	 * @param string $default  The default value
	 *
	 * @return mixed <string, State>
	 */
	public function getState($property = null, $default = null)
	{
		$result = $default;

		if (is_null($property))
		{
			$result = $this->state;
		}
		else
		{
			if (isset($this->state->$property))
			{
				$result = $this->state->$property;
			}
		}

		return $result;
	}

	/**
	 * Reset all cached data and reset the model state to it's default
	 *
	 * @param boolean $default If TRUE use defaults when resetting. Default is TRUE
	 *
	 * @return $this
	 */
	public function reset($default = true)
	{
		$this->data  = null;
		$this->total = null;
		$this->state->reset($default);
		$this->query = $this->db->getQuery(true);

		return $this;
	}

	/**
	 * Get the dbo
	 *
	 * @return JDatabaseDriver
	 */
	public function getDbo()
	{
		return $this->db;
	}

	/**
	 * Get name of the model
	 *
	 * @return string
	 */
	public function getName()
	{
		return $this->name;
	}

	/**
	 * This blank method give child class a chance to init the class further after being constructed
	 */
	protected function initialize()
	{

	}

	/**
	 * Supports a simple form Fluent Interfaces.
	 * Allows you to set states by
	 * using the state name as the method name.
	 *
	 * For example : $model->filter_order('name')->filter_order_Dir('DESC')->limit(10)->getData();
	 *
	 * @param string $method Method name
	 *
	 * @param array  $args   Array containing all the arguments for the original call
	 *
	 * @return $this
	 */
	public function __call($method, $args)
	{
		if (isset($this->state->$method))
		{
			$this->state->set($method, $args[0]);

			return $this;
		}

		return null;
	}

	/**
	 * Gets the value of a user state variable.
	 *
	 * @param Input  $input   The input object
	 * @param string $key     The key of the user state variable.
	 * @param string $request The name of the variable passed in a request.
	 * @param string $default The default value for the variable if not found. Optional.
	 * @param string $type    Filter for the variable, for valid values see {@link JFilterInput::clean()}. Optional.
	 *
	 * @return  object  The request user state.
	 */
	protected function getUserStateFromRequest($input, $key, $request, $default = null, $type = 'none')
	{
		$app = $this->container->app;

		$currentState = $app->getUserState($key, $default);
		$newState     = $input->get($request, null, $type);

		// Save the new value only if it was set in this request.
		if ($newState !== null)
		{
			$app->setUserState($key, $newState);
		}
		else
		{
			$newState = $currentState;
		}

		return $newState;
	}

	/**
	 * Clean the cache
	 *
	 * @param string  $group     The cache group
	 * @param integer $client_id The ID of the client
	 *
	 * @return  void
	 *
	 */
	protected function cleanCache($group = null, $client_id = 0)
	{
		$conf = $this->container->appConfig;

		$options = [
			'defaultgroup' => ($group) ? $group : $this->container->option,
			'cachebase'    => ($client_id) ? JPATH_ADMINISTRATOR . '/cache' : $conf->get('cache_path', JPATH_SITE . '/cache')];

		$cache = JCache::getInstance('callback', $options);
		$cache->clean();

		// Trigger the onContentCleanCache event.
		if (!empty($this->eventCleanCache))
		{
			$this->container->app->triggerEvent($this->eventCleanCache, $options);
		}
	}
}