<?php
/**
 * @package     OSL
 * @subpackage  Controller
 *
 * @copyright   Copyright (C) 2016 Ossolution Team, Inc. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE
 */

namespace OSL\Utils;

use JComponentHelper, JFactory, JObject;

class Helper
{
	/**
	 * Get list of language uses on the site
	 *
	 * @return array
	 */
	public static function getLanguages()
	{
		$db    = JFactory::getDbo();
		$query = $db->getQuery(true);

		$params  = JComponentHelper::getParams('com_languages');
		$default = $params->get('site', 'en-GB');

		$query->select('lang_id, lang_code, title, `sef`')
			->from('#__languages')
			->where('published = 1')
			->where('lang_code != ' . $db->quote($default))
			->order('ordering');
		$db->setQuery($query);
		$languages = $db->loadObjectList();

		return $languages;
	}

	/**
	 * Get actions can be performed by the current user on the view of a component
	 *
	 * @param string $option Name of the component is being dispatched
	 *
	 * @return JObject    Actions which can be performed by the current user
	 */
	public static function getActions($option)
	{
		$result = new JObject();

		$user    = JFactory::getUser();
		$actions = array('core.admin', 'core.manage', 'core.create', 'core.edit', 'core.edit.own', 'core.edit.state', 'core.delete');

		foreach ($actions as $action)
		{
			$result->set($action, $user->authorise($action, $option));
		}

		return $result;
	}

	/**
	 * Apply some fixes for request data, this method should be called before the dispatcher is called
	 *
	 * @return void
	 */
	public static function prepareRequestData()
	{
		//Remove cookie vars from request data
		$cookieVars = array_keys($_COOKIE);

		if (count($cookieVars))
		{
			foreach ($cookieVars as $key)
			{
				if (!isset($_POST[$key]) && !isset($_GET[$key]))
				{
					unset($_REQUEST[$key]);
				}
			}
		}

		if (isset($_REQUEST['start']) && !isset($_REQUEST['limitstart']))
		{
			$_REQUEST['limitstart'] = $_REQUEST['start'];
		}

		if (!isset($_REQUEST['limitstart']))
		{
			$_REQUEST['limitstart'] = 0;
		}
	}

	/**
	 * Convert payment amount to from one currency to other currency
	 *
	 * @param float  $amount
	 *
	 * @param string $fromCurrency
	 *
	 * @param string $toCurrency
	 *
	 * @return float
	 */
	public static function convertPaymentAmount($amount, $fromCurrency, $toCurrency)
	{
		$http     = \JHttpFactory::getHttp();
		$url      = 'http://finance.yahoo.com/d/quotes.csv?e=.csv&f=sl1d1t1&s=' . $toCurrency . $fromCurrency . '=X';
		$response = $http->get($url);
		if ($response->code == 200)
		{
			$currencyData = explode(',', $response->body);
			$rate         = floatval($currencyData[1]);
			if ($rate > 0)
			{
				$amount = $amount / $rate;
			}
		}

		return round($amount, 2);
	}

	/**
	 * Credit Card Type
	 *
	 * Iterates through known/supported card brands to determine the brand of this card
	 *
	 * @param string $cardNumber
	 *
	 * @return string
	 */
	public static function getCardType($cardNumber)
	{
		$supportedCardTypes = array(
			'visa'               => '/^4\d{12}(\d{3})?$/',
			'mastercard'         => '/^(5[1-5]\d{4}|677189)\d{10}$/',
			'discover'           => '/^(6011|65\d{2}|64[4-9]\d)\d{12}|(62\d{14})$/',
			'amex'               => '/^3[47]\d{13}$/',
			'diners_club'        => '/^3(0[0-5]|[68]\d)\d{11}$/',
			'jcb'                => '/^35(28|29|[3-8]\d)\d{12}$/',
			'switch'             => '/^6759\d{12}(\d{2,3})?$/',
			'solo'               => '/^6767\d{12}(\d{2,3})?$/',
			'dankort'            => '/^5019\d{12}$/',
			'maestro'            => '/^(5[06-8]|6\d)\d{10,17}$/',
			'forbrugsforeningen' => '/^600722\d{10}$/',
			'laser'              => '/^(6304|6706|6709|6771(?!89))\d{8}(\d{4}|\d{6,7})?$/',
		);

		foreach ($supportedCardTypes as $brand => $val)
		{
			if (preg_match($val, $cardNumber))
			{
				return $brand;
			}
		}

		return '';
	}
}