<?php
/**
 * @package     OSL
 * @subpackage  Config
 *
 * @copyright   Copyright (C) 2016 Ossolution Team, Inc. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE
 */

namespace OSL\Config;

/**
 * Provide basic methods which help accessing to configuration data
 *
 * @package        OSL
 * @subpackage     Config
 * @since          1.0
 */
class Config implements \ArrayAccess
{
	/**
	 * The config data container
	 *
	 * @var array
	 */
	protected $data;

	/**
	 * OSFConfig constructor.
	 *
	 * @param array $data
	 */
	public function __construct(array $data = array())
	{
		$this->data = $data;
	}

	/**
	 * Retrieve data for a config option
	 *
	 * @param string $key     The key of the config option
	 *
	 * @param mixed  $default Default value if no data has been set for that config option
	 *
	 * @return mixed The config option value
	 */
	public function get($key, $default = null)
	{
		if (isset($this->data[$key]))
		{
			return $this->data[$key];
		}

		return $default;
	}

	/**
	 * Set data for a config option
	 *
	 * @param string $name The name of config option
	 * @param mixed  $value
	 *
	 * @return $this
	 */
	public function set($name, $value)
	{
		$this->data[$name] = $value;

		return $this;
	}

	/**
	 * Define value for a config option. The value will only be set if there's no value for the name or if it is null.
	 *
	 * @param   string $name  Name of the value to define.
	 * @param   mixed  $value Value to assign to the input.
	 *
	 * @return  void
	 *
	 */
	public function def($name, $value)
	{
		if (isset($this->data[$name]))
		{
			return;
		}

		$this->data[$name] = $value;
	}

	/**
	 * @param mixed $offset
	 *
	 * @return mixed
	 */
	public function offsetExists($offset)
	{
		return isset($this->data[$offset]);
	}

	/**
	 * @param mixed $offset
	 *
	 * @return mixed
	 */
	public function offsetGet($offset)
	{
		return $this->get($offset);
	}

	/**
	 * @param mixed $offset
	 * @param mixed $value
	 *
	 * @return mixed
	 */
	public function offsetSet($offset, $value)
	{
		$this->set($offset, $value);
	}

	/**
	 * @param mixed $offset
	 *
	 * @return mixed
	 */
	public function offsetUnset($offset)
	{
		unset($this->data[$offset]);
	}


	/**
	 * Magic method to get a config option value
	 *
	 * @param string
	 *
	 * @return mixed
	 */
	public function __get($name)
	{
		return $this->get($name);
	}

	/**
	 * Set config option value
	 *
	 * @param string $name  user-specified config option
	 *
	 * @param mixed  $value user-specified config option value.
	 *
	 * @return void
	 */
	public function __set($name, $value)
	{
		$this->set($name, $value);
	}

	/**
	 * Test existence of a config variable
	 *
	 * @param string
	 *
	 * @return boolean
	 */
	public function __isset($name)
	{
		return isset($this->data[$name]);
	}
}