<?php
/**
 * @package     OSL
 * @subpackage  Config
 *
 * @copyright   Copyright (C) 2016 Ossolution Team, Inc. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE
 */

namespace OSL\Factory;

use JDatabaseDriver, JTable;

use OSL\Container\Container;

class Factory
{
	/**
	 * Component container
	 *
	 * @var Container
	 */
	protected $container;

	/**
	 * Constructor.
	 *
	 * @param   Container  $container
	 */
	public function __construct(Container $container)
	{
		$this->container = $container;
	}

	/**
	 * Method to create a controller instance
	 *
	 * @param   string  $name
	 * @param   array   $config
	 * @param   string  $section
	 *
	 * @return \OSL\Controller\Controller
	 *
	 * @throws \InvalidArgumentException
	 *
	 */
	public function createController($name, $config = array(), $section = 'auto')
	{
		$section = strtolower($section);

		if ($section == 'auto')
		{
			$namespace = $this->container->namespace;
		}
		elseif ($section == 'site')
		{
			$namespace = $this->container->feNamespace;
		}
		elseif ($section == 'admin')
		{
			$namespace = $this->container->beNamespace;
		}
		else
		{
			throw new \InvalidArgumentException(printf('The provided section %s is invalid', $section));
		}

		$classes   = [];
		$classes[] = $namespace . '\\Override\\Controller\\' . ucfirst($name);
		$classes[] = $namespace . '\\Controller\\' . ucfirst($name);
		$classes[] = $namespace . '\\Controller\\Controller';

		foreach ($classes as $class)
		{
			if (class_exists($class))
			{
				$config['name'] = $name;

				return new $class($this->container, $config);
			}
		}

		throw new \InvalidArgumentException(printf('Could not find the controller %s ', $name));
	}

	/**
	 * Method to create a model instance
	 *
	 * @param   string  $name
	 * @param   array   $config
	 * @param   string  $section
	 *
	 * @return \OSL\Model\Model
	 *
	 * @throws \InvalidArgumentException
	 */
	public function createModel($name, $config = array(), $section = 'auto')
	{
		$section = strtolower($section);

		if ($section == 'auto')
		{
			$namespace = $this->container->namespace;
		}
		elseif ($section == 'site')
		{
			$namespace = $this->container->feNamespace;
		}
		elseif ($section == 'admin')
		{
			$namespace = $this->container->beNamespace;
		}
		else
		{
			throw new \InvalidArgumentException(printf('The provided section %s is invalid', $section));
		}

		$classes   = [];
		$classes[] = $namespace . '\\Override\\Model\\' . ucfirst($name);
		$classes[] = $namespace . '\\Model\\' . ucfirst($name);

		if ($this->container->inflector->isPlural($name))
		{
			$classes[] = 'OSL\\Model\\ListModel';
		}
		else
		{
			if ($this->container->app->isClient('administrator'))
			{
				$classes[] = 'OSL\\Model\\AdminModel';
			}
			else
			{
				$classes[] = 'OSL\\Model\\Model';
			}
		}


		$config['name'] = $name;

		foreach ($classes as $class)
		{
			if (class_exists($class))
			{
				return new $class($this->container, $config);
			}
		}

		throw new \InvalidArgumentException(printf('Could not find the model %s', ucfirst($name)));
	}

	/**
	 * Method to create a view instance
	 *
	 * @param   string  $name
	 * @param   string  $type
	 * @param   array   $config
	 * @param   string  $section
	 *
	 * @return \OSL\View\AbstractView
	 */
	public function createView($name, $type = 'html', $config = array(), $section = 'auto')
	{
		$section = strtolower($section);

		if ($section == 'auto')
		{
			$namespace = $this->container->namespace;
		}
		elseif ($section == 'site')
		{
			$namespace = $this->container->feNamespace;
		}
		elseif ($section == 'admin')
		{
			$namespace = $this->container->beNamespace;
		}
		else
		{
			throw new \InvalidArgumentException(printf('The provided section %s is invalid', $section));
		}

		$classes   = [];
		$classes[] = $namespace . '\\Override\\View\\' . ucfirst($name) . '\\' . ucfirst($type);;
		$classes[] = $namespace . '\\View\\' . ucfirst($name) . '\\' . ucfirst($type);;

		if ($this->container->inflector->isPlural($name))
		{
			$classes[] = 'OSL\\View\\ListView';
		}
		else
		{
			$classes[] = 'OSL\\View\\ItemView';
		}

		$config['name'] = $name;

		foreach ($classes as $class)
		{
			if (class_exists($class))
			{
				return new $class($this->container, $config);
			}
		}

		throw new \InvalidArgumentException(printf('Could not find the view %s, type %s', ucfirst($name), ucfirst($type)));
	}

	/**
	 * @param   string           $name
	 * @param   JDatabaseDriver  $db
	 *
	 * @return JTable
	 */
	public function createTable($name, JDatabaseDriver $db = null)
	{
		$className = $this->container->beNamespace . '\\Table\\' . ucfirst($name);

		if (!class_exists($className))
		{
			throw new \RuntimeException(sprintf('Table class %s does not exist', $name));
		}

		if (is_null($db))
		{
			$db = $this->container->db;
		}

		return new $className($db);
	}
}