<?php
/**
* Community Builder (TM)
* @version $Id: $
* @package CommunityBuilder
* @copyright (C)2005-2015 www.joomlapolis.com / Lightning MultiCom SA - and its licensors, all rights reserved
* @license http://www.gnu.org/licenses/old-licenses/gpl-2.0.html GNU/GPL version 2
*/

use CBLib\Application\Application;
use CB\Database\Table\TabTable;
use CB\Database\Table\UserTable;
use CBLib\Registry\GetterInterface;
use CBLib\Language\CBTxt;
use CB\Plugin\Privacy\CBPrivacy;
use CB\Plugin\Privacy\Privacy;
use CB\Plugin\Privacy\PrivacyInterface;
use CB\Plugin\Privacy\Table\PrivacyTable;
use CB\Plugin\Privacy\Table\ClosedTable;
use CB\Plugin\Privacy\Table\BlockedTable;
use CBLib\Input\Get;

if ( ! ( defined( '_VALID_CB' ) || defined( '_JEXEC' ) || defined( '_VALID_MOS' ) ) ) { die( 'Direct Access to this location is not allowed.' ); }

global $_PLUGINS;

$_PLUGINS->loadPluginGroup( 'user' );

class CBplug_cbprivacy extends cbPluginHandler
{

	/**
	 * @param  TabTable   $tab       Current tab
	 * @param  UserTable  $user      Current user
	 * @param  int        $ui        1 front, 2 admin UI
	 * @param  array      $postdata  Raw unfiltred POST data
	 */
	public function getCBpluginComponent( $tab, $user, $ui, $postdata )
	{
		$action				=	$this->input( 'action', null, GetterInterface::STRING );

		if ( $action == 'privacy' ) {
			$this->getPrivacy();
			return;
		}

		$raw				=	( $this->input( 'format', null, GetterInterface::STRING ) == 'raw' );
		$function			=	$this->input( 'func', null, GetterInterface::STRING );
		$id					=	$this->input( 'id', null, GetterInterface::INT );
		$user				=	CBuser::getMyUserDataInstance();

		if ( ! $raw ) {
			outputCbJs();
			outputCbTemplate();

			ob_start();
		}

		switch ( $action ) {
			case 'blocked':
				switch ( $function ) {
					case 'block':
						$this->blockUser( $id, $user );
						break;
					case 'unblock':
						$this->unblockUser( $id, $user );
						break;
					case 'show':
					default:
						$this->showBlocked( $user );
						break;
				}
				break;
			case 'disable':
				switch ( $function ) {
					case 'confirm':
						$this->confirmDisable( null, $user );
						break;
					case 'cancel':
						$this->cancelDisable( $id, $user );
						break;
					case 'enable':
						$this->saveEnable( $id, $user );
						break;
					case 'save':
						cbSpoofCheck( 'plugin' );
						$this->saveDisable( $id, $user );
						break;
					case 'show':
					default:
						$this->showDisableEdit( $id, $user );
						break;
				}
				break;
			case 'delete':
				switch ( $function ) {
					case 'confirm':
						$this->confirmDelete( null, $user );
						break;
					case 'cancel':
						$this->cancelDelete( $id, $user );
						break;
					case 'save':
						cbSpoofCheck( 'plugin' );
						$this->saveDelete( $id, $user );
						break;
					case 'show':
					default:
						$this->showDeleteEdit( $id, $user );
						break;
				}
				break;
			case 'export':
				switch ( $function ) {
					case 'export':
						$this->confirmExportData( $id, $user );
						break;
					case 'request':
					default:
						$this->requestExportData( $id, $user );
						break;
				}
				break;
			default:
				CBRedirect( 'index.php', CBTxt::T( 'Not authorized.' ), 'error' );
				break;
		}

		if ( ! $raw ) {
			$html			=	ob_get_contents();
			ob_end_clean();

			if ( ! $html ) {
				return;
			}

			$class			=	$this->params->getString( 'general_class', '' );

			$return			=	'<div class="cbPrivacy' . ( $class ? ' ' . htmlspecialchars( $class ) : null ) . '">'
							.		$html
							.	'</div>';

			echo $return;
		}
	}

	/**
	 * Loads in a privacy directly or by URL
	 *
	 * @param null|Privacy $privacy
	 * @param null|string  $view
	 */
	public function getPrivacy( $privacy = null, $view = null )
	{
		global $_PLUGINS;

		$viewer							=	CBuser::getMyUserDataInstance();
		$raw							=	false;
		$inline							=	false;
		$access							=	true;

		$privacyLoaded					=	false;

		if ( $privacy ) {
			if ( ! ( $privacy instanceof PrivacyInterface ) ) {
				return;
			}

			$function					=	( $view ? $view : 'edit' );
			$inline						=	true;

			$privacyLoaded				=	true;
		} else {
			$raw						=	( $this->input( 'format', null, GetterInterface::STRING ) == 'raw' );
			$function					=	$this->input( 'func', null, GetterInterface::STRING );
			$privacyId					=	$this->input( 'privacy', null, GetterInterface::STRING );

			$privacy					=	new Privacy( null, $viewer );

			if ( $privacyId ) {
				if ( $privacy->load( $privacyId ) ) {
					$privacyLoaded		=	true;
				} else {
					$access				=	false;
				}
			} else {
				$access					=	false;
			}
		}

		if ( ! $privacy->asset() ) {
			$access						=	false;
		}

		$_PLUGINS->trigger( 'privacy_onPrivacy', array( &$privacy, &$access, $privacyLoaded ) );

		if ( ! $access ) {
			if ( $inline ) {
				return;
			} elseif ( $raw ) {
				header( 'HTTP/1.0 401 Unauthorized' );
				exit();
			} else {
				cbRedirect( 'index.php', CBTxt::T( 'Not authorized.' ), 'error' );
			}
		}

		if ( ! $privacy->id() ) {
			$privacy->cache();
		}

		if ( ! $raw ) {
			outputCbJs();
			outputCbTemplate();

			ob_start();
		}

		switch ( $function ) {
			case 'users':
				if ( $raw ) {
					$this->showPrivacyUsers( $privacy );
				}
				break;
			case 'edit':
				$this->showPrivacyEdit( $viewer, $privacy );
				break;
			case 'save':
				$this->savePrivacy( $viewer, $privacy, $raw );
				break;
		}

		if ( ! $raw ) {
			$html						=	ob_get_contents();
			ob_end_clean();

			if ( ! $html ) {
				return;
			}

			$class						=	$this->params->getString( 'general_class', '' )
										.	' ' . $privacy->getString( 'class', '' );

			$return						=	'<span class="cbPrivacy' . ( $class ? ' ' . htmlspecialchars( trim( $class ) ) : null ) . '">'
										.		$html
										.	'</span>';

			echo $return;
		}
	}

	/**
	 * Shows list of blocked users
	 *
	 * @param UserTable $user
	 */
	private function showBlocked( $user )
	{
		global $_CB_framework, $_CB_database;

		if ( ( ! $user->get( 'id', 0, GetterInterface::INT ) ) || ( ! CBPrivacy::getGlobalParams()->get( 'profile_blocking', true, GetterInterface::BOOLEAN ) ) ) {
			CBPrivacy::returnRedirect( 'index.php', CBTxt::T( 'Not authorized.' ), 'error' );
		}

		$prefix					=	'blocked_users_';
		$limit					=	$this->params->get( 'profile_blocking_limit', 30, GetterInterface::INT );
		$limitstart				=	(int) $_CB_framework->getUserStateFromRequest( $prefix . 'limitstart{com_comprofiler}', $prefix . 'limitstart', 0 );
		$search					=	$_CB_framework->getUserStateFromRequest( $prefix . 'search{com_comprofiler}', $prefix . 'search', '' );
		$where					=	null;

		if ( $search && $this->params->get( 'profile_blocking_search', true, GetterInterface::BOOLEAN ) ) {
			$where				.=	"\n AND ( j." . $_CB_database->NameQuote( 'name' ) . " LIKE " . $_CB_database->Quote( '%' . $_CB_database->getEscaped( $search, true ) . '%', false )
								.	" OR j." . $_CB_database->NameQuote( 'username' ) . " LIKE " . $_CB_database->Quote( '%' . $_CB_database->getEscaped( $search, true ) . '%', false ) . " )";
		}

		$searching				=	( $where ? true : false );

		$query					=	'SELECT COUNT(*)'
								.	"\n FROM " . $_CB_database->NameQuote( '#__comprofiler_plugin_privacy_blocked' ) . " AS b"
								.	"\n LEFT JOIN " . $_CB_database->NameQuote( '#__comprofiler' ) . " AS cb"
								.	' ON cb.' . $_CB_database->NameQuote( 'id' ) . ' = b.' . $_CB_database->NameQuote( 'block' )
								.	"\n LEFT JOIN " . $_CB_database->NameQuote( '#__users' ) . " AS j"
								.	' ON j.' . $_CB_database->NameQuote( 'id' ) . ' = cb.' . $_CB_database->NameQuote( 'id' )
								.	"\n WHERE b." . $_CB_database->NameQuote( 'user_id' ) . " = " . $user->get( 'id', 0, GetterInterface::INT )
								.	"\n AND cb." . $_CB_database->NameQuote( 'approved' ) . " = 1"
								.	"\n AND cb." . $_CB_database->NameQuote( 'confirmed' ) . " = 1"
								.	"\n AND j." . $_CB_database->NameQuote( 'block' ) . " = 0"
								.	$where;
		$_CB_database->setQuery( $query );
		$total					=	(int) $_CB_database->loadResult();

		$pageNav				=	new \cbPageNav( $total, $limitstart, $limit );

		$pageNav->setInputNamePrefix( $prefix );
		$pageNav->setStaticLimit( true );
		$pageNav->setBaseURL( $_CB_framework->pluginClassUrl( $this->element, false, array( 'action' => 'blocked', $prefix . 'search' => ( $searching ? $search : null ) ) ) );

		$query					=	'SELECT b.*'
								.	"\n FROM " . $_CB_database->NameQuote( '#__comprofiler_plugin_privacy_blocked' ) . " AS b"
								.	"\n LEFT JOIN " . $_CB_database->NameQuote( '#__comprofiler' ) . " AS cb"
								.	' ON cb.' . $_CB_database->NameQuote( 'id' ) . ' = b.' . $_CB_database->NameQuote( 'block' )
								.	"\n LEFT JOIN " . $_CB_database->NameQuote( '#__users' ) . " AS j"
								.	' ON j.' . $_CB_database->NameQuote( 'id' ) . ' = cb.' . $_CB_database->NameQuote( 'id' )
								.	"\n WHERE b." . $_CB_database->NameQuote( 'user_id' ) . " = " . $user->get( 'id', 0, GetterInterface::INT )
								.	"\n AND cb." . $_CB_database->NameQuote( 'approved' ) . " = 1"
								.	"\n AND cb." . $_CB_database->NameQuote( 'confirmed' ) . " = 1"
								.	"\n AND j." . $_CB_database->NameQuote( 'block' ) . " = 0"
								.	$where
								.	"\n ORDER BY b." . $_CB_database->NameQuote( 'date' ) . " DESC";
		if ( $this->params->get( 'profile_blocking_paging', true, GetterInterface::BOOLEAN ) ) {
			$_CB_database->setQuery( $query, $pageNav->limitstart, $pageNav->limit );
		} else {
			$_CB_database->setQuery( $query );
		}
		$rows					=	$_CB_database->loadObjectList( null, '\CB\Plugin\Privacy\Table\BlockedTable', array( $_CB_database ) );

		$users					=	array();

		/** @var BlockedTable[] $rows */
		foreach ( $rows as $row ) {
			$userId				=	$row->get( 'block', 0, GetterInterface::INT );

			if ( $userId && ( ! in_array( $userId, $users ) ) ) {
				$users[]		=	$userId;
			}
		}

		if ( $users ) {
			CBuser::advanceNoticeOfUsersNeeded( $users );
		}

		$input					=	array();

		$input['search']		=	'<input type="text" name="' . htmlspecialchars( $prefix ) . 'search" value="' . htmlspecialchars( $search ) . '" onchange="document.privacyBlockedForm.submit();" placeholder="' . htmlspecialchars( CBTxt::T( 'Search Blocked...' ) ) . '" class="form-control" />';

		require CBPrivacy::getTemplate( null, 'blocked' );
	}

	/**
	 * Blocks the supplied $userId
	 *
	 * @param int       $userId
	 * @param UserTable $user
	 */
	private function blockUser( $userId, $user )
	{
		global $_CB_framework;

		if ( ( ! $user->get( 'id', 0, GetterInterface::INT ) ) || ( ! CBPrivacy::getGlobalParams()->get( 'profile_blocking', true, GetterInterface::BOOLEAN ) ) ) {
			CBPrivacy::returnRedirect( 'index.php', CBTxt::T( 'Not authorized.' ), 'error' );
		}

		if ( ! $userId ) {
			$block				=	$this->input( 'post/block', null, GetterInterface::STRING );

			if ( $block ) {
				if ( is_string( $block ) ) {
					$blockUser	=	new UserTable();

					$blockUser->loadByUsername( $block );

					$userId		=	$blockUser->get( 'id', 0, GetterInterface::INT );
				}

				if ( ! $userId ) {
					$userId		=	(int) $block;
				}
			}
		}

		if ( ( ! $userId ) || ( ! Application::User( $userId )->getUserId() ) ) {
			CBPrivacy::returnRedirect( $_CB_framework->pluginClassUrl( $this->element, false, array( 'action' => 'blocked' ) ), CBTxt::T( 'User does not exist.' ), 'error' );
		}

		if ( $userId == $user->get( 'id', 0, GetterInterface::INT ) ) {
			CBPrivacy::returnRedirect( $_CB_framework->pluginClassUrl( $this->element, false, array( 'action' => 'blocked' ) ), CBTxt::T( 'You can not block yourself.' ), 'error' );
		}

		if ( Application::User( $userId )->isGlobalModerator() ) {
			CBPrivacy::returnRedirect( $_CB_framework->pluginClassUrl( $this->element, false, array( 'action' => 'blocked' ) ), CBTxt::T( 'You can not block this user.' ), 'error' );
		}

		$blocked				=	new BlockedTable();

		$blocked->load( array( 'user_id' => $user->get( 'id', 0, GetterInterface::INT ), 'block' => $userId ) );

		if ( $blocked->get( 'id', 0, GetterInterface::INT ) ) {
			CBPrivacy::returnRedirect( $_CB_framework->pluginClassUrl( $this->element, false, array( 'action' => 'blocked' ) ), CBTxt::T( 'User already blocked.' ) );
		}

		$blocked->set( 'user_id', $user->get( 'id', 0, GetterInterface::INT ) );
		$blocked->set( 'block', $userId );

		if ( $blocked->getError() || ( ! $blocked->check() ) ) {
			CBPrivacy::returnRedirect( $_CB_framework->pluginClassUrl( $this->element, false, array( 'action' => 'blocked' ) ), CBTxt::T( 'USER_FAILED_TO_BLOCK', 'User failed to block! Error: [error]', array( '[error]' => $blocked->getError() ) ), 'error' );
		}

		if ( $blocked->getError() || ( ! $blocked->store() ) ) {
			CBPrivacy::returnRedirect( $_CB_framework->pluginClassUrl( $this->element, false, array( 'action' => 'blocked' ) ), CBTxt::T( 'USER_FAILED_TO_BLOCK', 'User failed to block! Error: [error]', array( '[error]' => $blocked->getError() ) ), 'error' );
		}

		CBPrivacy::returnRedirect( $_CB_framework->pluginClassUrl( $this->element, false, array( 'action' => 'blocked' ) ), CBTxt::T( 'User blocked successfully!' ) );
	}

	/**
	 * Unblocks the supplied $userId
	 *
	 * @param int       $userId
	 * @param UserTable $user
	 */
	private function unblockUser( $userId, $user )
	{
		global $_CB_framework;

		if ( ( ! $user->get( 'id', 0, GetterInterface::INT ) ) || ( ! CBPrivacy::getGlobalParams()->get( 'profile_blocking', true, GetterInterface::BOOLEAN ) ) ) {
			CBPrivacy::returnRedirect( 'index.php', CBTxt::T( 'Not authorized.' ), 'error' );
		}

		if ( ( ! $userId ) || ( ! Application::User( $userId )->getUserId() ) ) {
			CBPrivacy::returnRedirect( $_CB_framework->pluginClassUrl( $this->element, false, array( 'action' => 'blocked' ) ), CBTxt::T( 'User does not exist.' ), 'error' );
		}

		$blocked	=	new BlockedTable();

		$blocked->load( array( 'user_id' => $user->get( 'id', 0, GetterInterface::INT ), 'block' => $userId ) );

		if ( ! $blocked->get( 'id', 0, GetterInterface::INT ) ) {
			CBPrivacy::returnRedirect( $_CB_framework->pluginClassUrl( $this->element, false, array( 'action' => 'blocked' ) ), CBTxt::T( 'User is not blocked.' ) );
		}

		if ( $blocked->getError() || ( ! $blocked->canDelete() ) ) {
			CBPrivacy::returnRedirect( $_CB_framework->pluginClassUrl( $this->element, false, array( 'action' => 'blocked' ) ), CBTxt::T( 'USER_FAILED_TO_UNBLOCK', 'User failed to unblock! Error: [error]', array( '[error]' => $blocked->getError() ) ), 'error' );
		}

		if ( $blocked->getError() || ( ! $blocked->delete() ) ) {
			CBPrivacy::returnRedirect( $_CB_framework->pluginClassUrl( $this->element, false, array( 'action' => 'blocked' ) ), CBTxt::T( 'USER_FAILED_TO_UNBLOCK', 'User failed to unblock! Error: [error]', array( '[error]' => $blocked->getError() ) ), 'error' );
		}

		CBPrivacy::returnRedirect( $_CB_framework->pluginClassUrl( $this->element, false, array( 'action' => 'blocked' ) ), CBTxt::T( 'User unblocked successfully!' ) );
	}

	/**
	 * Checks if the disable account field is accessible to a user
	 *
	 * @param int       $userId
	 * @param UserTable $user
	 * @return bool
	 */
	private function getDisableField( $userId, $user )
	{
		if ( ( ! $userId )
			 || ( ( $userId != $user->get( 'id', 0, GetterInterface::INT ) ) && ( ! Application::User( $user->get( 'id', 0, GetterInterface::INT ) )->isGlobalModerator() ) )
			 || ( Application::User( $userId )->isGlobalModerator() && ( ! $this->params->get( 'disable_moderator', false, GetterInterface::BOOLEAN ) ) )
		) {
			return false;
		}

		$fields		=	CBuser::getInstance( $userId, false )->_getCbTabs( false )->_getTabFieldsDb( null, $user, 'edit', 'privacy_disable_me' );

		return ( $fields ? true : false );
	}

	/**
	 * Displays account disable form
	 *
	 * @param int       $userId
	 * @param UserTable $user
	 */
	private function showDisableEdit( $userId, $user )
	{
		if ( ! $userId ) {
			$userId		=	$user->get( 'id', 0, GetterInterface::INT );
		}

		if ( ! $this->getDisableField( $userId, $user ) ) {
			cbRedirect( 'index.php', CBTxt::T( 'Not authorized.' ), 'error' );
		}

		$closed			=	new ClosedTable();

		$closed->load( array( 'user_id' => $userId, 'type' => 'disable_request' ) );

		if ( $closed->get( 'id', 0, GetterInterface::INT ) ) {
			require CBPrivacy::getTemplate( null, 'disable_pending' );
		} else {
			require CBPrivacy::getTemplate( null, 'disable' );
		}
	}

	/**
	 * Requests to disable a users account
	 *
	 * @param int       $userId
	 * @param UserTable $user
	 */
	private function saveDisable( $userId, $user )
	{
		global $_CB_framework, $_PLUGINS;

		if ( ! $userId ) {
			$userId			=	$user->get( 'id', 0, GetterInterface::INT );
		}

		if ( ! $this->getDisableField( $userId, $user ) ) {
			cbRedirect( 'index.php', CBTxt::T( 'Not authorized.' ), 'error' );
		}

		if ( ( $userId != $user->get( 'id', 0, GetterInterface::INT ) ) && Application::User( $user->get( 'id', 0, GetterInterface::INT ) )->isGlobalModerator() ) {
			// Moderator is disabling another users account so directly bypass to the confirm disable process:
			$this->confirmDisable( $userId, $user );
			return;
		}

		$cbUser				=	CBuser::getInstance( $userId, false );
		$disableUser		=	$cbUser->getUserData();

		if ( ! $disableUser->get( 'id', 0, GetterInterface::INT ) ) {
			cbRedirect( 'index.php', CBTxt::T( 'Not authorized.' ), 'error' );
		}

		$_PLUGINS->trigger( 'privacy_onBeforeAccountDisableRequest', array( &$disableUser, $user ) );

		$closed				=	new ClosedTable();

		$closed->set( 'user_id', $disableUser->get( 'id', 0, GetterInterface::INT ) );
		$closed->set( 'username', $disableUser->getString( 'username', '' ) );
		$closed->set( 'name', $disableUser->getString( 'name', '' ) );
		$closed->set( 'email', $disableUser->getString( 'email', '' ) );
		$closed->set( 'type', 'disable_request' );
		$closed->set( 'reason', $this->input( 'reason', null, GetterInterface::STRING ) );
		$closed->set( 'token', md5( 'disable_account_' . $disableUser->get( 'id', 0, GetterInterface::INT ) . '_' . $_CB_framework->getCfg( 'secret' ) ) );

		if ( $closed->getError() || ( ! $closed->check() ) ) {
			$_CB_framework->enqueueMessage( CBTxt::T( 'ACCOUNT_FAILED_TO_DISABLE', 'Account failed to disable! Error: [error]', array( '[error]' => $closed->getError() ) ), 'error' );

			$this->showDisableEdit( $userId, $user );
			return;
		}

		if ( $closed->getError() || ( ! $closed->store() ) ) {
			$_CB_framework->enqueueMessage( CBTxt::T( 'ACCOUNT_FAILED_TO_DISABLE', 'Account failed to disable! Error: [error]', array( '[error]' => $closed->getError() ) ), 'error' );

			$this->showDisableEdit( $userId, $user );
			return;
		}

		$notification		=	new cbNotification();

		$confirmUrl			=	$_CB_framework->pluginClassUrl( $this->element, false, array( 'action' => 'disable', 'func' => 'confirm', 'confirm' => $closed->getString( 'token' ) ) );

		$extra				=	array(	'ip_address'	=>	Application::Input()->getRequestIP(),
										'reason'		=>	$closed->getString( 'reason', '' ),
										'date'			=>	cbFormatDate( $closed->getString( 'date', '' ) )
									);

		$savedLanguage		=	CBTxt::setLanguage( $disableUser->getUserLanguage() );

		$subject			=	CBTxt::T( 'Confirm Account Disable' );
		$body				=	CBTxt::T( 'NOTICE_CONFIRM_ACCOUNT_DISABLE', 'This is a notice that your account [username] on [siteurl] has been requested to be disabled. Please confirm the account disable by clicking the following link.<br /><br />[confirm]', array( '[confirm]' => '<a href="' . htmlspecialchars( $confirmUrl ) . '">' . $confirmUrl . '</a>', '[confirm_url]' => htmlspecialchars( $confirmUrl ) ) );

		CBTxt::setLanguage( $savedLanguage );

		if ( $subject && $body ) {
			$notification->sendFromSystem( $disableUser, $subject, $body, true, 1, null, null, null, $extra );
		}

		$_PLUGINS->trigger( 'privacy_onAfterAccountDisableRequest', array( $disableUser, $user ) );

		$_CB_framework->enqueueMessage( CBTxt::T( 'Your disable account request requires confirmation. Please check your email and confirm.' ) );

		$this->showDisableEdit( $userId, $user );
	}

	/**
	 * Requests to disable a users account
	 *
	 * @param int       $userId
	 * @param UserTable $user
	 */
	private function cancelDisable( $userId, $user )
	{
		global $_CB_framework;

		if ( ! $userId ) {
			$userId		=	$user->get( 'id', 0, GetterInterface::INT );
		}

		if ( ! $this->getDisableField( $userId, $user ) ) {
			cbRedirect( 'index.php', CBTxt::T( 'Not authorized.' ), 'error' );
		}

		$cbUser			=	CBuser::getInstance( $userId, false );
		$disableUser	=	$cbUser->getUserData();

		if ( ! $disableUser->get( 'id', 0, GetterInterface::INT ) ) {
			cbRedirect( 'index.php', CBTxt::T( 'Not authorized.' ), 'error' );
		}

		$closed			=	new ClosedTable();

		$closed->load( array( 'user_id' => $disableUser->get( 'id', 0, GetterInterface::INT ), 'type' => 'disable_request' ) );

		if ( ! $closed->get( 'id', 0, GetterInterface::INT ) ) {
			cbRedirect( 'index.php', CBTxt::T( 'Not authorized.' ), 'error' );
		}

		if ( $closed->getError() || ( ! $closed->delete() ) ) {
			$_CB_framework->enqueueMessage( CBTxt::T( 'ACCOUNT_DISABLE_FAILED_TO_CANCEL', 'Account disable request failed to cancel! Error: [error]', array( '[error]' => $closed->getError() ) ), 'error' );

			$this->showDisableEdit( $userId, $user );
			return;
		}

		cbRedirect( $_CB_framework->userProfileUrl( $userId, false ), CBTxt::T( 'Account disable request cancelled.' ) );
	}

	/**
	 * Disables a users account
	 *
	 * @param int       $userId
	 * @param UserTable $user
	 */
	private function confirmDisable( $userId, $user )
	{
		global $_CB_framework, $_PLUGINS;

		$token				=	$this->input( 'confirm', null, GetterInterface::STRING );

		if ( ! $userId ) {
			$userId			=	$user->get( 'id', 0, GetterInterface::INT );
		}

		$closed				=	new ClosedTable();

		if ( $token ) {
			$closed->load( array( 'type' => 'disable_request', 'token' => $token ) );

			if ( ! $closed->get( 'id', 0, GetterInterface::INT ) ) {
				cbRedirect( 'index.php', CBTxt::T( 'Not authorized.' ), 'error' );
			}

			$userId			=	$closed->get( 'user_id', 0, GetterInterface::INT );

			if ( ! $user->get( 'id', 0, GetterInterface::INT ) ) {
				$user		=	CBuser::getUserDataInstance( $userId );
			}
		} elseif ( ( ! Application::User( $user->get( 'id', 0, GetterInterface::INT ) )->isGlobalModerator() ) || ( $userId == $user->get( 'id', 0, GetterInterface::INT ) ) ) {
			// Only moderators can directly bypass confirmation unless the confirmation is for themself:
			cbRedirect( 'index.php', CBTxt::T( 'Not authorized.' ), 'error' );
		}

		if ( ( ! $token ) && ( ! $this->getDisableField( $userId, $user ) ) ) {
			// Only check for field access if we don't have a valid token, which confirms we do have access:
			cbRedirect( 'index.php', CBTxt::T( 'Not authorized.' ), 'error' );
		}

		$cbUser				=	CBuser::getInstance( $userId, false );
		$disableUser		=	$cbUser->getUserData();

		if ( ! $disableUser->get( 'id', 0, GetterInterface::INT ) ) {
			cbRedirect( 'index.php', CBTxt::T( 'Not authorized.' ), 'error' );
		}

		$_PLUGINS->trigger( 'privacy_onBeforeAccountDisable', array( &$disableUser, $user ) );

		$disableUser->set( 'block', 1 );

		if ( ! $disableUser->storeBlock() ) {
			cbRedirect( $_CB_framework->userProfileUrl( $userId, false ), CBTxt::T( 'ACCOUNT_FAILED_TO_DISABLE', 'Account failed to disable! Error: [error]', array( '[error]' => $disableUser->getError() ) ), 'error' );
		}

		if ( ! $closed->get( 'id', 0, GetterInterface::INT ) ) {
			$closed->set( 'user_id', $disableUser->get( 'id', 0, GetterInterface::INT ) );
			$closed->set( 'username', $disableUser->getString( 'username', '' ) );
			$closed->set( 'name', $disableUser->getString( 'name', '' ) );
			$closed->set( 'email', $disableUser->getString( 'email', '' ) );
			$closed->set( 'reason', $this->input( 'reason', null, GetterInterface::STRING ) );
			$closed->set( 'token', md5( 'disable_account_' . $disableUser->get( 'id', 0, GetterInterface::INT ) . '_' . $_CB_framework->getCfg( 'secret' ) ) );
		}

		$closed->set( 'type', 'disable' );
		$closed->set( 'date', Application::Database()->getUtcDateTime() );

		if ( $closed->getError() || ( ! $closed->check() ) ) {
			cbRedirect( $_CB_framework->userProfileUrl( $userId, false ), CBTxt::T( 'ACCOUNT_FAILED_TO_DISABLE', 'Account failed to disable! Error: [error]', array( '[error]' => $closed->getError() ) ), 'error' );
		}

		if ( $closed->getError() || ( ! $closed->store() ) ) {
			cbRedirect( $_CB_framework->userProfileUrl( $userId, false ), CBTxt::T( 'ACCOUNT_FAILED_TO_DISABLE', 'Account failed to disable! Error: [error]', array( '[error]' => $closed->getError() ) ), 'error' );
		}

		$notification		=	new cbNotification();

		$extra				=	array(	'ip_address'	=>	Application::Input()->getRequestIP(),
										'reason'		=>	$closed->getString( 'reason', '' ),
										'date'			=>	cbFormatDate( $closed->getString( 'date', '' ) )
									);

		if ( $this->params->get( 'disable_notify', true, GetterInterface::BOOLEAN ) ) {
			$subject		=	$cbUser->replaceUserVars( CBTxt::T( 'User Account Disabled' ), true, false, $extra, false );
			$body			=	$cbUser->replaceUserVars( CBTxt::T( 'Name: [name]<br />Username: [username]<br />Email: [email]<br />IP Address: [ip_address]<br />Date: [date]<br /><br />[reason]<br /><br />' ), false, false, $extra, false );

			if ( $subject && $body ) {
				$notification->sendToModerators( $subject, $body, false, 1 );
			}
		}

		$savedLanguage		=	CBTxt::setLanguage( $disableUser->getUserLanguage() );

		$subject			=	CBTxt::T( 'Your Account has been Disabled' );
		$body				=	CBTxt::T( 'This is a notice that your account [username] on [siteurl] has been disabled.' );

		CBTxt::setLanguage( $savedLanguage );

		if ( $subject && $body ) {
			$notification->sendFromSystem( $disableUser, $subject, $body, true, 1, null, null, null, $extra );
		}

		$_PLUGINS->trigger( 'privacy_onAfterAccountDisable', array( $disableUser, $user ) );

		if ( ( $closed->get( 'user_id', 0, GetterInterface::INT ) == $user->get( 'id', 0, GetterInterface::INT ) ) && Application::MyUser()->getUserId() ) {
			// Only logout if the user being disabled is themself:
			$_CB_framework->logout();
		}

		cbRedirect( 'index.php', CBTxt::T( 'Account disabled successfully!' ) );
	}

	/**
	 * Displays account enable form
	 *
	 * @param int       $userId
	 * @param UserTable $user
	 */
	private function saveEnable( $userId, $user )
	{
		global $_CB_framework;

		if ( ! $userId ) {
			$userId			=	$user->get( 'id', 0, GetterInterface::INT );
		}

		if ( ( ! $userId ) || $user->get( 'id', 0, GetterInterface::INT ) || ( ! $this->params->get( 'disabled_enable', true, GetterInterface::BOOLEAN ) ) ) {
			cbRedirect( 'index.php', CBTxt::T( 'Not authorized.' ), 'error' );
		}

		$cbUser				=	CBuser::getInstance( $userId, false );
		$disabledUser		=	$cbUser->getUserData();

		if ( ( ! $disabledUser->get( 'block', 0, GetterInterface::INT ) ) || ( ! $disabledUser->get( 'approved', 1, GetterInterface::INT ) ) || ( ! $disabledUser->get( 'confirmed', 1, GetterInterface::INT ) ) ) {
			cbRedirect( 'index.php', CBTxt::T( 'Not authorized.' ), 'error' );
		}

		$closed				=	new ClosedTable();

		$closed->load( array( 'user_id' => $disabledUser->get( 'id', 0, GetterInterface::INT ), 'type' => 'disable' ) );

		if ( ! $closed->get( 'id', 0, GetterInterface::INT ) ) {
			$closed->load( array( 'user_id' => $disabledUser->get( 'id', 0, GetterInterface::INT ), 'type' => 'pending' ) );

			if ( $closed->get( 'id', 0, GetterInterface::INT ) ) {
				cbRedirect( 'index.php', CBTxt::T( 'Your account is already awaiting confirmation to re-enable. Please check your email for your confirmation.' ) );
			}

			cbRedirect( 'index.php', CBTxt::T( 'Not authorized.' ), 'error' );
		}

		$closed->set( 'type', 'pending' );
		$closed->set( 'date', Application::Database()->getUtcDateTime() );

		$closed->store();

		cbRedirect( $_CB_framework->viewUrl( 'login' ), CBTxt::T( 'Please login to re-enable your account.' ) );
	}

	/**
	 * Checks if the delete account field is accessible to a user
	 *
	 * @param int       $userId
	 * @param UserTable $user
	 * @return bool
	 */
	private function getDeleteField( $userId, $user )
	{
		if ( ( ! $userId )
			 || ( ( $userId != $user->get( 'id', 0, GetterInterface::INT ) ) && ( ! Application::User( $user->get( 'id', 0, GetterInterface::INT ) )->isGlobalModerator() ) )
			 || ( Application::User( $userId )->isGlobalModerator() && ( ! $this->params->get( 'delete_moderator', false, GetterInterface::BOOLEAN ) ) )
		) {
			return false;
		}

		$fields		=	CBuser::getInstance( $userId, false )->_getCbTabs( false )->_getTabFieldsDb( null, $user, 'edit', 'privacy_delete_me' );

		return ( $fields ? true : false );
	}

	/**
	 * Displays account delete form
	 *
	 * @param int       $userId
	 * @param UserTable $user
	 */
	private function showDeleteEdit( $userId, $user )
	{
		if ( ! $userId ) {
			$userId		=	$user->get( 'id', 0, GetterInterface::INT );
		}

		if ( ! $this->getDeleteField( $userId, $user ) ) {
			cbRedirect( 'index.php', CBTxt::T( 'Not authorized.' ), 'error' );
		}

		$closed			=	new ClosedTable();

		$closed->load( array( 'user_id' => $userId, 'type' => 'delete_request' ) );

		if ( $closed->get( 'id', 0, GetterInterface::INT ) ) {
			require CBPrivacy::getTemplate( null, 'delete_pending' );
		} else {
			require CBPrivacy::getTemplate( null, 'delete' );
		}
	}

	/**
	 * Requests to delete a users account
	 *
	 * @param int       $userId
	 * @param UserTable $user
	 */
	private function saveDelete( $userId, $user )
	{
		global $_CB_framework, $_PLUGINS;

		if ( ! $userId ) {
			$userId			=	$user->get( 'id', 0, GetterInterface::INT );
		}

		if ( ! $this->getDeleteField( $userId, $user ) ) {
			cbRedirect( 'index.php', CBTxt::T( 'Not authorized.' ), 'error' );
		}

		if ( ( $userId != $user->get( 'id', 0, GetterInterface::INT ) ) && Application::User( $user->get( 'id', 0, GetterInterface::INT ) )->isGlobalModerator() ) {
			// Moderator is deleting another users account so directly bypass to the confirm delete process:
			$this->confirmDelete( $userId, $user );
			return;
		}

		$cbUser				=	CBuser::getInstance( $userId, false );
		$deleteUser			=	$cbUser->getUserData();

		if ( ! $deleteUser->get( 'id', 0, GetterInterface::INT ) ) {
			cbRedirect( 'index.php', CBTxt::T( 'Not authorized.' ), 'error' );
		}

		$_PLUGINS->trigger( 'privacy_onBeforeAccountDeleteRequest', array( &$deleteUser, $user ) );

		$closed				=	new ClosedTable();

		$closed->set( 'user_id', $deleteUser->get( 'id', 0, GetterInterface::INT ) );
		$closed->set( 'username', $deleteUser->getString( 'username', '' ) );
		$closed->set( 'name', $deleteUser->getString( 'name', '' ) );
		$closed->set( 'email', $deleteUser->getString( 'email', '' ) );
		$closed->set( 'type', 'delete_request' );
		$closed->set( 'reason', $this->input( 'reason', null, GetterInterface::STRING ) );
		$closed->set( 'token', md5( 'delete_account_' . $deleteUser->get( 'id', 0, GetterInterface::INT ) . '_' . $_CB_framework->getCfg( 'secret' ) ) );

		if ( $closed->getError() || ( ! $closed->check() ) ) {
			$_CB_framework->enqueueMessage( CBTxt::T( 'ACCOUNT_FAILED_TO_DELETE', 'Account failed to delete! Error: [error]', array( '[error]' => $closed->getError() ) ), 'error' );

			$this->showDeleteEdit( $userId, $user );
			return;
		}

		if ( $closed->getError() || ( ! $closed->store() ) ) {
			$_CB_framework->enqueueMessage( CBTxt::T( 'ACCOUNT_FAILED_TO_DELETE', 'Account failed to delete! Error: [error]', array( '[error]' => $closed->getError() ) ), 'error' );

			$this->showDeleteEdit( $userId, $user );
			return;
		}

		$notification		=	new cbNotification();

		$confirmUrl			=	$_CB_framework->pluginClassUrl( $this->element, false, array( 'action' => 'delete', 'func' => 'confirm', 'confirm' => $closed->getString( 'token' ) ) );

		$extra				=	array(	'ip_address'	=>	Application::Input()->getRequestIP(),
										'reason'		=>	$closed->getString( 'reason', '' ),
										'date'			=>	cbFormatDate( $closed->getString( 'date', '' ) )
									);

		$savedLanguage		=	CBTxt::setLanguage( $deleteUser->getUserLanguage() );

		$subject			=	CBTxt::T( 'Confirm Account Delete' );
		$body				=	CBTxt::T( 'NOTICE_CONFIRM_ACCOUNT_DELETE', 'This is a notice that your account [username] on [siteurl] has been requested to be deleted. Please confirm the account delete by clicking the following link.<br /><br />[confirm]', array( '[confirm]' => '<a href="' . htmlspecialchars( $confirmUrl ) . '">' . $confirmUrl . '</a>', '[confirm_url]' => htmlspecialchars( $confirmUrl ) ) );

		CBTxt::setLanguage( $savedLanguage );

		if ( $subject && $body ) {
			$notification->sendFromSystem( $deleteUser, $subject, $body, true, 1, null, null, null, $extra );
		}

		$_PLUGINS->trigger( 'privacy_onAfterAccountDeleteRequest', array( $deleteUser, $user ) );

		$_CB_framework->enqueueMessage( CBTxt::T( 'Your account delete request requires confirmation. Please check your email and confirm.' ) );

		$this->showDeleteEdit( $userId, $user );
	}

	/**
	 * Requests to delete a users account
	 *
	 * @param int       $userId
	 * @param UserTable $user
	 */
	private function cancelDelete( $userId, $user )
	{
		global $_CB_framework;

		if ( ! $userId ) {
			$userId		=	$user->get( 'id', 0, GetterInterface::INT );
		}

		if ( ! $this->getDeleteField( $userId, $user ) ) {
			cbRedirect( 'index.php', CBTxt::T( 'Not authorized.' ), 'error' );
		}

		$cbUser			=	CBuser::getInstance( $userId, false );
		$deleteUser		=	$cbUser->getUserData();

		if ( ! $deleteUser->get( 'id', 0, GetterInterface::INT ) ) {
			cbRedirect( 'index.php', CBTxt::T( 'Not authorized.' ), 'error' );
		}

		$closed			=	new ClosedTable();

		$closed->load( array( 'user_id' => $deleteUser->get( 'id', 0, GetterInterface::INT ), 'type' => 'delete_request' ) );

		if ( ! $closed->get( 'id', 0, GetterInterface::INT ) ) {
			cbRedirect( 'index.php', CBTxt::T( 'Not authorized.' ), 'error' );
		}

		if ( $closed->getError() || ( ! $closed->delete() ) ) {
			$_CB_framework->enqueueMessage( CBTxt::T( 'ACCOUNT_DELETE_FAILED_TO_CANCEL', 'Account delete request failed to cancel! Error: [error]', array( '[error]' => $closed->getError() ) ), 'error' );

			$this->showDeleteEdit( $userId, $user );
			return;
		}

		cbRedirect( $_CB_framework->userProfileUrl( $userId, false ), CBTxt::T( 'Account delete request cancelled.' ) );
	}

	/**
	 * Deletes a users account
	 *
	 * @param int       $userId
	 * @param UserTable $user
	 */
	private function confirmDelete( $userId, $user )
	{
		global $_CB_framework, $_PLUGINS;

		$token				=	$this->input( 'confirm', null, GetterInterface::STRING );

		if ( ! $userId ) {
			$userId			=	$user->get( 'id', 0, GetterInterface::INT );
		}

		$closed				=	new ClosedTable();

		if ( $token ) {
			$closed->load( array( 'type' => 'delete_request', 'token' => $token ) );

			if ( ! $closed->get( 'id', 0, GetterInterface::INT ) ) {
				cbRedirect( 'index.php', CBTxt::T( 'Not authorized.' ), 'error' );
			}

			$userId			=	$closed->get( 'user_id', 0, GetterInterface::INT );

			if ( ! $user->get( 'id', 0, GetterInterface::INT ) ) {
				$user		=	CBuser::getUserDataInstance( $userId );
			}
		} elseif ( ( ! Application::User( $user->get( 'id', 0, GetterInterface::INT ) )->isGlobalModerator() ) || ( $userId == $user->get( 'id', 0, GetterInterface::INT ) ) ) {
			// Only moderators can directly bypass confirmation unless the confirmation is for themself:
			cbRedirect( 'index.php', CBTxt::T( 'Not authorized.' ), 'error' );
		}

		if ( ( ! $token ) && ( ! $this->getDeleteField( $userId, $user ) ) ) {
			// Only check for field access if we don't have a valid token, which confirms we do have access:
			cbRedirect( 'index.php', CBTxt::T( 'Not authorized.' ), 'error' );
		}

		$cbUser				=	CBuser::getInstance( $userId, false );
		$deleteUser			=	$cbUser->getUserData();

		if ( ! $deleteUser->get( 'id', 0, GetterInterface::INT ) ) {
			cbRedirect( 'index.php', CBTxt::T( 'Not authorized.' ), 'error' );
		}

		$_PLUGINS->trigger( 'privacy_onBeforeAccountDelete', array( &$deleteUser, $user ) );

		$deleteUser->set( 'block', 1 );

		if ( ! $deleteUser->delete( $userId ) ) {
			cbRedirect( $_CB_framework->userProfileUrl( $userId, false ), CBTxt::T( 'ACCOUNT_FAILED_TO_DELETE', 'Account failed to delete! Error: [error]', array( '[error]' => $deleteUser->getError() ) ), 'error' );
		}

		if ( ! $closed->get( 'id', 0, GetterInterface::INT ) ) {
			$closed->set( 'user_id', $deleteUser->get( 'id', 0, GetterInterface::INT ) );
			$closed->set( 'username', $deleteUser->getString( 'username', '' ) );
			$closed->set( 'name', $deleteUser->getString( 'name', '' ) );
			$closed->set( 'email', $deleteUser->getString( 'email', '' ) );
			$closed->set( 'reason', $this->input( 'reason', null, GetterInterface::STRING ) );
			$closed->set( 'token', md5( 'delete_account_' . $deleteUser->get( 'id', 0, GetterInterface::INT ) . '_' . $_CB_framework->getCfg( 'secret' ) ) );
		}

		$closed->set( 'type', 'delete' );
		$closed->set( 'date', Application::Database()->getUtcDateTime() );

		if ( $closed->getError() || ( ! $closed->check() ) ) {
			cbRedirect( $_CB_framework->userProfileUrl( $userId, false ), CBTxt::T( 'ACCOUNT_FAILED_TO_DELETE', 'Account failed to delete! Error: [error]', array( '[error]' => $closed->getError() ) ), 'error' );
		}

		if ( $closed->getError() || ( ! $closed->store() ) ) {
			cbRedirect( $_CB_framework->userProfileUrl( $userId, false ), CBTxt::T( 'ACCOUNT_FAILED_TO_DELETE', 'Account failed to delete! Error: [error]', array( '[error]' => $closed->getError() ) ), 'error' );
		}

		$notification		=	new cbNotification();

		$extra				=	array(	'ip_address'	=>	Application::Input()->getRequestIP(),
										'reason'		=>	$closed->getString( 'reason', '' ),
										'date'			=>	cbFormatDate( $closed->getString( 'date', '' ) )
									);

		if ( $this->params->get( 'delete_notify', true, GetterInterface::BOOLEAN ) ) {
			$subject		=	$cbUser->replaceUserVars( CBTxt::T( 'User Account Deleted' ), true, false, $extra, false );
			$body			=	$cbUser->replaceUserVars( CBTxt::T( 'Name: [name]<br />Username: [username]<br />Email: [email]<br />IP Address: [ip_address]<br />Date: [date]<br /><br />[reason]<br /><br />' ), false, false, $extra, false );

			if ( $subject && $body ) {
				$notification->sendToModerators( $subject, $body, false, 1 );
			}
		}

		$savedLanguage		=	CBTxt::setLanguage( $deleteUser->getUserLanguage() );

		$subject			=	CBTxt::T( 'Your Account has been Deleted' );
		$body				=	CBTxt::T( 'This is a notice that your account [username] on [siteurl] has been deleted.' );

		CBTxt::setLanguage( $savedLanguage );

		if ( $subject && $body ) {
			$notification->sendFromSystem( $deleteUser, $subject, $body, true, 1, null, null, null, $extra );
		}

		$_PLUGINS->trigger( 'privacy_onAfterAccountDelete', array( $deleteUser, $user ) );

		if ( ( $closed->get( 'user_id', 0, GetterInterface::INT ) == $user->get( 'id', 0, GetterInterface::INT ) ) && Application::MyUser()->getUserId() ) {
			// Only logout if the user being deleted is themself:
			$_CB_framework->logout();
		}

		cbRedirect( 'index.php', CBTxt::T( 'Account deleted successfully!' ) );
	}

	/**
	 * Checks if the export account field is accessible to a user
	 *
	 * @param int       $userId
	 * @param UserTable $user
	 * @return bool
	 */
	private function getExportField( $userId, $user )
	{
		if ( ( ! $userId )
			 || ( ( $userId != $user->get( 'id', 0, GetterInterface::INT ) ) && ( ! Application::User( $user->get( 'id', 0, GetterInterface::INT ) )->isGlobalModerator() ) )
		) {
			return false;
		}

		$fields		=	CBuser::getInstance( $userId, false )->_getCbTabs( false )->_getTabFieldsDb( null, $user, 'edit', 'privacy_export' );

		return ( $fields ? true : false );
	}

	/**
	 * Requests to have user data exported
	 *
	 * @param int       $userId
	 * @param UserTable $user
	 */
	private function requestExportData( $userId, $user )
	{
		global $_CB_framework, $_PLUGINS;

		if ( ! $userId ) {
			$userId			=	$user->get( 'id', 0, GetterInterface::INT );
		}

		if ( ! $this->getExportField( $userId, $user ) ) {
			cbRedirect( 'index.php', CBTxt::T( 'Not authorized.' ), 'error' );
		}

		if ( ( $userId != $user->get( 'id', 0, GetterInterface::INT ) ) && Application::User( $user->get( 'id', 0, GetterInterface::INT ) )->isGlobalModerator() ) {
			// Moderator is exporting another users account so directly bypass to the confirm export process:
			$this->confirmExportData( $userId, $user );
			return;
		}

		$cbUser				=	CBuser::getInstance( $userId, false );
		$exportUser			=	$cbUser->getUserData();

		if ( ! $exportUser->get( 'id', 0, GetterInterface::INT ) ) {
			cbRedirect( 'index.php', CBTxt::T( 'Not authorized.' ), 'error' );
		}

		$_PLUGINS->trigger( 'privacy_onBeforeAccountExportRequest', array( &$exportUser, $user ) );

		$notification		=	new cbNotification();

		$exportUrl			=	$_CB_framework->pluginClassUrl( $this->element, false, array( 'action' => 'export', 'func' => 'export', 'id' => $exportUser->get( 'id', 0, GetterInterface::INT ), 'confirm' => md5( 'export_account_' . $userId . '_' . $_CB_framework->getCfg( 'secret' ) . '_' . Application::Date( 'now', 'UTC' )->format( 'Y-m-d' ) ) ) );

		$extra				=	array(	'ip_address'	=>	Application::Input()->getRequestIP(),
										'date'			=>	cbFormatDate( 'now' )
									);

		$savedLanguage		=	CBTxt::setLanguage( $exportUser->getUserLanguage() );

		$subject			=	CBTxt::T( 'Confirm Account Export' );
		$body				=	CBTxt::T( 'NOTICE_CONFIRM_ACCOUNT_EXPORT', 'This is a notice that your account [username] on [siteurl] has been requested to be exported. Your exported data can be accessed at the following link.<br /><br />[export]', array( '[export]' => '<a href="' . htmlspecialchars( $exportUrl ) . '">' . $exportUrl . '</a>', '[export_url]' => htmlspecialchars( $exportUrl ) ) );

		CBTxt::setLanguage( $savedLanguage );

		if ( $subject && $body ) {
			$notification->sendFromSystem( $exportUser, $subject, $body, true, 1, null, null, null, $extra );
		}

		$_PLUGINS->trigger( 'privacy_onAfterAccountExportRequest', array( $exportUser, $user ) );

		cbRedirect( $_CB_framework->userProfileEditUrl( (int) $userId, false ), CBTxt::T( 'Your account export request requires email confirmation. Please check your email for a link to export your data. This request is only valid for 24 hours.' ) );
	}

	/**
	 * Prepares and delivers user data in a portable format
	 *
	 * @param int       $userId
	 * @param UserTable $user
	 */
	private function confirmExportData( $userId, $user )
	{
		global $_CB_framework, $_PLUGINS;

		$token								=	$this->input( 'confirm', null, GetterInterface::STRING );

		if ( ! $userId ) {
			$userId							=	$user->get( 'id', 0, GetterInterface::INT );
		}

		if ( $token ) {
			if ( $token != md5( 'export_account_' . $userId . '_' . $_CB_framework->getCfg( 'secret' ) . '_' . Application::Date( 'now', 'UTC' )->format( 'Y-m-d' ) ) ) {
				cbRedirect( 'index.php', CBTxt::T( 'Confirmation token has expired or is invalid.' ), 'error' );
			}

			if ( ! $user->get( 'id', 0, GetterInterface::INT ) ) {
				$user						=	CBuser::getUserDataInstance( $userId );
			}
		} elseif ( ( ! Application::User( $user->get( 'id', 0, GetterInterface::INT ) )->isGlobalModerator() ) || ( $userId == $user->get( 'id', 0, GetterInterface::INT ) ) ) {
			// Only moderators can directly bypass confirmation unless the confirmation is for themself:
			cbRedirect( 'index.php', CBTxt::T( 'Not authorized.' ), 'error' );
		}

		if ( ! $this->getExportField( $userId, $user ) ) {
			cbRedirect( 'index.php', CBTxt::T( 'Not authorized.' ), 'error' );
		}

		$exportFormat						=	$this->params->getString( 'export_format', 'xml' );
		$exportEmpty						=	$this->params->get( 'export_empty', false, GetterInterface::BOOLEAN );

		// Exclude some core columns as they're not relevant to the user:
		$coreExclude						=	array(	'user_id', 'password', 'gids', 'sendEmail', 'block', 'confirmed', 'approved',
														'banned', 'resetCount', 'lastResetTime', 'requireReset', 'message_last_sent',
														'message_number_sent', 'params', 'otpKey', 'otep'
													);

		// Add the optional exclude fields:
		$exclude							=	array_unique( array_merge( $coreExclude, explode( '|*|', $this->params->getString( 'export_exclude', 'avatar|*|canvas' ) ) ) );

		$_PLUGINS->loadPluginGroup( 'user' );

		$cbUser								=	CBuser::getInstance( (int) $userId, false );
		$row								=	$cbUser->getUserData();

		if ( ! $row->get( 'id', 0, GetterInterface::INT ) ) {
			cbRedirect( 'index.php', CBTxt::T( 'Not authorized.' ), 'error' );
		}

		$rowData							=	array();
		$dataArray							=	get_object_vars( $row );

		if ( isset( $dataArray['id'] ) ) {
			// Ensure the primary key is always ordered first:
			$dataArray						=	( array( 'id' => $dataArray['id'] ) + $dataArray );
		}

		$childColumns						=	array( 'approved', 'position', 'consent' );

		foreach ( $dataArray as $k => $v ) {
			if ( ( ! $k ) || in_array( $k, $exclude ) || ( $k[0] == '_' ) ) {
				// Skip empty keys, excluded keys, and private variables:
				continue;
			}

			foreach ( $childColumns as $childColumn ) {
				$colPosition				=	strpos( $k, $childColumn );

				if ( $colPosition !== false ) {
					if ( substr( $k, -strlen( $childColumn ) ) !== $childColumn ) {
						// These are always at the end so if it's not skip checking parent:
						continue;
					}

					$parentCol				=	substr( $k, 0, $colPosition );

					if ( ! isset( $rowData[$parentCol] ) ) {
						// This column belongs to another field, but that field doesn't exist so skip this one:
						continue 2;
					}
				}
			}

			if ( ( ( $v === null ) || ( $v === '' ) ) && ( ! $exportEmpty ) ) {
				// Skip empty values if we're excluding them from the export:
				continue;
			}

			$rowData[$k]					=	$v;
		}

		foreach ( $this->params->subTree( 'export_custom' ) as $customExport ) {
			$k								=	$customExport->getString( 'key', '' );

			if ( ! $k ) {
				continue;
			}

			$k								=	$cbUser->replaceUserVars( $k );

			// Always use lowercase names:
			$k								=	strtolower( $k );
			// Replace all invalid characters:
			$k								=	preg_replace( '/[^a-zA-Z0-9_]+/', '', $k );
			// Replace duplicate underscores:
			$k								=	preg_replace( '/(_{2,})+/', '', $k );
			// Replace leading underscores:
			$k								=	preg_replace( '/^_/', '', $k );

			if ( ! $k ) {
				continue;
			}

			$v								=	$customExport->get( 'value', '', GetterInterface::HTML );

			if ( ( ( $v === null ) || ( $v === '' ) ) && ( ! $exportEmpty ) ) {
				// Skip empty values if we're excluding them from the export:
				continue;
			}

			$v								=	$cbUser->replaceUserVars( $v );

			if ( ( ( $v === null ) || ( $v === '' ) ) && ( ! $exportEmpty ) ) {
				// Skip empty values if we're excluding them from the export:
				continue;
			}

			$rowData[$k]					=	$v;
		}

		// Class specific export trigger (e.g. onExportUserTable):
		$_PLUGINS->trigger( 'onExportUserTable', array( $row, &$rowData, $exportFormat ) );

		if ( ! $rowData ) {
			// Skip export as this row has no data to export:
			cbRedirect( 'index.php', CBTxt::T( 'Nothing to export.' ), 'error' );
		}

		$exportFormatted					=	null;

		switch ( $exportFormat ) {
			case 'csv':
				$csvHeaders					=	array();

				foreach ( $rowData as $dataName => $dataValue ) {
					if ( ! in_array( $dataName, $csvHeaders ) ) {
						$csvHeaders[]		=	$dataName;
					}
				}

				$exportFormatted			.=	implode( ',', $csvHeaders ) . "\n";

				$csvData					=	array();

				foreach ( $csvHeaders as $csvHeader ) {
					if ( ! isset( $rowData[$csvHeader] ) ) {
						// Column doesn't exist for this row, but we need it anyway for proper CSV formatting so set to empty:
						$csvData[]			=	'""';
						continue;
					}

					$dataValue				=	$this->getExportValue( $rowData[$csvHeader] );

					if ( is_bool( $dataValue ) ) {
						$dataValue			=	( $dataValue ? 'true' : 'false' );
					} elseif ( $dataValue && ( ! is_numeric( $dataValue ) ) ) {
						if ( ( Get::clean( $dataValue, GetterInterface::STRING ) != $dataValue ) // Check if contains any HTML
							 || ( strpos( $dataValue, ',' ) !== false ) // Check if contains a comma
							 || ( strpos( $dataValue, '"' ) !== false ) // Check if contains double quote
							 || ( strpos( $dataValue, "\r" ) !== false ) // Check if contains linebreak
							 || ( strpos( $dataValue, "\n" ) !== false ) // Check if contains linebreak
							 || ( $dataValue[0] === '=' ) // Check if begins with a formula
						) {
							// Contains characters that could malform the CSV so enclose in quotes to treat as string; note quotes must be double quoted to reverse correctly:
							$dataValue		=	'"' . str_replace( '"', '""', $dataValue ) . '"';
						}
					}

					if ( ( $dataValue === null ) || ( $dataValue === '' ) ) {
						// CSV can't have a completely empty space so just treat it as empty string:
						$dataValue			=	'""';
					}

					$csvData[]				=	$dataValue;
				}

				$exportFormatted			.=	implode( ',', $csvData ) . "\n";
				break;
			case 'xml':
				$exportFormatted			.=	'<?xml version="1.0" encoding="utf-8"?>'
											.	"\n<rows>"
											.		"\n\t<row>";

				foreach ( $rowData as $dataName => $dataValue ) {
					$dataValue				=	$this->getExportValue( $dataValue );

					if ( is_bool( $dataValue ) ) {
						$dataValue			=	( $dataValue ? 'true' : 'false' );
					} elseif ( $dataValue && ( ! is_numeric( $dataValue ) ) ) {
						if ( ( Get::clean( $dataValue, GetterInterface::STRING ) != $dataValue ) // Check if contains any HTML
							 || ( strpos( $dataValue, '<' ) !== false ) // Check if contains a lt
							 || ( strpos( $dataValue, '&' ) !== false ) // Check if contains a amp
							 || ( strpos( $dataValue, ']]>' ) !== false ) // Check if it contains CDATA closing token
						) {
							// Contains characters that could malform the XML so enclose in CDATA:
							$dataValue		=	'<![CDATA[' . str_replace( ']]>', ']]]]><![CDATA[>', $dataValue ) . ']]>';
						}
					}

					if ( ( $dataValue === null ) || ( $dataValue === '' ) ) {
						$exportFormatted	.=		"\n\t\t<" . htmlspecialchars( $dataName ) . " />";
					} else {
						$exportFormatted	.=		"\n\t\t<" . htmlspecialchars( $dataName ) . ">"
											.			$dataValue
											.		'</' . htmlspecialchars( $dataName ) . '>';
					}
				}

				$exportFormatted			.=		"\n\t</row>"
											.	"\n</rows>";
				break;
			case 'json':
				$exportFormatted			=	json_encode( $rowData );
				break;
		}

		while ( @ob_end_clean() );

		header( 'HTTP/1.0 200 OK' );

		switch ( $exportFormat ) {
			case 'csv':
				header( 'Content-Disposition: attachment; filename="profile.csv"' );
				header( 'Content-Type: application/csv; charset=UTF-8' );
				break;
			case 'json':
				header( 'Content-Disposition: attachment; filename="profile.txt"' );
				header( 'Content-Type: text/plain; charset=UTF-8' );
				break;
			case 'xml':
				header( 'Content-Disposition: attachment; filename="profile.xml"' );
				header( 'Content-Type: application/xml; charset=UTF-8' );
				break;
		}

		header( 'Content-Transfer-Encoding: binary' );
		header( 'Pragma: private' );
		header( 'Cache-Control: private' );
		header( 'Content-Length: ' . strlen( $exportFormatted ) );
		header( 'Connection: close' );

		echo $exportFormatted;

		exit();
	}

	/**
	 * Properly converts array or object data values
	 *
	 * @param mixed $dataValue
	 * @return mixed
	 */
	private function getExportValue( $dataValue )
	{
		if ( is_array( $dataValue ) || is_object( $dataValue ) ) {
			$dataValue			=	json_encode( $dataValue );
		} elseif ( is_object( $dataValue ) ) {
			if ( method_exists( $dataValue, 'asArray' ) ) {
				$dataValue		=	$dataValue->asArray();
			} else {
				$dataValue		=	get_object_vars( $dataValue );
			}

			if ( $dataValue ) {
				$dataValue		=	json_encode( $dataValue );
			} else {
				$dataValue		=	null;
			}
		}

		return $dataValue;
	}

	/**
	 * Displays privacy edit
	 *
	 * @param UserTable $viewer
	 * @param Privacy   $privacy
	 */
	private function showPrivacyEdit( $viewer, $privacy )
	{
		global $_CB_framework;

		if ( ! $privacy->rules() ) {
			// No rules are available so skip display:
			return;
		}

		if ( $privacy->getString( 'layout', 'button' ) == 'hidden' ) {
			// This is a hidden privacy control so don't bother querying for or outputting anything:
			return;
		}

		if ( $privacy->get( 'ajax', false, GetterInterface::BOOLEAN ) && ( ( ! $privacy->user()->get( 'id', 0, GetterInterface::INT ) ) || preg_match( '/\.0$/', $privacy->asset() ) ) ) {
			$privacy->set( 'ajax', false );
		}

		/** @noinspection PhpUnusedLocalVariableInspection */
		$rows						=	array();
		$selected					=	array();

		if ( $privacy->get( 'query', true, GetterInterface::BOOLEAN ) ) {
			$rows					=	CBPrivacy::getPrivacy( $privacy->user() );

			if ( ! isset( $rows[$privacy->asset()] ) ) {
				/** @noinspection PhpUnusedLocalVariableInspection */
				$rows				=	array();
			} else {
				$rows				=	$rows[$privacy->asset()];

				/** @var PrivacyTable[] $rows */
				foreach ( $rows as $row ) {
					$rule			=	$row->getString( 'rule', '' );

					if ( $rule === null ) {
						continue;
					}

					$selected[]		=	$rule;
				}
			}
		}

		$ajaxUrl					=	null;

		if ( $privacy->get( 'ajax', false, GetterInterface::BOOLEAN ) ) {
			$ajaxUrl				=	$_CB_framework->pluginClassUrl( $this->element, true, array( 'action' => 'privacy', 'func' => 'save', 'privacy' => $privacy->id() ), 'raw', 0, true );
		}

		$usersUrl					=	null;

		if ( array_key_exists( 'USER-ADD', $privacy->rules( true ) ) ) {
			$usersUrl				=	$_CB_framework->pluginClassUrl( $this->element, true, array( 'action' => 'privacy', 'func' => 'users', 'privacy' => $privacy->id() ), 'raw', 0, true );
		}

		require CBPrivacy::getTemplate( $privacy->getString( 'template', '' ), 'edit' );
	}

	/**
	 * Save privacy
	 *
	 * @param UserTable $viewer
	 * @param Privacy   $privacy
	 * @param bool      $ajax
	 */
	private function savePrivacy( $viewer, $privacy, $ajax = false )
	{
		global $_CB_database, $_PLUGINS;

		if ( ! $privacy->rules() ) {
			// No rules are available so skip save:
			if ( $ajax ) {
				header( 'HTTP/1.0 401 Unauthorized' );
				exit();
			}

			return;
		}

		if ( $privacy->getString( 'layout', 'button' ) == 'hidden' ) {
			// This privacy is hidden so it's being stored somewhere else so lets skip its save behavior:
			if ( $ajax ) {
				header( 'HTTP/1.0 401 Unauthorized' );
				exit();
			}

			return;
		}

		if ( $ajax && ( ! $privacy->get( 'ajax', false, GetterInterface::BOOLEAN ) ) ) {
			// We're ajax, but this control doesn't allow for ajax so skip the save behavior:
			header( 'HTTP/1.0 401 Unauthorized' );
			exit();
		}

		if ( ! $privacy->get( 'guest', false, GetterInterface::BOOLEAN ) ) {
			if ( ( ! $viewer->get( 'id', 0, GetterInterface::INT ) ) || ( $viewer->get( 'id', 0, GetterInterface::INT ) != $privacy->user()->get( 'id', 0, GetterInterface::INT ) ) && ( ! Application::MyUser()->isGlobalModerator() ) ) {
				if ( $ajax ) {
					header( 'HTTP/1.0 401 Unauthorized' );
					exit();
				}

				return;
			}
		}

		if ( ! $privacy->user()->get( 'id', 0, GetterInterface::INT ) ) {
			$privacy->user( $viewer );
		}

		if ( ! $privacy->user()->get( 'id', 0, GetterInterface::INT ) ) {
			if ( $ajax ) {
				header( 'HTTP/1.0 401 Unauthorized' );
				exit();
			}

			return;
		}

		$rules				=	array();

		if ( $privacy->name() ) {
			// First check by specified name:
			$rules			=	$this->input( $privacy->name(), array(), GetterInterface::RAW );
		}

		if ( ! $rules ) {
			// No specified name was supplied so check by asset:
			$rules			=	$this->input( md5( 'privacy_' . $privacy->asset() ), array(), GetterInterface::RAW );
		}

		if ( ! $rules ) {
			// Check if for post values if this is a new entry:
			$rules			=	$this->input( md5( 'privacy_' . preg_replace( '/\.(\d+)$/', '.0', $privacy->asset() ) ), array(), GetterInterface::RAW );
		}

		if ( ! $rules ) {
			// We found no rules being set for this privacy control so skip its save behavior; SOMETHING always has to be set:
			if ( $ajax ) {
				header( 'HTTP/1.0 401 Unauthorized' );
				exit();
			}

			return;
		}

		// Ensure the selected rules are allowed in this privacy usage (frontend only):
		$userRules					=	array();

		if ( ! Application::Cms()->getClientId() ) {
			foreach ( $rules as $k => $v ) {
				if ( array_key_exists( 'USER-ADD', $privacy->rules( true ) ) && preg_match( '/USER-(\d+)/', $v, $userMatch ) ) {
					$userRuleID			=	(int) $userMatch[1];

					if ( ! in_array( $userRuleID, $userRules ) ) {
						$userRules[]	=	$userRuleID;
					}

					unset( $rules[$k] ); // We'll add back in later after we validate the user exists

					continue;
				}

				if ( ! array_key_exists( $v, $privacy->rules( true ) ) ) {
					unset( $rules[$k] );
				}
			}
		}

		if ( $userRules ) {
			$query					=	"SELECT " . $_CB_database->NameQuote( 'id' )
									.	"\n FROM " . $_CB_database->NameQuote( '#__users' )
									.	"\n WHERE " . $_CB_database->NameQuote( 'id' ) . " IN " . $_CB_database->safeArrayOfIntegers( $userRules );
			$_CB_database->setQuery( $query );
			$privacyUsers			=	$_CB_database->loadResultArray();

			foreach ( $privacyUsers as $privacyUserID ) {
				$rules[]			=	'USER-' . $privacyUserID;
			}
		}

		// Clean out any duplicates or deleted rules:
		foreach ( $privacy->reset()->rows( 'all' ) as $rule ) {
			/** @var PrivacyTable $rule */
			if ( ! in_array( $rule->getString( 'rule', '' ), $rules ) ) {
				// Privacy rule no longer exists so trigger a delete for it:
				$_PLUGINS->trigger( 'privacy_onBeforeDeletePrivacyRule', array( $privacy, &$rule ) );

				if ( $rule->getError() || ( ! $rule->canDelete() ) || ( ! $rule->delete() ) ) {
					continue;
				}

				$_PLUGINS->trigger( 'privacy_onAfterDeletePrivacyRule', array( $privacy, $rule ) );
			} else {
				// Privacy rule already exists for this asset so skip it as we don't want duplicates:
				$key		=	array_search( $rule->getString( 'rule', '' ), $rules );

				if ( $key !== false ) {
					unset( $rules[$key] );
				}
			}
		}

		foreach ( $rules as $privacyRule ) {
			$rule			=	new PrivacyTable(); // No need to load as we handle duplicate cleanup above:

			$rule->set( 'user_id', $privacy->user()->get( 'id', 0, GetterInterface::INT ) );
			$rule->set( 'asset', $privacy->asset() );
			$rule->set( 'rule', Get::clean( $privacyRule, GetterInterface::STRING ) );

			$_PLUGINS->trigger( 'privacy_onBeforeCreatePrivacyRule', array( $privacy, &$rule ) );

			if ( $rule->getError() || ( ! $rule->check() ) || ( ! $rule->store() ) ) {
				continue;
			}

			$_PLUGINS->trigger( 'privacy_onAfterCreatePrivacyRule', array( $privacy, $rule ) );
		}

		$privacy->clear();

		if ( $ajax ) {
			header( 'HTTP/1.0 200 OK' );
			exit();
		}
	}

	/**
	 * @param Privacy $privacy
	 */
	private function showPrivacyUsers( $privacy )
	{
		global $_CB_database;

		if ( ! $privacy->rules() ) {
			// No rules are available so skip display:
			header( 'HTTP/1.0 401 Unauthorized' );
			exit();
		}

		if ( $privacy->getString( 'layout', 'button' ) == 'hidden' ) {
			// This is a hidden privacy control so don't bother querying for or outputting anything:
			header( 'HTTP/1.0 401 Unauthorized' );
			exit();
		}

		if ( ! array_key_exists( 'USER-ADD', $privacy->rules( true ) ) ) {
			header( 'HTTP/1.0 401 Unauthorized' );
			exit();
		}

		$value			=	$this->input( 'value', null, GetterInterface::STRING );

		if ( ! $value ) {
			header( 'HTTP/1.0 401 Unauthorized' );
			exit();
		}

		$query			=	"SELECT *"
						.	"\n FROM " . $_CB_database->NameQuote( '#__comprofiler' ) . " AS c"
						.	"\n INNER JOIN " . $_CB_database->NameQuote( '#__users' ) . ' AS u'
						.	' ON u.' . $_CB_database->NameQuote( 'id' ) . ' = c.' . $_CB_database->NameQuote( 'id' )
						.	"\n WHERE c." . $_CB_database->NameQuote( 'approved' ) . " = 1"
						.	"\n AND c." . $_CB_database->NameQuote( 'confirmed' ) . " = 1"
						.	"\n AND u." . $_CB_database->NameQuote( 'id' ) . " != " . Application::MyUser()->getUserId()
						.	"\n AND u." . $_CB_database->NameQuote( 'block' ) . " = 0"
						.	"\n AND ( u." . $_CB_database->NameQuote( 'name' ) . " LIKE " . $_CB_database->Quote( $_CB_database->getEscaped( $value, true ) . '%', false )
						.	" OR u." . $_CB_database->NameQuote( 'username' ) . " LIKE " . $_CB_database->Quote( $_CB_database->getEscaped( $value, true ) . '%', false )
						.	" OR u." . $_CB_database->NameQuote( 'email' ) . " LIKE " . $_CB_database->Quote( $_CB_database->getEscaped( $value, true ) . '%', false ) . " )";
		$_CB_database->setQuery( $query, null, 10 );
		$users			=	$_CB_database->loadObjectList( null, '\CB\Database\Table\UserTable', array( $_CB_database ) );

		$options		=	array();

		/** @var UserTable[] $users */
		foreach ( $users as $user ) {
			$options[]	=	array( 'value' => $user->get( 'id', 0, GetterInterface::INT ), 'label' => $user->getFormattedName() );
		}

		echo json_encode( $options );

		header( 'HTTP/1.0 200 OK' );
		exit();
	}
}
