<?php
/**
* Community Builder (TM)
* @version $Id: $
* @package CommunityBuilder
* @copyright (C) 2004-2019 www.joomlapolis.com / Lightning MultiCom SA - and its licensors, all rights reserved
* @license http://www.gnu.org/licenses/old-licenses/gpl-2.0.html GNU/GPL version 2
*/

namespace CB\Plugin\Activity;

use CB\Plugin\Activity\Table\ActivityTable;
use CB\Plugin\Activity\Table\CommentTable;
use CB\Plugin\Activity\Table\NotificationTable;
use CBLib\Application\Application;
use CB\Database\Table\UserTable;
use CBLib\Registry\ParametersStore;
use CBLib\Registry\ParamsInterface;
use CBLib\Registry\Registry;

defined('CBLIB') or die();

class Stream extends ParametersStore implements StreamInterface
{
	/** @var string $type */
	private $type						=	null;
	/** @var string $ns */
	private $ns							=	null;

	/** @var string $id */
	protected $id						=	null;
	/** @var array $assets */
	protected $assets					=	array();
	/** @var UserTable $user */
	protected $user						=	null;
	/** @var array $ini */
	protected $ini						=	array();

	/** @var array $defaults */
	protected $defaults					=	array();

	/** @var bool $clearRowCount */
	protected $clearRowCount			=	false;
	/** @var bool $clearRowSelect */
	protected $clearRowSelect			=	false;

	/**
	 * Constructor for stream object
	 *
	 * @param null|string|array  $assets
	 * @param null|int|UserTable $user
	 * @param null|string        $namespace
	 */
	public function __construct( $assets = null, $user = null, $namespace = null )
	{
		global $_PLUGINS;

		$this->type				=	strtolower( substr( strrchr( get_class( $this ), '\\' ), 1 ) );
		$this->ns				=	( $namespace ? $namespace : $this->type );

		$_PLUGINS->loadPluginGroup( 'user' );

		$_PLUGINS->trigger( 'activity_on' . ucfirst( $this->type ), array( &$assets, &$user, &$this->defaults, &$this ) );

		if ( $user === null ) {
			$user				=	\CBuser::getMyUserDataInstance();
		}

		$this->user( $user );
		$this->assets( $assets );

		$pluginParams			=	CBActivity::getGlobalParams();
		$defaultParams			=	new Registry();

		foreach ( $this->defaults as $param => $default ) {
			$paramName			=	$this->ns . '_' . $param;

			if ( is_int( $default ) ) {
				$defaultValue	=	$pluginParams->getInt( $paramName, $default );
			} elseif ( is_bool( $default ) ) {
				$defaultValue	=	$pluginParams->getBool( $paramName, $default );
			} elseif ( is_array( $default ) ) {
				$defaultValue	=	$pluginParams->getRaw( $paramName, $default );
			} else {
				$defaultValue	=	$pluginParams->getString( $paramName, $default );
			}

			$defaultParams->set( $param, $defaultValue );
		}

		$this->setParent( $defaultParams );
	}

	/**
	 * Reloads the stream from session by id
	 *
	 * @param string $id
	 * @return bool
	 */
	public function load( $id )
	{
		$cache					=	Application::Session()->getRaw( 'cb.' . $this->type . '.' . $id);

		if ( ! is_array( $cache ) ) {
			$inherit			=	Application::Session()->getString( 'cb.' . $this->type . '.' . $id );

			if ( $inherit ) {
				if ( ! $this->id ) {
					$this->id	=	$id;
				}

				// We want to keep our current ID so cache it since the inherit load will override it:
				$initialId		=	$this->id;

				$loadInherit	=	$this->load( $inherit );

				// Restore our initial id:
				$this->id		=	$initialId;

				return $loadInherit;
			}

			return false;
		}

		$session				=	Application::Session()->subTree( 'cb.' . $this->type . '.' . $id );

		if ( $session->count() ) {
			$this->__construct( $session->getRaw( 'assets', array() ),
								$session->getInt( 'user', 0 ),
								$session->getString( 'ns' ) );

			$this->id			=	$id;
			$this->ini			=	$session->asArray();

			parent::load( $session );

			return true;
		}

		return false;
	}

	/**
	 * Resets the stream filters
	 *
	 * @return static
	 */
	public function reset()
	{
		$stream		=	new static( $this->assets(), $this->user(), $this->ns );

		return $stream->parse( $this );
	}

	/**
	 * Parses parameters into the stream
	 *
	 * @param ParamsInterface|array|string $params
	 * @param null|string                  $namespace
	 * @param bool                         $override
	 * @return static
	 */
	public function parse( $params, $namespace = null, $override = true )
	{
		if ( ! $params ) {
			return $this;
		}

		if ( $params instanceof static ) {
			$this->ns					=	$params->ns();
			$this->id					=	$params->id();
			$this->assets				=	$params->assets();
			$this->user					=	$params->user();
			$this->ini					=	$params->ini;

			parent::load( $params->ini );
		} else {
			if ( is_array( $params ) ) {
				$params					=	new Registry( $params );
			}

			foreach ( $this->defaults as $param => $default ) {
				if ( ( ! $override ) && $this->hasInThis( $param ) ) {
					continue;
				}

				$value					=	$params->getRaw( $namespace . $param );

				if ( ( $param === 'filters' ) && is_array( $value ) ) {
					$value				=	array_filter( $value, static function( $v ) {
												return array_filter( $v );
											});
				}

				if ( is_array( $default ) && ( ! $value ) ) {
					$value				=	null;
				}

				if ( ( $value !== null ) && ( $value !== '' ) && ( $value !== '-1' ) ) {
					$paramName			=	$namespace . $param;

					if ( is_int( $default ) ) {
						$paramValue		=	$params->getInt( $paramName );
					} elseif ( is_bool( $default ) ) {
						$paramValue		=	$params->getBool( $paramName );
					} elseif ( is_array( $default ) ) {
						$paramValue		=	$params->getRaw( $paramName );
					} else {
						$paramValue		=	$params->getString( $paramName );
					}

					$this->set( $param, $paramValue );
				}
			}
		}

		return $this;
	}

	/**
	 * Gets the stream namespace
	 *
	 * @return string
	 */
	public function ns()
	{
		return $this->ns;
	}

	/**
	 * Gets the stream id
	 *
	 * @return string
	 */
	public function id()
	{
		return $this->id;
	}

	/**
	 * Gets the primary stream asset
	 *
	 * @return string
	 */
	public function asset()
	{
		$assets		=	$this->assets( false );

		if ( ! $assets ) {
			return null;
		}

		return $assets[0];
	}

	/**
	 * Gets or sets the raw stream assets
	 *
	 * @param null|array|string|bool $assets   null|true: get with wildcards; false: get without wildcards; string: set assets
	 * @param bool                   $filtered true: we're parsing filtered assets so just return the assets but do not set them; false: we're setting the assets
	 * @return array
	 */
	public function assets( $assets = null, $filtered = false )
	{
		global $_CB_framework, $_PLUGINS;

		if ( ( $assets !== null ) && ( $assets !== true ) && ( $assets !== false ) ) {
			$extras							=	array(	'displayed_id'	=>	$_CB_framework->displayedUser(),
														'viewer_id'		=>	Application::MyUser()->getUserId()
													);

			if ( $this->getInt( 'tab', 0 ) ) {
				$extras['tab_id']			=	$this->getInt( 'tab', 0 );
			}

			if ( $this->getInt( 'field', 0 ) ) {
				$extras['field_id']			=	$this->getInt( 'field', 0 );
			}

			if ( ! is_array( $assets ) ) {
				$assets						=	explode( ',', $assets );
			}

			$primaryUser					=	null;

			foreach ( $assets as $k => $asset ) {
				$assetsUser					=	null;

				if ( in_array( $asset, array( 'self', 'self.connections', 'self.connectionsonly', 'self.following', 'self.followingonly', 'self.likes', 'self.likesonly', 'user', 'user.connections', 'user.connectionsonly', 'user.following', 'user.followingonly', 'user.likes', 'user.likesonly' ), true ) ) {
					$assetsUser				=	\CBuser::getMyUserDataInstance();
				}

				if ( in_array( $asset, array( 'user', 'user.connections', 'user.connectionsonly', 'user.following', 'user.followingonly', 'user.likes', 'user.likesonly', 'displayed', 'displayed.connections', 'displayed.connectionsonly', 'displayed.following', 'displayed.followingonly', 'displayed.likes', 'displayed.likesonly' ), true ) ) {
					if ( $_CB_framework->displayedUser() ) {
						$assetsUser			=	\CBuser::getUserDataInstance( $_CB_framework->displayedUser() );
					} elseif ( ! in_array( $asset, array( 'user', 'user.connections', 'user.connectionsonly', 'user.following', 'user.followingonly', 'user.likes', 'user.likesonly' ), true ) ) {
						$assetsUser			=	\CBuser::getUserDataInstance( 0 );
					}
				}

				if ( $assetsUser === null ) {
					$assetsUser				=	$this->user();
				} elseif ( $k === 0 ) {
					$primaryUser			=	$assetsUser;
				}

				if ( ( $asset === null ) || in_array( $asset, array( 'profile', 'following', 'followingonly', 'likes', 'likesonly', 'connections', 'connectionsonly', 'self', 'self.connections', 'self.connectionsonly', 'self.following', 'self.followingonly', 'self.likes', 'self.likesonly', 'user', 'user.connections', 'user.connectionsonly', 'user.following', 'user.followingonly', 'user.likes', 'user.likesonly', 'displayed', 'displayed.connections', 'displayed.connectionsonly', 'displayed.following', 'displayed.followingonly', 'displayed.likes', 'displayed.likesonly' ), true ) ) {
					$newAsset				=	'profile.' . $assetsUser->getInt( 'id', 0 );

					if ( Application::Config()->getBool( 'allowConnections', true ) ) {
						if ( strpos( $asset, 'connectionsonly' ) !== false ) {
							$newAsset		.=	'.connectionsonly';
						} elseif ( strpos( $asset, 'connections' ) !== false ) {
							$newAsset		.=	'.connections';
						}
					}

					if ( strpos( $asset, 'followingonly' ) !== false ) {
						$newAsset			.=	'.followingonly';
					} elseif ( strpos( $asset, 'following' ) !== false ) {
						$newAsset			.=	'.following';
					}

					if ( strpos( $asset, 'likesonly' ) !== false ) {
						$newAsset			.=	'.likesonly';
					} elseif ( strpos( $asset, 'likes' ) !== false ) {
						$newAsset			.=	'.likes';
					}

					$asset					=	$newAsset;
				}

				$_PLUGINS->trigger( 'activity_onStreamAsset', array( &$asset, &$primaryUser, &$assetsUser, $filtered, &$this ) );

				$assets[$k]					=	\CBuser::getInstance( $assetsUser->getInt( 'id', 0 ), false )->replaceUserVars( str_replace( '*', '%', $asset ), true, false, $extras, false );
			}

			if ( ! $filtered ) {
				if ( $primaryUser !== null ) {
					$this->user( $primaryUser );
				}

				$this->assets				=	$assets;
			}
		}

		if ( $filtered ) {
			if ( ! $assets ) {
				return array();
			}

			return $assets;
		}

		if ( ! $this->assets ) {
			return array();
		}

		if ( $assets === false ) {
			$assets							=	$this->assets;

			foreach ( $assets as $k => $asset ) {
				$asset						=	strtolower( trim( preg_replace( '/[^a-zA-Z0-9.%_-]/i', '', $asset ) ) );

				// Replace profile wildcard for storage purposes:
				if ( strpos( $asset, 'profile.' ) !== false ) {
					$asset					=	preg_replace( '/profile\.%(\.|$)/i', 'profile.' . $this->user()->getInt( 'id', 0 ) . '$1', $asset );
				}

				// Replace tab wildcard for storage purposes:
				if ( strpos( $asset, 'tab.' ) !== false ) {
					$asset					=	preg_replace( '/tab\.%(\.|$)/i', 'tab.' . $this->getInt( 'tab', 0 ) . '$1', $asset );
				}

				// Replace field wildcard for storage purposes:
				if ( strpos( $asset, 'field.' ) !== false ) {
					$asset					=	preg_replace( '/field\.%(\.|$)/i', 'field.' . $this->getInt( 'field', 0 ) . '$1', $asset );
				}

				if ( ( $asset === 'all' ) || ( strpos( $asset, 'connections' ) !== false ) || ( strpos( $asset, 'following' ) !== false ) ) {
					$asset					=	'profile.' . $this->user()->getInt( 'id', 0 );
				}

				$assets[$k]					=	$asset;
			}

			return array_unique( $assets );
		}

		return $this->assets;
	}

	/**
	 * Prepares assets array for query usage
	 *
	 * @param array $queryAssets
	 * @return array
	 */
	protected function queryAssets( $queryAssets = array() )
	{
		global $_CB_database;

		$assets									=	array();
		$wildcards								=	array();
		$users									=	array();
		$exists									=	array();

		foreach ( ( $queryAssets ? $queryAssets : $this->assets() ) as $asset ) {
			if ( $asset === 'all' ) {
				continue;
			}

			if ( strpos( $asset, 'connections' ) !== false ) {
				if ( preg_match( '/^profile\.(\d+)\.connections/', $asset, $matches ) ) {
					$profileId					=	(int) $matches[1];
				} else {
					$profileId					=	$this->user()->getInt( 'id', 0 );
				}

				if ( $profileId ) {
					if ( strpos( $asset, 'connectionsonly' ) === false ) {
						$users[]				=	$profileId;
						$assets[]				=	'profile.' . $profileId;
					}

					$connections				=	CBActivity::getConnections( $profileId );

					if ( ! $connections ) {
						continue;
					}

					if ( count( $connections ) > 50 ) {
						// There's too many assets or users which could cause a massive IN statement so lets use a subquery instead:
						$exists[]				=	"SELECT 1"
												.	" FROM " . $_CB_database->NameQuote( '#__comprofiler_members' ) . " AS m"
												.	" LEFT JOIN " . $_CB_database->NameQuote( '#__comprofiler' ) . " AS mcb"
												.	" ON mcb." . $_CB_database->NameQuote( 'id' ) . " = m." . $_CB_database->NameQuote( 'memberid' )
												.	" LEFT JOIN " . $_CB_database->NameQuote( '#__users' ) . " AS mj"
												.	" ON mj." . $_CB_database->NameQuote( 'id' ) . " = m." . $_CB_database->NameQuote( 'memberid' )
												.	" WHERE mcb." . $_CB_database->NameQuote( 'approved' ) . " = 1"
												.	" AND mcb." . $_CB_database->NameQuote( 'confirmed' ) . " = 1"
												.	" AND mj." . $_CB_database->NameQuote( 'block' ) . " = 0"
												.	" AND m." . $_CB_database->NameQuote( 'referenceid' ) . " = ". $profileId
												.	" AND ( m." . $_CB_database->NameQuote( 'memberid' ) . " = a." . $_CB_database->NameQuote( 'user_id' )
												.	" OR CONCAT( 'profile.', m." . $_CB_database->NameQuote( 'memberid' ) . " ) = a." . $_CB_database->NameQuote( 'asset' ) . " )"
												.	" AND m." . $_CB_database->NameQuote( 'accepted' ) . " = 1";
					} else {
						foreach( $connections as $connection ) {
							$users[]			=	$connection;
							$assets[]			=	'profile.' . $connection;
						}
					}
				}
			} elseif ( strpos( $asset, 'following' ) !== false ) {
				if ( preg_match( '/^profile\.(\d+)\.following/', $asset, $matches ) ) {
					$profileId					=	(int) $matches[1];
				} else {
					$profileId					=	$this->user()->getInt( 'id', 0 );
				}

				if ( $profileId ) {
					if ( strpos( $asset, 'followingonly' ) === false ) {
						$users[]				=	$profileId;
						$assets[]				=	'profile.' . $profileId;
					}

					$following					=	CBActivity::getFollowing( $profileId );

					if ( ! $following ) {
						continue;
					}

					if ( count( $following ) > 50 ) {
						// There's too many assets or users which could cause a massive IN statement so lets use a subquery instead:
						foreach( $following as $follow ) {
							if ( ( strpos( $follow, '%' ) !== false ) || ( strpos( $follow, '_' ) !== false ) ) {
								$wildcards[]	=	$follow;
							}
						}

						$exists[]				=	"SELECT 1"
												.	" FROM " . $_CB_database->NameQuote( '#__comprofiler_plugin_activity_following' ) . " AS f"
												.	" WHERE f." . $_CB_database->NameQuote( 'user_id' ) . " = ". $profileId
												.	" AND f." . $_CB_database->NameQuote( 'asset' ) . " IN ( CONCAT( 'profile.', a." . $_CB_database->NameQuote( 'user_id' ) . " ), a." . $_CB_database->NameQuote( 'asset' ) . " )";
					} else {
						foreach( $following as $follow ) {
							if ( preg_match( '/^profile\.(\d+)/', $follow, $matches ) ) {
								$users[]		=	(int) $matches[1];
							}

							if ( ( strpos( $follow, '%' ) !== false ) || ( strpos( $follow, '_' ) !== false ) ) {
								$wildcards[]	=	$follow;
							} else {
								$assets[]		=	$follow;
							}
						}
					}
				}
			} elseif ( strpos( $asset, 'likes' ) !== false ) {
				if ( preg_match( '/^profile\.(\d+)\.likes/', $asset, $matches ) ) {
					$profileId					=	(int) $matches[1];
				} else {
					$profileId					=	$this->user()->getInt( 'id', 0 );
				}

				if ( $profileId ) {
					if ( strpos( $asset, 'likesonly' ) === false ) {
						$users[]				=	$profileId;
						$assets[]				=	'profile.' . $profileId;
					}

					$likes						=	CBActivity::getLikes( $profileId );

					if ( ! $likes ) {
						continue;
					}

					if ( count( $likes ) > 50 ) {
						foreach( $likes as $like ) {
							if ( ( strpos( $like, '%' ) !== false ) || ( strpos( $like, '_' ) !== false ) ) {
								$wildcards[]	=	$like;
							}
						}

						$exists[]				=	"SELECT 1"
												.	" FROM " . $_CB_database->NameQuote( '#__comprofiler_plugin_activity_likes' ) . " AS l"
												.	" WHERE l." . $_CB_database->NameQuote( 'user_id' ) . " = ". $profileId
												.	" AND l." . $_CB_database->NameQuote( 'asset' ) . " IN ( CONCAT( 'profile.', a." . $_CB_database->NameQuote( 'user_id' ) . " ), a." . $_CB_database->NameQuote( 'asset' ) . " )";
					} else {
						foreach( $likes as $like ) {
							if ( preg_match( '/^profile\.(\d+)/', $like, $matches ) ) {
								$users[]		=	(int) $matches[1];
							}

							if ( ( strpos( $like, '%' ) !== false ) || ( strpos( $like, '_' ) !== false ) ) {
								$wildcards[]	=	$like;
							} else {
								$assets[]		=	$like;
							}
						}
					}
				}
			} elseif ( ( strpos( $asset, '%' ) !== false ) || ( strpos( $asset, '_' ) !== false ) ) {
				$wildcards[]					=	$asset;
			} else {
				if ( preg_match( '/^profile\.(\d+)$/', $asset, $matches ) ) {
					$users[]					=	(int) $matches[1];
				}

				$assets[]						=	$asset;
			}
		}

		$assets									=	array_unique( $assets );
		$wildcards								=	array_unique( $wildcards );
		$users									=	array_unique( $users );
		$exists									=	array_unique( $exists );

		return array( 'assets' => $assets, 'wildcards' => $wildcards, 'users' => $users, 'exists' => $exists );
	}

	/**
	 * Gets or sets the stream target user (owner)
	 *
	 * @param null|UserTable|int $user
	 * @return UserTable|int|null
	 */
	public function user( $user = null )
	{
		if ( $user !== null ) {
			if ( is_numeric( $user ) ) {
				$user		=	\CBuser::getUserDataInstance( (int) $user );
			}

			$this->user		=	$user;
		}

		return $this->user;
	}

	/**
	 * Clears the data cache
	 *
	 * @return static
	 */
	public function clear()
	{
		$this->clearRowCount		=	true;
		$this->clearRowSelect		=	true;

		return $this;
	}

	/**
	 * Returns a parser object for parsing stream content
	 *
	 * @param string                                       $string
	 * @param ActivityTable|CommentTable|NotificationTable $source
	 * @return Parser
	 */
	public function parser( $string = '', $source = null )
	{
		return new Parser( $string, $source, $this );
	}

	/**
	 * Returns an array of the stream variables
	 *
	 * @return array
	 */
	public function asArray()
	{
		$params					=	parent::asArray();

		if ( isset( $params['paging_total'] ) ) {
			unset( $params['paging_total'] );
		}

		if ( isset( $params['paging_limitstart'] ) ) {
			unset( $params['paging_limitstart'] );
		}

		if ( isset( $params['filter'] ) ) {
			unset( $params['filter'] );
		}

		if ( isset( $params['search'] ) ) {
			unset( $params['search'] );
		}

		if ( isset( $params['query'] ) ) {
			unset( $params['query'] );
		}

		if ( isset( $params['query_count'] ) ) {
			unset( $params['query_count'] );
		}

		$params['ns']			=	$this->ns();
		$params['assets']		=	$this->assets();
		$params['user']			=	$this->user()->getInt( 'id', 0 );

		return $params;
	}

	/**
	 * Caches the stream into session; this is normally only done on creation or parse to preserve parameters between loads
	 * It is not advised to call this manually unless stream parameters have changed after creation and desired result is for them to persist
	 *
	 * @return static
	 */
	public function cache()
	{
		$newId				=	md5( $this->asJson() );

		if ( $this->id() !== $newId ) {
			$session		=	Application::Session();
			$stream			=	$session->subTree( 'cb.' . $this->type );

			if ( $this->id() ) {
				$stream->set( $this->id(), $newId );
			}

			$this->id		=	$newId;
			$this->ini		=	$this->asArray();

			$stream->set( $this->id(), $this->ini );

			$session->set( 'cb.' . $this->type, $stream->asArray() );
		}

		return $this;
	}

	/**
	 * Outputs stream HTML
	 *
	 * @param null|string $view
	 * @param int         $id
	 * @param array       $params
	 * @return string
	 */
	protected function display( $view = null, $id = 0, $params = array() )
	{
		global $_CB_framework, $_PLUGINS;

		static $plugin		=	null;

		if ( ! $plugin ) {
			$plugin			=	$_PLUGINS->getLoadedPlugin( 'user', 'cbactivity' );
		}

		if ( ! $plugin ) {
			return null;
		}

		if ( ! class_exists( 'CBplug_cbactivity' ) ) {
			$component		=	$_CB_framework->getCfg( 'absolute_path' ) . '/components/com_comprofiler/plugin/user/plug_cbactivity/component.cbactivity.php';

			if ( file_exists( $component ) ) {
				include_once( $component );
			}
		}

		$stream				=	$this;

		if ( $params ) {
			// We're overriding the params for this instance only so lets make a clone first so $this is preserved:
			$stream			=	clone $stream;
			$stream->id		=	null;

			$stream->parse( $params );
		}

		$stream->cache();

		ob_start();
		$pluginArguements	=	array( &$stream, $view, $id );

		$_PLUGINS->call( $plugin->id, 'getStream', 'CBplug_cbactivity', $pluginArguements );
		return ob_get_clean();
	}
}