<?php
/**
* Community Builder (TM)
* @version $Id: $
* @package CommunityBuilder
* @copyright (C) 2004-2019 www.joomlapolis.com / Lightning MultiCom SA - and its licensors, all rights reserved
* @license http://www.gnu.org/licenses/old-licenses/gpl-2.0.html GNU/GPL version 2
*/

namespace CB\Plugin\Activity;

use CB\Database\Table\UserTable;
use CBLib\Application\Application;
use CBLib\Input\Get;
use CBLib\Registry\GetterInterface;
use CBLib\Registry\Registry;
use CB\Plugin\Activity\Table\ActivityTable;
use CB\Plugin\Activity\Table\CommentTable;
use CB\Plugin\Activity\Table\NotificationTable;

defined('CBLIB') or die();

class Parser
{
	/** @var array  */
	protected $regexp		=	array(	'hashtag'	=>	'/^#(\w+)$/iu',
										'profile'	=>	'/^@(\w+)$/iu',
										'reaction'	=>	'/\(giphy:([a-zA-z0-9]+)\)/iu',
										'link'		=>	'%^(?:(?:(?:(?:https?|ftp):)?\/\/)|www\.)(?:\S+(?::\S*)?@)?(?:(?!(?:10|127)(?:\.\d{1,3}){3})(?!(?:169\.254|192\.168)(?:\.\d{1,3}){2})(?!172\.(?:1[6-9]|2\d|3[0-1])(?:\.\d{1,3}){2})(?:[1-9]\d?|1\d\d|2[01]\d|22[0-3])(?:\.(?:1?\d{1,2}|2[0-4]\d|25[0-5])){2}(?:\.(?:[1-9]\d?|1\d\d|2[0-4]\d|25[0-4]))|(?:(?:[a-z0-9\x{00a1}-\x{ffff}][a-z0-9\x{00a1}-\x{ffff}_-]{0,62})?[a-z0-9\x{00a1}-\x{ffff}]\.)+(?:[a-z\x{00a1}-\x{ffff}]{2,}\.?))(?::\d{2,5})?(?:[/?#]\S*)?$%iuS',
										'email'		=>	'/^[a-z0-9!#$%&\'*+\\\\\/=?^_`{|}~-]+(?:\.[a-z0-9!#$%&\'*+\\\\\/=?^_`{|}~-]+)*@(?:[a-z0-9](?:[a-z0-9-]*[a-z0-9])?\.)+[a-z0-9](?:[a-z0-9-]*[a-z0-9])?$/iu'
									);
	/** @var null|ActivityTable|CommentTable|NotificationTable  */
	protected $source		=	null;
	/** @var null|Activity|Comments|Notifications  */
	protected $stream		=	null;
	/** @var string  */
	protected $string		=	null;
	/** @var string  */
	protected $parsed		=	null;
	/** @var array  */
	protected $words		=	array();
	/** @var Registry[]  */
	protected $attachments	=	array();

	/**
	 * Constructor
	 *
	 * @param string                                            $string
	 * @param null|ActivityTable|CommentTable|NotificationTable $source
	 * @param null|Activity|Comments|Notifications              $stream
	 */
	public function __construct( $string, $source = null, $stream = null )
	{
		global $_PLUGINS;

		$_PLUGINS->loadPluginGroup( 'user' );

		$this->source	=	$source;
		$this->stream	=	$stream;
		$this->string	=	$string;
		$this->parsed	=	$string;
		$this->words	=	preg_split( '/\s/', $string );
	}

	/**
	 * Returns parser stream
	 *
	 * @return null|Activity|Comments
	 */
	public function stream()
	{
		return $this->stream;
	}

	/**
	 * Returns the original unmodified string
	 *
	 * @return string
	 */
	public function original()
	{
		return $this->string;
	}

	/**
	 * Replaces user substitutions
	 *
	 * @return string
	 */
	public function substitutions()
	{
		if ( $this->source && ( ! $this->source->getBool( 'system', false ) ) ) {
			// If there's a source object and it's not a system object then don't parse substitutions:
			return $this->parsed;
		}

		if ( $this->stream ) {
			$cbUser		=	\CBuser::getInstance( $this->stream->user()->getInt( 'id', 0 ), false );
		} else {
			$cbUser		=	\CBuser::getMyInstance();
		}

		$this->parsed	=	$cbUser->replaceUserVars( $this->parsed, true, false, null, false );

		return $this->parsed;
	}

	/**
	 * Replaces (giphy:ID) with gyphy icons
	 *
	 * @return string
	 */
	public function reactions()
	{
		if ( ! CBActivity::getGlobalParams()->getString( 'reactions_giphy' ) ) {
			return $this->parsed;
		}

		$method				=	CBActivity::getGlobalParams()->getString( 'reactions_method', 'gif' );

		$this->parsed		=	preg_replace_callback( $this->regexp['reaction'], static function( array $matches ) use ( $method ) {
									switch ( $method ) {
										case 'html5':
											$gif	=	'<video src="https://media.giphy.com/media/' . htmlspecialchars( $matches[1] ) . '/giphy.mp4" type="video/mp4" class="d-block mw-100 streamIconReaction" controls autoplay loop></video>';
											break;
										case 'iframe':
											$gif	=	'<div class="position-relative w-100 mh-100 streamIconReactionEmbed">'
													.		'<iframe width="100%" height="100%" style="width: 100%; height: 100%;" frameborder="0" allowfullscreen class="position-absolute streamIconReaction" src="https://giphy.com/embed/' . htmlspecialchars( $matches[1] ) . '" loading="lazy"></iframe>'
													.	'</div>';
											break;
										case 'gif':
										default:
											$gif	=	'<img src="https://media.giphy.com/media/' . htmlspecialchars( $matches[1] ) . '/giphy.gif" loading="lazy" class="d-block img-fluid streamIconReaction" />';
											break;
									}

									return $gif;
								}, $this->parsed );

		return $this->parsed;
	}

	/**
	 * Replaces :EMOTE: with emote icons
	 *
	 * @return string
	 */
	public function emotes()
	{
		if ( $this->source ) {
			// If there's a source object we need to try and restrict access based off the object owner:
			$this->parsed	=	strtr( $this->parsed, CBActivity::loadEmoteOptions( true, false, $this->source->getInt( 'user_id', 0 ) ) );
		} else {
			$this->parsed	=	strtr( $this->parsed, CBActivity::loadEmoteOptions( true ) );
		}

		return $this->parsed;
	}

	/**
	 * Replaces #HASHTAG with stream filter urls
	 *
	 * @return string
	 */
	public function hashtags()
	{
		global $_CB_framework;

		foreach ( $this->words as $word ) {
			if ( preg_match( $this->regexp['hashtag'], $word, $match ) ) {
				$this->parsed	=	str_replace( $word, '<a href="' . $_CB_framework->pluginClassUrl( 'cbactivity', true, array( 'action' => 'recentactivity', 'hashtag' => $match[1] ) ) . '">' . htmlspecialchars( $match[0] ) . '</a>', $this->parsed );
			}
		}

		return $this->parsed;
	}

	/**
	 * Replaces @MENTION with profile urls
	 *
	 * @return string
	 */
	public function profiles()
	{
		global $_CB_database, $_CB_framework;

		/** @var UserTable[] $users */
		static $users						=	array();

		foreach ( $this->words as $k => $word ) {
			if ( preg_match( $this->regexp['profile'], $word, $match ) ) {
				$cleanWord					=	Get::clean( $match[1], GetterInterface::STRING );

				if ( ! isset( $users[$cleanWord] ) ) {
					$user					=	new UserTable();

					if ( is_numeric( $match[1] ) ) {
						$user				=	\CBuser::getUserDataInstance( (int) $match[1] );
					}

					if ( ! $user->getInt( 'id', 0 ) ) {
						$wordNext2			=	( isset( $this->words[$k+1] ) && ( ! preg_match( $this->regexp['profile'], $this->words[$k+1] ) ) ? $cleanWord . ' ' . Get::clean( $this->words[$k+1], GetterInterface::STRING ) : null );
						$wordNext3			=	( $wordNext2 && isset( $this->words[$k+2] ) && ( ! preg_match( $this->regexp['profile'], $this->words[$k+2] ) ) ? $wordNext2 . ' ' . Get::clean( $this->words[$k+2], GetterInterface::STRING ) : null );
						$wordNext4			=	( $wordNext3 && isset( $this->words[$k+3] ) && ( ! preg_match( $this->regexp['profile'], $this->words[$k+3] ) ) ? $wordNext3 . ' ' . Get::clean( $this->words[$k+3], GetterInterface::STRING ) : null );
						$wordNext5			=	( $wordNext4 && isset( $this->words[$k+4] ) && ( ! preg_match( $this->regexp['profile'], $this->words[$k+4] ) ) ? $wordNext4 . ' ' . Get::clean( $this->words[$k+4], GetterInterface::STRING ) : null );
						$wordNext6			=	( $wordNext5 && isset( $this->words[$k+5] ) && ( ! preg_match( $this->regexp['profile'], $this->words[$k+5] ) ) ? $wordNext5 . ' ' . Get::clean( $this->words[$k+5], GetterInterface::STRING ) : null );

						$query				=	'SELECT c.*, u.*'
											.	"\n FROM " . $_CB_database->NameQuote( '#__users' ) . " AS u"
											.	"\n LEFT JOIN " . $_CB_database->NameQuote( '#__comprofiler' ) . " AS c"
											.	' ON c.' . $_CB_database->NameQuote( 'id' ) . ' = u.' . $_CB_database->NameQuote( 'id' )
											.	"\n WHERE ( u." . $_CB_database->NameQuote( 'username' ) . ' = ' . $_CB_database->Quote( $cleanWord )		// Match username exactly
											.	' OR u.' . $_CB_database->NameQuote( 'name' ) . ' = ' . $_CB_database->Quote( $cleanWord );					// Match name exactly

						if ( $wordNext2 ) { // 2 Words
							$query			.=	' OR u.' . $_CB_database->NameQuote( 'username' ) . ' = ' . $_CB_database->Quote( $wordNext2 )				// Match username +1 word exactly
											.	' OR u.' . $_CB_database->NameQuote( 'name' ) . ' = ' . $_CB_database->Quote( $wordNext2 );					// Match name +1 word exactly
						}

						if ( $wordNext3 ) { // 3 Words
							$query			.=	' OR u.' . $_CB_database->NameQuote( 'username' ) . ' = ' . $_CB_database->Quote( $wordNext3 )				// Match username +2 words exactly
											.	' OR u.' . $_CB_database->NameQuote( 'name' ) . ' = ' . $_CB_database->Quote( $wordNext3 );					// Match name +2 words exactly
						}

						if ( $wordNext4 ) { // 4 Words
							$query			.=	' OR u.' . $_CB_database->NameQuote( 'username' ) . ' = ' . $_CB_database->Quote( $wordNext4 )				// Match username +3 words exactly
											.	' OR u.' . $_CB_database->NameQuote( 'name' ) . ' = ' . $_CB_database->Quote( $wordNext4 );					// Match name +3 words exactly
						}

						if ( $wordNext5 ) { // 5 Words
							$query			.=	' OR u.' . $_CB_database->NameQuote( 'username' ) . ' = ' . $_CB_database->Quote( $wordNext5 )				// Match username +4 words exactly
											.	' OR u.' . $_CB_database->NameQuote( 'name' ) . ' = ' . $_CB_database->Quote( $wordNext5 );					// Match name +4 words exactly
						}

						if ( $wordNext6 ) { // 6 Words
							$query			.=	' OR u.' . $_CB_database->NameQuote( 'username' ) . ' = ' . $_CB_database->Quote( $wordNext6 )				// Match username +5 words exactly
											.	' OR u.' . $_CB_database->NameQuote( 'name' ) . ' = ' . $_CB_database->Quote( $wordNext6 );					// Match name +5 words exactly
						}

						$query				.=	' )'
											.	"\n ORDER BY u." . $_CB_database->NameQuote( 'username' ) . ", u." . $_CB_database->NameQuote( 'name' );
						$_CB_database->setQuery( $query );
						$_CB_database->loadObject( $user );
					}

					$users[$cleanWord]		=	$user;
				}

				$found						=	$users[$cleanWord];

				if ( $found->getInt( 'id', 0 ) ) {
					$this->parsed			=	preg_replace( '/@' . $found->getInt( 'id', 0 ) . '\b|@' . preg_quote( $found->getString( 'name' ), '/' ) . '\b|@' . preg_quote( $found->getString( 'username' ), '/' ) . '\b|' . preg_quote( $word, '/' ) . '\b/i', '<a href="' . $_CB_framework->userProfileUrl( $found->getInt( 'id', 0 ) ) . '">' . htmlspecialchars( $found->getFormattedName() ) . '</a>', $this->parsed );
				}
			}
		}

		return $this->parsed;
	}

	/**
	 * Replaces URLs with clickable html URLs
	 *
	 * @return string
	 */
	public function links()
	{
		foreach ( $this->words as $word ) {
			if ( preg_match( $this->regexp['link'], $word, $match ) ) {
				$link				=	$match[0];

				if ( strpos( $link, 'www' ) === 0 ) {
					$link			=	'http://' . $link;
				}

				$newWindow			=	( ! Application::Router()->isInternal( $link ) );

				if ( ! $newWindow ) {
					$extension		=	strtolower( pathinfo( $link, PATHINFO_EXTENSION ) );

					if ( $extension && ( ! preg_match( '/^(php|asp|html)/', $extension ) ) ) {
						$newWindow	=	true;
					}
				}

				$this->parsed		=	str_replace( $word, '<a href="' . htmlspecialchars( $link ) . '"' . ( $newWindow ? ' target="_blank" rel="nofollow noopener"' : null ) . '>' . htmlspecialchars( $match[0] ) . '</a>', $this->parsed );
			} elseif ( preg_match( $this->regexp['email'], $word, $match ) ) {
				$this->parsed		=	str_replace( $word, '<a href="mailto:' . htmlspecialchars( $match[0] ) . '" rel="nofollow noopener" target="_blank">' . htmlspecialchars( $match[0] ) . '</a>', $this->parsed );
			}
		}

		return $this->parsed;
	}

	/**
	 * Parses for URLs
	 *
	 * @return array
	 */
	public function urls()
	{
		$urls				=	array();

		foreach ( $this->words as $word ) {
			if ( preg_match( $this->regexp['link'], $word, $match ) ) {
				$link		=	$match[0];

				if ( strpos( $link, 'www' ) === 0 ) {
					$link	=	'http://' . $link;
				}

				$urls[]		=	$link;
			}
		}

		return $urls;
	}

	/**
	 * Replaces bbcode with html
	 *
	 * @return string
	 */
	public function bbcode()
	{
		$that				=	$this;

		// Left Align:
		$this->parsed		=	preg_replace( '%\[left](.*)\[/left]%siU', '<div class="text-left">$1</div>', $this->parsed );

		// Center Align:
		$this->parsed		=	preg_replace( '%\[center](.*)\[/center]%siU', '<div class="text-center">$1</div>', $this->parsed );

		// Right Align:
		$this->parsed		=	preg_replace( '%\[right](.*)\[/right]%siU', '<div class="text-right">$1</div>', $this->parsed );

		// Bold:
		$this->parsed		=	preg_replace( '%\[b](.*)\[/b]%siU', '<strong>$1</strong>', $this->parsed );
		$this->parsed		=	preg_replace( '%\[bold](.*)\[/bold]%siU', '<strong>$1</strong>', $this->parsed );
		$this->parsed		=	preg_replace( '%\[strong](.*)\[/strong]%siU', '<strong>$1</strong>', $this->parsed );

		// Small:
		$this->parsed		=	preg_replace( '%\[small](.*)\[/small]%siU', '<small>$1</small>', $this->parsed );

		// Italic:
		$this->parsed		=	preg_replace( '%\[i](.*)\[/i]%siU', '<em>$1</em>', $this->parsed );

		// Underline:
		$this->parsed		=	preg_replace( '%\[u](.*)\[/u]%siU', '<u>$1</u>', $this->parsed );
		$this->parsed		=	preg_replace( '%\[ins](.*)\[/ins]%siU', '<ins>$1</ins>', $this->parsed );

		// Subscript:
		$this->parsed		=	preg_replace( '%\[sub](.*)\[/sub]%siU', '<sub>$1</sub>', $this->parsed );

		// Superscript:
		$this->parsed		=	preg_replace( '%\[sup](.*)\[/sup]%siU', '<sup>$1</sup>', $this->parsed );

		// Mark:
		$this->parsed		=	preg_replace( '%\[mark](.*)\[/mark]%siU', '<mark>$1</mark>', $this->parsed );

		// Strikethrough:
		$this->parsed		=	preg_replace( '%\[s](.*)\[/s]%siU', '<s>$1</s>', $this->parsed );
		$this->parsed		=	preg_replace( '%\[del](.*)\[/del]%siU', '<del>$1</del>', $this->parsed );
		$this->parsed		=	preg_replace( '%\[strike](.*)\[/strike]%siU', '<s>$1</s>', $this->parsed );

		// Font size:
		$this->parsed		=	preg_replace( '#\[size="?(8[0-9]|9[0-9]|1[0-9]{2}|200)(?:px|em|pt|%)?"?](.*)\[/size]#siU', '<span style="font-size: $1%;">$2</span>', $this->parsed );
		$this->parsed		=	preg_replace( '#\[style size="?(8[0-9]|9[0-9]|1[0-9]{2}|200)(?:px|em|pt|%)?"?](.*)\[/style]#siU', '<span style="font-size: $1%;">$2</span>', $this->parsed );

		// Font color:
		$this->parsed		=	preg_replace( '%\[color="?([a-zA-Z]+|#[a-zA-Z0-9]+)"?](.*)\[/color]%siU', '<span style="color: $1;">$2</span>', $this->parsed );
		$this->parsed		=	preg_replace( '%\[style color="?([a-zA-Z]+|#[a-zA-Z0-9]+)"?](.*)\[/style]%siU', '<span style="color: $1;">$2</span>', $this->parsed );

		// Code:
		$this->parsed		=	preg_replace( '%\[code](.*)\[/code]%siU', '<code>$1</code>', $this->parsed );

		// Quote:
		$this->parsed		=	preg_replace( '%\[quote](.*)\[/quote]%siU', '<blockquote>$1</blockquote>', $this->parsed );

		// URLs
		$this->parsed		=	preg_replace_callback( '%\[url(?:="?(.*)"?)?](.*)\[/url]%iU', static function( array $matches ) use ( $that ) {
									$url				=	( isset( $matches[1] ) ? $matches[1] : null );
									$hypertext			=	( isset( $matches[2] ) ? $matches[2] : null );

									if ( ! $url ) {
										$url			=	$hypertext;
									}

									if ( ! preg_match( $that->regexp['link'], $url, $match ) ) {
										return $matches[0];
									}

									if ( strpos( $url, 'www' ) === 0 ) {
										$url			=	'http://' . $url;
									}

									$newWindow			=	( ! Application::Router()->isInternal( $url ) );

									if ( ! $newWindow ) {
										$extension		=	strtolower( pathinfo( $url, PATHINFO_EXTENSION ) );

										if ( $extension && ( ! preg_match( '/^(php|asp|html)/', $extension ) ) ) {
											$newWindow	=	true;
										}
									}

									return '<a href="' . htmlspecialchars( $url ) . '"' . ( $newWindow ? ' target="_blank" rel="nofollow noopener"' : null ) . '>' . htmlspecialchars( $hypertext ) . '</a>';
								}, $this->parsed );

		// Emails
		$this->parsed		=	preg_replace_callback( '%\[email](.*)\[/email]%iU', static function( array $matches ) use ( $that ) {
									$email			=	( isset( $matches[1] ) ? $matches[1] : null );

									if ( ! preg_match( $that->regexp['email'], $email, $match ) ) {
										return $matches[0];
									}

									return '<a href="mailto:' . htmlspecialchars( $email ) . '">' . htmlspecialchars( $email ) . '</a>';
								}, $this->parsed );

		// Images
		$this->parsed		=	preg_replace_callback( '%\[img](.*)\[/img]%iU', static function( array $matches ) use ( $that ) {
									$image			=	( isset( $matches[1] ) ? $matches[1] : null );

									if ( ! preg_match( $that->regexp['link'], $image, $match ) ) {
										return $matches[0];
									}

									if ( strpos( $image, 'www' ) === 0 ) {
										$image		=	'http://' . $image;
									}

									$extension		=	strtolower( preg_replace( '/[^-a-zA-Z0-9_]/', '', pathinfo( $image, PATHINFO_EXTENSION ) ) );

									if ( ! in_array( $extension, array( 'jpg', 'jpeg', 'gif', 'png', 'svg', 'ico', 'bmp' ), true ) ) {
										return $matches[0];
									}

									return '<img src="' . htmlspecialchars( $image ) . '" loading="lazy" class="img-fluid" />';
								}, $this->parsed );

		// Lists
		$this->parsed		=	preg_replace_callback( '%(?:\[list](.*)\[/list])|(?:\[ol](.*)\[ol])|(?:\[ul](.*)\[ul])%siU', static function( array $matches ) use ( $that ) {
									$listType		=	'ul';
									$list			=	( isset( $matches[1] ) ? $matches[1] : ( isset( $matches[2] ) ? $matches[2] : null ) );

									if ( ! $list ) {
										$listType	=	'ol';
										$list		=	( isset( $matches[3] ) ? $matches[3] : null );
									}

									return "<$listType>" . preg_replace( '%\[li](.*)\[/li]%siU', '<li>$1</li>', $list ) . "</$listType>";
								}, $this->parsed );

		// Tables
		$this->parsed		=	preg_replace_callback( '%\[table](.*)\[/table]%siU', static function( array $matches ) use ( $that ) {
									$table			=	( isset( $matches[1] ) ? $matches[1] : null );

									// Table Header:
									$table			=	preg_replace( '%\[thead](.*)\[/thead]%siU', '<thead>$1</thead>', $table );

									// Table Body:
									$table			=	preg_replace( '%\[tbody](.*)\[/tbody]%siU', '<tbody>$1</tbody>', $table );

									// Table Footer:
									$table			=	preg_replace( '%\[tfoot](.*)\[/tfoot]%siU', '<tfoot>$1</tfoot>', $table );

									// Table Row:
									$table			=	preg_replace( '%\[tr](.*)\[/tr]%siU', '<tr>$1</tr>', $table );

									// Table Header Column:
									$table			=	preg_replace( '%\[th](.*)\[/th]%siU', '<th>$1</th>', $table );

									// Table Column:
									$table			=	preg_replace( '%\[td](.*)\[/td]%siU', '<td>$1</td>', $table );

									return '<div class="table-responsive"><table class="table table-bordered">' . $table . '</table></div>';
								}, $this->parsed );

		return $this->parsed;
	}

	/**
	 * Sends string through content.prepare
	 *
	 * @return string
	 */
	public function prepare()
	{
		if ( $this->source && ( ! $this->source->getBool( 'system', false ) ) ) {
			// If there's a source object and it's not a system object then don't parse content plugins:
			return $this->parsed;
		}

		$this->parsed	=	Application::Cms()->prepareHtmlContentPlugins( $this->parsed, 'activity.parser' );

		return $this->parsed;
	}

	/**
	 * Replaces linebreaks with html breaks
	 *
	 * @return string
	 */
	public function linebreaks()
	{
		$this->parsed	=	str_replace( array( "\r\n", "\r", "\n" ), '<br />', $this->parsed );

		return $this->parsed;
	}

	/**
	 * Replaces duplicate space, tabs, and linebreaks
	 *
	 * @return string
	 */
	public function duplicates()
	{
		// Remove duplicate spaces:
		$this->parsed		=	preg_replace( '/ {2,}/i', ' ', $this->parsed );

		// Remove duplicate tabs:
		$this->parsed		=	preg_replace( '/\t{2,}/i', "\t", $this->parsed );

		// Remove duplicate linebreaks:
		$this->parsed		=	preg_replace( '/((?:\r\n|\r|\n){2})(?:\r\n|\r|\n)*/i', '$1', $this->parsed );

		return $this->parsed;
	}

	/**
	 * Parsers urls for their media information
	 *
	 * @return Registry[]
	 */
	public function attachments()
	{
		foreach ( $this->words as $word ) {
			$this->attachment( $word );
		}

		return $this->attachments;
	}

	/**
	 * Parsers a url for media information
	 *
	 * @param string $url
	 * @return null|Registry
	 */
	public function attachment( $url )
	{
		global $_CB_framework;

		if ( ( ! $url ) || ( ! preg_match( $this->regexp['link'], $url ) ) ) {
			return null;
		}

		static $cache										=	array();

		if ( isset( $cache[$url] ) ) {
			return $cache[$url];
		}

		$cachePath											=	$_CB_framework->getCfg( 'absolute_path' ) . '/cache/activity_links';
		$cacheFile											=	$cachePath . '/' . md5( $url ) . '.json';
		$attachment											=	null;

		if ( file_exists( $cacheFile ) ) {
			if ( ( ( Application::Date( 'now', 'UTC' )->getTimestamp() - filemtime( $cacheFile ) ) / 3600 ) < 24 ) {
				$request									=	true;
			} else {
				$attachment									=	trim( file_get_contents( $cacheFile ) );
				$request									=	false;
			}
		} else {
			$request										=	true;
		}

		if ( $request ) {
			$trustedDomains									=	array( 'youtube', 'youtu', 'vimeo' );

			$attachment										=	array(	'type'			=>	'url',
																		'title'			=>	array(),
																		'description'	=>	array(),
																		'media'			=>	array( 'video' => array(), 'audio' => array(), 'image' => array(), 'file' => array() ),
																		'url'			=>	$url,
																		'thumbnail'		=>	null,
																		'internal'		=>	Application::Router()->isInternal( $url ),
																		'date'			=>	Application::Date( 'now', 'UTC' )->getTimestamp()
																	);

			$urlType										=	'url';
			$urlDomain										=	preg_replace( '/^(?:(?:\w+\.)*)?(\w+)\..+$/', '\1', parse_url( $url, PHP_URL_HOST ) );
			$urlExt											=	strtolower( pathinfo( $url, PATHINFO_EXTENSION ) );

			if ( $urlExt === 'exe' ) {
				// For security reject all exe attachments:
				$cache[$url]								=	null;

				return null;
			}

			$urlFilename									=	pathinfo( $url, PATHINFO_FILENAME ) . '.' . $urlExt;
			$urlFileSize									=	0;
			$urlYouTube										=	( in_array( $urlDomain, array( 'youtube', 'youtu' ), true ) && ( ! $urlExt ) && preg_match( '%(?:(?:watch\?v=)|(?:embed/)|(?:be/))([A-Za-z0-9_-]+)%', $url ) );
			$urlVimeo										=	( ( $urlDomain === 'vimeo' ) && ( ! $urlExt ) && preg_match( '%(?:/\d+|/[A-Za-z0-9_-]+/[A-Za-z0-9_-]+)$%i', $url ) );

			if ( $urlYouTube ) {
				$urlMimeType								=	'video/x-youtube';
			} elseif ( $urlVimeo ) {
				$urlMimeType								=	'video/x-vimeo';
			} elseif ( $urlExt === 'm4v' ) {
				$urlMimeType								=	'video/mp4';
			} elseif ( $urlExt === 'mp3' ) {
				$urlMimeType								=	'audio/mp3';
			} elseif ( $urlExt === 'm4a' ) {
				$urlMimeType								=	'audio/mp4';
			} else {
				$urlMimeType								=	cbGetMimeFromExt( $urlExt );
			}

			$fileExtensions									=	array( 'zip', 'rar', 'doc', 'pdf', 'txt', 'xls' );

			if ( $this->stream ) {
				$streamExtensions							=	$this->stream->getString( 'links_file_extensions', 'zip,rar,doc,pdf,txt,xls' );

				$fileExtensions								=	( $streamExtensions && ( $streamExtensions !== 'none' ) ? explode( ',', $streamExtensions ) : array() );
			}

			if ( in_array( $urlExt, array( 'mp4', 'ogv', 'ogg', 'webm', 'm4v' ), true ) || $urlYouTube || $urlVimeo ) {
				$urlType									=	'video';
			} elseif ( in_array( $urlExt, array( 'mp3', 'oga', 'ogg', 'weba', 'wav', 'm4a' ), true ) ) {
				$urlType									=	'audio';
			} elseif ( in_array( $urlExt, array( 'jpg', 'jpeg', 'gif', 'png' ), true ) ) {
				$urlType									=	'image';
			} elseif ( $fileExtensions && in_array( $urlExt, $fileExtensions, true ) ) {
				$urlType									=	'file';
			}

			$client											=	@new \GuzzleHttp\Client();

			try {
				$mediaUrl									=	$url;
				$requestUrl									=	$mediaUrl;

				if ( $urlVimeo ) {
					$requestUrl								=	'https://vimeo.com/api/oembed.xml?url=' . urlencode( $requestUrl );
				}

				$result										=	$client->get( $requestUrl, array( 'timeout' => 10 ) );

				if ( (int) $result->getStatusCode() === 200 ) {
					if ( $urlMimeType === 'application/octet-stream' ) {
						$urlFilename						=	null;
						$urlExt								=	null;

						if ( cbGuzzleVersion() >= 6 ) {
							$contentType					=	$result->getHeaderLine( 'Content-Type' );
							$contentDisposition				=	$result->getHeaderLine( 'Content-Disposition' );
							$lastModified					=	$result->getHeaderLine( 'last-modified' );
						} else {
							$contentType					=	(string) $result->getHeader( 'Content-Type' );
							$contentDisposition				=	(string) $result->getHeader( 'Content-Disposition' );
							$lastModified					=	(string) $result->getHeader( 'last-modified' );
						}

						// We can't find the mimetype or extension from the URL so lets check the headers
						list( $urlMimeType )				=	explode( ';', $contentType, 2 );

						foreach ( cbGetMimeMap() as $ext => $type ) {
							if ( is_array( $type ) ) {
								foreach ( $type as $subExt => $subType ) {
									if ( $urlMimeType === $subType ) {
										$urlExt				=	$subExt;

										break 2;
									}
								}
							} elseif ( $urlMimeType === $type ) {
								$urlExt						=	$ext;

								break;
							}
						}

						if ( preg_match( '/\s*filename\s?=\s?(.*)/', $contentDisposition, $filenameMatches ) ) {
							$filenameParts					=	explode( ';', $filenameMatches[1] );
							$urlFilename					=	trim( $filenameParts[0], '"' );
						}
					}

					if ( $urlExt === 'exe' ) {
						// For security reject all exe attachments:
						$cache[$url]						=	null;

						return null;
					}

					if ( $lastModified ) {
						$attachment['date']					=	Application::Date( $lastModified, 'UTC' )->getTimestamp();
					}

					// If attachment is a URL lets see if we can find some metadata:
					if ( CBActivity::checkDOMDocument() && ( ( $urlType === 'url' ) || $urlYouTube || $urlVimeo ) ) {
						$body								=	(string) $result->getBody();

						if ( function_exists( 'mb_convert_encoding' ) ) {
							$body							=	mb_convert_encoding( $body, 'HTML-ENTITIES', 'UTF-8' );
						} else {
							$body							=	'<?xml encoding="UTF-8">' . $body;
						}

						$document							=	@new \DOMDocument();

						@$document->loadHTML( $body );

						$xpath								=	@new \DOMXPath( $document );

						$paths								=	array(	'title'			=>	array(	'//meta[@name="og:title"]/@content',
																									'//meta[@name="twitter:title"]/@content',
																									'//meta[@name="title"]/@content',
																									'//meta[@property="og:title"]/@content',
																									'//meta[@property="twitter:title"]/@content',
																									'//meta[@property="title"]/@content',
																									'//oembed/title',
																									'//title'
																								),
																		'description'	=>	array(	'//meta[@name="og:description"]/@content',
																									'//meta[@name="twitter:description"]/@content',
																									'//meta[@name="description"]/@content',
																									'//meta[@property="og:description"]/@content',
																									'//meta[@property="twitter:description"]/@content',
																									'//meta[@property="description"]/@content',
																									'//oembed/description'
																								),
																		'media'			=>	array(	'video'	=>	array(	'//meta[@name="og:video"]/@content',
																														'//meta[@name="og:video:url"]/@content',
																														'//meta[@name="twitter:player"]/@content',
																														'//meta[@property="og:video"]/@content',
																														'//meta[@property="og:video:url"]/@content',
																														'//meta[@property="twitter:player"]/@content',
																														'//video/@src'
																													),
																									'audio'	=>	array(	'//meta[@name="og:audio"]/@content',
																														'//meta[@name="og:audio:url"]/@content',
																														'//meta[@property="og:audio"]/@content',
																														'//meta[@property="og:audio:url"]/@content',
																														'//audio/@src'
																													),
																									'image'	=>	array(	'//meta[@name="og:image"]/@content',
																														'//meta[@name="og:image:url"]/@content',
																														'//meta[@name="twitter:image"]/@content',
																														'//meta[@name="image"]/@content',
																														'//meta[@property="og:image"]/@content',
																														'//meta[@property="og:image:url"]/@content',
																														'//meta[@property="twitter:image"]/@content',
																														'//meta[@property="image"]/@content',
																														'//oembed/thumbnail_url',
																														'//img/@src'
																													)
																								)
																	);

						foreach ( $paths as $item => $itemPaths ) {
							$attachment[$item]										=	array();

							foreach ( $itemPaths as $subItem => $itemPath ) {
								if ( $item === 'media' ) {
									$attachment[$item][$subItem]					=	array();
									$existing										=	array();

									foreach ( $itemPath as $subItemPath ) {
										$nodes										=	@$xpath->query( $subItemPath );
										$limit										=	0;

										if ( ( $nodes !== false ) && isset( $nodes->length ) && $nodes->length ) {
											foreach ( $nodes as $node ) {
												$limit++;

												if ( $limit >= 10 ) {
													break;
												}

												$nodeUrl							=	( isset( $node->nodeValue ) ? $node->nodeValue : null );

												if ( ! $nodeUrl ) {
													continue;
												}

												if ( isset( $node->baseURI ) ) {
													if ( $nodeUrl[0] === '/' ) {
														if ( substr( $node->baseURI, -1, 1 ) === '/' ) {
															$nodeUrl				=	$node->baseURI . substr( $nodeUrl, 1 );
														} else {
															$nodeUrl				=	$node->baseURI . $nodeUrl;
														}
													} elseif ( ! preg_match( $this->regexp['link'], $nodeUrl ) ) {
														$nodeUrl					=	$node->baseURI . $nodeUrl;
													}
												} elseif ( $nodeUrl[0] === '/' ) {
													if ( $url[strlen( $url ) - 1] === '/' ) {
														$nodeUrl					=	$url . substr( $nodeUrl, 1 );
													} else {
														$nodeUrl					=	$url . $nodeUrl;
													}
												} elseif ( ! preg_match( $this->regexp['link'], $nodeUrl ) ) {
													$nodeUrl						=	$url . $nodeUrl;
												}

												if ( preg_match( $this->regexp['link'], $nodeUrl ) ) {
													if ( in_array( $nodeUrl, $existing, true ) ) {
														continue;
													}

													$itemDomain						=	preg_replace( '/^(?:(?:\w+\.)*)?(\w+)\..+$/', '\1', parse_url( $nodeUrl, PHP_URL_HOST ) );
													$itemExt						=	strtolower( pathinfo( $nodeUrl, PATHINFO_EXTENSION ) );
													$itemYouTube					=	( in_array( $itemDomain, array( 'youtube', 'youtu' ), true ) && ( ! $itemExt ) && preg_match( '%(?:(?:watch\?v=)|(?:embed/)|(?:be/))([A-Za-z0-9_-]+)%', $nodeUrl ) );
													$itemVimeo						=	( ( $itemDomain === 'vimeo' ) && ( ! $itemExt ) && preg_match( '%/(\d+)$%', $nodeUrl ) );

													if ( $itemYouTube ) {
														$itemMimeType				=	'video/x-youtube';
													} elseif ( $itemVimeo ) {
														$itemMimeType				=	'video/x-vimeo';
													} elseif ( $itemExt === 'm4v' ) {
														$itemMimeType				=	'video/mp4';
													} elseif ( $itemExt === 'mp3' ) {
														$itemMimeType				=	'audio/mp3';
													} elseif ( $itemExt === 'm4a' ) {
														$itemMimeType				=	'audio/mp4';
													} else {
														$itemMimeType				=	cbGetMimeFromExt( $itemExt );

														if ( $itemMimeType === 'application/octet-stream' ) {
															continue;
														}
													}

													$attachment[$item][$subItem][]	=	array(	'type'		=>	$subItem,
																								'url'		=>	$nodeUrl,
																								'filename'	=>	( ! ( $itemYouTube || $itemVimeo ) ? pathinfo( $nodeUrl, PATHINFO_FILENAME ) . '.' . $itemExt : null ),
																								'mimetype'	=>	$itemMimeType,
																								'extension'	=>	$itemExt,
																								'filesize'	=>	0, // We don't need the filesize for link media
																								'internal'	=>	Application::Router()->isInternal( $nodeUrl )
																							);

													$existing[]						=	$nodeUrl;
												}
											}
										}
									}
								} else {
									$nodes											=	@$xpath->query( $itemPath );

									if ( ( $nodes !== false ) && isset( $nodes->length ) && $nodes->length ) {
										foreach ( $nodes as $node ) {
											$nodeUrl								=	( isset( $node->nodeValue ) ? $node->nodeValue : null );

											if ( ! $nodeUrl ) {
												continue;
											}

											if ( in_array( $nodeUrl, $attachment[$item], true ) ) {
												continue;
											}

											$attachment[$item][]					=	$nodeUrl;
										}
									}
								}
							}
						}

						$urlFileSize												=	( ! ( $urlYouTube || $urlVimeo ) ? (int) $result->getHeader( 'Content-Length' ) : 0 );

						if ( $urlVimeo ) {
							$nodes													=	@$xpath->query( '//oembed/video_id' );
							$vimeoId												=	null;

							if ( ( $nodes !== false ) && isset( $nodes->length ) && $nodes->length ) {
								foreach ( $nodes as $node ) {
									if ( ! isset( $node->nodeValue ) ) {
										continue;
									}

									$vimeoId										=	Get::clean( $node->nodeValue, GetterInterface::INT );
									break;
								}
							}

							if ( $vimeoId ) {
								$mediaUrl											=	'https://vimeo.com/' . $vimeoId;
							}
						}
					}
				} else {
					// Allow trusted domains to fallthrough:
					if ( ! in_array( $urlDomain, $trustedDomains, true ) ) {
						$cache[$url]						=	null;

						return null;
					}
				}
			} catch( \Exception $e ) {
				// Allow trusted domains to fallthrough:
				if ( ! in_array( $urlDomain, $trustedDomains, true ) ) {
					$cache[$url]							=	null;

					return null;
				}
			}

			if ( in_array( $urlType, array( 'video', 'audio', 'image', 'file' ), true ) ) {
				$attachment['media'][$urlType][]			=	array(	'type'		=>	$urlType,
																		'url'		=>	$mediaUrl,
																		'filename'	=>	( ! ( $urlYouTube || $urlVimeo ) ? $urlFilename : null ),
																		'mimetype'	=>	$urlMimeType,
																		'extension'	=>	$urlExt,
																		'filesize'	=>	$urlFileSize,
																		'internal'	=>	Application::Router()->isInternal( $url )
																	);

				$attachment['type']							=	$urlType;
			}

			$attachment										=	json_encode( $attachment );

			if ( ! is_dir( $cachePath ) ) {
				$oldMask									=	@umask( 0 );

				if ( @mkdir( $cachePath, 0755, true ) ) {
					@umask( $oldMask );
					@chmod( $cachePath, 0755 );
				} else {
					@umask( $oldMask );
				}
			}

			file_put_contents( $cacheFile, $attachment );
		}

		if ( $attachment ) {
			$attachment										=	new Registry( $attachment );

			$this->attachments[]							=	$attachment;
		} else {
			$attachment										=	null;
		}

		$cache[$url]										=	$attachment;

		return $cache[$url];
	}

	/**
	 * Runs string through all parsers
	 *
	 * @param array $ignore
	 * @param bool  $html
	 * @return null|string
	 */
	public function parse( $ignore = array(), $html = true )
	{
		global $_PLUGINS;

		if ( ! $this->string ) {
			return null;
		}

		if ( $this->stream ) {
			if ( ! $this->stream->getBool( 'parser_substitutions', false ) ) {
				$ignore[]	=	'substitutions';
			}

			if ( ! $this->stream->getBool( 'parser_reactions', true ) ) {
				$ignore[]	=	'reactions';
			}

			if ( ! $this->stream->getBool( 'parser_emotes', true ) ) {
				$ignore[]	=	'emotes';
			}

			if ( ! $this->stream->getBool( 'parser_hashtags', true ) ) {
				$ignore[]	=	'hashtags';
			}

			if ( ! $this->stream->getBool( 'parser_profiles', true ) ) {
				$ignore[]	=	'profiles';
			}

			if ( ! $this->stream->getBool( 'parser_links', true ) ) {
				$ignore[]	=	'links';
			}

			if ( ! $this->stream->getBool( 'parser_prepare', false ) ) {
				$ignore[]	=	'prepare';
			}

			if ( ! $this->stream->getBool( 'parser_bbcode', false ) ) {
				$ignore[]	=	'bbcode';
			}
		}

		if ( ! in_array( 'prepare', $ignore, true ) ) {
			$this->prepare();
		}

		if ( ! in_array( 'substitutions', $ignore, true ) ) {
			$this->substitutions();
		}

		if ( ! in_array( 'reactions', $ignore, true ) ) {
			$this->reactions();
		}

		if ( ! in_array( 'emotes', $ignore, true ) ) {
			$this->emotes();
		}

		if ( ! in_array( 'hashtags', $ignore, true ) ) {
			$this->hashtags();
		}

		if ( ! in_array( 'profiles', $ignore, true ) ) {
			$this->profiles();
		}

		if ( ! in_array( 'bbcode', $ignore, true ) ) {
			$this->bbcode();
		}

		if ( ! in_array( 'links', $ignore, true ) ) {
			$this->links();
		}

		if ( ! in_array( 'duplicates', $ignore, true ) ) {
			$this->duplicates();
		}

		if ( ! in_array( 'linebreaks', $ignore, true ) ) {
			$this->linebreaks();
		}

		if ( ! $html ) {
			$this->parsed	=	htmlspecialchars( Get::clean( $this->parsed, GetterInterface::STRING ) );
		}

		$this->parsed		=	trim( $this->parsed );

		$_PLUGINS->trigger( 'activity_onParse', array( &$this->parsed, $this->string, $this->words, $this->source, $this->stream, $ignore, $html ) );

		return $this->parsed;
	}
}
