<?php
/**
* Community Builder (TM)
* @version $Id: $
* @package CommunityBuilder
* @copyright (C) 2004-2019 www.joomlapolis.com / Lightning MultiCom SA - and its licensors, all rights reserved
* @license http://www.gnu.org/licenses/old-licenses/gpl-2.0.html GNU/GPL version 2
*/

namespace CB\Plugin\Activity;

use CBLib\Application\Application;
use CB\Database\Table\UserTable;
use CB\Plugin\Activity\Table\CommentTable;

defined('CBLIB') or die();

/**
 * @method string getAsset()
 * @method Comments setAsset( $asset )
 * @method array getAssets()
 * @method Comments setAssets( $assets )
 * @method string getParentAsset()
 * @method Comments setParentAsset( $asset )
 * @method UserTable getUser()
 * @method Comments setUser( $user )
 * @method int|array getId()
 * @method Comments setId( $id )
 * @method array getModerators()
 * @method Comments setModerators( $moderators )
 * @method int|array  getUserId()
 * @method Comments setUserId( $userId )
 * @method string|array getDate()
 * @method Activity setDate( $datetime )
 * @method string getSearch()
 * @method Comments setSearch( $search )
 * @method string getMessage()
 * @method Comments setMessage( $message )
 * @method string getHidden()
 * @method Comments setHidden( $hidden )
 * @method bool getInline()
 * @method Comments setInline( $inline )
 * @method int getPublished()
 * @method Comments setPublished( $published )
 */
class Comments extends Stream implements CommentsInterface
{
	/** @var array $defaults */
	protected $defaults					=	array(	'template'							=>	null,
													'layout'							=>	'stream',
													'direction'							=>	'down',
													'auto_update'						=>	false,
													'auto_load'							=>	false,
													'pinned'							=>	true,
													'create'							=>	true,
													'create_access'						=>	2,
													'create_connected'					=>	false,
													'create_system'						=>	false,
													'message_limit'						=>	400,
													'collapsed'							=>	true,
													'paging'							=>	true,
													'paging_first_limit'				=>	15,
													'paging_limit'						=>	15,
													'parser_substitutions'				=>	false,
													'parser_emotes'						=>	true,
													'parser_reactions'					=>	true,
													'parser_hashtags'					=>	true,
													'parser_profiles'					=>	true,
													'parser_links'						=>	true,
													'parser_prepare'					=>	false,
													'parser_bbcode'						=>	false,
													'actions'							=>	false,
													'actions_message_limit'				=>	100,
													'actions_include'					=>	'0',
													'actions_exclude'					=>	'0',
													'locations'							=>	false,
													'locations_address_limit'			=>	200,
													'locations_include'					=>	'0',
													'locations_exclude'					=>	'0',
													'links'								=>	false,
													'links_file_extensions'				=>	'zip,rar,doc,pdf,txt,xls',
													'links_embedded'					=>	false,
													'links_link_limit'					=>	5,
													'tags'								=>	false,
													'likes'								=>	false,
													'likes_include'						=>	'0',
													'likes_exclude'						=>	'0',
													'themes'							=>	false,
													'themes_include'					=>	'0',
													'themes_exclude'					=>	'0',
													'replies'							=>	false,
													'replies_direction'					=>	'up',
													'replies_auto_update'				=>	false,
													'replies_auto_load'					=>	false,
													'replies_pinned'					=>	false,
													'replies_create'					=>	true,
													'replies_create_access'				=>	2,
													'replies_create_connected'			=>	false,
													'replies_create_system'				=>	false,
													'replies_message_limit'				=>	400,
													'replies_collapsed'					=>	false,
													'replies_paging'					=>	true,
													'replies_paging_first_limit'		=>	5,
													'replies_paging_limit'				=>	5,
													'replies_parser_substitutions'		=>	false,
													'replies_parser_emotes'				=>	true,
													'replies_parser_reactions'			=>	true,
													'replies_parser_hashtags'			=>	true,
													'replies_parser_profiles'			=>	true,
													'replies_parser_links'				=>	true,
													'replies_parser_prepare'			=>	false,
													'replies_parser_bbcode'				=>	false,
													'replies_actions'					=>	false,
													'replies_actions_message_limit'		=>	100,
													'replies_actions_include'			=>	'0',
													'replies_actions_exclude'			=>	'0',
													'replies_locations'					=>	false,
													'replies_locations_address_limit'	=>	200,
													'replies_locations_include'			=>	'0',
													'replies_locations_exclude'			=>	'0',
													'replies_links'						=>	false,
													'replies_links_file_extensions'		=>	'zip,rar,doc,pdf,txt,xls',
													'replies_links_embedded'			=>	false,
													'replies_links_link_limit'			=>	5,
													'replies_tags'						=>	false,
													'replies_likes'						=>	false,
													'replies_likes_include'				=>	'0',
													'replies_likes_exclude'				=>	'0',
													'replies_themes'					=>	false,
													'replies_themes_include'			=>	'0',
													'replies_themes_exclude'			=>	'0'
												);

	/** @var CommentTable[] $loadedRows */
	protected static $loadedRows		=	array();

	/**
	 * Constructor for comments object
	 *
	 * @param null|string|array  $assets
	 * @param null|int|UserTable $user
	 * @param null|string        $namespace
	 */
	public function __construct( $assets = null, $user = null, $namespace = null )
	{
		$this->set( 'published', 1 );

		parent::__construct( $assets, $user, $namespace );
	}

	/**
	 * @param string $name
	 * @param array  $arguments
	 * @return self|string|int|array|null
	 */
	public function __call( $name, $arguments )
	{
		$method									=	substr( $name, 0, 3 );

		if ( in_array( $method, array( 'get', 'set' ), true ) ) {
			$variables							=	array( 'asset', 'assets', 'user', 'id', 'parent', 'moderators', 'user_id', 'date', 'search', 'message', 'hidden', 'inline', 'published' );
			$variable							=	strtolower( substr( $name, 3 ) );

			switch ( $variable ) {
				case 'userid':
					$variable					=	'user_id';
					break;
				case 'parentasset':
					$variable					=	'parent';
					break;
			}

			if ( in_array( $variable, $variables, true ) ) {
				switch ( $method ) {
					case 'get':
						switch ( $variable ) {
							case 'asset':
								return $this->asset();
							case 'assets':
								return $this->assets();
							case 'user':
								return $this->user();
							case 'id':
							case 'user_id':
								if ( is_array( $this->getRaw( $variable ) ) ) {
									return $this->getRaw( $variable, array() );
								}

								return $this->getInt( $variable, 0 );
							case 'moderators':
								return $this->getRaw( $variable, array() );
							case 'inline':
							case 'hidden':
								return $this->getBool( $variable, false );
							case 'published':
								return $this->getInt( $variable, 0 );
							default:
								if ( is_array( $this->getRaw( $variable ) ) ) {
									return $this->getRaw( $variable, array() );
								}

								return $this->getString( $variable );
						}
					case 'set':
						switch ( $variable ) {
							case 'asset':
							case 'assets':
								$this->assets( ( $arguments ? $arguments[0] : null ) );
								break;
							case 'user':
								$this->user( ( $arguments ? $arguments[0] : null ) );
								break;
							default:
								$this->set( $variable, ( $arguments ? $arguments[0] : null ) );
								break;
						}

						return $this;
				}
			}
		}

		trigger_error( 'Call to undefined method ' . __CLASS__ . '::' . $name . '()', E_USER_ERROR );
	}

	/**
	 * Retrieves comments rows or row count
	 *
	 * @param string $output
	 * @return CommentTable[]|int
	 */
	public function rows( $output = null )
	{
		global $_CB_database, $_PLUGINS;

		static $cache						=	array();

		$myId								=	Application::MyUser()->getUserId();
		$id									=	$this->getRaw( 'id' );
		$hasId								=	( ( ( $id !== null ) && ( $id !== '' ) ) || ( is_array( $id ) && $id ) );

		if ( $this->getInt( 'paging_limitstart', 0 ) === 0 ) {
			$pageLimit						=	$this->getInt( 'paging_first_limit', 15 );
		} else {
			$pageLimit						=	$this->getInt( 'paging_limit', 15 );
		}

		$paging								=	( ( ! $hasId ) && $pageLimit && ( $output !== 'all' ) );
		$select								=	array();
		$join								=	array();
		$where								=	array();

		if ( $output === 'count' ) {
			$select[]						=	'COUNT(*)';
		} else {
			$select[]						=	'a.*';
		}

		if ( $hasId ) {
			if ( is_array( $this->getRaw( 'id' ) ) ) {
				$where[]					=	"a." . $_CB_database->NameQuote( 'id' ) . " IN " . $_CB_database->safeArrayOfIntegers( $id );
			} else {
				$where[]					=	"a." . $_CB_database->NameQuote( 'id' ) . " = " . (int) $id;
			}
		}

		$userId								=	$this->getRaw( 'user_id' );

		if ( ( ( $userId !== null ) && ( $userId !== '' ) ) || ( is_array( $userId ) && $userId ) ) {
			if ( is_array( $userId ) ) {
				$where[]					=	"a." . $_CB_database->NameQuote( 'user_id' ) . " IN " . $_CB_database->safeArrayOfIntegers( $userId );
			} else {
				$where[]					=	"a." . $_CB_database->NameQuote( 'user_id' ) . " = " . (int) $userId;
			}
		}

		if ( $this->assets() && ( ! in_array( 'all', $this->assets(), true ) ) )  {
			$queryAssets					=	$this->queryAssets();

			if ( $queryAssets['assets'] || $queryAssets['wildcards'] || $queryAssets['exists'] ) {
				$assetsWhere				=	array();

				if ( $queryAssets['assets'] ) {
					$assetsWhere[]			=	"a." . $_CB_database->NameQuote( 'asset' ) . ( count( $queryAssets['assets'] ) > 1 ? " IN " . $_CB_database->safeArrayOfStrings( $queryAssets['assets'] ) : " = " . $_CB_database->Quote( $queryAssets['assets'][0] ) );
				}

				if ( $queryAssets['wildcards'] ) {
					foreach ( $queryAssets['wildcards'] as $wildcard ) {
						$assetsWhere[]		=	"a." . $_CB_database->NameQuote( 'asset' ) . " LIKE " . $_CB_database->Quote( $wildcard );
					}
				}

				if ( $queryAssets['exists'] ) {
					foreach ( $queryAssets['exists'] as $exist ) {
						$assetsWhere[]		=	"EXISTS ( " . $exist . " )";
					}
				}

				$where[]					=	( count( $assetsWhere ) > 1 ? "( " . implode( " OR ", $assetsWhere ) . " )" : $assetsWhere[0] );
			} elseif ( $output === 'count' ) {
				return 0;
			} else {
				return array();
			}
		}

		if ( ! $hasId ) {
			$date							=	$this->getRaw( 'date' );

			if ( is_array( $date ) ) {
				if ( count( $date ) > 1 ) {
					if ( ! in_array( $date[1], array( '=', '<>', '<', '>', '<=', '>=', 'REGEXP', 'NOT REGEXP', 'LIKE', 'NOT LIKE' ), true ) ) {
						$date[1]			=	'>';
					}

					$where[]				=	"a." . $_CB_database->NameQuote( 'date' ) . " " . $date[1] . " " . $_CB_database->Quote( ( is_int( $date[0] ) ? Application::Database()->getUtcDateTime( $date[0] ) : $date[0] ) );
				} else {
					$where[]				=	"a." . $_CB_database->NameQuote( 'date' ) . " > " . $_CB_database->Quote( ( is_int( $date[0] ) ? Application::Database()->getUtcDateTime( $date[0] ) : $date[0] ) );
				}
			} elseif ( $date ) {
				$where[]					=	"a." . $_CB_database->NameQuote( 'date' ) . " = " . $_CB_database->Quote( ( is_int( $date ) ? Application::Database()->getUtcDateTime( $date ) : $date ) );
			}

			if ( $this->getString( 'message', '' ) !== '' ) {
				if ( strpos( $this->getString( 'message' ), '%' ) !== false ) {
					$where[]				=	"a." . $_CB_database->NameQuote( 'message' ) . " LIKE " . $_CB_database->Quote( $this->getString( 'message' ) );
				} else {
					$where[]				=	"a." . $_CB_database->NameQuote( 'message' ) . " = " . $_CB_database->Quote( $this->getString( 'message' ) );
				}
			}

			if ( $this->getString( 'search', '' ) !== '' ) {
				$where[]					=	"( a." . $_CB_database->NameQuote( 'message' ) . " LIKE " . $_CB_database->Quote( '%' . $_CB_database->getEscaped( $this->getString( 'search' ), true ) . '%', false )
											.	" OR a." . $_CB_database->NameQuote( 'date' ) . " LIKE " . $_CB_database->Quote( '%' . $_CB_database->getEscaped( $this->getString( 'search' ), true ) . '%', false ) . " )";
			}

			if ( $myId ) {
				if ( $this->getBool( 'hidden', false ) ) {
					$hiddenWhere			=	array();

					$select[]				=	"h1." . $_CB_database->NameQuote( 'id' ) . " AS _hidden_id";

					$join[]					=	"LEFT JOIN " . $_CB_database->NameQuote( '#__comprofiler_plugin_activity_hidden' ) . " AS h1"
											.	" ON h1." . $_CB_database->NameQuote( 'user_id' ) . " = " . $myId
											.	" AND h1." . $_CB_database->NameQuote( 'type' ) . " = " . $_CB_database->Quote( 'comment' )
											.	" AND h1." . $_CB_database->NameQuote( 'object' ) . " = a." . $_CB_database->NameQuote( 'id' );

					$hiddenWhere[]			=	"h1." . $_CB_database->NameQuote( 'id' ) . " IS NOT NULL";

					$select[]				=	"h2." . $_CB_database->NameQuote( 'id' ) . " AS _hidden_user";

					$join[]					=	"LEFT JOIN " . $_CB_database->NameQuote( '#__comprofiler_plugin_activity_hidden' ) . " AS h2"
											.	" ON h2." . $_CB_database->NameQuote( 'user_id' ) . " = " . $myId
											.	" AND h2." . $_CB_database->NameQuote( 'type' ) . " = " . $_CB_database->Quote( 'comment.user' )
											.	" AND h2." . $_CB_database->NameQuote( 'object' ) . " = a." . $_CB_database->NameQuote( 'user_id' );

					$hiddenWhere[]			=	"h2." . $_CB_database->NameQuote( 'id' ) . " IS NOT NULL";

					$select[]				=	"h3." . $_CB_database->NameQuote( 'id' ) . " AS _hidden_asset";

					$join[]					=	"LEFT JOIN " . $_CB_database->NameQuote( '#__comprofiler_plugin_activity_hidden' ) . " AS h3"
											.	" ON h3." . $_CB_database->NameQuote( 'user_id' ) . " = " . $myId
											.	" AND h3." . $_CB_database->NameQuote( 'type' ) . " = " . $_CB_database->Quote( 'comment.asset' )
											.	" AND h3." . $_CB_database->NameQuote( 'asset' ) . " = a." . $_CB_database->NameQuote( 'asset' );

					$hiddenWhere[]			=	"h3." . $_CB_database->NameQuote( 'id' ) . " IS NOT NULL";

					$where[]				=	"( " . implode( " OR ", $hiddenWhere ) . " )";
				} else {
					$where[]				=	"NOT EXISTS ("
											.	" SELECT 1 FROM " . $_CB_database->NameQuote( '#__comprofiler_plugin_activity_hidden' ) . " AS h1"
											.	" WHERE h1." . $_CB_database->NameQuote( 'user_id' ) . " = " . $myId
											.	" AND h1." . $_CB_database->NameQuote( 'type' ) . " = " . $_CB_database->Quote( 'comment' )
											.	" AND h1." . $_CB_database->NameQuote( 'object' ) . " = a." . $_CB_database->NameQuote( 'id' )
											.	" )";

					$where[]				=	"NOT EXISTS ("
											.	" SELECT 1 FROM " . $_CB_database->NameQuote( '#__comprofiler_plugin_activity_hidden' ) . " AS h2"
											.	" WHERE h2." . $_CB_database->NameQuote( 'user_id' ) . " = " . $myId
											.	" AND h2." . $_CB_database->NameQuote( 'type' ) . " = " . $_CB_database->Quote( 'comment.user' )
											.	" AND h2." . $_CB_database->NameQuote( 'object' ) . " = a." . $_CB_database->NameQuote( 'user_id' )
											.	" )";

					$where[]				=	"NOT EXISTS ("
											.	" SELECT 1 FROM " . $_CB_database->NameQuote( '#__comprofiler_plugin_activity_hidden' ) . " AS h3"
											.	" WHERE h3." . $_CB_database->NameQuote( 'user_id' ) . " = " . $myId
											.	" AND h3." . $_CB_database->NameQuote( 'type' ) . " = " . $_CB_database->Quote( 'comment.asset' )
											.	" AND h3." . $_CB_database->NameQuote( 'asset' ) . " = a." . $_CB_database->NameQuote( 'asset' )
											.	" )";
				}
			}
		}

		if ( ( $this->getRaw( 'published' ) !== null ) && ( $this->getRaw( 'published' ) !== '' ) ) {
			$where[]						=	"a." . $_CB_database->NameQuote( 'published' ) . " = " . $this->getInt( 'published' );
		}

		$_PLUGINS->trigger( 'activity_onQueryCommentsStream', array( $output, &$select, &$join, &$where, &$this ) );

		$query								=	"SELECT " . implode( ", ", $select )
											.	"\n FROM " . $_CB_database->NameQuote( '#__comprofiler_plugin_activity_comments' ) . " AS a"
											.	"\n LEFT JOIN " . $_CB_database->NameQuote( '#__comprofiler' ) . " AS cb"
											.	" ON cb." . $_CB_database->NameQuote( 'id' ) . " = a." . $_CB_database->NameQuote( 'user_id' )
											.	"\n LEFT JOIN " . $_CB_database->NameQuote( '#__users' ) . " AS j"
											.	" ON j." . $_CB_database->NameQuote( 'id' ) . " = a." . $_CB_database->NameQuote( 'user_id' )
											.	( $join ? "\n " . implode( "\n ", $join ) : null )
											.	"\n WHERE cb." . $_CB_database->NameQuote( 'approved' ) . " = 1"
											.	"\n AND cb." . $_CB_database->NameQuote( 'confirmed' ) . " = 1"
											.	"\n AND j." . $_CB_database->NameQuote( 'block' ) . " = 0"
											.	( $where ? "\n AND " . implode( "\n AND ", $where ) : null );

		if ( ( ! $hasId ) && ( $output !== 'count' ) ) {
			$query							.=	"\n ORDER BY ";

			if ( $this->getBool( 'pinned', true ) ) {
				$query						.=	"a." . $_CB_database->NameQuote( 'pinned' ) . " DESC, ";
			}

			$query							.=	"a." . $_CB_database->NameQuote( 'date' ) . " DESC";
		}

		$cacheId							=	md5( $query . ( $output === 'count' ? $output : ( $output ? $output : null ) . ( $paging ? $this->getInt( 'paging_limitstart', 0 ) . $pageLimit : null ) ) );

		if ( ( ! isset( $cache[$cacheId] ) ) || ( ( ( $output === 'count' ) && $this->clearRowCount ) || $this->clearRowSelect ) ) {
			if ( $output === 'count' ) {
				$this->clearRowCount		=	false;

				$_CB_database->setQuery( $query );

				$cache[$cacheId]			=	(int) $_CB_database->loadResult();

				$this->set( 'query_count', $cache[$cacheId] );
			} else {
				$this->clearRowSelect		=	false;

				if ( $paging ) {
					$_CB_database->setQuery( $query, $this->getInt( 'paging_limitstart', 0 ), ( $pageLimit + 1 ) );
				} else {
					$_CB_database->setQuery( $query );
				}

				$this->set( 'paging_limitstart', ( $this->getInt( 'paging_limitstart', 0 ) + $pageLimit ) );

				$rows						=	$_CB_database->loadObjectList( 'id', '\CB\Plugin\Activity\Table\CommentTable', array( $_CB_database ) );
				$rowsCount					=	count( $rows );

				if ( $paging ) {
					$this->set( 'paging_total', $rowsCount );

					$rows					=	array_slice( $rows, 0, $pageLimit, true );
					$rowsCount				=	count( $rows );
				}

				$userIds					=	array();

				/** @var CommentTable[] $rows */
				foreach ( $rows as $row ) {
					if ( preg_match( '/^profile\.(\d+)/', $row->getString( 'asset' ), $matches ) ) {
						$userIds[]			=	(int) $matches[1];
					}

					$userIds[]				=	$row->getInt( 'user_id', 0 );
				}

				if ( $userIds ) {
					\CBuser::advanceNoticeOfUsersNeeded( $userIds );
				}

				$_PLUGINS->trigger( 'activity_onLoadCommentsStream', array( &$rows, $this ) );

				if ( $rows ) {
					self::$loadedRows		+=	$rows;
				}

				$newCount					=	count( $rows );

				if ( $paging && $rowsCount && ( $newCount < round( $rowsCount / 1.25 ) ) ) {
					$pagingTotal			=	$this->getInt( 'paging_total', 0 );
					$nextLimit				=	( $pageLimit - $newCount );

					if ( $nextLimit <= 0 ) {
						$nextLimit			=	1;
					}

					$this->set( 'paging_limit', $nextLimit );

					$cache[$cacheId]		=	( $rows + $this->rows( $output ) );

					$this->set( 'paging_total', $pagingTotal );
					$this->set( 'paging_limit', $pageLimit );
				} else {
					$cache[$cacheId]		=	$rows;
				}

				if ( $cache[$cacheId] ) {
					if ( $this->getBool( 'replies', false ) ) {
						CBActivity::prefetchAssets( 'comments', $cache[$cacheId], $this );
					}

					if ( $this->getBool( 'tags', false ) ) {
						CBActivity::prefetchAssets( 'tags', $cache[$cacheId], $this );
					}

					if ( $this->getBool( 'likes', false ) ) {
						CBActivity::prefetchAssets( 'likes', $cache[$cacheId], $this );
					}
				}
			}
		} elseif ( $output !== 'count' ) {
			$this->set( 'paging_limitstart', ( $this->getInt( 'paging_limitstart', 0 ) + count( $cache[$cacheId] ) ) );
		}

		return $cache[$cacheId];
	}

	/**
	 * Retrieves comments row
	 *
	 * @param int $id
	 * @return CommentTable
	 */
	public function row( $id )
	{
		if ( ! $id ) {
			return new CommentTable();
		}

		if ( isset( self::$loadedRows[$id] ) ) {
			return self::$loadedRows[$id];
		}

		static $cache		=	array();

		if ( ! isset( $cache[$id] ) ) {
			$rows			=	$this->reset()->setId( $id )->rows();

			if ( isset( $rows[$id] ) ) {
				$row		=	$rows[$id];
			} else {
				$row		=	new CommentTable();
			}

			$cache[$id]		=	$row;
		}

		return $cache[$id];
	}

	/**
	 * Outputs comments HTML
	 *
	 * @param null|string $view
	 * @param int         $id
	 * @param array       $params
	 * @return string
	 */
	public function comments( $view = null, $id = 0, $params = array() )
	{
		return $this->display( $view, $id, $params );
	}
}