<?php
/**
* Community Builder (TM)
* @version $Id: $
* @package CommunityBuilder
* @copyright (C) 2004-2019 www.joomlapolis.com / Lightning MultiCom SA - and its licensors, all rights reserved
* @license http://www.gnu.org/licenses/old-licenses/gpl-2.0.html GNU/GPL version 2
*/

namespace CB\Plugin\Activity;

use CB\Plugin\Activity\Table\ActivityTable;
use CB\Plugin\Activity\Table\NotificationTable;
use CB\Plugin\Activity\Table\CommentTable;
use CB\Plugin\Activity\Table\TagTable;
use CB\Plugin\Activity\Table\FollowTable;
use CB\Plugin\Activity\Table\LikeTable;
use CBLib\Application\Application;
use CBLib\Registry\ParamsInterface;
use CB\Database\Table\FieldTable;
use CB\Database\Table\TabTable;
use CBLib\Registry\Registry;
use CB\Plugin\Activity\Table\ThemeTable;
use CB\Plugin\Activity\Table\ActionTable;
use CB\Plugin\Activity\Table\LocationTable;
use CB\Plugin\Activity\Table\EmoteTable;
use CB\Plugin\Activity\Table\LikeTypeTable;
use CB\Database\Table\UserTable;
use CBLib\Language\CBTxt;
use CBuser;

defined('CBLIB') or die();

class CBActivity
{

	/**
	 * @return Registry
	 */
	public static function getGlobalParams()
	{
		global $_PLUGINS;

		static $params	=	null;

		if ( ! $params ) {
			$plugin		=	$_PLUGINS->getLoadedPlugin( 'user', 'cbactivity' );
			$params		=	new Registry();

			if ( $plugin ) {
				$params->load( $plugin->getRaw( 'params' ) );
			}
		}

		return $params;
	}

	/**
	 * Loads the stream default properties and binds the stream to the supplied selector
	 *
	 * @param string $selector
	 */
	public static function bindStream( $selector = '.streamBind' )
	{
		global $_CB_framework;

		static $cache				=	array();

		$js							=	null;

		if ( ! isset( $cache['core'] ) ) {
			\cbValidator::loadValidation();
			\initToolTip();

			$_CB_framework->addJQueryPlugin( 'cbactivity', '/components/com_comprofiler/plugin/user/plug_cbactivity/js/jquery.cbactivity.js', array( -1 => array( 'ui-all', 'form', 'autosize', 'livestamp', 'qtip', 'cbtooltip', 'cbmoreless', 'cbrepeat', 'cbselect', 'cbtimeago', 'cbvalidate' ) ) );

			$autoUpdateInterval		=	self::getGlobalParams()->getInt( 'general_auto_update_interval', 60 );

			if ( ! $autoUpdateInterval ) {
				$autoUpdateInterval	=	60;
			}

			$autoLoadInterval		=	self::getGlobalParams()->getInt( 'general_auto_load_interval', 2 );

			if ( ! $autoLoadInterval ) {
				$autoLoadInterval	=	2;
			}

			$js						=	"$.fn.cbactivity.loader.autoUpdaterInterval = " . round( $autoUpdateInterval * 1000 ) . ";"
									.	"$.fn.cbactivity.loader.autoUpdaterMaxAttempts = " . self::getGlobalParams()->getInt( 'general_auto_update_attempts', 60 ) . ";"
									.	"$.fn.cbactivity.loader.autoLoaderInterval = " . round( $autoLoadInterval * 1000 ) . ";"
									.	"$.fn.cbactivity.loader.autoLoaderMaxAttempts = " . self::getGlobalParams()->getInt( 'general_auto_load_attempts', 150 ) . ";";

			$cache['core']			=	true;
		}

		if ( $selector && ( ! isset( $cache[$selector] ) ) ) {
			$js					.=	"$( " . json_encode( $selector, JSON_HEX_TAG ) . " ).cbactivity();";

			$cache[$selector]	=	true;
		}

		if ( $js ) {
			$_CB_framework->outputCbJQuery( $js, 'cbactivity' );
		}
	}

	/**
	 * Try to find the stream asset from object id
	 *
	 * @param string $type
	 * @param int    $id
	 * @return null|string
	 */
	public static function getAsset( $type, $id )
	{
		if ( ! $id ) {
			return null;
		}

		static $cache				=	array();

		if ( ! isset( $cache[$type][$id] ) ) {
			$asset					=	null;

			switch ( $type ) {
				case 'likes':
					$row			=	new LikeTable();

					$row->load( $id );

					$asset			=	$row->getString( 'asset' );
					break;
				case 'following':
					$row			=	new FollowTable();

					$row->load( $id );

					$asset			=	$row->getString( 'asset' );
					break;
				case 'tags':
					$row			=	new TagTable();

					$row->load( $id );

					$asset			=	$row->getString( 'asset' );
					break;
				case 'comments':
					$row			=	new CommentTable();

					$row->load( $id );

					$asset			=	$row->getString( 'asset' );
					break;
				case 'activity':
					$row			=	new ActivityTable();

					$row->load( $id );

					$asset			=	$row->getString( 'asset' );
					break;
				case 'notification':
					$row			=	new NotificationTable();

					$row->load( $id );

					$asset			=	$row->getString( 'asset' );
					break;
			}

			$cache[$type][$id]		=	$asset;
		}

		return $cache[$type][$id];
	}

	/**
	 * Try to build the asset source
	 * 
	 * @param string $asset
	 * @return mixed
	 */
	public static function getSource( $asset )
	{
		global $_PLUGINS;

		if ( ! $asset ) {
			return null;
		}

		static $cache				=	array();

		if ( ! isset( $cache[$asset] ) ) {
			$source					=	null;

			if ( preg_match( '/^(?:(.+)\.)?activity\.(\d+)/', $asset, $matches ) ) {
				static $activity	=	array();

				$id					=	( isset( $matches[2] ) ? (int) $matches[2] : 0 );

				if ( ! isset( $activity[$id] ) ) {
					$row			=	new ActivityTable();

					$row->load( $id );

					$activity[$id]	=	$row;
				}

				$source				=	$activity[$id];
			} elseif ( preg_match( '/^(?:(.+)\.)?comment\.(\d+)/', $asset, $matches ) ) {
				static $comments	=	array();

				$id					=	( isset( $matches[2] ) ? (int) $matches[2] : 0 );

				if ( ! isset( $comments[$id] ) ) {
					$row			=	new CommentTable();

					$row->load( $id );

					$comments[$id]	=	$row;
				}

				$source				=	$comments[$id];
			} elseif ( preg_match( '/^(?:(.+)\.)?tag\.(\d+)/', $asset, $matches ) ) {
				static $tags		=	array();

				$id					=	( isset( $matches[2] ) ? (int) $matches[2] : 0 );

				if ( ! isset( $tags[$id] ) ) {
					$row			=	new TagTable();

					$row->load( $id );

					$tags[$id]		=	$row;
				}

				$source				=	$tags[$id];
			} elseif ( preg_match( '/^(?:(.+)\.)?follow\.(\d+)/', $asset, $matches ) ) {
				static $follow		=	array();

				$id					=	( isset( $matches[2] ) ? (int) $matches[2] : 0 );

				if ( ! isset( $follow[$id] ) ) {
					$row			=	new FollowTable();

					$row->load( $id );

					$follow[$id]	=	$row;
				}

				$source				=	$follow[$id];
			} elseif ( preg_match( '/^(?:(.+)\.)?like\.(\d+)/', $asset, $matches ) ) {
				static $like		=	array();

				$id					=	( isset( $matches[2] ) ? (int) $matches[2] : 0 );

				if ( ! isset( $like[$id] ) ) {
					$row			=	new LikeTable();

					$row->load( $id );

					$like[$id]		=	$row;
				}

				$source				=	$like[$id];
			} elseif ( preg_match( '/^(?:(.+)\.)?notification\.(\d+)/', $asset, $matches ) ) {
				static $notif		=	array();

				$id					=	( isset( $matches[2] ) ? (int) $matches[2] : 0 );

				if ( ! isset( $notif[$id] ) ) {
					$row			=	new NotificationTable();

					$row->load( $id );

					$notif[$id]		=	$row;
				}

				$source				=	$notif[$id];
			} elseif ( preg_match( '/^profile\.(\d+)/', $asset, $matches ) ) {
				$source				=	\CBuser::getInstance( ( isset( $matches[1] ) ? (int) $matches[1] : 0 ) )->getUserData();
			}

			$_PLUGINS->trigger( 'activity_onAssetSource', array( $asset, &$source ) );

			$cache[$asset]			=	$source;
		}

		return $cache[$asset];
	}

	/**
	 * Utility function for grabbing a field while also ensuring proper display access to it
	 *
	 * @param int      $fieldId
	 * @param int|null $profileId
	 * @return FieldTable|null
	 */
	public static function getField( $fieldId, $profileId = null )
	{
		if ( ! $fieldId ) {
			return null;
		}

		$userId								=	Application::MyUser()->getUserId();

		if ( ! $profileId ) {
			$profileId						=	$userId;
		}

		static $fields						=	array();

		if ( ! isset( $fields[$profileId][$userId] ) ) {
			$profileUser					=	\CBuser::getInstance( $profileId, false );

			$fields[$profileId][$userId]	=	$profileUser->_getCbTabs( false )->_getTabFieldsDb( null, $profileUser->getUserData(), 'profile' );
		}

		if ( ! isset( $fields[$profileId][$userId][$fieldId] ) ) {
			return null;
		}

		$field								=	$fields[$profileId][$userId][$fieldId];

		if ( ! ( $field->getRaw( 'params' ) instanceof ParamsInterface ) ) {
			$field->set( 'params', new Registry( $field->getRaw( 'params' ) ) );
		}

		return $field;
	}

	/**
	 * Utility function for grabbing the activity tab while also ensuring proper display access to it
	 *
	 * @param int      $tabId
	 * @param int|null $profileId
	 * @return TabTable|null
	 */
	public static function getTab( $tabId, $profileId = null )
	{
		static $profileTab					=	null;

		if ( ! $tabId ) {
			if ( $profileTab === null ) {
				$profileTab					=	new TabTable();

				$profileTab->load( array( 'pluginclass' => 'cbactivityTab' ) );
			}

			$tabId							=	$profileTab->getInt( 'tabid', 0 );
		}

		if ( ! $tabId ) {
			return null;
		}

		$userId								=	Application::MyUser()->getUserId();

		if ( ! $profileId ) {
			$profileId						=	$userId;
		}

		static $tabs						=	array();

		if ( ! isset( $tabs[$profileId][$userId] ) ) {
			$profileUser					=	\CBuser::getInstance( $profileId, false );

			$tabs[$profileId][$userId]		=	$profileUser->_getCbTabs( false )->_getTabsDb( $profileUser->getUserData(), 'profile', false );
		}

		if ( ! isset( $tabs[$profileId][$userId][$tabId] ) ) {
			return null;
		}

		$tab								=	$tabs[$profileId][$userId][$tabId];

		if ( ! ( $tab->getRaw( 'params' ) instanceof ParamsInterface ) ) {
			$tab->set( 'params', new Registry( $tab->getRaw( 'params' ) ) );
		}

		return $tab;
	}

	/**
	 * Returns an array of user ids connected to the supplied profile id
	 *
	 * @param int|null $profileId
	 * @return int[]
	 */
	public static function getConnections( $profileId = null )
	{
		global $_CB_database;

		if ( ! $profileId ) {
			$profileId				=	Application::MyUser()->getUserId();
		}

		if ( ( ! $profileId ) || ( ! Application::Config()->getBool( 'allowConnections', true ) ) ) {
			return array();
		}

		static $cache				=	array();

		if ( ! isset( $cache[$profileId] ) ) {
			$query					=	"SELECT m." . $_CB_database->NameQuote( 'memberid' )
									.	"\n FROM " . $_CB_database->NameQuote( '#__comprofiler_members' ) . " AS m"
									.	"\n WHERE m." . $_CB_database->NameQuote( 'referenceid' ) . " = " . (int) $profileId
									.	"\n AND m." . $_CB_database->NameQuote( 'accepted' ) . " = 1";
			$_CB_database->setQuery( $query );
			$cache[$profileId]		=	cbToArrayOfInt( $_CB_database->loadResultArray() );
		}

		return $cache[$profileId];
	}

	/**
	 * Returns an array of assets the supplied profile id is following
	 *
	 * @param int|null $profileId
	 * @return string[]
	 */
	public static function getFollowing( $profileId = null )
	{
		global $_CB_database;

		if ( ! $profileId ) {
			$profileId				=	Application::MyUser()->getUserId();
		}

		if ( ! $profileId ) {
			return array();
		}

		static $cache				=	array();

		if ( ! isset( $cache[$profileId] ) ) {
			$query					=	"SELECT DISTINCT " . $_CB_database->NameQuote( 'asset' )
									.	"\n FROM " . $_CB_database->NameQuote( '#__comprofiler_plugin_activity_following' )
									.	"\n WHERE " . $_CB_database->NameQuote( 'user_id' ) . " = " . (int) $profileId;
			$_CB_database->setQuery( $query );
			$cache[$profileId]		=	$_CB_database->loadResultArray();
		}

		return $cache[$profileId];
	}

	/**
	 * Returns an array of assets the supplied profile id liked
	 *
	 * @param int|null $profileId
	 * @return string[]
	 */
	public static function getLikes( $profileId = null )
	{
		global $_CB_database;

		if ( ! $profileId ) {
			$profileId				=	Application::MyUser()->getUserId();
		}

		if ( ! $profileId ) {
			return array();
		}

		static $cache				=	array();

		if ( ! isset( $cache[$profileId] ) ) {
			$query					=	"SELECT DISTINCT " . $_CB_database->NameQuote( 'asset' )
									.	"\n FROM " . $_CB_database->NameQuote( '#__comprofiler_plugin_activity_likes' )
									.	"\n WHERE " . $_CB_database->NameQuote( 'user_id' ) . " = " . (int) $profileId;
			$_CB_database->setQuery( $query );
			$cache[$profileId]		=	$_CB_database->loadResultArray();
		}

		return $cache[$profileId];
	}

	/**
	 * Returns the last read date of an asset for a user
	 *
	 * @param string   $asset
	 * @param int|null $profileId
	 * @return string
	 */
	public static function getRead( $asset, $profileId = null )
	{
		global $_CB_database;

		if ( ! $profileId ) {
			$profileId				=	Application::MyUser()->getUserId();
		}

		if ( ! $profileId ) {
			return '0000-00-00 00:00:00';
		}

		static $cache				=	array();

		if ( ! isset( $cache[$profileId][$asset] ) ) {
			$query					=	"SELECT " . $_CB_database->NameQuote( 'asset' )
									.	", " . $_CB_database->NameQuote( 'date' )
									.	"\n FROM " . $_CB_database->NameQuote( '#__comprofiler_plugin_activity_read' )
									.	"\n WHERE " . $_CB_database->NameQuote( 'user_id' ) . " = " . (int) $profileId;
			$_CB_database->setQuery( $query );

			$cache[$profileId]		=	$_CB_database->loadAssocList( 'asset', 'date' );
		}

		if ( ! isset( $cache[$profileId][$asset] ) ) {
			return '0000-00-00 00:00:00';
		}

		return $cache[$profileId][$asset];
	}

	/**
	 * Finds the asset based off override parameters for a row
	 *
	 * @param string                     $type
	 * @param ActivityTable|CommentTable $row
	 * @param Activity|Comments          $stream
	 * @return string
	 */
	public static function findAssetOverride( $type, $row, $stream )
	{
		$asset								=	$row->getString( 'asset' );

		if ( $type === 'comments' ) {
			if ( $row instanceof CommentTable ) {
				$asset						=	self::findParamOverride( $row, 'replies_asset', 'comment' );

				if ( ! $asset ) {
					$asset					=	'comment';
				}

				$defaultAsset				=	'comment.' . $row->getInt( 'id', 0 );

				switch ( $asset ) {
					case 'asset':
						$asset				=	$row->getString( 'asset' );
						break;
					case 'stream':
						if ( $stream ) {
							$asset			=	$stream->asset();
						} else {
							$asset			=	$defaultAsset;
						}
						break;
					case 'comment':
						$asset				=	$defaultAsset;
						break;
				}
			} else {
				$asset						=	self::findParamOverride( $row, 'comments_asset', 'activity' );

				if ( ! $asset ) {
					$asset					=	'activity';
				}

				$defaultAsset				=	'activity.' . $row->getInt( 'id', 0 );

				switch ( $asset ) {
					case 'asset':
						$asset				=	$row->getString( 'asset' );
						break;
					case 'stream':
						if ( $stream ) {
							$asset			=	$stream->asset();
						} else {
							$asset			=	$defaultAsset;
						}
						break;
					case 'activity':
						$asset				=	$defaultAsset;
						break;
				}
			}
		} elseif ( $type === 'tags' ) {
			if ( $row instanceof CommentTable ) {
				$asset						=	self::findParamOverride( $row, 'tags_asset', 'comment' );

				if ( ! $asset ) {
					$asset					=	'comment';
				}

				$defaultAsset				=	'comment.' . $row->getInt( 'id', 0 );

				switch ( $asset ) {
					case 'asset':
						$asset				=	$row->getString( 'asset' );
						break;
					case 'stream':
						if ( $stream ) {
							$asset			=	$stream->asset();
						} else {
							$asset			=	$defaultAsset;
						}
						break;
					case 'comment':
						$asset				=	$defaultAsset;
						break;
				}
			} else {
				$asset						=	self::findParamOverride( $row, 'tags_asset', 'activity' );

				if ( ! $asset ) {
					$asset					=	'activity';
				}

				$defaultAsset				=	'activity.' . $row->getInt( 'id', 0 );

				switch ( $asset ) {
					case 'asset':
						$asset				=	$row->getString( 'asset' );
						break;
					case 'stream':
						if ( $stream ) {
							$asset			=	$stream->asset();
						} else {
							$asset			=	$defaultAsset;
						}
						break;
					case 'activity':
						$asset				=	$defaultAsset;
						break;
				}
			}
		} elseif ( $type === 'likes' ) {
			if ( $row instanceof CommentTable ) {
				$asset						=	self::findParamOverride( $row, 'likes_asset', 'comment' );

				if ( ! $asset ) {
					$asset					=	'comment';
				}

				$defaultAsset				=	'comment.' . $row->getInt( 'id', 0 );

				switch ( $asset ) {
					case 'asset':
						$asset				=	$row->getString( 'asset' );
						break;
					case 'stream':
						if ( $stream ) {
							$asset			=	$stream->asset();
						} else {
							$asset			=	$defaultAsset;
						}
						break;
					case 'comment':
						$asset				=	$defaultAsset;
						break;
				}
			} else {
				$asset						=	self::findParamOverride( $row, 'likes_asset', 'activity' );

				if ( ! $asset ) {
					$asset					=	'activity';
				}

				$defaultAsset				=	'activity.' . $row->getInt( 'id', 0 );

				switch ( $asset ) {
					case 'asset':
						$asset				=	$row->getString( 'asset' );
						break;
					case 'stream':
						if ( $stream ) {
							$asset			=	$stream->asset();
						} else {
							$asset			=	$defaultAsset;
						}
						break;
					case 'activity':
						$asset				=	$defaultAsset;
						break;
				}
			}
		}

		return $asset;
	}

	/**
	 * Finds the parameter override value if available
	 *
	 * @param ActivityTable|CommentTable|TagTable|FollowTable|LikeTable|NotificationTable $row
	 * @param string                                                                      $param
	 * @param mixed                                                                       $default
	 * @param Activity|Comments|Tags|Following|Likes|Notifications                        $stream
	 * @return mixed
	 */
	public static function findParamOverride( $row, $param, $default = null, $stream = null )
	{
		if ( $row instanceof ActivityTable ) {
			$override	=	$row->params()->getString( 'defaults.' . $param );
		} else {
			$override	=	null;
		}

		$value			=	$row->params()->getRaw( 'overrides.' . $param, $override );

		if ( ( $value === '' ) || ( $value === null ) || ( $value === '-1' ) ) {
			if ( ! $stream ) {
				return $default;
			}

			if ( is_bool( $default ) ) {
				return $stream->getBool( $param, $default );
			}

			if ( is_int( $default ) ) {
				return $stream->getInt( $param, $default );
			}

			if ( is_array( $default ) ) {
				return $stream->getRaw( $param, $default );
			}

			return $stream->getString( $param, $default );
		}

		return $value;
	}

	/**       
	 * Prefetch assets from rows based off type and return the total for all assets requested
	 * This primarily helps determine if a row has any comments, tags, etc..
	 *
	 * @param string                                               $type
	 * @param ActivityTable[]|CommentTable[]                       $rows
	 * @param Activity|Comments|Tags|Likes|Following|Notifications $stream
	 * @return int
	 */
	public static function prefetchAssets( $type, $rows, $stream )
	{
		global $_CB_database;

		static $cache							=	array();
		static $notifications					=	array();

		if ( ! isset( $cache[$type] ) ) {
			$cache[$type]						=	array();
		}

		$userId									=	$stream->user()->getInt( 'id', 0 );

		if ( ! isset( $notifications[$userId] ) ) {
			$notifications[$userId]				=	array();
		}

		// Determine how specific the caching needs to be:
		if ( $type === 'notifications' ) {
			$caching							=	&$notifications[$userId];
		} else {
			$caching							=	&$cache[$type];
		}

		$totalAssets							=	array();
		$assets									=	array();

		$queryAssets							=	array();
		$queryWildcards							=	array();

		// Prefetch row counts:
		foreach ( $rows as $row ) {
			$rowAssets							=	self::findAssetOverride( $type, $row, $stream );

			if ( ! is_array( $rowAssets ) ) {
				$rowAssets						=	explode( ',', $rowAssets );
			}

			foreach ( $rowAssets as $rowAsset ) {
				if ( ! in_array( $rowAsset, $totalAssets, true ) ) {
					$totalAssets[]				=	$rowAsset;
				}

				if ( ! isset( $caching[$rowAsset] ) && ( ! in_array( $rowAsset, $assets, true ) ) ) {
					$assets[]					=	$rowAsset;

					if ( ( strpos( $rowAsset, '%' ) !== false ) || ( strpos( $rowAsset, '_' ) !== false ) ) {
						$queryWildcards[]		=	$rowAsset;
					} else {
						$queryAssets[]			=	$rowAsset;
					}
				}
			}
		}

		// Prefetch stream counts:
		$allAssets								=	false;

		if ( ( ( $type === 'activity' ) && ( $stream instanceof Activity ) )
			 || ( ( $type === 'comments' ) && ( $stream instanceof Comments ) )
			 || ( ( $type === 'tags' ) && ( $stream instanceof Tags ) )
			 || ( ( $type === 'likes' ) && ( $stream instanceof Likes ) )
			 || ( ( $type === 'following' ) && ( $stream instanceof Following ) )
			 || ( ( $type === 'notifications' ) && ( $stream instanceof Notifications ) )
		) {
			foreach ( $stream->assets() as $streamAsset ) {
				if ( $streamAsset === 'all' ) {
					$allAssets					=	true;

					continue;
				}

				if ( ! in_array( $streamAsset, $totalAssets, true ) ) {
					$totalAssets[]				=	$streamAsset;
				}

				if ( ! isset( $caching[$streamAsset] ) && ( ! in_array( $streamAsset, $assets, true ) ) ) {
					$assets[]					=	$streamAsset;

					if ( ( strpos( $streamAsset, '%' ) !== false ) || ( strpos( $streamAsset, '_' ) !== false ) ) {
						$queryWildcards[]		=	$streamAsset;
					} else {
						$queryAssets[]			=	$streamAsset;
					}
				}
			}
		}

		if ( ( ! $assets ) && $allAssets ) {
			$assets[]							=	'all';
		} else {
			$allAssets							=	false;
		}

		if ( $assets ) {
			switch ( $type ) {
				case 'activity':
					$table						=	'#__comprofiler_plugin_activity';
					break;
				case 'comments':
				case 'replies':
					$table						=	'#__comprofiler_plugin_activity_comments';
					break;
				case 'tags':
					$table						=	'#__comprofiler_plugin_activity_tags';
					break;
				case 'likes':
					$table						=	'#__comprofiler_plugin_activity_likes';
					break;
				case 'following':
					$table						=	'#__comprofiler_plugin_activity_following';
					break;
				case 'notifications':
					$table						=	'#__comprofiler_plugin_activity_notifications';
					break;
				default:
					return 0;
					break;
			}

			$assetsWhere						=	array();
			$queryAssets						=	array_unique( $queryAssets );

			if ( $queryAssets ) {
				$assetsWhere[]					=	"a." . $_CB_database->NameQuote( 'asset' ) . ( count( $queryAssets ) > 1 ? " IN " . $_CB_database->safeArrayOfStrings( $queryAssets ) : " = " . $_CB_database->Quote( $queryAssets[0] ) );
			}

			$queryWildcards						=	array_unique( $queryWildcards );

			if ( $queryWildcards ) {
				foreach ( $queryWildcards as $queryWildcard ) {
					$assetsWhere[]				=	"a." . $_CB_database->NameQuote( 'asset' ) . " LIKE " . $_CB_database->Quote( $queryWildcard );
				}
			}

			if ( ( count( $queryAssets ) > 1 ) || ( count( $assetsWhere ) > 1 ) ) {
				$count							=	"( SELECT COUNT(*)"
												.	" FROM " . $_CB_database->NameQuote( $table ) . " AS b";

				if ( $type === 'notifications' ) {
					$count						.=	" WHERE b." . $_CB_database->NameQuote( 'user' ) . " = " . $stream->user()->getInt( 'id', 0 )
												.	" AND b." . $_CB_database->NameQuote( 'asset' ) . " = a." . $_CB_database->NameQuote( 'asset' );
				} else {
					$count						.=	" WHERE b." . $_CB_database->NameQuote( 'asset' ) . " = a." . $_CB_database->NameQuote( 'asset' );
				}

				if ( in_array( $type, array( 'activity', 'comments', 'notifications' ), true ) ) {
					$count						.=	" AND b." . $_CB_database->NameQuote( 'published' ) . " = 1";
				}

				$count							.=	" )";
			} else {
				$count							=	"COUNT(*)";
			}

			$where								=	array();

			if ( $type === 'notifications' ) {
				$where[]						=	"a." . $_CB_database->NameQuote( 'user' ) . " = " . $stream->user()->getInt( 'id', 0 );
			}

			if ( $assetsWhere ) {
				$where[]						=	( count( $assetsWhere ) > 1 ? "( " . implode( " OR ", $assetsWhere ) . " )" : $assetsWhere[0] );
			}

			if ( in_array( $type, array( 'activity', 'comments', 'notifications' ), true ) ) {
				$where[]						=	"a." . $_CB_database->NameQuote( 'published' ) . " = 1";
			}

			// If we're doing a full count then lets consolidate the query as we don't need any of the grouping:
			if ( $allAssets ) {
				$query							=	"SELECT " . $count
												.	"\n FROM " . $_CB_database->NameQuote( $table ) . " AS a"
												.	( $where ? "\n WHERE " . implode( "\n AND ", $where ) : null );
				$_CB_database->setQuery( $query );
				$caching['all']					=	$_CB_database->loadResult();
			} else {
				if ( ( count( $queryAssets ) > 1 ) || ( count( $assetsWhere ) > 1 ) ) {
					$query						=	"SELECT a." . $_CB_database->NameQuote( 'asset' )
												.	", $count AS total"
												.	"\n FROM " . $_CB_database->NameQuote( $table ) . " AS a"
												.	( $where ? "\n WHERE " . implode( "\n AND ", $where ) : null );

					if ( ( count( $queryAssets ) > 1 ) || ( count( $assetsWhere ) > 1 ) ) {
						$query					.=	"\n GROUP BY a." . $_CB_database->NameQuote( 'asset' );
					}

					$_CB_database->setQuery( $query );
					$prefetch					=	$_CB_database->loadAssocList( 'asset', 'total' );
				} else {
					// Do a fast count and nothing else if we're only looking for the count of a single asset:
					$prefetchAsset				=	( isset( $queryAssets[0] ) ? $queryAssets[0] : ( isset( $queryWildcards[0] ) ? $queryWildcards[0] : null ) );

					$query						=	"SELECT " . $count
												.	"\n FROM " . $_CB_database->NameQuote( $table ) . " AS a"
												.	( $where ? "\n WHERE " . implode( "\n AND ", $where ) : null );
					$_CB_database->setQuery( $query );
					$prefetch					=	array( $prefetchAsset => $_CB_database->loadResult() );
				}

				foreach ( $assets as $asset ) {
					if ( ( strpos( $asset, '%' ) !== false ) || ( strpos( $asset, '_' ) !== false ) ) {
						$wildcardExists			=	preg_grep( '/^' . str_replace( '%', '.+', preg_quote( $asset, '/' ) ) . '/', array_keys( $prefetch ) );

						if ( $wildcardExists ) {
							$assetExists		=	0;

							foreach ( $wildcardExists as $matchedAsset ) {
								if ( ! isset( $prefetch[$matchedAsset] ) ) {
									continue;
								}

								$assetExists	+=	(int) $prefetch[$matchedAsset];
							}

							if ( ! $assetExists ) {
								$assetExists	=	true;
							}
						} else {
							$assetExists		=	false;
						}
					} else {
						$assetExists			=	( array_key_exists( $asset, $prefetch ) ? ( isset( $prefetch[$asset] ) ? (int) $prefetch[$asset] : true ) : false );
					}

					$caching[$asset]			=	$assetExists;
				}
			}
		}

		if ( $caching ) {
			if ( $allAssets ) {
				if ( ! isset( $caching['all'] ) ) {
					return 0;
				}

				return $caching['all'];
			}

			foreach ( $rows as $row ) {
				$rowAssets						=	self::findAssetOverride( $type, $row, $stream );

				if ( ! is_array( $rowAssets ) ) {
					$rowAssets					=	explode( ',', $rowAssets );
				}

				$rowCount						=	0;

				foreach ( $rowAssets as $rowAsset ) {
					if ( isset( $caching[$rowAsset] ) ) {
						if ( ! is_bool( $caching[$rowAsset] ) ) {
							$rowCount			+=	$caching[$rowAsset];
						}

						$row->set( '_' . $type, $caching[$rowAsset] );
					}
				}

				if ( $rowCount ) {
					$row->set( '_' . $type, $rowCount );
				}
			}

			$total								=	0;

			foreach ( $totalAssets as $asset ) {
				if ( ( ! isset( $caching[$asset] ) ) || ( ! is_int( $caching[$asset] ) ) ) {
					continue;
				}

				$total							+=	$caching[$asset];
			}

			return $total;
		}

		return 0;
	}

	/**
	 * Checks if a user can create posts in the supplied stream
	 *
	 * @param string                                               $type
	 * @param Activity|Comments|Tags|Following|Likes|Notifications $stream
	 * @param null|UserTable                                       $user
	 * @return bool
	 */
	public static function canCreate( $type, $stream, $user = null )
	{
		global $_PLUGINS;

		static $cache											=	array();

		if ( $stream instanceof NotificationsInterface ) {
			// Notifications can not be created from frontend operations and must be generated by the system:
			return false;
		}

		if ( ! $user ) {
			$user												=	\CBuser::getMyUserDataInstance();
		} elseif ( is_int( $user ) ) {
			$user												=	\CBuser::getInstance( $user, false )->getUserData();
		}

		$userId													=	$user->getInt( 'id', 0 );

		if ( ( ! $userId ) || ( ! $user->getInt( 'approved', 1 ) ) || ( ! $user->getInt( 'confirmed', 1 ) ) || $user->getInt( 'block', 0 ) ) {
			return false;
		}

		$streamId												=	$stream->id();

		if ( ! isset( $cache[$userId][$type][$streamId] ) ) {
			$checks												=	array(	'moderate'		=>	true,
																			'moderateonly'	=>	false,
																			'accesslevel'	=>	false,
																			'stream'		=>	array(	'connected'	=>	false,
																										'owner'		=>	false,
																										'notowner'	=>	false
																									),
																			'asset'			=>	array(	'connected'	=>	false,
																										'owner'		=>	false,
																										'notowner'	=>	false
																									),
																			'assetonly'		=>	array(	'connected'	=>	false,
																										'owner'		=>	false,
																										'notowner'	=>	false
																									)
																		);

			switch ( $type ) {
				case 'activity':
					if ( $stream instanceof ActivityInterface ) {
						if ( ! $stream->getBool( 'create', true ) ) {
							$cache[$userId][$type][$streamId]	=	false;

							return false;
						}

						$accessLevel							=	$stream->getInt( 'create_access', 2 );

						if ( $accessLevel === -5 ) {
							$checks['moderateonly']				=	true;
						} elseif ( $accessLevel === -6 ) {
							$checks['asset']['notowner']		=	true;
						} elseif ( $accessLevel === -7 ) {
							$checks['asset']['owner']			=	true;
						} else {
							$checks['accesslevel']				=	$accessLevel;
						}

						$checks['asset']['connected']			=	$stream->getBool( 'create_connected', true );
					} else {
						$cache[$userId][$type][$streamId]		=	false;

						return false;
					}
					break;
				case 'comment':
					if ( $stream instanceof ActivityInterface ) {
						if ( ( ! $stream->getBool( 'comments', true ) ) || ( ! $stream->getBool( 'comments_create', true ) ) ) {
							$cache[$userId][$type][$streamId]	=	false;

							return false;
						}

						$accessLevel							=	$stream->getInt( 'comments_create_access', 2 );

						if ( $accessLevel === -5 ) {
							$checks['moderateonly']				=	true;
						} elseif ( $accessLevel === -6 ) {
							$checks['asset']['notowner']		=	true;
						} elseif ( $accessLevel === -7 ) {
							$checks['asset']['owner']			=	true;
						} else {
							$checks['accesslevel']				=	$accessLevel;
						}

						$checks['asset']['connected']			=	$stream->getBool( 'comments_create_connected', true );
					} elseif ( $stream instanceof CommentsInterface ) {
						if ( ! $stream->getBool( 'create', true ) ) {
							$cache[$userId][$type][$streamId]	=	false;

							return false;
						}

						$accessLevel							=	$stream->getInt( 'create_access', 2 );

						if ( $accessLevel === -5 ) {
							$checks['moderateonly']				=	true;
						} elseif ( $accessLevel === -6 ) {
							$checks['asset']['notowner']		=	true;
						} elseif ( $accessLevel === -7 ) {
							$checks['asset']['owner']			=	true;
						} else {
							$checks['accesslevel']				=	$accessLevel;
						}

						$checks['asset']['connected']			=	$stream->getBool( 'create_connected', true );
					} else {
						$cache[$userId][$type][$streamId]		=	false;

						return false;
					}
					break;
				case 'reply':
					if ( $stream instanceof ActivityInterface ) {
						if ( ( ! $stream->getBool( 'comments', true ) ) || ( ! $stream->getBool( 'comments_replies', false ) ) || ( ! $stream->getBool( 'comments_replies_create', true ) ) ) {
							$cache[$userId][$type][$streamId]	=	false;

							return false;
						}

						$accessLevel							=	$stream->getInt( 'comments_replies_create_access', 2 );

						if ( $accessLevel === -5 ) {
							$checks['moderateonly']				=	true;
						} elseif ( $accessLevel === -6 ) {
							$checks['asset']['notowner']		=	true;
						} elseif ( $accessLevel === -7 ) {
							$checks['asset']['owner']			=	true;
						} else {
							$checks['accesslevel']				=	$accessLevel;
						}

						$checks['asset']['connected']			=	$stream->getBool( 'comments_replies_create_connected', true );
					} elseif ( $stream instanceof CommentsInterface ) {
						if ( ( ! $stream->getBool( 'replies', false ) ) || ( ! $stream->getBool( 'replies_create', true ) ) ) {
							$cache[$userId][$type][$streamId]	=	false;

							return false;
						}

						$accessLevel							=	$stream->getInt( 'replies_create_access', 2 );

						if ( $accessLevel === -5 ) {
							$checks['moderateonly']				=	true;
						} elseif ( $accessLevel === -6 ) {
							$checks['asset']['notowner']		=	true;
						} elseif ( $accessLevel === -7 ) {
							$checks['asset']['owner']			=	true;
						} else {
							$checks['accesslevel']				=	$accessLevel;
						}

						$checks['asset']['connected']			=	$stream->getBool( 'replies_create_connected', true );
					} else {
						$cache[$userId][$type][$streamId]		=	false;

						return false;
					}
					break;
				case 'tag':
					if ( $stream instanceof ActivityInterface ) {
						if ( ! $stream->getBool( 'tags', true ) ) {
							$cache[$userId][$type][$streamId]	=	false;

							return false;
						}

						$checks['stream']['owner']				=	true;
					} elseif ( $stream instanceof CommentsInterface ) {
						if ( ! $stream->getBool( 'tags', true ) ) {
							$cache[$userId][$type][$streamId]	=	false;

							return false;
						}

						$checks['stream']['owner']				=	true;
					} elseif ( $stream instanceof TagsInterface ) {
						$checks['stream']['owner']				=	true;
					} else {
						$cache[$userId][$type][$streamId]		=	false;

						return false;
					}
					break;
				case 'follow':
					if ( $stream instanceof FollowingInterface ) {
						$checks['moderate']						=	false;
						$checks['assetonly']['connected']		=	$stream->getBool( 'connected', false );
						$checks['assetonly']['notowner']		=	true;
					} else {
						$cache[$userId][$type][$streamId]		=	false;

						return false;
					}
					break;
				case 'like':
					if ( $stream instanceof LikesInterface ) {
						$checks['moderate']						=	false;
					} else {
						$cache[$userId][$type][$streamId]		=	false;

						return false;
					}
					break;
				default:
					$cache[$userId][$type][$streamId]			=	false;

					return false;
			}

			$canModerate										=	self::canModerate( $stream, $user );

			if ( $checks['moderate'] === true ) {
				if ( $canModerate ) {
					$cache[$userId][$type][$streamId]			=	true;

					return true;
				}

				if ( $checks['moderateonly'] === true ) {
					$cache[$userId][$type][$streamId]			=	false;

					return false;
				}
			}

			$streamOwner										=	$stream->user()->getInt( 'id', 0 );

			if ( preg_match( '/^profile(?:\.(\d+)(?:\.field\.(\d+))?)?/', $stream->asset(), $matches ) ) {
				$assetOwner										=	( isset( $matches[1] ) ? (int) $matches[1] : 0 );

				$profileId										=	( isset( $matches[1] ) ? (int) $matches[1] : $streamOwner );
				$fieldId										=	( isset( $matches[2] ) ? (int) $matches[2] : $stream->getInt( 'field', 0 ) );
				$tabId											=	$stream->getInt( 'tab', 0 );

				if ( $fieldId ) {
					$field										=	self::getField( $fieldId, $profileId );

					if ( ! $field ) {
						$cache[$userId][$type][$streamId]		=	false;

						return false;
					}
				} else {
					$tab										=	self::getTab( $tabId, $profileId );

					if ( ! $tab ) {
						$cache[$userId][$type][$streamId]		=	false;

						return false;
					}
				}
			} else {
				$assetOwner										=	0;
			}

			$access												=	true;

			foreach ( $checks as $checkType => $check ) {
				if ( ! is_array( $check ) ) {
					continue;
				}

				switch ( $checkType ) {
					case 'stream':
						$checkUser								=	$streamOwner;
						break;
					case 'asset':
						$checkUser								=	( $assetOwner ? $assetOwner : $streamOwner );
						break;
					case 'assetonly':
						$checkUser								=	$assetOwner;
						break;
					default:
						$checkUser								=	0;
						break;
				}

				foreach ( $check as $subCheckType => $subCheck ) {
					if ( $subCheck === false ) {
						continue;
					}

					$denied										=	false;

					switch ( $subCheckType ) {
						case 'owner':
							$denied								=	( $checkUser !== $userId );
							break;
						case 'notowner':
							$denied								=	( $checkUser === $userId );
							break;
						case 'connected':
							if ( Application::Config()->getBool( 'allowConnections', true ) && ( ! $canModerate ) ) {
								if ( ( $checkUser !== $userId ) && ( ! in_array( $userId, self::getConnections( $checkUser ), true ) ) ) {
									// We want to fall through on soft connections access check to allow integrations to override this behavior, but other checks must remain as they're hard checks:
									$access						=	false;
								}
							}
							break;
					}

					if ( $denied ) {
						$cache[$userId][$type][$streamId]		=	false;

						return false;
					}
				}
			}

			if ( $checks['accesslevel'] !== false ) {
				if ( ! Application::User( $userId )->canViewAccessLevel( $checks['accesslevel'] ) ) {
					$cache[$userId][$type][$streamId]			=	false;

					return false;
				}
			}

			$_PLUGINS->trigger( 'activity_onStreamCreateAccess', array( &$access, $type, $user, $stream ) );

			$cache[$userId][$type][$streamId]					=	$access;
		}

		return $cache[$userId][$type][$streamId];
	}

	/**
	 * Checks if a user can edit an object on a stream
	 *
	 * @param ActivityTable|CommentTable $row
	 * @param Activity|Comments          $stream
	 * @param null|UserTable             $user
	 * @return bool
	 */
	public static function canEdit( $row, $stream, $user = null )
	{
		if ( $row->isHidden() ) {
			return false;
		}

		if ( ! $user ) {
			$user			=	\CBuser::getMyUserDataInstance();
		} elseif ( is_int( $user ) ) {
			$user			=	\CBuser::getInstance( $user, false )->getUserData();
		}

		$canEdit			=	self::findParamOverride( $row, 'edit', true );

		if ( $canEdit !== false ) {
			$canEdit		=	( ( $user->getInt( 'id', 0 ) === $row->getInt( 'user_id', 0 ) ) || self::canModerate( $stream, $user ) );

			if ( $row->getBool( 'system', false ) && ( ! Application::MyUser()->isGlobalModerator() ) ) {
				$canEdit	=	false;
			}
		}

		return $canEdit;
	}

	/**
	 * Checks if a user can delete an object on a stream
	 *
	 * @param ActivityTable|CommentTable $row
	 * @param Activity|Comments          $stream
	 * @param null|UserTable             $user
	 * @return bool
	 */
	public static function canDelete( $row, $stream, $user = null )
	{
		if ( ! $user ) {
			$user			=	\CBuser::getMyUserDataInstance();
		} elseif ( is_int( $user ) ) {
			$user			=	\CBuser::getInstance( $user, false )->getUserData();
		}

		$canDelete			=	false;

		if ( ( $user->getInt( 'id', 0 ) === $row->getInt( 'user_id', 0 ) ) || self::canModerate( $stream, $user ) ) {
			$canDelete		=	true;

			if ( $row->getBool( 'system', false ) && ( ! Application::MyUser()->isGlobalModerator() ) ) {
				$canDelete	=	false;
			}
		}

		return $canDelete;
	}

	/**
	 * Checks if a user can moderate the stream
	 *
	 * @param Activity|Comments|Tags|Following|Notifications $stream
	 * @param null|UserTable                                 $user
	 * @return bool
	 */
	public static function canModerate( $stream, $user = null )
	{
		global $_PLUGINS;

		static $cache						=	array();

		if ( ! $user ) {
			$user							=	\CBuser::getMyUserDataInstance();
		} elseif ( is_int( $user ) ) {
			$user							=	\CBuser::getInstance( $user, false )->getUserData();
		}

		$userId								=	$user->getInt( 'id', 0 );

		if ( ( ! $userId ) || ( ! $user->getInt( 'approved', 1 ) ) || ( ! $user->getInt( 'confirmed', 1 ) ) || $user->getInt( 'block', 0 ) ) {
			return false;
		}

		if ( Application::User( $userId )->isGlobalModerator() ) {
			return true;
		}

		if ( in_array( $userId, $stream->getRaw( 'moderators', array() ), true ) ) {
			return true;
		}

		$streamId							=	$stream->id();

		if ( ! isset( $cache[$userId][$streamId] ) ) {
			$access							=	false;

			$_PLUGINS->trigger( 'activity_onStreamModerateAccess', array( &$access, $user, $stream ) );

			$cache[$userId][$streamId]		=	$access;
		}

		return $cache[$userId][$streamId];
	}

	/**
	 * Outputs a JSON ajax response
	 *
	 * @param null|string $message
	 * @param null|string $type
	 * @param null|string $output
	 * @param null|string $target
	 */
	public static function ajaxResponse( $message = null, $type = 'html', $output = null, $target = null )
	{
		header( 'HTTP/1.0 200 OK' );
		header( 'Content-Type: application/json' );

		/** @noinspection PhpStatementHasEmptyBodyInspection */
		/** @noinspection LoopWhichDoesNotLoopInspection */
		/** @noinspection MissingOrEmptyGroupStatementInspection */
		while ( @ob_end_clean() ) {}

		if ( $message ) {
			switch ( $type ) {
				case 'notice':
					$message		=	'<div class="m-2 d-flex streamItemNotice">'
									.		'<div class="flex-grow-1 streamItemNoticeMessage">' . $message . '</div>'
									.		'<a href="javascript:void(0);" class="text-muted streamItemNoticeClose streamItemActionResponsesClose"><span class="fa fa-times streamIconClose"></span></a>'
									.	'</div>';
					break;
				case 'error':
				case 'warning':
				case 'info':
				case 'success':
					$message		=	'<div class="d-flex m-0 p-2 alert alert-' . htmlspecialchars( ( $type === 'error' ? 'danger' : $type ) ) . ' streamItemActionResponse streamItemAlert">'
									.		'<div class="flex-grow-1 streamItemAlertMessage">' . $message . '</div>'
									.		'<a href="javascript:void(0);" class="text-muted streamItemAlertClose streamItemActionResponseClose"><span class="fa fa-times streamIconClose"></span></a>'
									.	'</div>';

					if ( ! $output ) {
						$output		=	'prepend';
					}

					if ( ! $target ) {
						$target		=	'container';
					}
					break;
			}
		}

		if ( $type === 'raw' ) {
			echo json_encode( $message );
		} else {
			echo json_encode( array( 'message' => $message, 'type' => $type, 'output' => $output, 'target' => $target ) );
		}

		exit();
	}

	/**
	 * @param null|string $template
	 * @param null|string $file
	 * @param bool|array  $headers
	 * @return null|string
	 */
	public static function getTemplate( $template = null, $file = null, $headers = array( 'template', 'override', 'twemoji' ) )
	{
		global $_CB_framework, $_PLUGINS;

		if ( $headers === true ) {
			$headers					=	array( 'template', 'override', 'twemoji' );
		}

		$plugin							=	$_PLUGINS->getLoadedPlugin( 'user', 'cbactivity' );

		if ( ! $plugin ) {
			return null;
		}

		static $defaultTemplate			=	null;

		if ( $defaultTemplate === null ) {
			$defaultTemplate			=	self::getGlobalParams()->getString( 'general_template', 'default' );
		}

		if ( ( $template === '' ) || ( $template === null ) || ( $template === '-1' ) ) {
			$template					=	$defaultTemplate;
		}

		if ( ! $template ) {
			$template					=	'default';
		}

		$livePath						=	$_PLUGINS->getPluginLivePath( $plugin );
		$absPath						=	$_PLUGINS->getPluginPath( $plugin );

		if ( $headers !== false ) {
			static $loaded				=	array();

			$loaded[$template]			=	array();

			// Global CSS File:
			if ( in_array( 'template', $headers, true ) && ( ! in_array( 'template', $loaded[$template], true ) ) ) {
				$global					=	'/templates/' . $template . '/template.css';

				if ( ! file_exists( $absPath . $global ) ) {
					$global				=	'/templates/default/template.css';
				}

				if ( file_exists( $absPath . $global ) ) {
					$_CB_framework->document->addHeadStyleSheet( $livePath . $global );
				}

				$loaded[$template][]	=	'template';
			}

			// File or Custom CSS/JS Headers:
			foreach ( $headers as $header ) {
				if ( in_array( $header, $loaded[$template], true ) || in_array( $header, array( 'template', 'override' ), true ) ) {
					continue;
				}

				$header					=	preg_replace( '/[^-a-zA-Z0-9_]/', '', $header );

				if ( ! $header ) {
					continue;
				}

				$css					=	'/templates/' . $template . '/' . $header . '.css';
				$js						=	'/templates/' . $template . '/' . $header . '.js';

				if ( ! file_exists( $absPath . $css ) ) {
					$css				=	'/templates/default/' . $header . '.css';
				}

				if ( file_exists( $absPath . $css ) ) {
					$_CB_framework->document->addHeadStyleSheet( $livePath . $css );
				}

				if ( ! file_exists( $absPath . $js ) ) {
					$js					=	'/templates/default/' . $header . '.js';
				}

				if ( file_exists( $absPath . $js ) ) {
					$_CB_framework->document->addHeadScriptUrl( $livePath . $js );
				}

				$loaded[$template][]	=	$header;
			}

			// Override CSS File:
			if ( in_array( 'override', $headers, true ) && ( ! in_array( 'override', $loaded[$template], true ) ) ) {
				$override				=	'/templates/' . $template . '/override.css';

				if ( file_exists( $absPath . $override ) ) {
					$_CB_framework->document->addHeadStyleSheet( $livePath . $override );
				}

				$loaded[$template][]	=	'override';
			}
		}

		$path							=	array();

		foreach ( explode( '/', $file ) as $part ) {
			$path[]						=	preg_replace( '/[^-a-zA-Z0-9_]/', '', $part );
		}

		$file							=	implode( '/', $path );
		$return							=	null;

		if ( $file ) {
			if ( $headers !== false ) {
				$headers[]				=	$file;
			}

			$php						=	$absPath . '/templates/' . $template . '/' . $file . '.php';

			if ( ! file_exists( $php ) ) {
				$php					=	$absPath . '/templates/default/' . $file . '.php';
			}

			if ( file_exists( $php ) ) {
				$return					=	$php;
			}
		}

		return $return;
	}

	/**
	 * Reloads page headers for ajax responses
	 *
	 * @return null|string
	 */
	public static function reloadHeaders()
	{
		global $_CB_framework;

		if ( Application::Input()->getString( 'format' ) !== 'raw' ) {
			return null;
		}

		$_CB_framework->getAllJsPageCodes();

		// Reset meta headers as they can't be used inline anyway:
		$_CB_framework->document->_head['metaTags']		=	array();

		// Reset custom headers as we can't guarantee they can output inline:
		$_CB_framework->document->_head['custom']		=	array();

		// Remove all non-jQuery scripts as they'll likely just cause errors due to redeclaration:
		foreach( $_CB_framework->document->_head['scriptsUrl'] as $url => $script ) {
			if ( ( strpos( $url, 'jquery.' ) === false ) || ( strpos( $url, 'migrate' ) !== false ) ) {
				unset( $_CB_framework->document->_head['scriptsUrl'][$url] );
			}
		}

		$header				=	$_CB_framework->document->outputToHead();

		if ( ! $header ) {
			return null;
		}

		$return				=	'<div class="streamItemHeaders hidden" style="position: absolute; display: none; height: 0; width: 0; z-index: -999;">';

		if ( ( strpos( $header, 'cbjQuery' ) !== false ) && ( strpos( $header, 'window.$ = cbjQuery;' ) === false ) ) {
			// The jQuery pointer is missing so lets add it so jQuery rebinds correctly:
			$return			.=		'<script type="text/javascript">window.$ = cbjQuery; window.jQuery = cbjQuery;</script>';
		}

		$return				.=		$header
							.	'</div>';

		return $return;
	}

	/**
	 * Returns file size formatted from bytes
	 *
	 * @param int $bytes
	 * @param int $decimals
	 * @return string
	 */
	public static function getFormattedFileSize( $bytes, $decimals = 0 )
	{
		if ( $bytes >= 1099511627776 ) {
			return CBTxt::T( 'FILESIZE_FORMATTED_TB', '%%COUNT%% TB|%%COUNT%% TBs', array( '%%COUNT%%' => (float) number_format( $bytes / 1099511627776, $decimals, '.', '' ) ) );
		}

		if ( $bytes >= 1073741824 ) {
			return CBTxt::T( 'FILESIZE_FORMATTED_GB', '%%COUNT%% GB|%%COUNT%% GBs', array( '%%COUNT%%' => (float) number_format( $bytes / 1073741824, $decimals, '.', '' ) ) );
		}

		if ( $bytes >= 1048576 ) {
			return CBTxt::T( 'FILESIZE_FORMATTED_MB', '%%COUNT%% MB|%%COUNT%% MBs', array( '%%COUNT%%' => (float) number_format( $bytes / 1048576, $decimals, '.', '' ) ) );
		}

		if ( $bytes >= 1024 ) {
			return CBTxt::T( 'FILESIZE_FORMATTED_KB', '%%COUNT%% KB|%%COUNT%% KBs', array( '%%COUNT%%' => (float) number_format( $bytes / 1024, $decimals, '.', '' ) ) );
		}

		return CBTxt::T( 'FILESIZE_FORMATTED_B', '%%COUNT%% B|%%COUNT%% Bs', array( '%%COUNT%%' => (float) number_format( $bytes, $decimals, '.', '' ) ) );
	}

	/**
	 * Formats total using comma separation and reduce thousands
	 *
	 * @param int $total
	 * @return string
	 */
	public static function getFormattedTotal( $total )
	{
		if ( $total >= 1000 ) {
			$total	=	round( $total / 1000, 1 );

			return CBTxt::T( 'TOTAL_FORMATTED_SHORT', '[total]K', array( '[total]' => number_format( $total, ( floor( $total ) !== $total ? 1 : 0 ) ) ) );
		}

		return number_format( $total );
	}

	/**
	 * Helper function for preparing post attachments
	 *
	 * @param ParamsInterface $attachments
	 * @return Registry
	 */
	public static function prepareAttachments( $attachments )
	{
		$links									=	new Registry();

		foreach ( $attachments as $i => $link ) {
			/** @var ParamsInterface $link */
			$type								=	$link->getString( 'type' );

			if ( ! $type ) {
				$link->set( 'type', 'url' );
			}

			if ( $type !== 'custom' ) {
				$url							=	$link->getString( 'url' );

				if ( ! $url ) {
					continue;
				}

				if ( substr( $link['url'], 0, 3 ) === 'www' ) {
					$url						=	'http://' . $url;

					$link->set( 'url', $url );
				}

				if ( Application::Router()->isInternal( $url ) ) {
					$link->set( 'internal', true );
				} else {
					$link->set( 'internal', false );
				}
			}

			$media								=	$link->subTree( 'media' );

			switch ( $type ) {
				case 'custom':
					if ( ! $media->getRaw( 'custom' ) ) {
						continue 2;
					}
					break;
				case 'video':
				case 'audio':
					if ( ( ! $media->getString( 'url' ) ) || ( ! $media->getString( 'mimetype' ) ) ) {
						continue 2;
					}
					break;
				case 'image':
				case 'file':
					if ( ! $media->getString( 'url' ) ) {
						continue 2;
					}
					break;
				case 'url':
				default:
					if ( ( ! $media->getString( 'url' ) ) || ( ! $link->getBool( 'thumbnail', true ) ) ) {
						$link->set( 'media', false );
					}
					break;
			}

			if ( $type !== 'custom' ) {
				if ( $link->getRaw( 'media' ) !== false ) {
					$mediaUrl					=	$media->getString( 'url' );

					if ( substr( $mediaUrl, 0, 3 ) === 'www' ) {
						$mediaUrl				=	'http://' . $mediaUrl;

						$media->set( 'url', $mediaUrl );
					}

					if ( Application::Router()->isInternal( $mediaUrl ) ) {
						$media->set( 'internal', true );
					} else {
						$media->set( 'internal', false );
					}

					if ( in_array( $type, array( 'url', 'video', 'audio' ), true ) ) {
						/** @var ParamsInterface $thumbnails */
						$thumbnails				=	$link->subTree( 'thumbnails' );

						foreach ( $thumbnails as $t => $thumbnail ) {
							/** @var ParamsInterface $thumbnail */
							$thumbnailUrl		=	$thumbnail->getString( 'url' );

							if ( ! $thumbnailUrl ) {
								$thumbnails->unsetEntry( $t );
								continue;
							}

							if ( substr( $thumbnailUrl, 0, 3 ) === 'www' ) {
								$thumbnailUrl	=	'http://' . $thumbnailUrl;

								$thumbnail->set( 'url', $thumbnailUrl );
							}

							if ( Application::Router()->isInternal( $thumbnailUrl ) ) {
								$thumbnail->set( 'internal', true );
							} else {
								$thumbnail->set( 'internal', false );
							}
						}

						if ( $thumbnails ) {
							$link->set( 'thumbnails', $thumbnails->asArray() );
						} else {
							$link->set( 'thumbnails', false );
						}
					} else {
						$link->set( 'thumbnails', false );
					}

					$link->set( 'media', $media->asArray() );
				} else {
					$link->set( 'thumbnails', false );
				}
			} else {
				$link->set( 'internal', true );
				$link->set( 'thumbnails', false );
			}

			$links->set( $i, $link->asArray() );
		}

		return $links;
	}

	/**
	 * Returns internal clean up urls
	 *
	 * @return string
	 */
	public static function loadCleanUpURL()
	{
		global $_CB_framework;

		return '<a href="' . $_CB_framework->pluginClassUrl( 'cbactivity', true, array( 'action' => 'cleanup', 'token' => md5( $_CB_framework->getCfg( 'secret' ) ) ), 'raw', 0, true ) . '" target="_blank">' . CBTxt::T( 'Click to Process' ) . '</a>';
	}

	/**
	 * Returns an options array of available themes
	 *
	 * @param bool                   $raw
	 * @param null|UserTable|int     $viewer
	 * @param null|Activity|Comments $stream
	 * @return array|ThemeTable[]
	 */
	public static function loadThemeOptions( $raw = false, $viewer = null, $stream = null )
	{
		global $_CB_database;

		if ( Application::Application()->isClient( 'administrator' ) ) {
			$raw								=	false;
			$viewer								=	null;
			$stream								=	null;
		}

		if ( is_int( $viewer ) ) {
			$userId								=	$viewer;
		} elseif ( $viewer ) {
			$userId								=	$viewer->getInt( 'id', 0 );
		} else {
			$userId								=	Application::MyUser()->getUserId();
		}

		if ( $stream ) {
			$streamId							=	$stream->id();
		} else {
			$streamId							=	0;
		}

		static $cache							=	array();
		static $themes							=	null;

		if ( $themes === null ) {
			$query								=	'SELECT *'
												.	"\n FROM " . $_CB_database->NameQuote( '#__comprofiler_plugin_activity_themes' )
												.	"\n WHERE " . $_CB_database->NameQuote( 'published' ) . " = 1"
												.	"\n ORDER BY " . $_CB_database->NameQuote( 'ordering' );
			$_CB_database->setQuery( $query );
			$themes								=	$_CB_database->loadObjectList( 'id', '\CB\Plugin\Activity\Table\ThemeTable', array( $_CB_database ) );
		}

		if ( ! $themes ) {
			return array();
		}

		if ( ! isset( $cache[$streamId][$userId][$raw] ) ) {
			$include							=	array();
			$exclude							=	array();

			if ( $stream ) {
				$streamInclude					=	$stream->getString( 'themes_include', '0' );

				if ( $streamInclude ) {
					if ( strpos( $streamInclude, '|*|' ) !== false ) {
						$include				=	cbToArrayOfInt( explode( '|*|', $streamInclude ) );
					} else {
						$include				=	cbToArrayOfInt( explode( ',', $streamInclude ) );
					}
				}

				$streamExclude					=	$stream->getString( 'themes_exclude', '0' );

				if ( $streamExclude ) {
					if ( strpos( $streamExclude, '|*|' ) !== false ) {
						$exclude				=	cbToArrayOfInt( explode( '|*|', $streamExclude ) );
					} else {
						$exclude				=	cbToArrayOfInt( explode( ',', $streamExclude ) );
					}
				}
			}

			$options							=	array();

			/** @var ThemeTable[] $themes */
			foreach ( $themes as $id => $theme ) {
				if ( ( $exclude && in_array( $id, $exclude, true ) )
					 || ( $include && ( ! in_array( $id, $include, true ) ) )
					 || ( ! in_array( $theme->getInt( 'access', 1 ), Application::User( $userId )->getAuthorisedViewLevels(), true ) )
				) {
					continue;
				}

				if ( $raw ) {
					$options[$id]				=	$theme;
				} else {
					$attributes					=	' data-cbactivity-option-icon="' . htmlspecialchars( $theme->preview() ) . '"';

					if ( $theme->getString( 'background' ) ) {
						$attributes				.=	' data-cbactivity-option-background="' . htmlspecialchars( $theme->background() ) . '"';
					}

					if ( $theme->getString( 'class' ) ) {
						$attributes				.=	' data-cbactivity-option-class="' . htmlspecialchars( $theme->getString( 'class' ) ) . '"';
					}

					$option						=	\moscomprofilerHTML::makeOption( $id, '&nbsp;', 'value', 'text', null, null, $attributes );
					$option->text				=	null; // We want absolutely no label in this case since we only want the preview so lets null it

					$options[]					=	$option;
				}
			}

			if ( $options && ( ! $raw ) && ( ! Application::Application()->isClient( 'administrator' ) ) ) {
				array_unshift( $options, \moscomprofilerHTML::makeOption( 0, '&nbsp;', 'value', 'text', null, null, 'data-cbactivity-option-icon="' . htmlspecialchars( '<span class="fa fa-times"></span>' ) . '"' ) );
			}

			$cache[$streamId][$userId][$raw]	=	$options;
		}

		return $cache[$streamId][$userId][$raw];
	}

	/**
	 * Returns an options array of available actions
	 *
	 * @param bool                   $raw
	 * @param null|UserTable|int     $viewer
	 * @param null|Activity|Comments $stream
	 * @return array|ActionTable[]
	 */
	public static function loadActionOptions( $raw = false, $viewer = null, $stream = null )
	{
		global $_CB_database;

		if ( Application::Application()->isClient( 'administrator' ) ) {
			$raw								=	false;
			$viewer								=	null;
			$stream								=	null;
		}

		if ( is_int( $viewer ) ) {
			$userId								=	$viewer;
		} elseif ( $viewer ) {
			$userId								=	$viewer->getInt( 'id', 0 );
		} else {
			$userId								=	Application::MyUser()->getUserId();
		}

		if ( $stream ) {
			$streamId							=	$stream->id();
		} else {
			$streamId							=	0;
		}

		static $cache							=	array();
		static $actions							=	null;

		if ( $actions === null ) {
			$query								=	'SELECT *'
												.	"\n FROM " . $_CB_database->NameQuote( '#__comprofiler_plugin_activity_actions' )
												.	"\n WHERE " . $_CB_database->NameQuote( 'published' ) . " = 1"
												.	"\n ORDER BY " . $_CB_database->NameQuote( 'ordering' );
			$_CB_database->setQuery( $query );
			$actions							=	$_CB_database->loadObjectList( 'id', '\CB\Plugin\Activity\Table\ActionTable', array( $_CB_database ) );
		}

		if ( ! $actions ) {
			return array();
		}

		if ( ! isset( $cache[$streamId][$userId][$raw] ) ) {
			$include							=	array();
			$exclude							=	array();

			if ( $stream ) {
				$streamInclude					=	$stream->getString( 'actions_include', '0' );

				if ( $streamInclude ) {
					if ( strpos( $streamInclude, '|*|' ) !== false ) {
						$include				=	cbToArrayOfInt( explode( '|*|', $streamInclude ) );
					} else {
						$include				=	cbToArrayOfInt( explode( ',', $streamInclude ) );
					}
				}

				$streamExclude					=	$stream->getString( 'actions_exclude', '0' );

				if ( $streamExclude ) {
					if ( strpos( $streamExclude, '|*|' ) !== false ) {
						$exclude				=	cbToArrayOfInt( explode( '|*|', $streamExclude ) );
					} else {
						$exclude				=	cbToArrayOfInt( explode( ',', $streamExclude ) );
					}
				}
			}

			$options							=	array();
			$emotes								=	self::loadEmoteOptions( false, true, $viewer );

			/** @var ActionTable[] $actions */
			foreach ( $actions as $id => $action ) {
				if ( ( $exclude && in_array( $id, $exclude, true ) )
					 || ( $include && ( ! in_array( $id, $include, true ) ) )
					 || ( ! in_array( $action->getInt( 'access', 1 ), Application::User( $userId )->getAuthorisedViewLevels(), true ) )
				) {
					continue;
				}

				if ( $raw ) {
					$options[$id]				=	$action;
				} else {
					$attributes					=	null;

					if ( $action->icon() ) {
						$attributes				.=	' data-cbactivity-option-icon="' . htmlspecialchars( $action->icon() ) . '"';
					}

					if ( $action->getString( 'description' ) ) {
						$attributes				.=	' data-cbactivity-toggle-placeholder="' . htmlspecialchars( CBTxt::T( $action->getString( 'description' ) ) ) . '"';
					}

					$suggestions				=	array();

					foreach ( $action->params()->getRaw( 'suggestions', array() ) as $suggestion ) {
						if ( ( ! isset( $suggestion['suggestion'] ) ) || ( ! $suggestion['suggestion'] ) ) {
							continue;
						}

						$message				=	CBuser::getInstance( $userId, false )->replaceUserVars( $suggestion['suggestion'], true, false );
						$emote					=	( isset( $suggestion['emote'] ) ? $suggestion['emote'] : null );
						$icon					=	null;

						if ( $emote && isset( $emotes[$emote] ) ) {
							$icon				=	$emotes[$emote]->icon();
						}

						$suggestions[]			=	array(	'value'	=>	$message,
															'label'	=>	( $icon ? $icon : null ) . $message,
															'emote'	=>	( $icon ? $emote : null )
														);
					}

					if ( $suggestions ) {
						$attributes				.=	' data-cbactivity-toggle-suggestions="' . htmlspecialchars( json_encode( $suggestions ) ) . '"';
					}

					$options[]					=	\moscomprofilerHTML::makeOption( $id, CBTxt::T( $action->getString( 'value' ) ), 'value', 'text', null, null, $attributes );
				}
			}

			if ( $options && ( ! $raw ) && ( ! Application::Application()->isClient( 'administrator' ) ) ) {
				array_unshift( $options, \moscomprofilerHTML::makeOption( 0, '&nbsp;', 'value', 'text', null, null, 'data-cbactivity-option-icon="' . htmlspecialchars( '<span class="fa fa-times"></span>' ) . '"' ) );
			}

			$cache[$streamId][$userId][$raw]	=	$options;
		}

		return $cache[$streamId][$userId][$raw];
	}

	/**
	 * Returns an options array of available locations
	 *
	 * @param bool                   $raw
	 * @param null|UserTable|int     $viewer
	 * @param null|Activity|Comments $stream
	 * @return array|ActionTable[]
	 */
	public static function loadLocationOptions( $raw = false, $viewer = null, $stream = null )
	{
		global $_CB_database;

		if ( Application::Application()->isClient( 'administrator' ) ) {
			$raw								=	false;
			$viewer								=	null;
			$stream								=	null;
		}

		if ( is_int( $viewer ) ) {
			$userId								=	$viewer;
		} elseif ( $viewer ) {
			$userId								=	$viewer->getInt( 'id', 0 );
		} else {
			$userId								=	Application::MyUser()->getUserId();
		}

		if ( $stream ) {
			$streamId							=	$stream->id();
		} else {
			$streamId							=	0;
		}

		static $cache							=	array();
		static $locations						=	null;

		if ( $locations === null ) {
			$query								=	'SELECT *'
												.	"\n FROM " . $_CB_database->NameQuote( '#__comprofiler_plugin_activity_locations' )
												.	"\n WHERE " . $_CB_database->NameQuote( 'published' ) . " = 1"
												.	"\n ORDER BY " . $_CB_database->NameQuote( 'ordering' );
			$_CB_database->setQuery( $query );
			$locations							=	$_CB_database->loadObjectList( 'id', '\CB\Plugin\Activity\Table\LocationTable', array( $_CB_database ) );
		}

		if ( ! $locations ) {
			return array();
		}

		if ( ! isset( $cache[$streamId][$userId][$raw] ) ) {
			$include							=	array();
			$exclude							=	array();

			if ( $stream ) {
				$streamInclude					=	$stream->getString( 'locations_include', '0' );

				if ( $streamInclude ) {
					if ( strpos( $streamInclude, '|*|' ) !== false ) {
						$include				=	cbToArrayOfInt( explode( '|*|', $streamInclude ) );
					} else {
						$include				=	cbToArrayOfInt( explode( ',', $streamInclude ) );
					}
				}

				$streamExclude					=	$stream->getString( 'locations_exclude', '0' );

				if ( $streamExclude ) {
					if ( strpos( $streamExclude, '|*|' ) !== false ) {
						$exclude				=	cbToArrayOfInt( explode( '|*|', $streamExclude ) );
					} else {
						$exclude				=	cbToArrayOfInt( explode( ',', $streamExclude ) );
					}
				}
			}

			$options							=	array();

			/** @var LocationTable[] $locations */
			foreach ( $locations as $id => $location ) {
				if ( ( $exclude && in_array( $id, $exclude, true ) )
					 || ( $include && ( ! in_array( $id, $include, true ) ) )
					 || ( ! in_array( $location->getInt( 'access', 1 ), Application::User( $userId )->getAuthorisedViewLevels(), true ) )
				) {
					continue;
				}

				if ( $raw ) {
					$options[$id]				=	$location;
				} else {
					$options[]					=	\moscomprofilerHTML::makeOption( $id, CBTxt::T( $location->getString( 'value' ) ) );
				}
			}

			if ( $options && ( ! $raw ) && ( ! Application::Application()->isClient( 'administrator' ) ) ) {
				array_unshift( $options, \moscomprofilerHTML::makeOption( 0, '&nbsp;', 'value', 'text', null, null, 'data-cbactivity-option-icon="' . htmlspecialchars( '<span class="fa fa-times"></span>' ) . '"' ) );
			}

			$cache[$streamId][$userId][$raw]	=	$options;
		}

		return $cache[$streamId][$userId][$raw];
	}

	/**
	 * Returns an options array of available emotes
	 *
	 * @param bool                   $substitutions
	 * @param bool                   $raw
	 * @param null|UserTable|int     $viewer
	 * @return array|EmoteTable[]
	 */
	public static function loadEmoteOptions( $substitutions = false, $raw = false, $viewer = null )
	{
		global $_CB_database;

		if ( Application::Application()->isClient( 'administrator' ) ) {
			$substitutions							=	false;
			$raw									=	false;
			$viewer									=	null;
			$stream									=	null;
		}

		if ( is_int( $viewer ) ) {
			$userId									=	$viewer;
		} elseif ( $viewer ) {
			$userId									=	$viewer->getInt( 'id', 0 );
		} else {
			$userId									=	Application::MyUser()->getUserId();
		}

		static $cache								=	array();
		static $emotes								=	null;

		if ( $emotes === null ) {
			$query									=	'SELECT *'
													.	"\n FROM " . $_CB_database->NameQuote( '#__comprofiler_plugin_activity_emotes' )
													.	"\n WHERE " . $_CB_database->NameQuote( 'published' ) . " = 1"
													.	"\n ORDER BY " . $_CB_database->NameQuote( 'ordering' );
			$_CB_database->setQuery( $query );
			$emotes									=	$_CB_database->loadObjectList( 'id', '\CB\Plugin\Activity\Table\EmoteTable', array( $_CB_database ) );
		}

		if ( ! $emotes ) {
			return array();
		}

		if ( ! isset( $cache[$substitutions][$userId][$raw] ) ) {
			$options								=	array();

			/** @var EmoteTable[] $emotes */
			foreach ( $emotes as $id => $emote ) {
				if ( ! in_array( $emote->getInt( 'access', 1 ), Application::User( $userId )->getAuthorisedViewLevels(), true ) ) {
					continue;
				}

				$key								=	':' . $emote->getString( 'value' ) . ':';

				if ( $raw ) {
					$options[$id]					=	$emote;
				} elseif ( $substitutions === true ) {
					$options[$key]					=	$emote->icon();

					foreach ( explode( ',', $emote->params()->getString( 'substitutions' ) ) as $substitution ) {
						if ( ! $substitution ) {
							continue;
						}

						$options[$substitution]		=	$emote->icon();
					}
				} else {
					$options[]						=	\moscomprofilerHTML::makeOption( $id, '&nbsp;', 'value', 'text', null, null, ' data-cbactivity-option-icon="' . htmlspecialchars( $emote->icon() ) . '" data-cbactivity-option-insert="' . htmlspecialchars( $key ) . '"' );
				}
			}

			if ( $options && ( $substitutions !== true ) && ( ! $raw ) && ( ! Application::Application()->isClient( 'administrator' ) ) ) {
				array_unshift( $options, \moscomprofilerHTML::makeOption( 0, '&nbsp;', 'value', 'text', null, null, 'data-cbactivity-option-icon="' . htmlspecialchars( '<span class="streamIconEmote streamIconEmoteBlank fa fa-smile-o fa-lg"></span>' ) . '"' ) );
			}

			$cache[$substitutions][$userId][$raw]	=	$options;
		}

		return $cache[$substitutions][$userId][$raw];
	}

	/**
	 * Returns an options array of available like types
	 *
	 * @param bool               $raw
	 * @param null|UserTable|int $viewer
	 * @param null|Likes         $stream
	 * @return array|LikeTypeTable[]
	 */
	public static function loadLikeOptions( $raw = false, $viewer = null, $stream = null )
	{
		global $_CB_database;

		if ( Application::Application()->isClient( 'administrator' ) ) {
			$raw								=	false;
			$viewer								=	null;
			$stream								=	null;
		}

		if ( is_int( $viewer ) ) {
			$userId								=	$viewer;
		} elseif ( $viewer ) {
			$userId								=	$viewer->getInt( 'id', 0 );
		} else {
			$userId								=	Application::MyUser()->getUserId();
		}

		if ( $stream ) {
			$streamId							=	$stream->id();
		} else {
			$streamId							=	0;
		}

		static $cache							=	array();
		static $types							=	null;

		if ( $types === null ) {
			$query								=	'SELECT *'
												.	"\n FROM " . $_CB_database->NameQuote( '#__comprofiler_plugin_activity_like_types' )
												.	"\n WHERE " . $_CB_database->NameQuote( 'published' ) . " = 1"
												.	"\n ORDER BY " . $_CB_database->NameQuote( 'ordering' );
			$_CB_database->setQuery( $query );
			$types								=	$_CB_database->loadObjectList( 'id', '\CB\Plugin\Activity\Table\LikeTypeTable', array( $_CB_database ) );
		}

		if ( ! $types ) {
			return array();
		}

		if ( ! isset( $cache[$streamId][$userId][$raw] ) ) {
			$include							=	array();
			$exclude							=	array();

			if ( $stream ) {
				$streamInclude					=	$stream->getString( 'include' );

				if ( $streamInclude ) {
					if ( strpos( $streamInclude, '|*|' ) !== false ) {
						$include				=	cbToArrayOfInt( explode( '|*|', $streamInclude ) );
					} else {
						$include				=	cbToArrayOfInt( explode( ',', $streamInclude ) );
					}
				}

				$streamExclude					=	$stream->getString( 'exclude' );

				if ( $streamExclude ) {
					if ( strpos( $streamExclude, '|*|' ) !== false ) {
						$exclude				=	cbToArrayOfInt( explode( '|*|', $streamExclude ) );
					} else {
						$exclude				=	cbToArrayOfInt( explode( ',', $streamExclude ) );
					}
				}
			}

			$options							=	array();

			/** @var LikeTypeTable[] $types */
			foreach ( $types as $id => $type ) {
				if ( ( $exclude && in_array( $id, $exclude, true ) )
					 || ( $include && ( ! in_array( $id, $include, true ) ) )
					 || ( ! in_array( $type->getInt( 'access', 1 ), Application::User( $userId )->getAuthorisedViewLevels(), true ) )
				) {
					continue;
				}

				if ( $raw ) {
					$options[$id]				=	$type;
				} else {
					$options[]					=	\moscomprofilerHTML::makeOption( $id, CBTxt::T( $type->getString( 'value' ) ), 'value', 'text', null, null, ' data-cbactivity-option-icon="' . htmlspecialchars( $type->icon() ) . '"' );
				}
			}

			$cache[$streamId][$userId][$raw]	=	$options;
		}

		return $cache[$streamId][$userId][$raw];
	}

	/**
	 * Returns an options array of available template images
	 *
	 * @return array
	 */
	public static function loadImageOptions()
	{
		global $_CB_framework;

		static $cache			=	null;

		if ( $cache === null ) {
			$template		=	self::getGlobalParams()->getString( 'general_template', 'default' );

			if ( ! $template ) {
				$template	=	'default';
			}

			$path			=	$_CB_framework->getCfg( 'absolute_path' ) . '/components/com_comprofiler/plugin/user/plug_cbactivity/templates/' . $template . '/images';

			if ( ( $template !== 'default' ) && ( ! is_dir( $path ) ) ) {
				$path		=	$_CB_framework->getCfg( 'absolute_path' ) . '/components/com_comprofiler/plugin/user/plug_cbactivity/templates/default/images';
			}

			$cache			=	array();

			foreach ( cbReadDirectory( $path, '\.png$|\.gif$|\.jpg$|\.bmp$|\.ico$|\.svg$', false ) as $image ) {
				$cache[]	=	\moscomprofilerHTML::makeOption( $image, $image );
			}
		}

		return $cache;
	}

	/**
	 * Renders stream unique select list
	 *
	 * @param array             $options
	 * @param string            $name
	 * @param null|string       $attributes
	 * @param StreamInterface   $stream
	 * @param null|string|array $selected
	 * @return string
	 */
	public static function renderSelectList( $options, $name, $attributes, $stream, $selected = null )
	{
		$id				=	\moscomprofilerHTML::htmlId( $name );
		$uniqueId		=	md5( $stream->id() . '_' . $id . '_' . mt_rand() );

		foreach ( $options as $k => $v ) {
			$v->id		=	$uniqueId . '__cbf' . $k;
		}

		$selectList		=	\moscomprofilerHTML::selectList( $options, $name, $attributes, 'value', 'text', $selected, 0, false, false, false );

		return str_replace( 'id="' . $id . '"', 'id="' . $uniqueId . '"', $selectList );
	}

	/**    
	 * Returns the formatted owner name of an object
	 *
	 * @param ActivityTable|CommentTable|NotificationTable $row
	 * @return string
	 */
	public static function getOwnerName( $row )
	{
		if ( $row->getBool( 'system', false ) ) {
			$cbUser							=	\CBuser::getMyInstance();
			$systemUser						=	self::getGlobalParams()->getInt( 'system_user', 0 );
			$name							=	CBTxt::T( self::getGlobalParams()->getHtml( 'system_name', 'System' ) );

			if ( ! $name ) {
				if ( $systemUser ) {
					$name					=	\CBuser::getInstance( $systemUser, false )->getField( 'formatname', null, 'html', 'none', 'profile', 0, true );
				} else {
					$name					=	$cbUser->replaceUserVars( CBTxt::T( 'System' ), true, false, null, false );
				}
			} else {
				$name						=	$cbUser->replaceUserVars( $name, true, false, null, false );
			}

			$link							=	$cbUser->replaceUserVars( self::getGlobalParams()->getString( 'system_url' ), false, false, null, false );

			if ( $link ) {
				$name						=	'<strong><a href="' . htmlspecialchars( $link ) . '">' . $name . '</a></strong>';
			} else {
				$name						=	'<strong>' . $name . '</strong>';
			}
		} else {
			$cbUser							=	\CBuser::getInstance( $row->getInt( 'user_id', 0 ), false );
			$name							=	'<strong>' . $cbUser->getField( 'formatname', null, 'html', 'none', 'list', 0, true ) . '</strong>';

			$names							=	$row->params()->getRaw( 'overrides.names', array() );

			if ( $names ) {
				$totalNames					=	( count( $names ) + 1 );
				$namesList					=	array( $name );

				foreach ( $names as $k => $nameId ) {
					if ( is_numeric( $nameId ) ) {
						$newUser			=	\CBuser::getInstance( $nameId, false );

						if ( count( $namesList ) > 1 ) {
							$newName		=	$newUser->getField( 'formatname', null, 'html', 'none', 'profile', 0, true );
						} else {
							$newName		=	$newUser->getField( 'formatname', null, 'html', 'none', 'list', 0, true );
						}
					} else {
						$newName			=	htmlspecialchars( $nameId );
					}

					if ( ! $newName ) {
						--$totalNames;
						continue;
					}

					if ( count( $namesList ) > 1 ) {
						$namesList[]		=	$newName;

						if ( count( $namesList ) > 15 ) {
							$namesList[]	=	CBTxt::T( 'more...' );
							break;
						}
					} else {
						$namesList[]		=	'<strong>' . $newName . '</strong>';
					}
				}

				if ( $totalNames > 2 ) {
					$nameOne				=	array_shift( $namesList );
					$nameTwo				=	array_shift( $namesList );

					$more					=	cbTooltip( null, implode( '<br />', $namesList ), null, 'auto', null, CBTxt::T( 'NAMES_MORE', '[names] other|[names] others|%%COUNT%%', array( '%%COUNT%%' => ( $totalNames - 2 ), '[names]' => self::getFormattedTotal( ( $totalNames - 2 ) ) ) ), 'javascript: void(0);', 'data-hascbtooltip="true" data-cbtooltip-position-my="bottom center" data-cbtooltip-position-at="top center" data-cbtooltip-classes="qtip-simple"' );

					$name					=	CBTxt::T( 'NAMES_MORE_THAN_TWO', '[name_1], [name_2], and [more]', array( '[name_1]' => $nameOne, '[name_2]' => $nameTwo, '[more]' => $more ) );
				} elseif ( $totalNames > 1 ) {
					$name					=	CBTxt::T( 'NAMES_TWO', '[name_1] and [name_2]', array( '[name_1]' => $namesList[0], '[name_2]' => $namesList[1] ) );
				}
			}
		}

		return $name;
	}

	/**
	 * Returns the formatted owner avatar of an object
	 *
	 * @param ActivityTable|CommentTable|NotificationTable $row
	 * @return string
	 */
	public static function getOwnerAvatar( $row )
	{
		global $_CB_framework;

		if ( $row->getBool( 'system', false ) ) {
			$systemUser				=	self::getGlobalParams()->getInt( 'system_user', 0 );
			$avatar					=	CBTxt::T( self::getGlobalParams()->getString( 'system_avatar' ) );

			if ( $avatar ) {
				if ( $avatar[0] === '/' ) {
					$avatar			=	$_CB_framework->getCfg( 'live_site' ) . $avatar;
				}

				switch ( self::getGlobalParams()->getString( 'system_avatar_style', 'roundedbordered' ) ) {
					case 'rounded':
						$style		=	' rounded';
						break;
					case 'roundedbordered':
						$style		=	' img-thumbnail';
						break;
					case 'circle':
						$style		=	' rounded-circle';
						break;
					case 'circlebordered':
						$style		=	' img-thumbnail rounded-circle';
						break;
					default:
						$style		=	null;
						break;
				}

				$avatar				=	'<img src="' . htmlspecialchars( $avatar ) . '" class="cbImgPict cbThumbPict' . htmlspecialchars( $style ) . '" />';
			} else {
				$avatar				=	\CBuser::getInstance( $systemUser, false )->getField( 'avatar', null, 'html', 'none', 'list', 0, true, array( '_allowProfileLink' => false ) );
			}

			$link					=	\CBuser::getMyInstance()->replaceUserVars( self::getGlobalParams()->getString( 'system_url' ), false, false, null, false );

			if ( $link ) {
				$avatar				=	'<a href="' . htmlspecialchars( $link ) . '">' . $avatar . '</a>';
			}
		} else {
			$avatar					=	\CBuser::getInstance( $row->getInt( 'user_id', 0 ), false )->getField( 'avatar', null, 'html', 'none', 'list', 0, true );
		}

		return $avatar;
	}

	/**
	 * @return bool
	 */
	public static function isHTTPS()
	{
		global $_CB_framework;

		static $cache	=	null;

		if ( $cache === null ) {
			$cache		=	false;

			if ( isset( $_SERVER['HTTPS'] ) && ( ! empty( $_SERVER['HTTPS'] ) ) && ( $_SERVER['HTTPS'] !== 'off' ) ) {
				$cache	=	true;
			} elseif ( parse_url( $_CB_framework->getCfg( 'live_site' ), PHP_URL_HOST ) === 'localhost' ) {
				$cache	=	true;
			}
		}

		return $cache;
	}

	/**
	 * @return bool
	 */
	public static function checkDOMDocument()
	{
		if ( ! class_exists( 'DOMDocument' ) ) {
			return false;
		}

		if ( ! class_exists( 'DOMXPath' ) ) {
			return false;
		}

		return true;
	}

	/**
	 * @return false|\stdClass
	 */
	public static function getCBBlogs()
	{
		global $_PLUGINS;

		static $plugin		=	null;

		if ( $plugin === null ) {
			if ( ! $_PLUGINS->getLoadedPlugin( 'user', 'cbblogs' ) ) {
				$plugin		=	false;
			} else {
				$model		=	\cbblogsClass::getModel();

				if ( ! $model->file ) {
					$plugin	=	false;
				} else {
					$plugin	=	$model;
				}
			}
		}

		return $plugin;
	}

	/**
	 * @return false|\stdClass
	 */
	public static function getCBArticles()
	{
		global $_PLUGINS;

		static $plugin		=	null;

		if ( $plugin === null ) {
			if ( ! $_PLUGINS->getLoadedPlugin( 'user', 'cbarticles' ) ) {
				$plugin		=	false;
			} else {
				$model		=	\cbarticlesClass::getModel();

				if ( ! $model->file ) {
					$plugin	=	false;
				} else {
					$plugin	=	$model;
				}
			}
		}

		return $plugin;
	}

	/**
	 * @param int $id
	 * @return Registry
	 */
	public static function getMenuItem( $id )
	{
		static $cache		=	array();

		if ( isset( $cache[$id] ) ) {
			return $cache[$id];
		}

		$menuItem			=	\JFactory::getApplication()->getMenu()->getItem( $id );
		$menu				=	new Registry( $menuItem );
		$params				=	null;

		if ( $menuItem ) {
			if ( is_callable( array( $menuItem, 'getParams' ) ) ) {
				$params		=	$menuItem->getParams();
			} else {
				$params		=	$menuItem->params;
			}

			if ( $params instanceof \Joomla\Registry\Registry ) {
				$params		=	$params->toArray();
			}
		}

		$cache[$id]			=	$menu->setNamespaceRegistry( 'params', new Registry( $params ) );

		return $cache[$id];
	}
}