<?php
/**
* Community Builder (TM)
* @version $Id: $
* @package CommunityBuilder
* @copyright (C) 2004-2022 www.joomlapolis.com / Lightning MultiCom SA - and its licensors, all rights reserved
* @license http://www.gnu.org/licenses/old-licenses/gpl-2.0.html GNU/GPL version 2
*/

use CBLib\Application\Application;
use CBLib\Language\CBTxt;
use Joomla\CMS\Date\Date;
use Joomla\CMS\Factory;

defined( '_JEXEC' ) or die;

/**
 * Suppressing class name notice as isn't a bug, but intentional as we need J3.x compatibility, which has no src autoloader
 * Replace with autoloader class once J3.x support is discontinued
 *
 * @noinspection AutoloadingIssuesInspection
 */
class CBContentModuleHelper
{
	/**
	 * @return bool
	 */
	public static function isCB(): bool
	{
		static $cache	=	null;

		if ( $cache === null ) {
			$cache		=	( file_exists( JPATH_SITE . '/libraries/CBLib/CBLib/Core/CBLib.php' ) ) && ( file_exists( JPATH_ADMINISTRATOR . '/components/com_comprofiler/plugin.foundation.php' ) );
		}

		return $cache;
	}

	/**
	 * @param $params
	 * @return void
	 */
	public static function loadCB( $params ): void
	{
		global $_PLUGINS;

		if ( ! self::isCB() ) {
			return;
		}

		static $CB_loaded	=	0;

		if ( ! $CB_loaded++ ) {
			include_once( JPATH_ADMINISTRATOR . '/components/com_comprofiler/plugin.foundation.php' );

			cbimport( 'cb.html' );

			if ( (int) $params->get( 'maincbtpl', 0 ) ) {
				outputCbTemplate();
			}

			if ( (int) $params->get( 'maincbjs', 0 ) ) {
				outputCbJs();
			}

			if ( (int) $params->get( 'load_tooltip', 0 ) ) {
				initToolTip();
			}

			if ( (int) $params->get( 'load_lang', 1 ) ) {
				cbimport( 'language.front' );
			}

			if ( (int) $params->get( 'load_plgs', 0 ) ) {
				$_PLUGINS->loadPluginGroup( 'user' );
			}
		}
	}

	/**
	 * @return null|CBuser
	 */
	private static function cbUser(): ?CBuser
	{
		if ( ! self::isCB() ) {
			return null;
		}

		static $cache	=	null;

		if ( $cache === null ) {
			$cache		=	CBuser::getMyInstance();
		}

		return $cache;
	}

	/**
	 * @param string $text
	 * @return string
	 */
	private static function prepareHtmlContentPlugins( string $text ): string
	{
		if ( ! $text ) {
			return '';
		}

		if ( ! self::isCB() ) {
			JPluginHelper::importPlugin( 'content' );

			return JHtml::_( 'content.prepare', $text, '', 'mod_cbcontent.content' );
		}

		return Application::Cms()->prepareHtmlContentPlugins( $text );
	}

	/**
	 * @return array
	 */
	private static function joomlaSubstitutions(): array
	{
		static $cache		=	null;

		if ( $cache === null ) {
			$user			=	JFactory::getUser();

			$cache			=	[	'[user_id]'		=>	(int) $user->id,
									'[username]'	=>	(string) $user->username,
									'[name]'		=>	(string) $user->name,
									'[email]'		=>	(string) $user->email,
								];
		}

		return $cache;
	}

	/**
	 * @param string      $key
	 * @param null|string $text
	 * @return string
	 */
	public static function T( string $key, ?string $text = null ): string
	{
		if ( ! self::isCB() ) {
			return JText::_( $key );
		}

		return CBTxt::T( $key, $text );
	}

	/**
	 * @param string $url
	 * @param bool   $prepareContent
	 * @return void
	 */
	public static function addHeadStyleSheet( string $url, bool $prepareContent = true ): void
	{
		global $_CB_framework;

		if ( $prepareContent ) {
			$url	=	self::prepareHtmlContentPlugins( $url );
		}

		if ( ! $url ) {
			return;
		}

		if ( ! self::isCB() ) {
			$url	=	strtr( $url, self::joomlaSubstitutions() );

			if ( version_compare( JVERSION, '4.0', '>=' ) ) {
				Factory::getApplication()->getDocument()->getWebAssetManager()->registerAndUseStyle( 'mod_modules', $url );
			} else {
				JFactory::getDocument()->addStyleSheet( $url );
			}
		} else {
			$url	=	self::cbUser()->replaceUserVars( $url, true, false, null, false );

			if ( ! $url ) {
				return;
			}

			$_CB_framework->document->addHeadStyleSheet( $url );
		}
	}

	/**
	 * @param string $css
	 * @param bool   $prepareContent
	 * @return void
	 */
	public static function addHeadStyleInline( string $css, bool $prepareContent = true ): void
	{
		global $_CB_framework;

		if ( $prepareContent ) {
			$css	=	self::prepareHtmlContentPlugins( $css );
		}

		if ( ! $css ) {
			return;
		}

		if ( ! self::isCB() ) {
			$css	=	strtr( $css, self::joomlaSubstitutions() );

			if ( version_compare( JVERSION, '4.0', '>=' ) ) {
				Factory::getApplication()->getDocument()->getWebAssetManager()->addInlineStyle( $css );
			} else {
				JFactory::getDocument()->addStyleDeclaration( $css );
			}
		} else {
			$css	=	self::cbUser()->replaceUserVars( $css, true, false, null, false );

			if ( ! $css ) {
				return;
			}

			$_CB_framework->document->addHeadStyleInline( $css );
		}
	}

	/**
	 * @param string $url
	 * @param bool   $prepareContent
	 * @return void
	 */
	public static function addHeadScriptUrl( string $url, bool $prepareContent = true ): void
	{
		global $_CB_framework;

		if ( $prepareContent ) {
			$url	=	self::prepareHtmlContentPlugins( $url );
		}

		if ( ! $url ) {
			return;
		}

		if ( ! self::isCB() ) {
			$url	=	strtr( $url, self::joomlaSubstitutions() );

			if ( version_compare( JVERSION, '4.0', '>=' ) ) {
				Factory::getApplication()->getDocument()->getWebAssetManager()->registerAndUseScript( 'mod_modules', $url );
			} else {
				JFactory::getDocument()->addScript( $url );
			}
		} else {
			$url	=	self::cbUser()->replaceUserVars( $url, true, false, null, false );

			if ( ! $url ) {
				return;
			}

			$_CB_framework->document->addHeadScriptUrl( $url );
		}
	}

	/**
	 * @param string $js
	 * @param bool   $prepareContent
	 * @return void
	 */
	public static function addHeadScriptDeclaration( string $js, bool $prepareContent = true ): void
	{
		global $_CB_framework;

		if ( $prepareContent ) {
			$js	=	self::prepareHtmlContentPlugins( $js );
		}

		if ( ! $js ) {
			return;
		}

		if ( ! self::isCB() ) {
			$js	=	strtr( $js, self::joomlaSubstitutions() );

			if ( version_compare( JVERSION, '4.0', '>=' ) ) {
				Factory::getApplication()->getDocument()->getWebAssetManager()->addInlineScript( $js );
			} else {
				JFactory::getDocument()->addStyleDeclaration( $js );
			}
		} else {
			$js	=	self::cbUser()->replaceUserVars( $js, true, false, null, false );

			if ( ! $js ) {
				return;
			}

			$_CB_framework->document->addHeadScriptDeclaration( $js );
		}
	}

	/**
	 * @param string   $js
	 * @param string[] $plugins
	 * @param bool     $prepareContent
	 * @return void
	 */
	public static function addHeadJqueryDeclaration( string $js, array $plugins = [], bool $prepareContent = true ): void
	{
		global $_CB_framework;

		if ( $prepareContent ) {
			$js	=	self::prepareHtmlContentPlugins( $js );
		}

		if ( ! $js ) {
			return;
		}

		if ( ! self::isCB() ) {
			JHtml::_( 'jquery.framework' );

			foreach ( $plugins as $plugin ) {
				JHtml::_( $plugin );
			}

			$js	=	"jQuery(document).ready(function($){" . strtr( $js, self::joomlaSubstitutions() ) . "});";

			if ( version_compare( JVERSION, '4.0', '>=' ) ) {
				Factory::getApplication()->getDocument()->getWebAssetManager()->addInlineScript( $js );
			} else {
				JFactory::getDocument()->addStyleDeclaration( $js );
			}
		} else {
			$js	=	self::cbUser()->replaceUserVars( $js, true, false, null, false );

			if ( ! $js ) {
				return;
			}

			$_CB_framework->outputCbJQuery( $js, $plugins );
		}
	}

	/**
	 * @param string|int $date
	 * @param string     $format
	 * @return string
	 */
	public static function formatDate( $date, string $format = 'Y-m-d H:i:s' ): string
	{
		if ( ! self::isCB() ) {
			return ( new Date( $date, 'UTC' ) )->format( $format );
		}

		return Application::Date( $date, 'UTC' )->format( $format );
	}

	/**
	 * @param string|int  $date
	 * @param null|string $modify
	 * @return int
	 */
	public static function dateTimestamp( $date, ?string $modify = null ): int
	{
		if ( ! self::isCB() ) {
			if ( $modify ) {
				return ( new Date( $date, 'UTC' ) )->modify( $modify )->toUnix();
			}

			return ( new Date( $date, 'UTC' ) )->toUnix();
		}

		if ( $modify ) {
			return Application::Date( $date, 'UTC' )->modify( $modify )->getTimestamp();
		}

		return Application::Date( $date, 'UTC' )->getTimestamp();
	}

	/**
	 * @param string|int  $date
	 * @param bool        $showTime
	 * @param null|string $dateFormat
	 * @param null|string $timeFormat
	 * @param bool        $offset
	 * @return string
	 */
	public static function displayDate( $date, bool $showTime = true, ?string $dateFormat = null, ?string $timeFormat = null, bool $offset = true ): string
	{
		if ( ( ! $date ) || ( $date === '0000-00-00 00:00:00' ) || ( $date === '0000-00-00' ) || ( $date === '00:00:00' ) ) {
			return '';
		}

		if ( ! self::isCB() ) {
			if ( in_array( $showTime, [ 'timeago', 'exacttimeago' ], true ) ) {
				$format			=	JText::_( 'DATE_FORMAT_LC2' );
			} else {
				$format			=	( $showTime ? JText::_( 'DATE_FORMAT_LC6' ) : JText::_( 'DATE_FORMAT_LC4' ) );
			}

			if ( $dateFormat !== null ) {
				$format			=	$dateFormat;

				if ( $showTime ) {
					if ( $timeFormat !== null ) {
						$format	.=	$timeFormat;
					} else {
						$format	.=	' H:i:s';
					}
				}
			}

			return JHTML::_( 'date', $date, $format, ( $offset ? true : 'UTC' ) );
		}

		return cbFormatDate( $date, $offset, $showTime, $dateFormat, $timeFormat );
	}

	/**
	 * @param string $text
	 * @param bool   $prepareContent
	 * @param array  $extra
	 * @return string
	 */
	public static function prepareContent( string $text, bool $prepareContent = true, array $extra = [] ): string
	{
		if ( $prepareContent ) {
			$text		=	self::prepareHtmlContentPlugins( $text );
		}

		if ( ! self::isCB() ) {
			$text		=	strtr( $text, self::joomlaSubstitutions() );

			if ( ! $extra ) {
				return $text;
			}

			foreach ( $extra as $k => $v ) {
				$text	=	str_ireplace( '[' . $k . ']', (string) $v, $text );
			}

			return $text;
		}

		return self::cbUser()->replaceUserVars( $text, false, false, $extra, false );
	}

	/**
	 * @param string $text
	 * @return string
	 */
	public static function parseSubstitutions( string $text ): string
	{
		if ( ! self::isCB() ) {
			return $text;
		}

		return self::cbUser()->replaceUserVars( $text, true, false, null, false );
	}

	/**
	 * @param string $field
	 * @return string
	 */
	public static function fieldValue( string $field ): string
	{
		if ( ! self::isCB() ) {
			$user		=	JFactory::getUser();

			if ( isset( $user->$field ) ) {
				return (string) $user->$field;
			}

			$profile	=	JUserHelper::getProfile( $user->id );

			if ( isset( $profile->$field ) ) {
				return (string) $profile->$field;
			}

			return '';
		}

		$value			=	self::cbUser()->getField( $field, null, 'php', 'none', 'profile', 0, true );

		if ( is_array( $value ) ) {
			$value		=	array_shift( $value );

			if ( is_array( $value ) ) {
				$value	=	implode( '|*|', $value );
			}
		}

		if ( $value === null ) {
			return '';
		}

		return (string) $value;
	}
}