<?php
/**
 * Community Builder (TM)
 * @version $Id: $
 * @package CommunityBuilder
 * @copyright (C) 2004-2021 www.joomlapolis.com / Lightning MultiCom SA - and its licensors, all rights reserved
 * @license http://www.gnu.org/licenses/old-licenses/gpl-2.0.html GNU/GPL version 2
 */

namespace CB\Plugin\Query\Field;

use CB\Database\Table\UserTable;
use CB\Database\Table\FieldTable;
use CB\Plugin\Query\CBQueryField;
use CBLib\Language\CBTxt;
use CBLib\Application\Application;
use CBLib\Registry\GetterInterface;
use CBLib\Input\Get;
use CBuser;

\defined( 'CBLIB') or die();

class QuerySelectField extends \cbFieldHandler
{
	/** @var null|string */
	private ?string $queryError	=	null;

	/**
	 * Accessor:
	 * Returns a field in specified format
	 *
	 * @param  FieldTable  $field
	 * @param  UserTable   $user
	 * @param  string      $output               'html', 'xml', 'json', 'php', 'csvheader', 'csv', 'rss', 'fieldslist', 'htmledit'
	 * @param  string      $reason               'profile' for user profile view, 'edit' for profile edit, 'register' for registration, 'search' for searches
	 * @param  int         $list_compare_types   IF reason == 'search' : 0 : simple 'is' search, 1 : advanced search with modes, 2 : simple 'any' search
	 * @return mixed
	 */
	public function getField( &$field, &$user, $output, $reason, $list_compare_types )
	{
		$value							=	(string) $user->getRaw( $field->getString( 'name', '' ), '' );

		switch ( $output ) {
			case 'html':
			case 'rss':
				$values					=	$this->_explodeCBvalues( $value );
				$options				=	$this->getOptions( $field, $user, $reason, true );
				$labels					=	[];

				if ( $field->getString( 'type', '' ) === 'querytag' ) {
					foreach ( $values as $v ) {
						foreach ( $options as $option ) {
							if ( $v !== $option['value'] ) {
								continue;
							}

							if ( ! \in_array( $option['text'], $labels, true ) ) {
								$labels[]	=	$option['text'];
							}

							continue 2;
						}

						if ( ! \in_array( $v, $labels, true ) ) {
							// Adds custom tag values:
							$labels[]		=	$v;
						}
					}
				} else {
					foreach ( $options as $option ) {
						if ( \in_array( $option['value'], $values, true ) && ( ! \in_array( $option['text'], $labels, true ) ) ) {
							$labels[]		=	$option['text'];
						}
					}
				}

				$displayStyle				=	$field->params->getInt( 'field_display_style', 0 );
				$return						=	null;

				if ( Application::MyUser()->isGlobalModerator() && isset( $this->queryError ) && $this->queryError ) {
					$return					.=	'<div class="alert alert-danger">' . $this->queryError . '</div>';
				}

				$return						.=	$this->formatFieldValueLayout( $this->_arrayToFormat( $field, $labels, $output, ( $displayStyle === 1 ? 'ul' : ( $displayStyle === 2 ? 'ol' : ( $displayStyle === 3 ? 'tag' : ', ' ) ) ), trim( $field->params->getString( 'field_display_class', '' ) ) ), $reason, $field, $user );

				return $return;
			case 'htmledit':
				return $this->getEdit( $value, $field, $user, $reason, $list_compare_types );
			case 'xml':
			case 'json':
			case 'php':
			case 'csv':
				if ( ( $output !== 'csv' ) && ( substr( $reason, -11 ) === ':translated' ) ) {
					if ( \in_array( $field->getString( 'type', '' ), [ 'queryradio', 'queryselect' ], true ) ) {
						return $this->_formatFieldOutput( $field->getString( 'name', '' ), CBTxt::T( $value ), $output, ( $output !== 'xml' ) );
					}

					$values					=	$this->_explodeCBvalues( $value );

					foreach ( $values as $k => $v ) {
						$values[$k]			=	CBTxt::T( $v );
					}

					return $this->_arrayToFormat( $field, $values, $output );
				}

				if ( \in_array( $field->getString( 'type', '' ), [ 'queryradio', 'queryselect' ], true ) ) {
					return $this->_formatFieldOutput( $field->getString( 'name', '' ), $value, $output, ( $output !== 'xml' ) );
				}

				return $this->_arrayToFormat( $field, $this->_explodeCBvalues( $value ), $output );
			case 'csvheader':
			case 'fieldslist':
			default:
				return parent::getField( $field, $user, $output, $reason, $list_compare_types );
		}
	}

	/**
	 * Mutator:
	 * Prepares field data for saving to database (safe transfer from $postdata to $user)
	 * Override
	 *
	 * @param  FieldTable  $field
	 * @param  UserTable   $user      RETURNED populated: touch only variables related to saving this field (also when not validating for showing re-edit)
	 * @param  array       $postdata  Typically $_POST (but not necessarily), filtering required.
	 * @param  string      $reason    'edit' for save user edit, 'register' for save registration
	 */
	public function prepareFieldDataSave( &$field, &$user, &$postdata, $reason )
	{
		$isTags								=	( $field->getString( 'type', '' ) === 'querytag' );

		$this->_prepareFieldMetaSave( $field, $user, $postdata, $reason );

		$options							=	$this->getOptions( $field, $user, $reason, 'value', $postdata );

		if ( ( ! $options ) && ( ! $isTags ) ) {
			// There are no options available to this field so just skip validation and storage
			return;
		}

		foreach ( $field->getTableColumns() as $col ) {
			$value							=	cbGetParam( $postdata, $col, null, _CB_ALLOWRAW );

			if ( \is_array( $value ) ) {
				if ( \count( $value ) > 0 ) {
					$okVals					=	[];

					foreach ( $value as $k => $v ) {
						$v					=	stripslashes( (string) $v );

						if ( \in_array( $v, $okVals, true ) ) {
							continue;
						}

						if ( \in_array( $v, $options, true ) ) {
							$okVals[$k]		=	$v;
						} elseif ( $isTags ) {
							// Allow unauthorized values for tags, but clean them to strings:
							$okVals[$k]		=	Get::clean( $v, GetterInterface::STRING );
						}
					}

					$value					=	$this->_implodeCBvalues( $okVals );
				} else {
					$value					=	'';
				}
			} elseif ( ( $value === null ) || ( $value === '' ) ) {
				$value						=	'';
			} else {
				$value						=	stripslashes( (string) $value );

				if ( ! \in_array( $value, $options, true ) ) {
					if ( $isTags ) {
						// Allow unauthorized value for tags, but clean it to string:
						$value				=	Get::clean( $value, GetterInterface::STRING );
					} else {
						$value				=	'';
					}
				}
			}

			if ( $this->validate( $field, $user, $col, $value, $postdata, $reason )
				 && isset( $user->$col )
				 && ( (string) $user->$col !== (string) $value )
			) {
				$this->_logFieldUpdate( $field, $user, $reason, $user->$col, $value );
			}

			$user->$col						=	$value;
		}
	}

	/**
	 * Finder:
	 * Prepares field data for saving to database (safe transfer from $postdata to $user)
	 * Override
	 *
	 * @param  FieldTable  $field
	 * @param  UserTable   $searchVals          RETURNED populated: touch only variables related to saving this field (also when not validating for showing re-edit)
	 * @param  array       $postdata            Typically $_POST (but not necessarily), filtering required.
	 * @param  int         $list_compare_types  IF reason == 'search' : 0 : simple 'is' search, 1 : advanced search with modes, 2 : simple 'any' search
	 * @param  string      $reason              'edit' for save user edit, 'register' for save registration
	 * @return \cbSqlQueryPart[]
	 */
	public function bindSearchCriteria( &$field, &$searchVals, &$postdata, $list_compare_types, $reason )
	{
		switch ( $field->getString( 'type', '' ) ) {
			case 'querytag':
				$fieldType				=	'tag';
				break;
			case 'querymulticheckbox':
				$fieldType				=	'multicheckbox';
				break;
			case 'queryradio':
				$fieldType				=	'radio';
				break;
			case 'querymultiselect':
				$fieldType				=	'multiselect';
				break;
			case 'queryselect':
			default:
				$fieldType				=	'select';
				break;
		}

		if ( ( $fieldType === 'radio' ) && \in_array( $list_compare_types, [ 0, 2 ], true ) ) {
			$fieldType					=	'multicheckbox';
		}

		$query							=	[];
		$searchMode						=	$this->_bindSearchMode( $field, $searchVals, $postdata, ( strpos( $fieldType, 'multi' ) === 0 ? 'multiplechoice' : 'singlechoice' ), $list_compare_types );

		if ( ! $searchMode ) {
			return $query;
		}

		$user							=	CBuser::getMyUserDataInstance();
		$options						=	$this->getOptions( $field, $user, $reason, 'value', $postdata );

		if ( ( ! $options ) && ( $fieldType !== 'tag' ) ) {
			// There are no options available to this field so skip searching
			return $query;
		}

		foreach ( $field->getTableColumns() as $col ) {
			$value						=	cbGetParam( $postdata, $col );

			if ( \is_array( $value ) ) {
				if ( \count( $value ) > 0 ) {
					foreach ( $value as $k => $v ) {
						if ( ( \count( $value ) === 1 ) && ( $v === '' ) ) {
							if ( $list_compare_types === 1 ) {
								$value	=	'';		// Advanced search: "None": checked: search for nothing selected
							} else {
								$value	=	null;	// Type 0 and 2 : Simple search: "Do not care" checked: do not search
							}
							break;
						}

						$v				=	stripslashes( (string) $v );

						if ( \in_array( $v, $options, true ) ) {
							$value[$k]	=	$v;
						} elseif ( $fieldType === 'tag' ) {
							// Allow unauthorized values for tags, but clean them to strings:
							$value[$k]	=	Get::clean( $v, GetterInterface::STRING );
						} else {
							unset( $value[$k] );
						}
					}
				} else {
					$value				=	null;
				}

				if ( ( $value !== null ) && ( $value !== '' ) && \in_array( $searchMode, [ 'is', 'isnot' ], true ) ) {
					$value				=	stripslashes( $this->_implodeCBvalues( $value ) );
				}
			} elseif ( ( $value !== null ) && ( $value !== '' ) ) {
				if ( ! \in_array( $value, $options, true ) ) {
					if ( $fieldType === 'tag' ) {
						// Allow unauthorized value for tags, but clean it to string:
						$value			=	Get::clean( $value, GetterInterface::STRING );
					} else {
						$value			=	null;
					}
				}
			} elseif ( ( $list_compare_types === 1 ) && \in_array( $searchMode, [ 'is', 'isnot' ], true ) ) {
				$value					=	'';
			} else {
				$value					=	null; // 'none' is not checked and no other is checked: search for DON'T CARE
			}

			if ( $value !== null ) {
				$searchVals->$col		=	$value;

				$sql					=	new \cbSqlQueryPart();
				$sql->tag				=	'column';
				$sql->name				=	$col;
				$sql->table				=	$field->getString( 'table', '' );
				$sql->type				=	'sql:field';
				$sql->operator			=	'=';
				$sql->value				=	$value;
				$sql->valuetype			=	'const:string';
				$sql->searchmode		=	$searchMode;

				$query[]				=	$sql;
			}
		}

		return $query;
	}

	/**
	 * @param mixed      $value
	 * @param FieldTable $field
	 * @param UserTable  $user
	 * @param string     $reason             'profile' for user profile view, 'edit' for profile edit, 'register' for registration, 'search' for searches
	 * @param int        $list_compare_types IF reason == 'search' : 0 : simple 'is' search, 1 : advanced search with modes, 2 : simple 'any' search
	 * @return array|string|string[]
	 */
	private function getEdit( $value, FieldTable $field, UserTable $user, string $reason, int $list_compare_types )
	{
		global $_CB_framework;

		$options							=	$this->getOptions( $field, $user, $reason );

		switch ( $field->getString( 'type', '' ) ) {
			case 'querytag':
				$fieldType					=	'tag';
				break;
			case 'querymulticheckbox':
				$fieldType					=	'multicheckbox';
				break;
			case 'queryradio':
				$fieldType					=	'radio';
				break;
			case 'querymultiselect':
				$fieldType					=	'multiselect';
				break;
			case 'queryselect':
			default:
				$fieldType					=	'select';
				break;
		}

		if ( ( ! $options ) && ( $fieldType !== 'tag' ) ) {
			if ( $field->getBool( '_isAjaxUpdate', false ) ) {
				// There are no options available to this field and its an ajax response so lets let the ajax respone know we're empty
				$field->set( '_isAjaxUpdateEmpty', true );
			}

			// There's nothing to display so make the field hidden and not required, but still exist on the page for ajax plugin purposes
			$field->set( 'cssclass', 'hidden' );
			$field->set( 'required', 0 );
		}

		$return								=	null;

		if ( Application::MyUser()->isGlobalModerator() && isset( $this->queryError ) && $this->queryError ) {
			$return							.=	'<div class="alert alert-danger">' . $this->queryError . '</div>';
		}

		if ( $fieldType === 'tag' ) {
			static $loaded	=	0;

			if ( ! $loaded++ ) {
				$js							=	"$( 'select.cbQuerySelectTag' ).cbselect({"
											.		"tags: true,"
											.		"language: {"
											.			"errorLoading: function() {"
											.				"return " . json_encode( CBTxt::T( 'The results could not be loaded.' ), JSON_HEX_TAG ) . ";"
											.			"},"
											.			"inputTooLong: function() {"
											.				"return " . json_encode( CBTxt::T( 'Search input too long.' ), JSON_HEX_TAG ) . ";"
											.			"},"
											.			"inputTooShort: function() {"
											.				"return " . json_encode( CBTxt::T( 'Search input too short.' ), JSON_HEX_TAG ) . ";"
											.			"},"
											.			"loadingMore: function() {"
											.				"return " . json_encode( CBTxt::T( 'Loading more results...' ), JSON_HEX_TAG ) . ";"
											.			"},"
											.			"maximumSelected: function() {"
											.				"return " . json_encode( CBTxt::T( 'You cannot select any more choices.' ), JSON_HEX_TAG ) . ";"
											.			"},"
											.			"noResults: function() {"
											.				"return " . json_encode( CBTxt::T( 'No results found.' ), JSON_HEX_TAG ) . ";"
											.			"},"
											.			"searching: function() {"
											.				"return " . json_encode( CBTxt::T( 'Searching...' ), JSON_HEX_TAG ) . ";"
											.			"}"
											.		"}"
											.	"});";

				$_CB_framework->outputCbJQuery( $js, 'cbselect' );
			}
		}

		if ( $reason === 'search' ) {
			$displayType					=	$fieldType;

			if ( ( $fieldType === 'radio' ) && ( ( $list_compare_types === 2 ) || ( \is_array( $value ) && ( \count( $value ) > 1 ) ) ) ) {
				$displayType				=	'multicheckbox';
			}

			if ( ( $fieldType === 'select' ) && ( ( $list_compare_types === 1 ) || ( \is_array( $value ) && ( \count( $value ) > 1 ) ) ) ) {
				$displayType				=	'multiselect';
			}

			if ( \in_array( $list_compare_types, [ 0, 2 ], true ) && ( ! \in_array( $displayType, [ 'multicheckbox', 'tag' ], true ) ) ) {
				if ( $options && ( $options[0]->value === '' ) ) {
					// About to add 'No preference' so remove custom blank
					unset( $options[0] );
				}

				array_unshift( $options, \moscomprofilerHTML::makeOption( '', CBTxt::T( 'UE_NO_PREFERENCE', 'No preference' ) ) );
			}

			$html							=	$this->_fieldEditToHtml( $field, $user, $reason, 'input', $displayType, $value, null, $options, true, null, false );
			$return							.=	$this->_fieldSearchModeHtml( $field, $user, $html, ( ( ( strpos( $displayType, 'multi' ) === 0 ) && ( ! \in_array( $fieldType, [ 'queryradio', 'queryselect' ], true ) ) ) || ( $displayType === 'tag' ) ? 'multiplechoice' : 'singlechoice' ), $list_compare_types );
		} else {
			if ( ( $fieldType === 'tag' ) && ( $value !== '' ) ) {
				// Since we're a tag usage we can have custom values so lets see if any exist to be added to available options:
				$values						=	$this->_explodeCBvalues( $value );

				foreach ( $values as $k => $v ) {
					foreach ( $options as $option ) {
						if ( $v !== $option->value ) {
							// Custom values we'll add further below:
							continue;
						}

						// Skip values that actually exist:
						continue 2;
					}

					// Add custom tags to the available values list:
					$options[]				=	\moscomprofilerHTML::makeOption( $v, $v );
				}
			}

			if ( \in_array( $fieldType, [ 'multicheckbox', 'radio' ], true ) && $field->params->getInt( 'field_edit_style', 0 ) ) {
				$return						.=	$this->_fieldEditToHtml( $field, $user, $reason, 'input', $fieldType . 'buttons', $value, ( \in_array( $fieldType, [ 'multicheckbox', 'multiselect', 'tag' ], true ) ? $this->getDataAttributes( $field, $user, 'htmledit', $reason ) : null ), $options, true, null, false );
			} else {
				$return						.=	$this->_fieldEditToHtml( $field, $user, $reason, 'input', $fieldType, $value, ( \in_array( $fieldType, [ 'multicheckbox', 'multiselect', 'tag' ], true ) ? $this->getDataAttributes( $field, $user, 'htmledit', $reason ) : null ), $options, true, null, false );
			}
		}

		if ( $fieldType === 'tag' ) {
			// Workaround to tag JS conflict from above with core tag JS:
			$return							=	str_replace( 'cbSelectTag', 'cbQuerySelectTag', $return );
		}

		return $return;
	}

	/**
	 * @param FieldTable  $field
	 * @param UserTable   $user
	 * @param string      $reason
	 * @param bool|string $raw
	 * @param array       $post
	 * @return array
	 */
	private function getOptions( FieldTable $field, UserTable $user, string $reason, $raw = false, array $post = [] ): array
	{
		global $_CB_database;

		static $cache							=	[];

		$options								=	[];
		$cacheId								=	$field->getInt( 'fieldid', 0 );

		if ( ! isset( $cache[$cacheId] ) ) {
			$query								=	"SELECT " . $_CB_database->NameQuote( 'fieldtitle' ) . " AS " . $_CB_database->NameQuote( 'value' )
												.	", if ( " . $_CB_database->NameQuote( 'fieldlabel' ) . " != '', " . $_CB_database->NameQuote( 'fieldlabel' ) . ", " . $_CB_database->NameQuote( 'fieldtitle' ) . " ) AS " . $_CB_database->NameQuote( 'text' )
												.	", " . $_CB_database->NameQuote( 'fieldgroup' ) . " AS " . $_CB_database->NameQuote( 'group' )
												.	"\n FROM " . $_CB_database->NameQuote( '#__comprofiler_field_values' )
												.	"\n WHERE " . $_CB_database->NameQuote( 'fieldid' ) . " = " . (int) $cacheId
												.	"\n ORDER BY" . $_CB_database->NameQuote( 'ordering' );
			$_CB_database->setQuery( $query );
			$cache[$cacheId]					=	$_CB_database->loadObjectList();
		}

		$rows									=	$cache[$cacheId];

		if ( $rows ) {
			foreach ( $rows as $row ) {
				if ( $raw ) {
					if ( $row->group ) {
						continue;
					}

					if ( $raw === 'value' ) {
						$options[]				=	(string) $row->value;
					} elseif ( $raw === 'text' ) {
						$options[]				=	CBTxt::T( (string) $row->text );
					} else {
						$options[]				=	[ 'value' => (string) $row->value, 'text' => CBTxt::T( (string) $row->text ) ];
					}
				} elseif ( $row->group ) {
					$options[]					=	\moscomprofilerHTML::makeOptGroup( CBTxt::T( (string) $row->text ) );
				} else {
					$options[]					=	\moscomprofilerHTML::makeOption( (string) $row->value, CBTxt::T( (string) $row->text ) );
				}
			}
		}

		if ( $post ) {
			$postUser							=	clone $user;

			foreach ( $post as $k => $v ) {
				if ( ! $k ) {
					continue;
				}

				if ( \is_array( $v ) ) {
					$multi						=	false;

					foreach ( $v as $kv => $cv ) {
						if ( is_numeric( $kv ) ) {
							$kv					=	(int) $kv;
						}

						if ( \is_object( $cv ) || \is_array( $cv ) || ( $kv && ( ! \is_int( $kv ) ) ) ) {
							$multi				=	true;
						}
					}

					if ( ! $multi ) {
						$v						=	implode( '|*|', $v );
					} else {
						continue;
					}
				} elseif ( \is_object( $v ) ) {
					continue;
				}

				$postUser->set( $k, Get::clean( $v, GetterInterface::STRING ) );
			}

			$tempUser							=	$postUser;
		} else {
			$tempUser							=	$user;
		}

		// Force the user object to CBuser instance to allow substitutions for non-existant users to work:
		$cbUser									=	new CBuser();
		$cbUser->_cbuser						=	$tempUser;

		$query									=	$cbUser->replaceUserVars( $field->params->getRaw( 'qry_query', '' ), [ '\CB\Plugin\Query\CBQueryField', 'escapeSQL' ], false, [ 'reason' => $reason ], false );

		if ( ! $query ) {
			return $options;
		}

		$valueColumn							=	$field->params->getString( 'qry_col_value', '' );
		$labelColumn							=	$field->params->getString( 'qry_col_label', '' );
		$customLabel							=	CBTxt::T( $field->params->getString( 'qry_col_label_custom', '' ) );
		$groupColumn							=	$field->params->getString( 'qry_col_optgrp', '' );
		$customGroup							=	CBTxt::T( $field->params->getString( 'qry_col_optgrp_custom', '' ) );
		$optgroups								=	[];
		$currentGroup							=	null;

		$cacheId								=	md5( $query );

		if ( ! isset( $cache[$cacheId] ) ) {
			try {
				$_SQL_database					=	CBQueryField::getDatabase( $field );

				$_SQL_database->setQuery( $query );

				$cache[$cacheId]				=	$_SQL_database->loadAssocList();
			} catch ( \Exception $e ) {
				if ( Application::MyUser()->isGlobalModerator() ) {
					$this->queryError			=	$e->getMessage();
				}

				$cache[$cacheId]				=	[];
			}
		}

		$rows									=	$cache[$cacheId];

		foreach ( $rows as $row ) {
			if ( \is_array( $row ) ) {
				foreach ( $row as $k => $v ) {
					$row[$k]					=	trim( (string) $v );
				}
			} else {
				$row							=	trim( (string) $row );
			}

			$value								=	null;

			if ( $valueColumn && isset( $row[$valueColumn] ) ) {
				$value							=	$row[$valueColumn];
			}

			$label								=	null;

			if ( $labelColumn && isset( $row[$labelColumn] ) ) {
				$label							=	$row[$labelColumn];

				if ( $customLabel ) {
					$label						=	$cbUser->replaceUserVars( $customLabel, true, false, [ 'reason' => $reason, 'value' => (string) ( $value ?? $label ), 'label' => (string) $label ], false );
				}

				if ( $value === null ) {
					$value						=	$label;
				} elseif ( $label === null ) {
					$label						=	$value;
				}
			} elseif ( $value !== null ) {
				$label							=	$value;
			}

			if ( ( $value === null ) && ( $label === null ) ) {
				if ( \is_array( $row ) ) {
					$value						=	array_shift( $row );
				} else {
					$value						=	$row;
				}

				$label							=	$value;
			}

			if ( ( $value !== null ) && ( $label !== null ) ) {
				if ( $raw ) {
					if ( $raw === 'value' ) {
						$options[]				=	(string) $value;
					} elseif ( $raw === 'text' ) {
						$options[]				=	CBTxt::T( (string) $label );
					} else {
						$options[]				=	[ 'value' => (string) $value, 'text' => CBTxt::T( (string) $label ) ];
					}
				} else {
					if ( $groupColumn && \in_array( $field->getString( 'type', '' ), [ 'queryselect', 'querymultiselect' ], true ) ) {
						$previousGroup			=	$currentGroup;

						if ( isset( $row[$groupColumn] ) ) {
							$currentGroup		=	( $row[$groupColumn] !== '' ? $row[$groupColumn] : null );

							if ( $previousGroup && ( $currentGroup !== $previousGroup ) ) {
								// We're starting a new optgroup while one already exists so lets close the previous:
								$options[]		=	\moscomprofilerHTML::makeOptGroup( null );
							}

							if ( ( $row[$groupColumn] !== '' ) && ( ! \in_array( $row[$groupColumn], $optgroups, true ) ) ) {
								$group			=	$row[$groupColumn];

								if ( $customGroup ) {
									$group		=	$cbUser->replaceUserVars( $customGroup, true, false, [ 'reason' => $reason, 'value' => (string) $value, 'label' => (string) $label, 'group' => (string) $group ], false );
								}

								$options[]		=	\moscomprofilerHTML::makeOptGroup( CBTxt::T( (string) $group ) );
								$optgroups[]	=	$row[$groupColumn];
							}
						} else {
							$currentGroup		=	null;

							if ( $previousGroup && ( $currentGroup !== $previousGroup ) ) {
								// We're starting a new optgroup while one already exists so lets close the previous:
								$options[]		=	\moscomprofilerHTML::makeOptGroup( null );
							}
						}
					}

					$options[]					=	\moscomprofilerHTML::makeOption( (string) $value, CBTxt::T( (string) $label ) );
				}
			}
		}

		return $options;
	}
}