<?php
/**
 * Community Builder (TM)
 * @version $Id: $
 * @package CommunityBuilder
 * @copyright (C) 2004-2021 www.joomlapolis.com / Lightning MultiCom SA - and its licensors, all rights reserved
 * @license http://www.gnu.org/licenses/old-licenses/gpl-2.0.html GNU/GPL version 2
 */

namespace CB\Plugin\Query\Field;

use CB\Database\Table\FieldTable;
use CB\Database\Table\UserTable;
use CB\Plugin\Query\CBQueryField;
use CBLib\Application\Application;
use CBLib\Language\CBTxt;
use CBuser;

\defined( 'CBLIB') or die();

class QueryField extends \cbFieldHandler
{

	/**
	 * Accessor:
	 * Returns a field in specified format
	 *
	 * @param  FieldTable  $field
	 * @param  UserTable   $user
	 * @param  string      $output               'html', 'xml', 'json', 'php', 'csvheader', 'csv', 'rss', 'fieldslist', 'htmledit'
	 * @param  string      $reason               'profile' for user profile view, 'edit' for profile edit, 'register' for registration, 'search' for searches
	 * @param  int         $list_compare_types   IF reason == 'search' : 0 : simple 'is' search, 1 : advanced search with modes, 2 : simple 'any' search
	 * @return mixed
	 */
	public function getField( &$field, &$user, $output, $reason, $list_compare_types )
	{
		static $cache										=	[];

		// Force the user object to CBuser instance to allow substitutions for non-existant users to work:
		$cbUser												=	new CBuser();
		$cbUser->_cbuser									=	$user;

		$query												=	$cbUser->replaceUserVars( $field->params->getRaw( 'qry_query', '' ), [ '\CB\Plugin\Query\CBQueryField', 'escapeSQL' ], false, [ 'reason' => $reason ], false );

		if ( $query && $field->params->getBool( 'qry_content_plugins', false ) ) {
			$query											=	Application::Cms()->prepareHtmlContentPlugins( $query, 'field.query', $user->getInt( 'id', 0 ) );
		}

		$return												=	null;

		if ( $query ) {
			$cacheId										=	md5( $query );

			try {
				switch ( $field->params->getInt( 'qry_output', 0 ) ) {
					case 2:
						ob_start();
						require CBQueryField::getTemplate( $field, 'display' );
						$return								=	ob_get_clean();
						break;
					case 1:
						if ( $field->params->getInt( 'qry_multi_columns', 1 ) ) {
							if ( ! isset( $cache[$cacheId] ) ) {
								$_SQL_database				=	CBQueryField::getDatabase( $field );

								$_SQL_database->setQuery( $query );

								$cache[$cacheId]			=	$_SQL_database->loadAssocList();
							}

							$rows							=	$cache[$cacheId];

							if ( $rows ) {
								$count						=	\count( $rows );
								$return						=	$cbUser->replaceUserVars( CBTxt::T( $field->params->getRaw( 'qry_header', '' ) ), false, false, [ 'reason' => $reason, 'row_count' => $count ], false );

								foreach ( $rows as $row ) {
									$extra					=	[ 'reason' => $reason, 'row_count' => $count ];

									CBQueryField::prepareExtras( 'column', $row, $extra );

									$return					.=	$cbUser->replaceUserVars( CBTxt::T( $field->params->getRaw( 'qry_row', '' ) ), false, false, $extra, false );
								}

								$return						.=	$cbUser->replaceUserVars( CBTxt::T( $field->params->getRaw( 'qry_footer', '' ) ), false, false, [ 'reason' => $reason, 'row_count' => $count ], false );
							}
						} else {
							if ( ! isset( $cache[$cacheId] ) ) {
								$_SQL_database				=	CBQueryField::getDatabase( $field );

								$_SQL_database->setQuery( $query );

								$cache[$cacheId]			=	$_SQL_database->loadResultArray();
							}

							$rows							=	$cache[$cacheId];

							if ( $rows ) {
								$return						=	$this->delimiterRows( $rows, $field->params->getInt( 'qry_multi_delimiter', 0 ) );
							}
						}
						break;
					case 0:
					default:
						if ( $field->params->getInt( 'qry_columns', 0 ) ) {
							if ( ! isset( $cache[$cacheId] ) ) {
								$_SQL_database				=	CBQueryField::getDatabase( $field );

								$_SQL_database->setQuery( $query );

								$cache[$cacheId]			=	$_SQL_database->loadAssoc();
							}

							$row							=	$cache[$cacheId];

							if ( $row ) {
								if ( $field->params->getInt( 'qry_display', 0 ) ) {
									$extra					=	[ 'reason' => $reason, 'row_count' => 1, 'column_count' => \count( $row ) ];

									CBQueryField::prepareExtras( 'column', $row, $extra );

									$return					=	$cbUser->replaceUserVars( CBTxt::T( $field->params->getRaw( 'qry_custom', '' ) ), false, false, $extra, false );
								} else {
									$return					=	$this->delimiterRows( $row, $field->params->getInt( 'qry_delimiter', 0 ) );
								}
							}
						} else {
							if ( ! isset( $cache[$cacheId] ) ) {
								$_SQL_database				=	CBQueryField::getDatabase( $field );

								$_SQL_database->setQuery( $query );

								$cache[$cacheId]			=	(string) $_SQL_database->loadResult();
							}

							$return							=	$cache[$cacheId];

							if ( \is_string( $return ) ) {
								$return						=	$cbUser->replaceUserVars( CBTxt::T( $return ), false, false, [ 'reason' => $reason ], false );
							}
						}
						break;
				}
			} catch ( \Exception $e ) {
				if ( Application::MyUser()->isGlobalModerator() ) {
					$return									=	'<div class="alert alert-danger">' . $e->getMessage() . '</div>';
				} else {
					$return									=	null;
				}
			}
		}

		switch ( $output ) {
			case 'html':
			case 'rss':
			case 'htmledit':
				if ( $reason === 'search' ) {
					return	null;
				}

				return $this->formatFieldValueLayout( $this->_formatFieldOutput( $field->getString( 'name', '' ), $return, $output, false ), $reason, $field, $user );
			default:
				return $this->_formatFieldOutput( $field->getString( 'name', '' ), $return, $output, false );
		}
	}

	/**
	 * @param array  $rows
	 * @param string $delimiter
	 * @return string
	 */
	private function delimiterRows( $rows, $delimiter )
	{
		$start					=	'';
		$end					=	'';

		switch( $delimiter ) {
			case 8:
				$start			=	'<p>';
				$delimiter		=	'</p><p>';
				$end			=	'</p>';
				break;
			case 7:
				$start			=	'<span>';
				$delimiter		=	'</span><span>';
				$end			=	'</span>';
				break;
			case 6:
				$start			=	'<div>';
				$delimiter		=	'</div><div>';
				$end			=	'</div>';
				break;
			case 5:
				$start			=	'<ol><li>';
				$delimiter		=	'</li><li>';
				$end			=	'</li></ol>';
				break;
			case 4:
				$start			=	'<ul><li>';
				$delimiter		=	'</li><li>';
				$end			=	'</li></ul>';
				break;
			case 3:
				$delimiter		=	'<br />';
				break;
			case 2:
				$delimiter		=	' ';
				break;
			case 1:
				$delimiter		=	' - ';
				break;
			case 0:
			default:
				$delimiter		=	', ';
				break;
		}

		return $start . implode( $delimiter, $rows ) . $end;
	}
}