<?php
/**
 * Community Builder (TM)
 * @version $Id: $
 * @package CommunityBuilder
 * @copyright (C) 2004-2021 www.joomlapolis.com / Lightning MultiCom SA - and its licensors, all rights reserved
 * @license http://www.gnu.org/licenses/old-licenses/gpl-2.0.html GNU/GPL version 2
 */

namespace CB\Plugin\FieldGroups;

use CBLib\Application\Application;
use CBLib\Input\Get;
use CBLib\Registry\GetterInterface;
use CBLib\Registry\ParamsInterface;
use CB\Database\Table\FieldTable;
use CBLib\Registry\Registry;
use CB\Database\Table\UserTable;
use CBuser;

defined('CBLIB') or die();

class CBFieldGroups
{

	/**
	 * @return Registry
	 */
	static public function getGlobalParams()
	{
		global $_PLUGINS;

		static $params	=	null;

		if ( ! $params ) {
			$plugin		=	$_PLUGINS->getLoadedPlugin( 'user', 'cbfieldgroups' );
			$params		=	new Registry();

			if ( $plugin ) {
				$params->load( $plugin->params );
			}
		}

		return $params;
	}

	/**
	 * @param null|string $template
	 * @param null|string $file
	 * @param bool|array  $headers
	 * @return null|string
	 */
	static public function getTemplate( $template = null, $file = null, $headers = array( 'template', 'override' ) )
	{
		global $_CB_framework, $_PLUGINS;

		$plugin							=	$_PLUGINS->getLoadedPlugin( 'user', 'cbfieldgroups' );

		if ( ! $plugin ) {
			return null;
		}

		static $defaultTemplate			=	null;

		if ( $defaultTemplate === null ) {
			$defaultTemplate			=	self::getGlobalParams()->get( 'general_template', 'default', GetterInterface::STRING );
		}

		if ( ( $template === '' ) || ( $template === null ) || ( $template === '-1' ) ) {
			$template					=	$defaultTemplate;
		}

		if ( ! $template ) {
			$template					=	'default';
		}

		$livePath						=	$_PLUGINS->getPluginLivePath( $plugin );
		$absPath						=	$_PLUGINS->getPluginPath( $plugin );

		$file							=	preg_replace( '/[^-a-zA-Z0-9_]/', '', $file );
		$return							=	null;

		if ( $file ) {
			if ( $headers !== false ) {
				$headers[]				=	$file;
			}

			$php						=	$absPath . '/templates/' . $template . '/' . $file . '.php';

			if ( ! file_exists( $php ) ) {
				$php					=	$absPath . '/templates/default/' . $file . '.php';
			}

			if ( file_exists( $php ) ) {
				$return					=	$php;
			}
		}

		if ( $headers !== false ) {
			static $loaded				=	array();

			$loaded[$template]			=	array();

			// Global CSS File:
			if ( in_array( 'template', $headers ) && ( ! in_array( 'template', $loaded[$template] ) ) ) {
				$global					=	'/templates/' . $template . '/template.css';

				if ( ! file_exists( $absPath . $global ) ) {
					$global				=	'/templates/default/template.css';
				}

				if ( file_exists( $absPath . $global ) ) {
					$_CB_framework->document->addHeadStyleSheet( $livePath . $global );
				}

				$loaded[$template][]	=	'template';
			}

			// File or Custom CSS/JS Headers:
			foreach ( $headers as $header ) {
				if ( in_array( $header, $loaded[$template] ) || in_array( $header, array( 'template', 'override' ) ) ) {
					continue;
				}

				$header					=	preg_replace( '/[^-a-zA-Z0-9_]/', '', $header );

				if ( ! $header ) {
					continue;
				}

				$css					=	'/templates/' . $template . '/' . $header . '.css';
				$js						=	'/templates/' . $template . '/' . $header . '.js';

				if ( ! file_exists( $absPath . $css ) ) {
					$css				=	'/templates/default/' . $header . '.css';
				}

				if ( file_exists( $absPath . $css ) ) {
					$_CB_framework->document->addHeadStyleSheet( $livePath . $css );
				}

				if ( ! file_exists( $absPath . $js ) ) {
					$js					=	'/templates/default/' . $header . '.js';
				}

				if ( file_exists( $absPath . $js ) ) {
					$_CB_framework->document->addHeadScriptUrl( $livePath . $js );
				}

				$loaded[$template][]	=	$header;
			}

			// Override CSS File:
			if ( in_array( 'override', $headers ) && ( ! in_array( 'override', $loaded[$template] ) ) ) {
				$override				=	'/templates/' . $template . '/override.css';

				if ( file_exists( $absPath . $override ) ) {
					$_CB_framework->document->addHeadStyleSheet( $livePath . $override );
				}

				$loaded[$template][]	=	'override';
			}
		}

		return $return;
	}

	/**
	 * Returns a field name based off its group and row
	 *
	 * @param string      $groupName
	 * @param null|int    $index
	 * @param null|string $fieldName
	 * @return string
	 */
	static public function getGroupedName( $groupName, $index = null, $fieldName = null )
	{
		return ( strpos( $groupName, '_' ) === 0 ? null : '_' )
			   . \moscomprofilerHTML::htmlId( $groupName . ( $index !== null ? '[' . $index . ']' : null ) . ( $fieldName !== null ? '[' . $fieldName . ']' : null )  );
	}

	/**
	 * Returns the number of rows for a fieldgroup from $postData
	 *
	 * @param FieldTable $field
	 * @param array      $postData
	 * @return int
	 */
	static public function getGroupedCount( $field, $postData )
	{
		$fieldName				=	$field->get( 'name', null, GetterInterface::STRING );
		$count					=	Get::get( $postData, self::getGroupedName( $fieldName, null, 'count' ), 0, GetterInterface::INT );

		if ( ! $count ) {
			$groupedName		=	self::getGroupedName( $fieldName );
			$foundRows			=	array();

			foreach ( $postData as $k => $v ) {
				if ( strpos( $k, $groupedName ) !== 0 ) {
					continue;
				}

				if ( ! preg_match( '/^' . preg_quote( $groupedName ) . '__(\d+)/', $k, $rowMatch ) ) {
					continue;
				}

				$i				=	(int) $rowMatch[1];

				if ( in_array( $i, $foundRows ) ) {
					continue;
				}

				$foundRows[]	=	$i;
			}

			$count				=	count( $foundRows );
		}

		return $count;
	}

	/**
	 * Returns array of grouped fields for this field
	 * Also updates user object with grouped fields value if not from post
	 *
	 * @param FieldTable      $field
	 * @param UserTable       $user
	 * @param string          $reason     'profile' for user profile view, 'edit' for profile edit, 'register' for registration, 'search' for searches
	 * @param null|array      $postData
	 * @param null|string|int $index      first|start: first row, last|end: last row, int: row by index number
	 * @param bool            $fullAccess ignores field availability access checks
	 * @return array
	 */
	static public function getGroupedFields( $field, &$user, $reason, $postData = null, $index = null, $fullAccess = false )
	{
		global $_PLUGINS;

		$fieldId								=	$field->get( 'fieldid', 0, GetterInterface::INT );
		$fieldName								=	$field->get( 'name', null, GetterInterface::STRING );
		$fieldLimit								=	$field->params->get( 'repeat_limit', 5, GetterInterface::INT );

		if ( $reason == 'search' ) {
			$existingValues						=	new Registry();
		} else {
			$existingValues						=	new Registry( $user->get( $fieldName, null, GetterInterface::RAW ) );

			if ( ( $postData !== null )
				 && $field->params->get( 'repeat_ordering', false, GetterInterface::BOOLEAN )
				 && $existingValues->count()
			) {
				// This field can be reordered so lets check if the order was changed and update it if so:
				$reorderedValues				=	array();
				$reordered						=	false;

				foreach ( $existingValues as $i => $existingValue ) {
					/** @var ParamsInterface $existingValue */
					$rowIndex					=	Get::get( $postData, self::getGroupedName( $fieldName, $i, 'index' ), $i, GetterInterface::INT );

					if ( $i == $rowIndex ) {
						$reorderedValues[$i]	=	$existingValue->asArray();

						continue;
					}

					$reorderedValues[$rowIndex]	=	$existingValue->asArray();
					$reordered					=	true;
				}

				if ( $reordered )  {
					ksort( $reorderedValues );

					$existingValues				=	new Registry( $reorderedValues );
				}
			}
		}

		if ( $postData !== null ) {
			if ( $reason == 'search' ) {
				$count							=	$field->params->get( 'repeat_limit', 5, GetterInterface::INT );

				if ( ! $count ) {
					$count						=	$field->params->get( 'repeat_search_limit', 5, GetterInterface::INT );
				}

				if ( ! $count ) {
					$count						=	1;
				}
			} else {
				$count							=	self::getGroupedCount( $field, $postData );
			}

			$values								=	new Registry();

			for ( $i = 0, $n = $count; $i < $n; $i++ ) {
				$values->set( $i, array() );
			}
		} else {
			$values								=	$existingValues;

			if ( in_array( $index, array( 'first', 'start' ) ) ) {
				$index							=	0;
			} elseif ( in_array( $index, array( 'last', 'end' ) ) ) {
				$index							=	( $values->count() ? ( $values->count() - 1 ) : 0 );
			} elseif ( $index !== null ) {
				$index							=	(int) $index;
			}

			if ( ( $index !== null ) && ( ! $values->has( $index ) ) ) {
				// We're looking for a specific index, but it doesn't exist so lets just add it as empty so we can prepare the field:
				$values->set( $index, array() );
			}

			if ( ! $values->count() ) {
				if ( in_array( $reason, array( 'edit', 'register', 'search' ) ) )  {
					// No values exist so lets create a 0 index array:
					$values						=	new Registry( array( 0 => array() ) );
				} else {
					// Nothing to display so just return empty array:
					return array();
				}
			}
		}

		$user->set( '_isFieldGroup', true );

		$availableFields						=	CBuser::getInstance( $user->get( 'id', 0, GetterInterface::INT ), false )->_getCbTabs( false )->_getTabFieldsDb( null, $user, $reason, null, true, $fullAccess );

		$user->set( '_isFieldGroup', false );

		$groupedFieldIds						=	cbToArrayOfInt( array_filter( explode( '|*|', $field->params->get( 'repeat_fields', null, GetterInterface::STRING ) ) ) );
		$groupedFields							=	array();

		foreach ( $availableFields as $groupField ) {
			if ( ! in_array( $groupField->get( 'fieldid', 0, GetterInterface::INT ), $groupedFieldIds ) ) {
				continue;
			} elseif ( $groupField->get( 'fieldid', 0, GetterInterface::INT ) == $fieldId ) {
				continue;
			}

			$groupedFields[]					=	$groupField;
		}

		$fields									=	array();

		foreach ( $values as $i => $value ) {
			/** @var ParamsInterface $value */
			if ( $fieldLimit && ( ( $i + 1 ) > $fieldLimit ) ) {
				break;
			}

			$fields[$i]							=	array();

			foreach ( $groupedFields as $groupField ) {
				switch ( $reason ) {
					case 'edit':
						if ( ( ! Application::Cms()->getClientId() ) && ( ! $groupField->get( 'edit', 1, GetterInterface::INT ) ) ) {
							continue 2;
						}
						break;
					case 'profile':
					case 'list':
						if ( ! $groupField->get( 'profile', 1, GetterInterface::INT ) ) {
							continue 2;
						}
						break;
					case 'register':
						if ( ! $groupField->get( 'registration', 1, GetterInterface::INT ) ) {
							continue 2;
						}
						break;
					case 'search':
						if ( ! $groupField->get( 'searchable', 0, GetterInterface::INT ) ) {
							continue 2;
						}
						break;
				}

				// We only want to modify a copy of the cached field object so the original field object is unaltered by other usages:
				$groupField						=	clone $groupField;

				// Force off various integrations that are not compatible with field groups:
				$groupField->set( '_isGrouped', true ); // CB Field Groups

				if ( ! in_array( $reason, array( 'register', 'edit', 'search' ) ) ) {
					$groupField->set( '_noAjax', true ); // CB Core Fields Ajax
				}

				$groupFieldId					=	$groupField->get( 'fieldid', 0, GetterInterface::INT );
				$groupFieldName					=	$groupField->get( 'name', null, GetterInterface::STRING );
				$groupFieldType					=	$groupField->get( 'type', null, GetterInterface::STRING );

				// Lets other plugins know what field group this field is a part of and what row it's in:
				$groupField->set( '_fieldGroup', $field );
				$groupField->set( '_fieldGroupId', $fieldId );
				$groupField->set( '_fieldGroupName', $fieldName );
				$groupField->set( '_fieldGroupIndex', $i );
				$groupField->set( '_fieldGroupFields', $groupedFieldIds );

				$groupField->set( '_name', $groupFieldName );
				// Use a puesedo name with an underscore so when adding values to user object they'll never attempt to store directly:
				$groupField->set( 'name', self::getGroupedName( $fieldName, $i, $groupFieldName ) );
				$groupField->set( '_fieldid', $fieldId . '__' . $i . '__' . $groupFieldId );

				$groupFieldColumns				=	array();

				foreach ( $groupField->getTableColumns() as $col ) {
					// Use a puesedo column with an underscore so when adding values to user object they'll never attempt to store directly:
					$groupFieldColumns[$col]	=	self::getGroupedName( $fieldName, $i, $col );
				}

				$groupField->set( '_table', $groupField->get( 'table', null, GetterInterface::STRING ) );
				$groupField->set( 'table', null ); // We don't want any of this trying to store to a table with a column that doesn't exist
				$groupField->set( '_tablecolumns', $groupFieldColumns ); // pass the old to new column array map
				$groupField->set( 'tablecolumns', implode( ',', $groupFieldColumns ) );

				if ( $reason != 'search' ) {
					// We need to be sure the existing stored values are mapped properly to the user:
					foreach ( $groupFieldColumns as $oldCol => $newCol ) {
						switch ( $groupFieldType ) {
							case 'text':
							case 'textarea':
							case 'radio':
							case 'queryradio':
							case 'coderadio':
							case 'select':
							case 'queryselect':
							case 'codeselect':
							case 'emailaddress':
							case 'webaddress':
							case 'date':
							case 'datetime':
							case 'time':
							case 'password':
							case 'image':
							case 'file':
							case 'video':
							case 'audio':
							case 'gravatar':
								$type			=	GetterInterface::STRING;

								if ( $groupFieldType == 'image' ) {
									if ( strpos( $oldCol, 'approved' ) !== false ) {
										$type	=	GetterInterface::INT;
									} elseif ( strpos( $oldCol, 'position' ) !== false ) {
										$type	=	GetterInterface::INT;
									}
								}
								break;
							case 'editorta':
								$type			=	GetterInterface::HTML;
								break;
							case 'integer':
							case 'checkbox':
							case 'terms':
								$type			=	GetterInterface::INT;
								break;
							case 'points':
							case 'rating':
							case 'float':
								$type			=	GetterInterface::FLOAT;
								break;
							case 'multicheckbox':
							case 'querymulticheckbox':
							case 'codemulticheckbox':
							case 'multiselect':
							case 'querymultiselect':
							case 'codemultiselect':
							case 'tag':
							case 'querytag':
							case 'codetag':
							case 'fieldgroup':
							default:
								$type			=	GetterInterface::RAW;
								break;
						}

						$colValue				=	$existingValues->get( $i . '.' . $oldCol, null, $type );

						if ( preg_match( '/multicheckbox|multiselect|tag/', $groupFieldType ) ) {
							// Be sure multiselect field types are imploded to string and sanitized to string:
							if ( is_array( $colValue ) ) {
								$colValue		=	implode( '|*|', $colValue );
							}

							$colValue			=	Get::clean( $colValue, GetterInterface::STRING );
						} elseif ( $groupFieldType == 'fieldgroup' ) {
							// Be sure fieldgroups are using valid json otherwise reset it to empty json array:
							if ( is_string( $colValue ) ) {
								$colValue		=	json_decode( $colValue, true );
							}

							if ( $colValue ) {
								$colValue		=	json_encode( $colValue );
							} else {
								$colValue		=	'[]';
							}
						}

						$user->set( $newCol, $colValue );
					}
				}

				$fields[$i][$groupFieldName]	=		$groupField;
			}
		}

		$_PLUGINS->trigger( 'fieldgroups_onAfterFieldsFetch', array( $field, &$fields, &$user, $reason, $fullAccess ) );

		if ( $index !== null ) {
			// Specific index lookup so either return it if it exists or return nothing:
			if ( isset( $fields[$index] ) ) {
				return $fields[$index];
			}

			return array();
		}

		return $fields;
	}

	/**
	 * Maps a rows grouped field values to the user for substitution usages
	 *
	 * @param FieldTable[] $fields
	 * @param UserTable    $user
	 * @param null|array   $postData
	 * @return CBuser
	 */
	static public function mapGroupedColumns( $fields, &$user, &$postData = null )
	{
		foreach ( $fields as $field ) {
			foreach ( $field->get( '_tablecolumns', array(), GetterInterface::RAW ) as $oldCol => $newCol ) {
				$user->set( $oldCol, $user->get( $newCol, null, GetterInterface::RAW ) ); // Already sanitized by getGroupedFields

				if ( $postData ) {
					// Just 1 to 1 map the post data from generated column names to old column names for field processing behaviors (they'll handle sanitizing):
					if ( Get::has( $postData, $newCol ) ) {
						Get::set( $postData, $oldCol, Get::get( $postData, $newCol, null, GetterInterface::RAW ) );
					} elseif ( Get::has( $postData, $oldCol ) ) {
						Get::unsetKey( $postData, $oldCol );
					}
				}
			}
		}

		// Update the cbUser object temporarily to allow substitutions to work for repeat values:
		$cbUser				=	CBuser::getInstance( $user->get( 'id', 0, GetterInterface::INT ), false );

		// Note: you should reverse this after usage of mapGroupedColumns:
		$cbUser->_cbuser	=	$user;

		return $cbUser;
	}

	/**
	 * Checks if MYSQL supports JSON functions
	 *
	 * @return bool
	 */
	static public function canSearch()
	{
		global $_CB_database;

		if ( ! $_CB_database->versionCompare( '5.7.8' ) ) {
			return false;
		}

		if ( ! property_exists( 'cbSqlQueryPart', 'paths' ) ) {
			return false;
		}

		return true;
	}
}
