<?php
/**
* Community Builder (TM)
* @version $Id: $
* @package CommunityBuilder
* @copyright (C) 2004-2021 www.joomlapolis.com / Lightning MultiCom SA - and its licensors, all rights reserved
* @license http://www.gnu.org/licenses/old-licenses/gpl-2.0.html GNU/GPL version 2
*/

namespace CB\Plugin\AutoActions;

use CB\Database\Table\UserTable;
use CB\Plugin\AutoActions\Action\Action;
use CB\Plugin\AutoActions\Table\AutoActionTable;
use CBLib\Application\Application;
use CBLib\Input\Get;
use CBLib\Language\CBTxt;
use CBLib\Registry\GetterInterface;
use CBLib\Registry\ParamsInterface;
use CBLib\Registry\Registry;

\defined( 'CBLIB' ) or die();

class CBAutoActions
{

	/**
	 * @return Registry
	 */
	public static function getGlobalParams(): Registry
	{
		global $_PLUGINS;

		static $params	=	null;

		if ( ! $params ) {
			$plugin		=	$_PLUGINS->getLoadedPlugin( 'user', 'cbautoactions' );
			$params		=	new Registry();

			if ( $plugin ) {
				$params->load( $plugin->params );
			}
		}

		return $params;
	}

	/**
	 * Checks if eval is available
	 *
	 * @return bool
	 */
	public static function canEval(): bool
	{
		static $cache					=	null;

		if ( $cache === null ) {
			$cache						=	true;

			if ( \function_exists( 'ini_get' ) ) {
				$disabledFunctions		=	ini_get( 'disabled_functions' );

				if ( $disabledFunctions !== false ) {
					$cache				=	( ! \in_array( 'eval', explode( ',', $disabledFunctions ), true ) );
				}

				if ( $cache ) {
					$evalDisabled		=	ini_get( 'suhosin.executor.disable_eval' );

					if ( ( $evalDisabled === true ) || ( $evalDisabled === 1 ) ) {
						$cache			=	false;
					}
				}
			}
		}

		return $cache;
	}

	/**
	 * Attempts to execute code from a string
	 *
	 * @param null|string          $code
	 * @param null|AutoActionTable $autoaction
	 * @param null|UserTable       $user
	 * @param null|array           $variables
	 * @param null|Action          $action
	 * @param mixed                $content
	 * @return mixed
	 * @throws \Exception
	 */
	public static function outputCode( ?string $code, ?AutoActionTable &$autoaction = null, ?UserTable &$user = null, ?array &$variables = null, ?Action &$action = null, &$content = null )
	{
		if ( ! $code ) {
			return null;
		}

		// For B/C:
		$trigger				=	$autoaction;
		$vars					=	$variables;
		$input					=	Application::Input();

		ob_start();
		if ( self::canEval() ) {
			$function			=	'$function = function( $autoaction, $user, $variables, $action, $content, $trigger, $vars, $input ) { ' . $code . ' };';

			try {
				eval( $function );

				$return			=	$function( $autoaction, $user, $variables, $action, $content, $trigger, $vars, $input );
			} catch ( \Error $e ) {
				throw new \Exception( $e->getMessage() );
			}
		} else {
			// Last resort fallback; this is slow so hopefully it doesn't come to this:
			$temp				=	tmpfile();

			fwrite( $temp, "<?php \n" . $code  );

			$tempData			=	stream_get_meta_data( $temp );

			try {
				$return			=	include $tempData['uri'];
			} catch ( \Error $e ) {
				fclose( $temp );

				throw new \Exception( $e->getMessage() );
			}

			fclose( $temp );

			if ( ! \is_string( $return ) ) {
				$return			=	null;
			}
		}
		$echos					=	ob_get_clean();

		if ( $echos && ( \is_string( $return ) || ( $return === '' ) || ( $return === null ) ) ) {
			$return				=	$echos . $return;
		}

		return $return;
	}

	/**
	 * Trys to finder a user object in an array of variables
	 *
	 * @param array $variables
	 * @return UserTable
	 */
	public static function getUser( array $variables ): UserTable
	{
		$user						=	null;

		// Lets first try to find a user object in the variables directly:
		foreach ( $variables as $variable ) {
			if ( $variable instanceof UserTable ) {
				$user				=	$variable;
				break;
			}
		}

		// We failed to find a user object so lets try to parse for one from the variables:
		if ( ! $user ) {
			foreach ( $variables as $variable ) {
				$variable			=	self::prepareUser( $variable );

				if ( ! $variable instanceof UserTable ) {
					$variableUser	=	\CBuser::getUserDataInstance( $variable );

					if ( $variableUser->getInt( 'id', 0 ) ) {
						$user		=	$variableUser;
						break;
					}
				} elseif ( $variable->getInt( 'id', 0 ) ) {
					$user			=	$variable;
					break;
				}
			}
		}

		// Still can't find a user so lets just fallback to self:
		if ( ! $user ) {
			$user					=	\CBuser::getMyUserDataInstance();
		}

		return $user;
	}

	/**
	 * Trys to load a user object from a variable
	 *
	 * @param object|int $userVariable
	 * @param array      $skip
	 * @return UserTable|int
	 */
	public static function prepareUser( $userVariable, array $skip = [] )
	{
		if ( \is_object( $userVariable ) ) {
			if ( $userVariable instanceof UserTable ) {
				return $userVariable;
			}

			if ( isset( $userVariable->user_id ) && ( ! \in_array( 'user_id', $skip, true ) ) ) {
				return (int) $userVariable->user_id;
			}

			if ( isset( $userVariable->user ) && ( ! \in_array( 'user', $skip, true ) ) ) {
				return (int) $userVariable->user;
			}

			if ( isset( $userVariable->id ) && ( ! \in_array( 'id', $skip, true ) ) ) {
				return (int) $userVariable->id;
			}
		} elseif ( is_numeric( $userVariable ) && ( ! \in_array( 'int', $skip, true ) ) ) {
			return (int) $userVariable;
		}

		return \CBuser::getUserDataInstance( 0 );
	}

	/**
	 * Parses substitution extras array from available variables
	 *
	 * @param array $variables
	 * @param int $maxDepth
	 * @return array
	 */
	public static function getExtras( array $variables = [], int $maxDepth = 2 ): array
	{
		static $cache					=	null;

		if ( $cache === null ) {
			$cache						=	[];

			$get						=	Application::Input()->getNamespaceRegistry( 'get' );

			if ( $get ) {
				self::prepareExtras( 'get', $get->asArray(), $cache, $maxDepth );
			}

			$post						=	Application::Input()->getNamespaceRegistry( 'post' );

			if ( $post ) {
				self::prepareExtras( 'post', $post->asArray(), $cache, $maxDepth );
			}

			$files						=	Application::Input()->getNamespaceRegistry( 'files' );

			if ( $files ) {
				self::prepareExtras( 'files', $files->asArray(), $cache, $maxDepth );
			}

			$cookie						=	Application::Input()->getNamespaceRegistry( 'cookie' );

			if ( $cookie ) {
				self::prepareExtras( 'cookie', $cookie->asArray(), $cache, $maxDepth );
			}

			$server						=	Application::Input()->getNamespaceRegistry( 'server' );

			if ( $server ) {
				self::prepareExtras( 'server', $server->asArray(), $cache, $maxDepth );
			}

			$env						=	Application::Input()->getNamespaceRegistry( 'env' );

			if ( $env ) {
				self::prepareExtras( 'env', $env->asArray(), $cache, $maxDepth );
			}

			$session					=	Application::Session();

			if ( $session ) {
				self::prepareExtras( 'session', $session->asArray(), $cache, $maxDepth );
			}
		}

		$extras							=	[];

		foreach ( $variables as $key => $variable ) {
			if ( \is_object( $variable ) || \is_array( $variable ) ) {
				/** @var array|object $variable */
				if ( \is_object( $variable ) ) {
					if ( $variable instanceof ParamsInterface ) {
						$paramsArray	=	$variable->asArray();
					} else {
						$paramsArray	=	get_object_vars( $variable );
					}
				} else {
					$paramsArray		=	$variable;
				}

				self::prepareExtras( $key, $paramsArray, $extras, $maxDepth );
			} else {
				$extras[$key]			=	$variable;
			}
		}

		return array_merge( $cache, $extras );
	}

	/**
	 * Converts array or object into pathed extras substitutions
	 *
	 * @param null|string  $prefix
	 * @param array|object $items
	 * @param array        $extras
	 * @param int          $maxDepth
	 * @param int          $depth
	 */
	public static function prepareExtras( ?string $prefix, $items, array &$extras, int $maxDepth = 2, int $depth = 0 ): void
	{
		if ( ! $maxDepth ) {
			$maxDepth					=	2;
		}

		if ( $depth > $maxDepth ) {
			return;
		}

		foreach ( $items as $k => $v ) {
			if ( \is_string( $k ) && ( $k[0] === '_' ) ) {
				continue;
			}

			if ( \is_array( $v ) ) {
				$multi					=	false;

				foreach ( $v as $kv => $cv ) {
					if ( is_numeric( $kv ) ) {
						$kv				=	(int) $kv;
					}

					if ( \is_object( $cv ) || \is_array( $cv ) || ( $kv && ( ! \is_int( $kv ) ) ) ) {
						$multi			=	true;
					}
				}

				if ( ! $multi ) {
					$v					=	implode( '|*|', $v );
				}
			}

			if ( $v && ( ! \is_object( $v ) ) && ( ! \is_array( $v ) ) && ( ( $k === 'params' ) || ( \is_string( $v ) && ( ( $v[0] === '{' ) || ( $v[0] === '[' ) ) ) ) ) {
				$params					=	new Registry( $v );
				$json					=	$params->asArray();

				if ( \is_string( $v ) && ( ( $v[0] === '{' ) || ( $v[0] === '[' ) ) ) {
					// If we parsed from a string only use the resulting array if we actually found some JSON data:
					if ( $json ) {
						$v				=	$json;
					}
				} else {
					$v					=	$json;
				}
			}

			$k							=	'_' . ltrim( str_replace( ' ', '_', strtolower( trim( $k ) ) ), '_' );

			if ( ( ! \is_object( $v ) ) && ( ! \is_array( $v ) ) ) {
				$extras[$prefix . $k]	=	$v;
			} elseif ( $v ) {
				$depth++;

				if ( \is_object( $v ) ) {
					/** @var object $v */
					$subItems			=	get_object_vars( $v );
				} else {
					$subItems			=	$v;
				}

				self::prepareExtras( $prefix . $k, $subItems, $extras, $maxDepth, $depth );
			}
		}
	}

	/**
	 * Parses a string for PHP functions
	 *
	 * @param null|string    $input
	 * @param array          $vars
	 * @param array          $extraStrings
	 * @param null|UserTable $user
	 * @param null|\CBuser   $cbUser
	 * @return null|string
	 */
	public static function formatFunction( ?string $input, array $vars = [], array $extraStrings = [], ?UserTable $user = null, ?\CBuser $cbUser = null ): ?string
	{
		if ( ( ! $input ) || strpos( $input, '[cb:parse' ) === false ) {
			return $input;
		}

		$regex		=	'%\[cb:parse(?: +function="([^"/\[\] ]+)")?( +(?: ?[a-zA-Z-_]+="(?:[^"]|\\\\")+")+)?(?:(?:\s*/])|(?:]((?:[^\[]+|\[(?!/?cb:parse[^\]]*])|(?R))+)?\[/cb:parse]))%i';

		$input		=	preg_replace_callback( $regex, static function( array $matches ) use ( $vars, $extraStrings, $user, $cbUser )
							{
								$function								=	( $matches[1] ?? '' );
								$value									=	( isset( $matches[3] ) ? self::formatFunction( $matches[3], $vars, $extraStrings, $user, $cbUser ) : '' );

								if ( ! $function ) {
									return $value;
								}

								$options								=	new Registry();

								if ( isset( $matches[2] ) && preg_match_all( '/(?:([a-zA-Z-_]+)="((?:[^"]|\\\\\\\\")+)")+/i', $matches[2], $optionResults, PREG_SET_ORDER ) ) {
									foreach( $optionResults as $option ) {
										$k								=	( $option[1] ?? '' );
										$v								=	( isset( $option[2] ) ? self::formatFunction( $option[2], $vars, $extraStrings, $user, $cbUser ) : '' );

										if ( $k ) {
											$options->set( $k, $v );
										}
									}
								}

								$method									=	$options->getString( 'method', '' );

								$options->unsetEntry( 'method' );

								$return									=	'';

								switch ( $function ) {
									case 'prepare':
										$return							=	Application::Cms()->prepareHtmlContentPlugins( $value, 'autoaction', ( $user ? $user->getInt( 'id', 0 ) : 0 ) );
										break;
									case 'substitutions':
										if ( $cbUser ) {
											$return						=	$cbUser->replaceUserVars( $value, false, false, $extraStrings, false );
										} elseif ( $user ) {
											$return						=	\CBuser::getInstance( $user->getInt( 'id', 0 ), false )->replaceUserVars( $value, false, false, $extraStrings, false );
										} else {
											$return						=	\CBuser::getMyInstance()->replaceUserVars( $value, false, false, $extraStrings, false );
										}
										break;
									case 'translate':
										$return							=	CBTxt::T( $value );
										break;
									case 'clean':
										switch( $method ) {
											case 'cmd':
												$return					=	Get::clean( $value, GetterInterface::COMMAND );
												break;
											case 'numeric':
												$return					=	Get::clean( $value, GetterInterface::NUMERIC );
												break;
											case 'unit':
												$return					=	Get::clean( $value, GetterInterface::UINT );
												break;
											case 'int':
											case 'integer':
												$return					=	Get::clean( $value, GetterInterface::INT );
												break;
											case 'bool':
											case 'boolean':
												$return					=	Get::clean( $value, GetterInterface::BOOLEAN );
												break;
											case 'str':
											case 'string':
												$return					=	Get::clean( $value, GetterInterface::STRING );
												break;
											case 'html':
												$return					=	Get::clean( $value, GetterInterface::HTML );
												break;
											case 'float':
												$return					=	Get::clean( $value, GetterInterface::FLOAT );
												break;
											case 'base64':
												$return					=	Get::clean( $value, GetterInterface::BASE64 );
												break;
											case 'tags':
												$return					=	strip_tags( $value );
												break;
										}
										break;
									case 'convert':
										switch( $method ) {
											case 'uppercase':
												$return					=	strtoupper( $value );
												break;
											case 'uppercasewords':
												$return					=	ucwords( $value );
												break;
											case 'uppercasefirst':
												$return					=	ucfirst( $value );
												break;
											case 'lowercase':
												$return					=	strtolower( $value );
												break;
											case 'lowercasefirst':
												$return					=	lcfirst( $value );
												break;
										}
										break;
									case 'math':
										$return							=	self::formatMath( $value );

										switch( $method ) {
											case 'round':
												$return					=	round( $value, $options->getInt( 'decimal', 0 ) );
												break;
											case 'ceil':
												$return					=	ceil( $value );
												break;
											case 'floor':
												$return					=	floor( $value );
												break;
											case 'abs':
												$return					=	abs( $value );
												break;
											case 'number':
												$return					=	number_format( $value, $options->getInt( 'decimals', 0 ), $options->getString( 'decimal', '.' ), $options->getString( 'separator', ',' ) );
												break;
										}
										break;
									case 'time':
										if ( $options->has( 'time' ) ) {
											$return						=	Application::Date( ( is_numeric( $value ) ? (int) $value : $value ), 'UTC' )->modify( $options->getString( 'time', '' ) )->getTimestamp();
										} else {
											$return						=	Application::Date( ( is_numeric( $value ) ? (int) $value : $value ), 'UTC' )->getTimestamp();
										}
										break;
									case 'date':
										$offset							=	$options->getString( 'offset' );
										$return							=	cbFormatDate( ( is_numeric( $value ) ? (int) $value : $value ), (bool) $offset, ( ! ( $options->getString( 'time', 'true' ) === 'false' ) ), $options->getString( 'date-format' ), $options->getString( 'time-format' ), ( $offset !== 'true' ? $offset : null ) );
										break;
									case 'length':
										$return							=	\strlen( $value );
										break;
									case 'replace':
										$count							=	$options->getInt( 'count', 0 );

										if ( $options->has( 'pattern' ) ) {
											$return						=	( $options->has( 'count' ) ? preg_replace( $options->getRaw( 'pattern', '' ), $options->getRaw( 'replace', '' ), $value, $count ) : preg_replace( $options->getRaw( 'pattern', '' ), $options->getRaw( 'replace', '' ), $value ) );
										} else {
											$return						=	( $options->has( 'count' ) ? str_replace( $options->getRaw( 'search', '' ), $options->getRaw( 'replace', '' ), $value, $count ) : str_replace( $options->getRaw( 'search', '' ), $options->getRaw( 'replace', '' ), $value ) );
										}
										break;
									case 'position':
										switch( $options->getString( 'occurrence', '' ) ) {
											case 'last':
												$return					=	strrpos( $value, $options->getRaw( 'search', '' ) );
												break;
											case 'first':
											default:
												$return					=	strpos( $value, $options->getRaw( 'search', '' ) );
												break;
										}
										break;
									case 'occurrence':
										$return							=	strstr( $value, $options->getRaw( 'search', '' ) );
										break;
									case 'repeat':
										$return							=	str_repeat( $value, $options->getInt( 'count', 0 ) );
										break;
									case 'extract':
										$return							=	( $options->has( 'length' ) ? substr( $value, $options->getInt( 'start', 0 ), $options->getInt( 'length', 0 ) ) : substr( $value, $options->getInt( 'start', 0 ) ) );
										break;
									case 'trim':
										switch( $options->getString( 'direction', '' ) ) {
											case 'left':
												$return					=	( $options->has( 'characters' ) ? ltrim( $value, $options->getString( 'characters', '' ) ) : ltrim( $value ) );
												break;
											case 'right':
												$return					=	( $options->has( 'characters' ) ? rtrim( $value, $options->getString( 'characters', '' ) ) : rtrim( $value ) );
												break;
											default:
												$return					=	( $options->has( 'characters' ) ? trim( $value, $options->getString( 'characters', '' ) ) : trim( $value ) );
												break;
										}
										break;
									case 'encode':
										switch( $method ) {
											case 'cslashes':
												$return					=	addcslashes( $value, $options->getString( 'characters', '' ) );
												break;
											case 'slashes':
												$return					=	addslashes( $value );
												break;
											case 'entity':
												$return					=	htmlentities( $value );
												break;
											case 'html':
												$return					=	htmlspecialchars( $value );
												break;
											case 'url':
												$return					=	urlencode( $value );
												break;
											case 'base64':
												$return					=	base64_encode( $value );
												break;
											case 'md5':
												$return					=	md5( $value );
												break;
											case 'sha1':
												$return					=	sha1( $value );
												break;
											case 'password':
												$user					=	new UserTable();

												$return					=	$user->hashAndSaltPassword( $value );
												break;
										}
										break;
									case 'decode':
										switch( $method ) {
											case 'cslashes':
												$return					=	stripcslashes( $value );
												break;
											case 'slashes':
												$return					=	stripslashes( $value );
												break;
											case 'entity':
												$return					=	html_entity_decode( $value );
												break;
											case 'html':
												$return					=	htmlspecialchars_decode( $value );
												break;
											case 'url':
												$return					=	urldecode( $value );
												break;
											case 'base64':
												$return					=	base64_decode( $value );
												break;
										}
										break;
									default:
										$class							=	$options->getString( 'class', '' );
										$subFunction					=	'';
										$static							=	false;
										$result							=	'';

										if ( strpos( $function, '::' ) !== false ) {
											[ $class, $function ]		=	explode( '::', $function, 2 );

											$static						=	true;
										} elseif ( strpos( $class, '::' ) !== false ) {
											$subFunction				=	$function;

											[ $class, $function ]		=	explode( '::', $class, 2 );

											$static						=	true;
										}

										$functionVariables				=	$options->asArray();

										if ( ! $functionVariables ) {
											$functionVariables[]		=	$value;
										}

										if ( $class ) {
											if ( isset( $functionVariables['class'] ) ) {
												unset( $functionVariables['class'] );
											}

											$object						=	null;

											if ( isset( $vars[$class] ) && \is_object( $vars[$class] ) ) {
												$object					=	$vars[$class];
												$class					=	\get_class( $object );
											}

											if ( $static ) {
												if ( $subFunction ) {
													if ( \is_callable( [ $class, $function ] ) ) {
														$object			=	\call_user_func_array( [ $class, $function ], [] );

														if ( method_exists( $object, $subFunction ) ) {
															$result		=	\call_user_func_array( [ $object, $subFunction ], $functionVariables );
														}
													}
												} elseif ( \is_callable( [ $class, $function ] ) ) {
													$result				=	\call_user_func_array( [ $class, $function ], $functionVariables );
												}
											} elseif ( $object || class_exists( $class ) ) {
												if ( ! $object ) {
													$object				=	new $class();

													if ( $value && method_exists( $object, 'load' ) ) {
														$object->load( $value );
													}
												}

												if ( method_exists( $object, $function ) ) {
													$result				=	\call_user_func_array( [ $object, $function ], $functionVariables );
												}
											}
										} elseif ( \function_exists( $function ) ) {
											$result						=	\call_user_func_array( $function, $functionVariables );
										}

										if ( $method && \is_object( $result ) && method_exists( $result, $method ) ) {
											$result						=	\call_user_func_array( [ $result, $method ], $functionVariables );
										}

										if ( ( ! \is_array( $result ) ) && ( ! \is_object( $result ) ) ) {
											$return						=	$result;
										}
										break;
								}

								return $return;
							},
							$input );

		return self::formatFunction( $input, $vars, $extraStrings, $user, $cbUser );
	}

	/**
	 * Parses a string for math expressions
	 *
	 * @param null|string $value
	 * @return null|string
	 */
	public static function formatMath( ?string $value ): ?string
	{
		if ( preg_match( '/(?:\(\s*)([^(]+?)(?:\s*\))/', $value, $expression ) ) {
			// Sub-Expression
			$value					=	str_replace( $expression[0], self::formatMath( $expression[1] ), $value );

			return self::formatMath( $value );
		}

		if ( preg_match( '/([+-]?\d*\.?\d+)\s*\*\s*([+-]?\d*\.?\d+)/', $value, $expression ) ) {
			// Multiply
			$left					=	( isset( $expression[1] ) ? trim( $expression[1] ) : null );
			$right					=	( isset( $expression[2] ) ? trim( $expression[2] ) : null );
			$value					=	str_replace( $expression[0], ( $left * $right ), $value );

			return self::formatMath( $value );
		}

		if ( preg_match( '%([+-]?\d*\.?\d+)\s*/\s*([+-]?\d*\.?\d+)%', $value, $expression ) ) {
			// Divide:
			$left					=	( isset( $expression[1] ) ? trim( $expression[1] ) : null );
			$right					=	( isset( $expression[2] ) ? trim( $expression[2] ) : null );
			$value					=	str_replace( $expression[0], ( $left / $right ), $value );

			return self::formatMath( $value );
		}

		if ( preg_match( '/([+-]?\d*\.?\d+)\s*([+%-])\s*([+-]?\d*\.?\d+)/', $value, $expression ) ) {
			// Add, Subtract, Modulus:
			$left					=	( isset( $expression[1] ) ? trim( $expression[1] ) : null );
			$operator				=	( isset( $expression[2] ) ? trim( $expression[2] ) : null );
			$right					=	( isset( $expression[3] ) ? trim( $expression[3] ) : null );

			if ( $operator ) {
				switch( $operator ) {
					case '+':
						$value		=	str_replace( $expression[0], ( $left + $right ), $value );
						break;
					case '-':
						$value		=	str_replace( $expression[0], ( $left - $right ), $value );
						break;
					case '%':
						$value		=	str_replace( $expression[0], ( $left % $right ), $value );
						break;
				}
			}

			return self::formatMath( $value );
		}

		return $value;
	}

	/**
	 * Compares condition values based off operator
	 *
	 * @param null|string|array $value
	 * @param int               $operator
	 * @param null|string|array $input
	 * @param null|string       $delimiter
	 * @return bool
	 */
	public static function getConditionMatch( $value, int $operator, $input, ?string $delimiter = null ): bool
	{
		if ( \is_array( $value ) ) {
			$value			=	implode( ( $delimiter ?: '|*|' ), $value );
		}

		if ( \is_array( $input ) ) {
			$input			=	implode( ( $delimiter ?: '|*|' ), $input );
		}

		$value				=	trim( $value );
		$input				=	trim( $input );

		switch ( $operator ) {
			case 1:
				/** @noinspection TypeUnsafeComparisonInspection */
				$match		=	( $value != $input );
				break;
			case 2:
				$match		=	( $value > $input );
				break;
			case 3:
				$match		=	( $value < $input );
				break;
			case 4:
				$match		=	( $value >= $input );
				break;
			case 5:
				$match		=	( $value <= $input );
				break;
			case 6:
				$match		=	( ! $value );
				break;
			case 7:
				$match		=	( $value );
				break;
			case 8:
				if ( $delimiter ) {
					$match	=	( \in_array( $input, explode( $delimiter, $value ), true ) );
				} else {
					if ( $input === '' ) {
						// Can't have an empty needle so fallback to simple equal to check:
						return self::getConditionMatch( $value, 0, $input, $delimiter );
					}

					$match	=	( stristr( $value, $input ) );
				}
				break;
			case 9:
				if ( $delimiter ) {
					$match	=	( ! \in_array( $input, explode( $delimiter, $value ), true ) );
				} else {
					if ( $input === '' ) {
						// Can't have an empty needle so fallback to simple not equal to check:
						return self::getConditionMatch( $value, 1, $input, $delimiter );
					}

					$match	=	( stripos( $value, $input ) === false );
				}
				break;
			case 10:
				if ( $input === '' ) {
					// Can't have an empty regexp so fallback to simple equal to check:
					return self::getConditionMatch( $value, 0, $input, $delimiter );
				}

				$match		=	( preg_match( $input, $value ) );
				break;
			case 11:
				if ( $input === '' ) {
					// Can't have an empty regexp so fallback to simple not equal to check:
					return self::getConditionMatch( $value, 1, $input, $delimiter );
				}

				$match		=	( ! preg_match( $input, $value ) );
				break;
			case 12:
				if ( $delimiter ) {
					$match	=	( \in_array( $value, explode( $delimiter, $input ), true ) );
				} else {
					if ( $value === '' ) {
						// Can't have an empty needle so fallback to simple equal to check:
						return self::getConditionMatch( $value, 0, $input, $delimiter );
					}

					$match	=	( stristr( $input, $value ) );
				}
				break;
			case 13:
				if ( $delimiter ) {
					$match	=	( ! \in_array( $value, explode( $delimiter, $input ), true ) );
				} else {
					if ( $value === '' ) {
						// Can't have an empty needle so fallback to simple not equal to check:
						return self::getConditionMatch( $value, 1, $input, $delimiter );
					}

					$match	=	( stripos( $input, $value ) === false );
				}
				break;
			case 0:
			default:
				/** @noinspection TypeUnsafeComparisonInspection */
				$match		=	( $value == $input );
				break;
		}

		return (bool) $match;
	}

	/**
	 * Returns string name of an operator from int
	 *
	 * @param int|string $operator
	 * @return string
	 */
	public static function getOperator( $operator ): string
	{
		switch ( $operator ) {
			case 1:
				return CBTxt::T( 'Not Equal To' );
			case 2:
				return CBTxt::T( 'Greater Than' );
			case 3:
				return CBTxt::T( 'Less Than' );
			case 4:
				return CBTxt::T( 'Greater Than or Equal To' );
			case 5:
				return CBTxt::T( 'Less Than or Equal To' );
			case 6:
				return CBTxt::T( 'Empty' );
			case 7:
				return CBTxt::T( 'Not Empty' );
			case 8:
				return CBTxt::T( 'Does Contain' );
			case 9:
				return CBTxt::T( 'Does Not Contain' );
			case 10:
				return CBTxt::T( 'Is REGEX' );
			case 11:
				return CBTxt::T( 'Is Not REGEX' );
			case 12:
				return CBTxt::T( 'Is In' );
			case 13:
				return CBTxt::T( 'Is Not In' );
			case 0:
			default:
				return CBTxt::T( 'Equal To' );
		}
	}

	/**
	 * Returns the current URL
	 * 
	 * @return string
	 */
	public static function getCurrentURL(): string
	{
		$isHttps		=	( isset( $_SERVER['HTTPS'] ) && ( ! empty( $_SERVER['HTTPS'] ) ) && ( $_SERVER['HTTPS'] !== 'off' ) );
		$url			=	'http' . ( $isHttps ? 's' : '' ) . '://' . $_SERVER['HTTP_HOST'];

		if ( ( ! empty( $_SERVER['PHP_SELF'] ) ) && ( ! empty( $_SERVER['REQUEST_URI'] ) ) ) {
			$url		.=	$_SERVER['REQUEST_URI'];
		} else {
			$url		.=	$_SERVER['SCRIPT_NAME'];

			if ( isset( $_SERVER['QUERY_STRING'] ) && ( ! empty( $_SERVER['QUERY_STRING'] ) ) ) {
				$url	.=	'?' . $_SERVER['QUERY_STRING'];
			}
		}

		return cbUnHtmlspecialchars( preg_replace( '/[\\\"\\\'][\\s]*javascript:(.*)[\\\"\\\']/', '""', preg_replace( '/eval\((.*)\)/', '', htmlspecialchars( urldecode( $url ) ) ) ) );
	}

	/**
	 * Encodes a string to URL safe
	 *
	 * @param null|string $str
	 * @return string
	 */
	public static function escapeURL( ?string $str ): string
	{
		return urlencode( trim( (string) $str ) );
	}

	/**
	 * Encodes a string to XML safe
	 *
	 * @param null|string $str
	 * @return string
	 */
	public static function escapeXML( ?string $str ): string
	{
		return htmlspecialchars( trim( (string) $str ), ENT_COMPAT );
	}

	/**
	 * Encodes a string to SQL safe
	 *
	 * @param null|string $str
	 * @return string
	 */
	public static function escapeSQL( ?string $str ): string
	{
		global $_CB_database;

		return $_CB_database->getEscaped( $str );
	}

	/**
	 * Returns the internal action URL for firing an action
	 *
	 * @param int $id
	 * @return string
	 */
	public static function loadInternalActionURL( int $id ): string
	{
		global $_CB_framework;

		if ( ! $id ) {
			return '';
		}

		return '<a href="' . $_CB_framework->pluginClassUrl( 'cbautoactions', true, [ 'action' => 'action', 'actions' => $id ], 'html', 0, true ) . '" target="_blank" class="text-wrapall">' . $_CB_framework->pluginClassUrl( 'cbautoactions', false, [ 'action' => 'action', 'actions' => $id ], 'html', 0, true ) . '</a>';
	}

	/**
	 * Returns the internal batch URL for firing an action
	 *
	 * @param int $id
	 * @return string
	 */
	public static function loadInternalBatchURL( int $id ): string
	{
		global $_CB_framework;

		if ( ! $id ) {
			return '';
		}

		return '<a href="' . $_CB_framework->pluginClassUrl( 'cbautoactions', true, [ 'action' => 'batch', 'batches' => $id, 'token' => md5( $_CB_framework->getCfg( 'secret' ) ) ], 'raw', 0, true ) . '" target="_blank" class="text-wrapall">' . $_CB_framework->pluginClassUrl( 'cbautoactions', false, [ 'action' => 'batch', 'batches' => $id, 'token' => md5( $_CB_framework->getCfg( 'secret' ) ) ], 'raw', 0, true ) . '</a>';
	}

	/**
	 * Checks if the system plugin is installed and published
	 *
	 * @return bool
	 */
	public static function botExistsCheck(): bool
	{
		global $_CB_database;

		$query	=	"SELECT " . $_CB_database->NameQuote( 'enabled' )
				.	"\n FROM " . $_CB_database->NameQuote( '#__extensions' )
				.	"\n WHERE " . $_CB_database->NameQuote( 'type' ) . " = " . $_CB_database->Quote( 'plugin' )
				.	"\n AND " . $_CB_database->NameQuote( 'element' ) . " = " . $_CB_database->Quote( 'cbautoactionsbot' );
		$_CB_database->setQuery( $query, 0, 1 );
		if ( $_CB_database->loadResult() ) {
			return true;
		}

		return false;
	}

	/**
	 * @return bool
	 */
	public static function checkCBSubsExists(): bool
	{
		global $_PLUGINS;

		if ( $_PLUGINS->getLoadedPlugin( 'user', 'cbpaidsubscriptions' ) ) {
			return true;
		}

		return false;
	}

	/**
	 * @return array
	 */
	public static function getCBSubsPlans(): array
	{
		if ( ! self::checkCBSubsExists() ) {
			return [];
		}

		$plansMgr				=	\cbpaidPlansMgr::getInstance();
		$plans					=	$plansMgr->loadPublishedPlans( null, true, 'any', null );
		$plansList				=	[];

		if ( $plans ) {
			$plansList			=	[];

			foreach ( $plans as $k => $plan ) {
				$plansList[]	=	\moscomprofilerHTML::makeOption( (string) $k, $plan->getString( 'alias', '' ) );
			}
		}

		return $plansList;
	}

	/**
	 * @param int $userId
	 * @return array
	 */
	public static function getCBSubsSubscriptionsStatus( int $userId ): array
	{
		if ( ( ! $userId ) || ( ! self::checkCBSubsExists() ) ) {
			return [];
		}

		static $cache						=	[];

		if ( ! isset( $cache[$userId] ) ) {
			$cache[$userId]					=	[];

			$paidUserExtension				=	\cbpaidUserExtension::getInstance( $userId );
			$subscriptions					=	$paidUserExtension->getUserSubscriptions();

			foreach ( $subscriptions as $subscription ) {
				$planId						=	$subscription->getInt( 'plan_id', 0 );

				$cache[$userId][$planId]	=	$subscription->getString( 'status', '' );
			}
		}

		return $cache[$userId];
	}
}