<?php
/**
* Community Builder (TM)
* @version $Id: $
* @package CommunityBuilder
* @copyright (C) 2004-2021 www.joomlapolis.com / Lightning MultiCom SA - and its licensors, all rights reserved
* @license http://www.gnu.org/licenses/old-licenses/gpl-2.0.html GNU/GPL version 2
*/

namespace CB\Plugin\AutoActions\Action;

use CB\Database\Table\UserTable;
use CB\Plugin\GroupJive\CBGroupJive;
use CB\Plugin\GroupJive\Table\CategoryTable;
use CB\Plugin\GroupJive\Table\GroupTable;
use CBLib\Language\CBTxt;
use CBLib\Registry\ParamsInterface;

\defined( 'CBLIB' ) or die();

class GroupjiveAction extends Action
{

	/**
	 * @param UserTable $user
	 */
	public function execute( UserTable $user ): void
	{
		global $_CB_database;

		if ( ! $this->installed() ) {
			$this->error( CBTxt::T( 'AUTO_ACTION_GROUPJIVE_NOT_INSTALLED', ':: Action [action] :: CB GroupJive is not installed', [ '[action]' => $this->autoaction()->getId() ] ) );
			return;
		}

		foreach ( $this->autoaction()->getParams()->subTree( 'groupjive' ) as $row ) {
			/** @var ParamsInterface $row */
			$owner									=	$row->getString( 'owner', '' );

			if ( ! $owner ) {
				$owner								=	$user->getInt( 'id', 0 );
			} else {
				$owner								=	(int) $this->string( $user, $owner );
			}

			if ( $user->getInt( 'id', 0 ) !== $owner ) {
				$actionOwner						=	\CBuser::getUserDataInstance( $owner );
			} else {
				$actionOwner						=	$user;
			}

			$userId									=	$row->getString( 'user', '' );

			if ( ! $userId ) {
				$userId								=	$user->getInt( 'id', 0 );
			} else {
				$userId								=	(int) $this->string( $actionOwner, $userId );
			}

			if ( $user->getInt( 'id', 0 ) !== $userId ) {
				$actionUser							=	\CBuser::getUserDataInstance( $userId );
			} else {
				$actionUser							=	$user;
			}

			switch( $row->getInt( 'mode', 1 ) ) {
				case 3:
					if ( ! $actionOwner->getInt( 'id', 0 ) ) {
						$this->error( CBTxt::T( 'AUTO_ACTION_GROUPJIVE_NO_OWNER', ':: Action [action] :: CB GroupJive skipped due to missing owner', [ '[action]' => $this->autoaction()->getId() ] ) );
						continue 2;
					}

					$name							=	$this->string( $actionOwner, $row->getString( 'name', '' ) );

					if ( ! $name ) {
						$this->error( CBTxt::T( 'AUTO_ACTION_GROUPJIVE_NO_NAME', ':: Action [action] :: CB GroupJive skipped due to missing name', [ '[action]' => $this->autoaction()->getId() ] ) );
						continue 2;
					}

					$category						=	new CategoryTable();

					$category->load( [ 'name' => $name ] );

					if ( $category->getInt( 'id', 0 ) ) {
						continue 2;
					}

					$category->set( 'published', 1 );
					$category->set( 'name', $name );
					$category->set( 'description', $this->string( $actionOwner, $row->getString( 'description', '' ) ) );
					$category->set( 'access', 1 );
					$category->set( 'create_access', 0 );
					$category->set( 'types', $row->getString( 'types', '1|*|2|*|3' ) );
					$category->set( 'ordering', 1 );

					$category->params()->set( 'forums', 1 );

					$category->set( 'params', $category->params()->asJson() );

					if ( ! $category->store() ) {
						$this->error( CBTxt::T( 'AUTO_ACTION_GROUPJIVE_FAILED', ':: Action [action] :: CB GroupJive failed to save. Error: [error]', [ '[action]' => $this->autoaction()->getId(), '[error]' => $category->getError() ] ) );
						continue 2;
					}
					break;
				case 2:
					if ( ! $actionOwner->getInt( 'id', 0 ) ) {
						$this->error( CBTxt::T( 'AUTO_ACTION_GROUPJIVE_NO_OWNER', ':: Action [action] :: CB GroupJive skipped due to missing owner', [ '[action]' => $this->autoaction()->getId() ] ) );
						continue 2;
					}

					$categoryId						=	$row->getString( 'category', '-1' );

					$category						=	new CategoryTable();

					if ( $categoryId !== '0' ) {
						if ( ( $categoryId === '-1' ) || ( $categoryId === '' ) ) {
							$name					=	$this->string( $actionOwner, $row->getString( 'category_name', '' ) );

							if ( ! $name ) {
								$this->error( CBTxt::T( 'AUTO_ACTION_GROUPJIVE_NO_CAT_NAME', ':: Action [action] :: CB GroupJive skipped due to missing category name', [ '[action]' => $this->autoaction()->getId() ] ) );
								continue 2;
							}

							$category->load( [ 'name' => $name ] );

							if ( ! $category->getInt( 'id', 0 ) ) {
								$category->set( 'published', 1 );
								$category->set( 'name', $name );
								$category->set( 'description', $this->string( $actionOwner, $row->getString( 'category_description', '' ) ) );
								$category->set( 'access', 1 );
								$category->set( 'create_access', 0 );
								$category->set( 'types', $row->getString( 'category_types', '1|*|2|*|3' ) );
								$category->set( 'ordering', 1 );

								$category->params()->set( 'forums', 1 );

								$category->set( 'params', $category->params()->asJson() );

								if ( ! $category->store() ) {
									$this->error( CBTxt::T( 'AUTO_ACTION_GROUPJIVE_FAILED', ':: Action [action] :: CB GroupJive failed to save. Error: [error]', [ '[action]' => $this->autoaction()->getId(), '[error]' => $category->getError() ] ) );
									continue 2;
								}
							}
						} else {
							$category->load( (int) $categoryId );
						}

						if ( ! $category->getInt( 'id', 0 ) ) {
							$this->error( CBTxt::T( 'AUTO_ACTION_GROUPJIVE_NO_CATEGORY', ':: Action [action] :: CB GroupJive skipped due to missing category', [ '[action]' => $this->autoaction()->getId() ] ) );
							continue 2;
						}

						$categoryId					=	$category->getInt( 'id', 0 );
					}

					$name							=	$this->string( $actionOwner, $row->getString( 'name', '' ) );

					if ( ! $name ) {
						$this->error( CBTxt::T( 'AUTO_ACTION_GROUPJIVE_NO_NAME', ':: Action [action] :: CB GroupJive skipped due to missing name', [ '[action]' => $this->autoaction()->getId() ] ) );
						continue 2;
					}

					$group							=	new GroupTable();
					$join							=	false;

					if ( $row->getBool( 'unique', true ) ) {
						$group->load( [ 'category' => (int) $categoryId, 'user_id' => $actionOwner->getInt( 'id', 0 ), 'name' => $name ] );
					} else {
						$group->load( [ 'category' => (int) $categoryId, 'name' => $name ] );

						if ( $row->getBool( 'autojoin', true ) ) {
							$join					=	true;
						}
					}

					if ( ! $group->getInt( 'id', 0 ) ) {
						$group->set( 'published', 1 );
						$group->set( 'category', (int) $categoryId );
						$group->set( 'user_id', $actionOwner->getInt( 'id', 0 ) );
						$group->set( 'name', $name );
						$group->set( 'description', $this->string( $actionOwner, $row->getString( 'description', '' ) ) );
						$group->set( 'type', $row->getInt( 'type', 1 ) );
						$group->set( 'ordering', 1 );

						$group->params()->set( 'users', 1 );
						$group->params()->set( 'invites', 1 );
						$group->params()->set( 'video', 1 );
						$group->params()->set( 'events', 1 );
						$group->params()->set( 'wall', 1 );
						$group->params()->set( 'photo', 1 );
						$group->params()->set( 'file', 1 );
						$group->params()->set( 'forums', 1 );

						$group->set( 'params', $group->params()->asJson() );

						if ( ! $group->store() ) {
							$this->error( CBTxt::T( 'AUTO_ACTION_GROUPJIVE_FAILED', ':: Action [action] :: CB GroupJive failed to save. Error: [error]', [ '[action]' => $this->autoaction()->getId(), '[error]' => $group->getError() ] ) );
							continue 2;
						}
					} elseif ( $join ) {
						$userId						=	$row->getString( 'group_user', '' );

						if ( ! $userId ) {
							$userId					=	$user->getInt( 'id', 0 );
						} else {
							$userId					=	(int) $this->string( $actionOwner, $userId );
						}

						if ( ! $userId ) {
							$this->error( CBTxt::T( 'AUTO_ACTION_GROUPJIVE_NO_OWNER', ':: Action [action] :: CB GroupJive skipped due to missing user', [ '[action]' => $this->autoaction()->getId() ] ) );
							continue 2;
						}

						$groupUser					=	new \CB\Plugin\GroupJive\Table\UserTable( $_CB_database );

						$groupUser->load( [ 'group' => $group->getInt( 'id', 0 ), 'user_id' => $userId ] );

						if ( $groupUser->getInt( 'id', 0 ) ) {
							$this->error( CBTxt::T( 'AUTO_ACTION_GROUPJIVE_NO_GROUP', ':: Action [action] :: CB GroupJive skipped due to missing group', [ '[action]' => $this->autoaction()->getId() ] ) );
							continue 2;
						}

						$groupUser->set( 'user_id', $userId );
						$groupUser->set( 'group', $group->getInt( 'id', 0 ) );
						$groupUser->set( 'status', $row->getInt( 'group_status', 1 ) );

						if ( ! $groupUser->store() ) {
							$this->error( CBTxt::T( 'AUTO_ACTION_GROUPJIVE_FAILED', ':: Action [action] :: CB GroupJive failed to save. Error: [error]', [ '[action]' => $this->autoaction()->getId(), '[error]' => $groupUser->getError() ] ) );
							continue 2;
						}
					}
					break;
				case 4:
					if ( ! $actionUser->getInt( 'id', 0 ) ) {
						$this->error( CBTxt::T( 'AUTO_ACTION_GROUPJIVE_NO_OWNER', ':: Action [action] :: CB GroupJive skipped due to missing user', [ '[action]' => $this->autoaction()->getId() ] ) );
						continue 2;
					}

					$groups							=	$row->getString( 'groups', '' );

					if ( ! $groups ) {
						$this->error( CBTxt::T( 'AUTO_ACTION_GROUPJIVE_NO_GROUPS', ':: Action [action] :: CB GroupJive skipped due to missing groups', [ '[action]' => $this->autoaction()->getId() ] ) );
						continue 2;
					}

					if ( $groups === 'all' ) {
						$query						=	'SELECT *'
													.	"\n FROM " . $_CB_database->NameQuote( '#__groupjive_users' )
													.	"\n WHERE " . $_CB_database->NameQuote( 'user_id' ) . " = " . $actionUser->getInt( 'id', 0 )
													.	"\n AND " . $_CB_database->NameQuote( 'status' ) . " != 4";
						$_CB_database->setQuery( $query );
						$groups						=	$_CB_database->loadObjectList( null, '\CB\Plugin\GroupJive\Table\UserTable', [ $_CB_database ] );
					} else {
						$groups						=	cbToArrayOfInt( explode( '|*|', $groups ) );
					}

					foreach ( $groups as $groupId ) {
						if ( $groupId instanceof \CB\Plugin\GroupJive\Table\UserTable ) {
							$groupUser				=	$groupId;
						} else {
							$group					=	new GroupTable();

							$group->load( (int) $groupId );

							if ( ! $group->getInt( 'id', 0 ) ) {
								continue;
							}

							$groupUser				=	new \CB\Plugin\GroupJive\Table\UserTable( $_CB_database );

							$groupUser->load( [ 'group' => $group->getInt( 'id', 0 ), 'user_id' => $actionUser->getInt( 'id', 0 ) ] );
						}

						if ( ( ! $groupUser->getInt( 'id', 0 ) ) || ( $groupUser->getInt( 'status', 0 ) === 4 ) ) {
							continue;
						}

						$groupUser->delete();
					}
					break;
				case 1:
				default:
					if ( ! $actionUser->getInt( 'id', 0 ) ) {
						$this->error( CBTxt::T( 'AUTO_ACTION_GROUPJIVE_NO_OWNER', ':: Action [action] :: CB GroupJive skipped due to missing user', [ '[action]' => $this->autoaction()->getId() ] ) );
						continue 2;
					}

					$groups							=	$row->getString( 'groups', '' );

					if ( ! $groups ) {
						$this->error( CBTxt::T( 'AUTO_ACTION_GROUPJIVE_NO_GROUPS', ':: Action [action] :: CB GroupJive skipped due to missing groups', [ '[action]' => $this->autoaction()->getId() ] ) );
						continue 2;
					}

					if ( $groups === 'all' ) {
						$query						=	'SELECT g.*'
													.	"\n FROM " . $_CB_database->NameQuote( '#__groupjive_groups' ) . " AS g"
													.	"\n LEFT JOIN " . $_CB_database->NameQuote( '#__groupjive_users' ) . " AS u"
													.	' ON u.' . $_CB_database->NameQuote( 'user_id' ) . ' = ' . $actionUser->getInt( 'id', 0 )
													.	' AND u.' . $_CB_database->NameQuote( 'group' ) . ' = g.' . $_CB_database->NameQuote( 'id' )
													.	"\n WHERE g." . $_CB_database->NameQuote( 'published' ) . " = 1"
													.	"\n AND g." . $_CB_database->NameQuote( 'user_id' ) . " != " . $actionUser->getInt( 'id', 0 )
													.	"\n AND u." . $_CB_database->NameQuote( 'id' ) . " IS NULL";
						$_CB_database->setQuery( $query );
						$groups						=	$_CB_database->loadObjectList( null, '\CB\Plugin\GroupJive\Table\GroupTable', [ $_CB_database ] );
					} else {
						$groups						=	cbToArrayOfInt( explode( '|*|', $groups ) );
					}

					foreach ( $groups as $groupId ) {
						$groupUser					=	new \CB\Plugin\GroupJive\Table\UserTable( $_CB_database );

						if ( $groupId instanceof GroupTable ) {
							$group					=	$groupId;
						} else {
							$group					=	new GroupTable();

							$group->load( (int) $groupId );

							if ( ! $group->getInt( 'id', 0 ) ) {
								continue;
							}
						}

						$groupUser->load( [ 'group' => $group->getInt( 'id', 0 ), 'user_id' => $actionUser->getInt( 'id', 0 ) ] );

						if ( ! $groupUser->getInt( 'id', 0 ) ) {
							$groupUser->set( 'user_id', $actionUser->getInt( 'id', 0 ) );
							$groupUser->set( 'group', $group->getInt( 'id', 0 ) );
						} elseif ( ( $groupUser->getInt( 'status', 1 ) === $row->getInt( 'status', 1 ) ) || ( $groupUser->getInt( 'status', 1 ) === 4 ) ) {
							continue;
						}

						$groupUser->set( 'status', $row->getInt( 'status', 1 ) );

						if ( ! $groupUser->store() ) {
							$this->error( CBTxt::T( 'AUTO_ACTION_GROUPJIVE_FAILED', ':: Action [action] :: CB GroupJive failed to save. Error: [error]', [ '[action]' => $this->autoaction()->getId(), '[error]' => $groupUser->getError() ] ) );
						}
					}
					break;
			}
		}
	}

	/**
	 * @return bool
	 */
	public function categoriesLimited(): bool
	{
		if ( $this->installed() && method_exists( '\CB\Plugin\GroupJive\CBGroupJive', 'checkTooManyCategories' ) ) {
			return CBGroupJive::checkTooManyCategories();
		}

		return false;
	}

	/**
	 * @return array
	 */
	public function categories(): array
	{
		$options		=	[];

		if ( $this->installed() ) {
			$options	=	CBGroupJive::getCategoryOptions();
		}

		return $options;
	}

	/**
	 * @return bool
	 */
	public function groupsLimited(): bool
	{
		if ( $this->installed() && method_exists( '\CB\Plugin\GroupJive\CBGroupJive', 'checkTooManyGroups' ) ) {
			return CBGroupJive::checkTooManyGroups();
		}

		return false;
	}

	/**
	 * @return array
	 */
	public function groups(): array
	{
		$options		=	[];

		if ( $this->installed() ) {
			$options	=	CBGroupJive::getGroupOptions();
		}

		return $options;
	}

	/**
	 * @return bool
	 */
	public function installed(): bool
	{
		global $_CB_framework, $_PLUGINS;

		static $installed			=	null;

		if ( $installed === null ) {
			if ( $_PLUGINS->getLoadedPlugin( 'user', 'cbgroupjive' ) ) {
				if ( file_exists( $_CB_framework->getCfg( 'absolute_path' ) . '/components/com_comprofiler/plugin/user/plug_cbgroupjive/cbgroupjive.class.php' ) ) {
					$installed		=	false;
				} else {
					$installed		=	true;
				}
			} else {
				$installed			=	false;
			}
		}

		return $installed;
	}
}