<?php
/**
* Community Builder (TM)
* @version $Id: $
* @package CommunityBuilder
* @copyright (C) 2004-2021 www.joomlapolis.com / Lightning MultiCom SA - and its licensors, all rights reserved
* @license http://www.gnu.org/licenses/old-licenses/gpl-2.0.html GNU/GPL version 2
*/

namespace CB\Plugin\GroupJive\Table;

use CBLib\Application\Application;
use CB\Plugin\GroupJive\CBGroupJive;
use CBLib\Database\Table\Table;
use CBLib\Language\CBTxt;
use CBLib\Registry\Registry;

defined('CBLIB') or die();

class InviteTable extends Table
{
	/** @var int  */
	public $id				=	null;
	/** @var int  */
	public $user_id			=	null;
	/** @var int  */
	public $group			=	null;
	/** @var string  */
	public $message			=	null;
	/** @var string  */
	public $invited			=	null;
	/** @var string  */
	public $accepted		=	null;
	/** @var string  */
	public $code			=	null;
	/** @var string  */
	public $email			=	null;
	/** @var string  */
	public $user			=	null;
	/** @var string  */
	public $params			=	null;

	/**
	 * Table name in database
	 *
	 * @var string
	 */
	protected $_tbl			=	'#__groupjive_invites';

	/**
	 * Primary key(s) of table
	 *
	 * @var string
	 */
	protected $_tbl_key		=	'id';

	/**
	 * @return bool
	 */
	public function check()
	{
		if ( ! $this->getInt( 'user_id', 0 ) ) {
			$this->setError( CBTxt::T( 'Owner not specified!' ) );

			return false;
		}

		if ( ! $this->getInt( 'group', 0 ) ) {
			$this->setError( CBTxt::T( 'Group not specified!' ) );

			return false;
		}

		if ( ( $this->getString( 'email', '' ) === '' ) && ( ! $this->getInt( 'user', 0 ) ) ) {
			$this->setError( CBTxt::T( 'Recipient not specified or invalid!' ) );

			return false;
		}

		$group				=	$this->group();

		if ( ! $group->getInt( 'id', 0 ) ) {
			$this->setError( CBTxt::T( 'Group does not exist!' ) );

			return false;
		}

		if ( ! $this->getInt( 'id', 0 ) ) {
			$user			=	new \CB\Database\Table\UserTable();

			if ( $this->getString( 'email' ) ) {
				$user->load( array( 'email' => $this->getString( 'email' ) ) );
			} elseif ( $this->getInt( 'user', 0 ) ) {
				$user->load( $this->getInt( 'user', 0 ) );

				if ( ! $user->getInt( 'id', 0 ) ) {
					$this->setError( CBTxt::T( 'The user you are inviting does not exist!' ) );

					return false;
				}
			}

			if ( $user->getInt( 'id', 0 ) ) {
				if ( $this->getInt( 'user_id', 0 ) === $user->getInt( 'id', 0 ) ) {
					$this->setError( CBTxt::T( 'You can not invite your self!' ) );

					return false;
				}

				if ( $group->getInt( 'user_id', 0 ) === $user->getInt( 'id', 0 ) ) {
					$this->setError( CBTxt::T( 'You can not invite the group owner!' ) );

					return false;
				}

				if ( $user->getInt( 'block', 0 ) || ( ! $user->getInt( 'approved', 0 ) ) || ( ! $user->getInt( 'confirmed', 0 ) ) ) {
					$this->setError( CBTxt::T( 'The user you are inviting does not exist!' ) );

					return false;
				}

				$groupUser	=	new UserTable();

				$groupUser->load( array( 'group' => $this->getInt( 'group', 0 ), 'user_id' => $user->getInt( 'id', 0 ) ) );

				if ( $groupUser->getInt( 'id', 0 ) ) {
					$this->setError( CBTxt::T( 'The user you are inviting already belongs to this group!' ) );

					return false;
				}
			}

			$invite			=	new InviteTable();

			if ( $this->getString( 'email' ) ) {
				$invite->load( array( 'group' => $this->getInt( 'group', 0 ), 'email' => $this->getString( 'email' ) ) );

				if ( $invite->getInt( 'id', 0 ) ) {
					$this->setError( CBTxt::T( 'The email address you are inviting has already been invited to this group!' ) );

					return false;
				}
			} elseif ( $this->getInt( 'user', 0 ) ) {
				$invite->load( array( 'group' => $this->getInt( 'group', 0 ), 'user' => $this->getInt( 'user', 0 ) ) );

				if ( $invite->getInt( 'id', 0 ) ) {
					$this->setError( CBTxt::T( 'The user you are inviting has already been invited to this group!' ) );

					return false;
				}
			}
		}

		return true;
	}

	/**
	 * @param bool $updateNulls
	 * @return bool
	 */
	public function store( $updateNulls = false )
	{
		global $_PLUGINS;

		$new		=	( ! $this->getInt( 'id', 0 ) );
		$old		=	new self();

		$this->set( 'code', $this->getString( 'code', md5( uniqid( '', true ) ) ) );

		if ( ! $new ) {
			$old->load( $this->getInt( 'id', 0 ) );

			$_PLUGINS->trigger( 'gj_onBeforeUpdateInvite', array( &$this, $old ) );
		} else {
			$_PLUGINS->trigger( 'gj_onBeforeCreateInvite', array( &$this ) );
		}

		if ( ! parent::store( $updateNulls ) ) {
			return false;
		}

		if ( ! $new ) {
			$_PLUGINS->trigger( 'gj_onAfterUpdateInvite', array( $this, $old ) );
		} else {
			$this->send();

			$_PLUGINS->trigger( 'gj_onAfterCreateInvite', array( $this ) );
		}

		return true;
	}

	/**
	 * @param null|int $id
	 * @return bool
	 */
	public function delete( $id = null )
	{
		global $_PLUGINS;

		$_PLUGINS->trigger( 'gj_onBeforeDeleteInvite', array( &$this ) );

		if ( ! parent::delete( $id ) ) {
			return false;
		}

		$_PLUGINS->trigger( 'gj_onAfterDeleteInvite', array( $this ) );

		return true;
	}

	/**
	 * @return Registry
	 */
	public function params()
	{
		if ( ! ( $this->getRaw( '_params' ) instanceof Registry ) ) {
			$this->set( '_params', new Registry( $this->getRaw( 'params' ) ) );
		}

		return $this->getRaw( '_params' );
	}

	/**
	 * @return GroupTable
	 */
	public function group()
	{
		return CBGroupJive::getGroup( $this->getInt( 'group', 0 ) );
	}

	/**
	 * @return bool
	 */
	public function invited()
	{
		return ( $this->getString( 'invited' ) && ( $this->getString( 'invited' ) !== $this->getDbo()->getNullDate() ) );
	}

	/**
	 * @return bool
	 */
	public function accepted()
	{
		return ( $this->getString( 'accepted' ) && ( $this->getString( 'accepted' ) !== $this->getDbo()->getNullDate() ) );
	}

	/**
	 * @return bool
	 */
	public function canResend()
	{
		if ( $this->accepted() ) {
			return false;
		}

		if ( ( ! $this->invited() ) || Application::Application()->isClient( 'administrator' ) ) {
			return true;
		}

		$days	=	CBGroupJive::getGlobalParams()->getInt( 'groups_invites_resend', 7 );

		if ( ! $days ) {
			return false;
		}

		$diff	=	Application::Date( 'now', 'UTC' )->diff( $this->getString( 'invited' ) );

		return ! ( ( $diff === false ) || ( $diff->days < $days ) );
	}

	/**
	 * @return bool
	 */
	public function send()
	{
		if ( $this->accepted() ) {
			$this->setError( CBTxt::T( 'Invite already accepted!' ) );

			return false;
		}

		if ( $this->invited() && ( ! $this->canResend() ) ) {
			$this->setError( CBTxt::T( 'Invite already sent!' ) );

			return false;
		}

		if ( ( $this->group()->getInt( 'published', 1 ) !== 1 ) || ( ! $this->group()->category()->getInt( 'published', 1 ) ) ) {
			$this->setError( CBTxt::T( 'Can not invite to an unpublished group!' ) );

			return false;
		}

		if ( $this->getInt( 'user', 0 ) ) {
			$to		=	$this->getInt( 'user', 0 );
		} else {
			$to		=	$this->getString( 'email' );
		}

		CBGroupJive::sendNotification( 'user_invite', 4, $this->getInt( 'user_id', 0 ), $to, CBTxt::T( 'Group invite' ), CBTxt::T( 'GROUP_INVITE_MESSAGE', "You have been invited to join the group [group] by [user]!\n\n[message]", array( '[message]' => htmlspecialchars( $this->getString( 'message' ) ) ) ), $this->group() );

		$this->set( 'invited', $this->getString( 'invited', Application::Database()->getUtcDateTime() ) );

		if ( ! $this->store() ) {
			return false;
		}

		return true;
	}

	/**
	 * @return bool
	 */
	public function accept()
	{
		if ( $this->accepted() ) {
			return true;
		}

		$userId				=	$this->getInt( 'user', 0 );

		if ( ! $userId ) {
			$email			=	$this->getString( 'email' );

			if ( $email ) {
				$user		=	new \CB\Database\Table\UserTable();

				$user->loadByEmail( $email );

				$userId		=	$user->getInt( 'id', 0 );
			}
		}

		if ( ! $userId ) {
			return false;
		}

		$row				=	new UserTable();

		$row->load( array( 'user_id' => $userId, 'group' => $this->getInt( 'group', 0 ) ) );

		if ( ! $row->getInt( 'id', 0 ) ) {
			$row->set( 'user_id', $userId );
			$row->set( 'group', $this->getInt( 'group', 0 ) );
			$row->set( 'status', 1 );

			if ( ! $row->store() ) {
				$this->setError( $row->getError() );

				return false;
			}
		}

		$this->set( 'accepted', Application::Database()->getUtcDateTime() );

		if ( ! $this->store() ) {
			return false;
		}

		return true;
	}
}