<?php
/**
* Community Builder (TM)
* @version $Id: $
* @package CommunityBuilder
* @copyright (C) 2004-2021 www.joomlapolis.com / Lightning MultiCom SA - and its licensors, all rights reserved
* @license http://www.gnu.org/licenses/old-licenses/gpl-2.0.html GNU/GPL version 2
*/

namespace CB\Plugin\GroupJive\Table;

use CBLib\Database\Table\OrderedTable;
use CBLib\Language\CBTxt;
use CBLib\Registry\Registry;
use CB\Plugin\GroupJive\CBGroupJive;
use CBLib\Image\Color;

defined('CBLIB') or die();

class CategoryTable extends OrderedTable
{
	/** @var int  */
	public $id				=	null;
	/** @var string  */
	public $canvas			=	null;
	/** @var string  */
	public $logo			=	null;
	/** @var string  */
	public $name			=	null;
	/** @var string  */
	public $description		=	null;
	/** @var string  */
	public $types			=	null;
	/** @var int  */
	public $access			=	null;
	/** @var int  */
	public $create_access	=	null;
	/** @var string  */
	public $css				=	null;
	/** @var int  */
	public $published		=	null;
	/** @var int  */
	public $ordering		=	null;
	/** @var string  */
	public $params			=	null;

	/** @var Registry  */
	protected $_params		=	null;

	/**
	 * Table name in database
	 *
	 * @var string
	 */
	protected $_tbl			=	'#__groupjive_categories';

	/**
	 * Primary key(s) of table
	 *
	 * @var string
	 */
	protected $_tbl_key		=	'id';

	/**
	 * Ordering keys and for each their ordering groups.
	 * E.g.; array( 'ordering' => array( 'tab' ), 'ordering_registration' => array() )
	 * @var array
	 */
	protected $_orderings	=	array( 'ordering' );

	/**
	 * @return bool
	 */
	public function check()
	{
		if ( $this->getString( 'name', '' ) === '' ) {
			$this->setError( CBTxt::T( 'Name not specified!' ) );

			return false;
		}

		if ( ! $this->getString( 'types' ) ) {
			$this->setError( CBTxt::T( 'Types not specified!' ) );

			return false;
		}

		return true;
	}

	/**
	 * @param bool $updateNulls
	 * @return bool
	 */
	public function store( $updateNulls = false )
	{
		global $_PLUGINS;

		$new	=	( ! $this->getInt( 'id', 0 ) );
		$old	=	new self();

		if ( ! $new ) {
			$old->load( $this->getInt( 'id', 0 ) );

			$_PLUGINS->trigger( 'gj_onBeforeUpdateCategory', array( &$this, $old ) );
		} else {
			$_PLUGINS->trigger( 'gj_onBeforeCreateCategory', array( &$this ) );
		}

		if ( ! parent::store( $updateNulls ) ) {
			return false;
		}

		if ( ! CBGroupJive::uploadImage( 'canvas', $this ) ) {
			return false;
		}

		if ( ! CBGroupJive::uploadImage( 'logo', $this ) ) {
			return false;
		}

		if ( ! parent::store( $updateNulls ) ) {
			return false;
		}

		if ( ! $new ) {
			$_PLUGINS->trigger( 'gj_onAfterUpdateCategory', array( $this, $old ) );
		} else {
			$_PLUGINS->trigger( 'gj_onAfterCreateCategory', array( $this ) );
		}

		return true;
	}

	/**
	 * @param null|self $object
	 * @return self|bool
	 */
	public function copy( $object = null )
	{
		global $_CB_framework, $_PLUGINS;

		if ( $object === null ) {
			$object			=	clone $this;
		}

		$old				=	new self();

		$old->load( $object->getInt( 'id', 0 ) );

		$_PLUGINS->trigger( 'gj_onBeforeCopyCategory', array( &$object, $old ) );

		$copy				=	parent::copy( $object );

		if ( ! $copy ) {
			return false;
		}

		// Copy the groups in this category:
		$query				=	'SELECT *'
							.	"\n FROM " . $this->getDbo()->NameQuote( '#__groupjive_groups' )
							.	"\n WHERE " . $this->getDbo()->NameQuote( 'category' ) . " = " . $old->getInt( 'id', 0 );
		$this->getDbo()->setQuery( $query );
		$groups				=	$this->getDbo()->loadObjectList( null, '\CB\Plugin\GroupJive\Table\GroupTable', array( $this->getDbo() ) );

		/** @var GroupTable[] $groups */
		foreach ( $groups as $group ) {
			$group->set( 'category', $object->getInt( 'id', 0 ) );

			$group->copy();
		}

		// Copy the canvas and logo:
		if ( $object->getString( 'canvas' ) || $object->getString( 'logo' ) ) {
			$basePath		=	$_CB_framework->getCfg( 'absolute_path' ) . '/images/comprofiler/plug_cbgroupjive';
			$oldPath		=	$basePath . '/' . $old->getInt( 'id', 0 );
			$newPath		=	$basePath . '/' . $object->getInt( 'id', 0 );

			if ( is_dir( $oldPath ) ) {
				CBGroupJive::createDirectory( $basePath, $object->getInt( 'id', 0 ) );
				CBGroupJive::copyDirectory( $oldPath, $newPath );
			}
		}

		$_PLUGINS->trigger( 'gj_onAfterCopyCategory', array( $object, $old ) );

		return $copy;
	}

	/**
	 * @param null|int $id
	 * @return bool
	 */
	public function delete( $id = null )
	{
		global $_CB_framework, $_PLUGINS;

		$_PLUGINS->trigger( 'gj_onBeforeDeleteCategory', array( &$this ) );

		if ( ! parent::delete( $id ) ) {
			return false;
		}

		// Delete groups in this category:
		$query				=	'SELECT *'
							.	"\n FROM " . $this->getDbo()->NameQuote( '#__groupjive_groups' )
							.	"\n WHERE " . $this->getDbo()->NameQuote( 'category' ) . " = " . $this->getInt( 'id', 0 );
		$this->getDbo()->setQuery( $query );
		$groups				=	$this->getDbo()->loadObjectList( null, '\CB\Plugin\GroupJive\Table\GroupTable', array( $this->getDbo() ) );

		/** @var GroupTable[] $groups */
		foreach ( $groups as $group ) {
			$group->delete();
		}

		// Delete canvas and logo:
		if ( $this->getString( 'canvas' ) || $this->getString( 'logo' ) ) {
			CBGroupJive::deleteDirectory( $_CB_framework->getCfg( 'absolute_path' ) . '/images/comprofiler/plug_cbgroupjive/' . $this->getInt( 'id', 0 ) );
		}

		$_PLUGINS->trigger( 'gj_onAfterDeleteCategory', array( $this ) );

		return true;
	}

	/**
	 * @return Registry
	 */
	public function params()
	{
		if ( ! ( $this->getRaw( '_params' ) instanceof Registry ) ) {
			$this->set( '_params', new Registry( $this->getRaw( 'params' ) ) );
		}

		return $this->getRaw( '_params' );
	}

	/**
	 * @param bool        $thumbnail
	 * @param bool        $html
	 * @param bool        $linked
	 * @param null|string $classes
	 * @return string
	 */
	public function canvas( $thumbnail = false, $html = true, $linked = false, $classes = null )
	{
		global $_CB_framework;

		static $cache		=	array();

		$id					=	$this->getInt( 'id', 0 ) . $thumbnail;

		if ( ! isset( $cache[$id] ) ) {
			$default		=	CBGroupJive::getGlobalParams()->getString( 'categories_canvas', 'canvas.png' );

			if ( ! $default ) {
				$default	=	'canvas.png';
			}

			$image			=	null;

			if ( $this->getString( 'canvas' ) ) {
				$path		=	'/images/comprofiler/plug_cbgroupjive/' . $this->getInt( 'id', 0 ) . '/' . ( $thumbnail ? 'tn' : null ) . preg_replace( '/[^-a-zA-Z0-9_.]/', '', $this->getString( 'canvas' ) );

				if ( file_exists( $_CB_framework->getCfg( 'absolute_path' ) . $path ) ) {
					$image	=	$_CB_framework->getCfg( 'live_site' ) . $path;
				}
			}

			if ( ! $image ) {
				$image		=	CBGroupJive::getDefaultImage( $default, $thumbnail );
			}

			$cache[$id]		=	$image;
		}

		$canvas				=	$cache[$id];

		if ( ( ! $canvas ) || ( $canvas === 'none' ) ) {
			return null;
		}

		if ( $html ) {
			if ( $canvas === 'color' ) {
				$canvas		=	'<div style="background: linear-gradient( 0deg, ' . htmlspecialchars( Color::stringToHex( $this->getString( 'name' ) ) ) . ' 0%, ' . htmlspecialchars( Color::stringToHex( $this->getString( 'name' ), 0.9 ) ) . ' 100% );" class="cbImgCanvas cbImgCanvasInitial' . ( $thumbnail ? ' cbThumbCanvas' : ' cbFullCanvas' ) . ' gjCanvas gjCanvasDefault' . ( $classes ? ' ' . htmlspecialchars( $classes ) : null ) . '"></div>';
			} else {
				$canvas		=	'<div style="background-image: url(' . htmlspecialchars( $canvas ) . ')" class="cbImgCanvas' . ( $thumbnail ? ' cbThumbCanvas' : ' cbFullCanvas' ) . ' gjCanvas' . ( $this->getString( 'canvas' ) ? ' gjCanvasCustom' : ' gjCanvasDefault' ) . ( $classes ? ' ' . htmlspecialchars( $classes ) : null ) . '"></div>';
			}
		}

		if ( $linked ) {
			$canvas			=	'<a href="' . $_CB_framework->pluginClassUrl( 'cbgroupjive', true, array( 'action' => 'categories', 'func' => 'show', 'id' => $this->getInt( 'id', 0 ) ) ) . '">' . $canvas . '</a>';
		}

		return $canvas;
	}

	/**
	 * @param bool        $thumbnail
	 * @param bool        $html
	 * @param bool        $linked
	 * @param null|string $classes
	 * @return string
	 */
	public function logo( $thumbnail = false, $html = true, $linked = false, $classes = null )
	{
		global $_CB_framework;

		static $cache		=	array();

		$id					=	$this->getInt( 'id', 0 ) . $thumbnail;

		if ( ! isset( $cache[$id] ) ) {
			$default		=	CBGroupJive::getGlobalParams()->getString( 'categories_logo', 'logo.png' );

			if ( ! $default ) {
				$default	=	'logo.png';
			}

			$image			=	null;

			if ( $this->getString( 'logo' ) ) {
				$path		=	'/images/comprofiler/plug_cbgroupjive/' . $this->getInt( 'id', 0 ) . '/' . ( $thumbnail ? 'tn' : null ) . preg_replace( '/[^-a-zA-Z0-9_.]/', '', $this->getString( 'logo' ) );

				if ( file_exists( $_CB_framework->getCfg( 'absolute_path' ) . $path ) ) {
					$image	=	$_CB_framework->getCfg( 'live_site' ) . $path;
				}
			}

			if ( ! $image ) {
				$image		=	CBGroupJive::getDefaultImage( $default, $thumbnail );
			}

			$cache[$id]		=	$image;
		}

		$logo				=	$cache[$id];

		if ( ( ! $logo ) || ( $logo === 'none' ) ) {
			return null;
		}

		if ( $html ) {
			switch ( CBGroupJive::getGlobalParams()->getString( 'categories_logo_style', 'roundedbordered' ) ) {
				case 'rounded':
					$classes	.=	' rounded';
					break;
				case 'roundedbordered':
					$classes	.=	' img-thumbnail';
					break;
				case 'circle':
					$classes	.=	' rounded-circle';
					break;
				case 'circlebordered':
					$classes	.=	' img-thumbnail rounded-circle';
					break;
			}

			$logo			=	'<img alt="' . htmlspecialchars( CBTxt::T( 'Logo' ) ) . '" src="' . htmlspecialchars( $logo ) . '" class="cbImgPict' . ( $thumbnail ? ' cbThumbPict' : ' cbFullPict' ) . ' gjLogo' . ( $this->getString( 'logo' ) ? ' gjLogoCustom' : ' gjLogoDefault' ) . ( $classes ? ' ' . htmlspecialchars( $classes ) : null ) . '" />';
		}

		if ( $linked ) {
			$logo			=	'<a href="' . $_CB_framework->pluginClassUrl( 'cbgroupjive', true, array( 'action' => 'categories', 'func' => 'show', 'id' => $this->getInt( 'id', 0 ) ) ) . '">' . $logo . '</a>';
		}

		return $logo;
	}
}