<?php
/**
 * Community Builder (TM)
 * @version $Id: $
 * @package CommunityBuilder
 * @copyright (C) 2004-2021 www.joomlapolis.com / Lightning MultiCom SA - and its licensors, all rights reserved
 * @license http://www.gnu.org/licenses/old-licenses/gpl-2.0.html GNU/GPL version 2
 */

namespace CB\Plugin\Code\Trigger;

use CB\Plugin\Code\CBCodeField;
use CBLib\Input\Get;
use CBLib\Registry\Registry;
use CBLib\Registry\ParamsInterface;
use CB\Database\Table\UserTable;
use CB\Database\Table\FieldTable;
use CBLib\Language\CBTxt;
use CBLib\Application\Application;
use CBLib\Registry\GetterInterface;

if ( ! ( defined( '_VALID_CB' ) || defined( '_JEXEC' ) || defined( '_VALID_MOS' ) ) ) { die( 'Direct Access to this location is not allowed.' ); }

class FieldTrigger extends \cbFieldHandler
{

	/**
	 * Checks if the field can be code validated
	 *
	 * @param FieldTable $field
	 * @param UserTable  $user
	 * @param string     $reason
	 * @param bool       $checkAjax
	 * @return bool
	 */
	private function canValidate( &$field, &$user, $reason, $checkAjax = true )
	{
		if ( ! ( $user instanceof UserTable ) ) {
			$user				=	new UserTable();
		}

		if ( ( $field instanceof FieldTable )
			 && ( ( ! Application::Cms()->getClientId() ) || ( Application::Cms()->getClientId() && Application::Config()->get( 'adminrequiredfields', true, GetterInterface::BOOLEAN ) ) )
			 && in_array( $reason, array( 'edit', 'register' ) )
		) {
			if ( ! ( $field->params instanceof ParamsInterface ) ) {
				$field->params	=	new Registry( $field->params );
			}

			$readOnly			=	$field->get( 'readonly', false, GetterInterface::BOOLEAN );

			if ( $field->get( 'name', null, GetterInterface::STRING ) == 'username' ) {
				if ( ! Application::Config()->get( 'usernameedit', true, GetterInterface::BOOLEAN ) ) {
					$readOnly	=	true;
				}
			}

			if ( ( ( ! $readOnly ) || ( $reason == 'register' ) || Application::Cms()->getClientId() )
				 && $field->params->get( 'code_validate', false, GetterInterface::BOOLEAN )
				 && ( ( $checkAjax && $field->params->get( 'code_validate_ajax', false, GetterInterface::BOOLEAN ) ) || ( ! $checkAjax ) )
			) {
				return true;
			}
		}

		return false;
	}

	/**
	 * Checks if the field can be code auto completed
	 *
	 * @param FieldTable $field
	 * @param UserTable  $user
	 * @param string     $reason
	 * @return bool
	 */
	private function canAutoComplete( &$field, &$user, $reason )
	{
		if ( ! ( $user instanceof UserTable ) ) {
			$user				=	new UserTable();
		}

		if ( ( $field instanceof FieldTable )
			 && ( $field->get( 'type', null, GetterInterface::STRING ) == 'text' )
			 && in_array( $reason, array( 'edit', 'register' ) )
		) {
			if ( ! ( $field->params instanceof ParamsInterface ) ) {
				$field->params	=	new Registry( $field->params );
			}

			$readOnly			=	$field->get( 'readonly', false, GetterInterface::BOOLEAN );

			if ( $field->get( 'name', null, GetterInterface::STRING ) == 'username' ) {
				if ( ! Application::Config()->get( 'usernameedit', true, GetterInterface::BOOLEAN ) ) {
					$readOnly	=	true;
				}
			}

			if ( ( ( ! $readOnly ) || ( $reason == 'register' ) || Application::Cms()->getClientId() )
				 && $field->params->get( 'code_autocomplete', false, GetterInterface::BOOLEAN )
			) {
				return true;
			}
		}

		return false;
	}

	/**
	 * Sends the field value through the code and tests for validity
	 *
	 * @param FieldTable   $field
	 * @param UserTable    $user
	 * @param string       $reason
	 * @param mixed        $value
	 * @param null|\CBuser $cbUser
	 * @return bool
	 */
	private function codeValidate( $field, $user, $reason, $value, $cbUser = null )
	{
		$tempUser				=	null;

		if ( ! $cbUser ) {
			$cbUser				=	\CBuser::getInstance( $user->get( 'id', 0, GetterInterface::INT ), false );
			$tempUser			=	$cbUser->_cbuser;

			$cbUser->_cbuser	=	$user;
		}

		$code					=	$cbUser->replaceUserVars( $field->params->get( 'code_validate_code', '', GetterInterface::RAW ), false, false, array( 'reason' => $reason, 'value' => $value ), false );

		if ( $tempUser ) {
			$cbUser->_cbuser	=	$tempUser;
		}

		if ( ! $code ) {
			return true;
		}

		try {
			$valid				=	CBCodeField::outputCode( $code, $field, $user, $reason, $value );
		} catch ( \Exception $e ) {
			if ( Application::MyUser()->isGlobalModerator() ) {
				$user->setError( $e->getMessage() );
			}

			$valid				=	false;
		}

		return $valid;
	}

	/**
	 * Checks if the field is empty based off fieldtype
	 * 
	 * @param FieldTable $field
	 * @param mixed      $value
	 * @return bool
	 */
	private function isEmpty( $field, $value )
	{
		if ( ( $value === '' ) || ( $value === null ) ) {
			return true;
		}

		switch ( $field->get( 'type', null, GetterInterface::STRING ) ) {
			case 'fieldgroup':
				return ( $value === '[]' );
				break;
			case 'date':
				return ( $value === '0000-00-00' );
				break;
			case 'datetime':
				return ( $value === '0000-00-00 00:00:00' );
				break;
			case 'time':
				return ( $value === '00:00:00' );
				break;
			case 'integer':
			case 'points':
			case 'checkbox':
			case 'terms':
			case 'counter':
				return ( (int) $value === 0 );
				break;
			case 'rating':
			case 'float':
				return ( (float) $value === 0 );
				break;
		}

		return false;
	}

	/**
	 * Direct access to field for custom operations, like for Ajax
	 *
	 * WARNING: direct unchecked access, except if $user is set, then check well for the $reason ...
	 *
	 * @param FieldTable     $field
	 * @param null|UserTable $user
	 * @param array          $postdata
	 * @param string         $reason 'profile' for user profile view, 'edit' for profile edit, 'register' for registration, 'search' for searches
	 * @return string
	 */
	public function getResponse( &$field, &$user, &$postdata, $reason )
	{
		/** @var FieldTable[] $fields */
		static $fields							=	array();

		if ( ( $this->input( 'function', null, GetterInterface::STRING ) == 'codevalidate' ) && $this->canValidate( $field, $user, $reason ) ) {
			$addFields							=	cbToArrayOfInt( array_filter( explode( '|*|', $field->params->get( 'code_validate_fields', null, GetterInterface::STRING ) ) ) );

			if ( ! ( $user instanceof UserTable ) ) {
				$user							=	new UserTable();
			}

			$value								=	Get::get( $postdata, 'value', null, GetterInterface::RAW );

			if ( is_array( $value ) ) {
				$value							=	implode( '|*|', $value );
			}

			$value								=	Get::clean( $value, GetterInterface::STRING );

			if ( $this->isEmpty( $field, $value ) ) {
				return null;
			}

			$cbUser								=	\CBuser::getInstance( $user->get( 'id', 0, GetterInterface::INT ), false );

			if ( $addFields ) {
				$tempUser						=	clone $user;

				foreach ( $addFields as $addField ) {
					if ( ! isset( $fields[$addField] ) ) {
						$loadField				=	new FieldTable();

						$loadField->load( (int) $addField );

						$fields[$addField]		=	$loadField;
					}

					$addFieldName				=	$fields[$addField]->get( 'name', null, GetterInterface::STRING );

					if ( ! $addFieldName ) {
						continue;
					}

					$addFieldValue				=	Get::get( $postdata, $addFieldName, null, GetterInterface::RAW );

					if ( is_array( $addFieldValue ) ) {
						$addFieldValue			=	implode( '|*|', $addFieldValue );
					}

					$tempUser->set( $addFieldName, Get::clean( $addFieldValue, GetterInterface::STRING ) );
				}

				$cbUser->_cbuser				=	$tempUser;
			}

			$valid								=	$this->codeValidate( $field, $user, $reason, $value, $cbUser );

			if ( $addFields ) {
				$cbUser->_cbuser				=	$user;
			}

			if ( $valid ) {
				$message						=	CBTxt::T( $field->params->get( 'code_validate_success', null, GetterInterface::HTML ), null, array( '[title]' => CBTxt::T( $field->get( 'title', null, GetterInterface::HTML ) ), '[value]' => $value ) );
			} else {
				$message						=	CBTxt::T( $field->params->get( 'code_validate_error', null, GetterInterface::HTML ), null, array( '[title]' => CBTxt::T( $field->get( 'title', null, GetterInterface::HTML ) ), '[value]' => $value ) );

				if ( ! $message ) {
					$message					=	CBTxt::T( 'CODE_VALIDATION_ERROR', 'Not a valid input.', array( '[title]' => CBTxt::T( $field->get( 'title', null, GetterInterface::HTML ) ), '[value]' => $value ) );
				}
			}

			if ( Application::MyUser()->isGlobalModerator() && $user->getError() ) {
				$message						=	$user->getError();
			}

			return json_encode( array( 'valid' => $valid, 'message' => $message ) );
		}

		if ( ( $this->input( 'function', null, GetterInterface::STRING ) == 'codeautocomplete' ) && $this->canAutoComplete( $field, $user, $reason ) ) {
			$addFields							=	cbToArrayOfInt( array_filter( explode( '|*|', $field->params->get( 'code_autocomplete_fields', null, GetterInterface::STRING ) ) ) );

			if ( ! ( $user instanceof UserTable ) ) {
				$user							=	new UserTable();
			}

			$cbUser								=	\CBuser::getInstance( $user->get( 'id', 0, GetterInterface::INT ), false );

			if ( $addFields ) {
				$tempUser						=	clone $user;

				foreach ( $addFields as $addField ) {
					if ( ! isset( $fields[$addField] ) ) {
						$loadField				=	new FieldTable();

						$loadField->load( (int) $addField );

						$fields[$addField]		=	$loadField;
					}

					$addFieldName				=	$fields[$addField]->get( 'name', null, GetterInterface::STRING );

					if ( ! $addFieldName ) {
						continue;
					}

					$addFieldValue				=	Get::get( $postdata, $addFieldName, null, GetterInterface::RAW );

					if ( is_array( $addFieldValue ) ) {
						$addFieldValue			=	implode( '|*|', $addFieldValue );
					}

					$tempUser->set( $addFieldName, Get::clean( $addFieldValue, GetterInterface::STRING ) );
				}

				$cbUser->_cbuser				=	$tempUser;
			}

			$value								=	Get::get( $postdata, 'value', null, GetterInterface::RAW );

			if ( is_array( $value ) ) {
				$value							=	implode( '|*|', $value );
			}

			$value								=	Get::clean( $value, GetterInterface::STRING );
			$code								=	$cbUser->replaceUserVars( $field->params->get( 'code_autocomplete_code', '', GetterInterface::RAW ), false, false, array( 'reason' => $reason, 'value' => $value ), false );

			if ( $addFields ) {
				$cbUser->_cbuser				=	$user;
			}

			if ( ! $code ) {
				return null;
			}

			try {
				$rows							=	CBCodeField::outputCode( $code, $field, $user, $reason, $value );
			} catch ( \Exception $e ) {
				$rows							=	array();
			}

			$options							=	array();

			if ( $rows ) {
				foreach ( $rows as $k => $v ) {
					$options[]					=	array( 'value' => $k, 'label' => CBTxt::T( $v ) );
				}
			}

			return json_encode( $options );
		}

		return null;
	}

	/**
	 * Formatter:
	 * Returns a field in specified format
	 *
	 * @param  FieldTable  $field
	 * @param  UserTable   $user
	 * @param  string      $output               'html', 'xml', 'json', 'php', 'csvheader', 'csv', 'rss', 'fieldslist', 'htmledit'
	 * @param  string      $formatting           'tr', 'td', 'div', 'span', 'none',   'table'??
	 * @param  string      $reason               'profile' for user profile view, 'edit' for profile edit, 'register' for registration, 'search' for searches
	 * @param  int         $list_compare_types   IF reason == 'search' : 0 : simple 'is' search, 1 : advanced search with modes, 2 : simple 'any' search
	 */
	public function getDisplay( &$field, &$user, $output, $formatting, $reason, $list_compare_types )
	{
		global $_CB_framework;

		if ( $reason == 'search' ) {
			$selectorPrefix					=	'.cbUserListSearchFields ';
		} else {
			$selectorPrefix					=	null;
		}

		if ( $this->canValidate( $field, $user, $reason ) ) {
			static $RULE_STATIC				=	0;

			if ( ! $RULE_STATIC++ ) {
				$rule						=	"params.method = 'cbcodevalidate';"
											.	"params.data = {};"
											.	"if ( ( typeof params.fields != 'undefined' ) && ( params.fields !== null ) && $.isArray( params.fields ) ) {"
											.		"$.each( params.fields, function( index, fieldId ) {"
											.			"var fieldTarget = $( fieldId );"
											.			"if ( $( element ).closest( '.cbRepeatRow' ).length ) {"
											.				"repeatTarget = $( element ).closest( '.cbRepeatRow' ).find( fieldId );"
											.				"if ( repeatTarget.length ) {"
											.					"fieldTarget = repeatTarget;"
											.				"}"
											.			"}"
											.			"var target = null;"
											.			"if ( fieldTarget.is( 'input' ) || fieldTarget.is( 'select' ) || fieldTarget.is( 'textarea' ) ) {"
											.				"target = fieldTarget;"
											.			"} else {"
											.				"target = fieldTarget.find( 'input,select,textarea' ).not( '[name$=\"__srmch\"]' ).first();"
											.				"if ( target.is( ':checkbox' ) || target.is( ':radio' ) ) {"
											.					"target = fieldTarget.find( 'input[name=\"' + target.attr( 'name' ) + '\"]' );"
											.				"}"
											.			"}"
											.			"if ( target.length ) {"
											.				"var fieldName = target.attr( 'name' );"
											.				"if ( $( element ).closest( '.cbRepeatRow' ).length ) {"
											.					"fieldName = fieldName.replace( /^.+__\d+__/g, '' );"
											.				"}"
											.				"if ( fieldName == params.field ) {"
											.					"return true;"
											.				"}"
											.				"fieldValue = null;"
											.				"if ( target.is( 'input' ) || target.is( 'select' ) || target.is( 'textarea' ) ) {"
											.					"if ( target.is( 'input[type=\"checkbox\"]' ) || target.is( 'input[type=\"radio\"]' ) ) {"
											.						"fieldValue = [];"
											.						"target.each( function() {"
											.							"if ( $( this ).is( ':checked' ) ) {"
											.								"fieldValue.push( $( this ).val() );"
											.							"}"
											.						"});"
											.					"} else if ( target.is( 'select[multiple]' ) ) {"
											.						"fieldValue = target.val();"
											.						"if ( value && ( ! $.isArray( fieldValue ) ) ) {"
											.							"fieldValue = fieldValue.split( ',' );"
											.						"}"
											.					"} else {"
											.						"fieldValue = target.val();"
											.					"}"
											.				"}"
											.				"params.data[fieldName] = fieldValue;"
											.			"}"
											.		"});"
											.	"}"
											.	"return $.validator.methods.cbfield.call( this, value, element, params );";

				\cbValidator::addRule( 'cbcodevalidate', $rule );
			}

			static $RULE_LOADED				=	array();

			$fieldId						=	$field->get( 'fieldid', 0, GetterInterface::INT );

			if ( ! isset( $RULE_LOADED[$fieldId] ) ) {
				$addFields					=	cbToArrayOfInt( array_filter( explode( '|*|', $field->params->get( 'code_validate_fields', null, GetterInterface::STRING ) ) ) );
				$selectors					=	array();

				foreach ( $addFields as $addField ) {
					$selectors[]			=	$selectorPrefix . '#cbfr_' . (int) $addField;
					$selectors[]			=	$selectorPrefix . '#cbfrd_' . (int) $addField;
				}

				$jsTargets					=	array();
				$jsTargets[]				=	$selectorPrefix . '#cbfr_' . $fieldId;
				$jsTargets[]				=	$selectorPrefix . '#cbfrd_' . $fieldId;

				$js							=	"$( " . json_encode( implode( ',', $jsTargets ), JSON_HEX_TAG ) . " ).each( function() {"
											.		"var element = $( this ).find( 'input,select,textarea' ).not( '[name$=\"__srmch\"]' ).first();"
											.		"if ( element.is( ':checkbox' ) || element.is( ':radio' ) ) {"
											.			"element = $( this ).find( 'input[name=\"' + element.attr( 'name' ) + '\"]' );"
											.		"}"
											.		"if ( ! element.length ) {"
											.			"return;"
											.		"}"
											.		"element.attr({"
											.			"'data-rule-cbcodevalidate': '" . json_encode( array( 'user' => $user->get( 'id', 0, GetterInterface::INT ), 'field' => htmlspecialchars( $field->get( 'name', null, GetterInterface::STRING ) ), 'reason' => htmlspecialchars( $reason ), 'function' => 'codevalidate', 'fields' => $selectors ) ) . "'"
											.		"}).on( 'modified-name.cbcodefield', function( e, oldName, newName, index ) {"
											.			"if ( oldName != newName ) {"
											.				"$( this ).removeData( 'rule-cbcodevalidate' );"
											.				"$( this ).attr( 'data-rule-cbcodevalidate', $( this ).attr( 'data-rule-cbcodevalidate' ).replace( oldName.replace( '[]', '' ), newName.replace( '[]', '' ) ) );"
											.			"}"
											.		"});"
											.	"});";

				$_CB_framework->outputCbJQuery( $js );

				$RULE_LOADED[$fieldId]		=	true;
			}
		}

		if ( $this->canAutoComplete( $field, $user, $reason ) ) {
			static $COMPLETE_LOADED			=	array();

			$fieldId						=	$field->get( 'fieldid', 0, GetterInterface::INT );

			if ( ! isset( $COMPLETE_LOADED[$fieldId] ) ) {
				$cbSpoofField				=	cbSpoofField();
				$cbSpoofString				=	cbSpoofString( null, 'fieldclass' );
				$regAntiSpamFieldName		=	cbGetRegAntiSpamFieldName();
				$regAntiSpamValues			=	cbGetRegAntiSpams();

				if ( Application::Cms()->getClientId() ) {
					$updateUrl				=	$_CB_framework->backendViewUrl( 'fieldclass', false, array( 'field' => $field->get( 'name', null, GetterInterface::STRING ), 'function' => 'codeautocomplete', 'user' => $user->get( 'id', 0, GetterInterface::INT ), 'reason' => $reason, $cbSpoofField => $cbSpoofString, $regAntiSpamFieldName => $regAntiSpamValues[0] ), 'raw' );
				} else {
					$updateUrl				=	$_CB_framework->viewUrl( 'fieldclass', false, array( 'field' => $field->get( 'name', null, GetterInterface::STRING ), 'function' => 'codeautocomplete', 'user' => $user->get( 'id', 0, GetterInterface::INT ), 'reason' => $reason, $cbSpoofField => $cbSpoofString, $regAntiSpamFieldName => $regAntiSpamValues[0] ), 'raw' );
				}

				$minLength					=	$field->params->get( 'code_autocomplete_min', 3, GetterInterface::INT );

				if ( ! $minLength ) {
					$minLength				=	1;
				}

				$addFields					=	cbToArrayOfInt( array_filter( explode( '|*|', $field->params->get( 'code_autocomplete_fields', null, GetterInterface::STRING ) ) ) );
				$selectors					=	array();

				foreach ( $addFields as $addField ) {
					$selectors[]			=	$selectorPrefix . '#cbfr_' . (int) $addField;
					$selectors[]			=	$selectorPrefix . '#cbfrd_' . (int) $addField;
				}

				$jsTargets					=	array();
				$jsTargets[]				=	$selectorPrefix . '#cbfr_' . $fieldId;
				$jsTargets[]				=	$selectorPrefix . '#cbfrd_' . $fieldId;

				$js							=	"$( " . json_encode( implode( ',', $jsTargets ), JSON_HEX_TAG ) . " ).cbcodeautocomplete({"
											.		"url: " . json_encode( $updateUrl, JSON_HEX_TAG ) . ","
											.		"length: " . $minLength . ","
											.		"strict: " . json_encode( $field->params->get( 'code_autocomplete_strict', false, GetterInterface::BOOLEAN ), JSON_HEX_TAG ) . ","
											.		"fields: " . json_encode( $selectors, JSON_HEX_TAG )
											.	"});";

				$_CB_framework->outputCbJQuery( $js, 'cbcodeautocomplete' );

				$COMPLETE_LOADED[$fieldId]	=	true;
			}
		}
	}

	/**
	 * Mutator:
	 * Prepares field data for saving to database (safe transfer from $postdata to $user)
	 * Override
	 *
	 * @param  FieldTable  $field
	 * @param  UserTable   $user      RETURNED populated: touch only variables related to saving this field (also when not validating for showing re-edit)
	 * @param  array       $postdata  Typically $_POST (but not necessarily), filtering required.
	 * @param  string      $reason    'edit' for save user edit, 'register' for save registration
	 */
	public function checkValidation( &$field, &$user, &$postdata, $reason )
	{
		if ( $this->canValidate( $field, $user, $reason, false ) ) {
			$value				=	cbGetParam( $postdata, $field->get( 'name', null, GetterInterface::STRING ) );

			if ( is_array( $value ) ) {
				$value			=	$this->_implodeCBvalues( $value );
			}

			$value				=	stripslashes( $value );

			if ( ( ! $this->isEmpty( $field, $value ) ) && ( ! $this->codeValidate( $field, $user, $reason, $value ) ) ) {
				$message		=	CBTxt::T( $field->params->get( 'code_validate_error', null, GetterInterface::HTML ), null, array( '[title]' => CBTxt::T( $field->get( 'title', null, GetterInterface::HTML ) ), '[value]' => $value ) );

				if ( ! $message )  {
					$message	=	CBTxt::T( 'CODE_VALIDATION_ERROR', 'Not a valid input.', array( '[title]' => CBTxt::T( $field->get( 'title', null, GetterInterface::HTML ) ), '[value]' => $value ) );
				}

				$this->_setValidationError( $field, $user, $reason, $message );
			}
		}

		if ( $this->canAutoComplete( $field, $user, $reason )
			 && ( ( ! Application::Cms()->getClientId() ) || ( Application::Cms()->getClientId() && Application::Config()->get( 'adminrequiredfields', true, GetterInterface::BOOLEAN ) ) )
			 && $field->params->get( 'code_autocomplete_strict', false, GetterInterface::BOOLEAN )
		) {
			$value				=	cbGetParam( $postdata, $field->get( 'name', null, GetterInterface::STRING ) );

			if ( is_array( $value ) ) {
				$value			=	$this->_implodeCBvalues( $value );
			}

			$value				=	stripslashes( $value );

			$cbUser				=	\CBuser::getInstance( $user->get( 'id', 0, GetterInterface::INT ), false );
			$tempUser			=	$cbUser->_cbuser;

			$cbUser->_cbuser	=	$user;

			$code				=	$cbUser->replaceUserVars( $field->params->get( 'code_autocomplete_code', '', GetterInterface::RAW ), false, false, array( 'reason' => $reason, 'value' => $value ), false );

			$cbUser->_cbuser	=	$tempUser;

			if ( ! $code ) {
				return;
			}

			try {
				$rows			=	CBCodeField::outputCode( $code, $field, $user, $reason, $value );
			} catch ( \Exception $e ) {
				$rows			=	array();
			}

			if ( ( ! $this->isEmpty( $field, $value ) ) && ( ! in_array( $value, array_keys( $rows ) ) ) ) {
				$message		=	CBTxt::T( $field->params->get( 'code_autocomplete_strict_error', null, GetterInterface::HTML ), null, array( '[title]' => CBTxt::T( $field->get( 'title', null, GetterInterface::HTML ) ), '[value]' => $value ) );

				if ( ! $message )  {
					$message	=	CBTxt::T( 'CODE_AUTOCOMPLETE_VALIDATION_ERROR', 'Not a valid input.', array( '[title]' => CBTxt::T( $field->get( 'title', null, GetterInterface::HTML ) ), '[value]' => $value ) );
				}

				$this->_setValidationError( $field, $user, $reason, $message );
			}
		}
	}
}