<?php
/**
* Community Builder (TM)
* @version $Id: $
* @package CommunityBuilder
* @copyright (C) 2004-2021 www.joomlapolis.com / Lightning MultiCom SA - and its licensors, all rights reserved
* @license http://www.gnu.org/licenses/old-licenses/gpl-2.0.html GNU/GPL version 2
*/

namespace CB\Plugin\Invites\Table;

use CBLib\Application\Application;
use CBLib\Database\Table\Table;
use CB\Database\Table\UserTable;
use CB\Plugin\Invites\CBInvites;
use CBLib\Language\CBTxt;

defined('CBLIB') or die();

class InviteTable extends Table
{
	/** @var int */
	public $id;
	/** @var int */
	public $user_id;
	/** @var string */
	public $to;
	/** @var int */
	public $user;
	/** @var string */
	public $code;
	/** @var string */
	public $sent;
	/** @var string */
	public $accepted;

	/**
	 * Table name in database
	 * @var string
	 */
	protected $_tbl			=	'#__comprofiler_plugin_invites';

	/**
	 * Primary key(s) of table
	 * @var string
	 */
	protected $_tbl_key		=	'id';

	/**
	 * @return bool
	 */
	public function check()
	{
		if ( ! $this->getInt( 'user_id', 0 ) ) {
			$this->setError( CBTxt::T( 'Owner not specified!' ) );

			return false;
		}

		return true;
	}

	/**
	 * @param bool $updateNulls
	 * @return bool
	 */
	public function store( $updateNulls = false )
	{
		global $_PLUGINS;

		$new	=	( ! $this->getInt( 'id', 0 ) );
		$old	=	new self();

		$this->set( 'code', $this->getString( 'code', md5( uniqid( '', true ) ) ) );

		if ( ! $new ) {
			$old->load( $this->getInt( 'id', 0 ) );

			$_PLUGINS->trigger( 'invites_onBeforeUpdateInvite', array( &$this, $old ) );
		} else {
			$_PLUGINS->trigger( 'invites_onBeforeCreateInvite', array( &$this ) );
		}

		if ( ! parent::store( $updateNulls ) ) {
			return false;
		}

		if ( ! $new ) {
			$_PLUGINS->trigger( 'invites_onAfterUpdateInvite', array( $this, $old ) );
		} else {
			$_PLUGINS->trigger( 'invites_onAfterCreateInvite', array( $this ) );
		}

		return true;
	}

	/**
	 * @param null|int $id
	 * @return bool
	 */
	public function delete( $id = null )
	{
		global $_PLUGINS;

		$_PLUGINS->trigger( 'invites_onBeforeDeleteInvite', array( &$this ) );

		if ( ! parent::delete( $id ) ) {
			return false;
		}

		$_PLUGINS->trigger( 'invites_onAfterDeleteInvite', array( $this ) );

		return true;
	}

	/**
	 * @return bool
	 */
	public function send()
	{
		global $_PLUGINS;

		if ( ! $this->getString( 'to' ) ) {
			return false;
		}

		$this->set( 'sent', Application::Database()->getUtcDateTime() );

		$_PLUGINS->trigger( 'invites_onBeforeSendInvite', array( &$this ) );

		if ( ! CBInvites::sendInvite( $this ) ) {
			return false;
		}

		$_PLUGINS->trigger( 'invites_onAfterSendInvite', array( $this ) );

		if ( $this->getError() ) {
			return false;
		}

		return $this->store();
	}

	/**
	 * @param UserTable $user
	 * @return bool
	 */
	public function accept( $user )
	{
		global $_PLUGINS;

		if ( $this->isAccepted() || $this->isExpired() ) {
			return false;
		}

		$mode					=	CBInvites::getGlobalParams()->getInt( 'invite_connection', 2 );

		$this->set( 'accepted', Application::Database()->getUtcDateTime() );
		$this->set( 'user', $user->get( 'id', 0 ) );

		$_PLUGINS->trigger( 'invites_onBeforeAcceptInvite', array( &$this, $user ) );

		if ( $this->store() ) {
			if ( $mode ) {
				$connections	=	new \cbConnection( $this->getInt( 'user_id', 0 ) );

				$connections->addConnection( $user->getInt( 'id', 0 ), null, false );

				if ( $mode === 2 ) {
					$connections->acceptConnection( $this->getInt( 'user_id', 0 ), $user->getInt( 'id', 0 ), false );
				}
			}
		} else {
			return false;
		}

		$_PLUGINS->trigger( 'invites_onAfterAcceptInvite', array( $this, $user ) );

		return true;
	}

	/**
	 * @return bool
	 */
	public function isSent()
	{
		global $_CB_database;

		if ( ! $this->getString( 'to' ) ) {
			return true;
		}

		static $cache		=	array();

		$id					=	$this->getInt( 'id', 0 );

		if ( ! isset( $cache[$id] ) ) {
			if ( $this->getString( 'sent' ) && ( $this->getString( 'sent' ) !== $_CB_database->getNullDate() ) ) {
				$sent		=	true;
			} else {
				$sent		=	false;
			}

			$cache[$id]		=	$sent;
		}

		return $cache[$id];
	}

	/**
	 * @return bool
	 */
	public function isAccepted()
	{
		global $_CB_database;

		static $cache		=	array();

		$id					=	$this->getInt( 'id', 0 );

		if ( ! isset( $cache[$id] ) ) {
			if ( $this->getString( 'accepted' ) && ( $this->getString( 'accepted' ) !== $_CB_database->getNullDate() ) ) {
				$accepted	=	true;
			} else {
				$accepted	=	false;
			}

			$cache[$id]		=	$accepted;
		}

		return $cache[$id];
	}

	/**
	 * @return bool
	 */
	public function isExpired()
	{
		static $cache		=	array();

		$id					=	$this->getInt( 'id', 0 );

		if ( ! isset( $cache[$id] ) ) {
			$expire			=	CBInvites::getGlobalParams()->getString( 'invite_expire' );

			if ( ! $expire ) {
				$expired	=	false;
			} elseif ( ( ! $this->isSent() ) || $this->isAccepted() ) {
				$expired	=	true;
			} else {
				$expired	=	( Application::Date( 'now', 'UTC' )->getTimestamp() >= Application::Date( $this->getString( 'sent' ), 'UTC' )->modify( strtoupper( $expire ) )->getTimestamp() );
			}

			$cache[$id]		=	$expired;
		}

		return $cache[$id];
	}

	/**
	 * @return int
	 */
	public function dateDifference()
	{
		global $_CB_framework;

		static $cache		=	array();

		$id					=	$this->getInt( 'id', 0 );

		if ( ! isset( $cache[$id] ) ) {
			$cache[$id]		=	( ( $_CB_framework->getUTCNow() - strtotime( $this->getString( 'sent' ) ) ) / 86400 );
		}

		return $cache[$id];
	}

	/**
	 * @return bool
	 */
	public function canResend()
	{
		if ( ! $this->getString( 'to' ) ) {
			return false;
		}

		static $cache		=	array();

		$id					=	$this->getInt( 'id', 0 );

		if ( ! isset( $cache[$id] ) ) {
			if ( ( ! $this->isAccepted() ) && ( ( ! $this->isSent() ) || ( $this->dateDifference() >= CBInvites::getGlobalParams()->getInt( 'invite_resend', 7 ) ) ) ) {
				$resend		=	true;
			} else {
				$resend		=	false;
			}

			$cache[$id]		=	$resend;
		}

		return $cache[$id];
	}

	/**
	 * @param int|null $userId
	 * @return bool
	 */
	public function isDuplicate( $userId = null )
	{
		global $_CB_database;

		if ( ! $this->getString( 'to' ) ) {
			return false;
		}

		static $cache				=	array();

		$id							=	$this->getString( 'to' );

		if ( ! isset( $cache[$id][$userId] ) ) {
			$where					=	array();

			if ( $this->getInt( 'id', 0 ) ) {
				$where[]			=	$_CB_database->NameQuote( 'id' ) . " != " . $this->getInt( 'id', 0 );
			}

			if ( $userId ) {
				$where[]			=	$_CB_database->NameQuote( 'user_id' ) . " = " . (int) $userId;
			}

			$where[]				=	$_CB_database->NameQuote( 'to' ) . " = " . $_CB_database->Quote( $id );

			$query					=	'SELECT COUNT(*)'
									.	"\n FROM " . $_CB_database->NameQuote( '#__comprofiler_plugin_invites' )
									.	"\n WHERE " . implode( "\n AND ", $where );
			$_CB_database->setQuery( $query );
			$cache[$id][$userId]	=	(bool) $_CB_database->loadResult();
		}

		return $cache[$id][$userId];
	}

	/**
	 * @return null|string
	 */
	public function getLink()
	{
		global $_CB_framework, $_PLUGINS;

		if ( ! $this->getInt( 'id', 0 ) ) {
			return null;
		}

		$variables	=	array( 'action' => 'invites', 'func' => 'accept', 'code' => $this->getString( 'code' ) );
		$link		=	implode( '', $_PLUGINS->trigger( 'invites_onInviteLink', array( $this, &$variables ) ) );

		if ( ! $link ) {
			return $_CB_framework->pluginClassUrl( 'cbinvites', true, $variables );
		}

		return $link;
	}
}
