<?php
/**
* Community Builder (TM)
* @version $Id: $
* @package CommunityBuilder
* @copyright (C) 2004-2021 www.joomlapolis.com / Lightning MultiCom SA - and its licensors, all rights reserved
* @license http://www.gnu.org/licenses/old-licenses/gpl-2.0.html GNU/GPL version 2
*/

namespace CB\Plugin\AutoActions\Action;

use CB\Database\Table\UserTable;
use CBLib\Language\CBTxt;
use CBLib\Registry\ParamsInterface;
use Kunena\Forum\Libraries\Forum\Category\KunenaCategoryHelper;
use Kunena\Forum\Libraries\Forum\KunenaForum as Kunena6Forum;
use Kunena\Forum\Libraries\Forum\Message\KunenaMessageHelper;
use Kunena\Forum\Libraries\Route\KunenaRoute as Kunena6Route;
use Kunena\Forum\Libraries\Forum\Topic\KunenaTopicHelper;
use Kunena\Forum\Libraries\User\KunenaUserHelper as Kunena6UserHelper;

\defined( 'CBLIB' ) or die();

class KunenaAction extends Action
{
	/** @var int  */
	private int $version	=	0;

	/**
	 * @param UserTable $user
	 */
	public function execute( UserTable $user ): void
	{
		global $_CB_database;

		if ( ! $this->installed() ) {
			$this->error( CBTxt::T( 'AUTO_ACTION_KUNENA_NOT_INSTALLED', ':: Action [action] :: Kunena is not installed', [ '[action]' => $this->autoaction()->getId() ] ) );
			return;
		}

		foreach ( $this->autoaction()->getParams()->subTree( 'kunena' ) as $row ) {
			/** @var ParamsInterface $row */
			$owner								=	$row->getString( 'owner', '' );

			if ( ! $owner ) {
				$owner							=	$user->getInt( 'id', 0 );
			} else {
				$owner							=	(int) $this->string( $user, $owner );
			}

			if ( $user->getInt( 'id', 0 ) !== $owner ) {
				$actionUser						=	\CBuser::getUserDataInstance( $owner );
			} else {
				$actionUser						=	$user;
			}

			switch ( $row->getString( 'mode', 'category' ) ) {
				case 'sync':
					if ( ! $actionUser->getInt( 'id', 0 ) ) {
						$this->error( CBTxt::T( 'AUTO_ACTION_KUNENA_NO_USER', ':: Action [action] :: Kunena skipped due to missing user', [ '[action]' => $this->autoaction()->getId() ] ) );
						continue 2;
					}

					if ( $this->version() >= 6 ) {
						$kunenaUser				=	Kunena6UserHelper::get( $actionUser->getInt( 'id', 0 ) );
					} else {
						$kunenaUser				=	\KunenaUserHelper::get( $actionUser->getInt( 'id', 0 ) );
					}

					$kunenaUser->set( 'name', $actionUser->getString( 'name', '' ) );
					$kunenaUser->set( 'username', $actionUser->getString( 'username', '' ) );
					$kunenaUser->set( 'email', $actionUser->getString( 'email', '' ) );

					foreach ( $row->subTree( 'fields' ) as $r ) {
						/** @var ParamsInterface $r */
						$field					=	$r->getString( 'field', '' );

						if ( ! $field ) {
							continue;
						}

						$value					=	$this->string( $actionUser, $r->getRaw( 'value', '' ), false, $r->getBool( 'translate', false ) );

						switch ( $field ) {
							case 'ordering':
							case 'gender':
							case 'rank':
							case 'hideEmail':
							case 'showOnline':
							case 'canSubscribe':
								$value			=	(int) $value;
								break;
						}

						$kunenaUser->set( $field, $value );
					}

					$kunenaUser->save();
					break;
				case 'reply':
					if ( ! $actionUser->getInt( 'id', 0 ) ) {
						$this->error( CBTxt::T( 'AUTO_ACTION_KUNENA_NO_OWNER', ':: Action [action] :: Kunena skipped due to missing owner', [ '[action]' => $this->autoaction()->getId() ] ) );
						continue 2;
					}

					$message					=	$this->string( $actionUser, $row->getRaw( 'message', '' ), false );

					if ( ! $message ) {
						$this->error( CBTxt::T( 'AUTO_ACTION_KUNENA_NO_MSG', ':: Action [action] :: Kunena skipped due to missing message', [ '[action]' => $this->autoaction()->getId() ] ) );
						continue 2;
					}

					$topicId					=	$row->getInt( 'topic', 0 );

					if ( ! $topicId ) {
						$this->error( CBTxt::T( 'AUTO_ACTION_KUNENA_NO_TOPIC', ':: Action [action] :: Kunena skipped due to missing topic', [ '[action]' => $this->autoaction()->getId() ] ) );
						continue 2;
					}

					$subject					=	$this->string( $actionUser, $row->getString( 'subject', '' ) );

					if ( $this->version() >= 6 ) {
						$topic					=	KunenaMessageHelper::get( $topicId );
					} else {
						$topic					=	\KunenaForumMessageHelper::get( $topicId );
					}

					$fields						=	[ 'message' => $message ];

					if ( $subject ) {
						$fields['subject']		=	$subject;
					}

					$topic->newReply( $fields, $actionUser->getInt( 'id', 0 ) );
					break;
				case 'topic':
					if ( ! $actionUser->getInt( 'id', 0 ) ) {
						$this->error( CBTxt::T( 'AUTO_ACTION_KUNENA_NO_OWNER', ':: Action [action] :: Kunena skipped due to missing owner', [ '[action]' => $this->autoaction()->getId() ] ) );
						continue 2;
					}

					$subject					=	$this->string( $actionUser, $row->getString( 'subject', '' ) );

					if ( ! $subject ) {
						$this->error( CBTxt::T( 'AUTO_ACTION_KUNENA_NO_SUBJ', ':: Action [action] :: Kunena skipped due to missing subject', [ '[action]' => $this->autoaction()->getId() ] ) );
						continue 2;
					}

					$message					=	$this->string( $actionUser, $row->getRaw( 'message', '' ), false );

					if ( ! $message ) {
						$this->error( CBTxt::T( 'AUTO_ACTION_KUNENA_NO_MSG', ':: Action [action] :: Kunena skipped due to missing message', [ '[action]' => $this->autoaction()->getId() ] ) );
						continue 2;
					}

					$categoryId					=	$row->getInt( 'category', 0 );

					if ( ! $categoryId ) {
						$this->error( CBTxt::T( 'AUTO_ACTION_KUNENA_NO_CAT', ':: Action [action] :: Kunena skipped due to missing category', [ '[action]' => $this->autoaction()->getId() ] ) );
						continue 2;
					}

					if ( $this->version() >= 6 ) {
						$category				=	KunenaCategoryHelper::get( $categoryId );
					} else {
						$category				=	\KunenaForumCategoryHelper::get( $categoryId );
					}

					$fields						=	[	'catid'		=>	$categoryId,
														'subject'	=>	$subject,
														'message'	=>	$message,
													];

					$category->newTopic( $fields, $actionUser->getInt( 'id', 0 ) );
					break;
				case 'category':
				default:
					$name						=	$this->string( $actionUser, $row->getString( 'name', '' ) );

					if ( ! $name ) {
						$this->error( CBTxt::T( 'AUTO_ACTION_KUNENA_NO_NAME', ':: Action [action] :: Kunena skipped due to missing name', [ '[action]' => $this->autoaction()->getId() ] ) );
						continue 2;
					}

					$query						=	'SELECT ' . $_CB_database->NameQuote( 'id' )
												.	"\n FROM " . $_CB_database->NameQuote( '#__kunena_categories' )
												.	"\n WHERE " . $_CB_database->NameQuote( 'name' ) . " = " . $_CB_database->Quote( $name );
					$_CB_database->setQuery( $query );

					if ( ! $_CB_database->loadResult() ) {
						if ( $this->version() >= 6 ) {
							$category			=	KunenaCategoryHelper::get();

							$category->set( 'parentid', $row->getInt( 'parent', 0 ) );
							$category->set( 'alias', Kunena6Route::stringURLSafe( $name ) );

							if ( $category->get( 'channels' ) === null ) {
								$category->set( 'channels', 'THIS' );
							}

							if ( $category->get( 'icon' ) === null ) {
								$category->set( 'icon', '' );
							}

							if ( $category->get( 'iconset' ) === null ) {
								$category->set( 'iconset', 'default' );
							}

							if ( $category->get( 'headerdesc' ) === null ) {
								$category->set( 'headerdesc', '' );
							}

							if ( $category->get( 'topictemplate' ) === null ) {
								$category->set( 'topictemplate', '' );
							}

							if ( $category->get( 'class_sfx' ) === null ) {
								$category->set( 'class_sfx', '' );
							}
						} else {
							$category			=	\KunenaForumCategoryHelper::get();

							$category->set( 'parent_id', $row->getInt( 'parent', 0 ) );
							$category->set( 'alias', \KunenaRoute::stringURLSafe( $name ) );
						}

						$category->set( 'name', $name );
						$category->set( 'accesstype', 'joomla.group' );
						$category->set( 'access', $row->getInt( 'access', 1 ) );
						$category->set( 'published', $row->getInt( 'published', 1 ) );
						$category->set( 'description', $this->string( $actionUser, $row->getString( 'description', '' ) ) );

						if ( $category->save() && $actionUser->getInt( 'id', 0 ) ) {
							$category->addModerator( $actionUser->getInt( 'id', 0 ) );
						}
					}
					break;
			}
		}
	}

	/**
	 * @return array
	 */
	public function categories(): array
	{
		$options			=	[];

		if ( $this->installed() ) {
			if ( $this->version() >= 6 ) {
				$rows		=	KunenaCategoryHelper::getChildren( 0, 10 );
			} else {
				$rows		=	\KunenaForumCategoryHelper::getChildren( 0, 10 );
			}

			foreach ( $rows as $row ) {
				$options[]	=	\moscomprofilerHTML::makeOption( (string) $row->id, str_repeat( '- ', $row->level + 1  ) . ' ' . $row->name );
			}
		}

		return $options;
	}

	/**
	 * @return array
	 */
	public function topics(): array
	{
		$options			=	[];

		if ( $this->installed() ) {
			if ( $this->version() >= 6 ) {
				$rows		=	KunenaTopicHelper::getLatestTopics();
			} else {
				$rows		=	\KunenaForumTopicHelper::getLatestTopics();
			}

			foreach ( $rows[1] as $row ) {
				$options[]	=	\moscomprofilerHTML::makeOption( (string) $row->id, $row->subject );
			}
		}

		return $options;
	}

	/**
	 * @return int
	 */
	public function version(): int
	{
		return $this->version;
	}

	/**
	 * @return bool
	 */
	public function installed(): bool
	{
		global $_CB_framework;

		$api	=	$_CB_framework->getCfg( 'absolute_path' ) . '/administrator/components/com_kunena/api.php';

		if ( file_exists( $api ) ) {
			require_once( $api );

			if ( class_exists( 'KunenaForum' ) && class_exists( 'KunenaForumCategoryHelper' ) && class_exists( 'KunenaForumTopicHelper' ) && class_exists( 'KunenaUserHelper' ) ) {
				if ( method_exists( 'KunenaForum', 'version' ) ) {
					$this->version	=	(int) substr( (string) KunenaForum::version(), 0, 1 );
				}

				return true;
			}
		} elseif ( class_exists( '\Kunena\Forum\Libraries\Forum\KunenaForum' ) ) {
			$this->version			=	(int) substr( (string) Kunena6Forum::version(), 0, 1 );

			return true;
		}

		return false;
	}
}