<?php
/**
* Community Builder (TM)
* @version $Id: $
* @package CommunityBuilder
* @copyright (C) 2004-2021 www.joomlapolis.com / Lightning MultiCom SA - and its licensors, all rights reserved
* @license http://www.gnu.org/licenses/old-licenses/gpl-2.0.html GNU/GPL version 2
*/

use CB\Database\Table\PluginTable;
use CB\Plugin\AutoActions\CBAutoActions;
use CB\Plugin\AutoActions\SystemActions;
use CB\Plugin\AutoActions\Table\BatchTable;
use CBLib\Core\AutoLoader;
use CBLib\Database\Table\Table;
use CBLib\Registry\ParamsInterface;
use CBLib\Registry\Registry;

\defined( 'CBLIB' ) or die();

AutoLoader::registerExactMap( '%^CB/Plugin/AutoActions/(.+)%i', __DIR__ . '/library/$1.php' );

/**
 * @param cbInstallerPlugin $installer
 * @param PluginTable       $plugin
 * @param bool|string       $oldVersion
 * @param bool|string       $newRelease
 */
function plug_cbautoactions_install( cbInstallerPlugin $installer, PluginTable $plugin, $oldVersion, $newRelease )
{
	SystemActions::install();
}

/**
 * @param cbInstallerPlugin $installer
 * @param PluginTable       $plugin
 * @param bool|string       $oldVersion
 * @param bool|string       $newRelease
 */
function plug_cbautoactions_update( cbInstallerPlugin $installer, PluginTable $plugin, $oldVersion, $newRelease )
{
	cbautoactionsMigrate::migrate( $oldVersion, $newRelease );

	SystemActions::install();
}

class cbautoactionsMigrate
{
	public static function migrate( $oldVersion, $newRelease ): void
	{
		global $_CB_database;

		if ( ( $oldVersion !== false )
			 && ( $newRelease !== false )
			 && version_compare( $oldVersion, '8.2.0', '>=' )
		) {
			// The following migration is all pre-8.2.0 migrations so skip them if we're not upgrading from older than 8.2.0
			return;
		}

		$table									=	'#__comprofiler_plugin_autoactions';
		$fields									=	$_CB_database->getTableFields( $table );

		if ( isset( $fields[$table]['field'] ) ) {
			$translateExists					=	isset( $fields[$table]['translate'] );
			$excludeExists						=	isset( $fields[$table]['exclude'] );
			$debugExists						=	isset( $fields[$table]['debug'] );

			$query								=	"SELECT *"
												.	"\n FROM " . $_CB_database->NameQuote( '#__comprofiler_plugin_autoactions' );
			$_CB_database->setQuery( $query );
			$rows								=	$_CB_database->loadObjectList( null, '\CBLib\Database\Table\Table', [ $_CB_database, '#__comprofiler_plugin_autoactions', 'id' ] );

			/** @var $rows Table[] */
			foreach ( $rows as $row ) {
				$row->set( 'system', '' );
				$row->set( 'trigger', str_replace( ',', '|*|', $row->getString( 'trigger', '' ) ) );

				$newParams						=	new Registry( $row->getRaw( 'params' ) );

				if ( $row->getRaw( 'field' ) ) {
					$fields						=	new Registry( $row->getRaw( 'field' ) );
					$operators					=	new Registry( $row->getRaw( 'operator' ) );
					$values						=	new Registry( $row->getRaw( 'value' ) );

					if ( $translateExists ) {
						$translates				=	new Registry( $row->getRaw( 'translate' ) );
					} else {
						$translates				=	null;
					}

					$conditionals				=	count( $fields );

					if ( $conditionals ) {
						$conditions				=	[];

						for ( $i = 0, $n = $conditionals; $i < $n; $i++ ) {
							$field				=	$fields->getString( "field$i", '' );
							$operator			=	$operators->getString( "operator$i", '' );
							$value				=	$values->getString( "value$i", '' );

							if ( $translateExists ) {
								$translate		=	$translates->getInt( "translate$i", 0 );
							} else {
								$translate		=	0;
							}

							if ( $operator !== '' ) {
								$conditions[]	=	[	'field'					=>	'customvalue',
														'field_custom'			=>	$field,
														'field_custom_options'	=>	( $translate ? 'translate' : null ),
														'operator'				=>	$operator,
														'value'					=>	$value,
														'value_options'			=>	( $translate ? 'translate' : null )
													];
							}
						}

						if ( $conditions ) {
							$row->set( 'conditions', json_encode( [ [ 'condition' => $conditions ] ] ) );
						}
					}

					$row->set( 'field', null );
					$row->set( 'operator', null );
					$row->set( 'value', null );

					if ( $translateExists ) {
						$row->set( 'translate', null );
					}
				}

				if ( $excludeExists ) {
					$exclude					=	$row->getString( 'exclude', '' );

					if ( $exclude ) {
						$newParams->set( 'exclude', $exclude );
						$row->set( 'exclude', null );
					}
				}

				if ( $debugExists ) {
					$debug						=	$row->getString( 'debug', '' );

					if ( $debug ) {
						$newParams->set( 'debug', $debug );
						$row->set( 'debug', null );
					}
				}

				if ( method_exists( 'cbautoactionsMigrate', $row->getString( 'type', '' ) ) ) {
					call_user_func_array( [ 'cbautoactionsMigrate', $row->getString( 'type', '' ) ], [ &$row, &$newParams ] );
				}

				$row->set( 'params', $newParams->asJson() );

				$row->store( true );
			}

			$_CB_database->dropColumn( $table, 'field' );
			$_CB_database->dropColumn( $table, 'operator' );
			$_CB_database->dropColumn( $table, 'value' );

			if ( $translateExists ) {
				$_CB_database->dropColumn( $table, 'translate' );
			}

			if ( $excludeExists ) {
				$_CB_database->dropColumn( $table, 'exclude' );
			}

			if ( $debugExists ) {
				$_CB_database->dropColumn( $table, 'debug' );
			}
		} else {
			// Set old non-system int usages to empty string:
			$query								=	"UPDATE " . $_CB_database->NameQuote( '#__comprofiler_plugin_autoactions' )
												.	"\n SET " . $_CB_database->NameQuote( 'system' ) . " = ''"
												.	"\n WHERE " . $_CB_database->NameQuote( 'system' ) . " = '0'";
			$_CB_database->setQuery( $query );
			$_CB_database->query();

			// Migrate actions:
			$query								=	"SELECT *"
												.	"\n FROM " . $_CB_database->NameQuote( '#__comprofiler_plugin_autoactions' )
												.	"\n WHERE " . $_CB_database->NameQuote( 'system' ) . " = ''";
			$_CB_database->setQuery( $query );
			$rows								=	$_CB_database->loadObjectList( null, '\CBLib\Database\Table\Table', [ $_CB_database, '#__comprofiler_plugin_autoactions', 'id' ] );

			$internalGeneral					=	[];
			$internalUsers						=	[];

			/** @var $rows Table[] */
			foreach ( $rows as $row ) {
				$migrated						=	false;
				$oldTriggers					=	$row->getString( 'trigger', '' );
				$newTriggers					=	explode( '|*|', str_replace( ',', '|*|', $oldTriggers ) );

				foreach ( $newTriggers as $i => $newTrigger ) {
					if ( $newTrigger === 'internalGeneral' ) {
						$internalGeneral[]		=	$row->getInt( 'id', 0 );

						unset( $newTriggers[$i] );
					}

					if ( $newTrigger === 'internalUsers' ) {
						$internalUsers[]		=	$row->getInt( 'id', 0 );

						unset( $newTriggers[$i] );
					}
				}

				$newTriggers					=	implode( '|*|', $newTriggers );

				if ( $oldTriggers !== $newTriggers ) {
					$row->set( 'trigger', $newTriggers );

					$migrated					=	true;
				}

				$rowParams						=	new Registry( $row->getRaw( 'params' ) );
				$oldParams						=	$rowParams->asJson();

				if ( $row->getString( 'object', '0' ) === '4' ) {
					// Move old object_custom to object_code with consolidated features param:
					$options					=	[];

					if ( $rowParams->getBool( 'object_custom_translate', false ) ) {
						$options[]				=	'translate';
					}

					if ( $rowParams->getBool( 'object_custom_substitutions', false ) ) {
						$options[]				=	'substitutions';
					}

					if ( $rowParams->getBool( 'object_custom_format', false ) ) {
						$options[]				=	'format';
					}

					if ( $rowParams->getBool( 'object_custom_content_plugins', false ) ) {
						$options[]				=	'content_plugins';
					}

					$rowParams->set( 'object_code', $rowParams->getRaw( 'object_custom' ) );
					$rowParams->set( 'object_code_options', implode( '|*|', $options ) );

					$rowParams->unsetEntry( 'object_custom' );
					$rowParams->unsetEntry( 'object_custom_translate' );
					$rowParams->unsetEntry( 'object_custom_substitutions' );
					$rowParams->unsetEntry( 'object_custom_format' );
					$rowParams->unsetEntry( 'object_custom_content_plugins' );
				}

				$rowConditions					=	new Registry( $row->getRaw( 'conditions' ) );
				$conditions						=	[];

				foreach ( $rowConditions as $condition ) {
					/** @var ParamsInterface $condition */
					$conditionField				=	$condition->getRaw( 'field' );

					if ( ( $conditionField === null ) || ( $conditionField === '' ) ) {
						continue;
					}

					$options					=	[];

					if ( $condition->getBool( 'translate', false ) ) {
						$options[]				=	'translate';
					}

					if ( $condition->getBool( 'format', false ) ) {
						$options[]				=	'format';
					}

					if ( $condition->getBool( 'content_plugins', false ) ) {
						$options[]				=	'content_plugins';
					}

					$conditions[]				=	[	'field'					=>	'customvalue',
														'field_custom'			=>	$conditionField,
														'field_custom_options'	=>	implode( '|*|', $options ),
														'operator'				=>	$condition->getInt( 'operator', 0 ),
														'value'					=>	$condition->getRaw( 'value' ),
														'value_options'			=>	implode( '|*|', $options )
													];
				}

				if ( $conditions ) {
					$row->set( 'conditions', json_encode( [ [ 'condition' => $conditions ] ] ) );

					$migrated					=	true;
				}

				if ( method_exists( 'cbautoactionsMigrate', $row->getString( 'type', '' ) ) ) {
					call_user_func_array( [ 'cbautoactionsMigrate', $row->getString( 'type', '' ) ], [ &$row, &$rowParams ] );

				}

				$newParams						=	$rowParams->asJson();

				if ( $oldParams !== $newParams ) {
					$row->set( 'params', $newParams );

					$migrated					=	true;
				}

				if ( $migrated ) {
					$row->store();
				}
			}

			// Old Internal General trigger usage exists; migrate this to batch usage:
			if ( $internalGeneral ) {
				$query							=	"SELECT *"
												.	"\n FROM " . $_CB_database->NameQuote( '#__comprofiler_plugin_autoactions_batches' )
												.	"\n WHERE ( " . $_CB_database->NameQuote( 'title' ) . " = " . $_CB_database->Quote( 'internalGeneral' )
												.	" OR " . $_CB_database->NameQuote( 'users' ) . " = 0 )";
				$_CB_database->setQuery( $query, 0, 1 );
				$batch							=	new BatchTable();
				$_CB_database->loadObject( $batch );

				if ( ! $batch->getInt( 'id', 0 ) ) {
					$batch->set( 'title', 'internalGeneral' );
					$batch->set( 'users', 0 );
					$batch->set( 'access', -1 );
					$batch->set( 'published', 1 );
					$batch->set( 'batch_limit', CBAutoActions::getGlobalParams()->getInt( 'batch_limit', 0 ) );
					$batch->set( 'batch_redirect', 0 );
				} else {
					$actions					=	new Registry( $batch->getString( 'actions', '' ) );

					foreach ( $actions as $action ) {
						/** @var ParamsInterface $action */
						$actionId				=	$action->getInt( 'action', 0 );

						if ( ! $actionId ) {
							continue;
						}

						$internalGeneral[]		=	$actionId;
					}

					$internalGeneral			=	array_unique( $internalGeneral );
				}

				$batchActions					=	[];

				foreach ( $internalGeneral as $batchActionId ) {
					$batchActions[]				=	[ 'action' => $batchActionId ];
				}

				$batch->set( 'actions', json_encode( $batchActions ) );

				$batch->store();
			}

			// Old Internal Users trigger usage exists; migrate this to batch usage:
			if ( $internalUsers ) {
				$query							=	"SELECT *"
												.	"\n FROM " . $_CB_database->NameQuote( '#__comprofiler_plugin_autoactions_batches' )
												.	"\n WHERE ( " . $_CB_database->NameQuote( 'title' ) . " = " . $_CB_database->Quote( 'internalUsers' )
												.	" OR " . $_CB_database->NameQuote( 'users' ) . " = 1 )";
				$_CB_database->setQuery( $query, 0, 1 );
				$batch							=	new BatchTable();
				$_CB_database->loadObject( $batch );

				if ( ! $batch->getInt( 'id', 0 ) ) {
					$batch->set( 'title', 'internalUsers' );
					$batch->set( 'users', 1 );
					$batch->set( 'access', -1 );
					$batch->set( 'published', 1 );
					$batch->set( 'batch_limit', CBAutoActions::getGlobalParams()->getInt( 'batch_limit', 0 ) );
					$batch->set( 'batch_redirect', 0 );
				} else {
					$actions					=	new Registry( $batch->getString( 'actions', '' ) );

					foreach ( $actions as $action ) {
						/** @var ParamsInterface $action */
						$actionId				=	$action->getInt( 'action', 0 );

						if ( ! $actionId ) {
							continue;
						}

						$internalUsers[]		=	$actionId;
					}

					$internalUsers				=	array_unique( $internalUsers );
				}

				$batchActions					=	[];

				foreach ( $internalUsers as $batchActionId ) {
					$batchActions[]				=	[ 'action' => $batchActionId ];
				}

				$batch->set( 'actions', json_encode( $batchActions ) );

				$batch->store();
			}
		}

		// Migrate old system int usages to new string usages (this helps maintain published state):
		$legacyMap								=	[	1 => 'autologin', 100 => 'activityprofilelogin', 101 => 'activityprofilelogout', 102 => 'activityprofileregister',
														103 => 'activityprofileupdate', 104 => 'activityprofileavatar', 105 => 'activityprofilecanvas', 106 => 'activityaddconn',
														110 => 'activityremoveconn', 200 => 'activitycomment', 201 => 'activitytag', 300 => 'activitygallerycreate', 301 => 'activitygallerycomments',
														303 => 'activitygallerydelete', 400 => 'activityblogcreate', 402 => 'activityblogdelete', 500 => 'activitykunenacreate',
														501 => 'activitykunenareply', 502 => 'activitykunenadelete', 601 => 'activitygjgrpcreate', 604 => 'activitygjgrpdelete',
														606 => 'activitygjgrpjoin', 607 => 'activitygjgrpleave', 608 => 'activitygjwallcreate', 611 => 'activitygjwalldelete',
														612 => 'activitygjphotocreate', 614 => 'activitygjphotodelete', 615 => 'activitygjfilecreate', 617 => 'activitygjfiledelete',
														618 => 'activitygjvideocreate', 620 => 'activitygjvideodelete', 621 => 'activitygjeventscreate', 623 => 'activitygjeventsdelete'
													];

		foreach ( $legacyMap as $systemFrom => $systemTo ) {
			$query								=	"UPDATE " . $_CB_database->NameQuote( '#__comprofiler_plugin_autoactions' )
												.	"\n SET " . $_CB_database->NameQuote( 'system' ) . " = " . $_CB_database->Quote( $systemTo )
												.	"\n WHERE " . $_CB_database->NameQuote( 'system' ) . " = " . $_CB_database->Quote( $systemFrom );
			$_CB_database->setQuery( $query );
			$_CB_database->query();
		}
	}

	/**
	 * @param Table $trigger
	 * @param Registry $params
	 */
	public static function activity( Table $trigger, Registry $params ): void
	{
		// Legacy Usage:
		$activityCount							=	substr_count( $params->asJson(), 'activity_owner' );

		if ( $activityCount ) {
			$newParams							=	[];
			$newParams['activity']				=	[];

			$paramsMap							=	[	'activity_owner' => 'owner', 'activity_user' => 'user', 'activity_type' => 'type',
														'activity_subtype' => 'subtype', 'activity_item' => 'item', 'activity_from' => 'from',
														'activity_to' => 'to', 'activity_title' => 'title', 'activity_message' => 'message',
														'activity_icon' => 'icon', 'activity_class' => 'class'
													];

			for ( $i = 0, $n = $activityCount; $i < $n; $i++ ) {
				if ( $i === 0 ) {
					$i							=	null;
				}

				$activity						=	[];

				foreach ( $paramsMap as $old => $new ) {
					$activity[$new]				=	$params->getRaw( $old . $i );
				}

				$newParams['activity'][]		=	$activity;
			}

			$params->load( $newParams );
		}

		// CB Activity 4.0.0+ Usage:
		$newParams								=	[];

		foreach ( $params->subTree( 'activity' ) as $i => $activityRow ) {
			/** @var ParamsInterface $activityRow */
			if ( ! $activityRow instanceof ParamsInterface ) {
				$activityRow					=	new Registry( $activityRow );
			}

			$type								=	$activityRow->getString( 'type', '' );
			$subtype							=	$activityRow->getString( 'subtype', '' );
			$item								=	$activityRow->getString( 'item', '' );
			$parent								=	$activityRow->getString( 'parent', '' );
			$output								=	$activityRow->getString( 'output', '' );

			if ( $type ) {
				$asset							=	$type;

				if ( $parent ) {
					$asset						.=	'.' . $parent;
				}

				if ( $subtype ) {
					$asset						.=	'.' . $subtype;
				}

				if ( $item ) {
					$asset						.=	'.' . $item;
				}

				$activityRow->set( 'asset', $asset );

				$activityRow->unsetEntry( 'type' );
				$activityRow->unsetEntry( 'subtype' );
				$activityRow->unsetEntry( 'item' );
				$activityRow->unsetEntry( 'parent' );

				$newParams['activity'][$i]		=	$activityRow->asArray();
			} else {
				$source							=	$activityRow->getString( 'source', '' );

				if ( $source === 'profile' ) {
					$activityRow->set( 'asset', 'user.connections,following,global' );

					$activityRow->unsetEntry( 'source' );

					$newParams['activity'][$i]	=	$activityRow->asArray();
				} elseif ( $source === 'recent' ) {
					$activityRow->set( 'asset', 'all' );

					$activityRow->unsetEntry( 'source' );

					$newParams['activity'][$i]	=	$activityRow->asArray();
				}
			}

			if ( $output ) {
				$newParams['display']			=	strtolower( $output );

				$activityRow->unsetEntry( 'output' );

				$newParams['activity'][$i]		=	$activityRow->asArray();
			}
		}

		if ( $newParams ) {
			$params->load( $newParams );
		}
	}

	/**
	 * @param Table $trigger
	 * @param Registry $params
	 */
	public static function acymailing( Table $trigger, Registry $params ): void
	{
		if ( $params->has( 'acymailing_subscribe' ) ) {
			$newParams							=	[];
			$newParams['acymailing']			=	[];

			$paramsMap							=	[	'acymailing_subscribe' => 'subscribe', 'acymailing_unsubscribe' => 'unsubscribe',
														'acymailing_remove' => 'remove', 'acymailing_pending' => 'pending'
													];

			foreach ( $paramsMap as $old => $new ) {
				$newParams['acymailing'][$new]	=	$params->getRaw( $old );
			}

			$params->load( $newParams );
		}
	}

	/**
	 * @param Table $trigger
	 * @param Registry $params
	 */
	public static function antispam( Table $trigger, Registry $params ): void
	{
		$antispamCount						=	substr_count( $params->asJson(), 'antispam_value' );

		if ( $antispamCount ) {
			$newParams						=	[];
			$newParams['antispam']			=	[];

			$paramsMap						=	[	'antispam_mode' => 'mode', 'antispam_type' => 'type', 'antispam_value' => 'value',
													'antispam_duration' => 'duration', 'antispam_reason' => 'reason'
												];

			for ( $i = 0, $n = $antispamCount; $i < $n; $i++ ) {
				if ( $i === 0 ) {
					$i						=	null;
				}

				$antispam					=	[];

				foreach ( $paramsMap as $old => $new ) {
					$antispam[$new]			=	$params->getRaw( $old . $i );
				}

				$newParams['antispam'][]	=	$antispam;
			}

			$params->load( $newParams );
		}
	}

	/**
	 * @param Table $trigger
	 * @param Registry $params
	 */
	public static function cbsubs30( Table $trigger, Registry $params ): void
	{
		$cbsubsCount						=	substr_count( $params->asJson(), 'cbsubs30_plans' );

		if ( $cbsubsCount ) {
			$trigger->set( 'type', 'cbsubs' );

			$newParams						=	[];
			$newParams['cbsubs']			=	[];

			$paramsMap						=	[ 'cbsubs30_plans' => 'plans', 'cbsubs30_mode' => 'mode' ];

			for ( $i = 0, $n = $cbsubsCount; $i < $n; $i++ ) {
				if ( $i === 0 ) {
					$i						=	null;
				}

				$cbsubs						=	[];

				foreach ( $paramsMap as $old => $new ) {
					$cbsubs[$new]			=	$params->getRaw( $old . $i );
				}

				$newParams['cbsubs'][]		=	$cbsubs;
			}

			$params->load( $newParams );
		}
	}

	/**
	 * @param Table $trigger
	 * @param Registry $params
	 */
	public static function code( Table $trigger, Registry $params ): void
	{
		// Legacy Usage:
		$codeCount							=	substr_count( $params->asJson(), 'code_method' );

		if ( $codeCount ) {
			$newParams						=	[];
			$newParams['code']				=	[];

			$paramsMap						=	[	'code_method' => 'method', 'code_code' => 'code', 'code_pluginurls' => 'pluginurls',
													'code_plugins' => 'plugins', 'code_url' => 'url', 'code_return' => 'return'
												];

			for ( $i = 0, $n = $codeCount; $i < $n; $i++ ) {
				if ( $i === 0 ) {
					$i						=	null;
				}

				$code						=	[];

				foreach ( $paramsMap as $old => $new ) {
					$code[$new]				=	$params->getRaw( $old . $i );
				}

				$newParams['code'][]		=	$code;
			}

			$params->load( $newParams );
		}

		// 7.0.0 Usage:
		$newParams							=	[];

		foreach ( $params->subTree( 'code' ) as $i => $codeRow ) {
			/** @var ParamsInterface $codeRow */
			if ( ! $codeRow instanceof ParamsInterface ) {
				$codeRow					=	new Registry( $codeRow );
			}

			$output							=	$codeRow->getString( 'return', '' );

			if ( $output ) {
				if ( $output === 'SILENT' ) {
					$output					=	'none';
				}

				$newParams['display']		=	strtolower( $output );

				$codeRow->unsetEntry( 'output' );

				$newParams['code'][$i]		=	$codeRow->asArray();
			}
		}

		if ( $newParams ) {
			$params->load( $newParams );
		}
	}

	/**
	 * @param Table $trigger
	 * @param Registry $params
	 */
	public static function connection( Table $trigger, Registry $params ): void
	{
		$connectionCount					=	substr_count( $params->asJson(), 'connection_users' );

		if ( $connectionCount ) {
			$newParams						=	[];
			$newParams['connection']		=	[];

			$paramsMap						=	[ 'connection_users' => 'users', 'connection_message' => 'message', 'connection_direction' => 'direction' ];

			for ( $i = 0, $n = $connectionCount; $i < $n; $i++ ) {
				if ( $i === 0 ) {
					$i						=	null;
				}

				$connection					=	[];

				foreach ( $paramsMap as $old => $new ) {
					$connection[$new]		=	$params->getRaw( $old . $i );
				}

				$newParams['connection'][]	=	$connection;
			}

			$params->load( $newParams );
		}
	}

	/**
	 * @param Table $trigger
	 * @param Registry $params
	 */
	public static function content( Table $trigger, Registry $params ): void
	{
		$contentCount					=	substr_count( $params->asJson(), 'content_title' );

		if ( $contentCount ) {
			$newParams					=	[];
			$newParams['content']		=	[];

			$paramsMap					=	[	'content_mode' => 'mode', 'content_title' => 'title', 'content_alias' => 'alias',
												'content_category_j' => 'category_j', 'content_category_k' => 'category_k', 'content_introtext' => 'introtext',
												'content_fulltext' => 'fulltext', 'content_metadesc' => 'metadesc', 'content_metakey' => 'metakey',
												'content_access' => 'access', 'content_published' => 'published', 'content_featured' => 'featured',
												'content_language' => 'language', 'content_owner' => 'owner'
											];

			for ( $i = 0, $n = $contentCount; $i < $n; $i++ ) {
				if ( $i === 0 ) {
					$i					=	null;
				}

				$content				=	[];

				foreach ( $paramsMap as $old => $new ) {
					$content[$new]		=	$params->getRaw( $old . $i );
				}

				$newParams['content'][]	=	$content;
			}

			$params->load( $newParams );
		}
	}

	/**
	 * @param Table $trigger
	 * @param Registry $params
	 */
	public static function email( Table $trigger, Registry $params ): void
	{
		$emailCount						=	substr_count( $params->asJson(), 'email_to' );

		if ( $emailCount ) {
			$newParams					=	[];
			$newParams['email']			=	[];

			$paramsMap					=	[	'email_to' => 'to', 'email_subject' => 'subject', 'email_body' => 'body',
												'email_mode' => 'mode', 'email_cc' => 'cc', 'email_bcc' => 'bcc',
												'email_attachment' => 'attachment', 'email_replyto_address' => 'replyto_address', 'email_replyto_name' => 'replyto_name',
												'email_address' => 'from_address', 'email_name' => 'from_name', 'email_mailer' => 'mailer',
												'email_mailer_sendmail' => 'mailer_sendmail', 'email_mailer_smtpauth' => 'mailer_smtpauth', 'email_mailer_smtpsecure' => 'mailer_smtpsecure',
												'email_mailer_smtpport' => 'mailer_smtpport', 'email_mailer_smtpuser' => 'mailer_smtpuser', 'email_mailer_smtppass' => 'mailer_smtppass',
												'email_mailer_smtphost' => 'mailer_smtphost'
											];

			for ( $i = 0, $n = $emailCount; $i < $n; $i++ ) {
				if ( $i === 0 ) {
					$i					=	null;
				}

				$email					=	[];

				foreach ( $paramsMap as $old => $new ) {
					$email[$new]		=	$params->getRaw( $old . $i );
				}

				$newParams['email'][]	=	$email;
			}

			$params->load( $newParams );
		}
	}

	/**
	 * @param Table $trigger
	 * @param Registry $params
	 */
	public static function field( Table $trigger, Registry $params ): void
	{
		$fieldsCount					=	substr_count( $params->asJson(), 'field_id' );

		if ( $fieldsCount ) {
			$newParams					=	[];
			$newParams['field']			=	[];

			$paramsMap					=	[	'field_id' => 'field', 'field_operator' => 'operator',
												'field_value' => 'value', 'field_translate' => 'translate'
											];

			for ( $i = 0, $n = $fieldsCount; $i < $n; $i++ ) {
				if ( $i === 0 ) {
					$i					=	null;
				}

				$field					=	[];

				foreach ( $paramsMap as $old => $new ) {
					$field[$new]		=	$params->getRaw( $old . $i );
				}

				$newParams['field'][]	=	$field;
			}

			$params->load( $newParams );
		}
	}

	/**
	 * @param Table $trigger
	 * @param Registry $params
	 */
	public static function groupjive20( Table $trigger, Registry $params ): void
	{
		if ( $params->has( 'gj20_auto' ) ) {
			$trigger->set( 'type', 'groupjive' );

			$newParams									=	[];
			$newParams['groupjive']						=	[];

			$paramsMap									=	[	'gj20_auto' => 'mode', 'gj20_groups' => 'groups', 'gj20_grp_parent' => 'group_parent',
																'gj20_category' => 'category', 'gj20_cat_parent' => [ 'parent', 'category_parent' ],
																'gj20_status' => [ 'status', 'group_status' ], 'gj20_cat_unique' => 'category_unique',
																'gj20_types' => [ 'types', 'category_types' ], 'gj20_grp_autojoin' => 'autojoin',
																'gj20_type' => 'type', 'gj20_cat_description' => '', 'gj20_cat_owner' => '',
																'gj20_cat_name' => '', 'gj20_grp_name' => '', 'gj20_grp_description' => '',
																'gj20_grp_unique' => '', 'gj20_grp_owner' => ''
															];

			switch ( $params->getInt( 'gj20_auto', 1 ) ) {
				case 3:
					$paramsMap['gj20_cat_name']			=	'name';
					$paramsMap['gj20_cat_description']	=	'description';
					$paramsMap['gj20_cat_unique']		=	'unique';
					$paramsMap['gj20_cat_owner']		=	'owner';
					break;
				case 2:
					$paramsMap['gj20_cat_name']			=	'category_name';
					$paramsMap['gj20_cat_description']	=	'category_description';
					$paramsMap['gj20_grp_name']			=	'name';
					$paramsMap['gj20_grp_description']	=	'description';
					$paramsMap['gj20_grp_unique']		=	'unique';
					$paramsMap['gj20_grp_owner']		=	'owner';
					break;
			}

			$groupJive									=	[];

			foreach ( $paramsMap as $old => $new ) {
				if ( $new ) {
					if ( is_array( $new ) ) {
						foreach ( $new as $n ) {
							$groupJive[$n]				=	$params->getRaw( $old );
						}
					} else {
						$groupJive[$new]				=	$params->getRaw( $old );
					}
				}
			}

			$newParams['groupjive'][]					=	$groupJive;

			$params->load( $newParams );
		}
	}

	/**
	 * @param Table $trigger
	 * @param Registry $params
	 */
	public static function k2( Table $trigger, Registry $params ): void
	{
		if ( $params->has( 'k2_mode' ) ) {
			$newParams					=	[];
			$newParams['k2']			=	[];

			$paramsMap					=	[	'k2_mode' => 'mode', 'k2_user_group' => 'group', 'k2_gender' => 'gender',
												'k2_description' => 'description', 'k2_url' => 'url', 'k2_notes' => 'notes'
											];

			foreach ( $paramsMap as $old => $new ) {
				$newParams['k2'][$new]	=	$params->getRaw( $old );
			}

			$params->load( $newParams );
		}
	}

	/**
	 * @param Table $trigger
	 * @param Registry $params
	 */
	public static function kunena17( Table $trigger, Registry $params ): void
	{
		$kunenaCount						=	substr_count( $params->asJson(), 'kunena17_name' );

		if ( $kunenaCount ) {
			$trigger->set( 'type', 'kunena' );

			$newParams						=	[];
			$newParams['kunena']			=	[];

			$paramsMap						=	[ 'kunena17_name' => 'name', 'kunena17_parent' => 'parent', 'kunena17_description' => 'description' ];

			for ( $i = 0, $n = $kunenaCount; $i < $n; $i++ ) {
				if ( $i === 0 ) {
					$i						=	null;
				}

				$kunena						=	[ 'mode' => 'category' ];

				foreach ( $paramsMap as $old => $new ) {
					$kunena[$new]			=	$params->getRaw( $old . $i );
				}

				$newParams['kunena'][]		=	$kunena;
			}

			$params->load( $newParams );
		}
	}

	/**
	 * @param Table $trigger
	 * @param Registry $params
	 */
	public static function kunena20( Table $trigger, Registry $params ): void
	{
		$kunenaCount						=	substr_count( $params->asJson(), 'kunena20_name' );

		if ( $kunenaCount ) {
			$trigger->set( 'type', 'kunena' );

			$newParams						=	[];
			$newParams['kunena']			=	[];

			$paramsMap						=	[ 'kunena20_name' => 'name', 'kunena20_parent' => 'parent', 'kunena20_description' => 'description' ];

			for ( $i = 0, $n = $kunenaCount; $i < $n; $i++ ) {
				if ( $i === 0 ) {
					$i						=	null;
				}

				$kunena						=	[ 'mode' => 'category' ];

				foreach ( $paramsMap as $old => $new ) {
					$kunena[$new]			=	$params->getRaw( $old . $i );
				}

				$newParams['kunena'][]		=	$kunena;
			}

			$params->load( $newParams );
		}
	}

	/**
	 * @param Table $trigger
	 * @param Registry $params
	 */
	public static function loginlogout( Table $trigger, Registry $params ): void
	{
		if ( $params->has( 'loginlogout_mode' ) ) {
			$newParams							=	[];
			$newParams['loginlogout']			=	[];

			$paramsMap							=	[	'loginlogout_mode' => 'mode', 'loginlogout_method' => 'method', 'loginlogout_username' => 'username',
														'loginlogout_email' => 'email', 'loginlogout_redirect' => 'redirect', 'loginlogout_message' => 'message'
													];

			foreach ( $paramsMap as $old => $new ) {
				$newParams['loginlogout'][$new]	=	$params->getRaw( $old );
			}

			$params->load( $newParams );
		}
	}

	/**
	 * @param Table $trigger
	 * @param Registry $params
	 */
	public static function menu( Table $trigger, Registry $params ): void
	{
		$menuCount						=	substr_count( $params->asJson(), 'menu_title' );

		if ( $menuCount ) {
			$newParams					=	[];
			$newParams['menu']			=	[];

			$paramsMap					=	[	'menu_title' => 'title', 'menu_type' => 'type', 'menu_class' => 'class',
												'menu_position' => 'position', 'menu_url' => 'url', 'menu_target' => 'target',
												'menu_tooltip' => 'tooltip', 'menu_img' => 'image'
											];

			for ( $i = 0, $n = $menuCount; $i < $n; $i++ ) {
				if ( $i === 0 ) {
					$i					=	null;
				}

				$menu					=	[];

				foreach ( $paramsMap as $old => $new ) {
					$menu[$new]			=	$params->getRaw( $old . $i );
				}

				$newParams['menu'][]	=	$menu;
			}

			$params->load( $newParams );
		}
	}

	/**
	 * @param Table $trigger
	 * @param Registry $params
	 */
	public static function pms( Table $trigger, Registry $params ): void
	{
		$pmsCount						=	substr_count( $params->asJson(), 'pms_from' );

		if ( $pmsCount ) {
			$newParams					=	[];
			$newParams['pms']			=	[];

			$paramsMap					=	[	'pms_from' => 'from', 'pms_to' => 'to',
												'pms_subject' => 'subject', 'pms_message' => 'body'
											];

			for ( $i = 0, $n = $pmsCount; $i < $n; $i++ ) {
				if ( $i === 0 ) {
					$i					=	null;
				}

				$pms					=	[];

				foreach ( $paramsMap as $old => $new ) {
					$pms[$new]			=	$params->getRaw( $old . $i );
				}

				$newParams['pms'][]		=	$pms;
			}

			$params->load( $newParams );
		}
	}

	/**
	 * @param Table $trigger
	 * @param Registry $params
	 */
	public static function privacy( Table $trigger, Registry $params ): void
	{
		// Legacy Usage:
		$privacyCount							=	substr_count( $params->asJson(), 'privacy_user' );

		if ( $privacyCount ) {
			$newParams							=	[];
			$newParams['privacy']				=	[];

			$paramsMap							=	[	'privacy_user' => 'owner', 'privacy_type' => 'type', 'privacy_subtype' => 'subtype',
														'privacy_item' => 'item', 'privacy_rule' => 'rule'
													];

			for ( $i = 0, $n = $privacyCount; $i < $n; $i++ ) {
				if ( $i === 0 ) {
					$i							=	null;
				}

				$privacy						=	[];

				foreach ( $paramsMap as $old => $new ) {
					$privacy[$new]				=	$params->getRaw( $old . $i );
				}

				$newParams['privacy'][]			=	$privacy;
			}

			$params->load( $newParams );
		}

		// CB Privacy 5.0.0+ Usage:
		$newParams								=	[];

		foreach ( $params->subTree( 'privacy' ) as $i => $privacyRow ) {
			$migrate							=	false;

			/** @var ParamsInterface $privacyRow */
			if ( ! $privacyRow instanceof ParamsInterface ) {
				$privacyRow						=	new Registry( $privacyRow );
			}

			$method								=	$privacyRow->getString( 'method', '' );

			if ( ! $method ) {
				$privacyRow->set( 'method', 'create' );

				$migrate						=	true;
			}

			$type								=	$privacyRow->getString( 'type', '' );
			$subtype							=	$privacyRow->getString( 'subtype', '' );
			$item								=	$privacyRow->getString( 'item', '' );

			if ( $type ) {
				$asset							=	$type;

				if ( $subtype ) {
					$asset						.=	'.' . $subtype;
				}

				if ( $item ) {
					$asset						.=	'.' . $item;
				}

				$privacyRow->set( 'asset', $asset );

				$privacyRow->unsetEntry( 'type' );
				$privacyRow->unsetEntry( 'subtype' );
				$privacyRow->unsetEntry( 'item' );

				$migrate						=	true;
			}

			$rule								=	$privacyRow->getString( 'rule', '' );

			if ( $rule ) {
				$privacyRow->set( 'rules', $rule );

				$privacyRow->unsetEntry( 'rule' );

				$migrate						=	true;
			}

			if ( $migrate ) {
				$newParams['privacy'][$i]		=	$privacyRow->asArray();
			}
		}

		if ( $newParams ) {
			$params->load( $newParams );
		}
	}

	/**
	 * @param Table $trigger
	 * @param Registry $params
	 */
	public static function query( Table $trigger, Registry $params ): void
	{
		if ( $params->has( 'query_sql' ) ) {
			$newParams					=	[];
			$newParams['query']			=	[];

			$paramsMap					=	[	'query_sql' => 'sql', 'query_mode' => 'mode', 'query_host' => 'host',
												'query_username' => 'username', 'query_password' => 'password', 'query_database' => 'database',
												'query_charset' => 'charset', 'query_prefix' => 'prefix'
											];

			$query						=	[];

			foreach ( $paramsMap as $old => $new ) {
				$query[$new]			=	$params->getRaw( $old );
			}

			$newParams['query'][]		=	$query;

			$params->load( $newParams );
		}
	}

	/**
	 * @param Table $trigger
	 * @param Registry $params
	 */
	public static function redirect( Table $trigger, Registry $params ): void
	{
		if ( $params->has( 'redirect_url' ) ) {
			$newParams							=	[];
			$newParams['redirect']				=	[];

			$paramsMap							=	[ 'redirect_url' => 'url', 'redirect_message' => 'message', 'redirect_type' => 'type' ];

			foreach ( $paramsMap as $old => $new ) {
				$newParams['redirect'][$new]	=	$params->getRaw( $old );
			}

			$params->load( $newParams );
		}
	}

	/**
	 * @param Table $trigger
	 * @param Registry $params
	 */
	public static function registration( Table $trigger, Registry $params ): void
	{
		if ( $params->has( 'registration_username' ) ) {
			$newParams									=	[];
			$newParams['registration']					=	[];

			$paramsMap									=	[	'registration_approve' => 'approve', 'registration_confirm' => 'confirm', 'registration_usergroup' => 'usergroup',
																'registration_username' => 'username', 'registration_password' => 'password', 'registration_email' => 'email',
																'registration_firstname' => 'firstname', 'registration_middlename' => 'middlename', 'registration_lastname' => 'lastname',
																'registration_supress' => 'supress', 'registration_fields' => ''
															];

			$fields										=	$params->getRaw( 'registration_fields', '' );
			$newFields									=	[];

			if ( $fields ) {
				$fields									=	explode( "\n", $fields );

				foreach ( $fields as $pair ) {
					$field								=	explode( '=', trim( $pair ), 2 );

					if ( count( $field ) === 2 ) {
						$newFields[]					=	[ 'field' => trim( $field[0] ), 'value' => trim( $field[1] ), 'translate' => '1' ];
					}
				}
			}

			$newParams['registration']['fields']		=	$newFields;

			foreach ( $paramsMap as $old => $new ) {
				if ( $new ) {
					$newParams['registration'][$new]	=	$params->getRaw( $old );
				}
			}

			$params->load( $newParams );
		}
	}

	/**
	 * @param Table $trigger
	 * @param Registry $params
	 */
	public static function request( Table $trigger, Registry $params ): void
	{
		// Legacy Usage:
		$requestCount						=	substr_count( $params->asJson(), 'request_url' );

		if ( $requestCount ) {
			$newParams						=	[];
			$newParams['request']			=	[];

			$paramsMap						=	[	'request_url' => 'url', 'request_method' => 'method', 'request_request' => '',
													'request_return' => '', 'request_error' => '', 'request_debug' => ''
												];

			for ( $i = 0, $n = $requestCount; $i < $n; $i++ ) {
				if ( $i === 0 ) {
					$i						=	null;
				}

				$request					=	[];

				$data						=	$params->getRaw( "request_request$i", '' );
				$newData					=	[];

				if ( $data ) {
					$data					=	explode( "\n", $data );

					foreach ( $data as $pair ) {
						$dataPair			=	explode( '=', trim( $pair ), 2 );

						if ( count( $dataPair ) === 2 ) {
							$newData[]		=	[ 'key' => trim( $dataPair[0] ), 'value' => trim( $dataPair[1] ), 'translate' => '1' ];
						}
					}
				}

				$request['request']			=	$newData;

				foreach ( $paramsMap as $old => $new ) {
					if ( $new ) {
						$request[$new]		=	$params->getRaw( $old . $i );
					}
				}

				$newParams['request'][]		=	$request;
			}

			$params->load( $newParams );
		}

		// 7.0.0 Usage:
		$newParams							=	[];

		foreach ( $params->subTree( 'request' ) as $i => $requestRow ) {
			/** @var ParamsInterface $requestRow */
			if ( ! $requestRow instanceof ParamsInterface ) {
				$requestRow					=	new Registry( $requestRow );
			}

			$output							=	$requestRow->getString( 'return', '' );

			if ( $output ) {
				if ( $output === 'SILENT' ) {
					$output					=	'none';
				}

				$newParams['display']		=	strtolower( $output );

				$requestRow->unsetEntry( 'output' );

				$newParams['request'][$i]	=	$requestRow->asArray();
			}
		}

		if ( $newParams ) {
			$params->load( $newParams );
		}
	}

	/**
	 * @param Table $trigger
	 * @param Registry $params
	 */
	public static function usergroup( Table $trigger, Registry $params ): void
	{
		$usergroupCount						=	substr_count( $params->asJson(), 'usergroup_mode' );

		if ( $usergroupCount ) {
			$newParams						=	[];
			$newParams['usergroup']			=	[];

			$paramsMap						=	[	'usergroup_mode' => 'mode', 'usergroup_parent' => 'parent', 'usergroup_title' => 'title',
													'usergroup_add' => 'add', 'usergroup_groups' => 'groups'
												];

			for ( $i = 0, $n = $usergroupCount; $i < $n; $i++ ) {
				if ( $i === 0 ) {
					$i						=	null;
				}

				$usergroup					=	[];

				foreach ( $paramsMap as $old => $new ) {
					$usergroup[$new]		=	$params->getRaw( $old . $i );
				}

				$newParams['usergroup'][]	=	$usergroup;
			}

			$params->load( $newParams );
		}
	}
}