<?php
/**
* Community Builder (TM)
* @version $Id: $
* @package CommunityBuilder
* @copyright (C) 2004-2021 www.joomlapolis.com / Lightning MultiCom SA - and its licensors, all rights reserved
* @license http://www.gnu.org/licenses/old-licenses/gpl-2.0.html GNU/GPL version 2
*/

namespace CB\Plugin\GroupJive;

use CBLib\Application\Application;
use CBLib\Language\CBTxt;
use CBLib\Database\Table\TableInterface;
use CB\Database\Table\UserTable;
use CBLib\Registry\Registry;
use CB\Plugin\GroupJive\Table\CategoryTable;
use CB\Plugin\GroupJive\Table\GroupTable;
use CB\Plugin\GroupJive\Table\NotificationTable;
use CB\Plugin\GroupJive\Table\InviteTable;

defined('CBLIB') or die();

class CBGroupJive
{
	/** @var GroupTable[]  */
	private static $groupCache		=	array();

	/** @var CategoryTable[]  */
	private static $categoryCache	=	array();

	/** @var InviteTable[]  */
	private static $inviteCache		=	array();

	/** @var \CB\Plugin\GroupJive\Table\UserTable[]  */
	private static $userCache		=	array();

	/**
	 * @return Registry
	 */
	public static function getGlobalParams()
	{
		global $_PLUGINS;

		static $params	=	null;

		if ( $params === null ) {
			$plugin		=	$_PLUGINS->getLoadedPlugin( 'user', 'cbgroupjive' );
			$params		=	new Registry();

			if ( $plugin ) {
				$params->load( $plugin->params );
			}
		}

		/** @var Registry $params */
		return $params;
	}

	/**
	 * @param null|array $files
	 * @param bool       $loadGlobal
	 * @param bool       $loadHeader
	 * @param string     $integration
	 */
	public static function getTemplate( $files = null, $loadGlobal = true, $loadHeader = true, $integration = null )
	{
		global $_CB_framework, $_PLUGINS;

		static $tmpl							=	array();

		if ( ! $files ) {
			$files								=	array();
		} elseif ( ! is_array( $files ) ) {
			$files								=	array( $files );
		}

		$id										=	md5( serialize( array( $files, $loadGlobal, $loadHeader, $integration ) ) );

		if ( ! isset( $tmpl[$id] ) ) {
			static $plugin						=	null;

			if ( ! $plugin ) {
				$plugin							=	$_PLUGINS->getLoadedPlugin( 'user', 'cbgroupjive' );

				if ( ! $plugin ) {
					return;
				}
			}

			if ( $integration ) {
				self::getTemplate();

				static $integrations			=	array();

				if ( ! isset( $integrations[$integration] ) ) {
					$integrationPlugin			=	$_PLUGINS->getLoadedPlugin( 'user/plug_cbgroupjive/plugins', $integration );

					if ( ! $integrationPlugin ) {
						return;
					}

					$integrations[$integration]	=	$integrationPlugin;
				}

				$integrationPlugin				=	$integrations[$integration];
				$livePath						=	$_PLUGINS->getPluginLivePath( $integrationPlugin );
				$absPath						=	$_PLUGINS->getPluginPath( $integrationPlugin );
			} else {
				$livePath						=	$_PLUGINS->getPluginLivePath( $plugin );
				$absPath						=	$_PLUGINS->getPluginPath( $plugin );
			}

			$template							=	self::getGlobalParams()->getString( 'general_template', 'default' );
			$globalCss							=	'/templates/' . $template . '/template.css';
			$overrideCss						=	'/templates/' . $template . '/override.css';

			if ( $loadGlobal && $loadHeader ) {
				if ( ! file_exists( $absPath . $globalCss ) ) {
					$globalCss					=	'/templates/default/template.css';
				}

				if ( file_exists( $absPath . $globalCss ) ) {
					$_CB_framework->document->addHeadStyleSheet( $livePath . $globalCss );

					$paths['global_css']		=	$livePath . $globalCss;
				}
			}

			$paths								=	array( 'global_css' => null, 'php' => null, 'css' => null, 'js' => null, 'override_css' => null );

			foreach ( $files as $file ) {
				$file							=	preg_replace( '/[^-a-zA-Z0-9_]/', '', $file );

				if ( $file ) {
					$php						=	$absPath . '/templates/' . $template . '/' . $file . '.php';
					$css						=	'/templates/' . $template . '/' . $file . '.css';
					$js							=	'/templates/' . $template . '/' . $file . '.js';
				} else {
					$php						=	null;
					$css						=	null;
					$js							=	null;
				}

				if ( $file ) {
					if ( ! file_exists( $php ) ) {
						$php					=	$absPath . '/templates/default/' . $file . '.php';
					}

					if ( file_exists( $php ) ) {
						require_once( $php );

						$paths['php']			=	$php;
					}

					if ( $loadHeader ) {
						if ( ! file_exists( $absPath . $css ) ) {
							$css				=	'/templates/default/' . $file . '.css';
						}

						if ( file_exists( $absPath . $css ) ) {
							$_CB_framework->document->addHeadStyleSheet( $livePath . $css );

							$paths['css']		=	$livePath . $css;
						}

						if ( ! file_exists( $absPath . $js ) ) {
							$js					=	'/templates/default/' . $file . '.js';
						}

						if ( file_exists( $absPath . $js ) ) {
							$_CB_framework->document->addHeadScriptUrl( $livePath . $js );

							$paths['js']		=	$livePath . $js;
						}
					}
				}
			}

			if ( $loadGlobal && $loadHeader && file_exists( $absPath . $overrideCss ) ) {
				$_CB_framework->document->addHeadStyleSheet( $livePath . $overrideCss );

				$paths['override_css']			=	$livePath . $overrideCss;
			}

			$tmpl[$id]							=	$paths;
		}
	}

	/**
	 * Sends a PM or Email notification with substitutions based off configuration
	 *
	 * @param string               $notificationType
	 * @param int                  $type 1: Email, 2: PM, 3: Moderators, 4: Auto
	 * @param UserTable|int|null   $from
	 * @param UserTable|int|string $to
	 * @param string               $subject
	 * @param string               $body
	 * @param GroupTable           $group
	 * @param array                $extra
	 * @return bool
	 */
	public static function sendNotification( $notificationType, $type, $from, $to, $subject, $body, $group, $extra = array() )
	{
		global $_CB_framework, $_PLUGINS;

		if ( ( ! $subject )
			 || ( ! $body )
			 || ( ! $group->getInt( 'id', 0 ) )
			 || ( ! $group->category()->getInt( 'published', 1 ) )
			 || ( ( $type !== 3 ) && ( ( $group->getInt( 'published', 1 ) !== 1 ) || ( ! $to ) ) )
		) {
			return false;
		}

		if ( $from instanceof UserTable ) {
			$fromUser			=	$from;
		} elseif ( is_int( $from ) ) {
			$fromUser			=	\CBuser::getUserDataInstance( $from );
		} else {
			$fromUser			=	null;
		}

		if ( $to instanceof UserTable ) {
			$toUser				=	$to;
		} elseif ( is_int( $to ) ) {
			$toUser				=	\CBuser::getUserDataInstance( $to );
		} else {
			$toUser				=	null;
		}

		if ( $fromUser && $toUser && ( $fromUser->getInt( 'id', 0 ) === $toUser->getInt( 'id', 0 ) ) ) {
			return false;
		}

		$notifyBy				=	self::getGlobalParams()->getInt( 'notifications_notifyby', 1 );
		$fromName				=	self::getGlobalParams()->getString( 'notifications_from_name' );
		$fromEmail				=	self::getGlobalParams()->getString( 'notifications_from_address' );
		$cbUser					=	\CBuser::getInstance( ( $fromUser ? $fromUser->getInt( 'id', 0 ) : ( $toUser ? $toUser->getInt( 'id', 0 ) : 0 ) ), false );
		$user					=	$cbUser->getUserData();

		$extras					=	array(	'category_id'			=>	$group->category()->getInt( 'id', 0 ),
											'category_name'			=>	( $group->category()->getInt( 'id', 0 ) ? CBTxt::T( $group->category()->getString( 'name' ) ) : CBTxt::T( 'Uncategorized' ) ),
											'category_description'	=>	( $group->category()->getInt( 'id', 0 ) ? htmlspecialchars( CBTxt::T( $group->category()->getString( 'description' ) ) ) : null ),
											'category'				=>	'<a href="' . $_CB_framework->pluginClassUrl( 'cbgroupjive', false, array( 'action' => 'categories', 'func' => 'show', 'id' => $group->getInt( 'category', 0 ) ) ) . '">' . ( $group->category()->getInt( 'id', 0 ) ? CBTxt::T( $group->category()->getString( 'name' ) ) : CBTxt::T( 'Uncategorized' ) ) . '</a>',
											'group_id'				=>	$group->getInt( 'id', 0 ),
											'group_name'			=>	htmlspecialchars( CBTxt::T( $group->getString( 'name' ) ) ),
											'group_description'		=>	htmlspecialchars( CBTxt::T( $group->getString( 'description' ) ) ),
											'group'					=>	'<a href="' . $_CB_framework->pluginClassUrl( 'cbgroupjive', false, array( 'action' => 'groups', 'func' => 'show', 'id' => $group->getInt( 'id', 0 ) ) ) . '">' . htmlspecialchars( CBTxt::T( $group->getString( 'name' ) ) ) . '</a>',
											'user'					=>	'<a href="' . $_CB_framework->viewUrl( 'userprofile', false, array( 'user' => $user->getInt( 'id', 0 ) ) ) . '">' . $user->getFormattedName() . '</a>'
										);

		if ( ! $toUser ) {
			$extras['email']	=	$to;
			$extras['name']		=	$to;
			$extras['username']	=	$to;
		}

		$extras					=	array_merge( $extras, $extra );
		$subject				=	$cbUser->replaceUserVars( $subject, true, false, $extras, false );
		$body					=	$cbUser->replaceUserVars( $body, false, false, $extras, false );

		if ( $type === 4 ) {
			$type				=	( $notifyBy === 2 ? 2 : 1 );
		}

		if ( ( $type === 2 ) && ( ! $toUser ) ) {
			// Can't send a PM to a user that doesn't exist so lets fallback to email:
			$type				=	1;
		}

		$_PLUGINS->trigger( 'gj_onSendNotification', array( $notificationType, &$type, &$fromUser, &$toUser, &$subject, &$body, $group, $extras ) );

		$notification			=	new \cbNotification();

		if ( $type === 3 ) {
			// Moderator Notification:
			$notification->sendToModerators( $subject, $body, false, 1 );
		} elseif ( ( $type === 2 ) && $toUser ) {
			// PM Notification:
			if ( ! $toUser->getInt( 'id', 0 ) ) {
				return false;
			}

			$notification->sendUserPMSmsg( $toUser, 0, $subject, $body, true, false, 1, $extras );
		} elseif ( $type === 1 ) {
			// Email Notification:
			if ( $toUser ) {
				if ( ! $toUser->getInt( 'id', 0 ) ) {
					return false;
				}

				$notification->sendFromSystem( $toUser, $subject, $body, 1, 1, null, null, null, $extras, true, $fromName, $fromEmail );
			} else {
				$userTo			=	new UserTable();

				$userTo->set( 'email', $to );
				$userTo->set( 'name', $to );
				$userTo->set( 'username', $to );

				$notification->sendFromSystem( $userTo, $subject, $body, 1, 1, null, null, null, $extras, true, $fromName, $fromEmail );
			}
		}

		return true;
	}

	/**
	 * Parses for users set to receive a notification and sends it to them
	 *
	 * @param string             $notification The notification to send
	 * @param string             $subject
	 * @param string             $body
	 * @param GroupTable         $group        Group for this notification
	 * @param UserTable|int|null $from         UserTable|int: Specific user to notify from (used for substitutions), Null: Notify from self
	 * @param UserTable|int|null $to           UserTable|int: Specific user to notify, Null: Notify everyone elegible
	 * @param array              $skip         Array of user ids to skip
	 * @param int                $status       Group status restriction for notifications (e.g. 2: Group Moderators and above)
	 * @param array              $extra
	 * @return bool
	 */
	public static function sendNotifications( $notification, $subject, $body, $group, $from = null, $to = null, $skip = array(), $status = 1, $extra = array() )
	{
		global $_CB_database;

		if ( is_int( $from ) ) {
			$from					=	\CBuser::getUserDataInstance( $from );
		}

		if ( is_int( $to ) ) {
			$to						=	\CBuser::getUserDataInstance( $to );
		}

		$myId						=	Application::MyUser()->getUserId();

		if ( ( ! $notification ) || ( ! $subject ) || ( ! $body ) ) {
			return false;
		}

		if ( $to && ( $to->getInt( 'id', 0 ) === $myId ) ) {
			return false;
		}

		if ( $from && $to && ( $from->getInt( 'id', 0 ) === $to->getInt( 'id', 0 ) ) ) {
			return false;
		}

		if ( ( ! $group->getInt( 'id', 0 ) ) || ( $group->getInt( 'published', 0 ) !== 1 ) ) {
			return false;
		}

		if ( $group->category()->getInt( 'id', 0 ) && ( ! $group->category()->getInt( 'published', 0 ) ) ) {
			return false;
		}

		if ( ( ! $group->category()->getInt( 'id', 0 ) ) && ( ! self::getGlobalParams()->getInt( 'groups_uncategorized', 1 ) ) ) {
			return false;
		}

		if ( ! self::getGlobalParams()->getInt( 'notifications', 1 ) ) {
			return false;
		}

		if ( ! $status ) {
			$status					=	1;
		}

		if ( ! is_array( $skip ) ) {
			$skip					=	array( $skip );
		}

		if ( $from ) {
			$skip[]					=	$from->getInt( 'id', 0 );
		}

		$moderators					=	Application::CmsPermissions()->getGroupsOfViewAccessLevel( Application::Config()->getInt( 'moderator_viewaccesslevel', 3 ), true );

		$query						=	'SELECT DISTINCT n.*'
									.	', u.' . $_CB_database->NameQuote( 'status' )
									.	"\n FROM " . $_CB_database->NameQuote( '#__groupjive_notifications' ) . " AS n"
									.	"\n LEFT JOIN " . $_CB_database->NameQuote( '#__groupjive_users' ) . " AS u"
									.	' ON u.' . $_CB_database->NameQuote( 'user_id' ) . ' = n.' . $_CB_database->NameQuote( 'user_id' )
									.	' AND u.' . $_CB_database->NameQuote( 'group' ) . ' = n.' . $_CB_database->NameQuote( 'group' )
									.	"\n LEFT JOIN " . $_CB_database->NameQuote( '#__comprofiler' ) . " AS cb"
									.	' ON cb.' . $_CB_database->NameQuote( 'id' ) . ' = n.' . $_CB_database->NameQuote( 'user_id' )
									.	"\n LEFT JOIN " . $_CB_database->NameQuote( '#__users' ) . " AS j"
									.	' ON j.' . $_CB_database->NameQuote( 'id' ) . ' = n.' . $_CB_database->NameQuote( 'user_id' )
									.	"\n LEFT JOIN " . $_CB_database->NameQuote( '#__user_usergroup_map' ) . " AS g"
									.	' ON g.' . $_CB_database->NameQuote( 'user_id' ) . ' = n.' . $_CB_database->NameQuote( 'user_id' )
									.	"\n WHERE n." . $_CB_database->NameQuote( 'group' ) . " = " . $group->getInt( 'id', 0 );

		if ( $to ) {
			$query					.=	"\n AND n." . $_CB_database->NameQuote( 'user_id' ) . " = " . $to->getInt( 'id', 0 );
		} else {
			$query					.=	"\n AND n." . $_CB_database->NameQuote( 'user_id' ) . " != " . $myId;
		}

		if ( $skip ) {
			$query					.=	"\n AND n." . $_CB_database->NameQuote( 'user_id' ) . " NOT IN " . $_CB_database->safeArrayOfIntegers( $skip );
		}

		$query						.=	"\n AND cb." . $_CB_database->NameQuote( 'approved' ) . " = 1"
									.	"\n AND cb." . $_CB_database->NameQuote( 'confirmed' ) . " = 1"
									.	"\n AND j." . $_CB_database->NameQuote( 'block' ) . " = 0"
									.	"\n AND ( n." . $_CB_database->NameQuote( 'user_id' ) . " = " . $group->getInt( 'user_id', 0 )
									.		' OR u.' . $_CB_database->NameQuote( 'status' ) . " >= " . (int) $status
									.		( $moderators ? ' OR g.' . $_CB_database->NameQuote( 'group_id' ) . " IN " . $_CB_database->safeArrayOfIntegers( $moderators ) : null ) . ' )';
		$_CB_database->setQuery( $query );
		$rows						=	$_CB_database->loadObjectList( null, '\CB\Plugin\GroupJive\Table\NotificationTable', array( $_CB_database ) );

		self::preFetchUsers( $rows );

		/** @var NotificationTable[] $rows */
		foreach ( $rows as $row ) {
			if ( ! $row->params()->getInt( $notification, 0 ) ) {
				continue;
			}

			if ( $to ) {
				$notifyUser			=	$to;
			} else {
				$notifyUser			=	\CBuser::getUserDataInstance( $row->getInt( 'user_id', 0 ) );
			}

			$group->set( '_user_status', $row->getInt( 'status', 0 ) );

			if ( ! self::canAccessGroup( $group, $notifyUser ) ) {
				continue;
			}

			self::sendNotification( $notification, 4, $from, $notifyUser, $subject, $body, $group, $extra );
		}

		return true;
	}

	/**
	 * Returns the current return url or generates one from current page
	 *
	 * @param bool|false $current
	 * @param bool|false $raw
	 * @return null|string
	 */
	public static function getReturn( $current = false, $raw = false )
	{
		static $cache				=	array();

		if ( ! isset( $cache[$current] ) ) {
			$url					=	null;

			if ( $current ) {
				$returnUrl			=	Application::Input()->getBase64( 'get/return' );

				if ( $returnUrl ) {
					$returnUrl		=	base64_decode( $returnUrl );

					if ( \JUri::isInternal( $returnUrl ) ) {
						$url		=	$returnUrl;
					}
				}
			} else {
				$isHttps			=	( isset( $_SERVER['HTTPS'] ) && ( ! empty( $_SERVER['HTTPS'] ) ) && ( $_SERVER['HTTPS'] !== 'off' ) );
				$returnUrl			=	'http' . ( $isHttps ? 's' : '' ) . '://' . $_SERVER['HTTP_HOST'];

				if ( ( ! empty( $_SERVER['PHP_SELF'] ) ) && ( ! empty( $_SERVER['REQUEST_URI'] ) ) ) {
					$returnUrl		.=	$_SERVER['REQUEST_URI'];
				} else {
					$returnUrl		.=	$_SERVER['SCRIPT_NAME'];

					if ( isset( $_SERVER['QUERY_STRING'] ) && ( ! empty( $_SERVER['QUERY_STRING'] ) ) ) {
						$returnUrl	.=	'?' . $_SERVER['QUERY_STRING'];
					}
				}

				$url				=	cbUnHtmlspecialchars( preg_replace( '/[\\\"\\\'][\\s]*javascript:(.*)[\\\"\\\']/', '""', preg_replace( '/eval\((.*)\)/', '', htmlspecialchars( urldecode( $returnUrl ) ) ) ) );
			}

			$cache[$current]		=	$url;
		}

		$return						=	$cache[$current];

		if ( ( ! $raw ) && $return ) {
			$return					=	base64_encode( $return );
		}

		return $return;
	}

	/**
	 * Redirects to the return url if available otherwise to the url specified
	 *
	 * @param string      $url
	 * @param null|string $message
	 * @param string      $messageType
	 */
	public static function returnRedirect( $url, $message = null, $messageType = 'message' )
	{
		$returnUrl		=	self::getReturn( true, true );

		cbRedirect( ( $returnUrl ? $returnUrl : $url ), $message, $messageType );
	}

	/**
	 * Creates a directory path from base to category or group
	 *
	 * @param string   $basePath
	 * @param int|null $category
	 * @param int|null $group
	 */
	public static function createDirectory( $basePath, $category = null, $group = null )
	{
		global $_CB_framework;

		if ( ! $basePath ) {
			return;
		}

		$indexPath					=	$_CB_framework->getCfg( 'absolute_path' ) . '/components/com_comprofiler/plugin/user/plug_cbgroupjive/index.html';

		if ( ! is_dir( $basePath ) ) {
			$oldMask				=	@umask( 0 );

			if ( @mkdir( $basePath, 0755, true ) ) {
				@umask( $oldMask );
				@chmod( $basePath, 0755 );

				if ( ! file_exists( $basePath . '/index.html' ) ) {
					@copy( $indexPath, $basePath . '/index.html' );
					@chmod( $basePath . '/index.html', 0755 );
				}
			} else {
				@umask( $oldMask );
			}
		}

		if ( $category !== null ) {
			$categoryPath			=	$basePath . '/' . (int) $category;

			if ( ! is_dir( $categoryPath ) ) {
				$oldMask			=	@umask( 0 );

				if ( @mkdir( $categoryPath, 0755, true ) ) {
					@umask( $oldMask );
					@chmod( $categoryPath, 0755 );

					if ( ! file_exists( $categoryPath . '/index.html' ) ) {
						@copy( $indexPath, $categoryPath . '/index.html' );
						@chmod( $categoryPath . '/index.html', 0755 );
					}
				} else {
					@umask( $oldMask );
				}
			}

			if ( $group !== null ) {
				$groupPath			=	$categoryPath . '/' . (int) $group;

				if ( ! is_dir( $groupPath ) ) {
					$oldMask		=	@umask( 0 );

					if ( @mkdir( $groupPath, 0755, true ) ) {
						@umask( $oldMask );
						@chmod( $groupPath, 0755 );

						if ( ! file_exists( $groupPath . '/index.html' ) ) {
							@copy( $indexPath, $groupPath . '/index.html' );
							@chmod( $groupPath . '/index.html', 0755 );
						}
					} else {
						@umask( $oldMask );
					}
				}
			}
		}
	}

	/**
	 * Recursively delete a folder and its contents
	 *
	 * @param string $source
	 */
	public static function deleteDirectory( $source )
	{
		if ( is_dir( $source ) ) {
			$source				=	str_replace( '\\', '/', realpath( $source ) );

			if ( is_dir( $source ) ) {
				$files			=	new \RecursiveIteratorIterator( new \RecursiveDirectoryIterator( $source ), \RecursiveIteratorIterator::CHILD_FIRST );

				if ( $files ) {
					foreach ( $files as $file ) {
						$file	=	str_replace( '\\', '/', realpath( $file ) );

						if ( is_dir( $file ) ) {
							@rmdir( $file );
						} elseif ( is_file( $file ) ) {
							@unlink( $file );
						}
					}
				}

				@rmdir( $source );
			}
		}
	}

	/**
	 * recursively copy a folder and its contents
	 *
	 * @param string $source
	 * @param string $destination
	 * @param bool   $deleteSource
	 */
	public static function copyDirectory( $source, $destination, $deleteSource = false )
	{
		if ( is_dir( $source ) ) {
			$source					=	str_replace( '\\', '/', realpath( $source ) );
			$oldmask				=	@umask( 0 );

			if ( ! file_exists( $destination ) ) {
				@mkdir( $destination, 0755 );
			}

			$destination			=	str_replace( '\\', '/', realpath( $destination ) );

			if ( is_dir( $destination ) ) {
				$files				=	new \RecursiveIteratorIterator( new \RecursiveDirectoryIterator( $source ), \RecursiveIteratorIterator::SELF_FIRST );

				if ( $files ) {
					foreach ( $files as $file ) {
						$file		=	str_replace( '\\', '/', realpath( $file ) );

						if ( is_dir( $file ) ) {
							@mkdir( str_replace( $source . '/', $destination . '/', $file . '/' ), 0755 );
						} elseif ( is_file( $file ) ) {
							$copy	=	str_replace( $source . '/', $destination . '/', $file );

							@copy( $file, $copy );
							@chmod( $copy, 0755 );
						}
					}
				}
			}

			@umask( $oldmask );

			if ( $deleteSource ) {
				self::deleteDirectory( $source );
			}
		}
	}

	/**
	 * Uploads category or group canvas or logo
	 *
	 * @param string                   $type
	 * @param CategoryTable|GroupTable $row
	 * @return bool
	 */
	public static function uploadImage( $type, &$row )
	{
		global $_CB_framework;

		if ( Application::Application()->isClient( 'administrator' ) ) {
			$input						=	Application::Input();
			$files						=	$input->getNamespaceRegistry( 'files' );
		} else {
			$input						=	$row->getRaw( '_input', new Registry() );
			$files						=	$row->getRaw( '_files', new Registry() );
		}

		if ( ( ! $type ) || ( ! in_array( $type, array( 'canvas', 'logo' ), true ) ) ) {
			return false;
		}

		$method							=	$input->getInt( 'post/' . $type . '_method' );

		if ( $method === 0 ) {
			return true;
		}

		$basePath						=	$_CB_framework->getCfg( 'absolute_path' ) . '/images/comprofiler/plug_cbgroupjive';

		if ( $row instanceof GroupTable ) {
			$imagePath					=	$basePath . '/' . $row->getInt( 'category', 0 ) . '/' . $row->getInt( 'id', 0 );
		} else {
			$imagePath					=	$basePath . '/' . $row->getInt( 'id', 0 );
		}

		$upload							=	$files->subTree( $type );
		$uploadFile						=	$upload->getString( 'tmp_name' );

		if ( ( ( $method === null ) || ( $method === 1 ) ) && $uploadFile ) {
			if ( $row instanceof GroupTable ) {
				self::createDirectory( $basePath, $row->getInt( 'category', 0 ), $row->getInt( 'id', 0 ) );
			} else {
				self::createDirectory( $basePath, $row->getInt( 'id', 0 ) );
			}

			$resample					=	self::getGlobalParams()->getInt( $type . '_resample', 1 );
			$aspectRatio				=	self::getGlobalParams()->getInt( $type . '_maintain_aspect_ratio', 1 );
			$imageHeight				=	self::getGlobalParams()->getInt( $type . '_image_height', 640 );

			if ( ! $imageHeight ) {
				$imageHeight			=	640;
			}

			$imageWidth					=	self::getGlobalParams()->getInt( $type . '_image_width', 1280 );

			if ( ! $imageWidth ) {
				$imageWidth				=	1280;
			}

			$thumbHeight				=	self::getGlobalParams()->getInt( $type . '_thumbnail_height', 320 );

			if ( ! $thumbHeight ) {
				$thumbHeight			=	320;
			}

			$thumbWidth					=	self::getGlobalParams()->getInt( $type . '_thumbnail_width', 640 );

			if ( ! $thumbWidth ) {
				$thumbWidth				=	640;
			}

			$conversionType				=	Application::Config()->getInt( 'conversiontype', 0 );
			$imageSoftware				=	( $conversionType === 5 ? 'gmagick' : ( $conversionType === 1 ? 'imagick' : ( $conversionType === 4 ? 'gd' : 'auto' ) ) );
			$imageId					=	uniqid( '', true );

			try {
				$image					=	new \CBLib\Image\Image( $imageSoftware, $resample, $aspectRatio );

				$image->setName( $imageId );
				$image->setSource( $upload->asArray() );
				$image->setDestination( $imagePath . '/' );

				$image->processImage( $imageWidth, $imageHeight );

				$newFileName			=	$image->getCleanFilename();

				$image->setName( 'tn' . $imageId );

				$image->processImage( $thumbWidth, $thumbHeight );

				if ( $row->getString( $type ) ) {
					$oldImage			=	$imagePath . '/' . $row->getString( $type );

					if ( file_exists( $oldImage ) ) {
						@unlink( $oldImage );
					}

					$oldThumbnail		=	$imagePath . '/tn' . $row->getString( $type );

					if ( file_exists( $oldThumbnail ) ) {
						@unlink( $oldThumbnail );
					}
				}

				$row->set( $type, $newFileName );
			} catch ( \Exception $e ) {
				$row->setError( $e->getMessage() );

				return false;
			}
		} elseif ( ( $method === 2 ) && $row->getString( $type ) ) {
			$image						=	$imagePath . '/' . $row->getString( $type );

			if ( file_exists( $image ) ) {
				@unlink( $image );
			}

			$thumbnail					=	$imagePath . '/tn' . $row->getString( $type );

			if ( file_exists( $thumbnail ) ) {
				@unlink( $thumbnail );
			}

			$row->set( $type, '' );
		}

		return true;
	}

	/**
	 * @param UserTable          $user
	 * @param CategoryTable|null $category
	 * @return bool
	 */
	public static function canCreateGroup( $user, $category = null )
	{
		global $_CB_database, $_PLUGINS;

		if ( ! $user->getInt( 'id', 0 ) ) {
			return false;
		}

		if ( self::isModerator( $user->getInt( 'id', 0 ) ) ) {
			return true;
		}

		static $cache							=	array();

		$id										=	md5( $user->getInt( 'id', 0 ) . ( $category ? $category->getInt( 'id', 0 ) : 0 ) );

		if ( ! isset( $cache[$id] ) ) {
			$cache[$id]							=	false;

			if ( $category !== null ) {
				if ( ! $category->getInt( 'id', 0 ) ) {
					if ( ! self::getGlobalParams()->getInt( 'groups_uncategorized', 1 ) ) {
						$cache[$id]				=	false;

						return false;
					}
				} else {
					if ( ( ! $category->getInt( 'published', 0 ) ) || ( ! self::canAccess( $category->getInt( 'access', 0 ), $user->getInt( 'id', 0 ) ) ) ) {
						$cache[$id]				=	false;

						return false;
					}

					$createAccess				=	$category->getInt( 'create_access', 0 );

					if ( ( $createAccess !== 0 ) && ( ( $createAccess === -1 ) || ( ! self::canAccess( $createAccess, $user->getInt( 'id', 0 ) ) ) ) ) {
						$cache[$id]				=	false;

						return false;
					}
				}
			}

			$createAccess						=	self::getGlobalParams()->getInt( 'groups_create_access', 2 );

			if ( $createAccess === -1 ) {
				$cache[$id]						=	false;

				return false;
			}

			if ( self::canAccess( $createAccess, $user->getInt( 'id', 0 ) ) ) {
				$createLimit					=	self::getCreateLimit( $user, self::getGlobalParams()->getString( 'groups_create_limit', 'custom' ), self::getGlobalParams()->getInt( 'groups_create_limit_custom', 0 ) );

				if ( $createLimit ) {
					static $count				=	array();

					$countId					=	$user->getInt( 'id', 0 );

					if ( ! isset( $count[$countId] ) ) {
						$query					=	'SELECT COUNT(*)'
												.	"\n FROM " . $_CB_database->NameQuote( '#__groupjive_groups' )
												.	"\n WHERE " . $_CB_database->NameQuote( 'user_id' ) . " = " . $countId;
						$_CB_database->setQuery( $query );
						$count[$countId]		=	(int) $_CB_database->loadResult();
					}

					if ( $count[$countId] >= $createLimit ) {
						$cache[$id]				=	false;

						return false;
					}
				}

				$access							=	true;

				$_PLUGINS->trigger( 'gj_onCanCreateGroup', array( &$access, $category, $user ) );

				$cache[$id]						=	$access;

				return $access;
			}
		}

		return $cache[$id];
	}

	/**
	 * Caches a prefetched array of categories for reuse
	 *
	 * @param CategoryTable[] $categories
	 */
	public static function prefetchCategories( $categories )
	{
		foreach ( $categories as $category ) {
			$rowId							=	$category->getInt( 'id', 0 );

			if ( ! $rowId ) {
				continue;
			}

			self::$categoryCache[$rowId]	=	$category;
		}
	}

	/**
	 * returns a cached category object
	 *
	 * @param int $id
	 * @return CategoryTable
	 */
	public static function getCategory( $id )
	{
		if ( ! $id ) {
			return new CategoryTable();
		}

		if ( ! isset( self::$categoryCache[$id] ) ) {
			$row						=	new CategoryTable();

			$row->load( (int) $id );

			self::$categoryCache[$id]	=	$row;
		}

		return self::$categoryCache[$id];
	}

	/**
	 * Caches a prefetched array of groups for reuse
	 *
	 * @param GroupTable[] $groups
	 */
	public static function prefetchGroups( $groups )
	{
		foreach ( $groups as $group ) {
			$rowId						=	$group->getInt( 'id', 0 );

			if ( ! $rowId ) {
				continue;
			}

			self::$groupCache[$rowId]	=	$group;
		}
	}

	/**
	 * returns a cached group object
	 *
	 * @param int $id
	 * @return GroupTable
	 */
	public static function getGroup( $id )
	{
		if ( ! $id ) {
			return new GroupTable();
		}

		if ( ! isset( self::$groupCache[$id] ) ) {
			$row					=	new GroupTable();

			$row->load( (int) $id );

			self::$groupCache[$id]	=	$row;
		}

		return self::$groupCache[$id];
	}

	/**
	 * Caches a prefetched array of group users for reuse
	 *
	 * @param \CB\Plugin\GroupJive\Table\UserTable[] $groups
	 */
	public static function prefetchGroupUsers( $users )
	{
		foreach ( $users as $user ) {
			$rowId						=	$user->getInt( 'id', 0 );

			if ( ! $rowId ) {
				continue;
			}

			self::$userCache[$rowId]	=	$user;
		}
	}

	/**
	 * returns a cached group user object
	 *
	 * @param int $id
	 * @return \CB\Plugin\GroupJive\Table\UserTable
	 */
	public static function getUser( $id )
	{
		if ( ! $id ) {
			return new \CB\Plugin\GroupJive\Table\UserTable();
		}

		if ( ! isset( self::$userCache[$id] ) ) {
			$row					=	new \CB\Plugin\GroupJive\Table\UserTable();

			$row->load( (int) $id );

			self::$userCache[$id]	=	$row;
		}

		return self::$userCache[$id];
	}

	/**
	 * Caches a prefetched array of invites for reuse
	 *
	 * @param InviteTable[] $invites
	 */
	public static function prefetchInvites( $invites )
	{
		foreach ( $invites as $invite ) {
			$rowId						=	$invite->getInt( 'id', 0 );

			if ( ! $rowId ) {
				continue;
			}

			self::$inviteCache[$rowId]	=	$invite;
		}
	}

	/**
	 * returns a cached group invite object
	 *
	 * @param int $id
	 * @return InviteTable
	 */
	public static function getInvite( $id )
	{
		if ( ! $id ) {
			return new InviteTable();
		}

		if ( ! isset( self::$inviteCache[$id] ) ) {
			$row						=	new InviteTable();

			$row->load( (int) $id );

			self::$inviteCache[$id]		=	$row;
		}

		return self::$inviteCache[$id];
	}

	/**
	 * @param UserTable  $user
	 * @param GroupTable $group
	 * @return null|int|bool    -1: Banned, 0: Pending, 1: Active|Member, 2: Moderator, 3: Admin, 4: Owner
	 */
	public static function getGroupStatus( $user, $group )
	{
		global $_CB_database;

		if ( ( ! $user->getInt( 'id', 0 ) ) || ( ! $group->getInt( 'id', 0 ) ) ) {
			return false;
		}

		if ( $user->getInt( 'id', 0 ) === $group->getInt( 'user_id', 0 ) ) {
			return 4;
		}

		if ( isset( $group->_user_status ) ) {
			return $group->getInt( '_user_status' );
		}

		static $users			=	array();

		$groupId				=	$group->getInt( 'id', 0 );

		if ( ! isset( $users[$groupId] ) ) {
			$query				=	'SELECT ' . $_CB_database->NameQuote( 'user_id' )
								.	', ' . $_CB_database->NameQuote( 'status' )
								.	"\n FROM " . $_CB_database->NameQuote( '#__groupjive_users' )
								.	"\n WHERE " . $_CB_database->NameQuote( 'group' ) . " = " . (int) $groupId;
			$_CB_database->setQuery( $query );
			$users[$groupId]	=	$_CB_database->loadAssocList( 'user_id', 'status' );
		}

		$userId					=	$user->getInt( 'id', 0 );

		return ( isset( $users[$groupId][$userId] ) ? $users[$groupId][$userId] : null );
	}

	/**
	 * @param UserTable   $user
	 * @param GroupTable  $group
	 * @return null|int
	 */
	public static function getGroupInvited( $user, $group )
	{
		global $_CB_database;

		if ( ( ! $user->getInt( 'id', 0 ) ) || ( ! $group->getInt( 'id', 0 ) ) || ( $user->getInt( 'id', 0 ) === $group->getInt( 'user_id', 0 ) ) ) {
			return null;
		}

		if ( isset( $group->_invite_id ) ) {
			return $group->getInt( '_invite_id' );
		}

		static $users					=	array();

		$userId							=	$user->getInt( 'id', 0 );
		$groupId						=	$group->getInt( 'id', 0 );

		if ( ! isset( $users[$userId][$groupId] ) ) {
			$query						=	'SELECT ' . $_CB_database->NameQuote( 'id' )
										.	"\n FROM " . $_CB_database->NameQuote( '#__groupjive_invites' )
										.	"\n WHERE " . $_CB_database->NameQuote( 'group' ) . " = " . $group->getInt( 'id', 0 )
										.	"\n AND " . $_CB_database->NameQuote( 'accepted' ) . ' IS NULL'
										.	"\n AND ( " . $_CB_database->NameQuote( 'email' ) . ' = ' . $_CB_database->Quote( $user->getString( 'email' ) )
										.	' OR ' . $_CB_database->NameQuote( 'user' ) . ' = ' . $user->getInt( 'id', 0 ) . ' )';
			$_CB_database->setQuery( $query, 0, 1 );
			$users[$userId][$groupId]	=	(int) $_CB_database->loadResult();
		}

		return $users[$userId][$groupId];
	}

	/**
	 * @param UserTable   $user
	 * @param GroupTable  $group
	 * @param null|string $param
	 * @param int         $status
	 * @return bool
	 */
	public static function canCreateGroupContent( $user, $group, $param = null, $status = 1 )
	{
		global $_CB_database, $_PLUGINS;

		if ( ( ! $user->getInt( 'id', 0 ) ) || ( ! $group->getInt( 'id', 0 ) ) ) {
			return false;
		}

		if ( self::isModerator( $user->getInt( 'id', 0 ) ) ) {
			return true;
		}

		static $cache								=	array();

		$id											=	md5( $user->getInt( 'id', 0 ) . $group->getInt( 'id', 0 ) . $param . $status );

		if ( ! isset( $cache[$id] ) ) {
			$cache[$id]								=	false;

			if ( ! self::canAccessGroup( $group, $user ) ) {
				$cache[$id]							=	false;

				return false;
			}

			if ( ( $group->getInt( 'published', 0 ) === -1 ) && self::getGlobalParams()->getInt( 'groups_create_approval', 0 ) ) {
				$cache[$id]							=	false;

				return false;
			}

			if ( $user->getInt( 'id', 0 ) === $group->getInt( 'user_id', 0 ) ) {
				$cache[$id]							=	true;

				return true;
			}

			if ( $group->getInt( 'type', 0 ) === 4 ) {
				if ( $param && ( ( ! $group->params()->getInt( $param, 1 ) ) || ( ! in_array( $param, array( 'wall', 'forums' ), true ) ) ) ) {
					$cache[$id]						=	false;

					return false;
				}
			} else {
				$userStatus							=	self::getGroupStatus( $user, $group );

				if ( $userStatus < $status ) {
					$cache[$id]						=	false;

					return false;
				}

				if ( ( $userStatus < 3 ) && $param && ( ! $group->params()->getInt( $param, 1 ) )  ) {
					$cache[$id]						=	false;

					return false;
				}

				if ( $param === 'invites' ) {
					$createLimit					=	self::getCreateLimit( $user, self::getGlobalParams()->getString( 'groups_invites_create_limit', 'custom' ), self::getGlobalParams()->getInt( 'groups_invites_create_limit_custom', 0 ) );

					if ( $createLimit ) {
						static $count				=	array();

						$countId					=	md5( $user->getInt( 'id', 0 ) . $group->getInt( 'id', 0 ) );

						if ( ! isset( $count[$countId] ) ) {
							$query					=	'SELECT COUNT(*)'
													.	"\n FROM " . $_CB_database->NameQuote( '#__groupjive_invites' )
													.	"\n WHERE " . $_CB_database->NameQuote( 'user_id' ) . " = " . $user->getInt( 'id', 0 )
													.	"\n AND " . $_CB_database->NameQuote( 'group' ) . " = " . $group->getInt( 'id', 0 )
													.	"\n AND " . $_CB_database->NameQuote( 'accepted' ) . " IS NULL";
							$_CB_database->setQuery( $query );
							$count[$countId]		=	(int) $_CB_database->loadResult();
						}

						if ( $count[$countId] >= $createLimit ) {
							$cache[$id]				=	false;

							return false;
						}
					}
				}
			}

			$access									=	true;

			$_PLUGINS->trigger( 'gj_onCanCreateGroupContent', array( &$access, $param, $group, $user ) );

			$cache[$id]								=	$access;
		}

		return $cache[$id];
	}

	/**
	 * @param UserTable $user
	 * @param string    $createLimit
	 * @param int       $default
	 * @return int
	 */
	public static function getCreateLimit( $user, $createLimit, $default = 0 )
	{
		if ( is_numeric( $createLimit ) ) {
			// covers B/C case where this used to be an integer:
			return (int) $createLimit;
		}

		if ( $createLimit !== 'custom' ) {
			$limitField			=	\CBuser::getInstance( $user->getInt( 'id', 0 ), false )->getField( $createLimit, null, 'php', 'none', 'profile', 0, true );

			if ( is_array( $limitField ) ) {
				$limitField		=	array_shift( $limitField );

				if ( is_array( $limitField ) ) {
					$limitField	=	implode( '|*|', $limitField );
				}
			} else {
				$limitField		=	$user->getInt( $createLimit, 0 );
			}

			return (int) $limitField;
		}

		return $default;
	}

	/**
	 * @param GroupTable  $group
	 * @param UserTable   $user
	 * @return bool
	 */
	public static function canAccessGroup( $group, $user )
	{
		if ( ! $group->getInt( 'id', 0 ) ) {
			return false;
		}

		if ( self::isModerator( $user->getInt( 'id', 0 ) ) ) {
			return true;
		}

		static $cache					=	array();

		$groupId						=	$group->getInt( 'id', 0 );
		$userId							=	$user->getInt( 'id', 0 );

		if ( ! isset( $cache[$userId][$groupId] ) ) {
			$access						=	true;

			if ( ( ! $group->category()->getInt( 'id', 0 ) ) && ( ! self::getGlobalParams()->getInt( 'groups_uncategorized', 1 ) ) ) {
				$access					=	false;
			} elseif ( $group->category()->getInt( 'id', 0 ) && ( ( ! $group->category()->getInt( 'published', 0 ) ) || ( ! self::canAccess( $group->category()->getInt( 'access', 0 ), $user->getInt( 'id', 0 ) ) ) ) ) {
				$access					=	false;
			} elseif ( $user->getInt( 'id', 0 ) !== $group->getInt( 'user_id', 0 ) ) {
				if ( $group->getInt( 'published', 0 ) !== 1 ) {
					$access				=	false;
				} elseif ( $group->getInt( 'type', 0 ) !== -4 ) {
					$userStatus			=	self::getGroupStatus( $user, $group );

					if ( $userStatus === -1 ) {
						$access			=	false;
					} elseif ( ( $group->getInt( 'type', 0 ) === 3 ) && ( ( $userStatus === false ) || ( $userStatus === null ) ) && ( ! self::getGroupInvited( $user, $group ) ) ) {
						$access			=	false;
					}
				}
			}

			$cache[$userId][$groupId]	=	$access;
		}

		return $cache[$userId][$groupId];
	}

	/**
	 * @param null|int $userId
	 * @return array
	 */
	public static function getAccess( $userId = null )
	{
		static $cache			=	array();

		if ( $userId === null ) {
			$userId				=	Application::MyUser()->getUserId();
		}

		if ( ! isset( $cache[$userId] ) ) {
			$cache[$userId]		=	Application::User( (int) $userId )->getAuthorisedViewLevels();
		}

		return $cache[$userId];
	}

	/**
	 * @param int      $viewAccessLevel
	 * @param null|int $userId
	 * @return bool
	 */
	public static function canAccess( $viewAccessLevel, $userId = null )
	{
		static $cache							=	array();

		if ( $userId === null ) {
			$userId								=	Application::MyUser()->getUserId();
		}

		if ( ! isset( $cache[$userId][$viewAccessLevel] ) ) {
			$cache[$userId][$viewAccessLevel]	=	Application::User( (int) $userId )->canViewAccessLevel( (int) $viewAccessLevel );
		}

		return $cache[$userId][$viewAccessLevel];
	}

	/**
	 * @param null|int $userId
	 * @return bool
	 */
	public static function isModerator( $userId = null )
	{
		static $cache			=	array();

		if ( $userId === null ) {
			$userId				=	Application::MyUser()->getUserId();
		}

		if ( ! isset( $cache[$userId] ) ) {
			$cache[$userId]		=	Application::User( (int) $userId )->isGlobalModerator();
		}

		return $cache[$userId];
	}

	/**
	 * Returns file size formatted from bytes
	 *
	 * @param int $bytes
	 * @param int $decimals
	 * @return string
	 */
	public static function getFormattedFileSize( $bytes, $decimals = 0 )
	{
		if ( $bytes >= 1099511627776 ) {
			return CBTxt::T( 'FILESIZE_FORMATTED_TB', '%%COUNT%% TB|%%COUNT%% TBs', array( '%%COUNT%%' => (float) number_format( $bytes / 1099511627776, $decimals, '.', '' ) ) );
		}

		if ( $bytes >= 1073741824 ) {
			return CBTxt::T( 'FILESIZE_FORMATTED_GB', '%%COUNT%% GB|%%COUNT%% GBs', array( '%%COUNT%%' => (float) number_format( $bytes / 1073741824, $decimals, '.', '' ) ) );
		}

		if ( $bytes >= 1048576 ) {
			return CBTxt::T( 'FILESIZE_FORMATTED_MB', '%%COUNT%% MB|%%COUNT%% MBs', array( '%%COUNT%%' => (float) number_format( $bytes / 1048576, $decimals, '.', '' ) ) );
		}

		if ( $bytes >= 1024 ) {
			return CBTxt::T( 'FILESIZE_FORMATTED_KB', '%%COUNT%% KB|%%COUNT%% KBs', array( '%%COUNT%%' => (float) number_format( $bytes / 1024, $decimals, '.', '' ) ) );
		}

		return CBTxt::T( 'FILESIZE_FORMATTED_B', '%%COUNT%% B|%%COUNT%% Bs', array( '%%COUNT%%' => (float) number_format( $bytes, $decimals, '.', '' ) ) );
	}

	/**
	 * Prefetches users
	 *
	 * @param TableInterface[]|int[] $rows
	 */
	public static function prefetchUsers( $rows )
	{
		if ( ! $rows ) {
			return;
		}

		$users					=	array();

		/** @var TableInterface[] $rows */
		foreach ( $rows as $row ) {
			if ( is_int( $row ) ) {
				if ( $row && ( ! in_array( $row, $users, true ) ) ) {
					$users[]	=	$row;
				}
			} elseif ( $row instanceof UserTable ) {
				$userId			=	$row->getInt( 'id', 0 );

				if ( $userId && ( ! in_array( $userId, $users, true ) ) ) {
					$users[]	=	$userId;
				}
			} else {
				$userId			=	$row->getInt( 'user_id', 0 );

				if ( $userId && ( ! in_array( $userId, $users, true ) ) ) {
					$users[]	=	$userId;
				}

				$userId			=	$row->getInt( 'user', 0 );

				if ( $userId && ( ! in_array( $userId, $users, true ) ) ) {
					$users[]	=	$userId;
				}
			}
		}

		if ( $users ) {
			\CBuser::advanceNoticeOfUsersNeeded( $users );
		}
	}

	/**
	 * Checks if there's too many categories to be using getCategoryOptions
	 * Note this is for XML purposes only
	 *
	 * @return bool
	 */
	public static function checkTooManyCategories()
	{
		global $_CB_database;

		static $cache	=	null;

		if ( $cache === null ) {
			$query		=	'SELECT ' . $_CB_database->NameQuote( 'id' )
						.	"\n FROM " . $_CB_database->NameQuote( '#__groupjive_categories' );
			$_CB_database->setQuery( $query, 0, 101 );
			$cache		=	count( $_CB_database->loadResultArray() );
		}

		return ( $cache > 100 );
	}

	/**
	 * Returns an XML options array of categories
	 *
	 * @return array|\stdClass[]
	 */
	public static function getCategoryXMLOptions()
	{
		return self::getCategoryOptions();
	}

	/**
	 * Returns an options array of available categories
	 *
	 * @param null|UserTable $user
	 * @param bool           $raw
	 * @return array|\stdClass[]
	 */
	public static function getCategoryOptions( $user = null, $raw = false )
	{
		global $_CB_database;

		static $cache			=	array();

		if ( $user ) {
			$userId				=	$user->getInt( 'id', 0 );
		} else {
			$userId				=	Application::MyUser()->getUserId();
		}

		if ( ! isset( $cache[$userId] ) ) {
			$query				=	'SELECT ' . $_CB_database->NameQuote( 'id' ) . ' AS value'
								.	', ' . $_CB_database->NameQuote( 'name' ) . ' AS text'
								.	"\n FROM " . $_CB_database->NameQuote( '#__groupjive_categories' );

			if ( ( ! self::isModerator( $userId ) ) && ( ! Application::Application()->isClient( 'administrator' ) ) ) {
				$query			.=	"\n WHERE " . $_CB_database->NameQuote( 'published' ) . " = 1"
								.	"\n AND " . $_CB_database->NameQuote( 'access' ) . " IN " . $_CB_database->safeArrayOfIntegers( self::getAccess( $userId ) )
								.	"\n AND ( " . $_CB_database->NameQuote( 'create_access' ) . " = 0"
								.		' OR ' . $_CB_database->NameQuote( 'create_access' ) . ' IN ' . $_CB_database->safeArrayOfIntegers( self::getAccess( $userId ) ) . ' )';
			}

			$query				.=	"\n ORDER BY " . $_CB_database->NameQuote( 'ordering' );
			$_CB_database->setQuery( $query );
			$cache[$userId]		=	$_CB_database->loadObjectList();
		}

		if ( $raw === true ) {
			return $cache[$userId];
		}

		$options				=	array();

		if ( Application::Application()->isClient( 'administrator' ) ) {
			$options[]			=	\moscomprofilerHTML::makeOption( 0, CBTxt::T( 'Uncategorized' ) );
		}

		foreach ( $cache[$userId] as $category ) {
			$options[]			=	\moscomprofilerHTML::makeOption( (int) $category->value, CBTxt::T( $category->text ) );
		}

		return $options;
	}

	/**
	 * Checks if there's too many groups to be using getGroupOptions
	 * Note this is for XML purposes only
	 *
	 * @return bool
	 */
	public static function checkTooManyGroups()
	{
		global $_CB_database;

		static $cache	=	null;

		if ( $cache === null ) {
			$query		=	'SELECT ' . $_CB_database->NameQuote( 'id' )
						.	"\n FROM " . $_CB_database->NameQuote( '#__groupjive_groups' );
			$_CB_database->setQuery( $query, 0, 101 );
			$cache		=	count( $_CB_database->loadResultArray() );
		}

		return ( $cache > 100 );
	}

	/**
	 * Returns an XML options array of groups
	 *
	 * @return array|\stdClass[]
	 */
	public static function getGroupXMLOptions()
	{
		return self::getGroupOptions();
	}

	/**
	 * Returns an options array of available groups
	 *
	 * @param null|UserTable    $user
	 * @param bool              $raw
	 * @param array             $excludeCategories
	 * @param array             $excludeGroups
	 * @param bool              $excludeJoined
	 * @param int               $offset
	 * @param int               $limit
	 * @param string|array|null $search
	 * @return array|\stdClass[]
	 */
	public static function getGroupOptions( $user = null, $raw = false, $excludeCategories = array(), $excludeGroups = array(), $excludeJoined = false, $offset = 0, $limit = 0, $search = null )
	{
		global $_CB_database;

		$excludeCategories			=	array_filter( $excludeCategories, function( $value ) {
											return ( ( $value !== null ) && ( $value !== false ) && ( $value !== '' ) );
										});
		$excludeGroups				=	array_filter( $excludeGroups );

		static $cache				=	array();

		if ( $user ) {
			$userId					=	$user->getInt( 'id', 0 );
		} else {
			$userId					=	Application::MyUser()->getUserId();
		}

		$ids						=	array();

		if ( is_array( $search ) ) {
			$ids					=	$search;
			$search					=	implode( '|*|', $search );
		}

		if ( ! isset( $cache[$userId][$offset][$limit][$search] ) ) {
			$query					=	'SELECT g.' . $_CB_database->NameQuote( 'id' ) . ' AS value'
									.	', g.' . $_CB_database->NameQuote( 'name' ) . ' AS text'
									.	', g.' . $_CB_database->NameQuote( 'category' )
									.	', c.' . $_CB_database->NameQuote( 'name' ) . ' AS category_name'
									.	"\n FROM " . $_CB_database->NameQuote( '#__groupjive_groups' ) . " AS g"
									.	"\n LEFT JOIN " . $_CB_database->NameQuote( '#__groupjive_categories' ) . " AS c"
									.	' ON c.' . $_CB_database->NameQuote( 'id' ) . ' = g.' . $_CB_database->NameQuote( 'category' );

			if ( ( ! self::isModerator( $userId ) ) && ( ! Application::Application()->isClient( 'administrator' ) ) ) {
				if ( $userId ) {
					$query			.=	"\n LEFT JOIN " . $_CB_database->NameQuote( '#__groupjive_users' ) . " AS u"
									.	' ON u.' . $_CB_database->NameQuote( 'group' ) . ' = g.' . $_CB_database->NameQuote( 'id' )
									.	' AND u.' . $_CB_database->NameQuote( 'user_id' ) . ' = ' . $userId
									.	( ! $excludeJoined ? ' AND u.' . $_CB_database->NameQuote( 'status' ) . ' >= 1' : null )
									.	"\n WHERE ( ( c." . $_CB_database->NameQuote( 'published' ) . " = 1"
									.		' AND c.' . $_CB_database->NameQuote( 'access' ) . ' IN ' . $_CB_database->safeArrayOfIntegers( self::getAccess( $userId ) ) . ' )'
									.		( self::getGlobalParams()->getBool( 'groups_uncategorized', true ) ? ' OR g.' . $_CB_database->NameQuote( 'category' ) . ' = 0 )' : ' )' );

					if ( $excludeJoined ) {
						$query		.=	"\n AND g." . $_CB_database->NameQuote( 'user_id' ) . " != " . $userId
									.	"\n AND g." . $_CB_database->NameQuote( 'published' ) . " = 1"
									.	"\n AND g." . $_CB_database->NameQuote( 'type' ) . " != 3"
									.	"\n AND u." . $_CB_database->NameQuote( 'id' ) . " IS NULL";
					} else {
						$query		.=	"\n AND ( g." . $_CB_database->NameQuote( 'user_id' ) . " = " . $userId
									.		' OR ( ( g.' . $_CB_database->NameQuote( 'published' ) . ' = 1 )'
									.		' AND ( ( g.' . $_CB_database->NameQuote( 'type' ) . ' != 3 )'
									.		' OR ( u.' . $_CB_database->NameQuote( 'id' ) . ' IS NOT NULL ) ) ) )';
					}
				} else {
					$query			.=	"\n WHERE ( ( c." . $_CB_database->NameQuote( 'published' ) . " = 1"
									.		' AND c.' . $_CB_database->NameQuote( 'access' ) . ' IN ' . $_CB_database->safeArrayOfIntegers( self::getAccess( $userId ) ) . ' )'
									.		( self::getGlobalParams()->getBool( 'groups_uncategorized', true ) ? ' OR g.' . $_CB_database->NameQuote( 'category' ) . ' = 0 )' : ' )' )
									.	"\n AND g." . $_CB_database->NameQuote( 'published' ) . " = 1"
									.	"\n AND g." . $_CB_database->NameQuote( 'type' ) . " != 3";
				}

				$query				.=	( $excludeCategories ? "\n AND c." . $_CB_database->NameQuote( 'id' ) . " NOT IN " . $_CB_database->safeArrayOfIntegers( $excludeCategories ) : null )
									.	( $excludeGroups ? "\n AND g." . $_CB_database->NameQuote( 'id' ) . " NOT IN " . $_CB_database->safeArrayOfIntegers( $excludeGroups ) : null );
			} elseif ( $excludeJoined && $userId ) {
				$query				.=	"\n LEFT JOIN " . $_CB_database->NameQuote( '#__groupjive_users' ) . " AS u"
									.	' ON u.' . $_CB_database->NameQuote( 'group' ) . ' = g.' . $_CB_database->NameQuote( 'id' )
									.	' AND u.' . $_CB_database->NameQuote( 'user_id' ) . ' = ' . $userId
									.	"\n WHERE g." . $_CB_database->NameQuote( 'user_id' ) . " != " . $userId
									.	"\n AND u." . $_CB_database->NameQuote( 'id' ) . " IS NULL"
									.	( $excludeCategories ? "\n AND c." . $_CB_database->NameQuote( 'id' ) . " NOT IN " . $_CB_database->safeArrayOfIntegers( $excludeCategories ) : null )
									.	( $excludeGroups ? "\n AND g." . $_CB_database->NameQuote( 'id' ) . " NOT IN " . $_CB_database->safeArrayOfIntegers( $excludeGroups ) : null );
			} else {
				$query				.=	( $excludeCategories ? "\n WHERE c." . $_CB_database->NameQuote( 'id' ) . " NOT IN " . $_CB_database->safeArrayOfIntegers( $excludeCategories ) : null )
									.	( $excludeGroups ? "\n " . ( $excludeCategories ? 'AND' : 'WHERE' ) . " g." . $_CB_database->NameQuote( 'id' ) . " NOT IN " . $_CB_database->safeArrayOfIntegers( $excludeGroups ) : null );
			}

			if ( $ids ) {
				$query				.=	"\n AND g." . $_CB_database->NameQuote( 'id' ) . " IN " . $_CB_database->safeArrayOfIntegers( $ids );
			} else {
				if ( $search ) {
					$query			.=	"\n AND g." . $_CB_database->NameQuote( 'name' ) . " LIKE " . $_CB_database->Quote( '%' . $_CB_database->getEscaped( $search, true ) . '%', false );
				}

				$query				.=	"\n ORDER BY g." . $_CB_database->NameQuote( 'category' ) . " ASC, g." . $_CB_database->NameQuote( 'date' ) . " DESC";
			}

			$_CB_database->setQuery( $query, $offset, $limit );
			$cache[$userId][$offset][$limit][$search]	=	$_CB_database->loadObjectList();
		}

		if ( $raw === true ) {
			return $cache[$userId][$offset][$limit][$search];
		}

		$optGroups					=	array();
		$options					=	array();

		foreach ( $cache[$userId][$offset][$limit][$search] as $group ) {
			$category				=	(int) $group->category;

			if ( ! in_array( $category, $optGroups, true ) ) {
				$options[]			=	\moscomprofilerHTML::makeOptGroup( ( $category ? CBTxt::T( $group->category_name ) : CBTxt::T( 'Uncategorized' ) ) );

				$optGroups[]		=	$category;
			}

			$options[]				=	\moscomprofilerHTML::makeOption( (int) $group->value, CBTxt::T( $group->text ) );
		}

		return $options;
	}

	/**
	 * @param string $default
	 * @param bool   $thumbnail
	 * @return string|null
	 */
	public static function getDefaultImage( $default, $thumbnail = false )
	{
		global $_CB_framework;

		static $cache	=	array();

		if ( ! isset( $cache[$default][$thumbnail] ) ) {
			if ( ( $default === 'none' ) || ( $default === 'color' ) ) {
				$image		=	$default;
			} else {
				$path		=	'/components/com_comprofiler/plugin/user/plug_cbgroupjive/templates/' . self::getGlobalParams()->getString( 'general_template', 'default' ) . '/images/' . ( $thumbnail ? 'tn' : null ) . $default;

				if ( ! file_exists( $_CB_framework->getCfg( 'absolute_path' ) . $path ) ) {
					$path	=	'/components/com_comprofiler/plugin/user/plug_cbgroupjive/templates/default/images/' . ( $thumbnail ? 'tn' : null ) . $default;
				}

				$image		=	$_CB_framework->getCfg( 'live_site' ) . $path;
			}

			$cache[$default][$thumbnail]	=	$image;
		}

		return $cache[$default][$thumbnail];
	}
}