<?php

use Joomla\CMS\Language\Text;
use OSSolution\HelpdeskPro\Site\Helper\Html as HelpdeskProHelperHtml;

/**
 * Abstract Form Field class for the HDP framework
 *
 * @package     Joomla.HDP
 * @subpackage  Form
 */
abstract class HDPFormField
{

	/**
	 * The form field type.
	 *
	 * @var    string
	 */
	protected $type;

	/**
	 * The name (and id) for the form field.
	 *
	 * @var    string
	 */
	protected $name;

	/**
	 * Title of the form field
	 *
	 * @var string
	 */
	protected $title;

	/**
	 * Description of the form field
	 * @var string
	 */
	protected $description;

	/**
	 * Default value for the field
	 *
	 * @var string
	 */
	protected $defaultValues;
	/**
	 * The current value of the form field.
	 *
	 * @var    mixed
	 */
	protected $value;
	/**
	 * The form field is required or not
	 *
	 * @var int
	 */
	protected $required;
	/**
	 * Any other extra attributes of the custom fields
	 *
	 * @var string
	 */
	protected $extraAttributes;

	/**
	 * This field is visible or hidden on the form
	 *
	 * @var bool
	 */
	protected $visible = true;

	/**
	 * ID of the custom field
	 *
	 * @var int
	 */
	protected $id = 0;

	/**
	 * ID of the category
	 *
	 * @var int
	 */
	protected $categoryId = 0;

	/**
	 * The html attributes of the field
	 *
	 * @var array
	 */
	protected $attributes = [];

	/**
	 * The input for the form field.
	 *
	 * @var    string
	 */
	protected $input;

	/**
	 * Method to instantiate the form field object.
	 *
	 * @param   JTable  $row    the table object store form field definitions
	 * @param   mixed   $value  the initial value of the form field
	 *
	 */
	public function __construct($row, $value = null)
	{
		$this->id              = $row->id;
		$this->name            = $row->name;
		$this->title           = Text::_($row->title);
		$this->description     = $row->description;
		$this->required        = $row->required;
		$this->extraAttributes = $row->extra_attributes;
		$this->value           = $value;
		$this->default_values  = $row->default_values;
		$this->categoryId      = $row->category_id;
		$this->row             = $row;
		$cssClasses            = [];

		if ($row->css_class)
		{
			$cssClasses[] = $row->css_class;
		}

		if ($row->validation_rules)
		{
			$cssClasses[] = $row->validation_rules;
		}

		if (count($cssClasses))
		{
			$this->attributes['class'] = implode(' ', $cssClasses);
		}
	}

	/**
	 * Method to get certain otherwise inaccessible properties from the form field object.
	 *
	 * @param   string  $name  The property name for which to the the value.
	 *
	 * @return  mixed  The property value or null.
	 *
	 */
	public function __get($name)
	{
		switch ($name)
		{
			case 'id':
			case 'type':
			case 'name':
			case 'title':
			case 'description':
			case 'value':
			case 'row':
			case 'extraAttributes':
			case 'required':
			case 'categoryId':
			case 'default_values':
				return $this->{$name};
				break;
			case 'input':
				// If the input hasn't yet been generated, generate it.
				if (empty($this->input))
				{
					$this->input = $this->getInput();
				}

				return $this->input;
				break;
		}

		return null;
	}

	/**
	 * Simple method to set the value for the form field
	 *
	 * @param   mixed  $value  Value to set
	 *
	 */
	public function setValue($value)
	{
		$this->value = $value;
	}

	/**
	 * Method to get the field input markup.
	 *
	 * @param   HelpdeskProHelperBootstrap  $bootstrapHelper
	 *
	 * @return  string  The field input markup.
	 *
	 */
	abstract protected function getInput($bootstrapHelper = null);

	/**
	 * Method to get the field label markup.
	 *
	 * @return  string  The field label markup.
	 */
	protected function getLabel()
	{
		$data = [
			'name'        => $this->name,
			'title'       => $this->title,
			'description' => $this->description,
			'row'         => $this->row,
		];

		return HelpdeskProHelperHtml::loadCommonLayout('fieldlayout/label.php', $data);
	}

	/**
	 * Method to get a control group with label and input.
	 *
	 * @param   bool                        $tableLess
	 * @param   HelpdeskProHelperBootstrap  $bootstrapHelper
	 *
	 * @return  string  A string containing the html for the control goup
	 */
	public function getControlGroup($tableLess = true, $bootstrapHelper = null)
	{
		if ($this->type == 'hidden')
		{
			return $this->getInput();
		}

		$controlGroupAttributes = 'id="field_' . $this->id . '"';

		if (!$this->visible)
		{
			$controlGroupAttributes .= ' style="display:none;" ';
		}

		$data = [
			'name'                   => $this->name,
			'controlGroupAttributes' => $controlGroupAttributes,
			'label'                  => $this->getLabel(),
			'input'                  => $this->getInput($bootstrapHelper),
			'bootstrapHelper'        => $bootstrapHelper,
			'row'                    => $this->row,
			'tableLess'              => $tableLess,
		];

		return HelpdeskProHelperHtml::loadCommonLayout('fieldlayout/controlgroup.php', $data);
	}

	/**
	 * Get output of the field using for sending email and display on the registration complete page
	 *
	 * @param   bool                        $tableLess
	 * @param   HelpdeskProHelperBootstrap  $bootstrapHelper
	 *
	 * @return string
	 */
	public function getOutput($tableLess = true, $bootstrapHelper = null)
	{
		if (!$this->visible)
		{
			return;
		}

		if (is_string($this->value) && is_array(json_decode($this->value)))
		{
			$fieldValue = implode(', ', json_decode($this->value));
		}
		else
		{
			$fieldValue = $this->value;
		}

		if ($tableLess)
		{
			$controlGroupClass = $bootstrapHelper ? $bootstrapHelper->getClassMapping('control-group') : 'control-group';
			$controlLabelClass = $bootstrapHelper ? $bootstrapHelper->getClassMapping('control-label') : 'control-label';
			$controlsClass     = $bootstrapHelper ? $bootstrapHelper->getClassMapping('controls') : 'controls';

			return '<div class="' . $controlGroupClass . '">' . '<div class="' . $controlLabelClass . '">' . $this->title . '</div>' . '<div class="' . $controlsClass . '">' . $fieldValue .
				'</div>' . '</div>';
		}
		else
		{
			return '<tr>' . '<td class="title_cell">' . $this->title . '</td>' . '<td class="field_cell">' . $fieldValue . '</td>' . '</tr>';
		}
	}

	/**
	 * Add attribute to the form field
	 *
	 * @param   string  $name
	 */
	public function setAttribute($name, $value)
	{
		$this->attributes[$name] = $value;
	}

	/**
	 * Get data of the given attribute
	 *
	 * @param   string  $name
	 *
	 * @return string
	 */
	public function getAttribute($name)
	{
		return $this->attributes[$name];
	}

	/**
	 * Set visibility status for the field on form
	 *
	 * @param $visible
	 */
	public function setVisibility($visible)
	{
		$this->visible = $visible;
	}

	/**
	 * Method to add new class to the field
	 *
	 * @param   string  $class
	 *
	 * @return void
	 */
	public function addClass($class)
	{
		$classes = $this->getAttribute('class');

		$this->setAttribute('class', $classes ? $classes . ' ' . $class : $class);
	}

	/**
	 * Build an HTML attribute string from an array.
	 *
	 * @param   array  $attributes
	 *
	 * @return string
	 */
	public function buildAttributes()
	{
		$html = [];

		foreach ((array) $this->attributes as $key => $value)
		{
			if (is_bool($value))
			{
				$html[] = " $key ";
			}
			else
			{

				$html[] = $key . '="' . htmlentities($value, ENT_QUOTES, 'UTF-8', false) . '"';
			}
		}

		if ($this->extraAttributes)
		{
			$html[] = $this->extraAttributes;
		}

		return count($html) > 0 ? ' ' . implode(' ', $html) : '';
	}
}
