<?php
/**
* Community Builder (TM)
* @version $Id: $
* @package CommunityBuilder
* @copyright (C) 2004-2021 www.joomlapolis.com / Lightning MultiCom SA - and its licensors, all rights reserved
* @license http://www.gnu.org/licenses/old-licenses/gpl-2.0.html GNU/GPL version 2
*/

namespace CB\Plugin\Query;

use CBLib\Registry\Registry;
use CBLib\Database\Driver\CmsDatabaseDriver;
use CB\Database\Table\FieldTable;
use CBLib\Database\DatabaseDriverInterface;
use CBLib\Language\CBTxt;

\defined( 'CBLIB') or die();

class CBQueryField
{

	/**
	 * @return Registry
	 */
	public static function getGlobalParams(): Registry
	{
		global $_PLUGINS;

		static $params	=	null;

		if ( ! $params ) {
			$plugin		=	$_PLUGINS->getLoadedPlugin( 'user', 'cbqueryfield' );
			$params		=	new Registry();

			if ( $plugin ) {
				$params->load( $plugin->params );
			}
		}

		return $params;
	}

	/**
	 * @param FieldTable  $field
	 * @param null|string $file
	 * @param bool|array  $headers
	 * @return null|string
	 */
	public static function getTemplate( FieldTable $field, ?string $file = null, $headers = [ 'template', 'override' ] ): ?string
	{
		global $_CB_framework, $_PLUGINS;

		$plugin							=	$_PLUGINS->getLoadedPlugin( 'user', 'cbqueryfield' );

		if ( ! $plugin ) {
			return null;
		}

		static $defaultTemplate			=	null;

		if ( $defaultTemplate === null ) {
			$defaultTemplate			=	self::getGlobalParams()->getString( 'general_template', 'default' );
		}

		$template						=	$field->params->getString( 'query_template', '' );

		if ( ( $template === '' ) || ( $template === null ) || ( $template === '-1' ) ) {
			$template					=	$defaultTemplate;
		}

		if ( ! $template ) {
			$template					=	'default';
		}

		$livePath						=	$_PLUGINS->getPluginLivePath( $plugin );
		$absPath						=	$_PLUGINS->getPluginPath( $plugin );
		$return							=	null;

		if ( $file ) {
			$file						=	preg_replace( '/[^-a-zA-Z0-9_]/', '', $file );

			if ( $headers !== false ) {
				$headers[]				=	$file;
			}

			$php						=	$absPath . '/templates/' . $template . '/' . $file . '.php';

			if ( ! file_exists( $php ) ) {
				$php					=	$absPath . '/templates/default/' . $file . '.php';
			}

			if ( file_exists( $php ) ) {
				$return					=	$php;
			}
		}

		if ( $headers !== false ) {
			static $loaded				=	[];

			$loaded[$template]			=	[];

			// Global CSS File:
			if ( \in_array( 'template', $headers, true ) && ( ! \in_array( 'template', $loaded[$template], true ) ) ) {
				$global					=	'/templates/' . $template . '/template.css';

				if ( ! file_exists( $absPath . $global ) ) {
					$global				=	'/templates/default/template.css';
				}

				if ( file_exists( $absPath . $global ) ) {
					$_CB_framework->document->addHeadStyleSheet( $livePath . $global );
				}

				$loaded[$template][]	=	'template';
			}

			// File or Custom CSS/JS Headers:
			foreach ( $headers as $header ) {
				if ( \in_array( $header, $loaded[$template], true ) || \in_array( $header, [ 'template', 'override' ], true ) ) {
					continue;
				}

				$header					=	preg_replace( '/[^-a-zA-Z0-9_]/', '', $header );

				if ( ! $header ) {
					continue;
				}

				$css					=	'/templates/' . $template . '/' . $header . '.css';
				$js						=	'/templates/' . $template . '/' . $header . '.js';

				if ( ! file_exists( $absPath . $css ) ) {
					$css				=	'/templates/default/' . $header . '.css';
				}

				if ( file_exists( $absPath . $css ) ) {
					$_CB_framework->document->addHeadStyleSheet( $livePath . $css );
				}

				if ( ! file_exists( $absPath . $js ) ) {
					$js					=	'/templates/default/' . $header . '.js';
				}

				if ( file_exists( $absPath . $js ) ) {
					$_CB_framework->document->addHeadScriptUrl( $livePath . $js );
				}

				$loaded[$template][]	=	$header;
			}

			// Override CSS File:
			if ( \in_array( 'override', $headers, true ) && ( ! \in_array( 'override', $loaded[$template], true ) ) ) {
				$override				=	'/templates/' . $template . '/override.css';

				if ( file_exists( $absPath . $override ) ) {
					$_CB_framework->document->addHeadStyleSheet( $livePath . $override );
				}

				$loaded[$template][]	=	'override';
			}
		}

		return $return;
	}

	/**
	 * Gets the internal or external database
	 *
	 * @param FieldTable $field
	 * @param string     $paramPrefix
	 * @return \cbDatabase|DatabaseDriverInterface
	 */
	public static function getDatabase( FieldTable $field, string $paramPrefix = 'qry' )
	{
		global $_CB_framework, $_CB_database;

		if ( $field->params->getInt( $paramPrefix . '_mode', 0 ) ) {
			$driver							=	$_CB_framework->getCfg( 'dbtype' );
			$host							=	$field->params->getString( $paramPrefix . '_host', '' );
			$username						=	$field->params->getString( $paramPrefix . '_username', '' );
			$password						=	$field->params->getString( $paramPrefix . '_password', '' );
			$database						=	$field->params->getString( $paramPrefix . '_database', '' );
			$charset						=	$field->params->getString( $paramPrefix . '_charset', '' );
			$prefix							=	$field->params->getString( $paramPrefix . '_prefix', '' );

			$options						=	array ( 'driver' => $driver, 'host' => $host, 'user' => $username, 'password' => $password, 'database' => $database, 'prefix' => $prefix );

			if ( checkJversion( '3.0+' ) ) {
				try {
					$_J_database			=	\JDatabaseDriver::getInstance( $options );
				} catch ( \RuntimeException $e ) {
					return null;
				}
			} else {
				$_J_database				=	\JDatabase::getInstance( $options );

				if ( \JError::isError( $_J_database ) ) {
					return null;
				}
			}

			$_SQL_database					=	new CmsDatabaseDriver( $_J_database, $prefix, checkJversion( 'release' ) );

			if ( $charset ) {
				$_SQL_database->setQuery( 'SET NAMES ' . $_SQL_database->Quote( $charset ) );
				$_SQL_database->query();
			}
		} else {
			$_SQL_database					=	$_CB_database;
		}

		return $_SQL_database;
	}

	/**
	 * @param string $str
	 * @return string
	 */
	public static function escapeSQL( ?string $str ): string
	{
		global $_CB_database;

		return $_CB_database->getEscaped( (string) $str );
	}

	/**
	 * Converts array or object into pathed extras substitutions
	 *
	 * @param string       $prefix
	 * @param array|object $items
	 * @param array        $extras
	 * @param int          $maxDepth
	 * @param int          $depth
	 * @param bool         $translate
	 */
	public static function prepareExtras( string $prefix, $items, array &$extras, int $maxDepth = 2, int $depth = 0, bool $translate = true ): void
	{
		if ( ! $maxDepth ) {
			$maxDepth					=	2;
		}

		if ( $depth > $maxDepth ) {
			return;
		}

		foreach ( $items as $k => $v ) {
			if ( \is_string( $k ) && ( $k[0] === '_' ) ) {
				continue;
			}

			if ( \is_array( $v ) ) {
				$multi					=	false;

				foreach ( $v as $kv => $cv ) {
					if ( is_numeric( $kv ) ) {
						$kv				=	(int) $kv;
					}

					if ( \is_object( $cv ) || \is_array( $cv ) || ( $kv && ( ! \is_int( $kv ) ) ) ) {
						$multi			=	true;
					}
				}

				if ( ! $multi ) {
					$v					=	implode( '|*|', $v );
				}
			}

			if ( $v && ( ! \is_object( $v ) ) && ( ! \is_array( $v ) ) && ( ( $k === 'params' ) || ( \is_string( $v ) && ( ( $v[0] === '{' ) || ( $v[0] === '[' ) ) ) ) ) {
				$params					=	new Registry( $v );

				$v						=	$params->asArray();
			}

			$k							=	'_' . ltrim( str_replace( ' ', '_', strtolower( trim( (string) $k ) ) ), '_' );

			if ( ( ! \is_object( $v ) ) && ( ! \is_array( $v ) ) ) {
				if ( $translate && ( ! is_numeric( $v ) ) && ( ! \is_bool( $v ) ) ) {
					$v					=	CBTxt::T( (string) $v );
				}

				$extras[$prefix . $k]	=	$v;
			} elseif ( $v ) {
				$depth++;

				if ( \is_object( $v ) ) {
					/** @var object $v */
					$subItems			=	get_object_vars( $v );
				} else {
					$subItems			=	$v;
				}

				self::prepareExtras( $prefix . $k, $subItems, $extras, $maxDepth, $depth, $translate );
			}
		}
	}
}