<?php
/**
 * Community Builder (TM)
 * @version $Id: $
 * @package CommunityBuilder
 * @copyright (C) 2004-2021 www.joomlapolis.com / Lightning MultiCom SA - and its licensors, all rights reserved
 * @license http://www.gnu.org/licenses/old-licenses/gpl-2.0.html GNU/GPL version 2
 */

defined( '_JEXEC' ) or die;

use Joomla\CMS\Application\CMSApplication;
use Joomla\CMS\Event\ErrorEvent;
use Joomla\CMS\Plugin\CMSPlugin;
use Joomla\CMS\Router\Route;
use Joomla\CMS\Uri\Uri;
use Joomla\Database\DatabaseInterface;
use Joomla\Event\SubscriberInterface;
use Joomla\String\StringHelper;

if ( JVersion::MAJOR_VERSION < 4 ) {
	JPluginHelper::importPlugin( 'system', 'redirect' );

	class plgSystemcbredirectbot extends JPlugin
	{
		/** @var bool  */
		protected $autoloadLanguage			=	false;
		/** @var array  */
		static $redirectsFromRegexp			=	[];
		/** @var array  */
		static $redirectsToRegexp			=	[];
		/** @var array  */
		static $redirectsFrom				=	[];
		/** @var array  */
		static $redirectsTo					=	[];
		/** @var int  */
		static $redirectHeader				=	301;
		/** @var callable  */
		static $previousExceptionHandler	=	null;

		/**
		 * plgSystemcbredirectbot constructor
		 *
		 * @param object $subject
		 * @param array  $config
		 */
		public function __construct( &$subject, $config )
		{
			parent::__construct( $subject, $config );

			try {
				// This is wrapped in a try catch as we don't want to risk Joomla API changes blocking an install
				JError::setErrorHandling( E_ERROR, 'callback', [ 'plgSystemcbredirectbot', 'handleError' ] );

				self::$previousExceptionHandler		=	set_exception_handler( [ 'plgSystemcbredirectbot', 'handleException' ] );

				$this->getRedirects();
			} catch ( Exception $e ) {}
		}

		/**
		 * Method to handle an error condition from JError
		 *
		 * @param JException $error
		 */
		public static function handleError( JException $error )
		{
			if ( ! self::doErrorHandling( $error ) ) {
				if ( class_exists( 'PlgSystemRedirect' ) ) {
					PlgSystemRedirect::handleError( $error );
				} elseif ( self::$previousExceptionHandler ) {
					call_user_func_array( self::$previousExceptionHandler, array( $error ) );
				} else {
					JErrorPage::render( $error );
				}
			}
		}

		/**
		 * Method to handle an uncaught exception
		 *
		 * @param Exception|Throwable $exception
		 * @throws InvalidArgumentException
		 */
		public static function handleException( $exception )
		{
			if ( ( ! $exception instanceof Throwable ) && ( ! $exception instanceof Exception ) ) {
				throw new InvalidArgumentException( sprintf( 'The error handler requires a Exception or Throwable object, a "%s" object was given instead.', get_class( $exception ) ) );
			}

			if ( ! self::doErrorHandling( $exception ) ) {
				if ( class_exists( 'PlgSystemRedirect' ) ) {
					PlgSystemRedirect::handleException( $exception );
				} elseif ( self::$previousExceptionHandler ) {
					call_user_func( self::$previousExceptionHandler, $exception );
				} else {
					JErrorPage::render( $exception );
				}
			}
		}

		/**
		 * Gets and prepares the redirect URLs
		 */
		private function getRedirects(): void
		{
			if ( ( ! self::$redirectsFromRegexp ) && ( ! self::$redirectsFrom ) ) {
				$redirects									=	$this->params->get( 'redirects' );
				$header										=	(int) $this->params->get( 'header' );

				if ( $header ) {
					self::$redirectHeader					=	$header;
				}

				if ( ! $redirects ) {
					return;
				}

				if ( is_string( $redirects ) ) {
					// Legacy B/C fallback encase migration failed for some reason:
					$redirects								=	json_decode( $redirects, true );

					if ( ( ! $redirects ) || ( ! isset( $redirects['from'] ) ) ) {
						return;
					}

					foreach ( $redirects['from'] as $k => $from ) {
						if ( ! $from ) {
							continue;
						}

						if ( ( $redirects['regexp'][$k] ?? 0 ) ) {
							self::$redirectsFromRegexp[]	=	$from;
							self::$redirectsToRegexp[]		=	( $redirects['to'][$k] ?? '' );
						} else {
							self::$redirectsFrom[]			=	$from;
							self::$redirectsTo[]			=	( $redirects['to'][$k] ?? '' );
						}
					}
				} else {
					foreach ( $redirects as $redirect ) {
						$from								=	( $redirect->from ?? '' );

						if ( ! $from ) {
							continue;
						}

						if ( ( $redirect->regexp ?? 0 ) ) {
							self::$redirectsFromRegexp[]	=	$from;
							self::$redirectsToRegexp[]		=	( $redirect->to ?? '' );
						} else {
							self::$redirectsFrom[]			=	$from;
							self::$redirectsTo[]			=	( $redirect->to ?? '' );
						}
					}
				}
			}
		}

		/**
		 * Internal processor for all error handlers
		 *
		 * @param Exception|Throwable $error
		 * @return bool
		 */
		private static function doErrorHandling( $error )
		{
			if ( ( ! self::$redirectsFromRegexp ) && ( ! self::$redirectsFrom ) ) {
				return false;
			}

			$app					=	JFactory::getApplication();

			if ( $app->isClient( 'administrator' ) || ( (int) $error->getCode() !== 404 ) ) {
				return false;
			}

			$uri					=	JUri::getInstance();

			// These are the original URLs
			$orgurl					=	rawurldecode( $uri->toString( [ 'scheme', 'host', 'port', 'path', 'query', 'fragment' ] ) );
			$orgurlRel				=	rawurldecode( $uri->toString( [ 'path', 'query', 'fragment' ] ) );

			// The above doesn't work for sub directories, so do this
			$orgurlRootRel			=	str_replace( JUri::root(), '', $orgurl );

			// For when users have added / to the url
			$orgurlRootRelSlash		=	str_replace( JUri::root(), '/', $orgurl );
			$orgurlWithoutQuery		=	rawurldecode( $uri->toString( [ 'scheme', 'host', 'port', 'path', 'fragment' ] ) );
			$orgurlRelWithoutQuery	=	rawurldecode( $uri->toString( [ 'path', 'fragment' ] ) );

			// These are the URLs we save and use
			$url					=	StringHelper::strtolower( rawurldecode( $uri->toString( [ 'scheme', 'host', 'port', 'path', 'query', 'fragment' ] ) ) );
			$urlRel					=	StringHelper::strtolower( rawurldecode( $uri->toString( [ 'path', 'query', 'fragment' ] ) ) );

			// The above doesn't work for sub directories, so do this
			$urlRootRel				=	str_replace( JUri::root(), '', $url );

			// For when users have added / to the url
			$urlRootRelSlash		=	str_replace( JUri::root(), '/', $url );
			$urlWithoutQuery		=	StringHelper::strtolower( rawurldecode( $uri->toString( [ 'scheme', 'host', 'port', 'path', 'fragment' ] ) ) );
			$urlRelWithoutQuery		=	StringHelper::strtolower( rawurldecode( $uri->toString( [ 'path', 'fragment' ] ) ) );

			if ( ( strpos( $url, 'mosConfig_' ) !== false ) || ( strpos( $url, '=http://' ) !== false ) ) {
				return false;
			}

			$possibleMatches		=	array_unique(
											[
												$url,
												$urlRel,
												$urlRootRel,
												$urlRootRelSlash,
												$urlWithoutQuery,
												$urlRelWithoutQuery,
												$orgurl,
												$orgurlRel,
												$orgurlRootRel,
												$orgurlRootRelSlash,
												$orgurlWithoutQuery,
												$orgurlRelWithoutQuery,
											]
										);

			$redirect				=	null;

			foreach ( $possibleMatches as $possibleMatch ) {
				$new				=	$possibleMatch;

				if ( self::$redirectsFromRegexp ) {
					$new			=	preg_replace( self::$redirectsFromRegexp, self::$redirectsToRegexp, $new );
				}

				if ( self::$redirectsFrom ) {
					$new			=	str_ireplace( self::$redirectsFrom, self::$redirectsTo, $new );
				}

				if ( ( ! $new ) || ( $new === $possibleMatch ) ) {
					continue;
				}

				$redirect			=	$new;
				break;
			}

			if ( ! $redirect ) {
				return false;
			}

			if ( ( self::$redirectHeader < 400 ) && ( self::$redirectHeader >= 300 ) ) {
				$app->redirect( ( JUri::isInternal( $redirect ) || ( strpos( $redirect, 'http' ) === false ) ? JRoute::_( $redirect ) : $redirect ), self::$redirectHeader );
			}

			JErrorPage::render( new RuntimeException( $error->getMessage(), self::$redirectHeader, $error ) );

			return true;
		}
	}
} else {
	class plgSystemcbredirectbot extends CMSPlugin implements SubscriberInterface
	{
		/** @var bool  */
		protected $autoloadLanguage			=	false;
		/** @var null|DatabaseInterface */
		protected $db						=	null;
		/** @var array  */
		protected $redirectsFromRegexp		=	[];
		/** @var array  */
		protected $redirectsToRegexp		=	[];
		/** @var array  */
		protected $redirectsFrom			=	[];
		/** @var array  */
		protected $redirectsTo				=	[];
		/** @var int  */
		protected $redirectHeader			=	301;

		/**
		 * @return string[]
		 */
		public static function getSubscribedEvents(): array
		{
			return [ 'onError' => 'handleError' ];
		}

		/**
		 * Gets and prepares the redirect URLs
		 */
		private function getRedirects(): void
		{
			if ( ( ! $this->redirectsFromRegexp ) && ( ! $this->redirectsFrom ) ) {
				$redirects									=	$this->params->get( 'redirects' );
				$header										=	(int) $this->params->get( 'header' );

				if ( $header ) {
					$this->redirectHeader					=	$header;
				}

				if ( ! $redirects ) {
					return;
				}

				if ( is_string( $redirects ) ) {
					// Legacy B/C fallback encase migration failed for some reason:
					$redirects								=	json_decode( $redirects, true );

					if ( ( ! $redirects ) || ( ! isset( $redirects['from'] ) ) ) {
						return;
					}

					foreach ( $redirects['from'] as $k => $from ) {
						if ( ! $from ) {
							continue;
						}

						if ( ( $redirects['regexp'][$k] ?? 0 ) ) {
							$this->redirectsFromRegexp[]	=	$from;
							$this->redirectsToRegexp[]		=	( $redirects['to'][$k] ?? '' );
						} else {
							$this->redirectsFrom[]			=	$from;
							$this->redirectsTo[]			=	( $redirects['to'][$k] ?? '' );
						}
					}
				} else {
					foreach ( $redirects as $redirect ) {
						$from								=	( $redirect->from ?? '' );

						if ( ! $from ) {
							continue;
						}

						if ( ( $redirect->regexp ?? 0 ) ) {
							$this->redirectsFromRegexp[]	=	$from;
							$this->redirectsToRegexp[]		=	( $redirect->to ?? '' );
						} else {
							$this->redirectsFrom[]			=	$from;
							$this->redirectsTo[]			=	( $redirect->to ?? '' );
						}
					}
				}
			}
		}

		/**
		 * @param ErrorEvent $event
		 */
		public function handleError( ErrorEvent $event ): void
		{
			$this->getRedirects();

			if ( ( ! $this->redirectsFromRegexp ) && ( ! $this->redirectsFrom ) ) {
				return;
			}

			/** @var CMSApplication $app */
			$app					=	$event->getApplication();

			if ( $app->isClient( 'administrator' ) || ( $event->getError()->getCode() !== 404 ) ) {
				return;
			}

			$uri					=	Uri::getInstance();

			// These are the original URLs
			$orgurl					=	rawurldecode( $uri->toString( [ 'scheme', 'host', 'port', 'path', 'query', 'fragment' ] ) );
			$orgurlRel				=	rawurldecode( $uri->toString( [ 'path', 'query', 'fragment' ] ) );

			// The above doesn't work for sub directories, so do this
			$orgurlRootRel			=	str_replace( Uri::root(), '', $orgurl );

			// For when users have added / to the url
			$orgurlRootRelSlash		=	str_replace( Uri::root(), '/', $orgurl );
			$orgurlWithoutQuery		=	rawurldecode( $uri->toString( [ 'scheme', 'host', 'port', 'path', 'fragment' ] ) );
			$orgurlRelWithoutQuery	=	rawurldecode( $uri->toString( [ 'path', 'fragment' ] ) );

			// These are the URLs we save and use
			$url					=	StringHelper::strtolower( rawurldecode( $uri->toString( [ 'scheme', 'host', 'port', 'path', 'query', 'fragment' ] ) ) );
			$urlRel					=	StringHelper::strtolower( rawurldecode( $uri->toString( [ 'path', 'query', 'fragment' ] ) ) );

			// The above doesn't work for sub directories, so do this
			$urlRootRel				=	str_replace( Uri::root(), '', $url );

			// For when users have added / to the url
			$urlRootRelSlash		=	str_replace( Uri::root(), '/', $url );
			$urlWithoutQuery		=	StringHelper::strtolower( rawurldecode( $uri->toString( [ 'scheme', 'host', 'port', 'path', 'fragment' ] ) ) );
			$urlRelWithoutQuery		=	StringHelper::strtolower( rawurldecode( $uri->toString( [ 'path', 'fragment' ] ) ) );

			if ( ( strpos( $url, 'mosConfig_' ) !== false ) || ( strpos( $url, '=http' ) !== false ) ) {
				return;
			}

			$possibleMatches		=	array_unique(
											[
												$url,
												$urlRel,
												$urlRootRel,
												$urlRootRelSlash,
												$urlWithoutQuery,
												$urlRelWithoutQuery,
												$orgurl,
												$orgurlRel,
												$orgurlRootRel,
												$orgurlRootRelSlash,
												$orgurlWithoutQuery,
												$orgurlRelWithoutQuery,
											]
										);

			$redirect				=	null;

			foreach ( $possibleMatches as $possibleMatch ) {
				$new				=	$possibleMatch;

				if ( $this->redirectsFromRegexp ) {
					$new			=	preg_replace( $this->redirectsFromRegexp, $this->redirectsToRegexp, $new );
				}

				if ( $this->redirectsFrom ) {
					$new			=	str_ireplace( $this->redirectsFrom, $this->redirectsTo, $new );
				}

				if ( ( ! $new ) || ( $new === $possibleMatch ) ) {
					continue;
				}

				$redirect			=	$new;
				break;
			}

			if ( ! $redirect ) {
				return;
			}

			if ( ( $this->redirectHeader < 400 ) && ( $this->redirectHeader >= 300 ) ) {
				$app->redirect( ( Uri::isInternal( $redirect ) || ( strpos( $redirect, 'http' ) === false ) ? Route::_( $redirect ) : $redirect ), $this->redirectHeader );
			}

			$event->setError( new RuntimeException( $event->getError()->getMessage(), $this->redirectHeader, $event->getError() ) );
		}
	}
}