<?php
/**
* Community Builder (TM)
* @version $Id: $
* @package CommunityBuilder
* @copyright (C) 2004-2021 www.joomlapolis.com / Lightning MultiCom SA - and its licensors, all rights reserved
* @license http://www.gnu.org/licenses/old-licenses/gpl-2.0.html GNU/GPL version 2
*/

namespace CB\Plugin\Gallery\Table;

use CBLib\Database\Table\Table;
use CBLib\Input\Get;
use CBLib\Language\CBTxt;
use CBLib\Registry\Registry;
use CBLib\Application\Application;
use CBLib\Registry\GetterInterface;
use CBLib\Image\Image;
use GuzzleHttp\Client;
use Exception;
use CB\Plugin\Gallery\CBGallery;
use CB\Plugin\Gallery\Gallery;
use CB\Database\Table\UserTable;

defined('CBLIB') or die();

class ItemTable extends Table
{
	/** @var int  */
	public $id				=	null;
	/** @var int  */
	public $user_id			=	null;
	/** @var string  */
	public $type			=	null;
	/** @var string  */
	public $asset			=	null;
	/** @var string  */
	public $value			=	null;
	/** @var string  */
	public $file			=	null;
	/** @var int  */
	public $folder			=	null;
	/** @var string  */
	public $title			=	null;
	/** @var string  */
	public $description		=	null;
	/** @var string  */
	public $thumbnail		=	null;
	/** @var string  */
	public $date			=	null;
	/** @var int  */
	public $published		=	null;
	/** @var string  */
	public $params			=	null;

	/** @var Registry  */
	protected $_input		=	null;
	/** @var Registry  */
	protected $_files		=	null;
	/** @var Registry  */
	protected $_params		=	null;

	/**
	 * Table name in database
	 * @var string
	 */
	protected $_tbl			=	'#__comprofiler_plugin_gallery_items';

	/**
	 * Primary key(s) of table
	 * @var string
	 */
	protected $_tbl_key		=	'id';

	/**
	 * @param null|Gallery $gallery
	 * @return bool
	 */
	public function check( $gallery = null )
	{
		$new								=	( ! $this->getInt( 'id', 0 ) );
		$old								=	new self();

		if ( ! $new ) {
			$old->load( $this->getInt( 'id', 0 ) );
		}

		$type								=	$this->discoverType( $gallery );

		if ( $gallery ) {
			$params							=	$gallery;
		} else {
			$params							=	CBGallery::getGlobalParams();
		}

		$minFileSize						=	$params->getInt( $type . '_min_size', 0 );
		$maxFileSize						=	$params->getInt( $type . '_max_size', 1024 );
		$thumbnails							=	$params->getBool( 'thumbnails', true );
		$thumbnailsUpload					=	$params->getBool( 'thumbnails_upload', true );
		$thumbnailsLink						=	$params->getBool( 'thumbnails_link', false );
		$minThumbnailSize					=	$params->getInt( 'thumbnails_min_size', 0 );
		$maxThumbnailSize					=	$params->getInt( 'thumbnails_max_size', 1024 );

		if ( ! $this->getInt( 'user_id', 0 ) ) {
			$this->setError( CBTxt::T( 'Owner not specified!' ) );

			return false;
		}

		if ( $type === '' ) {
			$this->setError( CBTxt::T( 'Type not specified!' ) );

			return false;
		}

		$upload								=	$this->getUpload();
		$uploadError						=	$this->getUploadError( $upload );

		if ( $uploadError ) {
			$this->setError( $uploadError );

			return false;
		}

		$uploadFile							=	$upload->getString( 'tmp_name' );
		$value								=	$this->getString( 'value' );

		if ( ( ! $this->getInt( 'id', 0 ) ) && ( ( ! $value ) && ( ! $uploadFile ) ) ) {
			$this->setError( CBTxt::T( 'Nothing to upload or link!' ) );

			return false;
		}

		if ( $uploadFile ) {
			$uploadExtension				=	CBGallery::getUploadExtension( $upload );

			if ( ( ! $uploadExtension ) || ( ! in_array( $uploadExtension, CBGallery::getExtensions( $type, $gallery, 'upload' ), true ) ) ) {
				$this->setError( CBTxt::T( 'FILE_UPLOAD_INVALID_EXT', 'Invalid file extension [extension]. Please upload only [extensions]!', array( '[extension]' => $uploadExtension, '[extensions]' => implode( ', ', CBGallery::getExtensions( 'all', $gallery, 'upload' ) ) ) ) );

				if ( $upload->getBool( 'is_data', false ) && file_exists( $uploadFile ) ) {
					@unlink( $uploadFile );
				}

				return false;
			}

			$uploadSize						=	$upload->getInt( 'size', 0 );

			if ( $minFileSize && ( ( $uploadSize / 1024 ) < $minFileSize ) ) {
				$this->setError( CBTxt::T( 'FILE_UPLOAD_TOO_SMALL', 'The file is too small. The minimum size is [size]!', array( '[size]' => CBGallery::getFormattedFileSize( $minFileSize * 1024 ) ) ) );

				if ( $upload->getBool( 'is_data', false ) && file_exists( $uploadFile ) ) {
					@unlink( $uploadFile );
				}

				return false;
			}

			if ( $maxFileSize && ( ( $uploadSize / 1024 ) > $maxFileSize ) ) {
				$this->setError( CBTxt::T( 'FILE_UPLOAD_TOO_LARGE', 'The file is too large. The maximum size is [size]!', array( '[size]' => CBGallery::getFormattedFileSize( $maxFileSize * 1024 ) ) ) );

				if ( $upload->getBool( 'is_data', false ) && file_exists( $uploadFile ) ) {
					@unlink( $uploadFile );
				}

				return false;
			}
		} elseif ( $value && ( $value !== $old->getString( 'value' ) ) ) {
			if ( $this->domain() ) {
				switch ( $this->domain() ) {
					case 'youtube':
					case 'youtu':
						$linkExtension		=	'youtube';
						break;
					case 'vimeo':
						$linkExtension		=	'vimeo';
						break;
					case 'facebook':
						$linkExtension		=	'facebook';
						break;
					default:
						$link				=	CBGallery::parseUrl( $value );

						if ( ! $link['exists'] ) {
							$this->setError( CBTxt::T( 'FILE_LINK_INVALID_URL', 'Invalid file URL. Please ensure the URL exists!' ) );

							return false;
						}

						$linkExtension		=	$link['extension'];
						break;
				}

				if ( ( ! $linkExtension ) || ( ! in_array( $linkExtension, CBGallery::getExtensions( $type, $gallery, 'link' ), true ) ) ) {
					$this->setError( CBTxt::T( 'FILE_LINK_INVALID_EXT', 'Invalid file URL extension [extension]. Please link only [extensions]!', array( '[extension]' => $linkExtension, '[extensions]' => implode( ', ', CBGallery::getExtensions( 'all', $gallery, 'link' ) ) ) ) );

					return false;
				}
			}
		}

		if ( ( ( ( $type === 'files' ) && ( ! ( ( $this->extension() === 'svg' ) || ( CBGallery::getExtensionType( $this->extension() ) === 'photos' ) ) ) ) || ( $type === 'music' ) )
			 && $thumbnails
		) {
			$uploadThumbnail				=	$this->getUpload( 'thumbnail_upload' );
			$uploadThumbnailError			=	$this->getUploadError( $uploadThumbnail );

			if ( $uploadThumbnailError ) {
				$this->setError( $uploadThumbnailError );

				return false;
			}

			$uploadThumbnailFile			=	$uploadThumbnail->getString( 'tmp_name' );
			$thumbnail						=	$this->getString( 'thumbnail' );

			if ( $thumbnailsUpload && $uploadThumbnailFile ) {
				$uploadThumbnailExtension	=	CBGallery::getUploadExtension( $uploadThumbnail );

				if ( ( ! $uploadThumbnailExtension ) || ( ! in_array( $uploadThumbnailExtension, CBGallery::getExtensions( 'photos' ), true ) ) ) {
					$this->setError( CBTxt::T( 'THUMBNAIL_UPLOAD_INVALID_EXT', 'Invalid thumbnail file extension [extension]. Please upload only [extensions]!', array( '[extension]' => $uploadThumbnailExtension, '[extensions]' => implode( ', ', CBGallery::getExtensions( 'photos' ) ) ) ) );

					if ( $uploadThumbnail->getBool( 'is_data', false ) && file_exists( $uploadThumbnailFile ) ) {
						@unlink( $uploadThumbnailFile );
					}

					return false;
				}

				$uploadThumbnailSize		=	$uploadThumbnail->getInt( 'size', 0 );

				if ( $minThumbnailSize && ( ( $uploadThumbnailSize / 1024 ) < $minThumbnailSize ) ) {
					$this->setError( CBTxt::T( 'THUMBNAIL_UPLOAD_TOO_SMALL', 'The thumbnail file is too small. The minimum size is [size]!', array( '[size]' => CBGallery::getFormattedFileSize( $minThumbnailSize * 1024 ) ) ) );

					if ( $uploadThumbnail->getBool( 'is_data', false ) && file_exists( $uploadThumbnailFile ) ) {
						@unlink( $uploadThumbnailFile );
					}

					return false;
				}

				if ( $maxThumbnailSize && ( ( $uploadThumbnailSize / 1024 ) > $maxThumbnailSize ) ) {
					$this->setError( CBTxt::T( 'THUMBNAIL_UPLOAD_TOO_LARGE', 'The thumbnail file is too large. The maximum size is [size]!', array( '[size]' => CBGallery::getFormattedFileSize( $maxThumbnailSize * 1024 ) ) ) );

					if ( $uploadThumbnail->getBool( 'is_data', false ) && file_exists( $uploadThumbnailFile ) ) {
						@unlink( $uploadThumbnailFile );
					}

					return false;
				}
			} elseif ( $thumbnailsLink && $thumbnail && ( $thumbnail !== $old->getString( 'thumbnail' ) ) ) {
				if ( $this->domain( true ) ) {
					$link					=	CBGallery::parseUrl( $thumbnail );

					if ( ! $link['exists'] ) {
						$this->setError( CBTxt::T( 'THUMBNAIL_LINK_INVALID_URL', 'Invalid thumbnail file URL. Please ensure the URL exists!' ) );

						return false;
					}

					$linkExtension			=	$link['extension'];

					if ( ( ! $linkExtension ) || ( ! in_array( $linkExtension, CBGallery::getExtensions( 'photos' ), true ) ) ) {
						$this->setError( CBTxt::T( 'THUMBNAIL_LINK_INVALID_EXT', 'Invalid thumbnail file URL extension [extension]. Please link only [extensions]!', array( '[extension]' => $linkExtension, '[extensions]' => implode( ', ', CBGallery::getExtensions( 'photos' ) ) ) ) );

						return false;
					}
				}
			}
		}

		return true;
	}

	/**
	 * @param bool         $updateNulls
	 * @param null|Gallery $gallery
	 * @return bool
	 */
	public function store( $updateNulls = false, $gallery = null )
	{
		global $_CB_framework, $_PLUGINS;

		if ( Application::Application()->isClient( 'administrator' ) ) {
			$input							=	Application::Input();
		} else {
			$input							=	$this->getRaw( '_input', new Registry() );
		}

		$new								=	( ! $this->getInt( 'id', 0 ) );
		$old								=	new self();

		if ( ! $new ) {
			$old->load( $this->getInt( 'id', 0 ) );

			$this->cache( true );
		}

		if ( ! $this->getString( 'asset' ) ) {
			$this->set( 'asset', 'profile.' . $this->getInt( 'user_id', 0 ) );
		}

		$this->set( 'published', $this->getInt( 'published', 0 ) );
		$this->set( 'date', $this->getString( 'date', Application::Database()->getUtcDateTime() ) );

		$type								=	$this->discoverType( $gallery );

		if ( $type !== $this->getString( 'type' ) ) {
			$this->set( 'type', $type );
		}

		if ( $gallery ) {
			$params							=	$gallery;
		} else {
			$params							=	CBGallery::getGlobalParams();
		}

		$resample							=	$params->getInt( 'photos_resample', 1 );
		$aspectRatio						=	$params->getInt( 'photos_maintain_aspect_ratio', 1 );
		$imageHeight						=	$params->getInt( 'photos_image_height', 640 );

		if ( ! $imageHeight ) {
			$imageHeight					=	640;
		}

		$imageWidth							=	$params->getInt( 'photos_image_width', 1280 );

		if ( ! $imageWidth ) {
			$imageWidth						=	1280;
		}

		$thumbHeight						=	$params->getInt( 'photos_thumbnail_height', 320 );

		if ( ! $thumbHeight ) {
			$thumbHeight					=	320;
		}

		$thumbWidth							=	$params->getInt( 'photos_thumbnail_width', 640 );

		if ( ! $thumbWidth ) {
			$thumbWidth						=	640;
		}

		$thumbnails							=	$params->getBool( 'thumbnails', true );
		$thumbnailsUpload					=	$params->getBool( 'thumbnails_upload', true );
		$thumbnailsLink						=	$params->getBool( 'thumbnails_link', false );
		$thumbnailsResample					=	$params->getInt( 'thumbnails_resample', 1 );
		$thumbnailsAspectRatio				=	$params->getInt( 'thumbnails_maintain_aspect_ratio', 1 );
		$thumbnailsImageHeight				=	$params->getInt( 'thumbnails_image_height', 320 );

		if ( ! $thumbnailsImageHeight ) {
			$thumbnailsImageHeight			=	320;
		}

		$thumbnailsImageWidth				=	$params->getInt( 'thumbnails_image_width', 640 );

		if ( ! $thumbnailsImageWidth ) {
			$thumbnailsImageWidth			=	640;
		}

		$checksumMD5						=	$params->getBool( 'files_md5', false );
		$checksumSHA1						=	$params->getBool( 'files_sha1', false );

		$conversionType						=	Application::Config()->getInt( 'conversiontype', 0 );
		$imageSoftware						=	( $conversionType === 5 ? 'gmagick' : ( $conversionType === 1 ? 'imagick' : ( $conversionType === 4 ? 'gd' : 'auto' ) ) );

		$upload								=	$this->getUpload();
		$uploadError						=	$this->getUploadError( $upload );

		if ( $uploadError ) {
			$this->setError( $uploadError );

			return false;
		}

		$uploadFile							=	$upload->getString( 'tmp_name' );
		$value								=	$this->getString( 'value' );

		$basePath							=	$_CB_framework->getCfg( 'absolute_path' ) . '/images/comprofiler/plug_cbgallery';
		$filePath							=	$basePath . '/' . $this->getInt( 'user_id', 0 ) . '/' . $type;

		if ( $uploadFile ) {
			CBGallery::createDirectory( $basePath, $this->getInt( 'user_id', 0 ), $type );

			$uploadExtension				=	CBGallery::getUploadExtension( $upload );

			if ( $uploadExtension === 'webp' ) {
				$uploadExtension			=	'png';

				$upload->set( 'type', 'image/png' );
			}

			$uploadName						=	$upload->getString( 'name' );

			if ( ( ! $uploadName ) || ( $uploadName === 'blob' ) ) {
				$uploadName					=	Application::Date( 'now', 'UTC' )->getTimestamp() . '.' . $uploadExtension;
			} else {
				$uploadName					=	pathinfo( parse_url( $uploadName, PHP_URL_PATH ), PATHINFO_FILENAME ) . '.' . $uploadExtension;
			}

			$upload->set( 'name', $uploadName );

			$uploadId						=	md5( uniqid( $uploadName ) );

			if ( $type === 'photos' ) {
				try {
					$image					=	new Image( $imageSoftware, $resample, $aspectRatio );

					$image->setName( $uploadId );
					$image->setSource( $upload->asArray() );
					$image->setDestination( $filePath . '/' );

					$image->processImage( $imageWidth, $imageHeight );

					$newFileName			=	$image->getCleanFilename();

					$image->setName( 'tn' . $uploadId );

					$image->processImage( $thumbWidth, $thumbHeight );

					if ( $value ) {
						if ( file_exists( $filePath . '/' . $value ) ) {
							@unlink( $filePath . '/' . $value );
						}

						if ( file_exists( $filePath . '/tn' . $value ) ) {
							@unlink( $filePath . '/tn' . $value );
						}
					}

					$this->set( 'value', $newFileName );
					$this->set( 'file', $uploadName );

					if ( $params->getBool( 'photos_metadata', false ) ) {
						$imagineImage		=	$image->getImage();

						if ( $imagineImage ) {
							$metadata		=	$imagineImage->metadata();

							if ( $metadata ) {
								$exif		=	new Registry();

								foreach ( $metadata->toArray() as $k => $v ) {
									if ( ( strpos( $k, 'exif' ) !== 0 ) && ( strpos( $k, 'ifd' ) !== 0 ) ) {
										continue;
									}

									$exif->set( $k, Get::clean( $v, GetterInterface::STRING ) );
								}

								$this->params()->set( 'metadata', $exif->asArray() );
							}
						}
					}

					if ( $upload->getBool( 'is_data', false ) && file_exists( $uploadFile ) ) {
						@unlink( $uploadFile );
					}
				} catch ( Exception $e ) {
					$this->setError( $e->getMessage() );

					if ( $upload->getBool( 'is_data', false ) && file_exists( $uploadFile ) ) {
						@unlink( $uploadFile );
					}

					return false;
				}
			} else {
				if ( $uploadExtension === 'mov' ) {
					$uploadExtension		=	'mp4';
				}

				$newFileName				=	$uploadId . '.' . $uploadExtension;

				if ( ! move_uploaded_file( $uploadFile, $filePath . '/' . $newFileName ) ) {
					$this->setError( CBTxt::T( 'FILE_FAILED_TO_UPLOAD', 'The file [file] failed to upload!', array( '[file]' => $newFileName ) ) );

					return false;
				}

				@chmod( $filePath . '/' . $newFileName, 0755 );

				if ( $value ) {
					if ( file_exists( $filePath . '/' . $value ) ) {
						@unlink( $filePath . '/' . $value );
					}

					if ( file_exists( $filePath . '/tn' . $value ) ) {
						@unlink( $filePath . '/tn' . $value );
					}
				}

				$this->set( 'value', $newFileName );
				$this->set( 'file', $uploadName );

				if ( $type === 'files' ) {
					if ( $checksumMD5 ) {
						$this->params()->set( 'checksum.md5', @md5_file( $filePath . '/' . $newFileName ) );
					}

					if ( $checksumSHA1 ) {
						$this->params()->set( 'checksum.sha1', @sha1_file( $filePath . '/' . $newFileName ) );
					}
				}
			}
		} elseif ( $this->domain() )  {
			$this->set( 'file', '' );

			if ( $value && ( $value !== $old->getString( 'value' ) ) ) {
				$link						=	CBGallery::parseUrl( $value );

				if ( $link['exists'] ) {
					$this->set( 'value', $link['url'] );

					switch ( preg_replace( '/^(?:(?:\w+\.)*)?(\w+)\..+$/', '\1', parse_url( $link['url'], PHP_URL_HOST ) ) ) {
						case 'youtube':
						case 'youtu':
							if ( ! preg_match( '%(?:(?:watch\?v=)|(?:embed/)|(?:shorts/)|(?:be/))([A-Za-z0-9_-]+)%', $link['url'] ) ) {
								$this->setError( CBTxt::T( 'FILE_LINK_INVALID_YOUTUBE_URL', 'Invalid Youtube URL. Please ensure the URL exists!' ) );

								return false;
							}
							break;
						case 'vimeo':
							if ( ! preg_match( '%/(\d+)$%', $link['url'] ) ) {
								$this->setError( CBTxt::T( 'FILE_LINK_INVALID_VIMEO_URL', 'Invalid Vime URL. Please ensure the URL exists!' ) );

								return false;
							}
							break;
						case 'facebook':
							if ( ! preg_match( '%/(\d+)/$%', $link['url'] ) ) {
								$this->setError( CBTxt::T( 'FILE_LINK_INVALID_FACEBOOK_URL', 'Invalid Facebook Video URL. Please ensure the URL exists!' ) );

								return false;
							}
							break;
					}

					if ( $new ) {
						if ( ! $this->getString( 'title' ) ) {
							$this->set( 'title', $link['title'] );
						}

						if ( ! $this->getString( 'description' ) ) {
							$this->set( 'description', $link['description'] );
						}

						if ( \in_array( $type, [ 'videos', 'music' ], true )
							 && ( ! \in_array( $this->domain(), [ 'youtube', 'youtu' ], true ) )
						) {
							$this->params()->set( 'link_thumbnail', $link['thumbnail'] );
						}
					}
				} else {
					$this->setError( CBTxt::T( 'FILE_LINK_INVALID_URL', 'Invalid file URL. Please ensure the URL exists!' ) );

					return false;
				}
			}
		} elseif ( ! $this->getString( 'file' ) ) {
			$this->set( 'file', $value );
		}

		if ( ( ( ( $type === 'files' ) && ( ! ( ( $this->extension() === 'svg' ) || ( CBGallery::getExtensionType( $this->extension() ) === 'photos' ) ) ) ) || ( $type === 'music' ) )
			 && $thumbnails
		) {
			$uploadThumbnail				=	$this->getUpload( 'thumbnail_upload' );
			$uploadThumbnailError			=	$this->getUploadError( $uploadThumbnail );

			if ( $uploadThumbnailError ) {
				$this->setError( $uploadThumbnailError );

				return false;
			}

			$uploadThumbnailFile			=	$uploadThumbnail->getString( 'tmp_name' );
			$thumbnail						=	$this->getString( 'thumbnail' );

			if ( $input->getInt( 'thumbnail_method', 0 ) === 3 ) {
				if ( $thumbnail ) {
					if ( ! $this->domain( true ) ) {
						if ( file_exists( $filePath . '/' . $thumbnail ) ) {
							@unlink( $filePath . '/' . $thumbnail );
						}
					}

					$this->set( 'thumbnail', '' );
				}
			} elseif ( $thumbnailsUpload && $uploadThumbnailFile ) {
				CBGallery::createDirectory( $basePath, $this->getInt( 'user_id', 0 ), $type );

				$uploadThumbnailExtension	=	CBGallery::getUploadExtension( $uploadThumbnail );

				if ( $uploadThumbnailExtension === 'webp' ) {
					$uploadThumbnailExtension	=	'png';

					$uploadThumbnail->set( 'type', 'image/png' );
				}

				$uploadThumbnailName		=	$uploadThumbnail->getString( 'name' );

				if ( ( ! $uploadThumbnailName ) || ( $uploadThumbnailName === 'blob' ) ) {
					$uploadThumbnailName	=	Application::Date( 'now', 'UTC' )->getTimestamp() . '.' . $uploadThumbnailExtension;
				} else {
					$uploadThumbnailName	=	pathinfo( parse_url( $uploadThumbnailName, PHP_URL_PATH ), PATHINFO_FILENAME ) . '.' . $uploadThumbnailExtension;
				}

				$uploadThumbnail->set( 'name', $uploadThumbnailName );

				try {
					$image					=	new Image( $imageSoftware, $thumbnailsResample, $thumbnailsAspectRatio );

					$image->setName( md5( uniqid( $uploadThumbnailName ) ) );
					$image->setSource( $uploadThumbnail->asArray() );
					$image->setDestination( $filePath . '/' );

					$image->processImage( $thumbnailsImageWidth, $thumbnailsImageHeight );

					if ( $thumbnail ) {
						if ( file_exists( $filePath . '/' . $thumbnail ) ) {
							@unlink( $filePath . '/' . $thumbnail );
						}
					}

					$this->set( 'thumbnail', $image->getCleanFilename() );

					if ( $uploadThumbnail->getBool( 'is_data', false ) && file_exists( $uploadThumbnailFile ) ) {
						@unlink( $uploadThumbnailFile );
					}
				} catch ( Exception $e ) {
					$this->setError( $e->getMessage() );

					if ( $uploadThumbnail->getBool( 'is_data', false ) && file_exists( $uploadThumbnailFile ) ) {
						@unlink( $uploadThumbnailFile );
					}

					return false;
				}
			} elseif ( $thumbnailsLink && $thumbnail && ( $thumbnail !== $old->getString( 'thumbnail' ) ) ) {
				if ( $this->domain( true ) ) {
					$linkThumbnail			=	CBGallery::parseUrl( $thumbnail );

					if ( $linkThumbnail['exists'] ) {
						$this->set( 'thumbnail', $linkThumbnail['url'] );
					}
				}
			}
		}

		$this->cache();

		if ( ! $new ) {
			$_PLUGINS->trigger( 'gallery_onBeforeUpdateItem', array( &$this, $old ) );
		} else {
			$_PLUGINS->trigger( 'gallery_onBeforeCreateItem', array( &$this ) );
		}

		if ( ! parent::store( $updateNulls ) ) {
			return false;
		}

		if ( $old->getInt( 'id', 0 ) && ( ! $old->domain() ) && ( $old->getInt( 'user_id', 0 ) !== $this->getInt( 'user_id', 0 ) ) ) {
			$basePath						=	$_CB_framework->getCfg( 'absolute_path' ) . '/images/comprofiler/plug_cbgallery';
			$oldPath						=	$basePath . '/' . $old->getInt( 'user_id', 0 ) . '/' . $old->getString( 'type' );
			$newPath						=	$basePath . '/' . $this->getInt( 'user_id', 0 ) . '/' . $type;

			if ( is_dir( $oldPath ) ) {
				CBGallery::createDirectory( $basePath, $this->getInt( 'user_id', 0 ), $type );

				if ( $this->getString( 'value' ) && ( ! $this->domain() ) ) {
					if ( file_exists( $oldPath . '/' . $this->getString( 'value' ) ) ) {
						@rename( $oldPath . '/' . $this->getString( 'value' ), $newPath . '/' . $this->getString( 'value' ) );
					}

					if ( file_exists( $oldPath . '/tn' . $this->getString( 'value' ) ) ) {
						@rename( $oldPath . '/tn' . $this->getString( 'value' ), $newPath . '/tn' . $this->getString( 'value' ) );
					}
				}

				if ( $this->getString( 'thumbnail' ) && ( ! $this->domain( true ) ) ) {
					if ( file_exists( $oldPath . '/' . $this->getString( 'thumbnail' ) ) ) {
						@rename( $oldPath . '/' . $this->getString( 'thumbnail' ), $newPath . '/' . $this->getString( 'thumbnail' ) );
					}
				}
			}
		}

		if ( ! $new ) {
			$_PLUGINS->trigger( 'gallery_onAfterUpdateItem', array( $this, $old ) );
		} else {
			$_PLUGINS->trigger( 'gallery_onAfterCreateItem', array( $this ) );
		}

		return true;
	}

	/**
	 * resets or sets the media details cache
	 *
	 * @param bool $resetOnly
	 */
	private function cache( $resetOnly = false )
	{
		$cache		=	$this->params();

		$cache->unsetEntry( 'name' );
		$cache->unsetEntry( 'extension' );
		$cache->unsetEntry( 'mimetype' );
		$cache->unsetEntry( 'modified' );
		$cache->unsetEntry( 'filesize' );
		$cache->unsetEntry( 'height' );
		$cache->unsetEntry( 'width' );

		$cache->unsetEntry( 'name_thumbnail' );
		$cache->unsetEntry( 'extension_thumbnail' );
		$cache->unsetEntry( 'mimetype_thumbnail' );
		$cache->unsetEntry( 'modified_thumbnail' );
		$cache->unsetEntry( 'filesize_thumbnail' );
		$cache->unsetEntry( 'height_thumbnail' );
		$cache->unsetEntry( 'width_thumbnail' );

		if ( $resetOnly ) {
			return;
		}

		$cache->set( 'name', $this->name() );
		$cache->set( 'extension', $this->extension() );
		$cache->set( 'mimetype', $this->mimeType() );
		$cache->set( 'modified', $this->modified() );
		$cache->set( 'filesize', $this->size( false, true ) );
		$cache->set( 'height', $this->height() );
		$cache->set( 'width', $this->width() );

		$cache->set( 'name_thumbnail', $this->name( true ) );
		$cache->set( 'extension_thumbnail', $this->extension( true ) );
		$cache->set( 'mimetype_thumbnail', $this->mimeType( true ) );
		$cache->set( 'modified_thumbnail', $this->modified( true ) );
		$cache->set( 'filesize_thumbnail', $this->size( true, true ) );
		$cache->set( 'height_thumbnail', $this->height( true ) );
		$cache->set( 'width_thumbnail', $this->width( true ) );

		$this->set( 'params', $cache->asJson() );
	}

	/**
	 * @param null|int $id
	 * @return bool
	 */
	public function delete( $id = null )
	{
		global  $_PLUGINS;

		$_PLUGINS->trigger( 'gallery_onBeforeDeleteItem', array( &$this ) );

		if ( ! parent::delete( $id ) ) {
			return false;
		}

		if ( ( ! $this->domain() ) && $this->exists() ) {
			@unlink( $this->path() );
		}

		if ( ( ! $this->domain( true ) ) && $this->exists( true ) ) {
			@unlink( $this->path( true ) );
		}

		if ( $this->getInt( 'folder', 0 ) ) {
			$folder		=	$this->folder();

			if ( $folder->getInt( 'id', 0 ) && ( $folder->getInt( 'thumbnail', 0 ) === $this->getInt( 'id', 0 ) ) ) {
				$folder->set( 'thumbnail', 0 );

				$folder->store();
			}
		}

		$_PLUGINS->trigger( 'gallery_onAfterDeleteItem', array( $this ) );

		return true;
	}

	/**
	 * @return Registry
	 */
	public function params()
	{
		if ( ! ( $this->getRaw( '_params' ) instanceof Registry ) ) {
			$this->set( '_params', new Registry( $this->getRaw( 'params' ) ) );
		}

		return $this->getRaw( '_params' );
	}

	/**
	 * @return UserTable|null
	 */
	public function source()
	{
		global $_PLUGINS;

		static $cache		=	array();

		$id					=	$this->getString( 'asset' );

		if ( ! isset( $cache[$id] ) ) {
			$source			=	CBGallery::getSource( $id );

			$_PLUGINS->trigger( 'gallery_onItemSource', array( $this, &$source ) );

			$cache[$id]		=	$source;
		}

		return $cache[$id];
	}

	/**
	 * Returns the domain if the item is a link
	 *
	 * @param bool $thumbnail
	 * @return string
	 */
	public function domain( $thumbnail = false )
	{
		static $cache		=	array();

		$id					=	$this->path( $thumbnail );

		if ( ! isset( $cache[$id] ) ) {
			$cache[$id]		=	preg_replace( '/^(?:(?:\w+\.)*)?(\w+)\..+$/', '\1', parse_url( $id, PHP_URL_HOST ) );
		}

		return $cache[$id];
	}

	/**
	 * Returns the clean absolute path to the items file
	 *
	 * @param bool $thumbnail
	 * @return null|string
	 */
	public function path( $thumbnail = false )
	{
		global $_CB_framework;

		static $cache					=	array();

		$type							=	$this->getString( 'type' );
		$id								=	null;

		if ( ( $type !== 'photos' ) && $thumbnail ) {
			$id							=	$this->getString( 'thumbnail' );

			if ( ! $id ) {
				$id						=	$this->params()->getString( 'link_thumbnail' );
			}
		}

		if ( ! $id ) {
			$id							=	$this->getString( 'value' );
		}

		if ( ! isset( $cache[$id][$thumbnail] ) ) {
			$domain						=	preg_replace( '/^(?:(?:\w+\.)*)?(\w+)\..+$/', '\1', parse_url( $id, PHP_URL_HOST ) );
			$path						=	null;
			$value						=	null;

			if ( $domain ) {
				if ( $thumbnail && in_array( $domain, array( 'youtube', 'youtu' ), true ) && ( ! $this->getString( 'thumbnail' ) ) ) {
					preg_match( '%(?:(?:watch\?v=)|(?:embed/)|(?:shorts/)|(?:be/))([A-Za-z0-9_-]+)%', $id, $matches );

					$path				=	'https://img.youtube.com/vi/' . ( isset( $matches[1] ) ? htmlspecialchars( $matches[1] ) : 'unknown' ) . '/0.jpg';
				}

				if ( ! $path ) {
					$path				=	$id;
				}
			} else {
				$idPath					=	parse_url( $id, PHP_URL_PATH );
				$value					=	pathinfo( $idPath, PATHINFO_FILENAME ) . '.' . strtolower( preg_replace( '/[^-a-zA-Z0-9_]/', '', pathinfo( $idPath, PATHINFO_EXTENSION ) ) );

				if ( $type === 'photos' ) {
					$value				=	( $thumbnail ? 'tn' : null ) . $value;
				}
			}

			if ( ! $path ) {
				$path					=	$_CB_framework->getCfg( 'absolute_path' ) . '/images/comprofiler/plug_cbgallery/' . $this->getInt( 'user_id', 0 ) . '/' . $type . '/' . $value;
			}

			$cache[$id][$thumbnail]		=	$path;
		}

		return $cache[$id][$thumbnail];
	}

	/**
	 * Checks if the file exists
	 *
	 * @param bool $thumbnail
	 * @return bool
	 */
	public function exists( $thumbnail = false )
	{
		static $cache						=	array();

		$id									=	$this->path( $thumbnail );

		if ( ! isset( $cache[$id] ) ) {
			$exists							=	false;

			if ( $id ) {
				$domain						=	$this->domain( $thumbnail );

				if ( $domain ) {
					if ( \in_array( $domain, [ 'youtube', 'youtu', 'vimeo', 'facebook' ], true ) && ( ! $thumbnail ) ) {
						$exists				=	true;
					} else {
						$link				=	CBGallery::parseUrl( $id );
						$exists				=	$link['exists'];
					}
				} else {
					$exists					=	file_exists( $id );
				}
			}

			$cache[$id]						=	$exists;
		}

		return $cache[$id];
	}

	/**
	 * Returns the file size raw or formatted to largest increment possible
	 *
	 * @param bool $thumbnail
	 * @param bool $raw
	 * @param bool $cacheOnly
	 * @return string|int
	 */
	public function size( $thumbnail = false, $raw = false, $cacheOnly = false )
	{
		if ( Application::Application()->isClient( 'administrator' ) && is_object( $raw ) ) {
			$thumbnail							=	( $thumbnail === '1' );
			$raw								=	false;
			$cacheOnly							=	false;
		}

		static $cache							=	array();

		$id										=	$this->path( $thumbnail );

		if ( ! isset( $cache[$id] ) ) {
			$fileSize							=	$this->params()->getInt( ( $thumbnail ? 'filesize_thumbnail' : 'filesize' ) );

			if ( ( $fileSize === null ) && $this->exists( $thumbnail ) ) {
				if ( $cacheOnly ) {
					if ( ! $raw ) {
						return CBGallery::getFormattedFileSize( 0 );
					}

					return 0;
				}

				$fileSize						=	0;

				if ( $this->exists( $thumbnail ) ) {
					$domain						=	$this->domain( $thumbnail );

					if ( $domain ) {
						if ( ( ! \in_array( $domain, [ 'youtube', 'youtu', 'vimeo', 'facebook' ], true ) ) || $thumbnail ) {
							$link				=	CBGallery::parseUrl( $id );

							if ( $link['exists'] ) {
								$fileSize		=	$link['size'];
							}
						}
					} else {
						$fileSize				=	@filesize( $id );
					}
				}
			}

			$cache[$id]							=	$fileSize;
		}

		if ( ! $raw ) {
			return CBGallery::getFormattedFileSize( $cache[$id] );
		}

		return $cache[$id];
	}

	/**
	 * Returns the file name cleaned of the unique id
	 *
	 * @param bool $thumbnail
	 * @param bool $cacheOnly
	 * @return string|null
	 */
	public function name( $thumbnail = false, $cacheOnly = false )
	{
		if ( Application::Application()->isClient( 'administrator' ) && is_object( $cacheOnly ) ) {
			$thumbnail					=	( $thumbnail === '1' );
			$cacheOnly					=	false;
		}

		static $cache					=	array();

		$id								=	$this->path( $thumbnail );

		if ( ! isset( $cache[$id] ) ) {
			$name						=	$this->params()->getString( ( $thumbnail ? 'name_thumbnail' : 'name' ) );

			if ( $name === null ) {
				if ( $cacheOnly ) {
					return null;
				}

				$domain					=	$this->domain( $thumbnail );

				if ( $domain ) {
					if ( \in_array( $domain, [ 'youtube', 'youtu', 'vimeo', 'facebook' ], true ) && ( ! $thumbnail ) ) {
						switch ( $domain ) {
							case 'youtube':
							case 'youtu':
								$name	=	preg_replace( '%^.*(?:v=|v/|/)([\w-]+).*%i', '$1', $id );
								break;
							case 'vimeo':
								$name	=	preg_replace( '%^.*/([\w-]+).*%', '$1', $id );
								break;
							case 'facebook':
								$name	=	preg_replace( '%^.*/([\w-]+)/.*%', '$1', $id );
								break;
						}
					} else {
						$link			=	CBGallery::parseUrl( $id );

						if ( $link['exists'] && $link['name'] ) {
							$name		=	$link['name'];
						}
					}
				}

				if ( $name === null ) {
					if ( $this->getString( 'file' ) && ( ! $thumbnail ) ) {
						$name			=	pathinfo( parse_url( $this->getString( 'file' ), PHP_URL_PATH ), PATHINFO_FILENAME ) . '.' . $this->extension( $thumbnail );
					} else {
						$name			=	pathinfo( parse_url( $id, PHP_URL_PATH ), PATHINFO_FILENAME ) . '.' . $this->extension( $thumbnail );
					}
				}
			}

			$cache[$id]					=	$name;
		}

		return $cache[$id];
	}

	/**
	 * Returns the items file extension
	 *
	 * @param bool $thumbnail
	 * @param bool $cacheOnly
	 * @return string|null
	 */
	public function extension( $thumbnail = false, $cacheOnly = false )
	{
		if ( Application::Application()->isClient( 'administrator' ) && is_object( $cacheOnly ) ) {
			$thumbnail						=	( $thumbnail === '1' );
			$cacheOnly						=	false;
		}

		static $cache						=	array();

		$id									=	$this->path( $thumbnail );

		if ( ! isset( $cache[$id] ) ) {
			$extension						=	$this->params()->getString( ( $thumbnail ? 'extension_thumbnail' : 'extension' ) );

			if ( $extension === null ) {
				if ( $cacheOnly ) {
					return null;
				}

				$domain						=	$this->domain( $thumbnail );

				if ( $domain ) {
					if ( \in_array( $domain, [ 'youtube', 'youtu', 'vimeo', 'facebook' ], true ) && ( ! $thumbnail ) ) {
						switch ( $domain ) {
							case 'youtube':
							case 'youtu':
								$extension	=	'youtube';
								break;
							case 'vimeo':
								$extension	=	'vimeo';
								break;
							case 'facebook':
								$extension	=	'facebook';
								break;
						}
					} else {
						$link				=	CBGallery::parseUrl( $id );

						if ( $link['exists'] && $link['extension'] ) {
							$extension		=	$link['extension'];
						}
					}
				}

				if ( $extension === null ) {
					$extension				=	strtolower( preg_replace( '/[^-a-zA-Z0-9_]/', '', pathinfo( parse_url( $id, PHP_URL_PATH ), PATHINFO_EXTENSION ) ) );
				}
			}

			$cache[$id]						=	$extension;
		}

		return $cache[$id];
	}

	/**
	 * Returns the files mimetype from extension
	 *
	 * @param bool $thumbnail
	 * @param bool $cacheOnly
	 * @return string|null
	 */
	public function mimeType( $thumbnail = false, $cacheOnly = false )
	{
		if ( Application::Application()->isClient( 'administrator' ) && is_object( $cacheOnly ) ) {
			$thumbnail							=	( $thumbnail === '1' );
			$cacheOnly							=	false;
		}

		static $cache							=	array();
		static $mimeTypes						=	array();

		$id										=	$this->path( $thumbnail );

		if ( ! isset( $cache[$id] ) ) {
			$mimeType							=	$this->params()->getString( ( $thumbnail ? 'mimetype_thumbnail' : 'mimetype' ) );

			if ( $mimeType === null ) {
				if ( $cacheOnly ) {
					return 'application/octet-stream';
				}

				$domain							=	$this->domain( $thumbnail );

				if ( $domain ) {
					if ( \in_array( $domain, [ 'youtube', 'youtu', 'vimeo', 'facebook' ], true ) && ( ! $thumbnail ) ) {
						switch ( $domain ) {
							case 'youtube':
							case 'youtu':
								$mimeType		=	'video/x-youtube';
								break;
							case 'vimeo':
								$mimeType		=	'video/x-vimeo';
								break;
							case 'facebook':
								$mimeType		=	'video/x-facebook';
								break;
						}
					} else {
						$link					=	CBGallery::parseUrl( $id );

						if ( $link['exists'] && $link['mimetype'] ) {
							$mimeType			=	$link['mimetype'];
						}
					}
				}

				if ( $mimeType === null ) {
					$extension					=	$this->extension( $thumbnail );

					if ( ! isset( $mimeTypes[$extension] ) ) {
						$mimeTypes[$extension]	=	CBGallery::getMimeTypes( $extension );
					}

					$mimeType					=	$mimeTypes[$extension];
				}
			}

			$cache[$id]							=	$mimeType;
		}

		return $cache[$id];
	}

	/**
	 * Returns the files modified timestamp
	 *
	 * @param bool $thumbnail
	 * @param bool $cacheOnly
	 * @return string
	 */
	public function modified( $thumbnail = false, $cacheOnly = false )
	{
		if ( Application::Application()->isClient( 'administrator' ) && is_object( $cacheOnly ) ) {
			$thumbnail						=	( $thumbnail === '1' );
			$cacheOnly						=	false;
		}

		static $cache						=	array();

		$id									=	$this->path( $thumbnail );

		if ( ! isset( $cache[$id] ) ) {
			$modified						=	$this->params()->getInt( ( $thumbnail ? 'modified_thumbnail' : 'modified' ) );

			if ( ( $modified === null ) && $this->exists( $thumbnail ) ) {
				if ( $cacheOnly ) {
					return null;
				}

				if ( $this->domain( $thumbnail ) ) {
					$link					=	CBGallery::parseUrl( $id );

					if ( $link['exists'] ) {
						$modified			=	$link['modified'];
					}
				} else {
					$modified				=	filemtime( $id );
				}

				if ( ! $modified ) {
					$modified				=	$this->getString( 'date' );
				}

				$modified					=	Application::Date( $modified, 'UTC' )->getTimestamp();
			}

			$cache[$id]						=	$modified;
		}

		return $cache[$id];
	}

	/**
	 * Returns the image height cleaned of the unique id
	 *
	 * @param bool $thumbnail
	 * @param bool $cacheOnly
	 * @return int
	 */
	public function height( $thumbnail = false, $cacheOnly = false )
	{
		if ( Application::Application()->isClient( 'administrator' ) && is_object( $cacheOnly ) ) {
			$thumbnail					=	( $thumbnail === '1' );
			$cacheOnly					=	false;
		}

		if ( ( $this->getString( 'type' ) !== 'photos' ) && ( ( $this->getString( 'type' ) === 'files' ) && ( CBGallery::getExtensionType( $this->extension() ) === 'photos' ) ) && ( ! ( ( $this->getString( 'thumbnail' ) && $thumbnail ) || ( $this->domain() && $this->params()->getString( 'link_thumbnail' ) && $thumbnail ) ) ) ) {
			return 0;
		}

		static $cache					=	array();

		$id								=	$this->path( $thumbnail );

		if ( ! isset( $cache[$id] ) ) {
			$height						=	$this->params()->getInt( ( $thumbnail ? 'height_thumbnail' : 'height' ) );

			if ( ( $height === null ) && $this->exists( $thumbnail ) ) {
				if ( $cacheOnly ) {
					return 0;
				}

				$height					=	0;
				$size					=	false;

				if ( $this->domain( $thumbnail ) ) {
					if ( PHP_VERSION_ID >= 50400 ) {
						try {
							$request	=	new Client();

							$request	=	$request->get( $id, array( 'timeout' => 5 ) );

							if ( $request && ( (int) $request->getStatusCode() === 200 ) ) {
								$size	=	@getimagesizefromstring( (string) $request->getBody() );
							}
						} catch( Exception $e ) {}
					}
				} else {
					$size				=	@getimagesize( $id );
				}

				if ( $size !== false ) {
					$height				=	(int) $size[1];
				}
			}

			$cache[$id]					=	$height;
		}

		return $cache[$id];
	}

	/**
	 * Returns the image width cleaned of the unique id
	 *
	 * @param bool $thumbnail
	 * @param bool $cacheOnly
	 * @return int
	 */
	public function width( $thumbnail = false, $cacheOnly = false )
	{
		if ( Application::Application()->isClient( 'administrator' ) && is_object( $cacheOnly ) ) {
			$thumbnail					=	( $thumbnail === '1' );
			$cacheOnly					=	false;
		}

		if ( ( $this->getString( 'type' ) !== 'photos' ) && ( ( $this->getString( 'type' ) === 'files' ) && ( CBGallery::getExtensionType( $this->extension() ) === 'photos' ) ) && ( ! ( ( $this->getString( 'thumbnail' ) && $thumbnail ) || ( $this->domain() && $this->params()->getString( 'link_thumbnail' ) && $thumbnail ) ) ) ) {
			return 0;
		}

		static $cache					=	array();

		$id								=	$this->path( $thumbnail );

		if ( ! isset( $cache[$id] ) ) {
			$width						=	$this->params()->getInt( ( $thumbnail ? 'width_thumbnail' : 'width' ) );

			if ( ( $width === null ) && $this->exists( $thumbnail ) ) {
				if ( $cacheOnly ) {
					return 0;
				}

				$width					=	0;
				$size					=	false;

				if ( $this->domain( $thumbnail ) ) {
					if ( PHP_VERSION_ID >= 50400 ) {
						try {
							$request	=	new Client();

							$request	=	$request->get( $id, array( 'timeout' => 5 ) );

							if ( $request && ( (int) $request->getStatusCode() === 200 ) ) {
								$size	=	@getimagesizefromstring( (string) $request->getBody() );
							}
						} catch( Exception $e ) {}
					}
				} else {
					$size				=	@getimagesize( $id );
				}

				if ( $size !== false ) {
					$width				=	(int) $size[0];
				}
			}

			$cache[$id]					=	$width;
		}

		return $cache[$id];
	}

	/**
	 * Rotates the item
	 *
	 * @param int $angle
	 * @return bool
	 */
	public function rotate( $angle = 0 )
	{
		if ( ( $this->getString( 'type' ) !== 'photos' ) || $this->domain() ) {
			return false;
		}

		$conversionType		=	Application::Config()->getInt( 'conversiontype', 0 );
		$imageSoftware		=	( $conversionType === 5 ? 'gmagick' : ( $conversionType === 1 ? 'imagick' : ( $conversionType === 4 ? 'gd' : 'auto' ) ) );

		try {
			// Fullsize:
			$image			=	new Image( $imageSoftware );
			$fullPath		=	parse_url( $this->path(), PHP_URL_PATH );

			$image->setName( pathinfo( $fullPath, PATHINFO_FILENAME ) );
			$image->setSource( $this->path() );
			$image->setDestination( pathinfo( $fullPath, PATHINFO_DIRNAME ) . '/' );

			$image->rotateImage( $angle, null, true );

			// Thumbnail:
			$image			=	new Image( $imageSoftware );
			$thumbPath		=	parse_url( $this->path( true ), PHP_URL_PATH );

			$image->setName( pathinfo( $thumbPath, PATHINFO_FILENAME ) );
			$image->setSource( $this->path( true ) );
			$image->setDestination( pathinfo( $thumbPath, PATHINFO_DIRNAME ) . '/' );

			$image->rotateImage( $angle, null, true );

			$this->cache();
		} catch ( Exception $e ) {
			$this->setError( $e->getMessage() );

			return false;
		}

		return true;
	}

	/**
	 * Previews the item
	 *
	 * @param bool $thumbnail
	 */
	public function preview( $thumbnail = false )
	{
		$this->output( true, $thumbnail );
	}

	/**
	 * Downloads the item
	 */
	public function download()
	{
		$this->output();
	}

	/**
	 * Outputs item to header
	 *
	 * @param bool $inline
	 * @param bool $thumbnail
	 */
	private function output( $inline = false, $thumbnail = false )
	{
		if ( ! $this->getInt( 'id', 0 ) ) {
			header( 'HTTP/1.0 404 Not Found' );
			exit();
		}

		$filePath					=	$this->path( $thumbnail );

		if ( $this->domain( $thumbnail ) ) {
			cbRedirect( $filePath );
		}

		if ( ! $this->exists( $thumbnail ) ) {
			header( 'HTTP/1.0 404 Not Found' );
			exit();
		}

		$fileExtension				=	$this->extension( $thumbnail );

		if ( ! $fileExtension ) {
			header( 'HTTP/1.0 406 Not Acceptable' );
			exit();
		}

		$fileName					=	$this->name( $thumbnail );

		if ( ! $fileName ) {
			header( 'HTTP/1.0 404 Not Found' );
			exit();
		}

		$fileModifedTime			=	$this->modified( $thumbnail );
		$fileModifedDate			=	Application::Date( $fileModifedTime, 'UTC' )->format( 'r', true, false );
		$fileEtag					=	md5_file( $filePath );

		if ( ! Application::Application()->isClient( 'administrator' ) ) {
			if ( ( isset( $_SERVER['HTTP_IF_MODIFIED_SINCE'] ) && ( strtotime( $_SERVER['HTTP_IF_MODIFIED_SINCE'] ) === $fileModifedTime ) )
				 || ( isset( $_SERVER['HTTP_IF_NONE_MATCH'] ) && ( trim( $_SERVER['HTTP_IF_NONE_MATCH'] ) === $fileEtag ) )
			) {
				header( 'HTTP/1.1 304 Not Modified' );
				exit();
			}
		}

		$fileMime					=	$this->mimeType( $thumbnail );
		$fileSize					=	@filesize( $filePath );

		/** @noinspection PhpStatementHasEmptyBodyInspection */
		/** @noinspection LoopWhichDoesNotLoopInspection */
		/** @noinspection MissingOrEmptyGroupStatementInspection */
		while ( @ob_end_clean() ) {}

		if ( ini_get( 'zlib.output_compression' ) ) {
			ini_set( 'zlib.output_compression', 'Off' );
		}

		if ( function_exists( 'apache_setenv' ) ) {
			apache_setenv( 'no-gzip', '1' );
		}

		header( "Content-Type: $fileMime" );
		header( 'Content-Disposition: ' . ( $inline ? 'inline' : 'attachment' ) . '; modification-date="' . $fileModifedDate . '"; size=' . $fileSize . '; filename="' . $fileName . '";' );
		header( 'Content-Transfer-Encoding: binary' );
		header( 'Pragma: private' );
		header( 'Cache-Control: private' );
		header( "Last-Modified: $fileModifedDate" );
		header( "Etag: $fileEtag" );
		header( 'Accept-Ranges: bytes' );

		$start						=	0;
		$end						=	( $fileSize - 1 );
		$length						=	$fileSize;

		$isLarge					=	( $fileSize >= 524288 ); // Larger Than or Equal To 512kb
		$isRange					=	false;

		if ( isset( $_SERVER['HTTP_RANGE'] ) ) {
			if ( ! preg_match( '/^bytes=\d*-\d*(,\d*-\d*)*$/i', $_SERVER['HTTP_RANGE'] ) ) {
				header( 'HTTP/1.1 416 Requested Range Not Satisfiable' );
				header( "Content-Range: bytes */$fileSize" );
				exit();
			}

			list( , $range )		=	explode( '=', $_SERVER['HTTP_RANGE'], 2 );

			if ( strpos( $range, ',' ) !== false ) {
				header('HTTP/1.1 416 Requested Range Not Satisfiable');
				header("Content-Range: bytes $start-$end/$fileSize");
				exit;
			}

			if ( $range === '-' ) {
				$rangeStart			=	( $fileSize - substr( $range, 1 ) );
				$rangeEnd			=	$end;
			} else {
				$range				=	explode( '-', $range );
				$rangeStart			=	$range[0];
				$rangeEnd			=	( ( isset( $range[1] ) && is_numeric( $range[1] ) ) ? $range[1] : $fileSize );
			}

			$rangeEnd				=	( ( $rangeEnd > $end ) ? $end : $rangeEnd );

			if ( ( $rangeStart > $rangeEnd ) || ( $rangeStart > ( $fileSize - 1 ) ) || ( $rangeEnd >= $fileSize ) ) {
				header( 'HTTP/1.1 416 Requested Range Not Satisfiable' );
				header( "Content-Range: bytes $start-$end/$fileSize" );
				exit;
			}

			$start					=	$rangeStart;
			$end					=	$rangeEnd;
			$length					=	( ( $end - $start ) + 1 );

			header( 'HTTP/1.1 206 Partial Content' );
			header( "Content-Range: bytes $start-$end/$fileSize" );
			header( "Content-Length: $length" );

			$isRange				=	true;
		} else {
			header( 'HTTP/1.0 200 OK' );
			header( "Content-Length: $fileSize" );
		}

		/** @noinspection DeprecatedIniOptionsInspection */
		if ( ! ini_get( 'safe_mode' ) ) {
			@set_time_limit( 0 );
		}

		if ( $isLarge || $isRange ) {
			$file					=	fopen( $filePath, 'rb' );

			if ( $file === false ) {
				header( 'HTTP/1.0 404 Not Found' );
				exit();
			}

			fseek( $file, $start );

			if ( $isLarge ) {
				$buffer				=	( 1024 * 8 );

				while ( ( ! feof( $file ) ) && ( ( $pos = ftell( $file ) ) <= $end ) ) {
					if ( ( $pos + $buffer ) > $end ) {
						$buffer		=	( ( $end - $pos ) + 1 );
					}

					echo fread( $file, $buffer );
					@ob_flush();
					flush();
				}
			} else {
				echo fread( $file, $length );
			}

			fclose( $file );
		} else {
			if ( readfile( $filePath ) === false ) {
				header( 'HTTP/1.0 404 Not Found' );
				exit();
			}
		}

		exit();
	}

	/**
	 * @param null|Gallery $gallery
	 * @return FolderTable
	 */
	public function folder( $gallery = null )
	{
		$id					=	$this->getInt( 'folder', 0 );

		if ( $gallery ) {
			return $gallery->folder( $id );
		}

		static $folders		=	array();

		if ( ! isset( $folders[$id] ) ) {
			$folder			=	new FolderTable();

			$folder->load( $id );

			$folders[$id]	=	$folder;
		}

		return $folders[$id];
	}

	/**
	 * Try to discover this items type
	 *
	 * @param null|Gallery $gallery
	 * @return string
	 */
	public function discoverType( $gallery = null )
	{
		if ( Application::Application()->isClient( 'administrator' ) ) {
			$input		=	Application::Input();
			$files		=	$input->getNamespaceRegistry( 'files' );
		} else {
			$input		=	$this->getRaw( '_input', new Registry() );
			$files		=	$this->getRaw( '_files', new Registry() );
		}

		$type			=	$this->getString( 'type' );
		$upload			=	$files->subTree( 'upload' );

		if ( $upload->getString( 'name' ) ) {
			return CBGallery::getExtensionType( CBGallery::getUploadExtension( $upload ), $gallery, 'upload', false );
		}

		$uploadData		=	$input->getString( 'upload_image_data' );

		if ( $uploadData ) {
			if ( preg_match( '%^data:(image/[A-Za-z]+);base64,.+%', $uploadData, $matches ) ) {
				return CBGallery::getExtensionType( CBGallery::getExtensionFromMimeType( $matches[1] ), $gallery, 'upload', false );
			}

			return null;
		}

		$value			=	$input->getString( 'value' );

		if ( $value && ( ( $value !== $this->getString( 'value' ) ) || ( ! $type ) ) ) {
			$domain		=	preg_replace( '/^(?:(?:\w+\.)*)?(\w+)\..+$/', '\1', parse_url( $value, PHP_URL_HOST ) );

			if ( \in_array( $domain, [ 'youtube', 'youtu', 'vimeo', 'facebook' ], true ) ) {
				return 'videos';
			}

			$link		=	CBGallery::parseUrl( $value );

			if ( $link['extension'] && ( ( $link['extension'] !== $this->extension() ) || ( ! $type ) ) ) {
				return CBGallery::getExtensionType( $link['extension'], $gallery, 'link', false );
			}
		}

		return $type;
	}

	/**
	 * Returns items thumbnail
	 *
	 * @param Gallery  $gallery
	 * @param int|null $tabIndex
	 * @return string
	 */
	public function thumbnail( $gallery = null, $tabIndex = null )
	{
		global $_CB_framework;

		$type					=	$this->getString( 'type' );

		if ( $gallery ) {
			$thumbnailPath		=	$_CB_framework->pluginClassUrl( 'cbgallery', false, [ 'action' => 'item', 'func' => 'preview', 'id' => $this->getInt( 'id', 0 ), 'gallery' => $gallery->id() ], 'raw', 0, true );
		} else {
			$thumbnailPath		=	$this->path( true );
		}

		if ( $this->domain( true ) ) {
			$previewPath		=	$this->path( true );

			if ( $this->params()->getString( 'link_thumbnail' ) && ( \in_array( $this->domain(), [ 'vimeo', 'facebook' ], true ) ) ) {
				$thumbnailPath	=	$previewPath;
			}
		} else {
			$previewPath		=	$thumbnailPath;
		}

		$title					=	( $this->getString( 'title' ) ?: $this->name() );
		$orientation			=	null;
		$hasThumbnail			=	( $this->getString( 'thumbnail' ) || ( $this->domain() && $this->params()->getString( 'link_thumbnail' ) ) );

		if ( ( $type === 'photos' ) || ( ( $type === 'files' ) && ( CBGallery::getExtensionType( $this->extension() ) === 'photos' ) ) || $hasThumbnail ) {
			if ( $this->height( true, true ) > $this->width( true, true ) ) {
				$orientation	=	' w-100 mw-100 mh-none galleryImagePortrait';
			} elseif ( $this->width( true, true ) > $this->height( true, true ) ) {
				$orientation	=	' h-100 mh-100 mw-none galleryImageLandscape';
			} else {
				$orientation	=	' h-100 w-100 mh-100 mw-100 galleryImageSquare';
			}
		}

		if ( ( $type === 'photos' )
			 || ( ( $type === 'files' ) && ( ( $this->extension() === 'svg' ) || ( CBGallery::getExtensionType( $this->extension() ) === 'photos' ) ) )
		) {
			return '<img alt="' . htmlspecialchars( $title ) . '" src="' . htmlspecialchars( $previewPath ) . '" loading="lazy" class="flex-auto' . $orientation . ' cbImgPict cbThumbPict galleryImage"' . ( $tabIndex ? ' tabindex="' . htmlspecialchars( $tabIndex ) . '"' : null ) . ' />';
		}

		if ( $hasThumbnail ) {
			return '<img alt="' . htmlspecialchars( $title ) . '" src="' . htmlspecialchars( $thumbnailPath ) . '" loading="lazy" class="flex-auto' . $orientation . ' cbImgPict cbThumbPict galleryImage"' . ( $tabIndex ? ' tabindex="' . htmlspecialchars( $tabIndex ) . '"' : null ) . ' />';
		}

		if ( ( ( $type === 'videos' ) || ( ( $type === 'files' ) && ( CBGallery::getExtensionType( $this->extension() ) === 'videos' ) ) ) && ( ! \in_array( $this->domain(), [ 'vimeo', 'facebook' ], true ) ) ) {
			if ( \in_array( $this->domain(), [ 'youtube', 'youtu' ], true ) ) {
				return '<img alt="' . htmlspecialchars( $title ) . '" src="' . htmlspecialchars( $previewPath ) . '" loading="lazy" class="flex-auto h-100 mh-100 mw-none galleryImageLandscape cbImgPict cbThumbPict galleryImage"' . ( $tabIndex ? ' tabindex="' . htmlspecialchars( $tabIndex ) . '"' : null ) . ' />';
			}

			return '<video src="' . htmlspecialchars( $previewPath ) . '" type="' . htmlspecialchars( $this->mimeType() ) . '" preload="auto" class="flex-auto h-100 w-auto mh-100 mw-none galleryVideoPlayer"' . ( $tabIndex ? ' tabindex="' . htmlspecialchars( $tabIndex ) . '"' : null ) . '></video>';
		}

		if ( ( $type === 'files' ) || ( \in_array( $this->domain(), [ 'vimeo', 'facebook' ], true ) ) ) {
			return '<span class="display-1 align-middle galleryImageIcon"><span class="fa ' . CBGallery::getExtensionIcon( $this->extension() ) . '"></span></span>';
		}

		return '<span class="display-1 align-middle galleryImageIcon"><span class="fa ' . CBGallery::getTypeIcon( $type ) . '"></span></span>';
	}

	/**
	 * Returns formatted media description
	 *
	 * @return string
	 */
	public function getDescription()
	{
		$description	=	$this->getString( 'description' );

		// Remove duplicate spaces:
		$description	=	preg_replace( '/ {2,}/i', ' ', $description );

		// Remove duplicate tabs:
		$description	=	preg_replace( '/\t{2,}/i', "\t", $description );

		// Remove duplicate linebreaks:
		$description	=	preg_replace( '/((?:\r\n|\r|\n){2})(?:\r\n|\r|\n)*/i', '$1', $description );

		// Convert line breaks to HTML breaks:
		$description	=	str_replace( [ "\r\n", "\r", "\n" ], '<br />', $description );

		return $description;
	}

	/**
	 * @param string $inputName
	 * @return Registry
	 */
	private function getUpload( $inputName = 'upload' )
	{
		global $_CB_framework;

		if ( Application::Application()->isClient( 'administrator' ) ) {
			$input						=	Application::Input();
			$files						=	$input->getNamespaceRegistry( 'files' );
		} else {
			$input						=	$this->getRaw( '_input', new Registry() );
			$files						=	$this->getRaw( '_files', new Registry() );
		}

		$uploadData						=	$input->getString( $inputName . '_image_data' );

		if ( $uploadData ) {
			static $cache				=	array();

			$id							=	md5( $uploadData );

			if ( ! isset( $cache[$id] ) ) {
				$upload					=	new Registry();

				if ( preg_match( '%^data:(image/[A-Za-z]+);base64,(.+)%', $uploadData, $matches ) ) {
					$name				=	md5( uniqid( mt_rand(), true ) );
					$tmpPath			=	$_CB_framework->getCfg( 'tmp_path' );
					$tmpName			=	null;
					$size				=	0;

					if ( ! is_dir( $tmpPath ) ) {
						$error			=	UPLOAD_ERR_NO_TMP_DIR;
					} else {
						$tmpFile		=	$tmpPath . '/' . $name . '.tmp';
						$error			=	UPLOAD_ERR_OK;

						if ( file_put_contents( $tmpFile, base64_decode( $matches[2] ) ) === false ) {
							$error		=	UPLOAD_ERR_NO_FILE;
						} else {
							$tmpName	=	$tmpFile;
							$size		=	@filesize( $tmpName );
						}
					}

					$uploadName			=	pathinfo( parse_url( $input->getString( $inputName . '_name' ), PHP_URL_PATH ), PATHINFO_FILENAME );

					if ( ! $uploadName ) {
						$uploadName		=	md5( $matches[2] );
					}

					$upload->load( array(	'name'		=>	$uploadName . '.' . CBGallery::getExtensionFromMimeType( $matches[1] ),
											'type'		=>	$matches[1],
											'tmp_name'	=>	$tmpName,
											'error'		=>	$error,
											'size'		=>	$size,
											'is_data'	=>	true
										) );
				}

				$cache[$id]				=	$upload;
			}

			return $cache[$id];
		}

		return $files->subTree( $inputName );
	}

	/**
	 * @param Registry $upload
	 * @return null|string
	 */
	private function getUploadError( $upload )
	{
		switch ( $upload->getString( 'error' ) ) {
			case 1:
			case 2:
				return CBTxt::T( 'The file is too large!' );
			case 3:
			case 6:
			case 7:
			case 8:
				return CBTxt::T( 'Failed to upload!' );
		}

		return null;
	}
}