<?php
/**
* Community Builder (TM)
* @version $Id: $
* @package CommunityBuilder
* @copyright (C) 2004-2021 www.joomlapolis.com / Lightning MultiCom SA - and its licensors, all rights reserved
* @license http://www.gnu.org/licenses/old-licenses/gpl-2.0.html GNU/GPL version 2
*/

use CBLib\Language\CBTxt;
use CB\Database\Table\UserTable;
use CB\Database\Table\TabTable;
use CBLib\Application\Application;
use CB\Plugin\Gallery\CBGallery;
use CB\Plugin\Gallery\Gallery;
use CB\Plugin\Gallery\GalleryInterface;
use CB\Plugin\Gallery\Table\FolderTable;
use CB\Plugin\Gallery\Table\ItemTable;
use CB\Database\Table\FieldTable;
use CBLib\Image\Image;
use CBLib\Registry\Registry;

if ( ! ( defined( '_VALID_CB' ) || defined( '_JEXEC' ) || defined( '_VALID_MOS' ) ) ) { die( 'Direct Access to this location is not allowed.' ); }

global $_PLUGINS;

$_PLUGINS->loadPluginGroup( 'user' );

class CBplug_cbgallery extends cbPluginHandler
{

	/**
	 * @param  TabTable   $tab       Current tab
	 * @param  UserTable  $user      Current user
	 * @param  int        $ui        1 front, 2 admin UI
	 * @param  array      $postdata  Raw unfiltred POST data
	 */
	public function getCBpluginComponent( $tab, $user, $ui, $postdata )
	{
		$this->getGallery();
	}

	/**
	 * Loads in a gallery directly or by URL
	 *
	 * @param null|Gallery $gallery
	 * @param null|string  $view
	 * @param int          $id
	 * @throws Exception
	 */
	public function getGallery( $gallery = null, $view = null, $id = 0 )
	{
		global $_CB_framework, $_PLUGINS;

		$viewer								=	CBuser::getMyUserDataInstance();
		$raw								=	false;
		$menu								=	null;
		$access								=	true;
		$output								=	null;

		if ( $gallery ) {
			if ( $gallery instanceof GalleryInterface ) {
				$action						=	'gallery';
			} else {
				return;
			}

			if ( $view === 'module' ) {
				$output						=	'module';
				$view						=	null;
			} elseif ( $view === 'compact' ) {
				$output						=	'compact';
				$view						=	null;
			}

			$function						=	( $view ? $view : 'show' );

			if ( strpos( $function, '.' ) !== false ) {
				list( $action, $function )	=	explode( '.', $function, 2 );
			}

			$inline							=	$gallery->getBool( 'inline', true );
		} else {
			$raw							=	( $this->getInput()->getString( 'format' ) === 'raw' );
			$action							=	$this->getInput()->getString( 'action' );

			if ( strpos( $action, '.' ) !== false ) {
				list( $action, $function )	=	explode( '.', $action, 2 );
			} else {
				$function					=	$this->getInput()->getString( 'func' );
			}

			$id								=	$this->getInput()->getInt( 'id', 0 );

			// TODO: For B/C: remove in next major release
			if ( ( $action === 'items' ) && ( ( ! $function ) || in_array( $function, array( 'preview', 'show' ), true ) ) ) {
				$action						=	'item';
			}

			$galleryId						=	$this->getInput()->getString( 'gallery' );
			$galleryAsset					=	null;

			if ( ! $galleryId ) {
				if ( $id && in_array( $action, array( 'item', 'folder' ), true ) ) {
					$galleryAsset			=	CBGallery::getAsset( $action, $id );
				}

				$menu						=	Application::Cms()->getActiveMenuWithParams();

				if ( $menu->getInt( 'id' )
					 && ( $menu->getString( 'query.option' ) === 'com_comprofiler' )
					 && ( $menu->getString( 'query.view' ) === 'pluginclass' )
					 && ( $menu->getString( 'query.plugin' ) === 'cbgallery' )
				) {
					if ( ! $galleryAsset ) {
						$galleryAsset		=	$menu->getString( 'params/gallery_asset' );
					}

					if ( ( $action === 'approval' ) && ( $menu->getString( 'query.action' ) !== 'approval' ) ) {
						// We're trying to reach the approval page, but the menu item isn't to an approval page so override the asset and turn off menu parsing:
						$galleryAsset		=	'all';
						$menu				=	null;
					}
				} else {
					$menu					=	null;
				}
			}

			if ( ( ! $galleryAsset ) && ( $action === 'approval' ) ) {
				// No asset found, but we're trying to reach approval page so override the asset:
				$galleryAsset				=	'all';
			}

			$gallery						=	new Gallery( $galleryAsset, $viewer );

			if ( $menu ) {
				$gallery->set( 'menu', $menu->getInt( 'id' ) );

				if ( $function === 'new' ) {
					$gallery->set( 'location', 'plugin' );
				} else {
					$gallery->set( 'location', JRoute::_( $menu->getString( 'link' ), false ) );
				}

				$gallery->parse( $menu->getNamespaceRegistry( 'params' ), 'gallery_' );
			}

			if ( $galleryId ) {
				if ( ! $gallery->load( $galleryId ) ) {
					if ( $id && in_array( $action, array( 'item', 'folder' ), true ) ) {
						$galleryAsset		=	CBGallery::getAsset( $action, $id );

						if ( $galleryAsset ) {
							$gallery->assets( $galleryAsset );
						} else {
							$access			=	false;
						}
					} else {
						$access				=	false;
					}
				}
			} elseif ( $function && ( ! in_array( $function, array( 'show', 'embed', 'new', 'preview', 'display', 'download' ), true ) ) ) {
				$access						=	false;
			}

			$inline							=	$gallery->getBool( 'inline', false );
		}

		if ( ! $gallery->asset() ) {
			$access							=	false;
		} elseif ( preg_match( '/^profile(?:\.(\d+)(?:\.field\.(\d+))?)?/', $gallery->asset(), $matches ) || $gallery->getInt( 'tab', 0 ) || $gallery->getInt( 'field', 0 ) ) {
			$profileId						=	( isset( $matches[1] ) ? (int) $matches[1] : $gallery->user()->getInt( 'id', 0 ) );
			$fieldId						=	( isset( $matches[2] ) ? (int) $matches[2] : $gallery->getInt( 'field', 0 ) );
			$tabId							=	$gallery->getInt( 'tab', 0 );

			if ( $profileId !== $gallery->user()->getInt( 'id', 0 ) ) {
				$gallery->user( $profileId );
			}

			if ( $fieldId ) {
				$field						=	CBGallery::getField( $fieldId, $profileId );

				if ( ! $field ) {
					$access					=	false;
				} else {
					// ALWAYS reload field params to prevent gallery going out of sync:
					$gallery->set( 'field', $field->getInt( 'fieldid', 0 ) );

					$gallery->parse( $field->params, 'gallery_', false );
				}
			} else {
				$tab						=	CBGallery::getTab( $tabId, $profileId );

				if ( ! $tab ) {
					if ( ! in_array( 'all', $gallery->assets(), true ) ) {
						$access				=	false;
					}
				} else {
					// ALWAYS reload tab params to prevent gallery going out of sync:
					$gallery->set( 'tab', $tab->getInt( 'tabid', 0 ) );

					$gallery->parse( $tab->params, 'gallery_', false );
				}
			}
		}

		$_PLUGINS->trigger( 'gallery_onGalleryAccess', array( &$gallery, &$access ) );

		if ( ! $access ) {
			if ( $inline ) {
				return;
			}

			if ( $raw ) {
				header( 'HTTP/1.0 401 Unauthorized' );
				exit();
			}

			CBGallery::returnRedirect( 'index.php', CBTxt::T( 'Not authorized.' ), 'error' );
		}

		if ( ! $gallery->id() ) {
			$gallery->cache();
		}

		if ( ! $raw ) {
			outputCbJs();
			outputCbTemplate();

			ob_start();
		}

		switch ( $action ) {
			case 'items':
				if ( $function === 'new' ) {
					$this->showItemsNew( $viewer, $gallery );
				}
				break;
			case 'item':
				switch ( $function ) {
					case 'download':
						$this->outputItem( 'download', $id, $viewer, $gallery );
						break;
					case 'edit':
						$this->showItemEdit( $id, $viewer, $gallery );
						break;
					case 'new':
						$this->showItemEdit( null, $viewer, $gallery );
						break;
					case 'save':
						cbSpoofCheck( 'plugin', ( $raw ? 'REQUEST' : 'POST' ) );
						$this->saveItemEdit( $id, $viewer, $gallery, ( $raw ? 'ajax' : null ) );
						break;
					case 'publish':
						$this->stateItem( 1, $id, $viewer, $gallery, ( $raw ? 'ajax' : null ) );
						break;
					case 'unpublish':
						$this->stateItem( 0, $id, $viewer, $gallery, ( $raw ? 'ajax' : null ) );
						break;
					case 'rotate':
						$this->rotateItem( $id, $viewer, $gallery );
						break;
					case 'avatar':
					case 'canvas':
						$this->saveItemField( $function, $id, $viewer, $gallery );
						break;
					case 'delete':
						$this->deleteItem( $id, $viewer, $gallery, ( $raw ? 'ajax' : null ) );
						break;
					case 'display':
						$this->showItem( $id, $viewer, $gallery );
						break;
					case 'embed':
						$this->showEmbed( $id, $viewer, $gallery );
						break;
					case 'preview':
						$this->outputItem( 'thumbnail', $id, $viewer, $gallery );
						break;
					case 'show':
					default:
						$this->outputItem( 'full', $id, $viewer, $gallery );
						break;
				}
				break;
			case 'folder':
				if ( ! $gallery->getBool( 'folders', true ) ) {
					CBGallery::returnRedirect( 'index.php', CBTxt::T( 'Not authorized.' ), 'error' );
				}

				switch ( $function ) {
					case 'cover':
						$this->saveFolderCover( $id, $viewer, $gallery );
						break;
					case 'edit':
						$this->showFolderEdit( $id, $viewer, $gallery );
						break;
					case 'new':
						$this->showFolderEdit( null, $viewer, $gallery );
						break;
					case 'save':
						cbSpoofCheck( 'plugin' );
						$this->saveFolderEdit( $id, $viewer, $gallery );
						break;
					case 'publish':
						$this->stateFolder( 1, $id, $viewer, $gallery );
						break;
					case 'unpublish':
						$this->stateFolder( 0, $id, $viewer, $gallery );
						break;
					case 'delete':
						$this->deleteFolder( $id, $viewer, $gallery );
						break;
					case 'show':
					default:
						$gallery->set( 'folder', $id );

						$this->showItems( $viewer, $gallery );
						break;
				}
				break;
			case 'gallery':
			case 'approval':
			default:
				$this->showItems( $viewer, $gallery, $inline, $output );
				break;
		}

		if ( ! $raw ) {
			$html							=	ob_get_clean();

			if ( ! $html ) {
				return;
			}

			if ( $inline ) {
				echo $html;
			} else {
				$class						=	$this->params->getString( 'general_class' );

				$return						=	'<' . ( $function === 'embed' ? 'span' : 'div' ) . ' class="cbGallery' . ( $class ? ' ' . htmlspecialchars( $class ) : null ) . '">'
											.		$html
											.	'</' . ( $function === 'embed' ? 'span' : 'div' ) . '>';

				echo $return;

				if ( $menu && $menu->getInt( 'id' ) ) {
					$_CB_framework->setMenuMeta();
				}
			}
		}
	}

	/**
	 * Outputs a JSON ajax response
	 *
	 * @param bool        $status
	 * @param null|string $mssage
	 */
	private function ajaxResponse( $status, $mssage = null )
	{
		header( 'HTTP/1.0 200 OK' );
		header( 'Content-Type: application/json' );

		/** @noinspection PhpStatementHasEmptyBodyInspection */
		/** @noinspection LoopWhichDoesNotLoopInspection */
		/** @noinspection MissingOrEmptyGroupStatementInspection */
		while ( @ob_end_clean() ) {}

		echo json_encode( array( 'status' => $status, 'message' => $mssage ) );

		exit();
	}

	/**
	 * Displays items gallery page
	 *
	 * @param UserTable $viewer
	 * @param Gallery   $gallery
	 * @param bool      $inline
	 * @param string    $output
	 */
	public function showItems( $viewer, $gallery, $inline = false, $output = null )
	{
		global $_CB_framework;

		$canModerate						=	CBGallery::canModerate( $gallery );

		CBGallery::getTemplate( array( 'gallery', 'folder', 'folders', 'folder_container', 'items', 'item_container' ) );

		if ( ! $canModerate ) {
			$gallery->set( 'published', 1 );
		} elseif ( $this->getInput()->getString( 'action' ) === 'approval' ) {
			$output							=	'approval';

			$gallery->set( 'location', $_CB_framework->pluginClassUrl( 'cbgallery', false, array( 'action' => 'approval' ) ) );
			$gallery->set( 'published', -1 );
		}

		$folderId							=	$this->getInput()->getInt( 'folder', $gallery->getInt( 'folder', 0 ) );

		if ( $folderId && ( $gallery->getInt( 'published' ) !== -1 ) ) {
			$gallery->set( 'folder', $folderId );
		}

		$galleryPrefix						=	'gallery_' . $gallery->id() . '_' . ( $folderId ? $folderId . '_' : null ) . ( ( $gallery->getInt( 'published' ) === -1 ) ? 'approval_' : null );
		$gallerySearch						=	$_CB_framework->getUserStateFromRequest( $galleryPrefix . 'search{com_comprofiler}', $galleryPrefix . 'search' );
		$searching							=	false;

		if ( $gallerySearch !== '' ) {
			$searching						=	true;

			$gallery->set( 'search', $gallerySearch );
		}

		$folder								=	null;
		$folders							=	null;
		$foldersPageNav						=	null;
		$pagingPrefix						=	null;
		$input								=	array();

		// Folders:
		if ( $gallery->getBool( 'folders', true ) ) {
			if ( ! $gallery->getInt( 'folder', 0 ) ) {
				$gallery->set( 'folder', 0 );

				$foldersPrefix				=	$galleryPrefix . 'folders_';
				$foldersLimitstart			=	(int) $_CB_framework->getUserStateFromRequest( $foldersPrefix . 'limitstart{com_comprofiler}', $foldersPrefix . 'limitstart' );

				if ( $gallery->getBool( 'query', true ) ) {
					$foldersTotal			=	$gallery->folders( 'count' );
				} else {
					$foldersTotal			=	0;
				}

				if ( $foldersTotal ) {
					if ( $foldersTotal <= $foldersLimitstart ) {
						$foldersLimitstart	=	0;
					}

					$foldersPageNav			=	new cbPageNav( $foldersTotal, $foldersLimitstart, $gallery->getInt( 'folders_paging_limit', 0 ) );

					$foldersPageNav->setInputNamePrefix( $foldersPrefix );
					$foldersPageNav->setStaticLimit( true );
					$foldersPageNav->setBaseURL( $gallery->location() );

					$gallery->set( 'folders_paging_limitstart', $foldersPageNav->limitstart );

					if ( ( $foldersLimitstart === 0 ) && ( $gallery->getString( 'folders_orderby', 'date_desc' ) === 'random' ) ) {
						Application::Session()->subTree( 'cb.gallery.random.' . $gallery->id() )->set( 'folders', mt_rand() );
					}

					$folders				=	array();

					if ( $gallery->getBool( 'query', true ) ) {
						$folders			=	$gallery->folders();
					}

					$thumbnails				=	array();

					foreach ( $folders as $fld ) {
						$thumbnail			=	$fld->getInt( 'thumbnail', 0 );

						if ( $thumbnail ) {
							$thumbnails[]	=	$thumbnail;
						}
					}

					if ( $thumbnails ) {
						$gallery->reset()->setId( $thumbnails )->items();
					}
				}
			} else {
				$folder						=	$gallery->folder( $folderId );

				if ( ( ! $folder->getInt( 'id', 0 ) ) || ( ( ! $folder->getInt( 'published', 1 ) ) && ( ( $viewer->getInt( 'id', 0 ) !== $folder->getInt( 'user_id', 0 ) ) && ( ! $canModerate ) ) ) ) {
					if ( $inline ) {
						return;
					}

					CBGallery::returnRedirect( $gallery->location(), CBTxt::T( 'Not authorized.' ), 'error' );
				}

				$pagingPrefix				=	'folders_';
			}
		} elseif ( $folderId ) {
			if ( $inline ) {
				return;
			}

			CBGallery::returnRedirect( $gallery->location(), CBTxt::T( 'Not authorized.' ), 'error' );
		}

		// Items:
		$itemsPrefix						=	$galleryPrefix . 'items_';
		$itemsType							=	$_CB_framework->getUserStateFromRequest( $itemsPrefix . 'type{com_comprofiler}', $itemsPrefix . 'type' );

		if ( $itemsType ) {
			$searching						=	true;

			$gallery->set( 'type', $itemsType );
		}

		$itemsLimitstart					=	(int) $_CB_framework->getUserStateFromRequest( $itemsPrefix . 'limitstart{com_comprofiler}', $itemsPrefix . 'limitstart' );

		if ( $gallery->getBool( 'query', true ) ) {
			$itemsTotal						=	$gallery->items( 'count' );
		} else {
			$itemsTotal						=	0;
		}

		if ( $itemsTotal <= $itemsLimitstart ) {
			$itemsLimitstart				=	0;
		}

		$itemsPageNav						=	new cbPageNav( $itemsTotal, $itemsLimitstart, $gallery->getInt( $pagingPrefix . 'items_paging_limit', 16 ) );

		$itemsPageNav->setInputNamePrefix( $itemsPrefix );
		$itemsPageNav->setStaticLimit( true );
		$itemsPageNav->setBaseURL( $gallery->location() );

		$gallery->set( $pagingPrefix . 'items_paging_limitstart', $itemsPageNav->limitstart );

		if ( ( $itemsLimitstart === 0 ) && ( $gallery->getString( $pagingPrefix . 'items_orderby', 'date_desc' ) === 'random' ) ) {
			Application::Session()->set( 'cb.gallery.random.' . $gallery->id() . '.' . $pagingPrefix . 'items', mt_rand() );
		}

		$items								=	array();

		if ( $gallery->getBool( 'query', true ) && $itemsTotal ) {
			$items							=	$gallery->items();
		}

		$input['type']						=	null;

		$galleryTypes						=	$gallery->types();

		if ( count( $galleryTypes ) > 1 ) {
			$filterTypes					=	array();
			$filterTypes[]					=	moscomprofilerHTML::makeOption( 0, CBTxt::T( 'All' ) );

			foreach ( $galleryTypes as $filterType ) {
				$filterTypes[]				=	moscomprofilerHTML::makeOption( $filterType, CBGallery::translateType( $filterType, true ) );
			}

			$input['type']					=	moscomprofilerHTML::selectList( $filterTypes, $itemsPrefix . 'type', 'class="w-auto flex-grow-0 form-control gallerySearchType" aria-label="' . htmlspecialchars( CBTxt::T( 'Search Type' ) ) . '"', 'value', 'text', $itemsType, 0, false, false );

			if ( $folderId ) {
				$searchPlaceholder			=	CBTxt::T( 'Search Album...' );
			} else {
				$searchPlaceholder			=	CBTxt::T( 'Search Gallery...' );
			}
		} else {
			// CBTxt::T( 'SEARCH_TYPE', 'Search [type]...', array( '[type]' => CBGallery::translateType( $galleryTypes[0] ) ) )
			$searchPlaceholder				=	CBTxt::T( 'SEARCH_' . strtoupper( $galleryTypes[0] ) . ' SEARCH_TYPE', 'Search [type]...', array( '[type]' => CBGallery::translateType( $galleryTypes[0], true ) ) );
		}

		$input['search']					=	'<input type="text" name="' . htmlspecialchars( $galleryPrefix . 'search' ) . '" value="' . htmlspecialchars( $gallerySearch ) . '" placeholder="' . htmlspecialchars( $searchPlaceholder ) . '" class="gallerySearch form-control" />';

		HTML_cbgalleryGallery::showGallery( $folder, $folders, $foldersPageNav, $items, $itemsPageNav, $searching, $input, $viewer, $gallery, $this, $output );
	}

	/**
	 * Displays items new page
	 *
	 * @param UserTable $viewer
	 * @param Gallery   $gallery
	 */
	public function showItemsNew( $viewer, $gallery )
	{
		CBGallery::getTemplate( 'items_new' );

		$folderId		=	$this->getInput()->getInt( 'folder', $gallery->getInt( 'folder', 0 ) );

		if ( $folderId ) {
			$gallery->set( 'folder', $folderId );
		}

		if ( ! CBGallery::canCreateItems( 'all', 'both', $gallery ) ) {
			CBGallery::returnRedirect( $gallery->location(), CBTxt::T( 'Not authorized.' ), 'error' );
		}

		HTML_cbgalleryItemsNew::showItemsNew( $viewer, $gallery, $this );
	}

	/**
	 * Outputs the header for an item
	 *
	 * @param string    $output
	 * @param int       $id
	 * @param UserTable $viewer
	 * @param Gallery   $gallery
	 */
	private function outputItem( $output, $id, $viewer, $gallery )
	{
		$row	=	$gallery->item( $id );

		if ( ( ! $row->getInt( 'id', 0 ) ) || ( ( ! $row->getInt( 'published', 1 ) ) && ( ( $viewer->getInt( 'id', 0 ) !== $row->getInt( 'user_id', 0 ) ) && ( ! CBGallery::canModerate( $gallery ) ) ) ) ) {
			header( 'HTTP/1.0 404 Not Found' );
			exit();
		}

		$type	=	$row->getString( 'type' );

		if ( ( $output === 'download' ) && ( ( $type === 'files' ) || ( $gallery->getBool( $type . '_download', false ) ) ) ) {
			$row->download();
		} elseif ( $output === 'thumbnail' ) {
			$row->preview( true );
		} else {
			$row->preview();
		}
	}

	/**
	 * Displays the embed for an item
	 *
	 * @param int       $id
	 * @param UserTable $viewer
	 * @param Gallery   $gallery
	 */
	private function showEmbed( $id, $viewer, $gallery )
	{
		$row	=	$gallery->item( $id );

		if ( ( ! $row->getInt( 'id', 0 ) ) || ( ( ! $row->getInt( 'published', 1 ) ) && ( ( $viewer->getInt( 'id', 0 ) !== $row->getInt( 'user_id', 0 ) ) && ( ! CBGallery::canModerate( $gallery ) ) ) ) ) {
			return;
		}

		CBGallery::getTemplate( 'embed' );

		echo HTML_cbgalleryEmbed::showEmbed( $row, $viewer, $gallery, $this );
	}

	/**
	 * Displays item modal page
	 *
	 * @param int|ItemTable       $id
	 * @param UserTable $viewer
	 * @param Gallery   $gallery
	 */
	public function showItem( $id, $viewer, $gallery )
	{
		if ( $id instanceof ItemTable ) {
			$row	=	$id;
		} else {
			$row	=	$gallery->item( $id );
		}

		if ( ( ! $row->getInt( 'id', 0 ) ) || ( ( ! $row->getInt( 'published', 1 ) ) && ( ( $viewer->getInt( 'id', 0 ) !== $row->getInt( 'user_id', 0 ) ) && ( ! CBGallery::canModerate( $gallery ) ) ) ) ) {
			header( 'HTTP/1.0 404 Not Found' );
			exit();
		}

		header( 'HTTP/1.0 200 OK' );

		CBGallery::getTemplate( 'item', false, false );

		HTML_cbgalleryItem::showItem( $row, $viewer, $gallery, $this );

		exit();
	}

	/**
	 * Displays item create/edit page
	 *
	 * @param int|ItemTable $id
	 * @param UserTable     $viewer
	 * @param Gallery       $gallery
	 * @param string        $output
	 * @return null|string
	 */
	public function showItemEdit( $id, $viewer, $gallery, $output = null )
	{
		global $_CB_framework;

		CBGallery::getTemplate( ( $output ? array( 'item_edit_mini', 'item_container' ) : array( 'item_edit', 'embed', 'integration_edit' ) ), ( $output !== 'ajax' ), ( $output !== 'ajax' ) );

		if ( $id instanceof ItemTable ) {
			$row											=	$id;
		} else {
			$row											=	$gallery->item( $id );
			$folderId										=	$this->getInput()->getInt( 'folder', $gallery->getInt( 'folder', 0 ) );

			if ( $folderId ) {
				$row->set( 'folder', $folderId );
			}
		}

		$canModerate										=	CBGallery::canModerate( $gallery );

		if ( $row->getInt( 'id', 0 ) ) {
			if ( ( $viewer->getInt( 'id', 0 ) !== $row->getInt( 'user_id', 0 ) ) && ( ! $canModerate ) ) {
				if ( $output ) {
					if ( $output === 'folder' ) {
						return null;
					}

					header( 'HTTP/1.0 404 Not Found' );
					exit();
				}

				CBGallery::returnRedirect( $gallery->location(), CBTxt::T( 'Not authorized.' ), 'error' );
			}
		} elseif ( ! CBGallery::canCreateItems( 'all', 'both', $gallery ) ) {
			if ( $output ) {
				if ( $output === 'folder' ) {
					return null;
				}

				header( 'HTTP/1.0 404 Not Found' );
				exit();
			}

			CBGallery::returnRedirect( $gallery->location(), CBTxt::T( 'Not authorized.' ), 'error' );
		}

		$input												=	array();

		$publishedTooltip									=	cbTooltip( null, CBTxt::T( 'Select publish status of the file. If unpublished the file will not be visible to the public.' ), null, null, null, null, null, 'data-hascbtooltip="true"' );

		$input['published']									=	moscomprofilerHTML::yesnoButtonList( 'published', $publishedTooltip, $this->getInput()->getInt( 'post/published', $row->getInt( 'published', 1 ) ) );

		$titleTooltip										=	cbTooltip( null, CBTxt::T( 'Optionally input a title. If no title is provided the filename will be displayed as the title.' ), null, null, null, null, null, 'data-hascbtooltip="true"' );

		$input['title']										=	'<input type="text" id="title' . ( $output ? $row->getInt( 'id' ) : null ) . '" name="title" value="' . htmlspecialchars( $this->getInput()->getString( 'post/title', $row->getString( 'title' ) ) ) . '" class="' . ( $output ? 'w-100 ' : null ) . 'form-control" size="25"' . $titleTooltip . ' />';

		$listFolders										=	array();

		$folders											=	$gallery->reset()->folders();

		if ( $folders ) {
			$listFolders[]									=	moscomprofilerHTML::makeOption( 0, CBTxt::T( 'None' ) );

			/** @var FolderTable[] $folders */
			foreach ( $folders as $folder ) {
				$listFolders[]								=	moscomprofilerHTML::makeOption( $folder->getInt( 'id', 0 ), ( $folder->getString( 'title' ) ? $folder->getString( 'title' ) : cbFormatDate( $folder->getString( 'date' ), true, false, 'F j, Y', ' g:i A' ) ) );
			}

			$folderTooltip									=	cbTooltip( null, CBTxt::T( 'Select the album for this file.' ), null, null, null, null, null, 'data-hascbtooltip="true"' );

			$input['folder']								=	moscomprofilerHTML::selectList( $listFolders, 'folder' . ( $output ? $row->getInt( 'id' ) : null ), 'class="' . ( $output ? 'w-100 ' : null ) . 'form-control galleryCBSelect"' . $folderTooltip, 'value', 'text', $this->getInput()->getInt( 'post/folder', $row->getInt( 'folder', 0 ) ), 1, false, false );
		} else {
			$input['folder']								=	null;
		}

		$descriptionTooltip									=	cbTooltip( null, CBTxt::T( 'Optionally input a description.' ), null, null, null, null, null, 'data-hascbtooltip="true"' );

		$input['description']								=	'<textarea id="description' . ( $output ? $row->getInt( 'id' ) : null ) . '" name="description" class="' . ( $output ? 'h-100 w-100 ' : null ) . 'form-control" cols="80" rows="' . ( $output ? 3 : 8 ) . '"' . $descriptionTooltip . '>' . $this->getInput()->getString( 'post/description', $row->getString( 'description' ) ) . '</textarea>';

		$canUpload											=	CBGallery::canCreateItems( 'all', 'upload', $gallery );
		$canLink											=	CBGallery::canCreateItems( 'all', 'link', $gallery );

		if ( $row->getInt( 'id', 0 ) || ( $canUpload && $canLink ) ) {
			static $JS_LOADED_METHOD						=	0;

			if ( ! $JS_LOADED_METHOD++ ) {
				$js											=	"$( '#method' ).on( 'change', function() {"
															.		"var value = $( this ).val();"
															.		"if ( value == 1 ) {"
															.			"$( '#itemUpload' ).removeClass( 'hidden' ).find( 'input' ).removeClass( 'cbValidationDisabled' );"
															.			"$( '#itemLink' ).addClass( 'hidden' ).find( 'input' ).addClass( 'cbValidationDisabled' );"
															.		"} else if ( value == 2 ) {"
															.			"$( '#itemUpload' ).addClass( 'hidden' ).find( 'input' ).addClass( 'cbValidationDisabled' ).val( '' );"
															.			"$( '#itemLink' ).removeClass( 'hidden' ).find( 'input' ).removeClass( 'cbValidationDisabled' );"
															.		"} else {"
															.			"$( '#itemUpload' ).addClass( 'hidden' ).find( 'input' ).addClass( 'cbValidationDisabled' ).val( '' );"
															.			"$( '#itemLink' ).addClass( 'hidden' ).find( 'input' ).addClass( 'cbValidationDisabled' );"
															.		"}"
															.	"}).change();";

				if ( $gallery->getBool( 'photos_client_resize', true ) ) {
					$js										.=	"$( '.galleryItemForm' ).on( 'cbvalidate.success', function( e, cbvalidate, label, element ) {"
															.		"if ( ! $( element ).is( '#upload' ) ) {"
															.			"return;"
															.		"}"
															.		"if ( $( element ).siblings( '.cbValidationImageData' ).length ) {"
															.			"$( '#upload_name' ).val( $( element ).val().replace( /^(.*[\\\/])?/, '' ) );"
															.		"} else {"
															.			"$( '#upload_name' ).val( '' );"
															.		"}"
															.	"});";
				}

				$_CB_framework->outputCbJQuery( $js );
			}

			$listMethods									=	array();

			if ( $row->getInt( 'id', 0 ) ) {
				$listMethods[]								=	moscomprofilerHTML::makeOption( 0, CBTxt::T( 'No Change' ) );
			}

			if ( $canUpload ) {
				$listMethods[]								=	moscomprofilerHTML::makeOption( 1, CBTxt::T( 'Upload' ) );
			}

			if ( $canLink ) {
				$listMethods[]								=	moscomprofilerHTML::makeOption( 2, CBTxt::T( 'Link' ) );
			}

			$input['method']								=	moscomprofilerHTML::selectList( $listMethods, 'method', 'class="form-control"', 'value', 'text', $this->getInput()->getInt( 'post/method', 0 ), 1, false, false );
		} else {
			$input['method']								=	null;
		}

		if ( $canUpload ) {
			$minFileSizes									=	array(	$gallery->getInt( 'photos_min_size', 0 ),
																		$gallery->getInt( 'videos_min_size', 0 ),
																		$gallery->getInt( 'files_min_size', 0 ),
																		$gallery->getInt( 'music_min_size', 0 )
																	);

			sort( $minFileSizes, SORT_NUMERIC );

			$maxFileSizes									=	array(	$gallery->getInt( 'photos_max_size', 0 ),
																		$gallery->getInt( 'videos_max_size', 0 ),
																		$gallery->getInt( 'files_max_size', 0 ),
																		$gallery->getInt( 'music_max_size', 0 )
																	);

			rsort( $maxFileSizes, SORT_NUMERIC );

			$minFileSize									=	( ! in_array( 0, $minFileSizes, true ) ? $minFileSizes[0] : 0 );
			$maxFileSize									=	( ! in_array( 0, $maxFileSizes, true ) ? $maxFileSizes[0] : 0 );
			$phpMaxFileSize									=	CBGallery::getPHPMaxUpload( 'KB' );

			if ( ( ! $maxFileSize ) || ( $maxFileSize > $phpMaxFileSize ) ) {
				$maxFileSize								=	$phpMaxFileSize;
			}

			$uploadExtLimit									=	CBGallery::getExtensions( 'all', $gallery, 'client' );
			$fileValidation									=	array();

			if ( $minFileSize || $maxFileSize ) {
				$fileValidation[]							=	cbValidator::getRuleHtmlAttributes( 'filesize', array( $minFileSize, $maxFileSize, 'KB' ) );
			}

			if ( $uploadExtLimit ) {
				$fileValidation[]							=	cbValidator::getRuleHtmlAttributes( 'extension', implode( ',', $uploadExtLimit ) );
			}

			if ( $gallery->getBool( 'photos_client_resize', true ) ) {
				$imageHeight								=	$gallery->getInt( 'photos_image_height', 640 );
				$imageWidth									=	$gallery->getInt( 'photos_image_width', 1280 );

				if ( ! $imageHeight ) {
					$imageHeight							=	640;
				}

				if ( ! $imageWidth ) {
					$imageWidth								=	1280;
				}

				$fileValidation[]							=	cbValidator::getRuleHtmlAttributes( 'resize', array( $imageWidth, $imageHeight, $gallery->getInt( 'photos_maintain_aspect_ratio', 1 ), $gallery->getInt( 'photos_client_resize', 1 ) ) );
			}

			$fileTooltip									=	cbTooltip( null, CBTxt::T( 'Select the file to upload.' ), null, null, null, null, null, 'data-hascbtooltip="true"' );

			$input['upload']								=	'<input type="file" id="upload" name="upload" value="" class="form-control' . ( ! $row->getInt( 'id', 0 ) ? ' required' : null ) . '"' . $fileTooltip . implode( ' ', $fileValidation ) . ' />';

			if ( $gallery->getBool( 'photos_client_resize', true ) ) {
				$input['upload']							.=	'<input type="hidden" id="upload_name" name="upload_name" value="" />';
			}

			$input['upload_limits']							=	array();

			if ( $uploadExtLimit ) {
				$input['upload_limits'][]					=	CBTxt::T( 'FILE_MUST_BE_EXTS', 'Your file must be of [extensions] type.', array( '[extensions]' => implode( ', ', $uploadExtLimit ) ) );
			}

			if ( $minFileSize ) {
				$input['upload_limits'][]					=	CBTxt::T( 'FILE_SHOULD_EXCEED_SIZE', 'Your file should exceed [size].', array( '[size]' => CBGallery::getFormattedFileSize( $minFileSize * 1024 ) ) );
			}

			if ( $maxFileSize ) {
				$input['upload_limits'][]					=	CBTxt::T( 'FILE_SHOUND_NOT_EXCEED_SIZE', 'Your file should not exceed [size].', array( '[size]' => CBGallery::getFormattedFileSize( $maxFileSize * 1024 ) ) );
			}
		} else {
			$input['upload']								=	null;
			$input['upload_limits']							=	null;
		}

		if ( $canLink ) {
			$linkExtLimit									=	CBGallery::getExtensions( 'all', $gallery, 'link' );

			$linkTooltip									=	cbTooltip( null, CBTxt::T( 'Input the URL to the file to link.' ), null, null, null, null, null, 'data-hascbtooltip="true"' );

			$input['link']									=	'<input type="text" id="value" name="value" value="' . htmlspecialchars( $this->getInput()->getString( 'post/value', ( $row->domain() ? $row->getString( 'value' ) : null ) ) ) . '" size="40" class="form-control' . ( ! $row->getInt( 'id', 0 ) ? ' required' : null ) . '"' . $linkTooltip . ' />';

			$input['link_limits']							=	array();

			if ( $linkExtLimit ) {
				$input['link_limits'][]						=	CBTxt::T( 'LINK_MUST_BE_EXTS', 'Your file link must be of [extensions] type.', array( '[extensions]' => implode( ', ', $linkExtLimit ) ) );
			}
		} else {
			$input['link']									=	null;
			$input['link_limits']							=	null;
		}

		if ( ( ( ( $row->getString( 'type' ) === 'files' ) && ( ! ( ( $row->extension() === 'svg' ) || ( CBGallery::getExtensionType( $row->extension() ) === 'photos' ) ) ) ) || ( $row->getString( 'type' ) === 'music' ) )
			 && $gallery->getBool( 'thumbnails', true )
		) {
			$canUploadThumbnail								=	$gallery->getBool( 'thumbnails_upload', true );
			$canLinkThumbnail								=	$gallery->getBool( 'thumbnails_link', false );

			if ( $row->getString( 'thumbnail' ) || ( $row->domain() && $row->params()->getString( 'link_thumbnail' ) ) ) {
				$input['thumbnail_preview']					=	$row->thumbnail( $gallery );
			} else {
				$input['thumbnail_preview']					=	null;
			}

			if ( $row->getInt( 'id', 0 ) || ( $canUploadThumbnail && $canLinkThumbnail ) ) {
				static $JS_LOADED_THUMBNAIL					=	0;

				if ( ! $JS_LOADED_THUMBNAIL++ ) {
					$js										=	"$( '#thumbnail_method' ).on( 'change', function() {"
															.		"var value = $( this ).val();"
															.		"if ( value == 1 ) {"
															.			"$( '#itemThumbnailUpload' ).removeClass( 'hidden' ).find( 'input' ).removeClass( 'cbValidationDisabled' );"
															.			"$( '#itemThumbnailLink' ).addClass( 'hidden' ).find( 'input' ).addClass( 'cbValidationDisabled' );"
															.		"} else if ( value == 2 ) {"
															.			"$( '#itemThumbnailUpload' ).addClass( 'hidden' ).find( 'input' ).addClass( 'cbValidationDisabled' ).val( '' );"
															.			"$( '#itemThumbnailLink' ).removeClass( 'hidden' ).find( 'input' ).removeClass( 'cbValidationDisabled' );"
															.		"} else {"
															.			"$( '#itemThumbnailUpload' ).addClass( 'hidden' ).find( 'input' ).addClass( 'cbValidationDisabled' ).val( '' );"
															.			"$( '#itemThumbnailLink' ).addClass( 'hidden' ).find( 'input' ).addClass( 'cbValidationDisabled' );"
															.		"}"
															.	"}).change();";

					$_CB_framework->outputCbJQuery( $js );
				}

				$listThumbnailMethods						=	array();

				if ( $row->getInt( 'id', 0 ) ) {
					$listThumbnailMethods[]					=	moscomprofilerHTML::makeOption( 0, CBTxt::T( 'No Change' ) );
				}

				if ( $canUploadThumbnail ) {
					$listThumbnailMethods[]					=	moscomprofilerHTML::makeOption( 1, CBTxt::T( 'Upload' ) );
				}

				if ( $canLinkThumbnail ) {
					$listThumbnailMethods[]					=	moscomprofilerHTML::makeOption( 2, CBTxt::T( 'Link' ) );
				}

				if ( $row->getString( 'thumbnail' ) && ( ! $row->domain( true ) ) ) {
					$listThumbnailMethods[]					=	moscomprofilerHTML::makeOption( 3, CBTxt::T( 'Delete' ) );
				}

				$input['thumbnail_method']					=	moscomprofilerHTML::selectList( $listThumbnailMethods, 'thumbnail_method', 'class="form-control"', 'value', 'text', $this->getInput()->getInt( 'post/thumbnail_method', 0 ), 1, false, false );
			} else {
				$input['thumbnail_method']					=	null;
			}

			if ( $canUploadThumbnail ) {
				$thumbnailExtLimit							=	CBGallery::getExtensions( 'photos', $gallery, 'client' );
				$minThumbnailSize							=	$gallery->getInt( 'thumbnails_min_size', 0 );
				$maxThumbnailSize							=	$gallery->getInt( 'thumbnails_max_size', 1024 );

				$thumbnailValidation						=	array();

				if ( $minThumbnailSize || $maxThumbnailSize ) {
					$thumbnailValidation[]					=	cbValidator::getRuleHtmlAttributes( 'filesize', array( $minThumbnailSize, $maxThumbnailSize, 'KB' ) );
				}

				$thumbnailValidation[]						=	cbValidator::getRuleHtmlAttributes( 'extension', implode( ',', $thumbnailExtLimit ) );

				if ( $gallery->getBool( 'photos_client_resize', true ) ) {
					$thumbnailHeight						=	$gallery->getInt( 'thumbnails_image_height', 320 );
					$thumbnailWidth							=	$gallery->getInt( 'thumbnails_image_width', 640 );

					if ( ! $thumbnailHeight ) {
						$thumbnailHeight					=	320;
					}

					if ( ! $thumbnailWidth ) {
						$thumbnailWidth						=	640;
					}

					$thumbnailValidation[]					=	cbValidator::getRuleHtmlAttributes( 'resize', array( $thumbnailWidth, $thumbnailHeight, $gallery->getInt( 'thumbnails_resample', 1 ), $gallery->getInt( 'photos_client_resize', 1 ) ) );
				}

				$thumbnailTooltip							=	cbTooltip( null, CBTxt::T( 'Optionally select the thumbnail file to upload.' ), null, null, null, null, null, 'data-hascbtooltip="true"' );

				$input['thumbnail_upload']					=	'<input type="file" id="thumbnail_upload" name="thumbnail_upload" value="" class="form-control"' . $thumbnailTooltip . implode( ' ', $thumbnailValidation ) . ' />';

				$input['thumbnail_upload_limits']			=	array();
				$input['thumbnail_upload_limits'][]			=	CBTxt::T( 'THUMBNAIL_MUST_BE_EXTS', 'Your thumbnail file must be of [extensions] type.', array( '[extensions]' => implode( ', ', $thumbnailExtLimit ) ) );

				if ( $minThumbnailSize ) {
					$input['thumbnail_upload_limits'][]		=	CBTxt::T( 'THUMBNAIL_SHOULD_EXCEED_SIZE', 'Your thumbnail file should exceed [size].', array( '[size]' => CBGallery::getFormattedFileSize( $minThumbnailSize * 1024 ) ) );
				}

				if ( $maxThumbnailSize ) {
					$input['thumbnail_upload_limits'][]		=	CBTxt::T( 'THUMBNAIL_SHOUND_NOT_EXCEED_SIZE', 'Your thumbnail file should not exceed [size].', array( '[size]' => CBGallery::getFormattedFileSize( $maxThumbnailSize * 1024 ) ) );
				}
			} else {
				$input['thumbnail_upload']					=	null;
				$input['thumbnail_upload_limits']			=	null;
			}

			if ( $canLinkThumbnail ) {
				$linkExtLimit								=	CBGallery::getExtensions( 'photos', $gallery, 'link' );

				$linkThumbnailTooltip						=	cbTooltip( null, CBTxt::T( 'Optionally input the URL to the thumbnail file to link.' ), null, null, null, null, null, 'data-hascbtooltip="true"' );

				$input['thumbnail_link']					=	'<input type="text" id="thumbnail" name="thumbnail" value="' . htmlspecialchars( $this->getInput()->getString( 'post/thumbnail', ( $row->domain( true ) ? $row->getString( 'thumbnail' ) : null ) ) ) . '" size="40" class="form-control' . ( ! $row->getInt( 'id', 0 ) ? ' required' : null ) . '"' . $linkThumbnailTooltip . ' />';

				$input['thumbnail_link_limits']				=	array();

				if ( $linkExtLimit ) {
					$input['thumbnail_link_limits'][]		=	CBTxt::T( 'THUMBNAIL_LINK_MUST_BE_EXTS', 'Your thumbnail file link must be of [extensions] type.', array( '[extensions]' => implode( ', ', $linkExtLimit ) ) );
				}
			} else {
				$input['thumbnail_link']					=	null;
				$input['thumbnail_link_limits']				=	null;
			}
		} else {
			$input['thumbnail_preview']						=	null;
			$input['thumbnail_method']						=	null;
			$input['thumbnail_upload']						=	null;
			$input['thumbnail_upload_limits']				=	null;
			$input['thumbnail_link']						=	null;
			$input['thumbnail_link_limits']					=	null;
		}

		$ownerTooltip										=	cbTooltip( null, CBTxt::T( 'Input owner as single integer user_id.' ), null, null, null, null, null, 'data-hascbtooltip="true"' );

		$input['user_id']									=	'<input type="text" id="user_id" name="user_id" value="' . $this->getInput()->getInt( 'post/user_id', $row->getInt( 'user_id', $viewer->getInt( 'id', 0 ) ) ) . '" class="digits required form-control" size="6"' . $ownerTooltip . ' />';

		if ( $output ) {
			$return											=	HTML_cbgalleryItemEditMini::showItemEditMini( $row, $input, $viewer, $gallery, $this, $output );

			if ( $output === 'folder' ) {
				return $return;
			}

			echo $return;
		} else {
			HTML_cbgalleryItemEdit::showItemEdit( $row, $input, $viewer, $gallery, $this );
		}

		return null;
	}

	/**
	 * Saves an item
	 *
	 * @param int       $id
	 * @param UserTable $viewer
	 * @param Gallery   $gallery
	 * @param string    $output
	 */
	private function saveItemEdit( $id, $viewer, $gallery, $output = null )
	{
		global $_CB_framework, $_PLUGINS;

		$input							=	$this->getInput();
		$files							=	$input->getNamespaceRegistry( 'files' );

		$row							=	$gallery->item( $id );

		$row->set( '_input', $input );
		$row->set( '_files', $files );

		$folderId						=	$this->getInput()->getInt( 'folder', $gallery->getInt( 'folder', 0 ) );

		if ( $folderId ) {
			$row->set( 'folder', $folderId );
		}

		$canModerate					=	CBGallery::canModerate( $gallery );
		$type							=	$row->discoverType( $gallery );

		$upload							=	$files->subTree( 'upload' );
		$link							=	$this->getInput()->getString( 'value' );
		$method							=	null;

		if ( $upload->getString( 'name' ) ) {
			$method						=	'upload';
		} elseif ( $link && ( $link !== $row->getString( 'value' ) ) ) {
			$method						=	'link';
		}

		if ( $row->getInt( 'id', 0 ) ) {
			if ( ( $viewer->getInt( 'id', 0 ) !== $row->getInt( 'user_id', 0 ) ) && ( ! $canModerate ) ) {
				if ( $output === 'ajax' ) {
					$this->ajaxResponse( false, CBTxt::T( 'You do not have permission to edit this file.' ) );
				} else {
					CBGallery::returnRedirect( $gallery->location(), CBTxt::T( 'Not authorized.' ), 'error' );
				}
			}
		} elseif ( ! CBGallery::canCreateItems( $type, $method, $gallery ) ) {
			if ( $method ) {
				if ( CBGallery::createLimitedItems( $type, $gallery ) ) {
					if ( $method === 'upload' ) {
						// CBTxt::T( 'YOU_CANNOT_UPLOAD_ANYMORE_TYPES', 'You can not upload anymore [types]. You have reached your quota.', array( '[type]' => CBGallery::translateType( $row, false, true ) ) )
						$error			=	CBTxt::T( 'YOU_CANNOT_UPLOAD_ANYMORE_' . strtoupper( $type ) . ' YOU_CANNOT_UPLOAD_ANYMORE_TYPES', 'You can not upload anymore [types]. You have reached your quota.', array( '[types]' => CBGallery::translateType( $type, true, true ) ) );
					} else {
						// CBTxt::T( 'YOU_CANNOT_LINK_ANYMORE_TYPES', 'You can not link anymore [types]. You have reached your quota.', array( '[type]' => CBGallery::translateType( $row, false, true ) ) )
						$error			=	CBTxt::T( 'YOU_CANNOT_LINK_ANYMORE_' . strtoupper( $type ) . ' YOU_CANNOT_LINK_ANYMORE_TYPES', 'You can not link anymore [types]. You have reached your quota.', array( '[types]' => CBGallery::translateType( $type, true, true ) ) );
					}
				} elseif ( $method === 'upload' ) {
					$error				=	CBTxt::T( 'FILE_UPLOAD_INVALID_UPLOAD_ONLY_EXTS', 'Invalid file. Please upload only [extensions]!', array( '[extensions]' => implode( ', ', CBGallery::getExtensions( 'all', $gallery, $method ) ) ) );
				} else {
					$error				=	CBTxt::T( 'FILE_LINK_INVALID_LINK_ONLY_EXTS', 'Invalid file URL. Please link only [extensions]!', array( '[extensions]' => implode( ', ', CBGallery::getExtensions( 'all', $gallery, $method ) ) ) );
				}

				if ( $output === 'ajax' ) {
					$this->ajaxResponse( false, $error );
				} else {
					$_CB_framework->enqueueMessage( $error, 'error' );

					$this->showItemEdit( $id, $viewer, $gallery );
					return;
				}
			} elseif ( $output === 'ajax' ) {
				// CBTxt::T( 'NO_PERMISSION_TO_CREATE_TYPES', 'You do not have permission to create [types] in this gallery.', array( '[types]' => CBGallery::translateType( $type, true, true ) ) )
				$this->ajaxResponse( false, CBTxt::T( 'NO_PERMISSION_TO_CREATE_' . strtoupper( $type ) . ' NO_PERMISSION_TO_CREATE_TYPES', 'You do not have permission to create [types] in this gallery.', array( '[types]' => CBGallery::translateType( $type, true, true ) ) ) );
			} else {
				CBGallery::returnRedirect( $gallery->location(), CBTxt::T( 'Not authorized.' ), 'error' );
			}
		}

		if ( $canModerate || ( ! $gallery->getBool( $type . '_create_approval', false ) ) || ( $gallery->getBool( $type . '_create_approval', false ) && ( $row->getInt( 'published', 1 ) !== -1 ) && ( ! $method ) ) ) {
			$row->set( 'published', $this->getInput()->getInt( 'post/published', $row->getInt( 'published', 1 ) ) );
		} else {
			$row->set( 'published', ( $gallery->getBool( $type . '_create_approval', false ) ? -1 : $row->getInt( 'published', 1 ) ) );
		}

		$row->set( 'asset', $row->getString( 'asset', $gallery->asset() ) );
		$row->set( 'title', $this->getInput()->getString( 'post/title', $row->getString( 'title' ) ) );
		$row->set( 'description', $this->getInput()->getString( 'post/description', $row->getString( 'description' ) ) );

		if ( $method === 'link' ) {
			$row->set( 'value', $link );
		}

		if ( Application::MyUser()->isGlobalModerator() ) {
			$row->set( 'user_id', $this->getInput()->getInt( 'post/user_id', $row->getInt( 'user_id', $viewer->getInt( 'id', 0 ) ) ) );
		} else {
			$row->set( 'user_id', $row->getInt( 'user_id', $viewer->getInt( 'id', 0 ) ) );
		}

		if ( $output !== 'ajax' ) {
			if ( $gallery->getBool( 'items_create_captcha', false ) && ( ! $canModerate ) ) {
				$_PLUGINS->loadPluginGroup( 'user' );

				$_PLUGINS->trigger( 'onCheckCaptchaHtmlElements', array() );

				if ( $_PLUGINS->is_errors() ) {
					$row->setError( $_PLUGINS->getErrorMSG() );
				}
			}

			if ( ( ( ( $type === 'files' ) && ( ! ( ( $row->extension() === 'svg' ) || ( CBGallery::getExtensionType( $row->extension() ) === 'photos' ) ) ) ) || ( $type === 'music' ) )
				 && $gallery->getBool( 'thumbnails', true )
			) {
				$thumbnail				=	$row->getString( 'thumbnail' );

				if ( $files->subTree( 'thumbnail_upload' )->getString( 'name' ) ) {
					if ( ! $gallery->getBool( 'thumbnails_upload', true ) ) {
						CBGallery::returnRedirect( $gallery->location(), CBTxt::T( 'Custom thumbnail files are not allowed in this gallery.' ), 'error' );
					}
				} elseif ( $thumbnail && ( $thumbnail !== $row->getString( 'thumbnail' ) ) ) {
					if ( ! $gallery->getBool( 'thumbnails_link', false ) ) {
						CBGallery::returnRedirect( $gallery->location(), CBTxt::T( 'Custom thumbnail links are not allowed in this gallery.' ), 'error' );
					}
				}
			}
		}

		// Create an uploads folder on demand:
		if ( $folderId === -1 ) {
			$folder						=	new FolderTable();

			$folder->load( array( 'user_id' => $row->getInt( 'user_id', 0 ), 'asset' => 'profile.' . $row->getInt( 'user_id', 0 ) . '.uploads' ) );

			if ( $folder->getInt( 'id', 0 ) ) {
				$row->set( 'folder', $folder->getInt( 'id', 0 ) );
			} else {
				$folder->set( 'user_id', $row->getInt( 'user_id', 0 ) );
				$folder->set( 'asset', 'profile.' . $row->getInt( 'user_id', 0 ) . '.uploads' );
				$folder->set( 'title', CBTxt::T( 'Uploads' ) );
				$folder->set( 'published', 1 );

				if ( $folder->store() ) {
					$row->set( 'folder', $folder->getInt( 'id', 0 ) );
				}
			}
		}

		$new							=	( ! $row->getInt( 'id', 0 ) );
		$old							=	new ItemTable();
		$source							=	$row->source();

		if ( ! $new ) {
			$old->load( $row->getInt( 'id', 0 ) );

			$_PLUGINS->trigger( 'gallery_onBeforeUpdateGalleryItem', array( $gallery, $source, &$row, $old ) );
		} else {
			$_PLUGINS->trigger( 'gallery_onBeforeCreateGalleryItem', array( $gallery, $source, &$row ) );
		}

		if ( $row->getError() || ( ! $row->check( $gallery ) ) ) {
			// CBTxt::T( 'TYPE_FAILED_TO_SAVE', '[type] failed to save! Error: [error]', array( '[type]' => CBGallery::translateType( $row ), '[error]' => $row->getError() ) )
			$error						=	CBTxt::T( strtoupper( $type ) . '_FAILED_TO_SAVE TYPE_FAILED_TO_SAVE', '[type] failed to save! Error: [error]', array( '[type]' => CBGallery::translateType( $row ), '[error]' => $row->getError() ) );

			if ( $output === 'ajax' ) {
				$this->ajaxResponse( false, $error );
			} else {
				$_CB_framework->enqueueMessage( $error, 'error' );

				$this->showItemEdit( $id, $viewer, $gallery );
				return;
			}
		}

		if ( $row->getError() || ( ! $row->store( null, $gallery ) ) ) {
			// CBTxt::T( 'TYPE_FAILED_TO_SAVE', '[type] failed to save! Error: [error]', array( '[type]' => CBGallery::translateType( $row ), '[error]' => $row->getError() ) )
			$error					=	CBTxt::T( strtoupper( $type ) . '_FAILED_TO_SAVE TYPE_FAILED_TO_SAVE', '[type] failed to save! Error: [error]', array( '[type]' => CBGallery::translateType( $row ), '[error]' => $row->getError() ) );

			if ( $output === 'ajax' ) {
				$this->ajaxResponse( false, $error );
			} else {
				$_CB_framework->enqueueMessage( $error, 'error' );

				$this->showItemEdit( $id, $viewer, $gallery );
				return;
			}
		}

		if ( $row->getInt( 'folder', 0 ) ) {
			$folder					=	$row->folder();

			// Update the folder thumbnail if one doesn't exist or if we're using the uploads folder always update to latest upload:
			if ( $folder->getInt( 'id', 0 ) && ( ( ! $folder->getInt( 'thumbnail', 0 ) ) || ( $folderId === -1 ) ) ) {
				$folder->set( 'thumbnail', $row->getInt( 'id', 0 ) );

				$folder->store();
			}
		}

		if ( ! $new ) {
			$_PLUGINS->trigger( 'gallery_onAfterUpdateGalleryItem', array( $gallery, $source, $row, $old ) );
		} else {
			$_PLUGINS->trigger( 'gallery_onAfterCreateGalleryItem', array( $gallery, $source, $row ) );
		}

		$newParams					=	clone $row->params();

		$newParams->unsetEntry( 'overrides' );

		$row->set( 'params', $newParams->asJson() );

		if ( $row->getInt( 'published', 1 ) === -1 ) {
			if ( $new && ( ! $canModerate ) && $gallery->getBool( 'items_create_approval_notify', true ) ) {
				$cbUser				=	CBuser::getInstance( $row->getInt( 'user_id', 0 ), false );

				if ( $row->domain() ) {
					$itemUrl		=	htmlspecialchars( $row->path() );
				} else {
					$itemUrl		=	$_CB_framework->pluginClassUrl( $this->element, true, array( 'action' => 'item', 'func' => 'show', 'type' => $type, 'id' => $row->getInt( 'id', 0 ), 'gallery' => $gallery->id() ), 'raw', 0, true );
				}

				$extraStrings		=	array(	'item_id'			=>	$row->getInt( 'id', 0 ),
												'item_value'		=>	$row->getString( 'value' ),
												'item_file'			=>	$row->getString( 'file' ),
												'item_title'		=>	( $row->getString( 'title' ) ? $row->getString( 'title' ) : $row->name() ),
												'item_description'	=>	$row->getDescription(),
												'item_extension'	=>	$row->extension(),
												'item_size'			=>	$row->size(),
												'item_date'			=>	cbFormatDate( $row->getString( 'date' ) ),
												'item_folder'		=>	$row->getInt( 'folder', 0 ),
												'item_type'			=>	$row->getString( 'type' ),
												'item_url'			=>	$itemUrl,
												'gallery_location'	=>	htmlspecialchars( $gallery->location() ),
												'user_url'			=>	$_CB_framework->viewUrl( 'userprofile', true, array( 'user' => $row->getInt( 'user_id', 0 ) ) )
											);

				// CBTxt::T( 'GALLERY_NEW_TYPE_CREATED', 'Gallery - New [type] Created!', array( '[type]' => CBGallery::translateType( $row ) ) )
				$subject			=	$cbUser->replaceUserVars( CBTxt::T( 'GALLERY_NEW_' . strtoupper( $type ) . '_CREATED GALLERY_NEW_TYPE_CREATED', 'Gallery - New [type] Created!', array( '[type]' => CBGallery::translateType( $row ) ) ), false, true, $extraStrings, false );
				// CBTxt::T( 'TYPE_PENDING_APPROVAL', '<a href="[user_url]">[formatname]</a> created [type] <a href="[item_url]">[item_title]</a> and requires <a href="[gallery_location]">approval</a>!', array( '[type]' => CBGallery::translateType( $row, false, true ) ) )
				$message			=	$cbUser->replaceUserVars( CBTxt::T( strtoupper( $type ) . '_PENDING_APPROVAL TYPE_PENDING_APPROVAL', '<a href="[user_url]">[formatname]</a> created [type] <a href="[item_url]">[item_title]</a> and requires <a href="[gallery_location]">approval</a>!', array( '[type]' => CBGallery::translateType( $row, false, true ) ) ), false, true, $extraStrings, false );

				$notifications		=	new cbNotification();

				$recipients			=	$gallery->getRaw( 'moderators', array() );

				if ( $recipients ) {
					cbToArrayOfInt( $recipients );

					foreach ( $recipients as $recipient ) {
						$notifications->sendFromSystem( $recipient, $subject, $message, false, 1 );
					}
				} else {
					$notifications->sendToModerators( $subject, $message, false, 1 );
				}
			}

			if ( $output === 'ajax' ) {
				if ( preg_match( '/^(?:profile\.(\d+)\.uploads|pm\.(\d+))/i', $gallery->asset() ) ) {
					CBGallery::getTemplate( array( 'item_edit_micro', 'item_container' ), false, false );

					echo HTML_cbgalleryItemEditMicro::showItemEditMicro( $row, $viewer, $gallery, $this, $output );
					return;
				}

				$this->showItemEdit( $row->getInt( 'id', 0 ), $viewer, $gallery, $output );
				return;
			}

			// CBTxt::T( 'TYPE_SAVED_SUCCESSFULLY_AND_AWAITING_APPROVAL', '[type] saved successfully and awaiting approval!', array( '[type]' => CBGallery::translateType( $row ) ) )
			$message				=	CBTxt::T( strtoupper( $type ) . '_SAVED_SUCCESSFULLY_AND_AWAITING_APPROVAL TYPE_SAVED_SUCCESSFULLY_AND_AWAITING_APPROVAL', '[type] saved successfully and awaiting approval!', array( '[type]' => CBGallery::translateType( $row ) ) );

			CBGallery::returnRedirect( $gallery->location(), $message );
		} else {
			if ( $output === 'ajax' ) {
				if ( preg_match( '/^(?:profile\.(\d+)\.uploads|pm\.(\d+))/i', $gallery->asset() ) ) {
					CBGallery::getTemplate( array( 'item_edit_micro', 'item_container' ), false, false );

					echo HTML_cbgalleryItemEditMicro::showItemEditMicro( $row, $viewer, $gallery, $this, $output );
					return;
				}

				$this->showItemEdit( $row->getInt( 'id', 0 ), $viewer, $gallery, $output );
				return;
			}

			// CBTxt::T( 'TYPE_SAVED_SUCCESSFULLY', '[type] saved successfully!', array( '[type]' => CBGallery::translateType( $row ) ) )
			$message				=	CBTxt::T( strtoupper( $type ) . '_SAVED_SUCCESSFULLY TYPE_SAVED_SUCCESSFULLY', '[type] saved successfully!', array( '[type]' => CBGallery::translateType( $row ) ) );

			CBGallery::returnRedirect( $gallery->location(), $message );
		}
	}

	/**
	 * Rotates the image in the angle specified
	 *
	 * @param int       $id
	 * @param UserTable $viewer
	 * @param Gallery   $gallery
	 */
	private function rotateItem( $id, $viewer, $gallery )
	{
		global $_PLUGINS;

		$row			=	$gallery->item( $id );

		if ( ( ! $row->getInt( 'id', 0 ) )
			 || ( ( $viewer->getInt( 'id', 0 ) !== $row->getInt( 'user_id', 0 ) ) && ( ! CBGallery::canModerate( $gallery ) ) )
			 || ( $row->getString( 'type' ) !== 'photos' )
			 || $row->domain()
		) {
			$row->setError( CBTxt::T( 'Nothing to rotate.' ) );

			$this->showItem( $row, $viewer, $gallery );
		}

		if ( $this->getInput()->getString( 'direction' ) === 'left' ) {
			$angle		=	-90;
		} else {
			$angle		=	90;
		}

		$row->rotate( $angle );

		$source			=	$row->source();

		$_PLUGINS->trigger( 'gallery_onBeforeRotateGalleryItem', array( $gallery, $source, &$row ) );

		if ( $row->getError() || ( ! $row->store() ) ) {
			$row->setError( CBTxt::T( 'PHOTO_FAILED_TO_ROTATE', 'Photo failed to rotate! Error: [error]', array( '[error]' => $row->getError() ) ) );

			$this->showItem( $row, $viewer, $gallery );
		}

		$_PLUGINS->trigger( 'gallery_onAfterRotateGalleryItem', array( $gallery, $source, $row ) );

		$this->showItem( $row, $viewer, $gallery );
	}

	/**
	 * Saves an item to a field
	 *
	 * @param string    $fieldName
	 * @param int       $id
	 * @param UserTable $viewer
	 * @param Gallery   $gallery
	 */
	private function saveItemField( $fieldName, $id, $viewer, $gallery )
	{
		global $_PLUGINS, $_CB_framework;

		$row										=	$gallery->item( $id );

		if ( ( ! $row->getInt( 'id', 0 ) )
			 || ( $row->getInt( 'published', 0 ) === -1 )
			 || $viewer->getInt( 'id', 0 ) !== $row->getInt( 'user_id', 0 )
			 || ( $row->getString( 'type' ) !== 'photos' )
			 || ( ! in_array( $fieldName, array( 'avatar', 'canvas' ), true ) )
			 || ( ! $gallery->getBool( 'photos_' . $fieldName, false ) )
			 || $row->domain()
		) {
			$this->showItem( $row, $viewer, $gallery );
		}

		$field										=	new FieldTable();

		$field->load( array( 'name' => $fieldName ) );

		$field->set( 'params', new Registry( $field->getRaw( 'params' ) ) );

		$isModerator								=	Application::MyUser()->isModeratorFor( Application::User( $viewer->getInt( 'id', 0 ) ) );
		$path										=	$row->path();

		$_PLUGINS->trigger( 'onBeforeUserAvatarUpdate', array( &$viewer, &$viewer, $isModerator, &$path ) );

		if ( $_PLUGINS->is_errors() ) {
			$this->_setErrorMSG( $_PLUGINS->getErrorMSG() );
		}

		$conversionType								=	Application::Config()->getInt( 'conversiontype', 0 );
		$imageSoftware								=	( $conversionType === 5 ? 'gmagick' : ( $conversionType === 1 ? 'imagick' : ( $conversionType === 4 ? 'gd' : 'auto' ) ) );
		$imagePath									=	$_CB_framework->getCfg( 'absolute_path' ) . '/images/comprofiler/';

		$resize										=	$field->params->getInt( 'avatarResizeAlways', '' );

		if ( $resize === '' ) {
			$resize									=	Application::Config()->getInt( 'avatarResizeAlways', 1 );
		}

		$aspectRatio								=	$field->params->getInt( 'avatarMaintainRatio', ( $fieldName === 'avatar' ? 3 : '' ) );

		if ( $aspectRatio === '' ) {
			$aspectRatio							=	Application::Config()->getInt( 'avatarMaintainRatio', 1 );
		}

		$width										=	$field->params->getInt( 'avatarWidth', ( $fieldName === 'canvas' ? 1280 : ( $fieldName === 'avatar' ? 160 : '' ) ) );

		if ( $width === '' ) {
			$width									=	Application::Config()->getInt( 'avatarWidth', 200 );
		}

		$height										=	$field->params->getInt( 'avatarHeight', ( $fieldName === 'canvas' ? 640 : ( $fieldName === 'avatar' ? 160 : '' ) ) );

		if ( $height === '' ) {
			$height									=	Application::Config()->getInt( 'avatarHeight', 500 );
		}

		$thumbWidth									=	$field->params->getInt( 'thumbWidth', ( $fieldName === 'canvas' ? 640 : ( $fieldName === 'avatar' ? 80 : '' ) ) );

		if ( $thumbWidth === '' ) {
			$thumbWidth								=	Application::Config()->getInt( 'thumbWidth', 60 );
		}

		$thumbHeight								=	$field->params->getInt( 'thumbHeight', ( $fieldName === 'canvas' ? 320 : ( $fieldName === 'avatar' ? 80 : '' ) ) );

		if ( $thumbHeight === '' ) {
			$thumbHeight							=	Application::Config()->getInt( 'thumbHeight', 86 );
		}

		$fileName									=	( $fieldName === 'canvas' ? 'canvas_' : '' ) . uniqid( $viewer->getInt( 'id', 0 ) . '_' );
		$newFileName								=	null;

		try {
			$image									=	new Image( $imageSoftware, $resize, $aspectRatio );

			$image->setName( $fileName );
			$image->setSource( $path );
			$image->setDestination( $imagePath );
			$image->processImage( $width, $height );

			$newFileName							=	$image->getCleanFilename();

			$image->setName( 'tn' . $fileName );

			$image->processImage( $thumbWidth, $thumbHeight );
		} catch ( Exception $e ) {
			if ( $fileName === 'canvas' ) {
				$row->setError( CBTxt::T( 'PROFILE_CANVAS_FAILED_TO_UPDATE', 'Profile canvas failed to update! Error: [error]', array( '[error]' => $e->getMessage() ) ) );
			} else {
				$row->setError( CBTxt::T( 'PROFILE_AVATAR_FAILED_TO_UPDATE', 'Profile avatar failed to update! Error: [error]', array( '[error]' => $e->getMessage() ) ) );
			}

			$this->showItem( $row, $viewer, $gallery );
		}

		$_PLUGINS->trigger( 'onLogChange', array( 'update', 'user', 'field', &$viewer, &$this, &$field, $viewer->getString( $fieldName ), $newFileName, 'edit' ) );

		if ( $viewer->getString( $fieldName ) ) {
			deleteAvatar( $viewer->getString( $fieldName ) );
		}

		$newValues									=	array(	$fieldName				=>	$newFileName,
																$fieldName . 'approved'	=>	1
															);

		if ( $fieldName === 'canvas' ) {
			$newValues[$fieldName . 'position']		=	50;
		}

		$_PLUGINS->trigger( 'onAfterUserAvatarUpdate', array( &$viewer, &$viewer, $isModerator, $newFileName ) );

		if ( $viewer->getError() || ( ! $viewer->storeDatabaseValues( $newValues ) ) ) {
			deleteAvatar( $viewer->getString( $fieldName ) );

			if ( $fieldName === 'canvas' ) {
				$row->setError( CBTxt::T( 'PROFILE_CANVAS_FAILED_TO_UPDATE', 'Profile canvas failed to update! Error: [error]', array( '[error]' => $viewer->getError() ) ) );
			} else {
				$row->setError( CBTxt::T( 'PROFILE_AVATAR_FAILED_TO_UPDATE', 'Profile avatar failed to update! Error: [error]', array( '[error]' => $viewer->getError() ) ) );
			}

			$this->showItem( $row, $viewer, $gallery );
		}

		if ( $fieldName === 'canvas' ) {
			$row->set( '_success', CBTxt::T( 'Profile canvas successfully set to this photo!' ) );
		} else {
			$row->set( '_success', CBTxt::T( 'Profile avatar successfully set to this photo!' ) );
		}

		$this->showItem( $row, $viewer, $gallery );
	}

	/**
	 * Sets the published state of an item
	 *
	 * @param int       $state
	 * @param int       $id
	 * @param UserTable $viewer
	 * @param Gallery   $gallery
	 * @param string    $output
	 */
	private function stateItem( $state, $id, $viewer, $gallery, $output = null )
	{
		global $_PLUGINS;

		$row		=	$gallery->item( $id );
		$type		=	$row->getString( 'type' );

		if ( ( ! $row->getInt( 'id', 0 ) )
			 || ( ( $viewer->getInt( 'id', 0 ) !== $row->getInt( 'user_id', 0 ) ) && ( ! CBGallery::canModerate( $gallery ) ) )
			 || ( ( $viewer->getInt( 'id', 0 ) === $row->getInt( 'user_id', 0 ) ) && ( $row->getInt( 'published', 1 ) === -1 ) && $gallery->getBool( $type . '_create_approval', false ) ) ) {
			$row->setError( CBTxt::T( 'Not authorized.' ) );

			if ( $output === 'ajax' ) {
				$this->showItem( $row, $viewer, $gallery );
			} else {
				CBGallery::returnRedirect( $gallery->location(), $row->getError(), 'error' );
			}
		}

		$row->set( 'published', (int) $state );

		$source		=	$row->source();

		if ( $state ) {
			$_PLUGINS->trigger( 'gallery_onBeforePublishGalleryItem', array( $gallery, $source, &$row ) );
		} else {
			$_PLUGINS->trigger( 'gallery_onBeforeUnpublishGalleryItem', array( $gallery, $source, &$row ) );
		}

		if ( $row->getError() || ( ! $row->store() ) ) {
			// CBTxt::T( 'TYPE_STATE_FAILED_TO_SAVE', '[type] state failed to save! Error: [error]', array( '[type]' => CBGallery::translateType( $row ), '[error]' => $row->getError() ) )
			$row->setError( CBTxt::T( strtoupper( $type ) . '_STATE_FAILED_TO_SAVE TYPE_STATE_FAILED_TO_SAVE', '[type] state failed to save! Error: [error]', array( '[type]' => CBGallery::translateType( $row ), '[error]' => $row->getError() ) ) );

			if ( $output === 'ajax' ) {
				$this->showItem( $row, $viewer, $gallery );
			} else {
				CBGallery::returnRedirect( $gallery->location(), $row->getError(), 'error' );
			}
		}

		if ( $state ) {
			$_PLUGINS->trigger( 'gallery_onAfterPublishGalleryItem', array( $gallery, $source, $row ) );
		} else {
			$_PLUGINS->trigger( 'gallery_onAfterUnpublishGalleryItem', array( $gallery, $source, $row ) );
		}

		// CBTxt::T( 'TYPE_STATE_SAVED_SUCCESSFULLY', '[type] state saved successfully!', array( '[type]' => CBGallery::translateType( $row ) ) )
		$row->set( '_success', CBTxt::T( strtoupper( $type ) . '_STATE_SAVED_SUCCESSFULLY TYPE_STATE_SAVED_SUCCESSFULLY', '[type] state saved successfully!', array( '[type]' => CBGallery::translateType( $row ) ) ) );

		if ( $output === 'ajax' ) {
			$this->showItem( $row, $viewer, $gallery );
		} else {
			CBGallery::returnRedirect( $gallery->location(), $row->getRaw( '_success' ) );
		}
	}

	/**
	 * Deletes an item
	 *
	 * @param int       $id
	 * @param UserTable $viewer
	 * @param Gallery   $gallery
	 * @param string    $output
	 */
	private function deleteItem( $id, $viewer, $gallery, $output = null )
	{
		global $_PLUGINS;

		$row		=	$gallery->item( $id );
		$type		=	$row->getString( 'type' );

		if ( ( ! $row->getInt( 'id', 0 ) ) || ( ( $viewer->getInt( 'id', 0 ) !== $row->getInt( 'user_id', 0 ) ) && ( ! CBGallery::canModerate( $gallery ) ) ) ) {
			if ( $output === 'ajax' ) {
				$this->ajaxResponse( false, CBTxt::T( 'Nothing to delete.' ) );
			} else {
				CBGallery::returnRedirect( $gallery->location(), CBTxt::T( 'Not authorized.' ), 'error' );
			}
		}

		$source		=	$row->source();

		$_PLUGINS->trigger( 'gallery_onBeforeDeleteGalleryItem', array( $gallery, $source, &$row ) );

		if ( $row->getError() || ( ! $row->canDelete() ) ) {
			// CBTxt::T( 'TYPE_FAILED_TO_DELETE', '[type] failed to delete! Error: [error]', array( '[type]' => CBGallery::translateType( $row ), '[error]' => $row->getError() ) )
			$error	=	CBTxt::T( strtoupper( $type ) . '_FAILED_TO_DELETE TYPE_FAILED_TO_DELETE', '[type] failed to delete! Error: [error]', array( '[type]' => CBGallery::translateType( $row ), '[error]' => $row->getError() ) );

			if ( $output === 'ajax' ) {
				$this->ajaxResponse( false, $error );
			} else {
				CBGallery::returnRedirect( $gallery->location(), $error, 'error' );
			}
		}

		if ( $row->getError() || ( ! $row->delete() ) ) {
			// CBTxt::T( 'TYPE_FAILED_TO_DELETE', '[type] failed to delete! Error: [error]', array( '[type]' => CBGallery::translateType( $row ), '[error]' => $row->getError() ) )
			$error	=	CBTxt::T( strtoupper( $type ) . '_FAILED_TO_DELETE TYPE_FAILED_TO_DELETE', '[type] failed to delete! Error: [error]', array( '[type]' => CBGallery::translateType( $row ), '[error]' => $row->getError() ) );

			if ( $output === 'ajax' ) {
				$this->ajaxResponse( false, $error );
			} else {
				CBGallery::returnRedirect( $gallery->location(), $error, 'error' );
			}
		}

		$_PLUGINS->trigger( 'gallery_onAfterDeleteGalleryItem', array( $gallery, $source, $row ) );

		if ( $output === 'ajax' ) {
			$this->ajaxResponse( true );
		} else {
			// CBTxt::T( 'TYPE_DELETED_SUCCESSFULLY', '[type] deleted successfully!', array( '[type]' => CBGallery::translateType( $row ) ) )
			CBGallery::returnRedirect( $gallery->location(), CBTxt::T( strtoupper( $type ) . '_DELETED_SUCCESSFULLY TYPE_DELETED_SUCCESSFULLY', '[type] deleted successfully!', array( '[type]' => CBGallery::translateType( $row ) ) ) );
		}
	}

	/**
	 * Displays folder create/edit page
	 *
	 * @param int       $id
	 * @param UserTable $viewer
	 * @param Gallery   $gallery
	 */
	public function showFolderEdit( $id, $viewer, $gallery )
	{
		global $_CB_framework;

		CBGallery::getTemplate( array( 'folder_edit', 'folder_container', 'integration_edit' ) );

		$row							=	$gallery->folder( $id );
		$canModerate					=	CBGallery::canModerate( $gallery );

		if ( $row->getInt( 'id', 0 ) ) {
			if ( ( $viewer->getInt( 'id', 0 ) !== $row->getInt( 'user_id', 0 ) ) && ( ! $canModerate ) ) {
				CBGallery::returnRedirect( $gallery->location(), CBTxt::T( 'Not authorized.' ), 'error' );
			}
		} elseif ( ! CBGallery::canCreateFolders( $gallery ) ) {
			CBGallery::returnRedirect( $gallery->location(), CBTxt::T( 'Not authorized.' ), 'error' );
		}

		$input							=	array();

		$publishedTooltip				=	cbTooltip( null, CBTxt::T( 'Select publish status of the album. If unpublished the album will not be visible to the public.' ), null, null, null, null, null, 'data-hascbtooltip="true"' );

		$input['published']				=	moscomprofilerHTML::yesnoButtonList( 'published', $publishedTooltip, $this->getInput()->getInt( 'post/published', $row->getInt( 'published', 1 ) ) );

		$titleTooltip					=	cbTooltip( null, CBTxt::T( 'Optionally input a title. If no title is provided the date will be displayed as the title.' ), null, null, null, null, null, 'data-hascbtooltip="true"' );

		$input['title']					=	'<input type="text" id="title" name="title" value="' . htmlspecialchars( $this->getInput()->getString( 'post/title', $row->getString( 'title' ) ) ) . '" class="form-control" size="25"' . $titleTooltip . ' />';

		$descriptionTooltip				=	cbTooltip( null, CBTxt::T( 'Optionally input a description.' ), null, null, null, null, null, 'data-hascbtooltip="true"' );

		$input['description']			=	'<textarea id="description" name="description" class="form-control" cols="80" rows="8"' . $descriptionTooltip . '>' . $this->getInput()->getString( 'post/description', $row->getString( 'description' ) ) . '</textarea>';

		$ownerTooltip					=	cbTooltip( null, CBTxt::T( 'Input owner as single integer user_id.' ), null, null, null, null, null, 'data-hascbtooltip="true"' );

		$input['user_id']				=	'<input type="text" id="user_id" name="user_id" value="' . $this->getInput()->getInt( 'post/user_id', $row->getInt( 'user_id', $viewer->getInt( 'id', 0 ) ) ) . '" class="digits required form-control" size="6"' . $ownerTooltip . ' />';

		$items							=	array();

		if ( $row->getInt( 'id', 0 ) ) {
			$gallery->set( 'folder', $row->getInt( 'id', 0 ) );

			$itemsPrefix				=	'gallery_' . $gallery->id() . '_' . $row->getInt( 'id', 0 ) . '_edit_items_';
			$itemsLimitstart			=	(int) $_CB_framework->getUserStateFromRequest( $itemsPrefix . 'limitstart{com_comprofiler}', $itemsPrefix . 'limitstart' );

			if ( ! $canModerate ) {
				$gallery->set( 'user_id', $viewer->getInt( 'id', 0 ) );
			}

			$itemsTotal					=	$gallery->items( 'count' );

			if ( $itemsTotal <= $itemsLimitstart ) {
				$itemsLimitstart		=	0;
			}

			$itemsPageNav				=	new cbPageNav( $itemsTotal, $itemsLimitstart, $gallery->getInt( 'folders_items_paging_limit', 16 ) );

			$itemsPageNav->setInputNamePrefix( $itemsPrefix );
			$itemsPageNav->setStaticLimit( true );
			$itemsPageNav->setBaseURL( $_CB_framework->pluginClassUrl( $this->element, false, array( 'action' => 'folder', 'func' => 'edit', 'id' => $row->getInt( 'id', 0 ), 'gallery' => $gallery->id(), 'return' => CBGallery::getReturn( true ) ) ) );

			$gallery->set( 'folders_items_paging_limitstart', $itemsPageNav->limitstart );

			foreach ( $gallery->items() as $item ) {
				$items[]				=	$this->showItemEdit( $item, $viewer, $gallery, 'folder' );
			}
		} else {
			$itemsPageNav				=	new cbPageNav( 0, 0, 0 );
		}

		HTML_cbgalleryFolderEdit::showFolderEdit( $row, $items, $itemsPageNav, $input, $viewer, $gallery, $this );
	}

	/**
	 * Saves a folder
	 *
	 * @param int       $id
	 * @param UserTable $viewer
	 * @param Gallery   $gallery
	 */
	private function saveFolderEdit( $id, $viewer, $gallery )
	{
		global $_CB_framework, $_PLUGINS;

		$row						=	$gallery->folder( $id );
		$canModerate				=	CBGallery::canModerate( $gallery );

		if ( $row->getInt( 'id', 0 ) ) {
			if ( ( $viewer->getInt( 'id', 0 ) !== $row->getInt( 'user_id', 0 ) ) && ( ! $canModerate ) ) {
				CBGallery::returnRedirect( $gallery->location(), CBTxt::T( 'Not authorized.' ), 'error' );
			}
		} elseif ( ! CBGallery::canCreateFolders( $gallery ) ) {
			CBGallery::returnRedirect( $gallery->location(), CBTxt::T( 'Not authorized.' ), 'error' );
		}

		if ( $canModerate || ( ! $gallery->getBool( 'folders_create_approval', false ) ) || ( $row->getInt( 'id', 0 ) && ( $row->getInt( 'published', 1 ) !== -1 ) ) ) {
			$row->set( 'published', $this->getInput()->getInt( 'post/published', $row->getInt( 'published', 1 ) ) );
		} else {
			$row->set( 'published', ( $gallery->getBool( 'folders_create_approval', false ) ? -1 : $row->getInt( 'published', 1 ) ) );
		}

		$row->set( 'asset', $row->getString( 'asset', $gallery->asset() ) );
		$row->set( 'title', $this->getInput()->getString( 'post/title', $row->getString( 'title' ) ) );
		$row->set( 'description', $this->getInput()->getString( 'post/description', $row->getString( 'description' ) ) );

		if ( Application::MyUser()->isGlobalModerator() ) {
			$row->set( 'user_id', $this->getInput()->getInt( 'post/user_id', $row->getInt( 'user_id', $viewer->getInt( 'id', 0 ) ) ) );
		} else {
			$row->set( 'user_id', $row->getInt( 'user_id', $viewer->getInt( 'id', 0 ) ) );
		}

		if ( $gallery->getBool( 'folders_create_captcha', false ) && ( ! $canModerate ) ) {
			$_PLUGINS->loadPluginGroup( 'user' );

			$_PLUGINS->trigger( 'onCheckCaptchaHtmlElements', array() );

			if ( $_PLUGINS->is_errors() ) {
				$row->setError( $_PLUGINS->getErrorMSG() );
			}
		}

		$new						=	( ! $row->getInt( 'id', 0 ) );
		$old						=	new ItemTable();
		$source						=	$row->source();

		if ( ! $new ) {
			$old->load( $row->getInt( 'id', 0 ) );

			$_PLUGINS->trigger( 'gallery_onBeforeUpdateGalleryFolder', array( $gallery, $source, &$row, $old ) );
		} else {
			$_PLUGINS->trigger( 'gallery_onBeforeCreateGalleryFolder', array( $gallery, $source, &$row ) );
		}

		$newParams					=	clone $row->params();

		$newParams->unsetEntry( 'overrides' );

		$row->set( 'params', $newParams->asJson() );

		if ( $row->getError() || ( ! $row->check() ) ) {
			$_CB_framework->enqueueMessage( CBTxt::T( 'ALBUM_FAILED_TO_SAVE', 'Album failed to save! Error: [error]', array( '[error]' => $row->getError() ) ), 'error' );

			$this->showFolderEdit( $id, $viewer, $gallery );
			return;
		}

		if ( $row->getError() || ( ! $row->store() ) ) {
			$_CB_framework->enqueueMessage( CBTxt::T( 'ALBUM_FAILED_TO_SAVE', 'Album failed to save! Error: [error]', array( '[error]' => $row->getError() ) ), 'error' );

			$this->showFolderEdit( $id, $viewer, $gallery );
			return;
		}

		if ( ! $new ) {
			$_PLUGINS->trigger( 'gallery_onAfterUpdateGalleryFolder', array( $gallery, $source, $row, $old ) );
		} else {
			$_PLUGINS->trigger( 'gallery_onAfterCreateGalleryFolder', array( $gallery, $source, $row ) );
		}

		if ( $row->getInt( 'published', 1 ) === -1 ) {
			if ( $new && ( ! $canModerate ) && $gallery->getBool( 'folders_create_approval_notify', true ) ) {
				$cbUser				=	CBuser::getInstance( $row->getInt( 'user_id', 0 ), false );

				$extraStrings		=	array(	'folder_id'				=>	$row->getInt( 'id', 0 ),
												'folder_title'			=>	( $row->getString( 'title' ) ? $row->getString( 'title' ) : cbFormatDate( $row->getString( 'date' ), true, false, 'M j, Y', ' g:i A' ) ),
												'folder_description'	=>	$row->getString( 'description' ),
												'folder_date'			=>	$row->getString( 'date' ),
												'folder_url'			=>	$_CB_framework->pluginClassUrl( $this->element, true, array( 'action' => 'folder', 'func' => 'show', 'id' => $row->getInt( 'id', 0 ), 'gallery' => $gallery->id() ) ),
												'gallery_location'		=>	htmlspecialchars( $gallery->location() ),
												'user_url'				=>	$_CB_framework->viewUrl( 'userprofile', true, array( 'user' => $row->getInt( 'user_id', 0 ) ) )
											);

				$subject			=	$cbUser->replaceUserVars( CBTxt::T( 'Gallery - New Album Created!' ), false, true, $extraStrings, false );
				$message			=	$cbUser->replaceUserVars( CBTxt::T( 'ALBUM_PENDING_APPROVAL', '<a href="[user_url]">[formatname]</a> created album <a href="[folder_url]">[folder_title]</a> and requires approval!' ), false, true, $extraStrings, false );

				$notifications		=	new cbNotification();

				$recipients			=	$gallery->getRaw( 'moderators', array() );

				if ( $recipients ) {
					cbToArrayOfInt( $recipients );

					foreach ( $recipients as $recipient ) {
						$notifications->sendFromSystem( $recipient, $subject, $message, false, 1 );
					}
				} else {
					$notifications->sendToModerators( $subject, $message, false, 1 );
				}
			}

			CBGallery::returnRedirect( $gallery->location(), CBTxt::T( 'Album saved successfully and awaiting approval!' ) );
		} else {
			CBGallery::returnRedirect( $gallery->location(), CBTxt::T( 'Album saved successfully!' ) );
		}
	}

	/**
	 * Sets the folder cover to an item in the folder
	 *
	 * @param int       $id
	 * @param UserTable $viewer
	 * @param Gallery   $gallery
	 */
	private function saveFolderCover( $id, $viewer, $gallery )
	{
		$row	=	$gallery->folder( $id );

		if ( ( ! $row->getInt( 'id', 0 ) ) || ( ( $viewer->getInt( 'id', 0 ) !== $row->getInt( 'user_id', 0 ) ) && ( ! CBGallery::canModerate( $gallery ) ) ) ) {
			header( 'HTTP/1.0 404 Not Found' );
			exit();
		}

		$item	=	$gallery->item( $this->getInput()->getInt( 'item', 0 ) );

		if ( ( ! $item->getInt( 'id', 0 ) ) || ( $item->getInt( 'folder', 0 ) !== $row->getInt( 'id', 0 ) ) ) {
			header( 'HTTP/1.0 404 Not Found' );
			exit();
		}

		$row->set( 'thumbnail', $item->getInt( 'id', 0 ) );

		if ( $row->getError() || ( ! $row->store() ) ) {
			$item->setError( CBTxt::T( 'ALBUM_COVER_FAILED_TO_SAVE', 'Album cover failed to save! Error: [error]', array( '[error]' => $row->getError() ) ) );

			$this->showItem( $item, $viewer, $gallery );
		}

		$item->set( '_success', CBTxt::T( 'Album cover saved successfully!' ) );

		$this->showItem( $item, $viewer, $gallery );
	}

	/**
	 * Sets the published state of a folder
	 *
	 * @param int       $state
	 * @param int       $id
	 * @param UserTable $viewer
	 * @param Gallery   $gallery
	 */
	private function stateFolder( $state, $id, $viewer, $gallery )
	{
		global $_PLUGINS;

		$row		=	$gallery->folder( $id );

		if ( ( ! $row->getInt( 'id', 0 ) )
			 || ( ( $viewer->getInt( 'id', 0 ) !== $row->getInt( 'user_id', 0 ) ) && ( ! CBGallery::canModerate( $gallery ) ) )
			 || ( ( $viewer->getInt( 'id', 0 ) === $row->getInt( 'user_id', 0 ) ) && ( $row->getInt( 'published', 1 ) === -1 ) && $gallery->getBool( 'folders_create_approval', false ) ) ) {
			CBGallery::returnRedirect( $gallery->location(), CBTxt::T( 'Not authorized.' ), 'error' );
		}

		$row->set( 'published', (int) $state );

		$source		=	$row->source();

		if ( $state ) {
			$_PLUGINS->trigger( 'gallery_onBeforePublishGalleryFolder', array( $gallery, $source, &$row ) );
		} else {
			$_PLUGINS->trigger( 'gallery_onBeforeUnpublishGalleryFolder', array( $gallery, $source, &$row ) );
		}

		if ( $row->getError() || ( ! $row->store() ) ) {
			CBGallery::returnRedirect( $gallery->location(), CBTxt::T( 'ALBUM_STATE_FAILED_TO_SAVE', 'Album state failed to save! Error: [error]', array( '[error]' => $row->getError() ) ), 'error' );
		}

		if ( $state ) {
			$_PLUGINS->trigger( 'gallery_onAfterPublishGalleryFolder', array( $gallery, $source, $row ) );
		} else {
			$_PLUGINS->trigger( 'gallery_onAfterUnpublishGalleryFolder', array( $gallery, $source, $row ) );
		}

		CBGallery::returnRedirect( $gallery->location(), CBTxt::T( 'Album state saved successfully!' ) );
	}

	/**
	 * Deletes a folder
	 *
	 * @param int       $id
	 * @param UserTable $viewer
	 * @param Gallery   $gallery
	 */
	private function deleteFolder( $id, $viewer, $gallery )
	{
		global $_PLUGINS;

		$row		=	$gallery->folder( $id );

		if ( ( ! $row->getInt( 'id', 0 ) ) || ( ( $viewer->getInt( 'id', 0 ) !== $row->getInt( 'user_id', 0 ) ) && ( ! CBGallery::canModerate( $gallery ) ) ) ) {
			CBGallery::returnRedirect( $gallery->location(), CBTxt::T( 'Not authorized.' ), 'error' );
		}

		$source		=	$row->source();

		$_PLUGINS->trigger( 'gallery_onBeforeDeleteGalleryFolder', array( $gallery, $source, &$row ) );

		if ( $row->getError() || ( ! $row->canDelete() ) ) {
			CBGallery::returnRedirect( $gallery->location(), CBTxt::T( 'ALBUM_FAILED_TO_DELETE', 'Album failed to delete! Error: [error]', array( '[error]' => $row->getError() ) ), 'error' );
		}

		if ( $row->getError() || ( ! $row->delete() ) ) {
			CBGallery::returnRedirect( $gallery->location(), CBTxt::T( 'ALBUM_FAILED_TO_DELETE', 'Album failed to delete! Error: [error]', array( '[error]' => $row->getError() ) ), 'error' );
		}

		$_PLUGINS->trigger( 'gallery_onAfterDeleteGalleryFolder', array( $gallery, $source, $row ) );

		CBGallery::returnRedirect( $gallery->location(), CBTxt::T( 'Album deleted successfully!' ) );
	}
}