<?php
/**
* Community Builder (TM)
* @version $Id: $
* @package CommunityBuilder
* @copyright (C) 2004-2021 www.joomlapolis.com / Lightning MultiCom SA - and its licensors, all rights reserved
* @license http://www.gnu.org/licenses/old-licenses/gpl-2.0.html GNU/GPL version 2
*/

namespace CB\Plugin\AutoActions\Action;

use CB\Database\Table\UserTable;
use CB\Plugin\Gallery\CBGallery;
use CB\Plugin\Gallery\Gallery;
use CB\Plugin\Gallery\Table\FolderTable;
use CB\Plugin\Gallery\Table\ItemTable;
use CBLib\Language\CBTxt;
use CBLib\Registry\ParamsInterface;
use CBLib\Registry\Registry;

\defined( 'CBLIB' ) or die();

class GalleryAction extends Action
{

	/**
	 * @param UserTable $user
	 * @return string
	 */
	public function execute( UserTable $user ): string
	{
		global $_CB_database;

		if ( ! $this->installed() ) {
			$this->error( CBTxt::T( 'AUTO_ACTION_GALLERY_NOT_INSTALLED', ':: Action [action] :: CB Gallery is not installed', [ '[action]' => $this->autoaction()->getId() ] ) );
			return '';
		}

		$return								=	'';

		foreach ( $this->autoaction()->getParams()->subTree( 'gallery' ) as $row ) {
			/** @var ParamsInterface $row */
			$owner							=	$row->getString( 'owner', '' );

			if ( ! $owner ) {
				$owner						=	$user->getInt( 'id', 0 );
			} else {
				$owner						=	(int) $this->string( $user, $owner );
			}

			if ( $user->getInt( 'id', 0 ) !== $owner ) {
				$actionUser					=	\CBuser::getUserDataInstance( $owner );
			} else {
				$actionUser					=	$user;
			}

			$mode							=	$row->getString( 'mode', '' );
			$asset							=	$this->string( $actionUser, $row->getString( 'asset', '' ) );
			$value							=	$this->string( $actionUser, $row->getString( 'value', '' ) );

			if ( $mode === 'gallery' ) {
				$location					=	$this->string( $actionUser, $row->getString( 'location', '' ), false );

				if ( ! $location ) {
					$location				=	'current';
				}

				$gallery					=	new Gallery( $asset, $actionUser );

				$gallery->set( 'location', $location );
				$gallery->set( 'moderators', explode( ',', $this->string( $actionUser, $row->getString( 'moderators', '' ) ) ) );
				$gallery->set( 'notify', explode( ',', $this->string( $actionUser, $row->getString( 'notify', '' ) ) ) );
				$gallery->set( 'autoaction', $this->autoaction()->getId() );

				$gallery->parse( $row->subTree( 'gallery_gallery' ), 'gallery_' );

				if ( ( ! $gallery->folders( 'count' ) ) && ( ! $gallery->items( 'count' ) ) && ( ! CBGallery::canCreateFolders( $gallery ) ) && ( ! CBGallery::canCreateItems( 'all', 'both', $gallery ) ) ) {
					continue;
				}

				$return						.=	$gallery->gallery();
			} elseif ( $row->getString( 'method', 'create' ) === 'delete' ) {
				switch ( $mode ) {
					case 'folder':
						$table				=	'#__comprofiler_plugin_gallery_folders';
						$class				=	'\CB\Plugin\Gallery\Table\FolderTable';
						break;
					case 'item':
					default:
						$table				=	'#__comprofiler_plugin_gallery_items';
						$class				=	'\CB\Plugin\Gallery\Table\ItemTable';
						break;
				}

				if ( ! $asset ) {
					$asset					=	'profile.' . $actionUser->getInt( 'id', 0 );
				}

				$where						=	[];

				switch ( $row->getString( 'delete_by', 'asset' ) ) {
					case 'link':
						if ( ! $value ) {
							$this->error( CBTxt::T( 'AUTO_ACTION_GALLERY_NO_VALUE', ':: Action [action] :: CB Gallery skipped due to missing link', [ '[action]' => $this->autoaction()->getId() ] ) );
							continue 2;
						}

						$where[]			=	$_CB_database->NameQuote( 'value' ) . ' = ' . $_CB_database->Quote( $value );
						break;
					case 'asset':
						$where[]			=	$_CB_database->NameQuote( 'asset' ) . ( strpos( $asset, '%' ) !== false ? ' LIKE ' : ' = ' ) . $_CB_database->Quote( $asset );
						break;
					case 'owner':
						if ( ! $actionUser->getInt( 'id', 0 ) ) {
							$this->error( CBTxt::T( 'AUTO_ACTION_GALLERY_NO_OWNER', ':: Action [action] :: CB Gallery skipped due to missing owner', [ '[action]' => $this->autoaction()->getId() ] ) );
							continue 2;
						}

						$where[]			=	$_CB_database->NameQuote( 'user_id' ) . ' = ' . $actionUser->getInt( 'id', 0 );
						break;
					case 'asset_link':
						if ( ! $value ) {
							$this->error( CBTxt::T( 'AUTO_ACTION_GALLERY_NO_VALUE', ':: Action [action] :: CB Gallery skipped due to missing link', [ '[action]' => $this->autoaction()->getId() ] ) );
							continue 2;
						}

						$where[]			=	$_CB_database->NameQuote( 'asset' ) . ( strpos( $asset, '%' ) !== false ? ' LIKE ' : ' = ' ) . $_CB_database->Quote( $asset );
						$where[]			=	$_CB_database->NameQuote( 'value' ) . ' = ' . $_CB_database->Quote( $value );
						break;
					case 'asset_owner':
						if ( ! $actionUser->getInt( 'id', 0 ) ) {
							$this->error( CBTxt::T( 'AUTO_ACTION_GALLERY_NO_OWNER', ':: Action [action] :: CB Gallery skipped due to missing owner', [ '[action]' => $this->autoaction()->getId() ] ) );
							continue 2;
						}

						$where[]			=	$_CB_database->NameQuote( 'user_id' ) . ' = ' . $actionUser->getInt( 'id', 0 );
						$where[]			=	$_CB_database->NameQuote( 'asset' ) . ( strpos( $asset, '%' ) !== false ? ' LIKE ' : ' = ' ) . $_CB_database->Quote( $asset );
						break;
					case 'link_owner':
						if ( ! $actionUser->getInt( 'id', 0 ) ) {
							$this->error( CBTxt::T( 'AUTO_ACTION_GALLERY_NO_OWNER', ':: Action [action] :: CB Gallery skipped due to missing owner', [ '[action]' => $this->autoaction()->getId() ] ) );
							continue 2;
						}

						if ( ! $value ) {
							$this->error( CBTxt::T( 'AUTO_ACTION_GALLERY_NO_VALUE', ':: Action [action] :: CB Gallery skipped due to missing link', [ '[action]' => $this->autoaction()->getId() ] ) );
							continue 2;
						}

						$where[]			=	$_CB_database->NameQuote( 'user_id' ) . ' = ' . $actionUser->getInt( 'id', 0 );
						$where[]			=	$_CB_database->NameQuote( 'value' ) . ' = ' . $_CB_database->Quote( $value );
						break;
					case 'asset_link_owner':
					default:
						if ( ! $actionUser->getInt( 'id', 0 ) ) {
							$this->error( CBTxt::T( 'AUTO_ACTION_GALLERY_NO_OWNER', ':: Action [action] :: CB Gallery skipped due to missing owner', [ '[action]' => $this->autoaction()->getId() ] ) );
							continue 2;
						}

						if ( ! $value ) {
							$this->error( CBTxt::T( 'AUTO_ACTION_GALLERY_NO_VALUE', ':: Action [action] :: CB Gallery skipped due to missing link', [ '[action]' => $this->autoaction()->getId() ] ) );
							continue 2;
						}

						$where[]			=	$_CB_database->NameQuote( 'user_id' ) . ' = ' . $actionUser->getInt( 'id', 0 );
						$where[]			=	$_CB_database->NameQuote( 'asset' ) . ( strpos( $asset, '%' ) !== false ? ' LIKE ' : ' = ' ) . $_CB_database->Quote( $asset );
						$where[]			=	$_CB_database->NameQuote( 'value' ) . ' = ' . $_CB_database->Quote( $value );
						break;
				}

				$query						=	'SELECT *'
											.	"\n FROM " . $_CB_database->NameQuote( $table )
											.	"\n WHERE " . implode( "\n AND ", $where );
				$_CB_database->setQuery( $query );
				$objects					=	$_CB_database->loadObjectList( null, $class, [ $_CB_database ] );

				/** @var FolderTable[]|ItemTable[] $objects */
				foreach ( $objects as $object ) {
					$object->delete();
				}
			} else {
				if ( ! $actionUser->getInt( 'id', 0 ) ) {
					$this->error( CBTxt::T( 'AUTO_ACTION_GALLERY_NO_OWNER', ':: Action [action] :: CB Gallery skipped due to missing owner', [ '[action]' => $this->autoaction()->getId() ] ) );
					continue;
				}

				if ( ! $asset ) {
					$asset					=	'profile.' . $actionUser->getInt( 'id', 0 );
				}

				$title						=	$this->string( $actionUser, $row->getString( 'title', '' ) );
				$description				=	$this->string( $actionUser, $row->getString( 'description', '' ) );

				switch ( $mode ) {
					case 'folder':
						$object				=	new FolderTable();
						break;
					case 'item':
					default:
						$object				=	new ItemTable();
						break;
				}

				switch ( $row->getString( 'create_by', 'asset' ) ) {
					case 'link':
						$object->load( [ 'value' => $value ] );
						break;
					case 'asset':
						$object->load( [ 'asset' => $asset ] );
						break;
					case 'owner':
						$object->load( [ 'user_id' => $actionUser->getInt( 'id', 0 ) ] );
						break;
					case 'asset_link':
						$object->load( [ 'asset' => $asset, 'value' => $value ] );
						break;
					case 'asset_owner':
						$object->load( [ 'user_id' => $actionUser->getInt( 'id', 0 ), 'asset' => $asset ] );
						break;
					case 'link_owner':
						$object->load( [ 'user_id' => $actionUser->getInt( 'id', 0 ), 'value' => $value ] );
						break;
					case 'asset_link_owner':
						$object->load( [ 'user_id' => $actionUser->getInt( 'id', 0 ), 'asset' => $asset, 'value' => $value ] );
						break;
				}

				$object->set( 'asset', $asset );
				$object->set( 'user_id', $actionUser->getInt( 'id', 0 ) );

				if ( $title ) {
					$object->set( 'title', $title );
				}

				if ( $description ) {
					$object->set( 'description', $description );
				}

				$published					=	$row->getString( 'published', '' );

				if ( ( $published === null ) || ( $published === '' ) ) {
					$object->set( 'published', 1 );
				} else {
					$object->set( 'published', ( (int) $this->string( $actionUser, $published ) ? 1 : 0 ) );
				}

				if ( $mode === 'item' ) {
					if ( ! $value ) {
						$this->error( CBTxt::T( 'AUTO_ACTION_GALLERY_NO_VALUE', ':: Action [action] :: CB Gallery skipped due to missing value', [ '[action]' => $this->autoaction()->getId() ] ) );
						continue;
					}

					$object->set( '_input', new Registry( [ 'value' => $value ] ) );
					$object->set( 'value', $value );
				}

				if ( ! $object->store() ) {
					$this->error( CBTxt::T( 'AUTO_ACTION_GALLERY_FAILED', ':: Action [action] :: CB Gallery failed to save. Error: [error]', [ '[action]' => $this->autoaction()->getId(), '[error]' => $object->getError() ] ) );
				}
			}
		}

		return $return;
	}

	/**
	 * @return bool
	 */
	public function installed(): bool
	{
		global $_PLUGINS;

		$plugin				=	$_PLUGINS->getLoadedPlugin( 'user', 'cbgallery' );

		if ( ! $plugin ) {
			return false;
		}

		$pluginVersion		=	str_replace( '+build.', '+', $_PLUGINS->getPluginVersion( $plugin, true ) );

		return version_compare( $pluginVersion, '2.0.0', '>=' ) && version_compare( $pluginVersion, '3.0.0', '<' );
	}
}