<?php
/**
* Community Builder (TM)
* @version $Id: $
* @package CommunityBuilder
* @copyright (C) 2004-2021 www.joomlapolis.com / Lightning MultiCom SA - and its licensors, all rights reserved
* @license http://www.gnu.org/licenses/old-licenses/gpl-2.0.html GNU/GPL version 2
*/

namespace CB\Plugin\AutoActions\Action;

use CB\Database\Table\FieldTable;
use CB\Database\Table\UserTable;
use CBLib\Application\Application;
use CBLib\Language\CBTxt;
use CBLib\Registry\ParamsInterface;
use CBLib\Registry\Registry;

\defined( 'CBLIB' ) or die();

class FieldAction extends Action
{

	/**
	 * @param UserTable $user
	 */
	public function execute( UserTable $user ): void
	{
		global $_CB_database;

		/** @var FieldTable[] $fields */
		static $fields				=	[];

		foreach ( $this->autoaction()->getParams()->subTree( 'field' ) as $row ) {
			/** @var ParamsInterface $row */
			$userId					=	$row->getString( 'user', '' );

			if ( ! $userId ) {
				$userId				=	$user->getInt( 'id', 0 );
			} else {
				$userId				=	(int) $this->string( $user, $userId );
			}

			if ( $user->getInt( 'id', 0 ) !== $userId ) {
				$actionUser			=	\CBuser::getUserDataInstance( $userId );
			} else {
				$actionUser			=	$user;
			}

			if ( ! $actionUser->getInt( 'id', 0 ) ) {
				$this->error( CBTxt::T( 'AUTO_ACTION_FIELD_NO_USER', ':: Action [action] :: Field skipped due to missing user', [ '[action]' => $this->autoaction()->getId() ] ) );
				continue;
			}

			$fieldId				=	$row->getInt( 'field', 0 );

			if ( ! $fieldId ) {
				$this->error( CBTxt::T( 'AUTO_ACTION_FIELD_NO_FIELD', ':: Action [action] :: Field skipped due to missing field', [ '[action]' => $this->autoaction()->getId() ] ) );
				continue;
			}

			if ( ! isset( $fields[$fieldId] ) ) {
				$field				=	new FieldTable();

				$field->load( $fieldId );

				$field->set( 'params', new Registry( $field->getRaw( 'params' ) ) );

				$fields[$fieldId]	=	$field;
			}

			if ( ! $fields[$fieldId] ) {
				$this->error( CBTxt::T( 'AUTO_ACTION_FIELD_DOES_NOT_EXIST', ':: Action [action] :: Field skipped due to field [field_id] does not exist', [ '[action]' => $this->autoaction()->getId(), '[field_id]' => $fieldId ] ) );
				continue;
			}

			$operator				=	$row->getString( 'operator', 'set' );
			$value					=	$this->string( $actionUser, $row->getRaw( 'value', '' ), false, $row->getBool( 'translate', false ) );
			$fieldName				=	$fields[$fieldId]->getString( 'name', '' );
			$fieldType				=	$fields[$fieldId]->getString( 'type', '' );

			switch ( $operator ) {
				case 'prefix':
					$fieldValue		=	( $value . $actionUser->getRaw( $fieldName, '' ) );
					break;
				case 'suffix':
					$fieldValue		=	( $actionUser->getRaw( $fieldName, '' ) . $value );
					break;
				case 'add':
					$fieldValue		=	( $actionUser->getFloat( $fieldName, 0 ) + (float) $value );
					break;
				case 'subtract':
					$fieldValue		=	( $actionUser->getFloat( $fieldName, 0 ) - (float) $value );
					break;
				case 'divide':
					$fieldValue		=	( $actionUser->getFloat( $fieldName, 0 ) / (float) $value );
					break;
				case 'multiply':
					$fieldValue		=	( $actionUser->getFloat( $fieldName, 0 ) * (float) $value );
					break;
				case 'set':
				default:
					$operator		=	'set';
					$fieldValue		=	$value;
					break;
			}

			if ( ( $fieldName === 'alias' ) && ( ! $this->fieldAlias( $actionUser, $fieldValue ) ) ) {
				continue;
			}

			if ( ( $fieldType === 'password' ) && ( $operator === 'set' ) && ( strpos( $fieldValue, 'random' ) !== false ) ) {
				$randomPassword		=	explode( ':', $fieldValue );
				$passwordLength		=	( $randomPassword[1] ?? null );

				$fieldValue			=	( $passwordLength ? $actionUser->getRandomPassword( $passwordLength ) : $actionUser->getRandomPassword() );
			}

			if ( $row->getBool( 'direct', true ) ) {
				if ( $fieldType === 'password' ) {
					$fieldValue		=	$actionUser->hashAndSaltPassword( $fieldValue );
				}

				$query				=	'UPDATE ' . $_CB_database->NameQuote( $fields[$fieldId]->getString( 'table', '' ) )
									.	"\n SET " . $_CB_database->NameQuote( $fieldName ) . " = " . $_CB_database->Quote( $fieldValue )
									.	"\n WHERE " . $_CB_database->NameQuote( 'id' ) . " = " . $actionUser->getInt( 'id', 0 );
				$_CB_database->setQuery( $query );
				$_CB_database->query();
			} else {
				$actionUser->storeDatabaseValue( $fieldName, $fieldValue );
			}

			$actionUser->set( $fieldName, $fieldValue );
		}
	}

	/**
	 * @param UserTable $user
	 * @param mixed     $alias
	 * @return bool
	 */
	private function fieldAlias( UserTable $user, &$alias ): bool
	{
		global $_CB_database;

		$alias				=	trim( Application::Router()->stringToAlias( $alias ) );

		if ( cbutf8_strlen( $alias ) < 2 ) {
			$this->error( CBTxt::T( 'AUTO_ACTION_FIELD_ALIAS_TOO_SHORT', ':: Action [action] :: Field skipped due to profile alias "[alias]" being less than 2 characters', [ '[action]' => $this->autoaction()->getId(), '[alias]' => $alias ] ) );
			return false;
		}

		$query				=	'SELECT u.' . $_CB_database->NameQuote( 'id' )
							.	"\n FROM " . $_CB_database->NameQuote( '#__comprofiler' ) . " AS c"
							.	"\n INNER JOIN " . $_CB_database->NameQuote( '#__users' ) . " AS u"
							.	' ON u.' . $_CB_database->NameQuote( 'id' ) . ' = c.' . $_CB_database->NameQuote( 'id' )
							.	"\n WHERE c." . $_CB_database->NameQuote( 'id' ) . " != " . $user->getInt( 'id', 0 )
							.	"\n AND ( u." . $_CB_database->NameQuote( 'username' ) . " = " . $_CB_database->Quote( $alias )
							.	' OR c.' . $_CB_database->NameQuote( 'alias' ) . ' = ' . $_CB_database->Quote( $alias ) . ' )';
		$_CB_database->setQuery( $query, 0, 1 );
		$exists				=	$_CB_database->loadResult();

		if ( $exists ) {
			$this->error( CBTxt::T( 'AUTO_ACTION_FIELD_ALIAS_EXISTS', ':: Action [action] :: Field skipped due to profile alias "[alias]" already exists', [ '[action]' => $this->autoaction()->getId(), '[alias]' => $alias ] ) );
			return false;
		}

		if ( \in_array( $alias, Application::Router()->getViews(), true ) ) {
			$this->error( CBTxt::T( 'AUTO_ACTION_FIELD_ALIAS_VIEW', ':: Action [action] :: Field skipped due to profile alias "[alias]" matching a core view', [ '[action]' => $this->autoaction()->getId(), '[alias]' => $alias ] ) );
			return false;
		}

		return true;
	}
}