<?php
/**
* Community Builder (TM)
* @version $Id: $
* @package CommunityBuilder
* @copyright (C) 2004-2021 www.joomlapolis.com / Lightning MultiCom SA - and its licensors, all rights reserved
* @license http://www.gnu.org/licenses/old-licenses/gpl-2.0.html GNU/GPL version 2
*/

namespace CB\Plugin\AutoActions\Action;

use CB\Database\Table\UserTable;
use CBLib\Application\Application;
use CBLib\Language\CBTxt;
use CBLib\Registry\ParamsInterface;

\defined( 'CBLIB' ) or die();

class CbsubsAction extends Action
{

	/**
	 * @param UserTable $user
	 */
	public function execute( UserTable $user ): void
	{
		if ( ! $this->installed() ) {
			$this->error( CBTxt::T( 'AUTO_ACTION_CBSUBS_NOT_INSTALLED', ':: Action [action] :: CB Paid Subscriptions is not installed', [ '[action]' => $this->autoaction()->getId() ] ) );
			return;
		}

		foreach ( $this->autoaction()->getParams()->subTree( 'cbsubs' ) as $row ) {
			/** @var ParamsInterface $row */
			$userId										=	$row->getString( 'user', '' );

			if ( ! $userId ) {
				$userId									=	$user->getInt( 'id', 0 );
			} else {
				$userId									=	(int) $this->string( $user, $userId );
			}

			if ( $user->getInt( 'id', 0 ) !== $userId ) {
				$actionUser								=	\CBuser::getUserDataInstance( $userId );
			} else {
				$actionUser								=	$user;
			}

			$mode										=	$row->getInt( 'mode', 1 );

			if ( $mode === 5 ) {
				if ( ! $this->promotions() ) {
					$this->error( CBTxt::T( 'AUTO_ACTION_CBSUBSPROMO_NOT_INSTALLED', ':: Action [action] :: CBSubs Promotion is not installed', [ '[action]' => $this->autoaction()->getId() ] ) );
					continue;
				}

				$promotion								=	$row->subTree( 'promotion' );

				foreach ( $promotion->asArray() as $k => $v ) {
					$promotion->set( $k, $this->string( $user, $v, false ) );
				}

				$object									=	new \cbpaidpromotionTotalizertype();

				switch ( $row->getString( 'create_by', 'name' ) ) {
					case 'name':
						$object->load( [ 'name' => $promotion->getString( 'name', '' ) ] );
						break;
					case 'coupon':
						$object->load( [ 'coupon_code' => $promotion->getString( 'coupon_code', '' ) ] );
						break;
					case 'name_coupon':
						$object->load( [ 'name' => $promotion->getString( 'name', '' ), 'coupon_code' => $promotion->getString( 'coupon_code', '' ) ] );
						break;
				}

				$object->bind( $promotion->asArray() );

				if ( $object->getError() || ( ! $object->check() ) || ( ! $object->store() ) ) {
					$this->error( CBTxt::T( 'AUTO_ACTION_CBSUBS_ERROR', ':: Action [action] :: CB Paid Subscriptions skipped due to error: [error]', [ '[action]' => $this->autoaction()->getId(), '[error]' => $object->getError() ] ) );
				}
			} else {
				if ( ! $actionUser->getInt( 'id', 0 ) ) {
					$this->error( CBTxt::T( 'AUTO_ACTION_CBSUBS_NO_USER', ':: Action [action] :: CB Paid Subscriptions skipped due to missing user', [ '[action]' => $this->autoaction()->getId() ] ) );
					continue;
				}

				$plans									=	$row->getString( 'plans', '' );

				if ( ! $plans ) {
					$this->error( CBTxt::T( 'AUTO_ACTION_CBSUBS_NO_PLANS', ':: Action [action] :: CB Paid Subscriptions skipped due to missing plans', [ '[action]' => $this->autoaction()->getId() ] ) );
					continue;
				}

				$plans									=	cbToArrayOfInt( explode( '|*|', $plans ) );
				$subscriptions							=	\cbpaidUserExtension::getInstance( $actionUser->getInt( 'id', 0 ) )->getUserSubscriptions( null, true );
				$activePlans							=	[];

				foreach ( $subscriptions as $subscription ) {
					$subscriptionStatus					=	$subscription->getString( 'status', '' );
					$planId								=	$subscription->getInt( 'plan_id', 0 );

					if ( ! \in_array( $planId, $plans, true ) ) {
						continue;
					}

					if ( ( $mode === 1 ) && ( $subscriptionStatus === 'A' ) && $row->getBool( 'renew', false ) ) {
						$mode							=	2;
					}

					switch ( $mode ) {
						case 2:
							$subscription->activate( $actionUser, Application::Date( 'now', 'UTC' )->getTimestamp(), true, 'R' );
							break;
						case 3:
							$cancel						=	$subscription->stopAutoRecurringPayments();

							if ( \is_string( $cancel ) ) {
								$cancel					=	false;
							}

							if ( $cancel !== false ) {
								$subscription->deactivate( $actionUser, 'C' );
							}
							break;
						case 4:
							if ( $subscription->canDelete() ) {
								$subscription->revert( $actionUser, 'Denied' );
								$subscription->delete();
							}
							break;
						case 1:
						default:
							if ( ( $subscriptionStatus === 'A' ) && ( ! \in_array( $planId, $activePlans, true ) ) ) {
								$activePlans[]			=	$planId;
							}
							break;
					}
				}

				if ( $mode === 1 ) {
					$plansMgr							=	\cbpaidPlansMgr::getInstance();
					$postData							=	[];
					$chosenPlans						=	[];

					foreach ( $plans as $planId ) {
						if ( ! \in_array( $planId, $activePlans, true ) ) {
							$chosenPlans[$planId]		=	$plansMgr->loadPlan( $planId );
						}
					}

					if ( $chosenPlans ) {
						\cbpaidControllerOrder::createSubscriptionsAndPayment( $actionUser, $chosenPlans, $postData, null, null, 'A', null, 'U', 'free' );
					}
				}
			}
		}
	}

	/**
	 * @return array
	 */
	public function plans(): array
	{
		$plansList					=	[];

		if ( $this->installed() ) {
			$plansMgr				=	\cbpaidPlansMgr::getInstance();
			$plans					=	$plansMgr->loadPublishedPlans( null, true, 'any', null );

			if ( $plans ) {
				$plansList			=	[];

				foreach ( $plans as $k => $plan ) {
					$plansList[]	=	\moscomprofilerHTML::makeOption( (string) $k, $plan->getString( 'alias', '' ) );
				}
			}
		}

		return $plansList;
	}

	/**
	 * @return bool
	 */
	public function promotions(): bool
	{
		global $_PLUGINS;

		$_PLUGINS->loadPluginGroup( 'user/plug_cbpaidsubscriptions/plugin' );

		if ( $_PLUGINS->getLoadedPlugin( 'user/plug_cbpaidsubscriptions/plugin', 'cbsubs.promotion' ) ) {
			return true;
		}

		return false;
	}

	/**
	 * @return bool
	 */
	public function installed(): bool
	{
		global $_PLUGINS;

		if ( $_PLUGINS->getLoadedPlugin( 'user', 'cbpaidsubscriptions' ) ) {
			return true;
		}

		return false;
	}
}