<?php
/**
* Community Builder (TM)
* @version $Id: $
* @package CommunityBuilder
* @copyright (C) 2004-2021 www.joomlapolis.com / Lightning MultiCom SA - and its licensors, all rights reserved
* @license http://www.gnu.org/licenses/old-licenses/gpl-2.0.html GNU/GPL version 2
*/

namespace CB\Plugin\AutoActions\Action;

use CB\Database\Table\UserTable;
use CB\Plugin\AutoActions\CBAutoActions;
use CB\Plugin\AutoActions\Table\AutoActionTable;
use CBLib\Application\Application;
use CBLib\Input\Get;
use CBLib\Language\CBTxt;
use CBLib\Registry\GetterInterface;
use CBLib\Registry\ParamsInterface;
use CBLib\Registry\Registry;
use CBuser;

\defined( 'CBLIB' ) or die();

class Action implements ActionInterface
{
	/** @var null|AutoActionTable  */
	private ?AutoActionTable $autoaction	=	null;
	/** @var array  */
	private array $substitutions			=	[];
	/** @var array  */
	private array $variables				=	[];
	/** @var array  */
	private array $errors					=	[];

	/**
	 * Constructor for action object
	 *
	 * @param mixed $autoaction
	 */
	public function __construct( $autoaction = null )
	{
		// Set as mixed to workaround CBLib XML function calls trying to treat this like a table class
		if ( $autoaction instanceof AutoActionTable ) {
			$this->autoaction	=	$autoaction;
		}
	}

	/**
	 * Triggers the action checking access and conditions then returning its results
	 *
	 * @param UserTable $user
	 * @return mixed
	 */
	public function trigger( UserTable $user )
	{
		global $_PLUGINS, $_CB_database;

		$autoaction				=	$this->autoaction();
		$variables				=	$this->variables();

		$userId					=	$user->getInt( 'id' );
		$viewerId				=	Application::MyUser()->getUserId();

		// Exclude:
		$excludeGlobal			=	explode( ',', CBAutoActions::getGlobalParams()->getString( 'exclude', '' ) );
		$excludeTrigger			=	explode( ',', $autoaction->getParams()->getString( 'exclude', '' ) );
		$exclude				=	array_filter( array_merge( $excludeGlobal, $excludeTrigger ) );

		if ( $exclude ) {
			cbArrayToInts( $exclude );

			$exclude			=	array_unique( $exclude );

			if ( \in_array( $userId, $exclude, true ) ) {
				$this->error( CBTxt::T( 'AUTO_ACTION_USER_EXCLUDED', ':: Action [action] :: User [user_id] excluded', [ '[action]' => $autoaction->getId(), '[user_id]' => $userId ] ) );
				return null;
			}
		}

		// Access:
		if ( $userId ) {
			$gids				=	Application::User( $userId )->getAuthorisedGroups( false );

			array_unshift( $gids, -3 );

			if ( Application::User( $userId )->isGlobalModerator() ) {
				array_unshift( $gids, -5 );
			} else {
				array_unshift( $gids, -4 );
			}
		} else {
			$gids				=	$user->getRaw( 'gids', [] );

			array_unshift( $gids, -2 );
			array_unshift( $gids, -4 );
		}

		if ( $userId === $viewerId ) {
			array_unshift( $gids, -7 );
		} else {
			array_unshift( $gids, -6 );
		}

		array_unshift( $gids, -1 );

		$access					=	explode( '|*|', $autoaction->getAccess() );

		if ( ! array_intersect( $access, $gids ) ) {
			$this->error( CBTxt::T( 'AUTO_ACTION_ACCESS_FAILED', ':: Action [action] :: Access check for user [user_id] failed: looking for [access] in [groups]', [ '[action]' => $autoaction->getId(), '[user_id]' => $userId, '[access]' => implode( ', ', $access ), '[groups]' => implode( ', ', $gids ) ] ) );
			return null;
		}

		// Conditions:
		if ( $autoaction->getConditions()->count() ) {
			$orMatched										=	false;
			$conditionErrors								=	[];

			foreach ( $autoaction->getConditions() as $orIndex => $orCondition ) {
				/** @var ParamsInterface $orCondition */
				$andMatched									=	true;

				foreach ( $orCondition->subTree( 'condition' ) as $andIndex => $andCondition ) {
					/** @var ParamsInterface $andCondition */
					$fieldName								=	$andCondition->getString( 'field', '' );

					if ( ! $fieldName ) {
						continue;
					}

					$fieldValue								=	null;
					$operator								=	$andCondition->getInt( 'operator', 0 );

					if ( \in_array( $operator, [ 8, 9, 12, 13 ], true ) ) {
						$delimiter							=	$andCondition->getString( 'delimiter', '' );
					} else {
						$delimiter							=	null;
					}

					$value									=	null;

					if ( ! \in_array( $fieldName, [ 'customviewaccesslevels', 'customusergroups', 'customlanguages', 'custommoderators', 'customusers' ], true ) ) {
						$valueOptions						=	explode( '|*|', $andCondition->getString( 'value_options', '' ) );
						$value								=	$this->string( $user, $andCondition->getRaw( 'value', '' ), true, \in_array( 'translate', $valueOptions, true ), \in_array( 'format', $valueOptions, true ), \in_array( 'content_plugins', $valueOptions, true ), true );

						if ( \in_array( $operator, [ 6, 7 ], true ) ) {
							$value							=	null;
						}
					}

					switch ( $fieldName ) {
						case 'customvalue':
							$valueOptions					=	explode( '|*|', $andCondition->getString( 'field_custom_options', '' ) );
							$fieldValue						=	$this->string( $user, $andCondition->getRaw( 'field_custom', '' ), true, \in_array( 'translate', $valueOptions, true ), \in_array( 'format', $valueOptions, true ), \in_array( 'content_plugins', $valueOptions, true ), true );
							break;
						case 'customviewaccesslevels':
							$accessLevels					=	cbToArrayOfInt( explode( '|*|', $andCondition->getString( 'field_viewaccesslevels', '' ) ) );

							if ( $andCondition->getString( 'user_viewaccesslevels', 'action' ) === 'viewer' ) {
								$userAccessLevels			=	Application::MyUser()->getAuthorisedViewLevels();
							} else {
								$userAccessLevels			=	Application::User( $userId )->getAuthorisedViewLevels();
							}

							$fieldValue						=	0;

							foreach ( $accessLevels as $accessLevel ) {
								if ( \in_array( $accessLevel, $userAccessLevels, true ) ) {
									$fieldValue				=	1;
									break;
								}
							}

							$operator						=	$andCondition->getInt( 'operator_viewaccesslevels', 0 );
							$value							=	1;
							break;
						case 'customusergroups':
							$userGroups						=	cbToArrayOfInt( explode( '|*|', $andCondition->getString( 'field_usergroups', '' ) ) );

							if ( $andCondition->getString( 'user_usergroups', 'action' ) === 'viewer' ) {
								$userUsergroups				=	Application::MyUser()->getAuthorisedGroups();
							} else {
								$userUsergroups				=	Application::User( $userId )->getAuthorisedGroups();
							}

							$fieldValue						=	0;

							foreach ( $userGroups as $userGroup ) {
								if ( \in_array( $userGroup, $userUsergroups, true ) ) {
									$fieldValue				=	1;
									break;
								}
							}

							$operator						=	$andCondition->getInt( 'operator_usergroups', 0 );
							$value							=	1;
							break;
						case 'customlanguages':
							if ( $andCondition->getString( 'from_languages', 'user' ) === 'site' ) {
								$fieldValue					=	Application::Cms()->getLanguageTag();
							} elseif ( $andCondition->getString( 'user_languages', 'action' ) === 'viewer' ) {
								$fieldValue					=	CBuser::getMyUserDataInstance()->getUserLanguage();
							} else {
								$fieldValue					=	$user->getUserLanguage();
							}

							$operator						=	$andCondition->getInt( 'operator_languages', 12 );
							$delimiter						=	'|*|';
							$value							=	$andCondition->getString( 'field_languages', '' );
							break;
						case 'custommoderators':
							if ( $andCondition->getString( 'user_moderators', 'action' ) === 'viewer' ) {
								$fieldValue					=	(int) Application::MyUser()->isGlobalModerator();
							} else {
								$fieldValue					=	(int) Application::User( $userId )->isGlobalModerator();
							}

							$operator						=	$andCondition->getInt( 'operator_moderators', 0 );
							$value							=	1;
							break;
						case 'customusers':
							if ( $andCondition->getString( 'user_users', 'action' ) === 'viewer' ) {
								$fieldValue					=	$viewerId;
							} else {
								$fieldValue					=	$userId;
							}

							$operator						=	$andCondition->getInt( 'operator_users', 12 );
							$delimiter						=	',';
							$value							=	$this->string( $user, $andCondition->getRaw( 'field_users', '' ), true, false, false, false, true );
							break;
						case 'customconnections':
							if ( $viewerId === $userId ) {
								$fieldValue					=	0;
							} elseif ( $andCondition->getString( 'user_connections', 'viewer' ) === 'viewer' ) {
								$connections				=	new \cbConnection( $viewerId );
								$connection					=	$connections->getConnectionDetails( $viewerId, $userId );
								$fieldValue					=	( ( $connection !== false ) && ( (int) $connection->accepted === 1 ) && ( (int) $connection->pending === 0 ) ? 1 : 0 );
							} else {
								$connections				=	new \cbConnection( $userId );
								$connection					=	$connections->getConnectionDetails( $userId, $viewerId );
								$fieldValue					=	( ( $connection !== false ) && ( (int) $connection->accepted === 1 ) && ( (int) $connection->pending === 0 ) ? 1 : 0 );
							}

							$operator						=	$andCondition->getInt( 'operator_connections', 0 );
							$value							=	1;
							break;
						case 'customcode':
							try {
								$codeOptions				=	explode( '|*|', $andCondition->getString( 'field_code_options', '' ) );
								$code						=	$this->string( $user, $andCondition->getRaw( 'field_code', '' ), false, \in_array( 'translate', $codeOptions, true ), \in_array( 'format', $codeOptions, true ), \in_array( 'content_plugins', $codeOptions, true ), true );
								$fieldValue					=	CBAutoActions::outputCode( $code, $autoaction, $user, $variables, $this );
							} catch ( \Exception $e ) {
								$conditionErrors[]			=	CBTxt::T( 'AUTO_ACTION_CONDITION_CODE_FAILED', ':: Action [action] :: Condition code failed. Error: [error]', [ '[action]' => $autoaction->getId(), '[error]' => $e->getMessage() ] );
							}
							break;
						case 'customquery':
							try {
								$queryOptions				=	explode( '|*|', $andCondition->getString( 'field_query_options', '' ) );
								$query						=	$this->string( $user, $andCondition->getRaw( 'field_query', '' ), [ '\CB\Plugin\AutoActions\CBAutoActions', 'escapeSQL' ], \in_array( 'translate', $queryOptions, true ), \in_array( 'format', $queryOptions, true ), \in_array( 'content_plugins', $queryOptions, true ), true );

								if ( stripos( trim( $query ), 'SELECT' ) === 0 ) {
									$_CB_database->setQuery( $query );

									if ( stripos( trim( $query ), 'SELECT COUNT' ) === 0  ) {
										// Count
										$fieldValue			=	$_CB_database->loadResult();
									} elseif ( \in_array( $operator, [ 8, 9 ], true ) ) {
										// Multiple Rows
										$fieldValue			=	$_CB_database->loadResultArray();
									} elseif ( preg_match( '/LIMIT (?:0,\s*1|1)/i', trim( $query ) ) ) {
										// Single Row
										$fieldValue			=	$_CB_database->loadResult();
									} else {
										// Multiple Rows
										$fieldValue			=	implode( '|*|', $_CB_database->loadResultArray() );
									}
								}
							} catch ( \Exception $e ) {
								$conditionErrors[]			=	CBTxt::T( 'AUTO_ACTION_CONDITION_QUERY_FAILED', ':: Action [action] :: Condition query failed. Error: [error]', [ '[action]' => $autoaction->getId(), '[error]' => $e->getMessage() ] );
							}
							break;
						case 'customsubscriptions':
							if ( $andCondition->getString( 'user_subscriptions', 'action' ) === 'viewer' ) {
								$subscriptions				=	CBAutoActions::getCBSubsSubscriptionsStatus( $viewerId );
							} else {
								$subscriptions				=	CBAutoActions::getCBSubsSubscriptionsStatus( $userId );
							}

							$plans							=	$andCondition->getString( 'plans_subscriptions' );

							if ( $plans ) {
								$plans						=	explode( '|*|', $plans );

								if ( \in_array( '', $plans, true ) ) {
									// Any Plan was selected so lets just enforce its selection:
									$plans					=	null;
								}
							}

							$status							=	$andCondition->getString( 'status_subscriptions' );

							if ( $status ) {
								$status						=	explode( '|*|', $status );

								if ( \in_array( '', $status, true ) ) {
									// Any Status was selected so lets just enforce its selection:
									$status					=	null;
								}
							}

							$fieldValue						=	0;

							if ( $plans ) {
								// Specific Plan with Any Status
								foreach ( $plans as $planId ) {
									if ( ! isset( $subscriptions[$planId] ) ) {
										continue;
									}

									// Specific Plan with Specific Status
									if ( $status && ( ! \in_array( $subscriptions[$planId], $status, true ) ) ) {
										continue;
									}

									$fieldValue				=	1;
									break;
								}
							} elseif ( $status ) {
								// Any Plan with Specific Status
								foreach ( $subscriptions as $subscriptionStatus ) {
									if ( ! \in_array( $subscriptionStatus, $status, true ) ) {
										continue;
									}

									$fieldValue				=	1;
									break;
								}
							} elseif ( $subscriptions ) {
								// Any Plan with Any Status
								$fieldValue					=	1;
							}

							$operator						=	$andCondition->getInt( 'operator_subscriptions', 0 );
							$value							=	1;
							break;
						default:
							if ( $andCondition->getString( 'user_fields', 'action' ) === 'viewer' ) {
								$fieldUser					=	CBuser::getMyUserDataInstance();
							} else {
								$fieldUser					=	$user;
							}

							$fieldValue						=	$fieldUser->getRaw( $fieldName );

							if ( $fieldValue === null ) {
								$fieldValue					=	CBuser::getInstance( $fieldUser->getInt( 'id', 0 ), false )->getField( $fieldName, null, 'php' );

								if ( \is_array( $fieldValue ) ) {
									$fieldValue				=	array_shift( $fieldValue );

									if ( \is_array( $fieldValue ) ) {
										$fieldValue			=	implode( '|*|', $fieldValue );
									}
								}
							}
							break;
					}

					$isMatched								=	CBAutoActions::getConditionMatch( $fieldValue, $operator, $value, $delimiter );

					if ( ! $isMatched ) {
						$conditionErrors[]					=	CBTxt::T( 'AUTO_ACTION_CONDITION_FAILED', ':: Action [action] :: Condition [cond] failed for user [user_id] with "[field]" [operator] "[value]"', [ '[action]' => $autoaction->getId(), '[cond]' => ( $orIndex + 1 ) . '-' . ( $andIndex + 1 ), '[user_id]' => $userId, '[field]' => $fieldValue, '[operator]' => CBAutoActions::getOperator( $operator ), '[value]' => $value ] );
					}

					if ( $andMatched ) {
						$andMatched							=	$isMatched;
					}
				}

				if ( ( ! $orMatched  ) && $andMatched ) {
					$orMatched								=	true;
				}
			}

			if ( ! $orMatched ) {
				foreach ( $conditionErrors as $conditionError ) {
					$this->error( $conditionError );
				}

				return null;
			}
		}

		// Action:
		$passwordCache			=	$user->getString( 'password' );

		$user->set( 'password', null );

		$content				=	$this->execute( $user );

		$user->set( 'password', $passwordCache );

		$substitutions			=	$this->substitutions();

		$_PLUGINS->trigger( 'autoactions_onAction', [ &$content, &$autoaction, &$user, &$variables, &$substitutions ] );

		// Layout:
		$display				=	$autoaction->getParams()->getString( 'display', 'none' );
		$url					=	$autoaction->getParams()->getString( 'display_url', '' );
		$layout					=	$autoaction->getParams()->getRaw( 'display_layout', '' );

		if ( ( $display !== 'none' ) && (
				( ( $content !== '' ) && ( $content !== null ) )
				|| ( ( $display === 'redirect' ) && $url )
				|| ( ( $display !== 'redirect' ) && $layout )
		) ) {
			if ( \is_object( $content ) || \is_array( $content ) ) {
				$registry		=	new Registry( $content );
				$content		=	$registry->asJson();
			}

			$return				=	null;

			if ( ( $display === 'redirect' ) && $url ) {
				$redirect			=	$this->string( $user, $url, ( preg_match( '/^\[[a-zA-Z0-9-_]+\]$/', $url ) ? false : [ '\CB\Plugin\AutoActions\CBAutoActions', 'escapeURL' ] ) );
				$message			=	$this->string( $user, $autoaction->getParams()->getRaw( 'display_url_message', '' ), false );
				$messageType		=	$autoaction->getParams()->getString( 'display_url_type', 'message' );

				if ( $messageType === 'custom' ) {
					$messageType	=	$this->string( $user, $autoaction->getParams()->getString( 'display_url_custom_type', '' ) );
				}

				if ( ! $redirect ) {
					$redirect		=	'index.php';
				} if ( strpos( $redirect, 'index.php?Itemid=' ) === 0 ) {
					// We're linking to another menu item so lets run through SEF before redirecting so we have the ACTUAL menu URL:
					$redirect		=	cbSef( $redirect, false );
				}

				cbRedirect( $redirect, $message, ( $message ? ( $messageType ?: null ) : null ) );
			} elseif ( ( $display !== 'redirect' ) && $layout ) {
				$translate		=	$autoaction->getParams()->getBool( 'display_translate', false );
				$substitute		=	$autoaction->getParams()->getBool( 'display_substitutions', false );
				$format			=	$autoaction->getParams()->getBool( 'display_format', false );
				$prepare		=	$autoaction->getParams()->getBool( 'display_content_plugins', false );

				$return			=	str_replace( '[content]', $content, $this->string( $user, $layout, false, $translate, $format, $prepare, $substitute, [ 'content' => $content ] ) );

				if ( $autoaction->getParams()->getString( 'display_method', 'html' ) === 'php' ) {
					try {
						$return	=	CBAutoActions::outputCode( $return, $autoaction, $user, $variables, $this, $content );
					} catch ( \Exception $e ) {
						$this->error( CBTxt::T( 'AUTO_ACTION_LAYOUT_CODE_FAILED', ':: Action [action] :: Layout code failed. Error: [error]', [ '[action]' => $autoaction->getId(), '[error]' => $e->getMessage() ] ) );
						return null;
					}
				}
			} else {
				$return			=	$content;
			}

			switch ( $display ) {
				case 'silent':
					break;
				case 'echo':
					echo $return;
					break;
				case 'var_dump':
					/** @noinspection ForgottenDebugOutputInspection */
					var_dump( $return );
					break;
				case 'print':
					print $return;
					break;
				case 'exit':
					exit( $return );
				case 'json':
					header( 'Content-Type: application/json' );

					echo $return;

					exit();
				case 'return':
					return $return;
			}
		}

		return null;
	}

	/**
	 * Executes the action directly and returns its results
	 *
	 * @param UserTable $user
	 * @return null|array|bool|string
	 */
	public function execute( UserTable $user )
	{
		$params				=	$this->autoaction()->getParams()->subTree( 'action' );
		$actions			=	$params->getString( 'actions', '' );

		if ( ! $actions ) {
			$this->error( CBTxt::T( 'AUTO_ACTION_ACTIN_NO_ACTIONS', ':: Action [action] :: Action skipped due to missing actions', [ '[action]' => $this->autoaction()->getId() ] ) );
			return null;
		}

		$mode				=	$params->getString( 'mode', 'trigger' );
		$userId				=	$params->getString( 'user', '' );

		if ( ! $userId ) {
			$userId			=	$user->getInt( 'id', 0 );
		} else {
			$userId			=	(int) $this->string( $user, $userId );
		}

		if ( $user->getInt( 'id', 0 ) !== $userId ) {
			$actionUser		=	CBuser::getUserDataInstance( $userId );
		} else {
			$actionUser		=	$user;
		}

		$variables			=	$this->variables();
		$substitutions		=	$this->substitutions();
		$actions			=	cbToArrayOfInt( explode( '|*|', $actions ) );
		$return				=	null;

		foreach ( $actions as $actionId ) {
			$action			=	new AutoActionTable();

			if ( ! $action->load( $actionId ) ) {
				continue;
			}

			if ( ( ! $action->getId() ) || ( ! $action->getPublished() ) ) {
				continue;
			}

			switch ( $mode ) {
				case 'run':
					$return	.=	$action->runAction( $variables );
					break;
				case 'execute':
					$return	.=	$action->executeAction( $actionUser );
					break;
				case 'trigger':
				default:
					$return	.=	$action->triggerAction( $actionUser, $variables, $substitutions );
					break;
			}

			if ( $action->getError() ) {
				$this->error( $action->getError() );
			}
		}

		return $return;
	}

	/**
	 * Gets the auto action associated with this action
	 *
	 * @return AutoActionTable
	 */
	public function autoaction(): AutoActionTable
	{
		if ( ! $this->autoaction ) {
			$this->autoaction	=	new AutoActionTable();
		}

		return $this->autoaction;
	}

	/**
	 * Gets or sets substitution variables for this action
	 *
	 * @param null|array $substitutions
	 * @return array
	 */
	public function substitutions( ?array $substitutions = null ): array
	{
		if ( $substitutions === null ) {
			return $this->substitutions;
		}

		$this->substitutions	=	$substitutions;

		return $this->substitutions;
	}

	/**
	 * Gets or sets action variables
	 *
	 * @param null|array $variables
	 * @return array
	 */
	public function variables( ?array $variables = null ): array
	{
		if ( $variables === null ) {
			return $this->variables;
		}

		$this->variables	=	$variables;

		return $this->variables;
	}

	/**
	 * Parses a string through action substitutions
	 *
	 * @param null|UserTable  $user
	 * @param null|string     $string
	 * @param null|array|bool $htmlspecialchars
	 * @param null|bool       $translate
	 * @param null|bool       $format
	 * @param null|bool       $prepare
	 * @param null|bool       $substitutions
	 * @param array           $extras
	 * @return string
	 */
	public function string( ?UserTable $user, ?string $string, $htmlspecialchars = true, ?bool $translate = null, ?bool $format = null, ?bool $prepare = null, ?bool $substitutions = null, array $extras = [] ): string
	{
		if ( ( $string === null ) || ( $string === '' ) ) {
			return '';
		}

		if ( ( $this->autoaction()->getParams()->getBool( 'translate', true ) && ( $translate === null ) ) || ( $translate === true ) ) {
			$string						=	CBTxt::T( $string );
		}

		if ( ( ( ! $this->autoaction()->getParams()->getBool( 'substitutions', true ) ) && ( $substitutions === null ) ) || ( $substitutions === false ) ) {
			return $string;
		}

		if ( ! $user ) {
			$user						=	CBuser::getMyInstance()->getUserData();
		}

		$cbUser							=	new CBuser();
		$cbUser->_cbuser				=	$user;

		$substitutions					=	$this->substitutions();
		$variables						=	$this->variables();

		$substitutions['action_id']		=	$this->autoaction()->getId();

		$password						=	( isset( $substitutions['password'] ) ? Get::clean( $substitutions['password'], GetterInterface::STRING ) : null );
		$actionUser						=	( isset( $variables['user'] ) ? $variables['user']->getInt( 'id', 0 ) : 0 );

		$substitutions['action_user']	=	$actionUser;

		$ignore							=	[];
		$ignoreId						=	0;

		$string							=	preg_replace_callback( '%\[cbautoactions:ignore\](.*?)\[/cbautoactions:ignore\]%si', static function( array $matches ) use ( &$ignore, &$ignoreId )
												{
													$ignoreId++;

													$ignore[$ignoreId]		=	$matches[1];

													return '[cbautoactions:ignored ' . $ignoreId . ']';
												},
												$string );

		if ( $password !== null ) {
			$string						=	str_ireplace( '[password]', $password, $string );
		}

		$string							=	str_ireplace( '[action_user]', $actionUser, $string );

		if ( ( $this->autoaction()->getParams()->getBool( 'content_plugins', false ) && ( $prepare === null ) ) || ( $prepare === true ) ) {
			$string						=	Application::Cms()->prepareHtmlContentPlugins( $string, 'autoaction', ( $user ? $user->getInt( 'id', 0 ) : 0 ) );
		}

		if ( $extras ) {
			$substitutions				=	array_merge( $substitutions, $extras );
		}

		$string							=	$cbUser->replaceUserVars( $string, $htmlspecialchars, false, $substitutions, false );

		if ( ( $this->autoaction()->getParams()->getBool( 'format', false ) && ( $format === null ) ) || ( $format === true ) ) {
			$string						=	CBAutoActions::formatFunction( $string, $variables, $substitutions, $user, $cbUser );
		}

		foreach ( $ignore as $id => $ignored ) {
			$string						=	str_replace( '[cbautoactions:ignored ' . (int) $id . ']', $ignored, $string );
		}

		return $string;
	}

	/**
	 * Checks if the actions dependency is installed
	 *
	 * @return bool
	 */
	public function installed(): bool
	{
		return true;
	}

	/**
	 * Gets or sets action errors
	 *
	 * @param null|string $error
	 * @return array
	 */
	public function error( ?string $error = null ): array
	{
		if ( $error ) {
			$this->errors[]		=	$error;
		}

		return $this->errors;
	}
}