<?php
/**
* Community Builder (TM)
* @version $Id: $
* @package CommunityBuilder
* @copyright (C) 2004-2021 www.joomlapolis.com / Lightning MultiCom SA - and its licensors, all rights reserved
* @license http://www.gnu.org/licenses/old-licenses/gpl-2.0.html GNU/GPL version 2
*/

namespace CB\Plugin\GroupJiveForums\Table\Kunena;

use CB\Plugin\GroupJive\Table\GroupTable;
use CB\Plugin\GroupJiveForums\CBGroupJiveForums;
use CB\Plugin\GroupJiveForums\Forum\ForumInterface;
use CBLib\Database\DatabaseDriverInterface;
use CBLib\Database\Table\Table;
use CB\Plugin\GroupJiveForums\Table\CategoryTableInterface;
use Kunena\Forum\Libraries\Forum\Category\KunenaCategory;

defined('CBLIB') or die();

class CategoryTable extends Table implements CategoryTableInterface
{
	/** @var int  */
	public $id				=	null;
	/** @var int  */
	public $parent			=	0;
	/** @var string  */
	public $name			=	'';
	/** @var string  */
	public $alias			=	'';
	/** @var string  */
	public $description		=	'';
	/** @var int  */
	public $published		=	0;

	/** @var null|ForumInterface  */
	protected $_forum		=	null;
	/** @var int  */
	protected $_version		=	0;
	/** @var KunenaCategory|\KunenaForumCategory  */
	protected $_category	=	null;

	/**
	 * Table name in database
	 *
	 * @var string
	 */
	protected $_tbl			=	'#__kunena_categories';

	/**
	 * Primary key(s) of table
	 *
	 * @var string
	 */
	protected $_tbl_key		=	'id';

	/**
	 * @param DatabaseDriverInterface|null $db
	 * @param string                       $table
	 * @param string|array                 $key
	 */
	public function __construct( DatabaseDriverInterface $db = null, $table = null, $key = null )
	{
		parent::__construct( $db, $table, $key );

		$this->_forum			=	CBGroupJiveForums::getForum();
		$this->_version			=	( $this->_forum ? $this->_forum->version : 0 );

		if ( $this->_version >= 6 ) {
			$this->_category	=	new KunenaCategory();
		} else {
			$this->_category	=	new \KunenaForumCategory();
		}

		$this->_category->load();
	}

	/**
	 * @param int|array $keys
	 * @return bool
	 */
	public function load( $keys = null )
	{
		if ( is_array( $keys ) ) {
			return false;
		}

		$this->_category->load( (int) $keys );

		if ( ! $this->_category->exists() ) {
			$this->_category->set( 'id', 0 );
		}

		$this->category( $this->_category );

		return true;
	}

	/**
	 * @param bool $updateNulls
	 * @return bool
	 */
	public function store( $updateNulls = false )
	{
		if ( ! $this->_category ) {
			return false;
		}

		if ( $this->_version >= 6 ) {
			if ( $this->_category->get( 'channels' ) === null ) {
				$this->_category->set( 'channels', 'THIS' );
			}

			if ( $this->_category->get( 'icon' ) === null ) {
				$this->_category->set( 'icon', '' );
			}

			if ( $this->_category->get( 'iconset' ) === null ) {
				$this->_category->set( 'iconset', 'default' );
			}

			if ( $this->_category->get( 'headerdesc' ) === null ) {
				$this->_category->set( 'headerdesc', '' );
			}

			if ( $this->_category->get( 'topictemplate' ) === null ) {
				$this->_category->set( 'topictemplate', '' );
			}

			if ( $this->_category->get( 'class_sfx' ) === null ) {
				$this->_category->set( 'class_sfx', '' );
			}
		}

		$store	=	$this->_category->save();

		$this->category( $this->_category );

		return $store;
	}

	/**
	 * @param string $key
	 * @return string
	 */
	private function map( $key )
	{
		if ( $this->_version >= 6 ) {
			if ( $key === 'parent' ) {
				$key	=	'parentid';
			}
		} elseif ( $key === 'parent' ) {
			$key		=	'parent_id';
		}

		return $key;
	}

	/**
	 * @param string $var
	 * @param mixed  $value
	 */
	public function set( $var, $value )
	{
		if ( ! $this->_category ) {
			return;
		}

		parent::set( $var, $value );

		$this->_category->set( $this->map( $var ), $value );
	}

	/**
	 * @param null|int $id
	 * @return bool
	 */
	public function delete( $id = null )
	{
		if ( ( ! $this->_category ) || $id ) {
			return false;
		}

		return $this->_category->delete();
	}

	/**
	 * @return bool
	 */
	public function check()
	{
		if ( ! $this->_category ) {
			return false;
		}

		return $this->_category->check();
	}

	/**
	 * @param array|object $array
	 * @param string       $ignore
	 * @param string       $prefix
	 * @return boolean
	 */
	public function bind( $array, $ignore = '', $prefix = null )
	{
		$bind				=	parent::bind( $array, $ignore, $prefix );

		if ( $bind ) {
			$data			=	array();

			foreach ( $this as $k => $v ) {
				$data[$this->map( $k )]	=	$v;
			}

			$bind			=	$this->_category->bind( $data, explode( ' ', $ignore ), false );
		}

		return $bind;
	}

	/**
	 * get or set the kunena category object
	 *
	 * @param KunenaCategory|\KunenaForumCategory|null $category
	 * @return KunenaCategory|\KunenaForumCategory|null
	 */
	public function category( $category = null )
	{
		if ( $category !== null ) {
			$this->_category	=	$category;

			$data				=	array();

			foreach ( $this->_category as $k => $v ) {
				$data[$this->map( $k )]		=	$v;
			}

			parent::bind( $data );
		}

		return $this->_category;
	}

	/**
	 * sets the forum access based off category or group object access
	 *
	 * @param \CB\Plugin\GroupJive\Table\CategoryTable|GroupTable $row
	 */
	public function access( $row )
	{
		if ( $row instanceof \CB\Plugin\GroupJive\Table\CategoryTable ) {
			$this->_category->set( 'accesstype', 'joomla.level' );
			$this->_category->set( 'access', $row->getInt( 'access', 0 ) );
		} elseif ( $row instanceof GroupTable ) {
			$this->_category->set( 'accesstype', 'communitybuilder' );
			$this->_category->set( 'access', $row->getInt( 'id', 0 ) );
		}
	}

	/**
	 * get, add, or delete moderators for this category
	 *
	 * @param null|array $addUsers
	 * @param null|array $deleteUsers
	 * @return array
	 */
	public function moderators( $addUsers = null, $deleteUsers = null )
	{
		if ( ! $this->_category ) {
			return array();
		}

		if ( $addUsers !== null ) {
			if ( is_array( $addUsers ) ) {
				$this->_category->addModerators( $addUsers );
			} else {
				$this->_category->addModerator( $addUsers );
			}
		}

		if ( $deleteUsers !== null ) {
			if ( is_array( $deleteUsers ) ) {
				foreach ( $deleteUsers as $deleteUser ) {
					$this->_category->removeModerator( $deleteUser );
				}
			} else {
				$this->_category->removeModerator( $deleteUsers );
			}
		}

		return $this->_category->getModerators( false, false );
	}

	/**
	 * returns the forum category url
	 *
	 * @return null|string
	 */
	public function url()
	{
		if ( ! $this->_category ) {
			return null;
		}

		return $this->_category->getUrl();
	}
}