<?php
/**
 * Community Builder (TM)
 * @version $Id: $
 * @package CommunityBuilder
 * @copyright (C) 2004-2021 www.joomlapolis.com / Lightning MultiCom SA - and its licensors, all rights reserved
 * @license http://www.gnu.org/licenses/old-licenses/gpl-2.0.html GNU/GPL version 2
 */

namespace CB\Plugin\FieldGroups\Field;

use CB\Database\Table\UserTable;
use CB\Database\Table\FieldTable;
use CBLib\Application\Application;
use CBLib\Language\CBTxt;
use CBLib\Registry\GetterInterface;
use CB\Plugin\FieldGroups\CBFieldGroups;
use CBLib\Registry\Registry;

defined('CBLIB') or die();

class FieldGroupField extends \cbFieldHandler
{

	/**
	 * Accessor:
	 * Returns a field in specified format
	 *
	 * @param  FieldTable  $field
	 * @param  UserTable   $user
	 * @param  string      $output               'html', 'xml', 'json', 'php', 'csvheader', 'csv', 'rss', 'fieldslist', 'htmledit'
	 * @param  string      $reason               'profile' for user profile view, 'edit' for profile edit, 'register' for registration, 'search' for searches
	 * @param  int         $list_compare_types   IF reason == 'search' : 0 : simple 'is' search, 1 : advanced search with modes, 2 : simple 'any' search
	 * @return mixed
	 */
	public function getField( &$field, &$user, $output, $reason, $list_compare_types )
	{
		global $_CB_framework, $_PLUGINS, $ueConfig;

		$fieldName								=	$field->get( 'name', null, GetterInterface::STRING );

		switch ( $output ) {
			case 'html':
			case 'rss':
				$template						=	$field->params->get( 'repeat_template', null, GetterInterface::STRING );
				$customTemplate					=	CBTxt::T( $field->params->get( 'repeat_template_custom', null, GetterInterface::RAW ) );
				$fieldFormatting				=	$field->params->get( 'repeat_formatting', 'div', GetterInterface::STRING );
				$showEmptyFields				=	$field->params->get( 'repeat_empty_fields', null, GetterInterface::STRING );
				$showEmptyFieldCurrent			=	Application::Config()->get( 'showEmptyFields', 1, GetterInterface::INT );

				if ( $showEmptyFields != '' ) {
					$ueConfig['showEmptyFields']	=	(int) $showEmptyFields;
				}

				$dummyUser						=	clone $user;

				$dummyUser->set( '_isFieldGroup', true );

				$hiddenPositions				=	array( 'not_on_profile_1', 'not_on_profile_2', 'not_on_profile_3', 'not_on_profile_4', 'not_on_profile_5', 'not_on_profile_6', 'not_on_profile_7', 'not_on_profile_8', 'not_on_profile_9' );

				if ( $reason == 'profile' ) {
					$availableTabs				=	\CBuser::getInstance( $user->get( 'id', 0, GetterInterface::INT ), false )->_getCbTabs( false )->_getTabsDb( $user, $reason );
				} else {
					$availableTabs				=	array();
				}

				$fieldGroups					=	CBFieldGroups::getGroupedFields( $field, $dummyUser, $reason );
				$rows							=	array();
				$rowsTitles						=	array();
				$rowsDescriptions				=	array();
				$rowsFields						=	array();

				foreach ( $fieldGroups as $row => $groupedFields ) {
					/** @var FieldTable[] $groupedFields */
					$cbUser						=	CBFieldGroups::mapGroupedColumns( $groupedFields, $dummyUser );
					$titles						=	array();
					$descriptions				=	array();
					$fields						=	array();
					$extras						=	array();

					foreach ( $groupedFields as $groupedField ) {
						$groupedFieldTab		=	$groupedField->get( 'tabid', 0, GetterInterface::INT );

						if ( ( $reason == 'profile' )
							 && ( $groupedFieldTab != $field->get( 'tabid', 0, GetterInterface::INT ) )
							 && isset( $availableTabs[$groupedFieldTab] )
							 && in_array( $availableTabs[$groupedFieldTab]->get( 'position', null, GetterInterface::STRING ), $hiddenPositions ) ) {
							// The grouped field is on a different tab that's in a not shown on profile position so lets respect that and hide it from profile view:
							continue;
						}

						$groupedFieldType		=	$groupedField->get( 'type', null, GetterInterface::STRING );
						$groupedFieldName		=	$groupedField->get( '_name', null, GetterInterface::STRING );

						$title					=	$_PLUGINS->callField( $groupedFieldType, 'getFieldTitle', array( &$groupedField, &$dummyUser, $output, $reason ), $groupedField );
						$description			=	$_PLUGINS->callField( $groupedFieldType, 'getFieldDescription', array( &$groupedField, &$dummyUser, $output, $reason ), $groupedField );
						$display				=	$_PLUGINS->callField( $groupedFieldType, 'getFieldRow', array( &$groupedField, &$dummyUser, $output, $fieldFormatting, $reason, $list_compare_types ), $groupedField );

						if ( ( $groupedFieldType == 'image' ) && ( strpos( $display, 'cbImagePendingApprovalButtons' ) !== false ) ) {
							$imgApprove			=	$_CB_framework->viewUrl( 'fieldclass', true, array( 'field' => $field->get( 'name', null, GetterInterface::STRING ), 'function' => 'image_approve', 'user' => $user->get( 'id', 0, GetterInterface::INT ), 'reason' => $reason, 'index' => $row, 'image' => $groupedFieldName ) );
							$imgReject			=	$_CB_framework->viewUrl( 'fieldclass', true, array( 'field' => $field->get( 'name', null, GetterInterface::STRING ), 'function' => 'image_reject', 'user' => $user->get( 'id', 0, GetterInterface::INT ), 'reason' => $reason, 'index' => $row, 'image' => $groupedFieldName ) );

							$imgApproval		=	'<div class="cbImagePendingApprovalButtons">'
												.		'<a href="' . $imgApprove . '" class="btn btn-sm btn-success cbImagePendingApprovalAccept">' . CBTxt::Th( 'UE_APPROVE', 'Approve' ) . '</a>'
												.		' <a href="' . $imgReject . '" class="btn btn-sm btn-danger cbImagePendingApprovalReject">' . CBTxt::Th( 'UE_REJECT', 'Reject' ) . '</a>'
												.	'</div>';

							$display			=	preg_replace( '%<div class="cbImagePendingApprovalButtons">.+</div>%iU', $imgApproval, $display );
						}

						ob_start();
						require CBFieldGroups::getTemplate( $template, 'display_field' );
						$html					=	ob_get_contents();
						ob_end_clean();

						if ( trim( $html ) != '' ) {
							$titles[$groupedFieldName]						=	$title;
							$descriptions[$groupedFieldName]				=	$description;
							$fields[$groupedFieldName]						=	$html;
						}

						if ( $customTemplate ) {
							// Always build the substitutions even if empty:
							$extras['title_' . $groupedFieldName]			=	$title;
							$extras['description_' . $groupedFieldName]		=	$description;
							$extras['field_' . $groupedFieldName]			=	( trim( $html ) != '' ? $html : '' );
						}
					}

					if ( $customTemplate && $fields ) {
						// Replace the fields array that's about to be imploded with the single custom template usage:
						$fields					=	array();
						$fields[]				=	$cbUser->replaceUserVars( $customTemplate, false, false, $extras, false );
					}

					ob_start();
					require CBFieldGroups::getTemplate( $template, 'display_row' );
					$html						=	ob_get_contents();
					ob_end_clean();

					if ( trim( $html ) != '' ) {
						$rows[]					=	$html;
						$rowsTitles[]			=	$titles;
						$rowsDescriptions[]		=	$descriptions;
						$rowsFields[]			=	$fields;
					}

					// Revert the cached user object change:
					$cbUser->_cbuser			=	$user;
				}

				$dummyUser->set( '_isFieldGroup', false );

				ob_start();
				require CBFieldGroups::getTemplate( $template, 'display' );
				$html							=	ob_get_contents();
				ob_end_clean();

				if ( $showEmptyFields != '' ) {
					$ueConfig['showEmptyFields']	=	$showEmptyFieldCurrent;
				}

				return $this->formatFieldValueLayout( $this->_formatFieldOutput( $fieldName, $html, $output, false ), $reason, $field, $user, false );
			case 'htmledit':
				$template						=	$field->params->get( 'repeat_template_edit', null, GetterInterface::STRING );
				$customTemplate					=	CBTxt::T( $field->params->get( 'repeat_template_edit_custom', null, GetterInterface::RAW ) );
				$fieldFormatting				=	$field->params->get( 'repeat_formatting_edit', 'div', GetterInterface::STRING );

				if ( ! $template ) {
					$template					=	$field->params->get( 'repeat_template', null, GetterInterface::STRING );
				}

				if ( $reason == 'search' ) {
					if ( ! CBFieldGroups::canSearch() ) {
						return null;
					}

					static $searchRepeat		=	0;

					if ( ! $searchRepeat++ ) {
						$_CB_framework->outputCbJQuery( "$( '.cbRepeat' ).cbrepeat();", 'cbrepeat' );
					}

					$dummyUser					=	clone $user; // using a dummy user object to avoid real user object being altered during grouped field parsing

					$dummyUser->set( '_isFieldGroup', true );

					$fieldGroups				=	CBFieldGroups::getGroupedFields( $field, $dummyUser, $reason );

					$rows						=	array();
					$rowsTitles					=	array();
					$rowsDescriptions			=	array();
					$rowsFields					=	array();

					foreach ( $fieldGroups as $row => $groupedFields ) {
						/** @var FieldTable[] $groupedFields */
						$cbUser					=	CBFieldGroups::mapGroupedColumns( $groupedFields, $dummyUser );
						$titles					=	array();
						$descriptions			=	array();
						$fields					=	array();

						foreach ( $groupedFields as $groupedField ) {
							$title				=	$_PLUGINS->callField( $groupedField->get( 'type', null, GetterInterface::STRING ), 'getFieldTitle', array( &$groupedField, &$dummyUser, $output, $reason ), $groupedField );
							$description		=	$_PLUGINS->callField( $groupedField->get( 'type', null, GetterInterface::STRING ), 'getFieldDescription', array( &$groupedField, &$dummyUser, $output, $reason ), $groupedField );
							$edit				=	$_PLUGINS->callField( $groupedField->get( 'type', null, GetterInterface::STRING ), 'getFieldRow', array( &$groupedField, &$dummyUser, $output, 'div', $reason, $list_compare_types ), $groupedField );

							ob_start();
							require CBFieldGroups::getTemplate( $template, 'search_field' );
							$html				=	ob_get_contents();
							ob_end_clean();

							if ( trim( $html ) != '' ) {
								$groupedFieldName					=	$groupedField->get( '_name', null, GetterInterface::STRING );

								$titles[$groupedFieldName]			=	$title;
								$descriptions[$groupedFieldName]	=	$description;
								$fields[$groupedFieldName]			=	$html;
							}
						}

						ob_start();
						require CBFieldGroups::getTemplate( $template, 'search_row' );
						$html					=	ob_get_contents();
						ob_end_clean();

						if ( trim( $html ) != '' ) {
							$rows[]				=	$html;
							$rowsTitles[]		=	$titles;
							$rowsDescriptions[]	=	$descriptions;
							$rowsFields[]		=	$fields;
						}

						// Revert the cached user object change:
						$cbUser->_cbuser		=	$user;
					}

					$dummyUser->set( '_isFieldGroup', false );

					ob_start();
					require CBFieldGroups::getTemplate( $template, 'search' );
					$html						=	ob_get_contents();
					ob_end_clean();

					return $this->formatFieldValueLayout( $this->_formatFieldOutput( $fieldName, $html, $output, false ), $reason, $field, $user, false );
				} else {
					static $editRepeat			=	0;

					if ( ! $editRepeat++ ) {
						$_CB_framework->outputCbJQuery( "$( '.cbRepeat' ).cbrepeat();", 'cbrepeat' );
					}

					$dummyUser					=	clone $user; // using a dummy user object to avoid real user object being altered during grouped field parsing

					$dummyUser->set( '_isFieldGroup', true );

					$fieldGroups				=	CBFieldGroups::getGroupedFields( $field, $dummyUser, $reason );

					$repeatLabel				=	CBTxt::T( $field->params->get( 'repeat_label', null, GetterInterface::STRING ) );
					/** @noinspection PhpUnusedLocalVariableInspection */
					$repeatOrdering				=	$field->params->get( 'repeat_ordering', false, GetterInterface::BOOLEAN );
					$repeatCount				=	$field->params->get( 'repeat_multiple', false, GetterInterface::BOOLEAN );
					/** @noinspection PhpUnusedLocalVariableInspection */
					$repeatMax					=	$field->params->get( 'repeat_limit', 5, GetterInterface::INT );

					if ( ! $repeatLabel ) {
						if ( $repeatCount ) {
							/** @noinspection PhpUnusedLocalVariableInspection */
							$repeatLabel		=	CBTxt::T( 'Add Rows' );
						} else {
							/** @noinspection PhpUnusedLocalVariableInspection */
							$repeatLabel		=	CBTxt::T( 'Add Row' );
						}
					}

					$rows						=	array();
					$rowsTitles					=	array();
					$rowsDescriptions			=	array();
					$rowsFields					=	array();
					$editorFields				=	array();

					foreach ( $fieldGroups as $row => $groupedFields ) {
						/** @var FieldTable[] $groupedFields */
						$cbUser					=	CBFieldGroups::mapGroupedColumns( $groupedFields, $dummyUser );
						$titles					=	array();
						$descriptions			=	array();
						$fields					=	array();
						$extras					=	array();

						foreach ( $groupedFields as $groupedField ) {
							$title				=	$_PLUGINS->callField( $groupedField->get( 'type', null, GetterInterface::STRING ), 'getFieldTitle', array( &$groupedField, &$dummyUser, $output, $reason ), $groupedField );
							$description		=	$_PLUGINS->callField( $groupedField->get( 'type', null, GetterInterface::STRING ), 'getFieldDescription', array( &$groupedField, &$dummyUser, $output, $reason ), $groupedField );
							$edit				=	$_PLUGINS->callField( $groupedField->get( 'type', null, GetterInterface::STRING ), 'getFieldRow', array( &$groupedField, &$dummyUser, $output, $fieldFormatting, $reason, $list_compare_types ), $groupedField );

							ob_start();
							require CBFieldGroups::getTemplate( $template, 'edit_field' );
							$html				=	ob_get_contents();
							ob_end_clean();

							$groupedFieldName									=	$groupedField->get( '_name', null, GetterInterface::STRING );

							if ( trim( $html ) != '' ) {
								if ( $groupedField->get( 'type', null, GetterInterface::STRING ) == 'editorta' ) {
									$editorFields[$groupedFieldName]			=	$groupedField;
								}

								$titles[$groupedFieldName]						=	$title;
								$descriptions[$groupedFieldName]				=	$description;
								$fields[$groupedFieldName]						=	$html;
							}

							if ( $customTemplate ) {
								// Always build the substitutions even if empty:
								$extras['title_' . $groupedFieldName]			=	$title;
								$extras['description_' . $groupedFieldName]		=	$description;
								$extras['field_' . $groupedFieldName]			=	( trim( $html ) != '' ? $html : '' );
							}
						}

						if ( $customTemplate && $fields ) {
							// Replace the fields array that's about to be imploded with the single custom template usage:
							$fields				=	array();
							$fields[]			=	$cbUser->replaceUserVars( $customTemplate, false, false, $extras, false );
						}

						ob_start();
						require CBFieldGroups::getTemplate( $template, 'edit_row' );
						$html					=	ob_get_contents();
						ob_end_clean();

						if ( trim( $html ) != '' ) {
							$rows[]				=	$html;
							$rowsTitles[]		=	$titles;
							$rowsDescriptions[]	=	$descriptions;
							$rowsFields[]		=	$fields;
						}

						// Revert the cached user object change:
						$cbUser->_cbuser		=	$user;
					}

					$dummyUser->set( '_isFieldGroup', false );

					ob_start();
					require CBFieldGroups::getTemplate( $template, 'edit' );
					$html						=	ob_get_contents();
					ob_end_clean();

					if ( $editorFields ) {
						$js						=	null;

						if ( stripos( $html, 'tinymce' ) !== false ) {
							static $tinyMCE		=	0;

							if ( ! $tinyMCE++ ) {
								$js				.=	"$( '.cbRepeat.cbFieldGroup' ).on( 'cbrepeat.add', function() {"
												.		"try {"
												.			"Joomla.JoomlaTinyMCE.setupEditors();"
												.		"} catch ( e ) {}"
												.	"}).find( 'textarea.mce_editable' ).on( 'cloning', function() {"
												.		"try {"
												.			"tinymce.remove();"
												.		"} catch ( e ) {}"
												.	"});";
							}
						} elseif ( stripos( $html, 'codemirror' ) !== false ) {
							static $codeMirror	=	0;

							if ( ! $codeMirror++ ) {
								$js				.=	"$( '.cbRepeat.cbFieldGroup' ).on( 'cbrepeat.add', function() {"
												.		"try {"
												.			"$( this ).find( 'textarea.codemirror-source' ).each( function() {"
												.				"var input = $( this ).removeClass( 'codemirror-source' );"
												.				"var id = input.prop( 'id' );"
												.				"Joomla.editors.instances[id] = CodeMirror.fromTextArea( this, input.data( 'options' ) );"
												.			"});"
												.		"} catch ( e ) {}"
												.	"}).find( 'textarea[data-options]' ).on( 'cloning', function() {"
												.		"try {"
												.			"var id = $( this ).prop( 'id' );"
												.			"if ( typeof Joomla.editors.instances[id] != 'undefined' ) {"
												.				"Joomla.editors.instances[id].toTextArea();"
												.			"}"
												.			"$( this ).addClass( 'codemirror-source' );"
												.		"} catch ( e ) {}"
												.	"});";
							}
						} elseif ( stripos( $html, 'wf-editor' ) !== false ) {
							static $JCE			=	0;

							if ( ! $JCE++ ) {
								$js				.=	"var JCECloning = false;"
												.	"$( '.cbRepeat.cbFieldGroup' ).on( 'cbrepeat.add', function() {"
												.		"try {"
												.			"if ( ! JCECloning ) {"
												.				"JCECloning = true;"
												.				"$( 'textarea.wf-editor' ).each( function() {"
												.					"$( this ).val( WFEditor.getContent( $( this ).prop( 'id' ) ) );"
												.					"$( this ).insertAfter( $( this ).parent() );"
												.					"$( this ).siblings( '.wf-editor-container' ).remove();"
												.					"$( this ).show();"
												.				"});"
												.				"tinymce.editors = [];"
												.				"WFEditor.load();"
												.				"setTimeout( function() {"
												.					"JCECloning = false;"
												.				"}, 1000 );"
												.			"}"
												.		"} catch ( e ) {}"
												.	"});";
							}
						}

						$_CB_framework->outputCbJQuery( $js );
					}

					return $this->formatFieldValueLayout( $this->_formatFieldOutput( $fieldName, $html, $output, false ), $reason, $field, $user, false )
						   . $this->_fieldIconsHtml( $field, $user, $output, $reason, null, 'html', null, null, null, true, 0 );
				}
			default:
				return $this->_formatFieldOutput( $fieldName, $user->get( $fieldName, null, GetterInterface::RAW ), $output, false );
		}
	}

	/**
	 * Mutator:
	 * Prepares field data for saving to database (safe transfer from $postdata to $user)
	 * Override
	 *
	 * @param  FieldTable  $field
	 * @param  UserTable   $user      RETURNED populated: touch only variables related to saving this field (also when not validating for showing re-edit)
	 * @param  array       $postdata  Typically $_POST (but not necessarily), filtering required.
	 * @param  string      $reason    'edit' for save user edit, 'register' for save registration
	 */
	public function prepareFieldDataSave( &$field, &$user, &$postdata, $reason )
	{
		global $_PLUGINS;

		$this->_prepareFieldMetaSave( $field, $user, $postdata, $reason );

		$dummyPostData							=	$postdata;
		$dummyUser								=	clone $user; // using a dummy user object to avoid real user object being altered during grouped field parsing

		$dummyUser->set( '_isFieldGroup', true );

		foreach ( $field->getTableColumns() as $col ) {
			$oldValues							=	array();
			$existingValues						=	$user->get( $col, null, GetterInterface::RAW );

			if ( $existingValues ) {
				$existingValues					=	new Registry( $existingValues );
				$oldValues						=	$existingValues->asArray();
			}

			$fields								=	CBFieldGroups::getGroupedFields( $field, $dummyUser, $reason, $dummyPostData );
			$values								=	array();

			$_PLUGINS->trigger( 'fieldgroups_onBeforePrepareFieldDataSave', array( $field, &$fields, $oldValues, &$values, &$dummyUser, &$dummyPostData, $reason ) );

			// Make sure the fields posted are allowed and validated:
			foreach ( $fields as $i => $groupedFields ) {
				/** @var FieldTable[] $groupedFields */
				$cbUser							=	CBFieldGroups::mapGroupedColumns( $groupedFields, $dummyUser, $dummyPostData );

				foreach ( $groupedFields as $name => $groupedField ) {
					$_PLUGINS->callField( $groupedField->get( 'type', null, GetterInterface::STRING ), 'prepareFieldDataSave', array( &$groupedField, &$dummyUser, &$dummyPostData, $reason ), $groupedField );

					if ( count( $_PLUGINS->getErrorMSG( false ) ) ) {
						break;
					}

					// Validation and storage processing passed so lets update the values to be stored:
					foreach ( $groupedField->get( '_tablecolumns', array(), GetterInterface::RAW ) as $oldCol => $newCol ) {
						$newValue				=	$dummyUser->get( $newCol, null, GetterInterface::RAW );

						if ( $groupedField->get( 'type', null, GetterInterface::STRING ) == 'fieldgroup' ) {
							$newValue			=	json_decode( $newValue, true );
						}

						$values[$i][$oldCol]	=	$newValue; // Already sanitized by prepareFieldDataSave or getGroupedFields
					}
				}

				// Revert the postdata and cached user object change:
				$cbUser->_cbuser				=	$user;
			}

			$_PLUGINS->trigger( 'fieldgroups_onAfterPrepareFieldDataSave', array( $field, &$fields, $oldValues, &$values, &$user, &$dummyPostData, $reason ) );

			$value								=	json_encode( $values );

			if ( $this->validate( $field, $user, $col, $value, $dummyPostData, $reason ) ) {
				if ( isset( $user->$col ) && ( (string) $user->$col ) !== (string) $value ) {
					$this->_logFieldUpdate( $field, $user, $reason, $user->$col, $value );
				}
			}

			$user->$col							=	$value;
		}

		$dummyUser->set( '_isFieldGroup', false );
	}

	/**
	 * Mutator:
	 * Prepares field data commit
	 * Override
	 *
	 * @param  FieldTable  $field
	 * @param  UserTable   $user      RETURNED populated: touch only variables related to saving this field (also when not validating for showing re-edit)
	 * @param  array       $postdata  Typically $_POST (but not necessarily), filtering required.
	 * @param  string      $reason    'edit' for save user edit, 'register' for save registration
	 */
	public function commitFieldDataSave( &$field, &$user, &$postdata, $reason )
	{
		global $_PLUGINS;

		$dummyPostData							=	$postdata;
		$dummyUser								=	clone $user; // using a dummy user object to avoid real user object being altered during grouped field parsing

		$dummyUser->set( '_isFieldGroup', true );

		foreach ( $field->getTableColumns() as $col ) {
			$oldValues							=	array();
			$existingValues						=	$user->get( $col, null, GetterInterface::RAW );

			if ( $existingValues ) {
				$existingValues					=	new Registry( $existingValues );
				$oldValues						=	$existingValues->asArray();
			}

			$fields								=	CBFieldGroups::getGroupedFields( $field, $dummyUser, $reason );
			$values								=	array();

			$_PLUGINS->trigger( 'fieldgroups_onBeforeCommitFieldDataSave', array( $field, &$fields, $oldValues, &$values, &$dummyUser, &$dummyPostData, $reason ) );

			foreach ( $fields as $i => $groupedFields ) {
				/** @var FieldTable[] $groupedFields */
				$cbUser							=	CBFieldGroups::mapGroupedColumns( $groupedFields, $dummyUser, $dummyPostData );

				foreach ( $groupedFields as $name => $groupedField ) {
					$_PLUGINS->callField( $groupedField->get( 'type', null, GetterInterface::STRING ), 'commitFieldDataSave', array( &$groupedField, &$dummyUser, &$dummyPostData, $reason ), $groupedField );

					if ( count( $_PLUGINS->getErrorMSG( false ) ) ) {
						break;
					}

					// Validation and storage processing passed so lets update the values to be stored:
					foreach ( $groupedField->get( '_tablecolumns', array(), GetterInterface::RAW ) as $oldCol => $newCol ) {
						$newValue				=	$dummyUser->get( $newCol, null, GetterInterface::RAW );

						if ( $groupedField->get( 'type', null, GetterInterface::STRING ) == 'fieldgroup' ) {
							$newValue			=	json_decode( $newValue, true );
						}

						$values[$i][$oldCol]	=	$newValue; // Already sanitized by commitFieldDataSave, prepareFieldDataSave, or getGroupedFields
					}
				}

				// Revert the cached user object change:
				$cbUser->_cbuser				=	$user;
			}

			$_PLUGINS->trigger( 'fieldgroups_onAfterCommitFieldDataSave', array( $field, &$fields, $oldValues, &$values, &$user, &$dummyPostData, $reason ) );

			$value								=	json_encode( $values );

			// Only update if fields changed the values in their commitFieldDataSave behavior:
			if ( isset( $user->$col ) && ( (string) $user->$col ) !== (string) $value ) {
				if ( $this->validate( $field, $user, $col, $value, $dummyPostData, $reason ) ) {
					$this->_logFieldUpdate( $field, $user, $reason, $user->$col, $value );

					$user->$col					=	$value;
				}
			}
		}

		$dummyUser->set( '_isFieldGroup', false );
	}

	/**
	 * Mutator:
	 * Prepares field data rollback
	 * Override
	 *
	 * @param  FieldTable  $field
	 * @param  UserTable   $user      RETURNED populated: touch only variables related to saving this field (also when not validating for showing re-edit)
	 * @param  array       $postdata  Typically $_POST (but not necessarily), filtering required.
	 * @param  string      $reason    'edit' for save user edit, 'register' for save registration
	 */
	public function rollbackFieldDataSave( &$field, &$user, &$postdata, $reason )
	{
		global $_PLUGINS;

		$dummyPostData				=	$postdata;
		$dummyUser					=	clone $user; // using a dummy user object to avoid real user object being altered during grouped field parsing

		$dummyUser->set( '_isFieldGroup', true );

		/** @noinspection PhpUnusedLocalVariableInspection */
		foreach ( $field->getTableColumns() as $col ) {
			$fields					=	CBFieldGroups::getGroupedFields( $field, $dummyUser, $reason );

			foreach ( $fields as $i => $groupedFields ) {
				/** @var FieldTable[] $groupedFields */
				$cbUser				=	CBFieldGroups::mapGroupedColumns( $groupedFields, $dummyUser, $dummyPostData );

				foreach ( $groupedFields as $name => $groupedField ) {
					$_PLUGINS->callField( $groupedField->get( 'type', null, GetterInterface::STRING ), 'rollbackFieldDataSave', array( &$groupedField, &$dummyUser, &$dummyPostData, $reason ), $groupedField );

					if ( count( $_PLUGINS->getErrorMSG( false ) ) ) {
						break;
					}
				}

				// Revert the cached user object change:
				$cbUser->_cbuser	=	$user;
			}

			$_PLUGINS->trigger( 'fieldgroups_onAfterRollbackFieldDataSave', array( $field, &$fields, &$user, &$dummyPostData, $reason ) );
		}

		$dummyUser->set( '_isFieldGroup', false );
	}

	/**
	 * Finder:
	 * Prepares field data for saving to database (safe transfer from $postdata to $user)
	 * Override
	 *
	 * @param  FieldTable  $field
	 * @param  UserTable   $searchVals          RETURNED populated: touch only variables related to saving this field (also when not validating for showing re-edit)
	 * @param  array       $postdata            Typically $_POST (but not necessarily), filtering required.
	 * @param  int         $list_compare_types  IF reason == 'search' : 0 : simple 'is' search, 1 : advanced search with modes, 2 : simple 'any' search
	 * @param  string      $reason              'edit' for save user edit, 'register' for save registration
	 * @return \cbSqlQueryPart[]
	 */
	public function bindSearchCriteria( &$field, &$searchVals, &$postdata, $list_compare_types, $reason )
	{
		global $_PLUGINS;

		if ( ! CBFieldGroups::canSearch() ) {
			return array();
		}

		$fieldLimit								=	$field->params->get( 'repeat_limit', 5, GetterInterface::INT );

		if ( ! $fieldLimit ) {
			$fieldLimit							=	$field->params->get( 'repeat_search_limit', 5, GetterInterface::INT );
		}

		$user									=	\CBuser::getMyUserDataInstance();

		$dummyPostData							=	$postdata;
		$dummySearchVals						=	new \stdClass();
		$query									=	array();

		static $simplified						=	array();

		foreach ( $field->getTableColumns() as $col ) {
			$groupedName						=	CBFieldGroups::getGroupedName( $col );

			$sql								=	new \cbSqlQueryPart();
			$sql->tag							=	'json';
			$sql->name							=	$col;
			$sql->table							=	$field->get( 'table', null, GetterInterface::STRING );
			$sql->type							=	'sql:operator';
			$sql->operator						=	'OR';
			$sql->searchmode					=	'is';
			$sql->children						=	array();

			// Find all the post data relevant to this field group:
			foreach ( $postdata as $k => $v ) {
				if ( strpos( $k, $groupedName ) !== 0 ) {
					continue;
				}

				// Map to the original field as well to handle substitutions cases:
				$oldName						=	str_replace( $groupedName . '__0__', '', $k );

				$dummyPostData[$oldName]		=	$v;

				if ( ! $fieldLimit ) {
					continue;
				}

				// Map up to the field limit as well if we're using one:
				for ( $i = 1; $i <= ( $fieldLimit - 1 ); $i++ ) {
					$name						=	str_replace( $groupedName . '__0', $groupedName . '__' . $i, $k );

					$dummyPostData[$name]		=	$v;
				}
			}

			$paths								=	array();

			foreach ( CBFieldGroups::getGroupedFields( $field, $user, $reason, $dummyPostData ) as $i => $groupedFields ) {
				/** @var FieldTable[] $groupedFields */
				foreach ( $groupedFields as $name => $groupedField ) {
					if ( in_array( $name, $simplified ) ) {
						continue;
					}

					$groupedField->set( 'table', $field->get( 'table', null, GetterInterface::STRING ) );

					$groupedFieldSearches		=	$_PLUGINS->callField( $groupedField->get( 'type', null, GetterInterface::STRING ), 'bindSearchCriteria', array( &$groupedField, &$dummySearchVals, &$dummyPostData, $list_compare_types, $reason ), $groupedField );

					if ( count( $_PLUGINS->getErrorMSG( false ) ) ) {
						break;
					}

					if ( ! count( $groupedFieldSearches ) ) {
						continue;
					}

					if ( ! $fieldLimit ) {
						// No limit so we can't be specific this means we need to change to a wildcard based matching:
						foreach ( $groupedFieldSearches as $k => $groupedFieldSearch ) {
							switch ( $groupedFieldSearch->searchmode ) {
								case 'is':
									$groupedFieldSearches[$k]->searchmode	=	'any';
									break;
								case 'isnot':
									$groupedFieldSearches[$k]->searchmode	=	'anynot';
									break;
							}
						}
					}

					$sql->children				=	$groupedFieldSearches;

					if ( $groupedField->get( 'type', null, GetterInterface::STRING ) == 'fieldgroup' ) {
						if ( ! $groupedFieldSearches[0]->paths ) {
							continue;
						}

						$sql->addChildren( $groupedFieldSearches[0]->children );

						foreach( $groupedFieldSearches[0]->paths as $pathName => $path ) {
							$prefixPath				=	( $fieldLimit ? $i : '*' );

							if ( strpos( $path, '*' ) !== false ) {
								$prefixPath			=	'*';
							}

							$paths[$pathName]		=	'$[' . $prefixPath . '].' . $name . substr( $path, 1 );
						}
					} else {
						$jsonPath					=	( $fieldLimit ? $i : '*' );

						if ( ( ( count( $groupedFieldSearches ) == 1 )
							 && in_array( $groupedFieldSearches[0]->operator, array( '=', '!=', '<>', 'LIKE', 'NOT LIKE' ) ) )
							 || ( $jsonPath === '*' )
						) {
							$simplified[]			=	$name;

							$jsonPath				=	'*';
						}

						$groupedFieldName			=	$groupedField->get( 'name', null, GetterInterface::STRING );

						$sql->addChildren( $groupedFieldSearches );

						$paths[$groupedFieldName]	=	'$[' . $jsonPath . '].' . $name;
					}

					// Push the first rows data to $searchVals so redisplay displays correctly:
					if ( $i == 0 ) {
						foreach ( $dummySearchVals as $k => $v ) {
							$searchVals->$k		=	$v;
						}
					}
				}
			}

			if ( $paths ) {
				$sql->paths						=	$paths;

				$query[]						=	$sql;
			}
		}

		return $query;
	}
	
	/**
	 * Direct access to field for custom operations, like for Ajax
	 *
	 * WARNING: direct unchecked access, except if $user is set, then check well for the $reason ...
	 *
	 * @param  FieldTable  $field
	 * @param  UserTable    $user
	 * @param  array                 $postdata
	 * @param  string                $reason     'profile' for user profile view, 'edit' for profile edit, 'register' for registration, 'search' for searches (always public!)
	 * @return string                            Expected output.
	 */
	public function fieldClass( &$field, &$user, &$postdata, $reason )
	{
		$function					=	$this->input( 'function', null, GetterInterface::STRING );

		switch ( $function ) {
			case 'image_approve':
			case 'image_reject':
				$index				=	$this->input( 'index', null, GetterInterface::INT );
				$image				=	$this->input( 'image', null, GetterInterface::STRING );

				if ( ( $index === null ) || ( ! $image ) ) {
					break;
				}

				$values				=	new Registry( $user->get( $field->get( 'name', null, GetterInterface::STRING ), null, GetterInterface::RAW ) );

				if ( ! $values->has( $index . '.' . $image ) ) {
					break;
				}

				$approved			=	$values->get( $index . '.' . $image . 'approved', 0, GetterInterface::INT );

				if ( $approved ) {
					break;
				}

				if ( $function == 'image_reject' ) {
					$values->set( $index . '.' . $image, '' );
				}

				$values->set( $index . '.' . $image . 'approved', 1 );

				if ( ! $user->storeDatabaseValue( $field->get( 'name', null, GetterInterface::STRING ), $values->asJson() ) ) {
					cbRedirectToProfile( $user->get( 'id', 0, GetterInterface::INT ), $user->getError() );

					return false;
				}

				$cbNotification		=	new \cbNotification();

				if ( $function == 'image_reject' ) {
					$cbNotification->sendFromSystem( $user, CBTxt::T( 'UE_IMAGEREJECTED_SUB', 'Image Rejected' ), CBTxt::T( 'UE_IMAGEREJECTED_MSG', 'Your image has been rejected by a moderator. Please log in and submit a new image.' ) );
				} else {
					$cbNotification->sendFromSystem( $user, CBTxt::T( 'UE_IMAGEAPPROVED_SUB', 'Image Approved' ), CBTxt::T( 'UE_IMAGEAPPROVED_MSG', 'Your image has been approved by a moderator.' ) );
				}

				cbRedirectToProfile( $user->get( 'id', 0, GetterInterface::INT ), CBTxt::Th( 'UE_USERIMAGEMODERATED_SUCCESSFUL', 'User Image Successfully Moderated!' ) );

				return false;
		}

		parent::fieldClass( $field, $user, $postdata, $reason );

		return false;
	}
}