<?php
/**
* Community Builder (TM)
* @version $Id: $
* @package CommunityBuilder
* @copyright (C) 2004-2021 www.joomlapolis.com / Lightning MultiCom SA - and its licensors, all rights reserved
* @license http://www.gnu.org/licenses/old-licenses/gpl-2.0.html GNU/GPL version 2
*/

namespace CB\Plugin\ProfileUpdateLogger\Trigger;

use CB\Database\Table\FieldTable;
use CB\Plugin\ProfileUpdateLogger\CBProfileUpdateLogger;
use CBLib\Application\Application;
use CB\Database\Table\UserTable;
use CB\Plugin\ProfileUpdateLogger\Table\UpdateLogTable;
use CBLib\Language\CBTxt;

defined('CBLIB') or die();

class UserTrigger extends \cbPluginHandler
{

	/**
	 * Deletes update log when the user is deleted
	 *
	 * @param  UserTable $user
	 * @param  int       $status
	 */
	public function deleteLog( $user, $status )
	{
		global $_CB_database;

		if ( ! CBProfileUpdateLogger::getGlobalParams()->getBool( 'general_delete', true ) ) {
			return;
		}

		// Delete all profile update logs for this user:
		$query		=	'SELECT *'
					.	"\n FROM " . $_CB_database->NameQuote( '#__comprofiler_plug_pulogger' )
					.	"\n WHERE " . $_CB_database->NameQuote( 'profileid' ) . " = " . $user->getInt( 'id', 0 );
		$_CB_database->setQuery( $query );
		$updates	=	$_CB_database->loadObjectList( null, '\CB\Plugin\ProfileUpdateLogger\Table\UpdateLogTable', array( $_CB_database ) );

		/** @var UpdateLogTable[] $updates */
		foreach ( $updates as $update ) {
			$update->delete();
		}
	}

	/**
	 * Deletes update log when the user is deleted
	 *
	 * @param  UserTable $new
	 * @param  UserTable $user
	 * @param  UserTable $old
	 */
	public function logUpdates( $new, $user, $old )
	{
		global $_CB_database;

		static $fields				=	array();

		$params						=	CBProfileUpdateLogger::getGlobalParams();

		if ( ( ! $new->getInt( 'id', 0 ) )
			 || ( ! $old->getInt( 'id', 0 ) )
			 || ( in_array( $new->getInt( 'id', 0 ), cbToArrayOfInt( explode( ',', $params->getString( 'logging_exclude_users' ) ) ), true ) )
			 || ( Application::Application()->isClient( 'administrator' ) && ( ! $params->getBool( 'pulBackEndLogging', true ) ) )
		) {
			return;
		}

		$ignoreTabs					=	cbToArrayOfInt( explode( '|*|', $params->getString( 'logging_exclude_tabs' ) ) );
		$ignoreFields				=	array_merge( explode( '|*|', $params->getString( 'logging_exclude_fields' ) ), array( 'id', 'password', 'params' ) );
		$ignoreTypes				=	array_merge( explode( '|*|', $params->getString( 'logging_exclude_types' ) ), array( 'password' ) );

		$formatTitles				=	CBProfileUpdateLogger::getGlobalParams()->getBool( 'notify_format_names', false );
		$formatValues				=	CBProfileUpdateLogger::getGlobalParams()->getBool( 'notify_format_values', false );

		$fieldNames					=	array_unique( array_merge( array_keys( get_object_vars( $new ) ), array_keys( get_object_vars( $old ) ) ) );

		$oldEmail					=	null;
		$changed					=	array();
		$pending					=	0;

		foreach ( $fieldNames as $k => $fieldName ) {
			if ( ( substr( $fieldName, 0, 1 ) === '_' ) || in_array( $fieldName, $ignoreFields, true ) ) {
				unset( $fieldNames[$k] );
				continue;
			}

			$newValue				=	$new->getRaw( $fieldName );
			$oldValue				=	$old->getRaw( $fieldName );

			// Lets see if there was a change of any kind:
			if ( $newValue === $oldValue ) {
				continue;
			}

			// Lets make sure what we're checking against is even a field:
			if ( ! isset( $fields[$fieldName] ) ) {
				$field				=	new FieldTable();

				$field->load( array( 'name' => $fieldName ) );

				$fields[$fieldName]	=	$field;
			}

			$field					=	$fields[$fieldName];

			if ( ( ! $field->getInt( 'fieldid', 0 ) )
				 || in_array( $field->getString( 'type' ), $ignoreTypes, true )
				 || in_array( $field->getInt( 'tabid' ), $ignoreTabs, true )
			) {
				continue;
			}

			// Now lets do fieldtype specific matching:
			switch ( $field->getString( 'type' ) ) {
				case 'checkbox':
				case 'integer':
				case 'terms':
				case 'points':
				case 'counter':
					$newValue		=	$new->getInt( $fieldName, 0 );
					$oldValue		=	$old->getInt( $fieldName, 0 );
					break;
				case 'rating':
				case 'float':
					$newValue		=	$new->getFloat( $fieldName, 0 );
					$oldValue		=	$old->getFloat( $fieldName, 0 );
					break;
				case 'date':
					$newValue		=	$new->getString( $fieldName, '0000-00-00' );

					if ( ! $newValue ) {
						$newValue	=	'0000-00-00';
					}

					$oldValue		=	$old->getString( $fieldName, '0000-00-00' );

					if ( ! $oldValue ) {
						$oldValue	=	'0000-00-00';
					}
					break;
				case 'datetime':
					$newValue		=	$new->getString( $fieldName, '0000-00-00 00:00:00' );

					if ( ! $newValue ) {
						$newValue	=	'0000-00-00 00:00:00';
					}

					$oldValue		=	$old->getString( $fieldName, '0000-00-00 00:00:00' );

					if ( ! $oldValue ) {
						$oldValue	=	'0000-00-00 00:00:00';
					}
					break;
				case 'time':
					$newValue		=	$new->getString( $fieldName, '00:00:00' );

					if ( ! $newValue ) {
						$newValue	=	'00:00:00';
					}

					$oldValue		=	$old->getString( $fieldName, '00:00:00' );

					if ( ! $oldValue ) {
						$oldValue	=	'00:00:00';
					}
					break;
				case 'fieldgroup':
					if ( ! $newValue ) {
						$newValue	=	'[]';
					}

					if ( ! $oldValue ) {
						$oldValue	=	'[]';
					}

					if ( CBProfileUpdateLogger::isArraysEqual( json_decode( $oldValue, true ), json_decode( $newValue, true ) ) ) {
						continue 2;
					}
					break;
			}

			if ( $newValue !== $oldValue ) {
				if ( ! $new->getInt( $fieldName . 'approved', 1 ) ) {
					$pending++;
				}

				$log				=	new UpdateLogTable();

				$log->set( 'profileid', $new->getInt( 'id', 0 ) );
				$log->set( 'fieldname', $fieldName );
				$log->set( 'oldvalue', $oldValue );
				$log->set( 'newvalue', $newValue );

				$log->store();

				if ( $fieldName === 'email' ) {
					$oldEmail		=	$oldValue;
				}

				$changed[]			=	array(	'field'		=>	$log->getFieldName( $formatTitles ),
												'old'		=>	$log->getOldValue( $formatValues ),
												'new'		=>	$log->getNewValue( $formatValues ),
											);
			}
		}

		if ( $changed && $params->getBool( 'pulEnableNotifications', true ) && ( ( ! Application::Application()->isClient( 'administrator' ) ) || ( Application::Application()->isClient( 'administrator' ) && $params->getBool( 'pulBackendNotifications', false ) ) ) ) {
			$myUser					=	\CBuser::getMyUserDataInstance();
			$cbUser					=	\CBuser::getInstance( $new->getInt( 'id', 0 ), false );
			$extras					=	array(	'editedbyid'	=>	$myUser->getInt( 'id', 0 ),
												'editedbyip'	=>	$myUser->getString( 'editedbyip' ),
												'changed'		=>	count( $changed ),
												'pending'		=>	$pending,
												'date'			=>	cbFormatDate( Application::Database()->getUtcDateTime() )
											);

			$changes				=	null;

			foreach ( $changed as $change ) {
				$changes			.=	$cbUser->replaceUserVars( CBTxt::T( '<p><strong>[field]:</strong> "[old]" to "[new]"</p>' ), false, false, array_merge( $extras, $change ), false );
			}

			if ( $myUser->getInt( 'id', 0 ) === $new->getInt( 'id', 0 ) ) {
				$subject			=	CBTxt::T( '[formatname] has updated their profile!' );
				$body				=	CBTxt::T( '[cb:userfield field="formatname" reason="list" /] has updated their profile. Changed: [changed]. Pending Changes: [pending].<p>[changes]</p>' );
			} else {
				$subject			=	CBTxt::T( '[cb:userdata field="formatname" user="[editedbyid]" /] has updated the profile of [formatname]!' );
				$body				=	CBTxt::T( '[cb:userfield field="formatname" user="[editedbyid]" reason="list" /] has updated the profile of [cb:userfield field="formatname" reason="list" /]. Changed: [changed]. Pending Changes: [pending].<p>[changes]</p>' );
			}

			$extras['changes']		=	$changes;

			$subject				=	$cbUser->replaceUserVars( $subject, true, false, $extras, false );
			$body					=	$cbUser->replaceUserVars( $body, false, false, $extras, false );

			$cbNotification			=	new \cbNotification( );

			switch ( $params->getInt( 'pulNotificationList', 0 ) ) {
				case 3: // Users Self
					if ( $myUser->getInt( 'id', 0 ) !== $new->getInt( 'id', 0 ) ) {
						if ( $oldEmail ) {
							// Email changed so lets send the notification to the old email address as well
							$cbNotification->sendFromSystem( $oldEmail, $subject, $body, false, true );
						}

						$cbNotification->sendFromSystem( $new->getInt( 'id', 0 ), $subject, $body, false, true );
					}
					break;
				case 2: // Specific Users
					$recipients		=	$params->getString( 'pulNotificationRecipientList' );

					if ( ! $recipients ) {
						return;
					}

					$recipients		=	cbToArrayOfInt( explode( ',', $recipients ) );

					foreach ( $recipients as $recipient ) {
						if ( $recipient === $new->getInt( 'id', 0 ) ) {
							continue;
						}

						$cbNotification->sendFromSystem( $recipient, $subject, $body, false, true );
					}
					break;
				case 4: // Specific Email Addresses
					$recipients		=	$params->getString( 'notify_email_addresses' );

					if ( ! $recipients ) {
						return;
					}

					$recipients		=	explode( ',', $recipients );

					foreach ( $recipients as $recipient ) {
						if ( $recipient === $new->getString( 'email' ) ) {
							continue;
						}

						$cbNotification->sendFromSystem( $recipient, $subject, $body, false, true );
					}
					break;
				case 1: // View Access Level
					$usergroups		=	Application::CmsPermissions()->getGroupsOfViewAccessLevel( $params->getInt( 'pulNotificationAclList', 6 ), true );

					if ( ! $usergroups ) {
						return;
					}

					$query			=	'SELECT DISTINCT u.' . $_CB_database->NameQuote( 'id' )
									.	"\n FROM " . $_CB_database->NameQuote( '#__users' ) . " AS u"
									.	"\n INNER JOIN " . $_CB_database->NameQuote( '#__comprofiler' ) . " AS c"
									.	' ON c.' . $_CB_database->NameQuote( 'id' ) . ' = u.' . $_CB_database->NameQuote( 'id' )
									.	"\n INNER JOIN " . $_CB_database->NameQuote( '#__user_usergroup_map' ) . " AS g"
									.	' ON g.' . $_CB_database->NameQuote( 'user_id' ) . ' = c.' . $_CB_database->NameQuote( 'id' )
									.	"\n WHERE g." . $_CB_database->NameQuote( 'group_id' ) . " IN " . $_CB_database->safeArrayOfIntegers( $usergroups )
									.	"\n AND u." . $_CB_database->NameQuote( 'block' ) . " = 0"
									.	"\n AND c." . $_CB_database->NameQuote( 'confirmed' ) . " = 1"
									.	"\n AND c." . $_CB_database->NameQuote( 'approved' ) . " = 1";
					$_CB_database->setQuery( $query );
					$recipients		=	cbToArrayOfInt( $_CB_database->loadResultArray() );

					if ( ! $recipients ) {
						return;
					}

					foreach ( $recipients as $recipient ) {
						if ( $recipient === $new->getInt( 'id', 0 ) ) {
							continue;
						}

						$cbNotification->sendFromSystem( $recipient, $subject, $body, false, true );
					}
					break;
				case 0: // Moderators
				default:
					$cbNotification->sendToModerators( $subject, $body, false, true );
					break;
			}

			// Also notify users self:
			if ( ( $params->getInt( 'pulNotificationList', 0 ) !== 3 ) && $params->getBool( 'pulNotificationSelf', false ) && ( $myUser->getInt( 'id', 0 ) !== $new->getInt( 'id', 0 ) ) ) {
				if ( $oldEmail ) {
					// Email changed so lets send the notification to the old email address as well
					$cbNotification->sendFromSystem( $oldEmail, $subject, $body, false, true );
				}

				$cbNotification->sendFromSystem( $new->getInt( 'id', 0 ), $subject, $body, false, true );
			}
		}
	}
}