<?php
/**
* Community Builder (TM)
* @version $Id: $
* @package CommunityBuilder
* @copyright (C) 2004-2019 www.joomlapolis.com / Lightning MultiCom SA - and its licensors, all rights reserved
* @license http://www.gnu.org/licenses/old-licenses/gpl-2.0.html GNU/GPL version 2
*/

namespace CB\Plugin\Quickstart;

use CB\Database\Table\ListTable;
use CB\Database\Table\PluginTable;
use CB\Database\Table\FieldTable;
use CB\Database\Table\TabTable;
use CBLib\Application\Application;
use CBLib\Registry\Registry;
use CBLib\Registry\GetterInterface;
use CBLib\Language\CBTxt;

defined('CBLIB') or die();

class Installer
{
	/** @var PluginTable[]  */
	static private $plugins			=	array(); // array of plugin objects by plugin element
	/** @var TabTable[]  */
	static private $tabs			=	array(); // array of tab objects by tab class
	/** @var int[]  */
	static private $tabOrder		=	array(); // incrementing id of order based off position: array( tab_position => int ); e.g. array( 'register' => 1, 'edit' => 2, 'not_on_profile_2' => 1 )
	/** @var FieldTable[]  */
	static private $fields			=	array(); // array of field objects by field name
	/** @var int[]  */
	static private $fieldOrder		=	array(); // incrementing id of order based off tab: array( tab_id => int ); e.g. array( 10 => 1, 11 => 2, 12 => 1 )

	static public function install()
	{
		global $_CB_framework, $_CB_database;

		// Default ACL:
		$permissions			=	Application::CmsPermissions();
		$modAccess				=	Application::Config()->get( 'moderator_viewaccesslevel', 3, GetterInterface::INT );
		$regAccess				=	2;
		$guestAccess			=	1;

		// Try to find the view access level for registered users and guest users based off configuration:
		foreach ( array_keys( $permissions->getAllViewAccessLevels() ) as $viewAccessLevel ) {
			$userGroups			=	$permissions->getGroupsOfViewAccessLevel( $viewAccessLevel, true );

			sort( $userGroups, SORT_NUMERIC );

			if ( $userGroups === array( (int) $_CB_framework->getCfg( 'new_usertype' ) ) ) {
				$regAccess		=	$viewAccessLevel;
			} elseif ( $userGroups === array( (int) $_CB_framework->getCfg( 'guest_usergroup' ) ) ) {
				$guestAccess	=	$viewAccessLevel;
			}
		}

		// Configuration:
		$corePlugin				=	new PluginTable();

		$corePlugin->load( 1 );

		if ( $corePlugin->get( 'id', 0, GetterInterface::INT ) ) {
			$configuration		=	new Registry( $corePlugin->get( 'params', null, GetterInterface::RAW ) );

			$configuration->set( 'allow_email_display', 1 );
			$configuration->set( 'allow_email_replyto', 2 );
			$configuration->set( 'calendar_type', 3 );
			$configuration->set( 'icons_display', 1 );
			$configuration->set( 'poweredBy', 0 );
			$configuration->set( 'reg_admin_allowcbregistration', 1 );
			$configuration->set( 'profile_viewaccesslevel', 1 );
			$configuration->set( 'showEmptyTabs', 0 );
			$configuration->set( 'showEmptyFields', 0 );
			$configuration->set( 'emptyFieldsText', '' );
			$configuration->set( 'frontend_userparams', 0 );
			$configuration->set( 'conNotifyType', 0 ); // Notifications handled by CB Activity

			$existingTypes		=	array_filter( explode( "\n", $configuration->get( 'connection_categories', null, GetterInterface::STRING ) ) );

			$newTypes			=	array(	'CONN_TYPE_ACQUAINTANCE'				=>	'Acquaintance', // CBTxt::T( 'CONN_TYPE_ACQUAINTANCE', 'Acquaintance' )
											'CONN_TYPE_ANCESTOR'					=>	'Ancestor', // CBTxt::T( 'CONN_TYPE_ANCESTOR', 'Ancestor' )
											'CONN_TYPE_AUNT'						=>	'Aunt', // CBTxt::T( 'CONN_TYPE_AUNT', 'Aunt' )
											'CONN_TYPE_BESTFRIEND'					=>	'Best friend', // CBTxt::T( 'CONN_TYPE_BESTFRIEND', 'Best friend' )
											'CONN_TYPE_BOSS'						=>	'Boss', // CBTxt::T( 'CONN_TYPE_BOSS', 'Boss' )
											'CONN_TYPE_BROTHER'						=>	'Brother', // CBTxt::T( 'CONN_TYPE_BROTHER', 'Brother' )
											'CONN_TYPE_BROTHERINLAW'				=>	'Brother-in-law', // CBTxt::T( 'CONN_TYPE_BROTHERINLAW', 'Brother-in-law' )
											'CONN_TYPE_CHILDGENDERNEUTRAL'			=>	'Child (gender neutral)', // CBTxt::T( 'CONN_TYPE_CHILDGENDERNEUTRAL', 'Child (gender neutral)' )
											'CONN_TYPE_CHILDOFSIBLINGGENDERNEUTRAL'	=>	'Child of Sibling (gender neutral)', // CBTxt::T( 'CONN_TYPE_CHILDOFSIBLINGGENDERNEUTRAL', 'Child of Sibling (gender neutral)' )
											'CONN_TYPE_CHILDINLAWGENDERNEUTRAL'		=>	'Child-in-law (gender neutral)', // CBTxt::T( 'CONN_TYPE_CHILDINLAWGENDERNEUTRAL', 'Child-in-law (gender neutral)' )
											'CONN_TYPE_CLOSEFRIEND'					=>	'Close friend', // CBTxt::T( 'CONN_TYPE_CLOSEFRIEND', 'Close friend' )
											'CONN_TYPE_COWORKER'					=>	'Co-worker', // CBTxt::T( 'CONN_TYPE_COWORKER', 'Co-worker' )
											'CONN_TYPE_COUSINFEMALE'				=>	'Cousin (female)', // CBTxt::T( 'CONN_TYPE_COUSINFEMALE', 'Cousin (female)' )
											'CONN_TYPE_COUSINGENDERNEUTRAL'			=>	'Cousin (gender neutral)', // CBTxt::T( 'CONN_TYPE_COUSINGENDERNEUTRAL', 'Cousin (gender neutral)' )
											'CONN_TYPE_COUSINMALE'					=>	'Cousin (male)', // CBTxt::T( 'CONN_TYPE_COUSINMALE', 'Cousin (male)' )
											'CONN_TYPE_DAUGHTER'					=>	'Daughter', // CBTxt::T( 'CONN_TYPE_DAUGHTER', 'Daughter' )
											'CONN_TYPE_DAUGHTERINLAW'				=>	'Daughter-in-law', // CBTxt::T( 'CONN_TYPE_DAUGHTERINLAW', 'Daughter-in-law' )
											'CONN_TYPE_FATHER'						=>	'Father', // CBTxt::T( 'CONN_TYPE_FATHER', 'Father' )
											'CONN_TYPE_FATHERINLAW'					=>	'Father-in-law', // CBTxt::T( 'CONN_TYPE_FATHERINLAW', 'Father-in-law' )
											'CONN_TYPE_FRIEND'						=>	'Friend', // CBTxt::T( 'CONN_TYPE_FRIEND', 'Friend' )
											'CONN_TYPE_GRANDCHILDGENDERNEUTRAL'		=>	'Grandchild (gender neutral)', // CBTxt::T( 'CONN_TYPE_GRANDCHILDGENDERNEUTRAL', 'Grandchild (gender neutral)' )
											'CONN_TYPE_GRANDDAUGHTER'				=>	'Granddaughter', // CBTxt::T( 'CONN_TYPE_GRANDDAUGHTER', 'Granddaughter' )
											'CONN_TYPE_GRANDFATHER'					=>	'Grandfather', // CBTxt::T( 'CONN_TYPE_GRANDFATHER', 'Grandfather' )
											'CONN_TYPE_GRANDMOTHER'					=>	'Grandmother', // CBTxt::T( 'CONN_TYPE_GRANDMOTHER', 'Grandmother' )
											'CONN_TYPE_GRANDPARENTGENDERNEUTRAL'	=>	'Grandparent (gender neutral)', // CBTxt::T( 'CONN_TYPE_GRANDPARENTGENDERNEUTRAL', 'Grandparent (gender neutral)' )
											'CONN_TYPE_GRANDSON'					=>	'Grandson', // CBTxt::T( 'CONN_TYPE_GRANDSON', 'Grandson' )
											'CONN_TYPE_MOTHER'						=>	'Mother', // CBTxt::T( 'CONN_TYPE_MOTHER', 'Mother' )
											'CONN_TYPE_MOTHERINLAW'					=>	'Mother-in-law', // CBTxt::T( 'CONN_TYPE_MOTHERINLAW', 'Mother-in-law' )
											'CONN_TYPE_NEPHEW'						=>	'Nephew', // CBTxt::T( 'CONN_TYPE_NEPHEW', 'Nephew' )
											'CONN_TYPE_NIECE'						=>	'Niece', // CBTxt::T( 'CONN_TYPE_NIECE', 'Niece' )
											'CONN_TYPE_PARENTGENDERNEUTRAL'			=>	'Parent (gender neutral)', // CBTxt::T( 'CONN_TYPE_PARENTGENDERNEUTRAL', 'Parent (gender neutral)' )
											'CONN_TYPE_PARENTINLAWGENDERNEUTRAL'	=>	'Parent-in-law (gender neutral)', // CBTxt::T( 'CONN_TYPE_PARENTINLAWGENDERNEUTRAL', 'Parent-in-law (gender neutral)' )
											'CONN_TYPE_PETGENDERNEUTRAL'			=>	'Pet (gender neutral)', // CBTxt::T( 'CONN_TYPE_PETGENDERNEUTRAL', 'Pet (gender neutral)' )
											'CONN_TYPE_RELATIVE'					=>	'Relative', // CBTxt::T( 'CONN_TYPE_RELATIVE', 'Relative' )
											'CONN_TYPE_SIBLINGGENDERNEUTRAL'		=>	'Sibling (gender neutral)', // CBTxt::T( 'CONN_TYPE_SIBLINGGENDERNEUTRAL', 'Sibling (gender neutral)' )
											'CONN_TYPE_SIBLINGOFPARENTGENDERNEUTRAL'=>	'Sibling of Parent (gender neutral)', // CBTxt::T( 'CONN_TYPE_SIBLINGOFPARENTGENDERNEUTRAL', 'Sibling of Parent (gender neutral)' )
											'CONN_TYPE_SIBLINGINLAWGENDERNEUTRAL'	=>	'Sibling-in-law (gender neutral)', // CBTxt::T( 'CONN_TYPE_SIBLINGINLAWGENDERNEUTRAL', 'Sibling-in-law (gender neutral)' )
											'CONN_TYPE_SISTER'						=>	'Sister', // CBTxt::T( 'CONN_TYPE_SISTER', 'Sister' )
											'CONN_TYPE_SISTERINLAW'					=>	'Sister-in-law', // CBTxt::T( 'CONN_TYPE_SISTERINLAW', 'Sister-in-law' )
											'CONN_TYPE_SON'							=>	'Son', // CBTxt::T( 'CONN_TYPE_SON', 'Son' )
											'CONN_TYPE_SONINLAW'					=>	'Son-in-law', // CBTxt::T( 'CONN_TYPE_SONINLAW', 'Son-in-law' )
											'CONN_TYPE_STEPCHILDGENDERNEUTRAL'		=>	'Step Child (gender neutral)', // CBTxt::T( 'CONN_TYPE_STEPCHILDGENDERNEUTRAL', 'Step Child (gender neutral)' )
											'CONN_TYPE_STEPPARENTGENDERNEUTRAL'		=>	'Step Parent (gender neutral)', // CBTxt::T( 'CONN_TYPE_STEPPARENTGENDERNEUTRAL', 'Step Parent (gender neutral)' )
											'CONN_TYPE_STEPSIBLINGGENDERNEUTRAL'	=>	'Step Sibling (gender neutral)', // CBTxt::T( 'CONN_TYPE_STEPSIBLINGGENDERNEUTRAL', 'Step Sibling (gender neutral)' )
											'CONN_TYPE_STEPBROTHER'					=>	'Stepbrother', // CBTxt::T( 'CONN_TYPE_STEPBROTHER', 'Stepbrother' )
											'CONN_TYPE_STEPDAUGHTER'				=>	'Stepdaughter', // CBTxt::T( 'CONN_TYPE_STEPDAUGHTER', 'Stepdaughter' )
											'CONN_TYPE_STEPFATHER'					=>	'Stepfather', // CBTxt::T( 'CONN_TYPE_STEPFATHER', 'Stepfather' )
											'CONN_TYPE_STEPMOTHER'					=>	'Stepmother', // CBTxt::T( 'CONN_TYPE_STEPMOTHER', 'Stepmother' )
											'CONN_TYPE_STEPSISTER'					=>	'Stepsister', // CBTxt::T( 'CONN_TYPE_STEPSISTER', 'Stepsister' )
											'CONN_TYPE_STEPSON'						=>	'Stepson', // CBTxt::T( 'CONN_TYPE_STEPSON', 'Stepson' )
											'CONN_TYPE_SUPERVISOR'					=>	'Supervisor', // CBTxt::T( 'CONN_TYPE_SUPERVISOR', 'Supervisor' )
											'CONN_TYPE_UNCLE'						=>	'Uncle' // CBTxt::T( 'CONN_TYPE_UNCLE', 'Uncle' )
										);

			$connTypes			=	array();

			foreach ( $newTypes as $k => $v ) {
				if ( in_array( $k, $existingTypes ) || in_array( $v, $existingTypes ) ) {
					continue;
				}

				$connTypes[$k]	=	$v;
			}

			if ( $connTypes ) {
				$configuration->set( 'connection_categories', implode( "\n", array_unique( array_merge( $existingTypes, array_keys( $connTypes ) ) ) ) );
			}

			$corePlugin->set( 'params', $configuration->asJson() );

			$corePlugin->store();
		}

		// CBTxt::T( 'PROFILE_FEATURES_PRIVACY', '<div class="bg-light text-center col-form-label">Features</div>' )

		/**
		 * Tab Edit Order
		 * 1. Account
		 * 2. Details
		 * 3. Canvas
		 * 4. Avatar
		 * 5. Notifications
		 * 6. Status
		 * 7. Hidden
		 * 8. Privacy
		 *
		 * Tab Profile Order
		 * 1. Activity
		 * 2. About
		 * 3. Connections
		 * 4. Gallery
		 * 5. Groups
		 *
		 * Tab Profile Nested Order
		 * 1. Articles
		 * 2. Blogs
		 * 3. Invites
		 * 4. Forums
		 */

		// Hidden Tab (used for strictly substitution only fields):
		$hiddenTab				=	new TabTable();

		$hiddenTab->load( array( 'title' => 'PROFILE_TAB_HIDDEN' ) );

		if ( ! $hiddenTab->get( 'tabid', 0, GetterInterface::INT ) ) {
			$hiddenTab->set( 'enabled', 1 );
		}

		$hiddenTab->set( 'title', 'PROFILE_TAB_HIDDEN' ); // CBTxt::T( 'PROFILE_TAB_HIDDEN', 'Hidden' )
		$hiddenTab->set( 'viewaccesslevel', 1 );
		$hiddenTab->set( 'displaytype', 'html' );
		$hiddenTab->set( 'position', 'not_on_profile_2' );
		$hiddenTab->set( 'fields', 1 );
		$hiddenTab->set( 'ordering', 1 );
		$hiddenTab->set( 'ordering_edit', 7 );
		$hiddenTab->set( 'ordering_register', 7 );

		$hiddenTab->store();

		self::setTab( 'hidden', $hiddenTab );

		// Account (Contact Info) Tab:
		$accountTab			=	new TabTable();

		$accountTab->load( array( 'pluginclass' => 'getContactTab' ) );

		if ( $accountTab->get( 'tabid', 0, GetterInterface::INT ) ) {
			$accountTab->set( 'title', 'PROFILE_TAB_ACCOUNT' ); // CBTxt::T( 'PROFILE_TAB_ACCOUNT', 'Account' )
			$accountTab->set( 'displaytype', 'menu' );
			$accountTab->set( 'position', 'not_on_profile_1' );
			$accountTab->set( 'ordering', 1 );
			$accountTab->set( 'ordering_edit', 1 );
			$accountTab->set( 'ordering_register', 1 );

			$accountTab->store();
		}

		self::setTab( 'account', $accountTab );

		// Details Tab:
		$detailsTab				=	new TabTable();

		$detailsTab->load( array( 'title' => 'PROFILE_TAB_DETAILS' ) );

		if ( ! $detailsTab->get( 'tabid', 0, GetterInterface::INT ) ) {
			$detailsTab->set( 'enabled', 1 );
		}

		$detailsTab->set( 'title', 'PROFILE_TAB_DETAILS' ); // CBTxt::T( 'PROFILE_TAB_DETAILS', 'Details' )
		$detailsTab->set( 'viewaccesslevel', 1 );
		$detailsTab->set( 'displaytype', 'menu' );
		$detailsTab->set( 'position', 'not_on_profile_1' );
		$detailsTab->set( 'fields', 1 );
		$detailsTab->set( 'ordering', 2 );
		$detailsTab->set( 'ordering_edit', 2 );
		$detailsTab->set( 'ordering_register', 2 );

		$detailsTab->store();

		self::setTab( 'details', $detailsTab );

		// Actions Tab:
		$actionsTab				=	new TabTable();

		$actionsTab->load( array( 'title' => 'PROFILE_TAB_ACTIONS' ) );

		if ( ! $actionsTab->get( 'tabid', 0, GetterInterface::INT ) ) {
			$actionsTab->set( 'enabled', 1 );
		}

		$actionsTab->set( 'title', 'PROFILE_TAB_ACTIONS' ); // CBTxt::T( 'PROFILE_TAB_ACTIONS', 'Actions' )
		$actionsTab->set( 'viewaccesslevel', 1 );
		$actionsTab->set( 'displaytype', 'html' );
		$actionsTab->set( 'position', 'canvas_info' );
		$actionsTab->set( 'fields', 1 );
		$actionsTab->set( 'ordering', 1 );
		$actionsTab->set( 'ordering_edit', 99 );
		$actionsTab->set( 'ordering_register', 99 );

		$actionsTab->store();

		self::setTab( 'actions', $actionsTab );

		// Notifications Tab:
		$notifyTab				=	new TabTable();

		$notifyTab->load( array( 'title' => 'PROFILE_TAB_NOTIFICATIONS' ) );

		if ( ! $notifyTab->get( 'tabid', 0, GetterInterface::INT ) ) {
			$notifyTab->set( 'enabled', 1 );
		}

		$notifyTab->set( 'title', 'PROFILE_TAB_NOTIFICATIONS' ); // CBTxt::T( 'PROFILE_TAB_NOTIFICATIONS', 'Notifications' )
		$notifyTab->set( 'viewaccesslevel', 1 );
		$notifyTab->set( 'displaytype', 'menu' );
		$notifyTab->set( 'position', 'not_on_profile_1' );
		$notifyTab->set( 'fields', 1 );
		$notifyTab->set( 'ordering', 3 );
		$notifyTab->set( 'ordering_edit', 5 );
		$notifyTab->set( 'ordering_register', 5 );

		$notifyTab->store();

		self::setTab( 'notifications', $notifyTab );

		// Status Tab:
		$statusTab				=	new TabTable();

		$statusTab->load( array( 'pluginclass' => 'getStatusTab' ) );

		if ( $statusTab->get( 'tabid', 0, GetterInterface::INT ) ) {
			$statusTab->set( 'displaytype', 'html' );
			$statusTab->set( 'position', 'not_on_profile_1' );
			$statusTab->set( 'ordering', 4 );
			$statusTab->set( 'ordering_edit', 6 );
			$statusTab->set( 'ordering_register', 6 );

			$statusTab->store();
		}

		self::setTab( 'status', $statusTab );

		// Stats Tab:
		$statsTab				=	new TabTable();

		$statsTab->load( array( 'pluginclass' => 'getStatsTab' ) );

		self::setTab( 'stats', $statsTab );

		// About Tab:
		$aboutTab				=	new TabTable();

		$aboutTab->load( array( 'title' => 'PROFILE_TAB_ABOUT' ) );

		if ( ! $aboutTab->get( 'tabid', 0, GetterInterface::INT ) ) {
			$aboutTab->set( 'enabled', 1 );
		}

		$aboutTab->set( 'title', 'PROFILE_TAB_ABOUT' ); // CBTxt::T( 'PROFILE_TAB_ABOUT', 'About' )
		$aboutTab->set( 'viewaccesslevel', 1 );
		$aboutTab->set( 'displaytype', 'menu' );
		$aboutTab->set( 'position', 'canvas_main_middle' );
		$aboutTab->set( 'fields', 1 );
		$aboutTab->set( 'ordering', 2 );
		$aboutTab->set( 'ordering_edit', 99 ); // last as this SHOULDN'T contain editable fields
		$aboutTab->set( 'ordering_register', 99 ); // last as this SHOULDN'T contain editable fields

		$aboutTab->store();

		self::setTab( 'about', $aboutTab );

		// About Field:
		$aboutField				=	new FieldTable();

		$aboutField->load( array( 'name' => 'cb_profile_about' ) );

		if ( ! $aboutField->get( 'fieldid', 0, GetterInterface::INT ) ) {
			$aboutField->set( 'published', 1 );
		}

		$aboutHTML				=	'[cb:if reason="profile" or reason="list"]'
								.	'<div class="row no-gutters">'
								.		'<div class="col-sm">[cb:usertab tab="' . self::getTab( 'details' )->get( 'tabid', 0, GetterInterface::INT ) . '" formatting="divs" /]</div>'
								.		'<div class="col-sm-3 pl-sm-2 text-right">[cb:usertab tab="' . self::getTab( 'status' )->get( 'tabid', 0, GetterInterface::INT ) . '" formatting="divs" /]</div>'
								.	'</div>'
								.	'[/cb:if]';

		$aboutField->set( 'name', 'cb_profile_about' );
		$aboutField->set( 'title', 'PROFILE_FIELD_ABOUT' ); // CBTxt::T( 'PROFILE_FIELD_ABOUT', 'About' )
		$aboutField->set( 'description', $aboutHTML );
		$aboutField->set( 'type', 'delimiter' );
		$aboutField->set( 'tabid', self::getTab( 'about' )->get( 'tabid', 0, GetterInterface::INT ) );
		$aboutField->set( 'edit', 0 );
		$aboutField->set( 'registration', 0 );
		$aboutField->set( 'profile', 4 );
		$aboutField->set( 'ordering', self::getFieldOrder( $aboutField->get( 'tabid', 0, GetterInterface::INT ) ) );

		$aboutField->store();

		self::setField( 'cb_profile_about', $aboutField );

		// Connections Tab:
		$connTab					=	new TabTable();

		$connTab->load( array( 'pluginclass' => 'getConnectionTab' ) );

		if ( $connTab->get( 'tabid', 0, GetterInterface::INT ) ) {
			$connTabParams			=	new Registry( $connTab->get( 'params', null, GetterInterface::RAW ) );

			$connTabParams->set( 'cbprivacy_display', 1 );
			$connTabParams->set( 'privacy_options_default', 0 );
			$connTabParams->set( 'cbprivacy_simple_group', 'PROFILE_FEATURES_PRIVACY' );

			$connTab->set( 'displaytype', 'menu' );
			$connTab->set( 'position', 'canvas_main_middle' );
			$connTab->set( 'ordering', 3 );
			$connTab->set( 'ordering_edit', 99 ); // last as this SHOULDN'T contain editable fields
			$connTab->set( 'ordering_register', 99 ); // last as this SHOULDN'T contain editable fields
			$connTab->set( 'params', $connTabParams->asJson() );

			$connTab->store();
		}

		self::setTab( 'cbconnections', $connTab );

		// Request Connection Button:
		$connectField			=	new FieldTable();

		$connectField->load( array( 'name' => 'cb_connect_button' ) );

		if ( ! $connectField->get( 'fieldid', 0, GetterInterface::INT ) ) {
			$connectField->set( 'published', 1 );
		}

		$connectFieldParams		=	new Registry( $connectField->get( 'params', null, GetterInterface::RAW ) );

		$conditions				=	array( array( 'condition' => array( array(	'field'						=>	'customconnections',
																				'user_connections'			=>	'viewer',
																				'operator_connections'		=>	1,
																				'locations'					=>	'profile|*|list'
																			),
																		array(	'field'						=>	'customusers',
																				'user_users'				=>	'displayed',
																				'operator_users'			=>	13,
																				'field_users'				=>	'[cb:userdata field="user_id" user="#me" /]',
																				'locations'					=>	'profile|*|list'
																			),
																		array(	'field'						=>	'customvalue',
																				'field_custom'				=>	'[cb:userdata field="user_id" user="#me" /]',
																				'operator'					=>	2,
																				'value'						=>	0,
																				'locations'					=>	'profile|*|list'
																			)
																		)
												)
										);

		$connectFieldParams->set( 'cbconditional_conditioned', 1 );
		$connectFieldParams->set( 'cbconditional_conditions', $conditions );

		$connectField->set( 'name', 'cb_connect_button' );
		$connectField->set( 'title', 'PROFILE_FIELD_CONNECT' ); // CBTxt::T( 'PROFILE_FIELD_CONNECT', 'Connect' )
		$connectField->set( 'description', 'REQUEST_CONNECTION_BUTTON' ); // CBTxt::T( 'REQUEST_CONNECTION_BUTTON', '[cb:if reason="profile" or reason="list"]<a href="index.php?option=com_comprofiler&view=addconnection&connectionid=[user_id]" class="btn btn-sm btn-success pt-0 pb-0 pl-3 pr-3 cbRequestConnBtn">Request Connection</a>[/cb:if]' )
		$connectField->set( 'type', 'delimiter' );
		$connectField->set( 'tabid', self::getTab( 'hidden' )->get( 'tabid', 0, GetterInterface::INT ) );
		$connectField->set( 'edit', 0 );
		$connectField->set( 'registration', 0 );
		$connectField->set( 'profile', 4 );
		$connectField->set( 'ordering', self::getFieldOrder( $connectField->get( 'tabid', 0, GetterInterface::INT ) ) );
		$connectField->set( 'params', $connectFieldParams->asJson() );

		$connectField->store();

		self::setField( 'cb_connect_button', $connectField );

		// Article Tab:
		$articleTab					=	new TabTable();

		$articleTab->load( array( 'pluginclass' => 'cbarticlesTab' ) );

		if ( $articleTab->get( 'tabid', 0, GetterInterface::INT ) ) {
			$articleTabParams		=	new Registry( $articleTab->get( 'params', null, GetterInterface::RAW ) );

			$articleTabParams->set( 'cbprivacy_display', 1 );
			$articleTabParams->set( 'privacy_options_default', 0 );
			$articleTabParams->set( 'cbprivacy_simple_group', 'PROFILE_FEATURES_PRIVACY' );

			$conditions										=	array( array( 'condition' => array( array(	'field'						=>	'custommoderators',
																											'operator_moderators'		=>	0,
																											'locations'					=>	'profile'
																										) ) )
																	);

			$articleTabParams->set( 'cbconditional_conditioned', 1 );
			$articleTabParams->set( 'cbconditional_conditions', $conditions );

			$articleTab->set( 'displaytype', 'menunested' );
			$articleTab->set( 'position', 'canvas_main_middle' );
			$articleTab->set( 'ordering', 6 );
			$articleTab->set( 'ordering_edit', 99 ); // last as this SHOULDN'T contain editable fields
			$articleTab->set( 'ordering_register', 99 ); // last as this SHOULDN'T contain editable fields
			$articleTab->set( 'params', $articleTabParams->asJson() );

			$articleTab->store();
		}

		self::setTab( 'cbarticles', $articleTab );

		// Blog Tab:
		$blogTab					=	new TabTable();

		$blogTab->load( array( 'pluginclass' => 'cbblogsTab' ) );

		if ( $blogTab->get( 'tabid', 0, GetterInterface::INT ) ) {
			$blogTabParams			=	new Registry( $blogTab->get( 'params', null, GetterInterface::RAW ) );

			$blogTabParams->set( 'cbprivacy_display', 1 );
			$blogTabParams->set( 'privacy_options_default', 0 );
			$blogTabParams->set( 'cbprivacy_simple_group', 'PROFILE_FEATURES_PRIVACY' );

			$blogTab->set( 'displaytype', 'menunested' );
			$blogTab->set( 'position', 'canvas_main_middle' );
			$blogTab->set( 'ordering', 7 );
			$blogTab->set( 'ordering_edit', 99 ); // last as this SHOULDN'T contain editable fields
			$blogTab->set( 'ordering_register', 99 ); // last as this SHOULDN'T contain editable fields
			$blogTab->set( 'params', $blogTabParams->asJson() );

			$blogTab->store();
		}

		self::setTab( 'cbblogs', $blogTab );

		// Forum Tab:
		$forumTab					=	new TabTable();

		$forumTab->load( array( 'pluginclass' => 'cbforumsTab' ) );

		if ( $forumTab->get( 'tabid', 0, GetterInterface::INT ) ) {
			$forumTabParams			=	new Registry( $forumTab->get( 'params', null, GetterInterface::RAW ) );

			$forumTabParams->set( 'cbprivacy_display', 1 );
			$forumTabParams->set( 'privacy_options_default', 0 );
			$forumTabParams->set( 'cbprivacy_simple_group', 'PROFILE_FEATURES_PRIVACY' );

			$forumTab->set( 'displaytype', 'menunested' );
			$forumTab->set( 'position', 'canvas_main_middle' );
			$forumTab->set( 'ordering', 9 );
			$forumTab->set( 'ordering_edit', 99 ); // last as this SHOULDN'T contain editable fields
			$forumTab->set( 'ordering_register', 99 ); // last as this SHOULDN'T contain editable fields
			$forumTab->set( 'params', $forumTabParams->asJson() );

			$forumTab->store();
		}

		self::setTab( 'cbforums', $forumTab );

		// Background (Canvas) Tab:
		$backgroundTab			=	new TabTable();

		$backgroundTab->load( array( 'pluginclass' => 'getCanvasTab' ) );

		if ( $backgroundTab->get( 'tabid', 0, GetterInterface::INT ) ) {
			$backgroundTab->set( 'title', 'PROFILE_FIELD_BACKGROUND' ); // CBTxt::T( 'PROFILE_FIELD_BACKGROUND', 'Background' )
			$backgroundTab->set( 'displaytype', 'html' );
			$backgroundTab->set( 'position', 'canvas_background' );
			$backgroundTab->set( 'ordering', 1 );
			$backgroundTab->set( 'ordering_edit', 3 );
			$backgroundTab->set( 'ordering_register', 3 );

			$backgroundTab->store();
		}

		self::setTab( 'canvas', $backgroundTab );

		$backgroundField			=	new FieldTable();

		$backgroundField->load( array( 'name' => 'canvas' ) );

		if ( $backgroundField->get( 'fieldid', 0, GetterInterface::INT ) ) {
			$backgroundFieldParams	=	new Registry( $backgroundField->get( 'params', null, GetterInterface::RAW ) );

			$backgroundFieldParams->set( 'ajax_profile', 1 );
			$backgroundFieldParams->set( 'ajax_profile_strict', 1 );
			$backgroundFieldParams->set( 'ajax_template', 'default' );
			$backgroundFieldParams->set( 'ajax_profile_output', 2 );

			$backgroundField->set( 'registration', 0 );
			$backgroundField->set( 'edit', 4 );
			$backgroundField->set( 'cssclass', 'form-group-block' );
			$backgroundField->set( 'params', $backgroundFieldParams->asJson() );
			$backgroundField->set( 'ordering', self::getFieldOrder( $backgroundTab->get( 'tabid', 0, GetterInterface::INT ) ) );

			$backgroundField->store();
		}

		self::setField( 'canvas', $backgroundField );

		// Portrait (Avatar) Tab:
		$portraitTab				=	new TabTable();

		$portraitTab->load( array( 'pluginclass' => 'getPortraitTab' ) );

		if ( $portraitTab->get( 'tabid', 0, GetterInterface::INT ) ) {
			$portraitTab->set( 'displaytype', 'html' );
			$portraitTab->set( 'position', 'canvas_photo' );
			$portraitTab->set( 'ordering', 1 );
			$portraitTab->set( 'ordering_edit', 4 );
			$portraitTab->set( 'ordering_register', 4 );

			$portraitTab->store();
		}

		self::setTab( 'avatar', $portraitTab );

		$portraitField				=	new FieldTable();

		$portraitField->load( array( 'name' => 'avatar' ) );

		if ( $portraitField->get( 'fieldid', 0, GetterInterface::INT ) ) {
			$portraitFieldParams	=	new Registry( $portraitField->get( 'params', null, GetterInterface::RAW ) );

			$portraitFieldParams->set( 'ajax_profile', 1 );
			$portraitFieldParams->set( 'ajax_profile_strict', 1 );
			$portraitFieldParams->set( 'ajax_template', 'default' );
			$portraitFieldParams->set( 'ajax_profile_output', 2 );

			$portraitField->set( 'registration', 0 );
			$portraitField->set( 'edit', 4 );
			$portraitField->set( 'cssclass', 'form-group-block' );
			$portraitField->set( 'params', $portraitFieldParams->asJson() );
			$portraitField->set( 'ordering', self::getFieldOrder( $portraitTab->get( 'tabid', 0, GetterInterface::INT ) ) );

			$portraitField->store();
		}

		self::setField( 'avatar', $portraitField );

		// PM Field:
		$pmField			=	new FieldTable();

		$pmField->load( array( 'name' => 'pm' ) );

		if ( $pmField->get( 'fieldid', 0, GetterInterface::INT ) ) {
			$pmFieldParams	=	new Registry( $pmField->get( 'params', null, GetterInterface::RAW ) );

			$conditions		=	array( array( 'condition' => array( array(	'field'						=>	'customusers',
																			'user_users'				=>	'displayed',
																			'operator_users'			=>	13,
																			'field_users'				=>	'[cb:userdata field="user_id" user="#me" /]',
																			'locations'					=>	'profile|*|list'
																		),
																	array(	'field'						=>	'customvalue',
																			'field_custom'				=>	'[cb:userdata field="user_id" user="#me" /]',
																			'operator'					=>	2,
																			'value'						=>	0,
																			'locations'					=>	'profile|*|list'
																		)
																	)
											)
									);

			$pmFieldParams->set( 'cbconditional_conditioned', 1 );
			$pmFieldParams->set( 'cbconditional_conditions', $conditions );
			$pmFieldParams->set( 'displayModeList', 1 );

			$pmField->set( 'tabid', self::getTab( 'hidden' )->get( 'tabid', 0, GetterInterface::INT ) );
			$pmField->set( 'edit', 0 );
			$pmField->set( 'registration', 0 );
			$pmField->set( 'profile', 4 );
			$pmField->set( 'ordering', self::getFieldOrder( $pmField->get( 'tabid', 0, GetterInterface::INT ) ) );
			$pmField->set( 'params', $pmFieldParams->asJson() );

			$pmField->store();
		}

		self::setField( 'pm', $pmField );

		// Plugins:
		$plugins				=	array(	'pms.mypmspro',
											'cbactivity',
											'cbgallery',
											'cbprivacy',
											'cbautoactions',
											'cbconditional',
											'cbantispam',
											'cbcorefieldsajax',
											'cbinvites',
											'cbfooter',
											'cbreconfirmemail',
											'cbcodefield',
											'cbqueryfield',
											'cbfieldgroups',
											'cbgroupjive',
											'cbgroupjiveabout',
											'cbgroupjiveevents',
											'cbgroupjiveforums',
											'cbpaidsubscriptions'
										);

		foreach ( $plugins as $element ) {
			$plugin				=	new PluginTable();

			$plugin->load( array( 'element' => $element ) );

			if ( ! $plugin->get( 'id', 0, GetterInterface::INT ) ) {
				continue;
			}

			$params				=	new Registry( $plugin->get( 'params', null, GetterInterface::RAW ) );

			switch ( $element ) {
				case 'pms.mypmspro':
					$params->set( 'general_model', 'internal' );
					$params->set( 'messages_notify', 0 ); // Notifications handled by CB Activity
					break;
				case 'cbautoactions':
					$autoActions			=	array(	'notificationpmscreate'			=>	'cb_notif_pm',
														'notificationpmsdelete'			=>	null,
														'notificationrequestconn'		=>	'cb_notif_request_conn',
														'notificationacceptedconn'		=>	'cb_notif_accepted_conn',
														'notificationrejectedconn'		=>	'cb_notif_rejected_conn',
														'activitycomment'				=>	null,
														'notificationactivity'			=>	'cb_notif_post',
														'notificationcomment'			=>	'cb_notif_comment',
														'notificationlike'				=>	'cb_notif_like',
														'notificationtag'				=>	'cb_notif_tag',
														'activitygallerylikes'			=>	null,
														'notificationgallerycomment'	=>	'cb_notif_gallery_comment',
														'notificationgallerylike'		=>	'cb_notif_gallery_like',
														'activitygjgrpstream'			=>	null,
														'notificationeditgjgrpstream'	=>	null,
														'notificationgjgrpstream'		=>	null,
														'gallerygjgrpgallery'			=>	null,
														'notificationeditgjgrpgallery'	=>	null,
														'notificationgjgrpgallery'		=>	null,
														'notificationgjgrpleave'		=>	null,
														'notificationgjgrpinvite'		=>	null,
														'notificationgjgrpmessage'		=>	null,
														'activitygjgrpfollow'			=>	null,
														'activitygjgrplikes'			=>	null,
														'notificationgjgrpgeneral'		=>	null,
														'activitygjeventslikes'			=>	null,
														'notificationgjevents'			=>	null
													);

					$query					=	"UPDATE " . $_CB_database->NameQuote( '#__comprofiler_plugin_autoactions' )
											.	"\n SET " . $_CB_database->NameQuote( 'published' ) . " = 1"
											.	"\n WHERE " . $_CB_database->NameQuote( 'system' ) . " IN " . $_CB_database->safeArrayOfStrings( array_keys( $autoActions ) );
					$_CB_database->setQuery( $query );
					$_CB_database->query();

					foreach ( array_values( $autoActions ) as $notifyFieldName ) {
						if ( ! $notifyFieldName ) {
							continue;
						}

						$notifyField		=	new FieldTable();

						$notifyField->load( array( 'name' => $notifyFieldName ) );

						if ( ! $notifyField->get( 'fieldid', 0, GetterInterface::INT ) ) {
							$notifyField->set( 'published', 1 );
						}

						$notifyField->set( 'description', $notifyField->getString( 'description', '' ) );

						switch ( $notifyFieldName ) {
							case 'cb_notif_pm':
								$notifyField->set( 'title', 'NOTIFY_PM' ); // CBTxt::T( 'NOTIFY_PM', 'Private Messages' )
								$notifyField->set( 'description', 'NOTIFY_PM_DESC' ); // CBTxt::T( 'NOTIFY_PM_DESC', 'Select if you would like to be notified of new private messages.' )
								break;
							case 'cb_notif_request_conn':
								$headerField								=	new FieldTable();

								$headerField->load( array( 'name' => 'cb_notif_conn' ) );

								if ( ! $headerField->get( 'fieldid', 0, GetterInterface::INT ) ) {
									$headerField->set( 'published', 1 );
								}

								$headerField->set( 'name', 'cb_notif_conn' );
								$headerField->set( 'title', 'NOTIFY_CONNECTIONS' ); // CBTxt::T( 'NOTIFY_CONNECTIONS', 'Connections' )
								$headerField->set( 'type', 'delimiter' );
								$headerField->set( 'description', '<span></span>' );
								$headerField->set( 'searchable', 0 );
								$headerField->set( 'profile', 2 );
								$headerField->set( 'edit', 2 );
								$headerField->set( 'registration', 0 );
								$headerField->set( 'tabid', self::getTab( 'notifications' )->get( 'tabid', 0, GetterInterface::INT ) );
								$headerField->set( 'ordering', self::getFieldOrder( $headerField->get( 'tabid', 0, GetterInterface::INT ) ) );
								$headerField->set( 'cssclass', 'form-group-section' );

								$headerField->store();

								self::setField( 'cb_notif_conn', $headerField );

								$notifyField->set( 'title', 'NOTIFY_REQUEST_CONN' ); // CBTxt::T( 'NOTIFY_REQUEST_CONN', 'Requests' )
								$notifyField->set( 'description', 'NOTIFY_REQUEST_CONN_DESC' ); // CBTxt::T( 'NOTIFY_REQUEST_CONN_DESC', 'Select if you would like to be notified of connection requests.' )
								break;
							case 'cb_notif_accepted_conn':
								$notifyField->set( 'title', 'NOTIFY_ACCEPT_CONN' ); // CBTxt::T( 'NOTIFY_ACCEPT_CONN', 'Accepted' )
								$notifyField->set( 'description', 'NOTIFY_ACCEPT_CONN_DESC' ); // CBTxt::T( 'NOTIFY_ACCEPT_CONN_DESC', 'Select if you would like to be notified for accepted connection requests.' )
								break;
							case 'cb_notif_rejected_conn':
								$notifyField->set( 'title', 'NOTIFY_REJECT_CONN' ); // CBTxt::T( 'NOTIFY_REJECT_CONN', 'Rejected' )
								$notifyField->set( 'description', 'NOTIFY_REJECT_CONN_DESC' ); // CBTxt::T( 'NOTIFY_REJECT_CONN_DESC', 'Select if you would like to be notified for rejected connection requests.' )
								break;
							case 'cb_notif_post':
								$headerField								=	new FieldTable();

								$headerField->load( array( 'name' => 'cb_notif_activity' ) );

								if ( ! $headerField->get( 'fieldid', 0, GetterInterface::INT ) ) {
									$headerField->set( 'published', 1 );
								}

								$headerField->set( 'name', 'cb_notif_activity' );
								$headerField->set( 'title', 'NOTIFY_ACTIVITY' ); // CBTxt::T( 'NOTIFY_ACTIVITY', 'Activity' )
								$headerField->set( 'type', 'delimiter' );
								$headerField->set( 'description', '<span></span>' );
								$headerField->set( 'searchable', 0 );
								$headerField->set( 'profile', 2 );
								$headerField->set( 'edit', 2 );
								$headerField->set( 'registration', 0 );
								$headerField->set( 'tabid', self::getTab( 'notifications' )->get( 'tabid', 0, GetterInterface::INT ) );
								$headerField->set( 'ordering', self::getFieldOrder( $headerField->get( 'tabid', 0, GetterInterface::INT ) ) );
								$headerField->set( 'cssclass', 'form-group-section' );

								$headerField->store();

								self::setField( 'cb_notif_activity', $headerField );

								$notifyField->set( 'title', 'NOTIFY_ACTIVITY_POST' ); // CBTxt::T( 'NOTIFY_ACTIVITY_POST', 'Posts' )
								$notifyField->set( 'description', 'NOTIFY_ACTIVITY_POST_DESC' ); // CBTxt::T( 'NOTIFY_ACTIVITY_POST_DESC', 'Select if you would like to be notified of activity posts on your profile.' )
								break;
							case 'cb_notif_comment':
								$notifyField->set( 'title', 'NOTIFY_ACTIVITY_COMMENT' ); // CBTxt::T( 'NOTIFY_ACTIVITY_COMMENT', 'Comments' )
								$notifyField->set( 'description', 'NOTIFY_ACTIVITY_COMMENT_DESC' ); // CBTxt::T( 'NOTIFY_ACTIVITY_COMMENT_DESC', 'Select if you would like to be notified of comments on your activity.' )
								break;
							case 'cb_notif_like':
								$notifyField->set( 'title', 'NOTIFY_ACTIVITY_LIKE' ); // CBTxt::T( 'NOTIFY_ACTIVITY_LIKE', 'Likes' )
								$notifyField->set( 'description', 'NOTIFY_ACTIVITY_DESC' ); // CBTxt::T( 'NOTIFY_ACTIVITY_DESC', 'Select if you would like to be notified of likes on your activity.' )
								break;
							case 'cb_notif_tag':
								$notifyField->set( 'title', 'NOTIFY_ACTIVITY_TAG' ); // CBTxt::T( 'NOTIFY_ACTIVITY_TAG', 'Tags' )
								$notifyField->set( 'description', 'NOTIFY_ACTIVITY_TAG_DESC' ); // CBTxt::T( 'NOTIFY_ACTIVITY_TAG_DESC', 'Select if you would like to be notified of activity you are tagged in.' )
								break;
							case 'cb_notif_gallery_comment':
								$headerField								=	new FieldTable();

								$headerField->load( array( 'name' => 'cb_notif_gallery' ) );

								if ( ! $headerField->get( 'fieldid', 0, GetterInterface::INT ) ) {
									$headerField->set( 'published', 1 );
								}

								$headerField->set( 'name', 'cb_notif_gallery' );
								$headerField->set( 'title', 'NOTIFY_GALLERY' ); // CBTxt::T( 'NOTIFY_GALLERY', 'Gallery' )
								$headerField->set( 'type', 'delimiter' );
								$headerField->set( 'description', '<span></span>' );
								$headerField->set( 'searchable', 0 );
								$headerField->set( 'profile', 2 );
								$headerField->set( 'edit', 2 );
								$headerField->set( 'registration', 0 );
								$headerField->set( 'tabid', self::getTab( 'notifications' )->get( 'tabid', 0, GetterInterface::INT ) );
								$headerField->set( 'ordering', self::getFieldOrder( $headerField->get( 'tabid', 0, GetterInterface::INT ) ) );
								$headerField->set( 'cssclass', 'form-group-section' );

								$headerField->store();

								self::setField( 'cb_notif_gallery', $headerField );

								$notifyField->set( 'title', 'NOTIFY_GALLERY_COMMENT' ); // CBTxt::T( 'NOTIFY_GALLERY_COMMENT', 'Comments' )
								$notifyField->set( 'description', 'NOTIFY_GALLERY_COMMENT_DESC' ); // CBTxt::T( 'NOTIFY_GALLERY_COMMENT_DESC', 'Select if you would like to be notified of comments on your gallery entries.' )
								break;
							case 'cb_notif_gallery_like':
								$notifyField->set( 'title', 'NOTIFY_GALLERY_LIKE' ); // CBTxt::T( 'NOTIFY_GALLERY_LIKE', 'Likes' )
								$notifyField->set( 'description', 'NOTIFY_GALLERY_LIKE_DESC' ); // CBTxt::T( 'NOTIFY_GALLERY_LIKE_DESC', 'Select if you would like to be notified of likes on your gallery entries.' )
								break;
						}

						$notifyFieldParams	=	new Registry( $notifyField->get( 'params', null, GetterInterface::RAW ) );

						$notifyFieldParams->set( 'field_edit_style', 1 );

						$notifyField->set( 'name', $notifyFieldName );
						$notifyField->set( 'type', 'checkbox' );
						$notifyField->set( 'searchable', 0 );
						$notifyField->set( 'profile', 0 );
						$notifyField->set( 'edit', 1 );
						$notifyField->set( 'tabid', self::getTab( 'notifications' )->get( 'tabid', 0, GetterInterface::INT ) );
						$notifyField->set( 'registration', 0 );
						$notifyField->set( 'ordering', self::getFieldOrder( $notifyField->get( 'tabid', 0, GetterInterface::INT ) ) );
						$notifyField->set( 'params', $notifyFieldParams->asJson() );

						$notifyField->store();

						self::setField( $notifyFieldName, $notifyField );
					}
					break;
				case 'cbactivity':
					$activityTab			=	new TabTable();

					$activityTab->load( array( 'pluginclass' => 'cbactivityTab' ) );

					if ( ! $activityTab->get( 'tabid', 0, GetterInterface::INT ) ) {
						continue 2;
					}

					$activityTabParams		=	new Registry( $activityTab->get( 'params', null, GetterInterface::RAW ) );

					$activityTabParams->set( 'activity_auto_update', 1 );
					$activityTabParams->set( 'activity_auto_load', 1 );
					$activityTabParams->set( 'cbprivacy_display', 1 );
					$activityTabParams->set( 'privacy_options_default', 0 );
					$activityTabParams->set( 'cbprivacy_simple_group', 'PROFILE_FEATURES_PRIVACY' );

					$filters				=	array(	array( 'filter' => 'PROFILE_ACTIVITY_FILTER_CONN', 'assets' => 'connectionsonly', 'users' => null, 'access' => 1 ), // CBTxt::T( 'PROFILE_ACTIVITY_FILTER_CONN', 'Connections' )
														array( 'filter' => 'PROFILE_ACTIVITY_FILTER_GALLERIES', 'assets' => 'gallery.%', 'users' => '[user_id],connections', 'access' => 1 ), // CBTxt::T( 'PROFILE_ACTIVITY_FILTER_GALLERIES', 'Galleries' )
														array( 'filter' => 'PROFILE_ACTIVITY_FILTER_GROUPS', 'assets' => 'groupjive.%', 'users' => '[user_id],connections', 'access' => 1 ), // CBTxt::T( 'PROFILE_ACTIVITY_FILTER_GROUPS', 'Groups' )
														array( 'filter' => 'PROFILE_ACTIVITY_FILTER_FOLLOWING', 'assets' => 'followingonly', 'users' => null, 'access' => 1 ) // CBTxt::T( 'PROFILE_ACTIVITY_FILTER_FOLLOWING', 'Following' )
													);

					$activityTabParams->set( 'activity_filters', $filters );

					$activityTab->set( 'displaytype', 'menu' );
					$activityTab->set( 'position', 'canvas_main_middle' );
					$activityTab->set( 'enabled', 1 );
					$activityTab->set( 'ordering', 1 );
					$activityTab->set( 'ordering_edit', 99 ); // last as this SHOULDN'T contain editable fields
					$activityTab->set( 'ordering_register', 99 ); // last as this SHOULDN'T contain editable fields
					$activityTab->set( 'params', $activityTabParams->asJson() );

					$activityTab->store();

					self::setTab( 'cbactivity', $activityTab );

					$followField			=	new FieldTable();

					$followField->load( array( 'name' => 'cb_follow' ) );

					if ( ! $followField->get( 'fieldid', 0, GetterInterface::INT ) ) {
						$followField->set( 'published', 1 );
					}

					$followFieldParams		=	new Registry( $followField->get( 'params', null, GetterInterface::RAW ) );

					$followFieldParams->set( 'following_asset', 'profile.[user_id]' );
					$followFieldParams->set( 'cbprivacy_asset', 'profile.tab.' . $activityTab->get( 'tabid', 0, GetterInterface::INT ) );
					$followFieldParams->set( 'cbprivacy_display', 1 );
					$followFieldParams->set( 'privacy_class', 'd-block mt-1' );

					$conditions				=	array( array( 'condition' => array( array(	'field'						=>	'customusers',
																							'user_users'				=>	'displayed',
																							'operator_users'			=>	13,
																							'field_users'				=>	'[cb:userdata field="user_id" user="#me" /]',
																							'locations'					=>	'profile|*|list'
																						) ) )
													);

					$followFieldParams->set( 'cbconditional_conditioned', 1 );
					$followFieldParams->set( 'cbconditional_conditions', $conditions );

					$followField->set( 'title', 'Follow' ); // already translated by CB Activity language strings
					$followField->set( 'name', 'cb_follow' );
					$followField->set( 'description', $followField->getString( 'description', '' ) );
					$followField->set( 'type', 'follow' );
					$followField->set( 'searchable', 0 );
					$followField->set( 'profile', 4 );
					$followField->set( 'edit', 0 );
					$followField->set( 'registration', 0 );
					$followField->set( 'tabid', self::getTab( 'actions' )->get( 'tabid', 0, GetterInterface::INT ) );
					$followField->set( 'params', $followFieldParams->asJson() );

					$followField->store();

					self::setField( 'cb_follow', $followField );

					$followField			=	new FieldTable();

					if ( ! $followField->get( 'fieldid', 0, GetterInterface::INT ) ) {
						$followField->set( 'published', 1 );
					}

					$followField->load( array( 'name' => 'cb_follow_button' ) );

					$followFieldParams		=	new Registry( $followField->get( 'params', null, GetterInterface::RAW ) );

					$followFieldParams->set( 'following_asset', 'profile.[user_id]' );
					$followFieldParams->set( 'following_count', 0 );
					$followFieldParams->set( 'cbprivacy_asset', 'profile.tab.' . $activityTab->get( 'tabid', 0, GetterInterface::INT ) );
					$followFieldParams->set( 'cbprivacy_display', 1 );
					$followFieldParams->set( 'privacy_class', 'd-block mt-1' );
					$followFieldParams->set( 'cbconditional_conditioned', 1 );
					$followFieldParams->set( 'cbconditional_conditions', $conditions );

					$followField->set( 'name', 'cb_follow_button' );
					$followField->set( 'title', 'Follow' ); // already translated by CB Activity language strings
					$followField->set( 'description', $followField->getString( 'description', '' ) );
					$followField->set( 'type', 'follow' );
					$followField->set( 'searchable', 0 );
					$followField->set( 'profile', 4 );
					$followField->set( 'edit', 0 );
					$followField->set( 'registration', 0 );
					$followField->set( 'tabid', self::getTab( 'hidden' )->get( 'tabid', 0, GetterInterface::INT ) );
					$followField->set( 'params', $followFieldParams->asJson() );

					$followField->store();

					self::setField( 'cb_follow_button', $followField );
					break;
				case 'cbgallery':
					// Gallery Tab:
					$galleryTab			=	new TabTable();

					$galleryTab->load( array( 'pluginclass' => 'cbgalleryTab' ) );

					if ( ! $galleryTab->get( 'tabid', 0, GetterInterface::INT ) ) {
						continue 2;
					}

					$galleryTabParams	=	new Registry( $galleryTab->get( 'params', null, GetterInterface::RAW ) );

					$galleryTabParams->set( 'cbprivacy_display', 1 );
					$galleryTabParams->set( 'privacy_options_default', 0 );
					$galleryTabParams->set( 'cbprivacy_simple_group', 'PROFILE_FEATURES_PRIVACY' );

					$galleryTab->set( 'displaytype', 'menu' );
					$galleryTab->set( 'position', 'canvas_main_middle' );
					$galleryTab->set( 'enabled', 1 );
					$galleryTab->set( 'ordering', 4 );
					$galleryTab->set( 'ordering_edit', 99 ); // last as this SHOULDN'T contain editable fields
					$galleryTab->set( 'ordering_register', 99 ); // last as this SHOULDN'T contain editable fields
					$galleryTab->set( 'params', $galleryTabParams->asJson() );

					$galleryTab->store();

					self::setTab( 'cbgallery', $galleryTab );

					// Photos Count Field:
					$photosAsset							=	$galleryTabParams->get( 'gallery_asset', null, GetterInterface::STRING );

					if ( ! $photosAsset ) {
						$photosAsset						=	'profile,uploads';
					}

					$photosField							=	new FieldTable();

					$photosField->load( array( 'name' => 'cb_photos' ) );

					if ( ! $photosField->get( 'fieldid', 0, GetterInterface::INT ) ) {
						$photosField->set( 'published', 1 );
					}

					$photosFieldParams						=	new Registry( $photosField->get( 'params', null, GetterInterface::RAW ) );

					$photosFieldParams->set( 'fieldLayout', '[cb:if cb_photos>"0"]<span class="fa fa-picture-o"></span> [value][/cb:if]' );
					$photosFieldParams->set( 'gallery_asset', $photosAsset );
					$photosFieldParams->set( 'gallery_layout', 'count' );
					$photosFieldParams->set( 'gallery_photos', 1 );
					$photosFieldParams->set( 'gallery_videos', 0 );
					$photosFieldParams->set( 'gallery_files', 0 );
					$photosFieldParams->set( 'gallery_music', 0 );
					$photosFieldParams->set( 'cbprivacy_asset', 'profile.tab.' . $galleryTab->get( 'tabid', 0, GetterInterface::INT ) );
					$photosFieldParams->set( 'cbprivacy_display', 1 );
					$photosFieldParams->set( 'privacy_class', 'd-block mt-1' );

					$photosField->set( 'name', 'cb_photos' );
					$photosField->set( 'title', 'Photos' ); // already translated by CB Gallery language strings
					$photosField->set( 'description', $photosField->getString( 'description', '' ) );
					$photosField->set( 'type', 'gallery' );
					$photosField->set( 'searchable', 0 );
					$photosField->set( 'profile', 1 );
					$photosField->set( 'edit', 0 );
					$photosField->set( 'registration', 0 );
					$photosField->set( 'tabid', self::getTab( 'stats' )->get( 'tabid', 0, GetterInterface::INT ) );
					$photosField->set( 'params', $photosFieldParams->asJson() );

					$photosField->store();

					self::setField( 'cb_photos', $photosField );
					break;
				case 'cbgroupjive':
					// Groups Tab:
					$groupsTab			=	new TabTable();

					$groupsTab->load( array( 'pluginclass' => 'cbgjTab' ) );

					if ( ! $groupsTab->get( 'tabid', 0, GetterInterface::INT ) ) {
						continue 2;
					}

					$groupsTabParams	=	new Registry( $groupsTab->get( 'params', null, GetterInterface::RAW ) );

					$groupsTabParams->set( 'cbprivacy_display', 1 );
					$groupsTabParams->set( 'privacy_options_default', 0 );
					$groupsTabParams->set( 'cbprivacy_simple_group', 'PROFILE_FEATURES_PRIVACY' );

					$groupsTab->set( 'displaytype', 'menu' );
					$groupsTab->set( 'position', 'canvas_main_middle' );
					$groupsTab->set( 'enabled', 1 );
					$groupsTab->set( 'ordering', 5 );
					$groupsTab->set( 'ordering_edit', 99 ); // last as this SHOULDN'T contain editable fields
					$groupsTab->set( 'ordering_register', 99 ); // last as this SHOULDN'T contain editable fields
					$groupsTab->set( 'params', $groupsTabParams->asJson() );

					$groupsTab->store();

					self::setTab( 'cbgroupjive', $groupsTab );

					// Group Count Field:
					$groupsField							=	new FieldTable();

					$groupsField->load( array( 'name' => 'cb_groups' ) );

					if ( ! $groupsField->get( 'fieldid', 0, GetterInterface::INT ) ) {
						$groupsField->set( 'published', 1 );
					}

					$groupsFieldParams						=	new Registry( $groupsField->get( 'params', null, GetterInterface::RAW ) );

					$groupsFieldParams->set( 'fieldLayout', '[cb:if cb_groups>"0"]<span class="fa fa-user-circle-o"></span> [value][/cb:if]' );
					$groupsFieldParams->set( 'cbprivacy_asset', 'profile.tab.' . $groupsTab->get( 'tabid', 0, GetterInterface::INT ) );
					$groupsFieldParams->set( 'cbprivacy_display', 1 );
					$groupsFieldParams->set( 'privacy_class', 'd-block mt-1' );

					$groupsField->set( 'name', 'cb_groups' );
					$groupsField->set( 'title', 'Groups' ); // already translated by CB GroupJive language strings
					$groupsField->set( 'description', $groupsField->getString( 'description', '' ) );
					$groupsField->set( 'type', 'groupautojoin' );
					$groupsField->set( 'searchable', 0 );
					$groupsField->set( 'profile', 1 );
					$groupsField->set( 'edit', 0 );
					$groupsField->set( 'registration', 0 );
					$groupsField->set( 'tabid', self::getTab( 'stats' )->get( 'tabid', 0, GetterInterface::INT ) );
					$groupsField->set( 'params', $groupsFieldParams->asJson() );

					$groupsField->store();

					self::setField( 'cb_groups', $groupsField );
					break;
				case 'cbinvites':
					$inviteTab		=	new TabTable();

					$inviteTab->load( array( 'pluginclass' => 'cbinvitesTab' ) );

					if ( $inviteTab->get( 'tabid', 0, GetterInterface::INT ) ) {
						$inviteTab->set( 'displaytype', 'menunested' );
						$inviteTab->set( 'position', 'canvas_main_middle' );
						$inviteTab->set( 'enabled', 1 );
						$inviteTab->set( 'ordering', 8 );
						$inviteTab->set( 'ordering_edit', 99 ); // last as this SHOULDN'T contain editable fields
						$inviteTab->set( 'ordering_register', 99 ); // last as this SHOULDN'T contain editable fields

						$inviteTab->store();

						self::setTab( 'cbinvites', $inviteTab );
					}

					$inviteField		=	new FieldTable();

					$inviteField->load( array( 'name' => 'invite_code' ) );

					if ( $inviteField->get( 'fieldid', 0, GetterInterface::INT ) ) {
						$inviteField->set( 'registration', 1 );
						$inviteField->set( 'edit', 0 );
						$inviteField->set( 'profile', 0 );
						$inviteField->set( 'cssclass', 'hidden' );

						$inviteField->store();

						self::setField( 'invite_code', $inviteField );
					}
					break;
				case 'cbprivacy':
					// Profile Privacy Field:
					$profilePrivacy		=	new FieldTable();

					$profilePrivacy->load( array( 'name' => 'privacy_profile' ) );

					if ( $profilePrivacy->get( 'fieldid', 0, GetterInterface::INT ) ) {
						$profilePrivacy->set( 'title', 'FIELD_PROFILE_PRIVACY' ); // CBTxt::T( 'FIELD_PROFILE_PRIVACY', 'Profile' )
						$profilePrivacy->set( 'published', 1 );
						$profilePrivacy->set( 'registration', 0 );
						$profilePrivacy->set( 'edit', 1 );
						$profilePrivacy->set( 'ordering', 1 );

						$profilePrivacy->store();

						self::setField( 'privacy_profile', $profilePrivacy );
					}

					// Privacy Fields:
					foreach ( array( 'privacy_disable_me', 'privacy_disable_me', 'privacy_export' ) as $fieldName ) {
						$privacyField	=	new FieldTable();

						$privacyField->load( array( 'name' => $fieldName ) );

						if ( ! $privacyField->get( 'fieldid', 0, GetterInterface::INT ) ) {
							continue;
						}

						$privacyField->set( 'published', 1 );
						$privacyField->set( 'edit', 1 );

						$privacyField->store();

						self::setField( $fieldName, $privacyField );
					}

					// Privacy Controls (Fields):
					// CBTxt::T( 'PROFILE_STATUS_PRIVACY', '<div class="bg-light text-center col-form-label">Status</div>' )
					foreach ( array( 'name', 'email', 'onlinestatus', 'registerDate', 'lastvisitDate', 'lastupdatedate' ) as $fieldName ) {
						$privacyField	=	new FieldTable();

						$privacyField->load( array( 'name' => $fieldName ) );

						if ( ! $privacyField->get( 'fieldid', 0, GetterInterface::INT ) ) {
							continue;
						}

						$privacyFieldParams	=	new Registry( $privacyField->get( 'params', null, GetterInterface::RAW ) );

						$privacyFieldParams->set( 'cbprivacy_display', 1 );
						$privacyFieldParams->set( 'privacy_class', 'd-block mt-1' );

						if ( ! in_array( $fieldName, array( 'onlinestatus', 'registerDate', 'lastvisitDate', 'lastupdatedate' ) ) ) {
							// Turn on ajax edit for fields that can be edited:
							$privacyField->set( 'profile', 1 );

							$privacyFieldParams->set( 'ajax_profile', 1 );
							$privacyFieldParams->set( 'ajax_profile_strict', 1 );
							$privacyFieldParams->set( 'ajax_template', 'cog' );
						}

						switch ( $fieldName ) {
							case 'name':
								$privacyFieldParams->set( 'cbprivacy_simple_group', 'PROFILE_DETAILS_PRIVACY' );
								$privacyFieldParams->set( 'privacy_options_default', 0 );
								break;
							case 'email':
								$privacyFieldParams->set( 'cbprivacy_simple_group', 'PROFILE_DETAILS_PRIVACY' );
								$privacyFieldParams->set( 'privacy_options_default', 2 );
								break;
							case 'onlinestatus':
							case 'registerDate':
							case 'lastvisitDate':
							case 'lastupdatedate':
								$privacyField->set( 'edit', 1 );
								$privacyFieldParams->set( 'cbprivacy_simple_group', 'PROFILE_STATUS_PRIVACY' );
								$privacyFieldParams->set( 'privacy_options_default', 0 );
								break;
						}

						$privacyField->set( 'params', $privacyFieldParams->asJson() );

						$privacyField->store();

						self::setField( $fieldName, $privacyField );
					}

					// Configuration:
					$params->set( 'profile_simple_edit', 1 );
					$params->set( 'profile_simple_registration', 1 );
					break;
				case 'cbantispam':
					// Captcha:
					$captchaField	=	new FieldTable();

					$captchaField->load( array( 'name' => 'antispam_captcha' ) );

					if ( $captchaField->get( 'fieldid', 0, GetterInterface::INT ) ) {
						$captchaField->set( 'edit', 0 );
						$captchaField->set( 'registration', 1 );
						$captchaField->set( 'published', 1 );

						$captchaField->store();

						self::setField( 'antispam_captcha', $captchaField );
					}

					// Configuration:
					$params->set( 'login_duplicate', 1 );
					$params->set( 'login_duplicate_method', 0 );
					$params->set( 'forgot_autoblock', 1 );
					$params->set( 'forgot_autoblock_method', 0 );
					$params->set( 'reg_duplicate', 1 );
					$params->set( 'reg_duplicate_count', 5 );
					$params->set( 'reg_duplicate_method', 0 );
					$params->set( 'captcha_legacy_forgot', 1 );

					// Tabs:
					foreach ( array( 'cbantispamTabBlocks', 'cbantispamTabWhitelists', 'cbantispamTabAttempts', 'cbantispamTabLog' ) as $tabClass ) {
						$antispamTab	=	new TabTable();

						$antispamTab->load( array( 'pluginclass' => $tabClass ) );

						if ( $antispamTab->get( 'tabid', 0, GetterInterface::INT ) ) {
							$antispamTab->set( 'position', 'not_on_profile_1' );
							$antispamTab->set( 'ordering', 99 ); // last as not in use
							$antispamTab->set( 'ordering_edit', 99 ); // last as this SHOULDN'T contain editable fields
							$antispamTab->set( 'ordering_register', 99 ); // last as this SHOULDN'T contain editable fields

							$antispamTab->store();

							self::setTab( $tabClass, $antispamTab );
						}
					}
					break;
			}

			$plugin->set( 'published', 1 );
			$plugin->set( 'params', $params->asJson() );

			$plugin->store();

			self::setPlugin( $element, $plugin );
		}

		// About Me:
		$aboutField					=	new FieldTable();

		$aboutField->load( array( 'name' => 'cb_about' ) );

		if ( ! $aboutField->get( 'fieldid', 0, GetterInterface::INT ) ) {
			$aboutField->set( 'published', 1 );
		}

		$aboutFieldParams			=	new Registry( $aboutField->get( 'params', null, GetterInterface::RAW ) );

		$aboutFieldParams->set( 'fieldPlaceholder', 'PROFILE_ABOUT_ME_PLACEHOLDER' ); // CBTxt::T( 'PROFILE_ABOUT_ME_PLACEHOLDER', 'Write some details about yourself...' )
		$aboutFieldParams->set( 'cbprivacy_display', 1 );
		$aboutFieldParams->set( 'privacy_class', 'd-block mt-1' );
		$aboutFieldParams->set( 'privacy_options_default', 0 );
		$aboutFieldParams->set( 'cbprivacy_simple_group', 'PROFILE_DETAILS_PRIVACY' );
		$aboutFieldParams->set( 'ajax_profile', 1 );
		$aboutFieldParams->set( 'ajax_profile_strict', 1 );
		$aboutFieldParams->set( 'ajax_template', 'cog' );
		$aboutFieldParams->set( 'ajax_fields_titles', 4 );

		$aboutField->set( 'name', 'cb_about' );
		$aboutField->set( 'title', 'PROFILE_ABOUT_ME' ); // CBTxt::T( 'PROFILE_ABOUT_ME', 'About Me' )
		$aboutField->set( 'description', $aboutField->getString( 'description', '' ) );
		$aboutField->set( 'type', 'textarea' );
		$aboutField->set( 'searchable', 1 );
		$aboutField->set( 'profile', 4 );
		$aboutField->set( 'edit', 4 );
		$aboutField->set( 'registration', 0 );
		$aboutField->set( 'cols', 0 );
		$aboutField->set( 'rows', 3 );
		$aboutField->set( 'cssclass', 'form-group-block' );
		$aboutField->set( 'tabid', self::getTab( 'details' )->get( 'tabid', 0, GetterInterface::INT ) );
		$aboutField->set( 'ordering', self::getFieldOrder( $aboutField->get( 'tabid', 0, GetterInterface::INT ) ) );
		$aboutField->set( 'params', $aboutFieldParams->asJson() );

		$aboutField->store();

		self::setField( 'cb_about', $aboutField );

		// Details:
		// CBTxt::T( 'PROFILE_DETAILS_PRIVACY', '<div class="bg-light text-center col-form-label">Details</div>' )
		foreach ( array( 'cb_details_header', 'cb_details_name', 'cb_details_email', 'cb_details_birthday', 'cb_details_birthday_style', 'cb_details_birthday_custom', 'cb_details_gender', 'cb_details_interestedin', 'cb_details_relationship', 'cb_details_language', 'cb_details_religion', 'cb_details_political' ) as $fieldName ) {
			$detailsField			=	new FieldTable();

			$detailsField->load( array( 'name' => $fieldName ) );

			if ( ! $detailsField->get( 'fieldid', 0, GetterInterface::INT ) ) {
				$detailsField->set( 'published', 1 );
			}

			$detailsFieldParams		=	new Registry( $detailsField->get( 'params', null, GetterInterface::RAW ) );

			$detailsField->set( 'description', $detailsField->getString( 'description', '' ) );
			$detailsField->set( 'tabid', self::getTab( 'details' )->get( 'tabid', 0, GetterInterface::INT ) );
			$detailsField->set( 'registration', 0 );

			switch ( $fieldName ) {
				case 'cb_details_header':
					$detailsField->set( 'title', 'PROFILE_DETAILS' ); // CBTxt::T( 'PROFILE_DETAILS', 'Details' )
					$detailsField->set( 'type', 'delimiter' );
					$detailsField->set( 'description', '<span></span>' );
					$detailsField->set( 'searchable', 0 );
					$detailsField->set( 'profile', 2 );
					$detailsField->set( 'edit', 2 );
					$detailsField->set( 'cssclass', 'form-group-section' );

					$conditions										=	array( array( 'condition' => array( array(	'field'						=>	'customvalue',
																													'field_custom'				=>	'[cb:userfield field="cb_details_name" /]',
																													'operator'					=>	7,
																													'locations'					=>	'profile'
																												) ) ),
																				array( 'condition' => array( array(	'field'						=>	'customvalue',
																													'field_custom'				=>	'[cb:userfield field="cb_details_email" /]',
																													'operator'					=>	7,
																													'locations'					=>	'profile'
																											 	) ) )
																			);

					$detailsFieldParams->set( 'cbconditional_conditioned', 1 );
					$detailsFieldParams->set( 'cbconditional_conditions', $conditions );
					break;
				case 'cb_details_name':
					$detailsField->set( 'title', 'PROFILE_DETAILS_NAME' ); // CBTxt::T( 'PROFILE_DETAILS_NAME', 'Name' )
					$detailsField->set( 'type', 'delimiter' );
					$detailsField->set( 'description', '[cb:userfield field="name" /]' );
					$detailsField->set( 'searchable', 0 );
					$detailsField->set( 'profile', 1 );
					$detailsField->set( 'edit', 0 );

					$conditions										=	array( array( 'condition' => array( array(	'field'						=>	'customvalue',
																													'field_custom'				=>	'[cb:userfield field="name" /]',
																													'operator'					=>	7,
																													'locations'					=>	'profile'
																												) ) )
																			);

					$detailsFieldParams->set( 'cbconditional_conditioned', 1 );
					$detailsFieldParams->set( 'cbconditional_conditions', $conditions );
					break;
				case 'cb_details_email':
					$detailsField->set( 'title', 'PROFILE_DETAILS_EMAIL' ); // CBTxt::T( 'PROFILE_DETAILS_EMAIL', 'Email' )
					$detailsField->set( 'type', 'delimiter' );
					$detailsField->set( 'description', '[cb:userfield field="email" /]' );
					$detailsField->set( 'searchable', 0 );
					$detailsField->set( 'profile', 1 );
					$detailsField->set( 'edit', 0 );

					$conditions										=	array( array( 'condition' => array( array(	'field'						=>	'customvalue',
																													'field_custom'				=>	'[cb:userfield field="email" /]',
																													'operator'					=>	7,
																													'locations'					=>	'profile'
																												) ) )
																			);

					$detailsFieldParams->set( 'cbconditional_conditioned', 1 );
					$detailsFieldParams->set( 'cbconditional_conditions', $conditions );
					break;
				case 'cb_details_birthday':
					$detailsField->set( 'title', 'PROFILE_DETAILS_BIRTHDAY' ); // CBTxt::T( 'PROFILE_DETAILS_BIRTHDAY', 'Birthday' )
					$detailsField->set( 'type', 'date' );
					$detailsField->set( 'searchable', 1 );
					$detailsField->set( 'profile', 1 );
					$detailsField->set( 'edit', 1 );

					$detailsFieldParams->set( 'duration_title', 'PROFILE_DETAILS_BIRTHDAY_PROFILE' ); // CBTxt::T( 'PROFILE_DETAILS_BIRTHDAY_PROFILE', '[cb:if cb_details_birthday_style="BIRTHDAY_YEAR"]Birth Year[cb:elseif cb_details_birthday_style="BIRTHDAY_MONTH"]Birth Month[/cb:elseif][cb:elseif cb_details_birthday_style!=""]Birthday[/cb:elseif][cb:else]Age[/cb:else][/cb:if]' )
					$detailsFieldParams->set( 'fieldLayout', '[cb:if cb_details_birthday_style=~"/AGE/"][value] - [/cb:if][cb:if cb_details_birthday_style!=""][cb:userfield field="cb_details_birthday_custom" /][cb:else][value][/cb:else][/cb:if]' );
					$detailsFieldParams->set( 'fieldLayoutList', '[cb:if cb_details_birthday_style=~"/AGE/"][value] - [/cb:if][cb:if cb_details_birthday_style!=""][cb:userfield field="cb_details_birthday_custom" /][cb:else][value][/cb:else][/cb:if]' );
					$detailsFieldParams->set( 'field_display_by', 1 );
					$detailsFieldParams->set( 'field_search_by', 1 );
					$detailsFieldParams->set( 'year_min', '-100' );
					$detailsFieldParams->set( 'year_max', '+0' );
					$detailsFieldParams->set( 'cbprivacy_display', 1 );
					$detailsFieldParams->set( 'cbprivacy_simple_group', 'PROFILE_DETAILS_PRIVACY' );
					$detailsFieldParams->set( 'privacy_class', 'd-block mt-1' );
					$detailsFieldParams->set( 'privacy_options_default', 0 );
					$detailsFieldParams->set( 'ajax_profile', 1 );
					$detailsFieldParams->set( 'ajax_profile_strict', 1 );
					$detailsFieldParams->set( 'ajax_template', 'cog' );
					$detailsFieldParams->set( 'ajax_fields_titles', 4 );
					break;
				case 'cb_details_birthday_style':
					$detailsField->set( 'title', 'PROFILE_DETAILS_BIRTHDAY' ); // CBTxt::T( 'PROFILE_DETAILS_BIRTHDAY', 'Birthday' )
					$detailsField->set( 'type', 'select' );
					$detailsField->set( 'searchable', 0 );
					$detailsField->set( 'profile', 0 );
					$detailsField->set( 'edit', 3 );

					$detailsField->set( '_fieldvalues', array(	array( 'fieldtitle' => '', 'fieldlabel' => 'PROFILE_DETAILS_BIRTHDAY_AGE' ), // CBTxt::T( 'PROFILE_DETAILS_BIRTHDAY_AGE', 'Display as Age' )
																array( 'fieldtitle' => 'BIRTHDAY_YEAR', 'fieldlabel' => 'PROFILE_DETAILS_BIRTHDAY_YEAR' ), // CBTxt::T( 'PROFILE_DETAILS_BIRTHDAY_YEAR', 'Display as Year' )
																array( 'fieldtitle' => 'BIRTHDAY_MONTH', 'fieldlabel' => 'PROFILE_DETAILS_BIRTHDAY_MONTH' ), // CBTxt::T( 'PROFILE_DETAILS_BIRTHDAY_MONTH', 'Display as Month' )
																array( 'fieldtitle' => 'BIRTHDAY_DAYMONTH', 'fieldlabel' => 'PROFILE_DETAILS_BIRTHDAY_DAYMONTH' ), // CBTxt::T( 'PROFILE_DETAILS_BIRTHDAY_DAYMONTH', 'Display as Day and Month' )
																array( 'fieldtitle' => 'BIRTHDAY_MONTHYEAR', 'fieldlabel' => 'PROFILE_DETAILS_BIRTHDAY_MONTHYEAR' ), // CBTxt::T( 'PROFILE_DETAILS_BIRTHDAY_MONTHYEAR', 'Display as Month and Year' )
																array( 'fieldtitle' => 'BIRTHDAY_DATE', 'fieldlabel' => 'PROFILE_DETAILS_BIRTHDAY_DATE' ), // CBTxt::T( 'PROFILE_DETAILS_BIRTHDAY_DATE', 'Display as Date' )
																array( 'fieldtitle' => 'BIRTHDAY_AGEYEAR', 'fieldlabel' => 'PROFILE_DETAILS_BIRTHDAY_AGEYEAR' ), // CBTxt::T( 'PROFILE_DETAILS_BIRTHDAY_AGEYEAR', 'Display as Age and Year' )
																array( 'fieldtitle' => 'BIRTHDAY_AGEMONTH', 'fieldlabel' => 'PROFILE_DETAILS_BIRTHDAY_AGEMONTH' ), // CBTxt::T( 'PROFILE_DETAILS_BIRTHDAY_AGEMONTH', 'Display as Age and Month' )
																array( 'fieldtitle' => 'BIRTHDAY_AGEDAYMONTH', 'fieldlabel' => 'PROFILE_DETAILS_BIRTHDAY_AGEDAYMONTH' ), // CBTxt::T( 'PROFILE_DETAILS_BIRTHDAY_AGEDAYMONTH', 'Display as Age, Day and Month' )
																array( 'fieldtitle' => 'BIRTHDAY_AGEMONTHYEAR', 'fieldlabel' => 'PROFILE_DETAILS_BIRTHDAY_AGEMONTHYEAR' ), // CBTxt::T( 'PROFILE_DETAILS_BIRTHDAY_AGEMONTHYEAR', 'Display as Age, Month and Year' )
																array( 'fieldtitle' => 'BIRTHDAY_AGEDATE', 'fieldlabel' => 'PROFILE_DETAILS_BIRTHDAY_AGEDATE' ) // CBTxt::T( 'PROFILE_DETAILS_BIRTHDAY_AGEDATE', 'Display as Age and Date' )
															) );

					$conditions											=	array( array( 'condition' => array( array(	'field'						=>	self::getField( 'cb_details_birthday' )->get( 'fieldid', 0, GetterInterface::INT ) . ',cb_details_birthday',
																														'operator'					=>	7,
																														'locations'					=>	'register|*|edit'
																													) ) )
																				);

					$detailsFieldParams->set( 'cbconditional_conditioned', 1 );
					$detailsFieldParams->set( 'cbconditional_conditions', $conditions );
					break;
				case 'cb_details_birthday_custom':
					$detailsField->set( 'title', 'PROFILE_DETAILS_BIRTHDAY' ); // CBTxt::T( 'PROFILE_DETAILS_BIRTHDAY', 'Birthday' )
					$detailsField->set( 'type', 'code' );
					$detailsField->set( 'searchable', 0 );
					$detailsField->set( 'profile', 1 );
					$detailsField->set( 'edit', 0 );
					$detailsField->set( 'tabid', self::getTab( 'hidden' )->get( 'tabid', 0, GetterInterface::INT ) );

					$birthdayCode		=	"switch ( '[cb_details_birthday_style]' ) {"
										.	"\n	case 'BIRTHDAY_YEAR':"
										.	"\n	case 'BIRTHDAY_AGEYEAR':"
										.	"\n		return cbFormatDate( '[cb_details_birthday]', false, false, 'Y' );"
										.	"\n		break;"
										.	"\n	case 'BIRTHDAY_MONTH':"
										.	"\n	case 'BIRTHDAY_AGEMONTH':"
										.	"\n		return cbFormatDate( '[cb_details_birthday]', false, false, 'F' );"
										.	"\n		break;"
										.	"\n	case 'BIRTHDAY_DAYMONTH':"
										.	"\n	case 'BIRTHDAY_AGEDAYMONTH':"
										.	"\n		return cbFormatDate( '[cb_details_birthday]', false, false, 'F j' );"
										.	"\n		break;"
										.	"\n	case 'BIRTHDAY_MONTHYEAR':"
										.	"\n	case 'BIRTHDAY_AGEMONTHYEAR':"
										.	"\n		return cbFormatDate( '[cb_details_birthday]', false, false, 'F Y' );"
										.	"\n		break;"
										.	"\n	case 'BIRTHDAY_DATE':"
										.	"\n	case 'BIRTHDAY_AGEDATE':"
										.	"\n		return cbFormatDate( '[cb_details_birthday]', false, false );"
										.	"\n		break;"
										.	"\n}";

					$detailsFieldParams->set( 'code', $birthdayCode );
					break;
				case 'cb_details_gender':
					$detailsField->set( 'title', 'PROFILE_DETAILS_GENDER' ); // CBTxt::T( 'PROFILE_DETAILS_GENDER', 'Gender' )
					$detailsField->set( 'type', 'tag' );
					$detailsField->set( 'searchable', 1 );
					$detailsField->set( 'profile', 1 );
					$detailsField->set( 'edit', 1 );

					$detailsField->set( '_fieldvalues', array(	array( 'fieldtitle' => 'GENDER_MALE', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_MALE' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_MALE', 'Male' )
																array( 'fieldtitle' => 'GENDER_FEMALE', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_FEMALE' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_FEMALE', 'Female' )
																array( 'fieldtitle' => 'GENDER_AGENDER', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_AGENDER' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_AGENDER', 'Agender' )
																array( 'fieldtitle' => 'GENDER_ANDROGYNE', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_ANDROGYNE' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_ANDROGYNE', 'Androgyne' )
																array( 'fieldtitle' => 'GENDER_ANDROGYNES', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_ANDROGYNES' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_ANDROGYNES', 'Androgynes' )
																array( 'fieldtitle' => 'GENDER_ANDROGYNOUS', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_ANDROGYNOUS' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_ANDROGYNOUS', 'Androgynous' )
																array( 'fieldtitle' => 'GENDER_ASEXUAL', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_ASEXUAL' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_ASEXUAL', 'Asexual' )
																array( 'fieldtitle' => 'GENDER_BIGENDER', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_BIGENDER' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_BIGENDER', 'Bigender' )
																array( 'fieldtitle' => 'GENDER_CIS', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_CIS' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_CIS', 'Cis' )
																array( 'fieldtitle' => 'GENDER_CISFEMALE', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_CISFEMALE' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_CISFEMALE', 'Cis Female' )
																array( 'fieldtitle' => 'GENDER_CISMALE', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_CISMALE' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_CISMALE', 'Cis Male' )
																array( 'fieldtitle' => 'GENDER_CISMAN', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_CISMAN' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_CISMAN', 'Cis Man' )
																array( 'fieldtitle' => 'GENDER_CISWOMAN', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_CISWOMAN' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_CISWOMAN', 'Cis Woman' )
																array( 'fieldtitle' => 'GENDER_CISGENDER', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_CISGENDER' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_CISGENDER', 'Cisgender' )
																array( 'fieldtitle' => 'GENDER_CISGENDERFEMALE', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_CISGENDERFEMALE' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_CISGENDERFEMALE', 'Cisgender Female' )
																array( 'fieldtitle' => 'GENDER_CISGENDERMALE', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_CISGENDERMALE' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_CISGENDERMALE', 'Cisgender Male' )
																array( 'fieldtitle' => 'GENDER_CISGENDERMAN', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_CISGENDERMAN' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_CISGENDERMAN', 'Cisgender Man' )
																array( 'fieldtitle' => 'GENDER_CISGENDERWOMAN', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_CISGENDERWOMAN' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_CISGENDERWOMAN', 'Cisgender Woman' )
																array( 'fieldtitle' => 'GENDER_F2M', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_F2M' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_F2M', 'F2M' )
																array( 'fieldtitle' => 'GENDER_FTM', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_FTM' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_FTM', 'FTM' )
																array( 'fieldtitle' => 'GENDER_FEMALETOMALE', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_FEMALETOMALE' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_FEMALETOMALE', 'Female to Male' )
																array( 'fieldtitle' => 'GENDER_FEMALETOMALETRANSMAN', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_FEMALETOMALETRANSMAN' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_FEMALETOMALETRANSMAN', 'Female to Male Trans Man' )
																array( 'fieldtitle' => 'GENDER_FEMALETOMALETRANSGENDERMAN', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_FEMALETOMALETRANSGENDERMAN' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_FEMALETOMALETRANSGENDERMAN', 'Female to Male Transgender Man' )
																array( 'fieldtitle' => 'GENDER_FEMALETOMALETRANSSEXUALMAN', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_FEMALETOMALETRANSSEXUALMAN' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_FEMALETOMALETRANSSEXUALMAN', 'Female to Male Transsexual Man' )
																array( 'fieldtitle' => 'GENDER_GENDERFLUID', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_GENDERFLUID' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_GENDERFLUID', 'Gender Fluid' )
																array( 'fieldtitle' => 'GENDER_GENDERNONCONFORMING', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_GENDERNONCONFORMING' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_GENDERNONCONFORMING', 'Gender Nonconforming' )
																array( 'fieldtitle' => 'GENDER_GENDERQUESTIONING', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_GENDERQUESTIONING' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_GENDERQUESTIONING', 'Gender Questioning' )
																array( 'fieldtitle' => 'GENDER_GENDERVARIANT', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_GENDERVARIANT' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_GENDERVARIANT', 'Gender Variant' )
																array( 'fieldtitle' => 'GENDER_GENDERNEUTRAL', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_GENDERNEUTRAL' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_GENDERNEUTRAL', 'Gender Neutral' )
																array( 'fieldtitle' => 'GENDER_GENDERQUEER', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_GENDERQUEER' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_GENDERQUEER', 'Genderqueer' )
																array( 'fieldtitle' => 'GENDER_HERMAPHRODITE', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_HERMAPHRODITE' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_HERMAPHRODITE', 'Hermaphrodite' )
																array( 'fieldtitle' => 'GENDER_INTERSEX', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_INTERSEX' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_INTERSEX', 'Intersex' )
																array( 'fieldtitle' => 'GENDER_INTERSEXMAN', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_INTERSEXMAN' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_INTERSEXMAN', 'Intersex Man' )
																array( 'fieldtitle' => 'GENDER_INTERSEXPERSON', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_INTERSEXPERSON' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_INTERSEXPERSON', 'Intersex Person' )
																array( 'fieldtitle' => 'GENDER_INTERSEXWOMAN', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_INTERSEXWOMAN' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_INTERSEXWOMAN', 'Intersex Woman' )
																array( 'fieldtitle' => 'GENDER_M2F', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_M2F' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_M2F', 'M2F' )
																array( 'fieldtitle' => 'GENDER_MTF', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_MTF' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_MTF', 'MTF' )
																array( 'fieldtitle' => 'GENDER_MALETOFEMALE', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_MALETOFEMALE' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_MALETOFEMALE', 'Male to Female' )
																array( 'fieldtitle' => 'GENDER_MALETOFEMALETRANSWOMAN', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_MALETOFEMALETRANSWOMAN' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_MALETOFEMALETRANSWOMAN', 'Male to Female Trans Woman' )
																array( 'fieldtitle' => 'GENDER_MALETOFEMALETRANSGENDERWOMAN', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_MALETOFEMALETRANSGENDERWOMAN' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_MALETOFEMALETRANSGENDERWOMAN', 'Male to Female Transgender Woman' )
																array( 'fieldtitle' => 'GENDER_MALETOFEMALETRANSSEXUALWOMAN', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_MALETOFEMALETRANSSEXUALWOMAN' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_MALETOFEMALETRANSSEXUALWOMAN', 'Male to Female Transsexual Woman' )
																array( 'fieldtitle' => 'GENDER_MAN', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_MAN' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_MAN', 'Man' )
																array( 'fieldtitle' => 'GENDER_NEITHER', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_NEITHER' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_NEITHER', 'Neither' )
																array( 'fieldtitle' => 'GENDER_NEUTROIS', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_NEUTROIS' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_NEUTROIS', 'Neutrois' )
																array( 'fieldtitle' => 'GENDER_NON-BINARY', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_NON-BINARY' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_NON-BINARY', 'Non-binary' )
																array( 'fieldtitle' => 'GENDER_OTHER', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_OTHER' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_OTHER', 'Other' )
																array( 'fieldtitle' => 'GENDER_PANGENDER', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_PANGENDER' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_PANGENDER', 'Pangender' )
																array( 'fieldtitle' => 'GENDER_POLYGENDER', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_POLYGENDER' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_POLYGENDER', 'Polygender' )
																array( 'fieldtitle' => 'GENDER_TOTHERMAN', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_TOTHERMAN' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_TOTHERMAN', 'T* Man' )
																array( 'fieldtitle' => 'GENDER_TOTHERWOMAN', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_TOTHERWOMAN' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_TOTHERWOMAN', 'T* Woman' )
																array( 'fieldtitle' => 'GENDER_TRANS', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_TRANS' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_TRANS', 'Trans' )
																array( 'fieldtitle' => 'GENDER_TRANSFEMALE', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_TRANSFEMALE' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_TRANSFEMALE', 'Trans Female' )
																array( 'fieldtitle' => 'GENDER_TRANSMALE', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_TRANSMALE' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_TRANSMALE', 'Trans Male' )
																array( 'fieldtitle' => 'GENDER_TRANSMAN', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_TRANSMAN' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_TRANSMAN', 'Trans Man' )
																array( 'fieldtitle' => 'GENDER_TRANSPERSON', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_TRANSPERSON' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_TRANSPERSON', 'Trans Person' )
																array( 'fieldtitle' => 'GENDER_TRANSOTHERFEMALE', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_TRANSOTHERFEMALE' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_TRANSOTHERFEMALE', 'Trans*Female' )
																array( 'fieldtitle' => 'GENDER_TRANSOTHERMALE', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_TRANSOTHERMALE' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_TRANSOTHERMALE', 'Trans*Male' )
																array( 'fieldtitle' => 'GENDER_TRANSOTHERMAN', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_TRANSOTHERMAN' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_TRANSOTHERMAN', 'Trans*Man' )
																array( 'fieldtitle' => 'GENDER_TRANSOTHERPERSON', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_TRANSOTHERPERSON' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_TRANSOTHERPERSON', 'Trans*Person' )
																array( 'fieldtitle' => 'GENDER_TRANSOTHERWOMAN', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_TRANSOTHERWOMAN' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_TRANSOTHERWOMAN', 'Trans*Woman' )
																array( 'fieldtitle' => 'GENDER_TRANSEXUAL', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_TRANSEXUAL' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_TRANSEXUAL', 'Transexual' )
																array( 'fieldtitle' => 'GENDER_TRANSEXUALFEMALE', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_TRANSEXUALFEMALE' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_TRANSEXUALFEMALE', 'Transexual Female' )
																array( 'fieldtitle' => 'GENDER_TRANSEXUALMALE', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_TRANSEXUALMALE' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_TRANSEXUALMALE', 'Transexual Male' )
																array( 'fieldtitle' => 'GENDER_TRANSEXUALMAN', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_TRANSEXUALMAN' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_TRANSEXUALMAN', 'Transexual Man' )
																array( 'fieldtitle' => 'GENDER_TRANSEXUALPERSON', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_TRANSEXUALPERSON' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_TRANSEXUALPERSON', 'Transexual Person' )
																array( 'fieldtitle' => 'GENDER_TRANSEXUALWOMAN', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_TRANSEXUALWOMAN' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_TRANSEXUALWOMAN', 'Transexual Woman' )
																array( 'fieldtitle' => 'GENDER_TRANSGENDERFEMALE', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_TRANSGENDERFEMALE' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_TRANSGENDERFEMALE', 'Transgender Female' )
																array( 'fieldtitle' => 'GENDER_TRANSGENDERPERSON', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_TRANSGENDERPERSON' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_TRANSGENDERPERSON', 'Transgender Person' )
																array( 'fieldtitle' => 'GENDER_TRANSMASCULINE', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_TRANSMASCULINE' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_TRANSMASCULINE', 'Transmasculine' )
																array( 'fieldtitle' => 'GENDER_TWOOTHERPERSON', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_TWOOTHERPERSON' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_TWOOTHERPERSON', 'Two* Person' )
																array( 'fieldtitle' => 'GENDER_TWOSPIRIT', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_TWOSPIRIT' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_TWOSPIRIT', 'Two-spirit' )
																array( 'fieldtitle' => 'GENDER_TWOSPIRITPERSON', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_TWOSPIRITPERSON' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_TWOSPIRITPERSON', 'Two-spirit Person' )
																array( 'fieldtitle' => 'GENDER_WOMAN', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_WOMAN' ) // CBTxt::T( 'PROFILE_DETAILS_GENDER_WOMAN', 'Woman' )
															) );

					$detailsFieldParams->set( 'field_display_style', 0 );
					$detailsFieldParams->set( 'cbprivacy_display', 1 );
					$detailsFieldParams->set( 'cbprivacy_simple_group', 'PROFILE_DETAILS_PRIVACY' );
					$detailsFieldParams->set( 'privacy_class', 'd-block mt-1' );
					$detailsFieldParams->set( 'privacy_options_default', 0 );
					$detailsFieldParams->set( 'ajax_profile', 1 );
					$detailsFieldParams->set( 'ajax_profile_strict', 1 );
					$detailsFieldParams->set( 'ajax_template', 'cog' );
					$detailsFieldParams->set( 'ajax_fields_titles', 4 );
					break;
				case 'cb_details_interestedin':
					$detailsField->set( 'title', 'PROFILE_DETAILS_INTERESTEDIN' ); // CBTxt::T( 'PROFILE_DETAILS_INTERESTEDIN', 'Interested In' )
					$detailsField->set( 'type', 'tag' );
					$detailsField->set( 'searchable', 1 );
					$detailsField->set( 'profile', 1 );
					$detailsField->set( 'edit', 1 );

					$detailsField->set( '_fieldvalues', array(	array( 'fieldtitle' => 'INTERESTEDIN_MEN', 'fieldlabel' => 'PROFILE_DETAILS_INTERESTEDIN_MEN' ), // CBTxt::T( 'PROFILE_DETAILS_INTERESTEDIN_MEN', 'Men' )
																array( 'fieldtitle' => 'INTERESTEDIN_WOMEN', 'fieldlabel' => 'PROFILE_DETAILS_INTERESTEDIN_WOMEN' ), // CBTxt::T( 'PROFILE_DETAILS_INTERESTEDIN_WOMEN', 'Women' )
																array( 'fieldtitle' => 'GENDER_AGENDER', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_AGENDER' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_AGENDER', 'Agender' )
																array( 'fieldtitle' => 'GENDER_ANDROGYNE', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_ANDROGYNE' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_ANDROGYNE', 'Androgyne' )
																array( 'fieldtitle' => 'GENDER_ANDROGYNES', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_ANDROGYNES' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_ANDROGYNES', 'Androgynes' )
																array( 'fieldtitle' => 'GENDER_ANDROGYNOUS', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_ANDROGYNOUS' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_ANDROGYNOUS', 'Androgynous' )
																array( 'fieldtitle' => 'GENDER_ASEXUAL', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_ASEXUAL' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_ASEXUAL', 'Asexual' )
																array( 'fieldtitle' => 'GENDER_BIGENDER', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_BIGENDER' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_BIGENDER', 'Bigender' )
																array( 'fieldtitle' => 'GENDER_CIS', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_CIS' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_CIS', 'Cis' )
																array( 'fieldtitle' => 'INTERESTEDIN_CISMEN', 'fieldlabel' => 'PROFILE_DETAILS_INTERESTEDIN_CISMEN' ), // CBTxt::T( 'PROFILE_DETAILS_INTERESTEDIN_CISMEN', 'Cis Men' )
																array( 'fieldtitle' => 'INTERESTEDIN_CISWOMEN', 'fieldlabel' => 'PROFILE_DETAILS_INTERESTEDIN_CISWOMEN' ), // CBTxt::T( 'PROFILE_DETAILS_INTERESTEDIN_CISWOMEN', 'Cis Women' )
																array( 'fieldtitle' => 'GENDER_CISGENDER', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_CISGENDER' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_CISGENDER', 'Cisgender' )
																array( 'fieldtitle' => 'INTERESTEDIN_CISGENDERMEN', 'fieldlabel' => 'PROFILE_DETAILS_INTERESTEDIN_CISGENDERMEN' ), // CBTxt::T( 'PROFILE_DETAILS_INTERESTEDIN_CISGENDERMEN', 'Cisgender Men' )
																array( 'fieldtitle' => 'INTERESTEDIN_CISGENDERWOMEN', 'fieldlabel' => 'PROFILE_DETAILS_INTERESTEDIN_CISGENDERWOMEN' ), // CBTxt::T( 'PROFILE_DETAILS_INTERESTEDIN_CISGENDERWOMEN', 'Cisgender Women' )
																array( 'fieldtitle' => 'GENDER_F2M', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_F2M' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_F2M', 'F2M' )
																array( 'fieldtitle' => 'GENDER_FTM', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_FTM' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_FTM', 'FTM' )
																array( 'fieldtitle' => 'GENDER_FEMALETOMALE', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_FEMALETOMALE' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_FEMALETOMALE', 'Female to Male' )
																array( 'fieldtitle' => 'INTERESTEDIN_FEMALETOMALETRANSMEN', 'fieldlabel' => 'PROFILE_DETAILS_INTERESTEDIN_FEMALETOMALETRANSMEN' ), // CBTxt::T( 'PROFILE_DETAILS_INTERESTEDIN_FEMALETOMALETRANSMEN', 'Female to Male Trans Men' )
																array( 'fieldtitle' => 'INTERESTEDIN_FEMALETOMALETRANSGENDERMEN', 'fieldlabel' => 'PROFILE_DETAILS_INTERESTEDIN_FEMALETOMALETRANSGENDERMEN' ), // CBTxt::T( 'PROFILE_DETAILS_INTERESTEDIN_FEMALETOMALETRANSGENDERMEN', 'Female to Male Transgender Men' )
																array( 'fieldtitle' => 'INTERESTEDIN_FEMALETOMALETRANSSEXUALMEN', 'fieldlabel' => 'PROFILE_DETAILS_INTERESTEDIN_FEMALETOMALETRANSSEXUALMEN' ), // CBTxt::T( 'PROFILE_DETAILS_INTERESTEDIN_FEMALETOMALETRANSSEXUALMEN', 'Female to Male Transsexual Men' )
																array( 'fieldtitle' => 'GENDER_GENDERFLUID', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_GENDERFLUID' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_GENDERFLUID', 'Gender Fluid' )
																array( 'fieldtitle' => 'GENDER_GENDERNONCONFORMING', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_GENDERNONCONFORMING' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_GENDERNONCONFORMING', 'Gender Nonconforming' )
																array( 'fieldtitle' => 'GENDER_GENDERQUESTIONING', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_GENDERQUESTIONING' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_GENDERQUESTIONING', 'Gender Questioning' )
																array( 'fieldtitle' => 'GENDER_GENDERVARIANT', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_GENDERVARIANT' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_GENDERVARIANT', 'Gender Variant' )
																array( 'fieldtitle' => 'GENDER_GENDERNEUTRAL', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_GENDERNEUTRAL' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_GENDERNEUTRAL', 'Gender Neutral' )
																array( 'fieldtitle' => 'GENDER_GENDERQUEER', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_GENDERQUEER' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_GENDERQUEER', 'Genderqueer' )
																array( 'fieldtitle' => 'GENDER_HERMAPHRODITE', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_HERMAPHRODITE' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_HERMAPHRODITE', 'Hermaphrodite' )
																array( 'fieldtitle' => 'GENDER_INTERSEX', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_INTERSEX' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_INTERSEX', 'Intersex' )
																array( 'fieldtitle' => 'INTERESTEDIN_INTERSEXMEN', 'fieldlabel' => 'PROFILE_DETAILS_INTERESTEDIN_INTERSEXMEN' ), // CBTxt::T( 'PROFILE_DETAILS_INTERESTEDIN_INTERSEXMEN', 'Intersex Men' )
																array( 'fieldtitle' => 'GENDER_INTERSEXPERSON', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_INTERSEXPERSON' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_INTERSEXPERSON', 'Intersex Person' )
																array( 'fieldtitle' => 'INTERESTEDIN_INTERSEXWOMEN', 'fieldlabel' => 'PROFILE_DETAILS_INTERESTEDIN_INTERSEXWOMEN' ), // CBTxt::T( 'PROFILE_DETAILS_INTERESTEDIN_INTERSEXWOMEN', 'Intersex Women' )
																array( 'fieldtitle' => 'GENDER_M2F', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_M2F' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_M2F', 'M2F' )
																array( 'fieldtitle' => 'GENDER_MTF', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_MTF' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_MTF', 'MTF' )
																array( 'fieldtitle' => 'GENDER_MALETOFEMALE', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_MALETOFEMALE' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_MALETOFEMALE', 'Male to Female' )
																array( 'fieldtitle' => 'INTERESTEDIN_MALETOFEMALETRANSWOMEN', 'fieldlabel' => 'PROFILE_DETAILS_INTERESTEDIN_MALETOFEMALETRANSWOMEN' ), // CBTxt::T( 'PROFILE_DETAILS_INTERESTEDIN_MALETOFEMALETRANSWOMEN', 'Male to Female Trans Women' )
																array( 'fieldtitle' => 'INTERESTEDIN_MALETOFEMALETRANSGENDERWOMEN', 'fieldlabel' => 'PROFILE_DETAILS_INTERESTEDIN_MALETOFEMALETRANSGENDERWOMEN' ), // CBTxt::T( 'PROFILE_DETAILS_INTERESTEDIN_MALETOFEMALETRANSGENDERWOMEN', 'Male to Female Transgender Women' )
																array( 'fieldtitle' => 'INTERESTEDIN_MALETOFEMALETRANSSEXUALWOMEN', 'fieldlabel' => 'PROFILE_DETAILS_INTERESTEDIN_MALETOFEMALETRANSSEXUALWOMEN' ), // CBTxt::T( 'PROFILE_DETAILS_INTERESTEDIN_MALETOFEMALETRANSSEXUALWOMEN', 'Male to Female Transsexual Women' )
																array( 'fieldtitle' => 'GENDER_NEITHER', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_NEITHER' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_NEITHER', 'Neither' )
																array( 'fieldtitle' => 'GENDER_NEUTROIS', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_NEUTROIS' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_NEUTROIS', 'Neutrois' )
																array( 'fieldtitle' => 'GENDER_NON-BINARY', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_NON-BINARY' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_NON-BINARY', 'Non-binary' )
																array( 'fieldtitle' => 'GENDER_OTHER', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_OTHER' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_OTHER', 'Other' )
																array( 'fieldtitle' => 'GENDER_PANGENDER', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_PANGENDER' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_PANGENDER', 'Pangender' )
																array( 'fieldtitle' => 'GENDER_POLYGENDER', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_POLYGENDER' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_POLYGENDER', 'Polygender' )
																array( 'fieldtitle' => 'INTERESTEDIN_TOTHERMEN', 'fieldlabel' => 'PROFILE_DETAILS_INTERESTEDIN_TOTHERMEN' ), // CBTxt::T( 'PROFILE_DETAILS_INTERESTEDIN_TOTHERMEN', 'T* Men' )
																array( 'fieldtitle' => 'INTERESTEDIN_TOTHERWOMEN', 'fieldlabel' => 'PROFILE_DETAILS_INTERESTEDIN_TOTHERWOMEN' ), // CBTxt::T( 'PROFILE_DETAILS_INTERESTEDIN_TOTHERWOMEN', 'T* Women' )
																array( 'fieldtitle' => 'GENDER_TRANS', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_TRANS' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_TRANS', 'Trans' )
																array( 'fieldtitle' => 'GENDER_TRANSFEMALE', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_TRANSFEMALE' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_TRANSFEMALE', 'Trans Female' )
																array( 'fieldtitle' => 'GENDER_TRANSMALE', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_TRANSMALE' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_TRANSMALE', 'Trans Male' )
																array( 'fieldtitle' => 'INTERESTEDIN_TRANSMEN', 'fieldlabel' => 'PROFILE_DETAILS_INTERESTEDIN_TRANSMEN' ), // CBTxt::T( 'PROFILE_DETAILS_INTERESTEDIN_TRANSMEN', 'Trans Men' )
																array( 'fieldtitle' => 'GENDER_TRANSPERSON', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_TRANSPERSON' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_TRANSPERSON', 'Trans Person' )
																array( 'fieldtitle' => 'GENDER_TRANSOTHERFEMALE', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_TRANSOTHERFEMALE' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_TRANSOTHERFEMALE', 'Trans*Female' )
																array( 'fieldtitle' => 'GENDER_TRANSOTHERMALE', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_TRANSOTHERMALE' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_TRANSOTHERMALE', 'Trans*Male' )
																array( 'fieldtitle' => 'INTERESTEDIN_TRANSOTHERMEN', 'fieldlabel' => 'PROFILE_DETAILS_INTERESTEDIN_TRANSOTHERMEN' ), // CBTxt::T( 'PROFILE_DETAILS_INTERESTEDIN_TRANSOTHERMEN', 'Trans*Men' )
																array( 'fieldtitle' => 'GENDER_TRANSOTHERPERSON', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_TRANSOTHERPERSON' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_TRANSOTHERPERSON', 'Trans*Person' )
																array( 'fieldtitle' => 'INTERESTEDIN_TRANSOTHERWOMEN', 'fieldlabel' => 'PROFILE_DETAILS_INTERESTEDIN_TRANSOTHERWOMEN' ), // CBTxt::T( 'PROFILE_DETAILS_INTERESTEDIN_TRANSOTHERWOMEN', 'Trans*Women' )
																array( 'fieldtitle' => 'GENDER_TRANSEXUAL', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_TRANSEXUAL' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_TRANSEXUAL', 'Transexual' )
																array( 'fieldtitle' => 'GENDER_TRANSEXUALFEMALE', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_TRANSEXUALFEMALE' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_TRANSEXUALFEMALE', 'Transexual Female' )
																array( 'fieldtitle' => 'GENDER_TRANSEXUALMALE', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_TRANSEXUALMALE' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_TRANSEXUALMALE', 'Transexual Male' )
																array( 'fieldtitle' => 'INTERESTEDIN_TRANSEXUALMEN', 'fieldlabel' => 'PROFILE_DETAILS_INTERESTEDIN_TRANSEXUALMEN' ), // CBTxt::T( 'PROFILE_DETAILS_INTERESTEDIN_TRANSEXUALMEN', 'Transexual Men' )
																array( 'fieldtitle' => 'GENDER_TRANSEXUALPERSON', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_TRANSEXUALPERSON' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_TRANSEXUALPERSON', 'Transexual Person' )
																array( 'fieldtitle' => 'INTERESTEDIN_TRANSEXUALWOMEN', 'fieldlabel' => 'PROFILE_DETAILS_INTERESTEDIN_TRANSEXUALWOMEN' ), // CBTxt::T( 'PROFILE_DETAILS_INTERESTEDIN_TRANSEXUALWOMEN', 'Transexual Women' )
																array( 'fieldtitle' => 'GENDER_TRANSGENDERFEMALE', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_TRANSGENDERFEMALE' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_TRANSGENDERFEMALE', 'Transgender Female' )
																array( 'fieldtitle' => 'GENDER_TRANSGENDERPERSON', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_TRANSGENDERPERSON' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_TRANSGENDERPERSON', 'Transgender Person' )
																array( 'fieldtitle' => 'GENDER_TRANSMASCULINE', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_TRANSMASCULINE' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_TRANSMASCULINE', 'Transmasculine' )
																array( 'fieldtitle' => 'GENDER_TWOOTHERPERSON', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_TWOOTHERPERSON' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_TWOOTHERPERSON', 'Two* Person' )
																array( 'fieldtitle' => 'GENDER_TWOSPIRIT', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_TWOSPIRIT' ), // CBTxt::T( 'PROFILE_DETAILS_GENDER_TWOSPIRIT', 'Two-spirit' )
																array( 'fieldtitle' => 'GENDER_TWOSPIRITPERSON', 'fieldlabel' => 'PROFILE_DETAILS_GENDER_TWOSPIRITPERSON' ) // CBTxt::T( 'PROFILE_DETAILS_GENDER_TWOSPIRITPERSON', 'Two-spirit Person' )
															) );

					$detailsFieldParams->set( 'field_display_style', 0 );
					$detailsFieldParams->set( 'cbprivacy_display', 1 );
					$detailsFieldParams->set( 'cbprivacy_simple_group', 'PROFILE_DETAILS_PRIVACY' );
					$detailsFieldParams->set( 'privacy_class', 'd-block mt-1' );
					$detailsFieldParams->set( 'privacy_options_default', 0 );
					$detailsFieldParams->set( 'ajax_profile', 1 );
					$detailsFieldParams->set( 'ajax_profile_strict', 1 );
					$detailsFieldParams->set( 'ajax_template', 'cog' );
					$detailsFieldParams->set( 'ajax_fields_titles', 4 );
					break;
				case 'cb_details_relationship':
					$detailsField->set( 'title', 'PROFILE_DETAILS_RELATIONSHIP' ); // CBTxt::T( 'PROFILE_DETAILS_RELATIONSHIP', 'Relationship' )
					$detailsField->set( 'type', 'select' );
					$detailsField->set( 'searchable', 1 );
					$detailsField->set( 'profile', 1 );
					$detailsField->set( 'edit', 1 );

					$detailsField->set( '_fieldvalues', array(	array( 'fieldtitle' => '', 'fieldlabel' => 'PROFILE_DETAILS_RELATIONSHIP_NONE' ), // CBTxt::T( 'PROFILE_DETAILS_RELATIONSHIP_NONE', 'Unspecified' )
																array( 'fieldtitle' => 'RELATIONSHIP_SINGLE', 'fieldlabel' => 'PROFILE_DETAILS_RELATIONSHIP_SINGLE' ), // CBTxt::T( 'PROFILE_DETAILS_RELATIONSHIP_SINGLE', 'Single' )
																array( 'fieldtitle' => 'RELATIONSHIP_IN', 'fieldlabel' => 'PROFILE_DETAILS_RELATIONSHIP_IN' ), // CBTxt::T( 'PROFILE_DETAILS_RELATIONSHIP_IN', 'In a Relationship' )
																array( 'fieldtitle' => 'RELATIONSHIP_ENGAGED', 'fieldlabel' => 'PROFILE_DETAILS_RELATIONSHIP_ENGAGED' ), // CBTxt::T( 'PROFILE_DETAILS_RELATIONSHIP_ENGAGED', 'Engaged' )
																array( 'fieldtitle' => 'RELATIONSHIP_MARRIED', 'fieldlabel' => 'PROFILE_DETAILS_RELATIONSHIP_MARRIED' ), // CBTxt::T( 'PROFILE_DETAILS_RELATIONSHIP_MARRIED', 'Married' )
																array( 'fieldtitle' => 'RELATIONSHIP_CIVIL', 'fieldlabel' => 'PROFILE_DETAILS_RELATIONSHIP_CIVIL' ), // CBTxt::T( 'PROFILE_DETAILS_RELATIONSHIP_CIVIL', 'In a Civil Union' )
																array( 'fieldtitle' => 'RELATIONSHIP_DOMESTIC', 'fieldlabel' => 'PROFILE_DETAILS_RELATIONSHIP_DOMESTIC' ), // CBTxt::T( 'PROFILE_DETAILS_RELATIONSHIP_DOMESTIC', 'In a Domestic Partnership' )
																array( 'fieldtitle' => 'RELATIONSHIP_OPEN', 'fieldlabel' => 'PROFILE_DETAILS_RELATIONSHIP_OPEN' ), // CBTxt::T( 'PROFILE_DETAILS_RELATIONSHIP_OPEN', 'In an Open Relationship' )
																array( 'fieldtitle' => 'RELATIONSHIP_COMPLICATED', 'fieldlabel' => 'PROFILE_DETAILS_RELATIONSHIP_COMPLICATED' ), // CBTxt::T( 'PROFILE_DETAILS_RELATIONSHIP_COMPLICATED', "It's Complicated" )
																array( 'fieldtitle' => 'RELATIONSHIP_SEPARATED', 'fieldlabel' => 'PROFILE_DETAILS_RELATIONSHIP_SEPARATED' ), // CBTxt::T( 'PROFILE_DETAILS_RELATIONSHIP_SEPARATED', 'Separated' )
																array( 'fieldtitle' => 'RELATIONSHIP_DIVORCED', 'fieldlabel' => 'PROFILE_DETAILS_RELATIONSHIP_DIVORCED' ), // CBTxt::T( 'PROFILE_DETAILS_RELATIONSHIP_DIVORCED', 'Divorced' )
																array( 'fieldtitle' => 'RELATIONSHIP_WIDOWED', 'fieldlabel' => 'PROFILE_DETAILS_RELATIONSHIP_WIDOWED' ) // CBTxt::T( 'PROFILE_DETAILS_RELATIONSHIP_WIDOWED', 'Widowed' )
															) );

					$detailsFieldParams->set( 'cbprivacy_display', 1 );
					$detailsFieldParams->set( 'cbprivacy_simple_group', 'PROFILE_DETAILS_PRIVACY' );
					$detailsFieldParams->set( 'privacy_class', 'd-block mt-1' );
					$detailsFieldParams->set( 'privacy_options_default', 0 );
					$detailsFieldParams->set( 'ajax_profile', 1 );
					$detailsFieldParams->set( 'ajax_profile_strict', 1 );
					$detailsFieldParams->set( 'ajax_template', 'cog' );
					break;
				case 'cb_details_language':
					$detailsField->set( 'title', 'PROFILE_DETAILS_LANGUAGES' ); // CBTxt::T( 'PROFILE_DETAILS_LANGUAGES', 'Languages' )
					$detailsField->set( 'type', 'tag' );
					$detailsField->set( 'searchable', 1 );
					$detailsField->set( 'profile', 1 );
					$detailsField->set( 'edit', 1 );

					$detailsField->set( '_fieldvalues', array(	array( 'fieldtitle' => 'LANGUAGE_ENGLISH', 'fieldlabel' => 'PROFILE_DETAILS_LANGUAGE_ENGLISH' ), // CBTxt::T( 'PROFILE_DETAILS_LANGUAGE_ENGLISH', 'English' )
																array( 'fieldtitle' => 'LANGUAGE_SPANISH', 'fieldlabel' => 'PROFILE_DETAILS_LANGUAGE_SPANISH' ), // CBTxt::T( 'PROFILE_DETAILS_LANGUAGE_SPANISH', 'Spanish' )
																array( 'fieldtitle' => 'LANGUAGE_MANDARIN', 'fieldlabel' => 'PROFILE_DETAILS_LANGUAGE_MANDARIN' ), // CBTxt::T( 'PROFILE_DETAILS_LANGUAGE_MANDARIN', 'Mandarin' )
																array( 'fieldtitle' => 'LANGUAGE_JAPANESE', 'fieldlabel' => 'PROFILE_DETAILS_LANGUAGE_JAPANESE' ), // CBTxt::T( 'PROFILE_DETAILS_LANGUAGE_JAPANESE', 'Japanese' )
																array( 'fieldtitle' => 'LANGUAGE_HINDI', 'fieldlabel' => 'PROFILE_DETAILS_LANGUAGE_HINDI' ), // CBTxt::T( 'PROFILE_DETAILS_LANGUAGE_HINDI', 'Hindi' )
																array( 'fieldtitle' => 'LANGUAGE_ARABIC', 'fieldlabel' => 'PROFILE_DETAILS_LANGUAGE_ARABIC' ), // CBTxt::T( 'PROFILE_DETAILS_LANGUAGE_ARABIC', 'Arabic' )
																array( 'fieldtitle' => 'LANGUAGE_PORTUQUESE', 'fieldlabel' => 'PROFILE_DETAILS_LANGUAGE_PORTUQUESE' ), // CBTxt::T( 'PROFILE_DETAILS_LANGUAGE_PORTUQUESE', 'Portuguese' )
																array( 'fieldtitle' => 'LANGUAGE_RUSSIAN', 'fieldlabel' => 'PROFILE_DETAILS_LANGUAGE_RUSSIAN' ), // CBTxt::T( 'PROFILE_DETAILS_LANGUAGE_RUSSIAN', 'Russian' )
																array( 'fieldtitle' => 'LANGUAGE_GERMAN', 'fieldlabel' => 'PROFILE_DETAILS_LANGUAGE_GERMAN' ), // CBTxt::T( 'PROFILE_DETAILS_LANGUAGE_GERMAN', 'German' )
																array( 'fieldtitle' => 'LANGUAGE_VIETNAMESE', 'fieldlabel' => 'PROFILE_DETAILS_LANGUAGE_VIETNAMESE' ), // CBTxt::T( 'PROFILE_DETAILS_LANGUAGE_VIETNAMESE', 'Vietnamese' )
																array( 'fieldtitle' => 'LANGUAGE_KOREAN', 'fieldlabel' => 'PROFILE_DETAILS_LANGUAGE_KOREAN' ), // CBTxt::T( 'PROFILE_DETAILS_LANGUAGE_KOREAN', 'Korean' )
																array( 'fieldtitle' => 'LANGUAGE_FRENCH', 'fieldlabel' => 'PROFILE_DETAILS_LANGUAGE_FRENCH' ), // CBTxt::T( 'PROFILE_DETAILS_LANGUAGE_FRENCH', 'French' )
																array( 'fieldtitle' => 'LANGUAGE_TURKISH', 'fieldlabel' => 'PROFILE_DETAILS_LANGUAGE_TURKISH' ), // CBTxt::T( 'PROFILE_DETAILS_LANGUAGE_TURKISH', 'Turkish' )
																array( 'fieldtitle' => 'LANGUAGE_ITALIAN', 'fieldlabel' => 'PROFILE_DETAILS_LANGUAGE_ITALIAN' ), // CBTxt::T( 'PROFILE_DETAILS_LANGUAGE_ITALIAN', 'Italian' )
																array( 'fieldtitle' => 'LANGUAGE_POLISH', 'fieldlabel' => 'PROFILE_DETAILS_LANGUAGE_POLISH' ), // CBTxt::T( 'PROFILE_DETAILS_LANGUAGE_POLISH', 'Polish' )
																array( 'fieldtitle' => 'LANGUAGE_UKRAINIAN', 'fieldlabel' => 'PROFILE_DETAILS_LANGUAGE_UKRAINIAN' ), // CBTxt::T( 'PROFILE_DETAILS_LANGUAGE_UKRAINIAN', 'Ukrainian' )
																array( 'fieldtitle' => 'LANGUAGE_ROMANIAN', 'fieldlabel' => 'PROFILE_DETAILS_LANGUAGE_ROMANIAN' ), // CBTxt::T( 'PROFILE_DETAILS_LANGUAGE_ROMANIAN', 'Romanian' )
																array( 'fieldtitle' => 'LANGUAGE_DUTCH', 'fieldlabel' => 'PROFILE_DETAILS_LANGUAGE_DUTCH' ), // CBTxt::T( 'PROFILE_DETAILS_LANGUAGE_DUTCH', 'Dutch' )
																array( 'fieldtitle' => 'LANGUAGE_KURDISH', 'fieldlabel' => 'PROFILE_DETAILS_LANGUAGE_KURDISH' ), // CBTxt::T( 'PROFILE_DETAILS_LANGUAGE_KURDISH', 'Kurdish' )
																array( 'fieldtitle' => 'LANGUAGE_SOMALI', 'fieldlabel' => 'PROFILE_DETAILS_LANGUAGE_SOMALI' ), // CBTxt::T( 'PROFILE_DETAILS_LANGUAGE_SOMALI', 'Somali' )
																array( 'fieldtitle' => 'LANGUAGE_HUNGARIAN', 'fieldlabel' => 'PROFILE_DETAILS_LANGUAGE_HUNGARIAN' ), // CBTxt::T( 'PROFILE_DETAILS_LANGUAGE_HUNGARIAN', 'Hungarian' )
																array( 'fieldtitle' => 'LANGUAGE_GREEK', 'fieldlabel' => 'PROFILE_DETAILS_LANGUAGE_GREEK' ), // CBTxt::T( 'PROFILE_DETAILS_LANGUAGE_GREEK', 'Greek' )
																array( 'fieldtitle' => 'LANGUAGE_CZECH', 'fieldlabel' => 'PROFILE_DETAILS_LANGUAGE_CZECH' ), // CBTxt::T( 'PROFILE_DETAILS_LANGUAGE_CZECH', 'Czech' )
																array( 'fieldtitle' => 'LANGUAGE_SWEDISH', 'fieldlabel' => 'PROFILE_DETAILS_LANGUAGE_SWEDISH' ), // CBTxt::T( 'PROFILE_DETAILS_LANGUAGE_SWEDISH', 'Swedish' )
																array( 'fieldtitle' => 'LANGUAGE_BENGALI', 'fieldlabel' => 'PROFILE_DETAILS_LANGUAGE_BENGALI' ) // CBTxt::T( 'PROFILE_DETAILS_LANGUAGE_BENGALI', 'Bengali' )
															) );

					$detailsFieldParams->set( 'field_display_style', 0 );
					$detailsFieldParams->set( 'cbprivacy_display', 1 );
					$detailsFieldParams->set( 'cbprivacy_simple_group', 'PROFILE_DETAILS_PRIVACY' );
					$detailsFieldParams->set( 'privacy_class', 'd-block mt-1' );
					$detailsFieldParams->set( 'privacy_options_default', 0 );
					$detailsFieldParams->set( 'ajax_profile', 1 );
					$detailsFieldParams->set( 'ajax_profile_strict', 1 );
					$detailsFieldParams->set( 'ajax_template', 'cog' );
					break;
				case 'cb_details_religion':
					$detailsField->set( 'title', 'PROFILE_DETAILS_RELIGION' ); // CBTxt::T( 'PROFILE_DETAILS_RELIGION', 'Religion' )
					$detailsField->set( 'type', 'tag' );
					$detailsField->set( 'searchable', 1 );
					$detailsField->set( 'profile', 1 );
					$detailsField->set( 'edit', 1 );

					$detailsField->set( '_fieldvalues', array(	array( 'fieldtitle' => 'RELIGION_CHRISTIAN', 'fieldlabel' => 'PROFILE_DETAILS_RELIGION_CHRISTIAN' ), // CBTxt::T( 'PROFILE_DETAILS_RELIGION_CHRISTIAN', 'Christian' )
																array( 'fieldtitle' => 'RELIGION_ATHEIST', 'fieldlabel' => 'PROFILE_DETAILS_RELIGION_ATHEIST' ), // CBTxt::T( 'PROFILE_DETAILS_RELIGION_ATHEIST', 'Atheist' )
																array( 'fieldtitle' => 'RELIGION_HINDUISM', 'fieldlabel' => 'PROFILE_DETAILS_RELIGION_HINDUISM' ), // CBTxt::T( 'PROFILE_DETAILS_RELIGION_HINDUISM', 'Hinduism' )
																array( 'fieldtitle' => 'RELIGION_ISLAM', 'fieldlabel' => 'PROFILE_DETAILS_RELIGION_ISLAM' ), // CBTxt::T( 'PROFILE_DETAILS_RELIGION_ISLAM', 'Islam' )
																array( 'fieldtitle' => 'RELIGION_BUDDHISM', 'fieldlabel' => 'PROFILE_DETAILS_RELIGION_BUDDHISM' ), // CBTxt::T( 'PROFILE_DETAILS_RELIGION_BUDDHISM', 'Buddhism' )
																array( 'fieldtitle' => 'RELIGION_CHINESE_TRADITIONAL', 'fieldlabel' => 'PROFILE_DETAILS_RELIGION_CHINESE_TRADITIONAL' ), // CBTxt::T( 'PROFILE_DETAILS_RELIGION_CHINESE_TRADITIONAL', 'Chinese Traditional' )
																array( 'fieldtitle' => 'RELIGION_SHINTO', 'fieldlabel' => 'PROFILE_DETAILS_RELIGION_SHINTO' ), // CBTxt::T( 'PROFILE_DETAILS_RELIGION_SHINTO', 'Shinto' )
																array( 'fieldtitle' => 'RELIGION_AFRICAN_TRADITIONAL', 'fieldlabel' => 'PROFILE_DETAILS_RELIGION_AFRICAN_TRADITIONAL' ), // CBTxt::T( 'PROFILE_DETAILS_RELIGION_AFRICAN_TRADITIONAL', 'African Traditional' )
																array( 'fieldtitle' => 'RELIGION_AGNOSTIC', 'fieldlabel' => 'PROFILE_DETAILS_RELIGION_AGNOSTIC' ), // CBTxt::T( 'PROFILE_DETAILS_RELIGION_AGNOSTIC', 'Agnostic' )
																array( 'fieldtitle' => 'RELIGION_BAHAI', 'fieldlabel' => 'PROFILE_DETAILS_RELIGION_BAHAI' ), // CBTxt::T( 'PROFILE_DETAILS_RELIGION_BAHAI', "Baha'i" )
																array( 'fieldtitle' => 'RELIGION_CAO_DAI', 'fieldlabel' => 'PROFILE_DETAILS_RELIGION_CAO_DAI' ), // CBTxt::T( 'PROFILE_DETAILS_RELIGION_CAO_DAI', 'Cao Dai' )
																array( 'fieldtitle' => 'RELIGION_JAINISM', 'fieldlabel' => 'PROFILE_DETAILS_RELIGION_JAINISM' ), // CBTxt::T( 'PROFILE_DETAILS_RELIGION_JAINISM', 'Jainism' )
																array( 'fieldtitle' => 'RELIGION_JUCHE', 'fieldlabel' => 'PROFILE_DETAILS_RELIGION_JUCHE' ), // CBTxt::T( 'PROFILE_DETAILS_RELIGION_JUCHE', 'Juche' )
																array( 'fieldtitle' => 'RELIGION_JUDAISM', 'fieldlabel' => 'PROFILE_DETAILS_RELIGION_JUDAISM' ), // CBTxt::T( 'PROFILE_DETAILS_RELIGION_JUDAISM', 'Judaism' )
																array( 'fieldtitle' => 'RELIGION_NEO_PAGANISM', 'fieldlabel' => 'PROFILE_DETAILS_RELIGION_NEO_PAGANISM' ), // CBTxt::T( 'PROFILE_DETAILS_RELIGION_NEO_PAGANISM', 'Neo-Paganism' )
																array( 'fieldtitle' => 'RELIGION_NON_RELIGIOUS', 'fieldlabel' => 'PROFILE_DETAILS_RELIGION_NON_RELIGIOUS' ), // CBTxt::T( 'PROFILE_DETAILS_RELIGION_NON_RELIGIOUS', 'Non-Religious' )
																array( 'fieldtitle' => 'RELIGION_PRIMAL_INDIGENOUS', 'fieldlabel' => 'PROFILE_DETAILS_RELIGION_PRIMAL_INDIGENOUS' ), // CBTxt::T( 'PROFILE_DETAILS_RELIGION_PRIMAL_INDIGENOUS', 'Primal Indigenous' )
																array( 'fieldtitle' => 'RELIGION_RASTAFARIANISM', 'fieldlabel' => 'PROFILE_DETAILS_RELIGION_RASTAFARIANISM' ), // CBTxt::T( 'PROFILE_DETAILS_RELIGION_RASTAFARIANISM', 'Rastafarianism' )
																array( 'fieldtitle' => 'RELIGION_SECULAR', 'fieldlabel' => 'PROFILE_DETAILS_RELIGION_SECULAR' ), // CBTxt::T( 'PROFILE_DETAILS_RELIGION_SECULAR', 'Secular' )
																array( 'fieldtitle' => 'RELIGION_SIKHISM', 'fieldlabel' => 'PROFILE_DETAILS_RELIGION_SIKHISM' ), // CBTxt::T( 'PROFILE_DETAILS_RELIGION_SIKHISM', 'Sikhism' )
																array( 'fieldtitle' => 'RELIGION_SPIRITISM', 'fieldlabel' => 'PROFILE_DETAILS_RELIGION_SPIRITISM' ), // CBTxt::T( 'PROFILE_DETAILS_RELIGION_SPIRITISM', 'Spiritism' )
																array( 'fieldtitle' => 'RELIGION_TENRIKYO', 'fieldlabel' => 'PROFILE_DETAILS_RELIGION_TENRIKYO' ), // CBTxt::T( 'PROFILE_DETAILS_RELIGION_TENRIKYO', 'Tenrikyo' )
																array( 'fieldtitle' => 'RELIGION_UNITARIAN_UNIVERSALISM', 'fieldlabel' => 'PROFILE_DETAILS_RELIGION_UNITARIAN_UNIVERSALISM' ), // CBTxt::T( 'PROFILE_DETAILS_RELIGION_UNITARIAN_UNIVERSALISM', 'Unitarian Universalism' )
																array( 'fieldtitle' => 'RELIGION_ZOROASTRIANISM', 'fieldlabel' => 'PROFILE_DETAILS_RELIGION_ZOROASTRIANISM' ), // CBTxt::T( 'PROFILE_DETAILS_RELIGION_ZOROASTRIANISM', 'Zoroastrianism' )
																array( 'fieldtitle' => 'RELIGION_PROTESTANT', 'fieldlabel' => 'PROFILE_DETAILS_RELIGION_PROTESTANT' ), // CBTxt::T( 'PROFILE_DETAILS_RELIGION_PROTESTANT', 'Protestant' )
																array( 'fieldtitle' => 'RELIGION_CATHOLIC', 'fieldlabel' => 'PROFILE_DETAILS_RELIGION_CATHOLIC' ), // CBTxt::T( 'PROFILE_DETAILS_RELIGION_CATHOLIC', 'Catholic' )
																array( 'fieldtitle' => 'RELIGION_MORMON', 'fieldlabel' => 'PROFILE_DETAILS_RELIGION_MORMON' ), // CBTxt::T( 'PROFILE_DETAILS_RELIGION_MORMON', 'Mormon' )
																array( 'fieldtitle' => 'RELIGION_BAPTIST', 'fieldlabel' => 'PROFILE_DETAILS_RELIGION_BAPTIST' ), // CBTxt::T( 'PROFILE_DETAILS_RELIGION_BAPTIST', 'Baptist' )
																array( 'fieldtitle' => 'RELIGION_METHODIST', 'fieldlabel' => 'PROFILE_DETAILS_RELIGION_METHODIST' ), // CBTxt::T( 'PROFILE_DETAILS_RELIGION_METHODIST', 'Methodist' )
																array( 'fieldtitle' => 'RELIGION_PENTECOSTAL', 'fieldlabel' => 'PROFILE_DETAILS_RELIGION_PENTECOSTAL' ), // CBTxt::T( 'PROFILE_DETAILS_RELIGION_PENTECOSTAL', 'Pentecostal' )
																array( 'fieldtitle' => 'RELIGION_LUTHERAN', 'fieldlabel' => 'PROFILE_DETAILS_RELIGION_LUTHERAN' ), // CBTxt::T( 'PROFILE_DETAILS_RELIGION_LUTHERAN', 'Lutheran' )
																array( 'fieldtitle' => 'RELIGION_PRESBYTERIAN', 'fieldlabel' => 'PROFILE_DETAILS_RELIGION_PRESBYTERIAN' ), // CBTxt::T( 'PROFILE_DETAILS_RELIGION_PRESBYTERIAN', 'Presbyterian' )
																array( 'fieldtitle' => 'RELIGION_ORTHODOX', 'fieldlabel' => 'PROFILE_DETAILS_RELIGION_ORTHODOX' ), // CBTxt::T( 'PROFILE_DETAILS_RELIGION_ORTHODOX', 'Orthodox' )
															) );

					$detailsFieldParams->set( 'field_display_style', 0 );
					$detailsFieldParams->set( 'cbprivacy_display', 1 );
					$detailsFieldParams->set( 'cbprivacy_simple_group', 'PROFILE_DETAILS_PRIVACY' );
					$detailsFieldParams->set( 'privacy_class', 'd-block mt-1' );
					$detailsFieldParams->set( 'privacy_options_default', 0 );
					$detailsFieldParams->set( 'ajax_profile', 1 );
					$detailsFieldParams->set( 'ajax_profile_strict', 1 );
					$detailsFieldParams->set( 'ajax_template', 'cog' );
					break;
				case 'cb_details_political':
					$detailsField->set( 'title', 'PROFILE_DETAILS_POLITICAL' ); // CBTxt::T( 'PROFILE_DETAILS_POLITICAL', 'Political' )
					$detailsField->set( 'type', 'tag' );
					$detailsField->set( 'searchable', 1 );
					$detailsField->set( 'profile', 1 );
					$detailsField->set( 'edit', 1 );

					$detailsField->set( '_fieldvalues', array(	array( 'fieldtitle' => 'DEMOCRATIC_PARTY', 'fieldlabel' => 'PROFILE_DETAILS_POLITICAL_DEMOCRATIC_PARTY' ), // CBTxt::T( 'PROFILE_DETAILS_POLITICAL_DEMOCRATIC_PARTY', 'Democratic Party' )
																array( 'fieldtitle' => 'REPUBLICAN_PARTY', 'fieldlabel' => 'PROFILE_DETAILS_POLITICAL_REPUBLICAN_PARTY' ), // CBTxt::T( 'PROFILE_DETAILS_POLITICAL_REPUBLICAN_PARTY', 'Republican Party' )
																array( 'fieldtitle' => 'LIBERAL_PARTY', 'fieldlabel' => 'PROFILE_DETAILS_POLITICAL_LIBERAL_PARTY' ), // CBTxt::T( 'PROFILE_DETAILS_POLITICAL_LIBERAL_PARTY', 'Liberal Party' )
																array( 'fieldtitle' => 'CONSERVATIVE_PARTY', 'fieldlabel' => 'PROFILE_DETAILS_POLITICAL_CONSERVATIVE_PARTY' ), // CBTxt::T( 'PROFILE_DETAILS_POLITICAL_CONSERVATIVE_PARTY', 'Conservative Party' )
																array( 'fieldtitle' => 'LIBERTARIAN_PARTY', 'fieldlabel' => 'PROFILE_DETAILS_POLITICAL_LIBERTARIAN_PARTY' ), // CBTxt::T( 'PROFILE_DETAILS_POLITICAL_LIBERTARIAN_PARTY', 'Libertarian Party' )
																array( 'fieldtitle' => 'FREEDOM_PARTY', 'fieldlabel' => 'PROFILE_DETAILS_POLITICAL_FREEDOM_PARTY' ), // CBTxt::T( 'PROFILE_DETAILS_POLITICAL_FREEDOM_PARTY', 'Freedom Party' )
																array( 'fieldtitle' => 'GREEN_PARTY', 'fieldlabel' => 'PROFILE_DETAILS_POLITICAL_GREEN_PARTY' ), // CBTxt::T( 'PROFILE_DETAILS_POLITICAL_GREEN_PARTY', 'Green Party' )
																array( 'fieldtitle' => 'ACTION_PARTY', 'fieldlabel' => 'PROFILE_DETAILS_POLITICAL_ACTION_PARTY' ), // CBTxt::T( 'PROFILE_DETAILS_POLITICAL_ACTION_PARTY', 'Action Party' )
																array( 'fieldtitle' => 'INDEPENDENCE_PARTY', 'fieldlabel' => 'PROFILE_DETAILS_POLITICAL_INDEPENDENCE_PARTY' ), // CBTxt::T( 'PROFILE_DETAILS_POLITICAL_INDEPENDENCE_PARTY', 'Independence Party' )
																array( 'fieldtitle' => 'EQUALITY_PARTY', 'fieldlabel' => 'PROFILE_DETAILS_POLITICAL_EQUALITY_PARTY' ), // CBTxt::T( 'PROFILE_DETAILS_POLITICAL_EQUALITY_PARTY', 'Equality Party' )
																array( 'fieldtitle' => 'CHRISTIAN_PARTY', 'fieldlabel' => 'PROFILE_DETAILS_POLITICAL_CHRISTIAN_PARTY' ), // CBTxt::T( 'PROFILE_DETAILS_POLITICAL_CHRISTIAN_PARTY', 'Christian Party' )
																array( 'fieldtitle' => 'LABOUR_PARTY', 'fieldlabel' => 'PROFILE_DETAILS_POLITICAL_LABOUR_PARTY' ), // CBTxt::T( 'PROFILE_DETAILS_POLITICAL_LABOUR_PARTY', 'Labour Party' )
																array( 'fieldtitle' => 'LEFT_PARTY', 'fieldlabel' => 'PROFILE_DETAILS_POLITICAL_LEFT_PARTY' ), // CBTxt::T( 'PROFILE_DETAILS_POLITICAL_LEFT_PARTY', 'Left Party' )
																array( 'fieldtitle' => 'AGRARIAN_PARTY', 'fieldlabel' => 'PROFILE_DETAILS_POLITICAL_AGRARIAN_PARTY' ), // CBTxt::T( 'PROFILE_DETAILS_POLITICAL_AGRARIAN_PARTY', 'Agrarian Party' )
																array( 'fieldtitle' => 'FATHERLAND_PARTY', 'fieldlabel' => 'PROFILE_DETAILS_POLITICAL_FATHERLAND_PARTY' ), // CBTxt::T( 'PROFILE_DETAILS_POLITICAL_FATHERLAND_PARTY', 'Fatherland Party' )
																array( 'fieldtitle' => 'MOTHERLAND_PARTY', 'fieldlabel' => 'PROFILE_DETAILS_POLITICAL_MOTHERLAND_PARTY' ), // CBTxt::T( 'PROFILE_DETAILS_POLITICAL_MOTHERLAND_PARTY', 'Motherland Party' )
																array( 'fieldtitle' => 'NATIONAL_PARTY', 'fieldlabel' => 'PROFILE_DETAILS_POLITICAL_NATIONAL_PARTY' ), // CBTxt::T( 'PROFILE_DETAILS_POLITICAL_NATIONAL_PARTY', 'National Party' )
																array( 'fieldtitle' => 'ALLIANCE_PARTY', 'fieldlabel' => 'PROFILE_DETAILS_POLITICAL_ALLIANCE_PARTY' ), // CBTxt::T( 'PROFILE_DETAILS_POLITICAL_ALLIANCE_PARTY', 'Alliance Party' )
																array( 'fieldtitle' => 'NEW_PARTY', 'fieldlabel' => 'PROFILE_DETAILS_POLITICAL_NEW_PARTY' ), // CBTxt::T( 'PROFILE_DETAILS_POLITICAL_NEW_PARTY', 'New Party' )
																array( 'fieldtitle' => 'NON_POLITICAL', 'fieldlabel' => 'PROFILE_DETAILS_POLITICAL_NON_POLITICAL' ), // CBTxt::T( 'PROFILE_DETAILS_POLITICAL_NON_POLITICAL', 'Non-Political' )
																array( 'fieldtitle' => 'PEOPLES_PARTY', 'fieldlabel' => 'PROFILE_DETAILS_POLITICAL_PEOPLES_PARTY' ), // CBTxt::T( 'PROFILE_DETAILS_POLITICAL_PEOPLES_PARTY', "People's Party" )
																array( 'fieldtitle' => 'POPULAR_PARTY', 'fieldlabel' => 'PROFILE_DETAILS_POLITICAL_POPULAR_PARTY' ), // CBTxt::T( 'PROFILE_DETAILS_POLITICAL_POPULAR_PARTY', 'Popular Party' )
																array( 'fieldtitle' => 'CENTRE_PARTY', 'fieldlabel' => 'PROFILE_DETAILS_POLITICAL_CENTRE_PARTY' ), // CBTxt::T( 'PROFILE_DETAILS_POLITICAL_CENTRE_PARTY', 'Centre Party' )
																array( 'fieldtitle' => 'PROGRESSIVE_PARTY', 'fieldlabel' => 'PROFILE_DETAILS_POLITICAL_PROGRESSIVE_PARTY' ), // CBTxt::T( 'PROFILE_DETAILS_POLITICAL_PROGRESSIVE_PARTY', 'Progressive Party' )
																array( 'fieldtitle' => 'RADICAL_PARTY', 'fieldlabel' => 'PROFILE_DETAILS_POLITICAL_RADICAL_PARTY' ), // CBTxt::T( 'PROFILE_DETAILS_POLITICAL_RADICAL_PARTY', 'Radical Party' )
																array( 'fieldtitle' => 'REFORM_PARTY', 'fieldlabel' => 'PROFILE_DETAILS_POLITICAL_REFORM_PARTY' ), // CBTxt::T( 'PROFILE_DETAILS_POLITICAL_REFORM_PARTY', 'Reform Party' )
																array( 'fieldtitle' => 'REVOLUTIONARY_PARTY', 'fieldlabel' => 'PROFILE_DETAILS_POLITICAL_REVOLUTIONARY_PARTY' ), // CBTxt::T( 'PROFILE_DETAILS_POLITICAL_REVOLUTIONARY_PARTY', 'Revolutionary Party' )
																array( 'fieldtitle' => 'SOCIAL_DEMOCRATIC_PARTY', 'fieldlabel' => 'PROFILE_DETAILS_POLITICAL_SOCIAL_DEMOCRATIC_PARTY' ), // CBTxt::T( 'PROFILE_DETAILS_POLITICAL_SOCIAL_DEMOCRATIC_PARTY', 'Social Democratic Party' )
																array( 'fieldtitle' => 'SOCIALIST_PARTY', 'fieldlabel' => 'PROFILE_DETAILS_POLITICAL_SOCIALIST_PARTY' ), // CBTxt::T( 'PROFILE_DETAILS_POLITICAL_SOCIALIST_PARTY', 'Socialist Party' )
																array( 'fieldtitle' => 'UNIONIST_PARTY', 'fieldlabel' => 'PROFILE_DETAILS_POLITICAL_UNIONIST_PARTY' ), // CBTxt::T( 'PROFILE_DETAILS_POLITICAL_UNIONIST_PARTY', 'Unionist Party' )
																array( 'fieldtitle' => 'CIVIC_PARTY', 'fieldlabel' => 'PROFILE_DETAILS_POLITICAL_CIVIC_PARTY' ), // CBTxt::T( 'PROFILE_DETAILS_POLITICAL_CIVIC_PARTY', 'Civic Party' )
																array( 'fieldtitle' => 'UNITY_PARTY', 'fieldlabel' => 'PROFILE_DETAILS_POLITICAL_UNITY_PARTY' ), // CBTxt::T( 'PROFILE_DETAILS_POLITICAL_UNITY_PARTY', 'Unity Party' )
																array( 'fieldtitle' => 'WORKERS_PARTY', 'fieldlabel' => 'PROFILE_DETAILS_POLITICAL_WORKERS_PARTY' ), // CBTxt::T( 'PROFILE_DETAILS_POLITICAL_WORKERS_PARTY', "Workers' Party" )
																array( 'fieldtitle' => 'COMMUNIST_PARTY', 'fieldlabel' => 'PROFILE_DETAILS_POLITICAL_COMMUNIST_PARTY' ), // CBTxt::T( 'PROFILE_DETAILS_POLITICAL_COMMUNIST_PARTY', 'Communist Party' )
															) );

					$detailsFieldParams->set( 'field_display_style', 0 );
					$detailsFieldParams->set( 'cbprivacy_display', 1 );
					$detailsFieldParams->set( 'cbprivacy_simple_group', 'PROFILE_DETAILS_PRIVACY' );
					$detailsFieldParams->set( 'privacy_class', 'd-block mt-1' );
					$detailsFieldParams->set( 'privacy_options_default', 0 );
					$detailsFieldParams->set( 'ajax_profile', 1 );
					$detailsFieldParams->set( 'ajax_profile_strict', 1 );
					$detailsFieldParams->set( 'ajax_template', 'cog' );
					break;
			}

			$detailsField->set( 'name', $fieldName );
			$detailsField->set( 'ordering', self::getFieldOrder( $detailsField->get( 'tabid', 0, GetterInterface::INT ) ) );
			$detailsField->set( 'params', $detailsFieldParams->asJson() );

			$detailsField->store();

			self::setField( $fieldName, $detailsField );

			switch ( $fieldName ) {
				case 'cb_details_birthday_style':
					$parentField			=	self::getField( 'cb_details_birthday' );
					$parentFieldParams		=	new Registry( $parentField->get( 'params', null, GetterInterface::RAW ) );

					$parentFieldParams->set( 'ajax_fields', $detailsField->get( 'fieldid', 0, GetterInterface::INT ) );

					$parentField->set( 'params', $parentFieldParams->asJson() );

					$parentField->store();

					self::setField( 'cb_details_birthday', $parentField );
					break;
			}
		}

		// Address:
		// CBTxt::T( 'PROFILE_ADDRESS_PRIVACY', '<div class="bg-light text-center col-form-label">Address</div>' )
		foreach ( array( 'cb_address_header', 'cb_address_street', 'cb_address_country', 'cb_address_state', 'cb_address_city', 'cb_address_zip' ) as $fieldName ) {
			$addressField			=	new FieldTable();

			$addressField->load( array( 'name' => $fieldName ) );

			if ( ! $addressField->get( 'fieldid', 0, GetterInterface::INT ) ) {
				$addressField->set( 'published', 1 );
			}

			$addressFieldParams		=	new Registry( $addressField->get( 'params', null, GetterInterface::RAW ) );

			if ( $fieldName != 'cb_address_header' ) {
				$addressField->set( 'type', 'text' );
				$addressField->set( 'searchable', 1 );
				$addressField->set( 'profile', 1 );
				$addressField->set( 'edit', 1 );

				$addressFieldParams->set( 'cbprivacy_display', 1 );
				$addressFieldParams->set( 'privacy_class', 'd-block mt-1' );
				$addressFieldParams->set( 'privacy_options_default', 2 );
				$addressFieldParams->set( 'ajax_profile', 1 );
				$addressFieldParams->set( 'ajax_profile_strict', 1 );
				$addressFieldParams->set( 'ajax_template', 'cog' );
				$addressFieldParams->set( 'cbprivacy_simple_group', 'PROFILE_ADDRESS_PRIVACY' );
			}

			$addressField->set( 'description', $addressField->getString( 'description', '' ) );
			$addressField->set( 'tabid', self::getTab( 'details' )->get( 'tabid', 0, GetterInterface::INT ) );
			$addressField->set( 'registration', 0 );

			switch ( $fieldName ) {
				case 'cb_address_header':
					$addressField->set( 'title', 'PROFILE_ADDRESS' ); // CBTxt::T( 'PROFILE_ADDRESS', 'Address' )
					$addressField->set( 'type', 'delimiter' );
					$addressField->set( 'description', '<span></span>' );
					$addressField->set( 'searchable', 0 );
					$addressField->set( 'profile', 2 );
					$addressField->set( 'edit', 2 );
					$addressField->set( 'cssclass', 'form-group-section' );

					$conditions											=	array( array( 'condition' => array( array(	'field'						=>	'customvalue',
																														'field_custom'				=>	'[cb:userfield field="cb_address_street" /]',
																														'operator'					=>	7,
																														'locations'					=>	'profile'
																													) ) ),
																					array( 'condition' => array( array(	'field'						=>	'customvalue',
																														'field_custom'				=>	'[cb:userfield field="cb_address_country" /]',
																														'operator'					=>	7,
																														'locations'					=>	'profile'
																												 	) ) ),
																					array( 'condition' => array( array(	'field'						=>	'customvalue',
																														'field_custom'				=>	'[cb:userfield field="cb_address_state" /]',
																														'operator'					=>	7,
																														'locations'					=>	'profile'
																												 	) ) ),
																					array( 'condition' => array( array(	'field'						=>	'customvalue',
																														'field_custom'				=>	'[cb:userfield field="cb_address_city" /]',
																														'operator'					=>	7,
																														'locations'					=>	'profile'
																												 	) ) ),
																					array( 'condition' => array( array(	'field'						=>	'customvalue',
																														'field_custom'				=>	'[cb:userfield field="cb_address_zip" /]',
																														'operator'					=>	7,
																														'locations'					=>	'profile'
																												 	) ) )
																				);

					$addressFieldParams->set( 'cbconditional_conditioned', 1 );
					$addressFieldParams->set( 'cbconditional_conditions', $conditions );
					break;
				case 'cb_address_street':
					$addressField->set( 'title', 'PROFILE_ADDRESS_STREET' ); // CBTxt::T( 'PROFILE_ADDRESS_STREET', 'Street' )
					$addressField->set( 'size', 50 );
					break;
				case 'cb_address_country':
					$addressField->set( 'title', 'PROFILE_ADDRESS_COUNTRY' ); // CBTxt::T( 'PROFILE_ADDRESS_COUNTRY', 'Country' )

					if ( self::getPlugin( 'cbpaidsubscriptions' ) || $_CB_database->getTableStatus( '#__comprofiler_countries' ) ) {
						$addressFieldParams->set( 'qry_autocomplete', 1 );

						$autoCompleteCountry							=	"SELECT `country_name`"
																		.	" FROM `#__comprofiler_countries`"
																		.	" WHERE ("
																		.		" `country_iso_code2` = '[value]'"
																		.		" OR `country_iso_code3` = '[value]'"
																		.		" OR `country_fips_code` = '[value]'"
																		.		" OR `country_name` LIKE '[value]%'"
																		.		" OR `country_region` LIKE '[value]%'"
																		.		" OR `country_sub_region` LIKE '[value]%'"
																		.		" OR `country_fips_name` LIKE '[value]%'"
																		.		" OR `country_alternate_names` LIKE '%[value]%'"
																		.	" )";

						$addressFieldParams->set( 'qry_autocomplete_query', $autoCompleteCountry );
						$addressFieldParams->set( 'qry_autocomplete_col_value', 'country_name' );
						$addressFieldParams->set( 'qry_autocomplete_col_label', 'country_name' );
						$addressFieldParams->set( 'qry_autocomplete_min', 2 );
					}
					break;
				case 'cb_address_state':
					$addressField->set( 'title', 'PROFILE_ADDRESS_STATE' ); // CBTxt::T( 'PROFILE_ADDRESS_STATE', 'State' )

					if ( self::getPlugin( 'cbpaidsubscriptions' ) || $_CB_database->getTableStatus( '#__comprofiler_provinces' ) ) {
						$addressFieldParams->set( 'qry_autocomplete', 1 );

						$autoCompleteState								=	"SELECT `province_latin_name`"
																		.	" FROM `#__comprofiler_provinces`"
																		.	" WHERE ("
																		.		" `province_iso_code` = '[value]'"
																		.		" OR `province_name` LIKE '[value]%'"
																		.		" OR `province_latin_name` LIKE '[value]%'"
																		.		" OR `province_alternate_names` LIKE '%[value]%'"
																		.	" )"
																		.	" AND `country_iso_code2` IN ("
																		.		" SELECT `country_iso_code2`"
																		.		" FROM `#__comprofiler_countries`"
																		.		" WHERE ("
																		.			" `country_iso_code2` = '[cb_address_country]'"
																		.			" OR `country_iso_code3` = '[cb_address_country]'"
																		.			" OR `country_fips_code` = '[cb_address_country]'"
																		.			" OR `country_name` LIKE '[cb_address_country]%'"
																		.			" OR `country_region` LIKE '[cb_address_country]%'"
																		.			" OR `country_sub_region` LIKE '[cb_address_country]%'"
																		.			" OR `country_fips_name` LIKE '[cb_address_country]%'"
																		.			" OR `country_alternate_names` LIKE '%[cb_address_country]%'"
																		.		" )"
																		.	" )";

						$addressFieldParams->set( 'qry_autocomplete_query', $autoCompleteState );
						$addressFieldParams->set( 'qry_autocomplete_col_value', 'province_latin_name' );
						$addressFieldParams->set( 'qry_autocomplete_col_label', 'province_latin_name' );
						$addressFieldParams->set( 'qry_autocomplete_min', 2 );
						$addressFieldParams->set( 'qry_autocomplete_fields', 'cb_address_country' );
					}
					break;
				case 'cb_address_city':
					$addressField->set( 'title', 'PROFILE_ADDRESS_CITY' ); // CBTxt::T( 'PROFILE_ADDRESS_CITY', 'City' )

//					if ( $_CB_database->getTableStatus( '#__comprofiler_cities' ) ) {
//						$addressFieldParams->set( 'qry_autocomplete', 1 );
//
//						$autoCompleteCity								=	"SELECT `city_name`"
//																		.	" FROM `#__comprofiler_cities`"
//																		.	" WHERE ("
//																		.		" OR `city_name` LIKE '[value]%'"
//																		.		" OR `city_alternate_names` LIKE '%[value]%'"
//																		.	" )"
//																		.	" AND `province_iso_code` IN ("
//																		.		" SELECT `province_iso_code`"
//																		.		" FROM `#__comprofiler_provinces`"
//																		.		" WHERE ("
//																		.			" `province_iso_code` = '[value]'"
//																		.			" OR `province_name` LIKE '[value]%'"
//																		.			" OR `province_latin_name` LIKE '[value]%'"
//																		.			" OR `province_alternate_names` LIKE '%[value]%'"
//																		.		" )"
//																		.	" )";
//
//						$addressFieldParams->set( 'qry_autocomplete_query', $autoCompleteCity );
//						$addressFieldParams->set( 'qry_autocomplete_col_value', 'city_name' );
//						$addressFieldParams->set( 'qry_autocomplete_col_label', 'city_name' );
//						$addressFieldParams->set( 'qry_autocomplete_min', 2 );
//						$addressFieldParams->set( 'qry_autocomplete_fields', 'cb_address_state' );
//					}
					break;
				case 'cb_address_zip':
					$addressField->set( 'title', 'PROFILE_ADDRESS_ZIP' ); // CBTxt::T( 'PROFILE_ADDRESS_ZIP', 'Zip' )
					$addressField->set( 'size', 10 );
					break;
			}

			$addressField->set( 'name', $fieldName );
			$addressField->set( 'ordering', self::getFieldOrder( $addressField->get( 'tabid', 0, GetterInterface::INT ) ) );
			$addressField->set( 'params', $addressFieldParams->asJson() );

			$addressField->store();

			self::setField( $fieldName, $addressField );
		}

		// Phone Numbers:
		// CBTxt::T( 'PROFILE_PHONE_NUMBERS_PRIVACY', '<div class="bg-light text-center col-form-label">Phone Numbers</div>' )
		foreach ( array( 'cb_phone_header', 'cb_phone_home', 'cb_phone_work', 'cb_phone_mobile' ) as $fieldName ) {
			$phoneField			=	new FieldTable();

			$phoneField->load( array( 'name' => $fieldName ) );

			if ( ! $phoneField->get( 'fieldid', 0, GetterInterface::INT ) ) {
				$phoneField->set( 'published', 1 );
			}

			$phoneFieldParams	=	new Registry( $phoneField->get( 'params', null, GetterInterface::RAW ) );

			if ( $fieldName != 'cb_phone_header' ) {
				$phoneField->set( 'type', 'text' );
				$phoneField->set( 'searchable', 1 );
				$phoneField->set( 'profile', 1 );
				$phoneField->set( 'edit', 1 );

				$phoneFieldParams->set( 'cbprivacy_display', 1 );
				$phoneFieldParams->set( 'privacy_class', 'd-block mt-1' );
				$phoneFieldParams->set( 'privacy_options_default', 2 );
				$phoneFieldParams->set( 'ajax_profile', 1 );
				$phoneFieldParams->set( 'ajax_profile_strict', 1 );
				$phoneFieldParams->set( 'ajax_template', 'cog' );
				$phoneFieldParams->set( 'cbprivacy_simple_group', 'PROFILE_PHONE_NUMBERS_PRIVACY' );
			}

			$phoneField->set( 'description', $phoneField->getString( 'description', '' ) );
			$phoneField->set( 'tabid', self::getTab( 'details' )->get( 'tabid', 0, GetterInterface::INT ) );
			$phoneField->set( 'registration', 0 );

			switch ( $fieldName ) {
				case 'cb_phone_header':
					$phoneField->set( 'title', 'PROFILE_PHONE_NUMBERS' ); // CBTxt::T( 'PROFILE_PHONE_NUMBERS', 'Phone Numbers' )
					$phoneField->set( 'type', 'delimiter' );
					$phoneField->set( 'description', '<span></span>' );
					$phoneField->set( 'searchable', 0 );
					$phoneField->set( 'profile', 2 );
					$phoneField->set( 'edit', 2 );
					$phoneField->set( 'cssclass', 'form-group-section' );

					$conditions										=	array( array( 'condition' => array( array(	'field'						=>	'customvalue',
																													'field_custom'				=>	'[cb:userfield field="cb_phone_home" /]',
																													'operator'					=>	7,
																													'locations'					=>	'profile'
																												) ) ),
																				array( 'condition' => array( array(	'field'						=>	'customvalue',
																													'field_custom'				=>	'[cb:userfield field="cb_phone_work" /]',
																													'operator'					=>	7,
																													'locations'					=>	'profile'
																											 	) ) ),
																				array( 'condition' => array( array(	'field'						=>	'customvalue',
																													'field_custom'				=>	'[cb:userfield field="cb_phone_mobile" /]',
																													'operator'					=>	7,
																													'locations'					=>	'profile'
																												 ) ) )
																			);

					$phoneFieldParams->set( 'cbconditional_conditioned', 1 );
					$phoneFieldParams->set( 'cbconditional_conditions', $conditions );
					break;
				case 'cb_phone_home':
					$phoneField->set( 'title', 'PROFILE_PHONE_HOME' ); // CBTxt::T( 'PROFILE_PHONE_HOME', 'Home' )
					$phoneFieldParams->set( 'fieldInputMask', 'phone' );
					break;
				case 'cb_phone_work':
					$phoneField->set( 'title', 'PROFILE_PHONE_WORK' ); // CBTxt::T( 'PROFILE_PHONE_WORK', 'Work' )
					$phoneFieldParams->set( 'fieldInputMask', 'phone' );
					break;
				case 'cb_phone_mobile':
					$phoneField->set( 'title', 'PROFILE_PHONE_MOBILE' ); // CBTxt::T( 'PROFILE_PHONE_MOBILE', 'Mobile' )
					$phoneFieldParams->set( 'fieldInputMask', 'phone' );
					break;
			}

			$phoneField->set( 'name', $fieldName );
			$phoneField->set( 'ordering', self::getFieldOrder( $phoneField->get( 'tabid', 0, GetterInterface::INT ) ) );
			$phoneField->set( 'params', $phoneFieldParams->asJson() );

			$phoneField->store();

			self::setField( $fieldName, $phoneField );
		}

		// Education:
		// CBTxt::T( 'PROFILE_EDUCATION_WORK_PRIVACY', '<div class="bg-light text-center col-form-label">Education & Work</div>' )
		foreach ( array( 'cb_education_school', 'cb_education_start', 'cb_education_end', 'cb_education_graduated', 'cb_education_description', 'cb_education_logo', 'cb_education' ) as $fieldName ) {
			$eduField			=	new FieldTable();

			$eduField->load( array( 'name' => $fieldName ) );

			if ( ! $eduField->get( 'fieldid', 0, GetterInterface::INT ) ) {
				$eduField->set( 'published', 1 );
			}

			$eduFieldParams		=	new Registry( $eduField->get( 'params', null, GetterInterface::RAW ) );

			$eduField->set( 'description', $eduField->getString( 'description', '' ) );
			$eduField->set( 'searchable', 0 );
			$eduField->set( 'profile', 1 );
			$eduField->set( 'edit', 1 );
			$eduField->set( 'registration', 0 );
			$eduField->set( 'tabid', self::getTab( 'details' )->get( 'tabid', 0, GetterInterface::INT ) );

			switch ( $fieldName ) {
				case 'cb_education_school':
					$eduField->set( 'title', 'PROFILE_EDUCATION_SCHOOL' ); // CBTxt::T( 'PROFILE_EDUCATION_SCHOOL', 'School' )
					$eduField->set( 'description', 'PROFILE_EDUCATION_SCHOOL_DESC' ); // CBTxt::T( 'PROFILE_EDUCATION_SCHOOL_DESC', 'Please provide the name of your school in full.' )
					$eduField->set( 'type', 'text' );
					$eduField->set( 'profile', 4 );
					$eduField->set( 'cssclass', 'form-group-block' );

					$eduFieldParams->set( 'fieldPlaceholder', 'PROFILE_EDUCATION_SCHOOL' );
					break;
				case 'cb_education_start':
					$eduField->set( 'title', 'PROFILE_EDUCATION_START' ); // CBTxt::T( 'PROFILE_EDUCATION_START', 'Start Year' )
					$eduField->set( 'description', 'PROFILE_EDUCATION_START_DESC' ); // CBTxt::T( 'PROFILE_EDUCATION_START_DESC', 'Optionally select the year you started school.' )
					$eduField->set( 'type', 'date' );

					$eduFieldParams->set( 'field_edit_format', 1 );
					$eduFieldParams->set( 'custom_date_edit_format', 'Y' );
					$eduFieldParams->set( 'field_display_by', 5 );
					$eduFieldParams->set( 'custom_date_format', 'Y' );
					$eduFieldParams->set( 'field_search_by', 2 );
					$eduFieldParams->set( 'custom_date_format', 'Y' );
					$eduFieldParams->set( 'calendar_type', 3 );
					break;
				case 'cb_education_end':
					$eduField->set( 'title', 'PROFILE_EDUCATION_END' ); // CBTxt::T( 'PROFILE_EDUCATION_END', 'End Year' )
					$eduField->set( 'description', 'PROFILE_EDUCATION_END_DESC' ); // CBTxt::T( 'PROFILE_EDUCATION_END_DESC', 'Optionally select the year you finished school.' )
					$eduField->set( 'type', 'date' );

					$eduFieldParams->set( 'field_edit_format', 1 );
					$eduFieldParams->set( 'custom_date_edit_format', 'Y' );
					$eduFieldParams->set( 'field_display_by', 5 );
					$eduFieldParams->set( 'custom_date_format', 'Y' );
					$eduFieldParams->set( 'field_search_by', 2 );
					$eduFieldParams->set( 'custom_date_format', 'Y' );
					$eduFieldParams->set( 'calendar_type', 3 );
					break;
				case 'cb_education_graduated':
					$eduField->set( 'title', 'PROFILE_EDUCATION_GRADUATED' ); // CBTxt::T( 'PROFILE_EDUCATION_GRADUATED', 'Graduated' )
					$eduField->set( 'description', 'PROFILE_EDUCATION_GRADUATED_DESC' ); // CBTxt::T( 'PROFILE_EDUCATION_GRADUATED_DESC', "Select if you've graduated." )
					$eduField->set( 'type', 'checkbox' );
					break;
				case 'cb_education_description':
					$eduField->set( 'title', 'PROFILE_EDUCATION_DESCRIPTION' ); // CBTxt::T( 'PROFILE_EDUCATION_DESCRIPTION', 'Description' )
					$eduField->set( 'type', 'textarea' );
					$eduField->set( 'cols', 0 );
					$eduField->set( 'rows', 3 );
					$eduField->set( 'cssclass', 'form-group-block' );

					$eduFieldParams->set( 'fieldPlaceholder', 'PROFILE_EDUCATION_DESCRIPTION_PLACEHOLDER' ); // CBTxt::T( 'PROFILE_EDUCATION_DESCRIPTION_PLACEHOLDER', 'Share your degree, major, school activities, etc..' )
					break;
				case 'cb_education_logo':
					$eduField->set( 'title', 'PROFILE_EDUCATION_LOGO' ); // CBTxt::T( 'PROFILE_EDUCATION_LOGO', 'Logo' )
					$eduField->set( 'description', 'PROFILE_EDUCATION_LOGO_DESC' ); // CBTxt::T( 'PROFILE_EDUCATION_LOGO_DESC', 'If your school has a logo you can optionally supply it here.' )
					$eduField->set( 'type', 'image' );
					$eduField->set( 'cssclass', 'form-group-block' );

					$eduFieldParams->set( 'imageStyle', '' );
					$eduFieldParams->set( 'pendingDefaultAvatar', 'none' );
					$eduFieldParams->set( 'defaultAvatar', 'none' );
					$eduFieldParams->set( 'altText', 2 );
					$eduFieldParams->set( 'altTextCustom', '[cb_education_school]' );
					$eduFieldParams->set( 'titleText', 2 );
					$eduFieldParams->set( 'titleTextCustom', '[cb_education_school]' );
					break;
				case 'cb_education':
					if ( ! self::getPlugin( 'cbfieldgroups' ) ) {
						continue 2;
					}

					$eduField->set( 'title', 'PROFILE_EDUCATION_SCHOOLS' ); // CBTxt::T( 'PROFILE_EDUCATION_SCHOOLS', 'Schools' )
					$eduField->set( 'type', 'fieldgroup' );
					$eduField->set( 'profile', 2 );
					$eduField->set( 'edit', 2 );
					$eduField->set( 'cssclass', 'form-group-section' );

					$eduFieldParams->set( 'repeat_fields', implode( '|*|', array( self::getField( 'cb_education_school' )->get( 'fieldid', 0, GetterInterface::INT ),
																				  self::getField( 'cb_education_start' )->get( 'fieldid', 0, GetterInterface::INT ),
																				  self::getField( 'cb_education_end' )->get( 'fieldid', 0, GetterInterface::INT ),
																				  self::getField( 'cb_education_graduated' )->get( 'fieldid', 0, GetterInterface::INT ),
																				  self::getField( 'cb_education_description' )->get( 'fieldid', 0, GetterInterface::INT ),
																				  self::getField( 'cb_education_logo' )->get( 'fieldid', 0, GetterInterface::INT ) ) ) );
					$eduFieldParams->set( 'repeat_label', 'PROFILE_EDUCATION_ADD_SCHOOL' ); // CBTxt::T( 'PROFILE_EDUCATION_ADD_SCHOOL', 'Add School' )
					$eduFieldParams->set( 'repeat_limit', 10 );
					$eduFieldParams->set( 'repeat_empty_fields', 0 );

					// CBTxt::T( 'PROFILE_EDUCATION_SCHOOL_DISPLAY', '[cb:if cb_education_school!=""]<div class="row no-gutters"><div class="col-3 text-center">[cb:if cb_education_logo=""]<span class="fa fa-university fa-4x"></span>[/cb:if][cb:if cb_education_logo!=""]<img src="[cb_education_logo]" alt="" class="img-fluid cbImgPict cbFullPict" style="max-height: 72px;" />[/cb:if]</div><div class="col-9 pl-1"><strong>[cb:userfield field="cb_education_school" /]</strong> [cb:if cb_education_graduated="1"]<span class="fa fa-graduation-cap"></span>[/cb:if][cb:if cb_education_start!="" and cb_education_start!="0000-00-00" or cb_education_end!="" and cb_education_end!="0000-00-00"]<div class="text-small">[cb:if cb_education_start="" or cb_education_start="0000-00-00"]Class of [/cb:if][cb:userfield field="cb_education_start" /][cb:if cb_education_start!="" and cb_education_start!="0000-00-00" and cb_education_end!="" and cb_education_end!="0000-00-00"] - [/cb:if][cb:if cb_education_end="" or cb_education_end="0000-00-00"] - Present[/cb:if][cb:userfield field="cb_education_end" /]</div>[/cb:if][cb:if cb_education_description!=""]<div class="text-small text-muted">[cb:userfield field="cb_education_description" /]</div>[/cb:if]</div></div>[/cb:if]' )

					$eduFieldParams->set( 'repeat_template_custom', 'PROFILE_EDUCATION_SCHOOL_DISPLAY' );

					// CBTxt::T( 'PROFILE_EDUCATION_SCHOOL_EDIT', '<div class="input-group mb-2"><div class="flex-fill">[field_cb_education_school]</div><div class="input-group-append"><div class="input-group-text">[field_cb_education_graduated] <span class="ml-1 fa fa-graduation-cap"></span></div></div></div><div class="mb-2">[field_cb_education_start] - [field_cb_education_end]</div><div class="mb-2">[field_cb_education_description]</div><div class="mb-3">[field_cb_education_logo]</div>' )

					$eduFieldParams->set( 'repeat_template_edit_custom', 'PROFILE_EDUCATION_SCHOOL_EDIT' );
					$eduFieldParams->set( 'repeat_formatting_edit', 'span' );

					$eduFieldParams->set( 'cbprivacy_display', 1 );
					$eduFieldParams->set( 'privacy_class', 'd-block mt-1' );
					$eduFieldParams->set( 'privacy_options_default', 0 );
					$eduFieldParams->set( 'ajax_profile', 1 );
					$eduFieldParams->set( 'ajax_profile_strict', 1 );
					$eduFieldParams->set( 'ajax_template', 'cog' );
					$eduFieldParams->set( 'cbprivacy_simple_group', 'PROFILE_EDUCATION_WORK_PRIVACY' );
					$eduFieldParams->set( 'cbprivacy_simple_title', 'PROFILE_EDUCATION_SCHOOLS' );
					break;
			}

			$eduField->set( 'name', $fieldName );
			$eduField->set( 'ordering', self::getFieldOrder( $eduField->get( 'tabid', 0, GetterInterface::INT ) ) );
			$eduField->set( 'params', $eduFieldParams->asJson() );

			$eduField->store();

			self::setField( $fieldName, $eduField );
		}

		// Work:
		foreach ( array( 'cb_work_workplace', 'cb_work_position', 'cb_work_start', 'cb_work_end', 'cb_work_description', 'cb_work_logo', 'cb_work' ) as $fieldName ) {
			$workField			=	new FieldTable();

			$workField->load( array( 'name' => $fieldName ) );

			if ( ! $workField->get( 'fieldid', 0, GetterInterface::INT ) ) {
				$workField->set( 'published', 1 );
			}

			$workFieldParams	=	new Registry( $workField->get( 'params', null, GetterInterface::RAW ) );

			$workField->set( 'description', $workField->getString( 'description', '' ) );
			$workField->set( 'tabid', self::getTab( 'details' )->get( 'tabid', 0, GetterInterface::INT ) );
			$workField->set( 'searchable', 0 );
			$workField->set( 'profile', 1 );
			$workField->set( 'edit', 1 );
			$workField->set( 'registration', 0 );

			switch ( $fieldName ) {
				case 'cb_work_workplace':
					$workField->set( 'title', 'PROFILE_WORK_WORKPLACE' ); // CBTxt::T( 'PROFILE_WORK_WORKPLACE', 'Workplace' )
					$workField->set( 'description', 'PROFILE_WORK_WORKPLACE_DESC' ); // CBTxt::T( 'PROFILE_WORK_WORKPLACE_DESC', 'Please provide the name of your workplace in full.' )
					$workField->set( 'type', 'text' );
					$workField->set( 'profile', 4 );
					$workField->set( 'cssclass', 'form-group-block' );

					$workFieldParams->set( 'fieldPlaceholder', 'PROFILE_WORK_WORKPLACE' );
					break;
				case 'cb_work_position':
					$workField->set( 'title', 'PROFILE_WORK_POSITION' ); // CBTxt::T( 'PROFILE_WORK_POSITION', 'Position' )
					$workField->set( 'description', 'PROFILE_WORK_POSITION_DESC' ); // CBTxt::T( 'PROFILE_WORK_POSITION_DESC', 'Optionally provide your position.' )
					$workField->set( 'type', 'text' );
					$workField->set( 'profile', 4 );
					$workField->set( 'cssclass', 'form-group-block' );

					$workFieldParams->set( 'fieldPlaceholder', 'PROFILE_WORK_POSITION' );
					break;
				case 'cb_work_start':
					$workField->set( 'title', 'PROFILE_WORK_START' ); // CBTxt::T( 'PROFILE_WORK_START', 'Start Date' )
					$workField->set( 'description', 'PROFILE_WORK_START_DESC' ); // CBTxt::T( 'PROFILE_WORK_START_DESC', 'Optionally select the date you started work.' )
					$workField->set( 'type', 'date' );

					$workFieldParams->set( 'calendar_type', 3 );
					break;
				case 'cb_work_end':
					$workField->set( 'title', 'PROFILE_WORK_END' ); // CBTxt::T( 'PROFILE_WORK_END', 'End Date' )
					$workField->set( 'description', 'PROFILE_WORK_END_DESC' ); // CBTxt::T( 'PROFILE_WORK_END_DESC', 'Optionally select the date you left work.' )
					$workField->set( 'type', 'date' );

					$workFieldParams->set( 'calendar_type', 3 );
					break;
				case 'cb_work_description':
					$workField->set( 'title', 'PROFILE_WORK_DESCRIPTION' ); // CBTxt::T( 'PROFILE_WORK_DESCRIPTION', 'Description' )
					$workField->set( 'type', 'textarea' );
					$workField->set( 'cols', 0 );
					$workField->set( 'rows', 3 );
					$workField->set( 'cssclass', 'form-group-block' );

					$workFieldParams->set( 'fieldPlaceholder', 'PROFILE_WORK_DESCRIPTION_PLACEHOLDER' ); // CBTxt::T( 'PROFILE_WORK_DESCRIPTION_PLACEHOLDER', 'Share your resposibilities, activites, etc..' )
					break;
				case 'cb_work_logo':
					$workField->set( 'title', 'PROFILE_WORK_LOGO' ); // CBTxt::T( 'PROFILE_WORK_LOGO', 'Logo' )
					$workField->set( 'description', 'PROFILE_WORK_LOGO_DESC' ); // CBTxt::T( 'PROFILE_WORK_LOGO_DESC', 'If your work has a logo you can optionally supply it here.' )
					$workField->set( 'type', 'image' );
					$workField->set( 'cssclass', 'form-group-block' );

					$workFieldParams->set( 'imageStyle', '' );
					$workFieldParams->set( 'pendingDefaultAvatar', 'none' );
					$workFieldParams->set( 'defaultAvatar', 'none' );
					$workFieldParams->set( 'altText', 2 );
					$workFieldParams->set( 'altTextCustom', '[cb_work_workplace]' );
					$workFieldParams->set( 'titleText', 2 );
					$workFieldParams->set( 'titleTextCustom', '[cb_work_workplace]' );
					break;
				case 'cb_work':
					if ( ! self::getPlugin( 'cbfieldgroups' ) ) {
						continue 2;
					}

					$workField->set( 'title', 'PROFILE_WORK_WORKPLACES' ); // CBTxt::T( 'PROFILE_WORK_WORKPLACES', 'Workplaces' )
					$workField->set( 'type', 'fieldgroup' );
					$workField->set( 'profile', 2 );
					$workField->set( 'edit', 2 );
					$workField->set( 'cssclass', 'form-group-section' );

					$workFieldParams->set( 'repeat_fields', implode( '|*|', array( self::getField( 'cb_work_workplace' )->get( 'fieldid', 0, GetterInterface::INT ),
																				  self::getField( 'cb_work_position' )->get( 'fieldid', 0, GetterInterface::INT ),
																				  self::getField( 'cb_work_start' )->get( 'fieldid', 0, GetterInterface::INT ),
																				  self::getField( 'cb_work_end' )->get( 'fieldid', 0, GetterInterface::INT ),
																				  self::getField( 'cb_work_description' )->get( 'fieldid', 0, GetterInterface::INT ),
																				  self::getField( 'cb_work_logo' )->get( 'fieldid', 0, GetterInterface::INT ) ) ) );
					$workFieldParams->set( 'repeat_label', 'PROFILE_WORK_ADD_WORKPLACE' ); // CBTxt::T( 'PROFILE_WORK_ADD_WORKPLACE', 'Add Workplace' )
					$workFieldParams->set( 'repeat_limit', 10 );
					$workFieldParams->set( 'repeat_empty_fields', 0 );

					// CBTxt::T( 'PROFILE_WORK_WORKPLACE_DISPLAY', '[cb:if cb_work_workplace!=""]<div class="row no-gutters"><div class="col-3 text-center">[cb:if cb_work_logo=""]<span class="fa fa-building fa-4x"></span>[/cb:if][cb:if cb_work_logo!=""]<img src="[cb_work_logo]" alt="" class="img-fluid cbImgPict cbFullPict" style="max-height: 72px;" />[/cb:if]</div><div class="col-9 pl-1"><strong>[cb:userfield field="cb_work_workplace" /]</strong>[cb:if cb_work_position!=""]<div>[cb:userfield field="cb_work_position" /]</div>[/cb:if][cb:if cb_work_start!="" and cb_work_start!="0000-00-00" or cb_work_end!="" and cb_work_end!="0000-00-00"]<div class="text-small">[cb:if cb_work_start="" or cb_work_start="0000-00-00"]Year of [/cb:if][cb:userfield field="cb_work_start" /][cb:if cb_work_start!="" and cb_work_start!="0000-00-00" and cb_work_end!="" and cb_work_end!="0000-00-00"] - [/cb:if][cb:if cb_work_end="" or cb_work_end="0000-00-00"] - Present[/cb:if][cb:userfield field="cb_work_end" /]</div>[/cb:if][cb:if cb_work_description!=""]<div class="text-small text-muted">[cb:userfield field="cb_work_description" /]</div>[/cb:if]</div></div>[/cb:if]' )

					$workFieldParams->set( 'repeat_template_custom', 'PROFILE_WORK_WORKPLACE_DISPLAY' );

					// CBTxt::T( 'PROFILE_WORK_WORKPLACE_EDIT', '<div class="mb-2">[field_cb_work_workplace]</div><div class="mb-2">[field_cb_work_position]</div><div class="mb-2">[field_cb_work_start] - [field_cb_work_end]</div><div class="mb-2">[field_cb_work_description]</div><div class="mb-3">[field_cb_work_logo]</div>' )

					$workFieldParams->set( 'repeat_template_edit_custom', 'PROFILE_WORK_WORKPLACE_EDIT' );
					$workFieldParams->set( 'repeat_formatting_edit', 'span' );

					$workFieldParams->set( 'cbprivacy_display', 1 );
					$workFieldParams->set( 'privacy_class', 'd-block mt-1' );
					$workFieldParams->set( 'privacy_options_default', 0 );
					$workFieldParams->set( 'ajax_profile', 1 );
					$workFieldParams->set( 'ajax_profile_strict', 1 );
					$workFieldParams->set( 'ajax_template', 'cog' );
					$workFieldParams->set( 'cbprivacy_simple_group', 'PROFILE_EDUCATION_WORK_PRIVACY' );
					$workFieldParams->set( 'cbprivacy_simple_title', 'PROFILE_WORK_WORKPLACES' );
					break;
			}

			$workField->set( 'name', $fieldName );
			$workField->set( 'ordering', self::getFieldOrder( $workField->get( 'tabid', 0, GetterInterface::INT ) ) );
			$workField->set( 'params', $workFieldParams->asJson() );

			$workField->store();

			self::setField( $fieldName, $workField );
		}

		// Create a default userlist using grid layout:
		$userlist				=	new ListTable();

		$userlist->load( array( 'title' => 'MEMBERS_LIST' ) ); // CBTxt::T( 'MEMBERS_LIST', 'Members' )

		if ( ! $userlist->get( 'listid', 0, GetterInterface::INT ) ) {
			$userlist->set( 'published', 1 );
			$userlist->set( 'default', 1 );
		}

		$userlist->set( 'title', 'MEMBERS_LIST' );
		$userlist->set( 'viewaccesslevel', 1 );
		$userlist->set( 'usergroupids', $regAccess );

		$userlistParams			=	new Registry( $userlist->get( 'params', null, GetterInterface::RAW ) );

		$userlistParams->set( 'sort_mode', 0 );
		$userlistParams->set( 'sort_basic', array( array( 'column' => 'registerDate', 'direction' => 'DESC' ) ) );
		$userlistParams->set( 'columns', array(	array(	'title'		=>	'',
														'size'		=>	6,
														'fields'	=>	array(	array( 'field' => self::getField( 'canvas' )->get( 'fieldid', 0, GetterInterface::INT ), 'display' => 4 ),
																				array( 'field' => self::getField( 'avatar' )->get( 'fieldid', 0, GetterInterface::INT ), 'display' => 4 ),
																				array( 'field' => self::getField( 'username' )->get( 'fieldid', 0, GetterInterface::INT ), 'display' => 4 ),
																				array( 'field' => self::getField( 'onlinestatus' )->get( 'fieldid', 0, GetterInterface::INT ), 'display' => 4 )
																			)
													),
												array(	'title'		=>	'',
														'size'		=>	0,
														'position'	=>	'canvas_top',
														'fields'	=>	array(	array( 'field' => self::getField( 'pm' )->get( 'fieldid', 0, GetterInterface::INT ), 'display' => 4 ),
																			)
													),
												array(	'title'		=>	'',
														'size'		=>	0,
														'position'	=>	'canvas_bottom',
														'fields'	=>	array(	array( 'field' => self::getField( 'cb_follow_button' )->get( 'fieldid', 0, GetterInterface::INT ), 'display' => 4 ),
																				array( 'field' => self::getField( 'cb_connect_button' )->get( 'fieldid', 0, GetterInterface::INT ), 'display' => 4 )
																			)
													),
												array(	'title'		=>	'',
														'size'		=>	6,
														'cssclass'	=>	'col-6 mt-1 pt-1',
														'fields'	=>	array(	array( 'field' => self::getField( 'cb_details_gender' )->get( 'fieldid', 0, GetterInterface::INT ), 'display' => 2 ),
																				array( 'field' => self::getField( 'cb_details_interestedin' )->get( 'fieldid', 0, GetterInterface::INT ), 'display' => 2 ),
																				array( 'field' => self::getField( 'cb_details_language' )->get( 'fieldid', 0, GetterInterface::INT ), 'display' => 2 )
																			)
													),
												array(	'title'		=>	'',
														'size'		=>	6,
														'cssclass'	=>	'col-6 ml-auto mt-1 pt-1 text-right',
														'fields'	=>	array(	array( 'field' => self::getField( 'cb_details_birthday' )->get( 'fieldid', 0, GetterInterface::INT ), 'display' => 2 ),
																				array( 'field' => self::getField( 'cb_details_relationship' )->get( 'fieldid', 0, GetterInterface::INT ), 'display' => 2 ),
																				array( 'field' => self::getField( 'cb_address_country' )->get( 'fieldid', 0, GetterInterface::INT ), 'display' => 2 )
																			)
													)
											) );
		$userlistParams->set( 'list_search', 3 );
		$userlistParams->set( 'list_search_fields', implode( '|*|', array(	self::getField( 'name' )->get( 'fieldid', 0, GetterInterface::INT ),
																			self::getField( 'username' )->get( 'fieldid', 0, GetterInterface::INT ),
																			self::getField( 'cb_details_birthday' )->get( 'fieldid', 0, GetterInterface::INT ),
																			self::getField( 'cb_details_gender' )->get( 'fieldid', 0, GetterInterface::INT ),
																			self::getField( 'cb_details_interestedin' )->get( 'fieldid', 0, GetterInterface::INT ),
																			self::getField( 'cb_details_relationship' )->get( 'fieldid', 0, GetterInterface::INT ),
																			self::getField( 'cb_details_language' )->get( 'fieldid', 0, GetterInterface::INT ),
																			self::getField( 'cb_details_religion' )->get( 'fieldid', 0, GetterInterface::INT ),
																			self::getField( 'cb_details_political' )->get( 'fieldid', 0, GetterInterface::INT ),
																			self::getField( 'cb_address_country' )->get( 'fieldid', 0, GetterInterface::INT ),
																			self::getField( 'cb_address_state' )->get( 'fieldid', 0, GetterInterface::INT ),
																			self::getField( 'cb_address_city' )->get( 'fieldid', 0, GetterInterface::INT )
																		) ) );
		$userlistParams->set( 'list_compare_types', 2 );
		$userlistParams->set( 'list_search_empty', 0 );
		$userlistParams->set( 'list_search_collapse', 1 );
		$userlistParams->set( 'list_show_selector', 0 );
		$userlistParams->set( 'list_show_in_selector', 1 );
		$userlistParams->set( 'list_grid_layout', 1 );
		$userlistParams->set( 'list_limit', 10 );
		$userlistParams->set( 'list_paging', 1 );

		$userlist->set( 'params', $userlistParams->asJson() );

		$userlist->store();

		// Make sure quickstart is actually published so our language strings load:
		$quickstartPlugin		=	new PluginTable();

		$quickstartPlugin->load( array( 'element' => 'cbquickstart' ) );

		if ( $quickstartPlugin->get( 'id', 0, GetterInterface::INT ) ) {
			$quickstartPlugin->set( 'published', 1 );

			$quickstartPlugin->store();
		}

		// Create default menu structure for navigating throughout CB:
		$menuRoot					=	self::createMenu( 'mainmenu', CBTxt::T( 'CB_COMMUNITY', 'Community' ), '#', array( 'type' => 'url', 'component_id' => 0 ) );

		if ( $menuRoot ) {
			// Critical menu endpoints like registration, profile, and profile edit:
			$menuJoin				=	self::createMenu( 'mainmenu', CBTxt::T( 'CB_COMMUNITY_JOIN', 'Join' ), 'index.php?option=com_comprofiler&view=registers', array( 'parent_id' => $menuRoot, 'params' => array( 'menu_show' => 0 ) ) );

			if ( $menuJoin ) {
				self::createMenu( 'mainmenu', CBTxt::T( 'CB_COMMUNITY_JOIN_US', 'Join Us' ), 'index.php?Itemid=' . $menuJoin, array( 'alias' => 'join-alias', 'type' => 'alias', 'component_id' => 0, 'access' => $guestAccess, 'parent_id' => $menuRoot, 'params' => array( 'aliasoptions' => $menuJoin ) ) );
			}

			$menuProfile			=	self::createMenu( 'mainmenu', CBTxt::T( 'CB_COMMUNITY_PROFILE', 'Profile' ), 'index.php?option=com_comprofiler&view=userprofile', array( 'parent_id' => $menuRoot, 'params' => array( 'menu_show' => 0 ) ) );

			if ( $menuProfile ) {
				self::createMenu( 'mainmenu', CBTxt::T( 'CB_COMMUNITY_MY_PROFILE', 'My Profile' ), 'index.php?Itemid=' . $menuProfile, array( 'alias' => 'profile-alias', 'type' => 'alias', 'component_id' => 0, 'access' => $regAccess, 'parent_id' => $menuRoot, 'params' => array( 'aliasoptions' => $menuProfile ) ) );
			}

			$menuEdit				=	self::createMenu( 'mainmenu', CBTxt::T( 'CB_COMMUNITY_PROFILE_EDIT', 'Profile Edit' ), 'index.php?option=com_comprofiler&view=userdetails', array( 'parent_id' => $menuRoot, 'params' => array( 'menu_show' => 0 ) ) );

			if ( $menuEdit ) {
				self::createMenu( 'mainmenu', CBTxt::T( 'CB_COMMUNITY_EDIT_PROFILE', 'Edit Profile' ), 'index.php?Itemid=' . $menuEdit, array( 'alias' => 'profile-edit-alias', 'type' => 'alias', 'component_id' => 0, 'access' => $regAccess, 'parent_id' => $menuRoot, 'params' => array( 'aliasoptions' => $menuEdit ) ) );
			}

			// Profile feature menu endpoints like connections, gallery, notifications, private messages, and more:
			self::createMenu( 'mainmenu', CBTxt::T( 'CB_COMMUNITY_FEATURES', 'Features' ), '', array( 'type' => 'separator', 'component_id' => 0, 'access' => $regAccess, 'parent_id' => $menuRoot ) );

			$menuConn				=	self::createMenu( 'mainmenu', CBTxt::T( 'CB_COMMUNITY_CONNECTIONS', 'Connections' ), 'index.php?option=com_comprofiler&view=manageconnections', array( 'parent_id' => $menuRoot, 'params' => array( 'menu_show' => 0 ) ) );

			if ( $menuConn ) {
				self::createMenu( 'mainmenu', CBTxt::T( 'CB_COMMUNITY_MY_CONNECTIONS', 'My Connections' ), 'index.php?Itemid=' . $menuConn, array( 'alias' => 'connections-alias', 'type' => 'alias', 'component_id' => 0, 'access' => $regAccess, 'parent_id' => $menuRoot, 'params' => array( 'aliasoptions' => $menuConn ) ) );
			}

			if ( self::getPlugin( 'cbgroupjive' ) ) {
				self::createMenu( 'mainmenu', CBTxt::T( 'CB_COMMUNITY_MY_GROUPS', 'My Groups' ), 'index.php?option=com_comprofiler&view=pluginclass&plugin=cbgroupjive&action=mygroups', array( 'access' => $regAccess, 'parent_id' => $menuRoot ) );
			}

			if ( self::getPlugin( 'cbgallery' ) ) {
				$menuGallery		=	self::createMenu( 'mainmenu', CBTxt::T( 'CB_COMMUNITY_GALLERY', 'Gallery' ), 'index.php?option=com_comprofiler&view=pluginclass&plugin=cbgallery&action=gallery', array( 'parent_id' => $menuRoot, 'params' => array( 'menu_show' => 0 ) ) );

				if ( $menuGallery ) {
					self::createMenu( 'mainmenu', CBTxt::T( 'CB_COMMUNITY_MY_GALLERY', 'My Gallery' ), 'index.php?Itemid=' . $menuGallery, array( 'alias' => 'gallery-alias', 'type' => 'alias', 'component_id' => 0, 'access' => $regAccess, 'parent_id' => $menuRoot, 'params' => array( 'aliasoptions' => $menuGallery ) ) );
				}
			}

			if ( self::getPlugin( 'cbactivity' ) ) {
				$menuActivity		=	self::createMenu( 'mainmenu', CBTxt::T( 'CB_COMMUNITY_ACTIVITY', 'Activity' ), 'index.php?option=com_comprofiler&view=pluginclass&plugin=cbactivity&action=myactivity', array( 'parent_id' => $menuRoot, 'params' => array( 'menu_show' => 0 ) ) );

				if ( $menuActivity ) {
					self::createMenu( 'mainmenu', CBTxt::T( 'CB_COMMUNITY_MY_ACTIVITY', 'My Activity' ), 'index.php?Itemid=' . $menuActivity, array( 'alias' => 'activity-alias', 'type' => 'alias', 'component_id' => 0, 'access' => $regAccess, 'parent_id' => $menuRoot, 'params' => array( 'aliasoptions' => $menuActivity ) ) );
				}
			}

			$menuPMS				=	self::createMenu( 'mainmenu', CBTxt::T( 'CB_COMMUNITY_PRIVATE_MESSAGES', 'Private Messages' ), 'index.php?option=com_comprofiler&view=pluginclass&plugin=pms.mypmspro&action=messages', array( 'parent_id' => $menuRoot, 'params' => array( 'menu_show' => 0 ) ) );

			if ( $menuPMS ) {
				self::createMenu( 'mainmenu', CBTxt::T( 'CB_COMMUNITY_PRIVATE_MESSAGES', 'Private Messages' ), 'index.php?Itemid=' . $menuPMS, array( 'alias' => 'pms-alias', 'type' => 'alias', 'component_id' => 0, 'access' => $regAccess, 'parent_id' => $menuRoot, 'params' => array( 'aliasoptions' => $menuPMS ) ) );
			}

			if ( self::getPlugin( 'cbactivity' ) ) {
				self::createMenu( 'mainmenu', CBTxt::T( 'CB_COMMUNITY_NOTIFICATIONS', 'Notifications' ), 'index.php?option=com_comprofiler&view=pluginclass&plugin=cbactivity&action=notifications', array( 'access' => $regAccess, 'parent_id' => $menuRoot ) );
			}

			// Profile task menu endpoints like hidden activity, hidden comments, blocked users, and more:
			if ( self::getPlugin( 'cbactivity' ) || self::getPlugin( 'cbprivacy' ) ) {
				self::createMenu( 'mainmenu', CBTxt::T( 'CB_COMMUNITY_TASKS', 'Tasks' ), '', array( 'type' => 'separator', 'component_id' => 0, 'access' => $regAccess, 'parent_id' => $menuRoot ) );
			}

			if ( self::getPlugin( 'cbactivity' ) ) {
				self::createMenu( 'mainmenu', CBTxt::T( 'CB_COMMUNITY_HIDDEN_ACTIVITY', 'Hidden Activity' ), 'index.php?option=com_comprofiler&view=pluginclass&plugin=cbactivity&action=hiddenactivity', array( 'access' => $regAccess, 'parent_id' => $menuRoot ) );
				self::createMenu( 'mainmenu', CBTxt::T( 'CB_COMMUNITY_HIDDEN_COMMENTS', 'Hidden Comments' ), 'index.php?option=com_comprofiler&view=pluginclass&plugin=cbactivity&action=hiddencomments', array( 'access' => $regAccess, 'parent_id' => $menuRoot ) );
				self::createMenu( 'mainmenu', CBTxt::T( 'CB_COMMUNITY_HIDDEN_NOTIFICATIONS', 'Hidden Notifications' ), 'index.php?option=com_comprofiler&view=pluginclass&plugin=cbactivity&action=hiddennotifications', array( 'access' => $regAccess, 'parent_id' => $menuRoot ) );
			}

			if ( self::getPlugin( 'cbprivacy' ) ) {
				self::createMenu( 'mainmenu', CBTxt::T( 'CB_COMMUNITY_BLOCKED_USERS', 'Blocked Users' ), 'index.php?option=com_comprofiler&view=pluginclass&plugin=cbprivacy&action=blocked', array( 'access' => $regAccess, 'parent_id' => $menuRoot ) );
			}

			// Common menu endpoints such as groups, userlist, recent activity, and in general anything public:
			self::createMenu( 'mainmenu', CBTxt::T( 'CB_COMMUNITY_COMMON', 'Common' ), '', array( 'type' => 'separator', 'component_id' => 0, 'parent_id' => $menuRoot ) );
			self::createMenu( 'mainmenu', CBTxt::T( 'CB_COMMUNITY_MEMBERS', 'Members' ), 'index.php?option=com_comprofiler&view=userslist', array( 'parent_id' => $menuRoot ) );

			if ( self::getPlugin( 'cbgroupjive' ) ) {
				self::createMenu( 'mainmenu', CBTxt::T( 'CB_COMMUNITY_GROUPS', 'Groups' ), 'index.php?option=com_comprofiler&view=pluginclass&plugin=cbgroupjive&action=allgroups', array( 'parent_id' => $menuRoot ) );
			}

			if ( self::getPlugin( 'cbactivity' ) ) {
				self::createMenu( 'mainmenu', CBTxt::T( 'CB_COMMUNITY_RECENT_ACTIVITY', 'Recent Activity' ), 'index.php?option=com_comprofiler&view=pluginclass&plugin=cbactivity&action=recentactivity', array( 'parent_id' => $menuRoot ) );
			}

			// Moderator actions like user approval, image approval, and user reports:
			self::createMenu( 'mainmenu', CBTxt::T( 'CB_COMMUNITY_MODERATION', 'Moderation' ), '', array( 'type' => 'separator', 'component_id' => 0, 'access' => $modAccess, 'parent_id' => $menuRoot ) );
			self::createMenu( 'mainmenu', CBTxt::T( 'CB_COMMUNITY_USER_APPROVAL', 'User Approval' ), 'index.php?option=com_comprofiler&view=pendingapprovaluser', array( 'access' => $modAccess, 'parent_id' => $menuRoot ) );
			self::createMenu( 'mainmenu', CBTxt::T( 'CB_COMMUNITY_IMAGE_APPROVAL', 'Image Approval' ), 'index.php?option=com_comprofiler&view=moderateimages', array( 'access' => $modAccess, 'parent_id' => $menuRoot ) );
			self::createMenu( 'mainmenu', CBTxt::T( 'CB_COMMUNITY_USER_REPORTS', 'User Reports' ), 'index.php?option=com_comprofiler&view=moderatereports', array( 'access' => $modAccess, 'parent_id' => $menuRoot ) );
			self::createMenu( 'mainmenu', CBTxt::T( 'CB_COMMUNITY_USER_BANS', 'User Bans' ), 'index.php?option=com_comprofiler&view=moderatebans', array( 'access' => $modAccess, 'parent_id' => $menuRoot ) );

			if ( self::getPlugin( 'cbgallery' ) ) {
				self::createMenu( 'mainmenu', CBTxt::T( 'CB_COMMUNITY_GALLERY_APPROVAL', 'Gallery Approval' ), 'index.php?option=com_comprofiler&view=pluginclass&plugin=cbgallery&action=approval', array( 'access' => $modAccess, 'parent_id' => $menuRoot ) );
			}

			// Actions menu endpoints not commonly used like login, logout, and forgot login:
			self::createMenu( 'mainmenu', CBTxt::T( 'CB_COMMUNITY_ACTIONS', 'Actions' ), '', array( 'type' => 'separator', 'component_id' => 0, 'parent_id' => $menuRoot ) );

			$menuLogin				=	self::createMenu( 'mainmenu', CBTxt::T( 'CB_COMMUNITY_LOGIN', 'Login' ), 'index.php?option=com_comprofiler&view=login', array( 'parent_id' => $menuRoot, 'params' => array( 'menu_show' => 0 ) ) );

			if ( $menuLogin ) {
				self::createMenu( 'mainmenu', CBTxt::T( 'CB_COMMUNITY_LOGIN', 'Login' ), 'index.php?Itemid=' . $menuLogin, array( 'alias' => 'login-alias', 'type' => 'alias', 'component_id' => 0, 'access' => $guestAccess, 'parent_id' => $menuRoot, 'params' => array( 'aliasoptions' => $menuLogin ) ) );
			}

			$menuLogout				=	self::createMenu( 'mainmenu', CBTxt::T( 'CB_COMMUNITY_LOGOUT', 'Logout' ), 'index.php?option=com_comprofiler&view=logout', array( 'parent_id' => $menuRoot, 'params' => array( 'menu_show' => 0 ) ) );

			if ( $menuLogout ) {
				self::createMenu( 'mainmenu', CBTxt::T( 'CB_COMMUNITY_LOGOUT', 'Logout' ), 'index.php?Itemid=' . $menuLogout, array( 'alias' => 'logout-alias', 'type' => 'alias', 'component_id' => 0, 'access' => $regAccess, 'parent_id' => $menuRoot, 'params' => array( 'aliasoptions' => $menuLogout ) ) );
			}

			$menuForgot				=	self::createMenu( 'mainmenu', CBTxt::T( 'CB_COMMUNITY_FORGOT_LOGIN', 'Forgot Login' ), 'index.php?option=com_comprofiler&view=lostpassword', array( 'parent_id' => $menuRoot, 'params' => array( 'menu_show' => 0 ) ) );

			if ( $menuForgot ) {
				self::createMenu( 'mainmenu', CBTxt::T( 'CB_COMMUNITY_FORGOT_LOGIN', 'Forgot Login' ), 'index.php?Itemid=' . $menuForgot, array( 'alias' => 'forgot-login-alias', 'type' => 'alias', 'component_id' => 0, 'access' => $guestAccess, 'parent_id' => $menuRoot, 'params' => array( 'aliasoptions' => $menuForgot ) ) );
			}

			// Hidden menu items specifically for SEF purposes:
			self::createMenu( 'mainmenu', CBTxt::T( 'CB_COMMUNITY_BLOGS', 'Blogs' ), 'index.php?option=com_comprofiler&view=pluginclass&plugin=cbblogs&action=blogs', array( 'parent_id' => $menuRoot, 'params' => array( 'menu_show' => 0 ) ) );
			self::createMenu( 'mainmenu', CBTxt::T( 'CB_COMMUNITY_INVITES', 'Invites' ), 'index.php?option=com_comprofiler&view=pluginclass&plugin=cbinvites&action=invites', array( 'parent_id' => $menuRoot, 'params' => array( 'menu_show' => 0 ) ) );
		}

		// Joomla Plugins:
		$extensions				=	array(	'cbauthorbot'	=>	'content',
											'cbactivitybot'	=>	'content'
										);

		foreach ( $extensions as $element => $type ) {
			$extension			=	\JTable::getInstance( 'extension' );

			if ( ! $extension->load( array( 'type' => 'plugin', 'folder' => $type, 'element' => $element ) ) ) {
				continue;
			}

			$extensionParams	=	new \JRegistry;

			$extensionParams->loadString( $extension->get( 'params' ) );

			switch ( $element ) {
				case 'cbactivitybot':
					$extensionParams->set( 'comments', 1 );
					$extensionParams->set( 'likes', 1 );
					break;
			}

			$extension->set( 'enabled', 1 );
			$extension->set( 'params', $extensionParams->toString() );

			$extension->store();
		}
	}

	/**
	 * XML triggered reinstall; should redirect away when done
	 */
	static public function reinstall()
	{
		global $_CB_framework;

		self::install();

		cbRedirect( $_CB_framework->backendViewUrl( 'showPlugins' ), CBTxt::T( 'Quickstart Reinstalled Successfully!' ) );
	}

	/**
	 * Caches a plugin by element
	 * 
	 * @param string $element
	 * @param PluginTable $plugin
	 */
	static private function setPlugin( $element, $plugin )
	{
		self::$plugins[$element]	=	$plugin;
	}

	/**
	 * Returns a plugin object by element
	 *
	 * @param string $element
	 * @return null|PluginTable
	 */
	static private function getPlugin( $element )
	{
		if ( ! isset( self::$plugins[$element] ) ) {
			return null;
		}

		return self::$plugins[$element];
	}

	/**
	 * Caches a tab object by title or plugin class
	 *
	 * @param string $title
	 * @param TabTable $tab
	 */
	static private function setTab( $title, $tab )
	{
		self::$tabs[$title]	=	$tab;
	}

	/**
	 * Returns a tab object by title or plugin class
	 *
	 * @param string $title
	 * @return null|TabTable
	 */
	static private function getTab( $title )
	{
		if ( ! isset( self::$tabs[$title] ) ) {
			$tab	=	new TabTable();

			$tab->load( array( 'title' => $title ) );

			if ( ! $tab->get( 'tabid', 0, GetterInterface::INT ) ) {
				self::setTab( $title, null );
			} else {
				self::setTab( $title, $tab );
			}
		}

		return self::$tabs[$title];
	}

	/**
	 * Caches a field object by name
	 *
	 * @param string $name
	 * @param FieldTable $field
	 */
	static private function setField( $name, $field )
	{
		self::$fields[$name]	=	$field;
	}

	/**
	 * Returns a field object by name if the field exists
	 *
	 * @param string $name
	 * @return null|FieldTable
	 */
	static private function getField( $name )
	{
		if ( ! isset( self::$fields[$name] ) ) {
			$field	=	new FieldTable();

			$field->load( array( 'name' => $name ) );

			if ( ! $field->get( 'fieldid', 0, GetterInterface::INT ) ) {
				self::setField( $name, null );
			} else {
				self::setField( $name, $field );
			}
		}

		return self::$fields[$name];
	}

	/**
	 * Returns the next field order then increments the field order by tab id
	 *
	 * @param int $tabId
	 * @return int
	 */
	static private function getFieldOrder( $tabId )
	{
		if ( ! isset( self::$fieldOrder[$tabId] ) ) {
			self::$fieldOrder[$tabId]		=	1;
		}

		$order								=	self::$fieldOrder[$tabId];

		self::$fieldOrder[$tabId]++;

		return $order;
	}

	/**
	 * Creates a Joomla menu item
	 *
	 * @param string $menuType
	 * @param string $title
	 * @param string $url
	 * @param array  $data
	 * @return int
	 */
	static private function createMenu( $menuType, $title, $url, $data = array() )
	{
		global $_CB_database;

		static $menuOrder				=	array();
		static $menuModel				=	null;

		if ( $menuModel === null ) {
			\JLoader::register( 'MenusHelper', JPATH_ADMINISTRATOR . '/components/com_menus/helpers/menus.php' );
			\JModelLegacy::addIncludePath( JPATH_ADMINISTRATOR . '/components/com_menus/models/', 'MenusModel' );
			\JTable::addIncludePath( JPATH_ADMINISTRATOR . '/components/com_menus/tables/' );

			$menuModel					=	\JModelLegacy::getInstance( 'Item', 'MenusModel' );
		}

		static $menuTypeExists			=	null;

		if ( $menuTypeExists === null ) {
			if ( ! \JTable::getInstance( 'MenuType' )->load( array( 'menutype' => $menuType ) ) ) {
				$menuTypeExists			=	false;
			} else {
				$menuTypeExists			=	true;
			}
		}

		if ( ! $menuTypeExists ) {
			// The menutype we're trying to add menu items to doesn't exist so lets just skip this process:
			return 0;
		}

		$menuModel->setState( 'item.id', 0 );

		// Establish the default values:
		$data							=	array_merge( array(	'id'					=>	0,
																'created_user_id'		=>	Application::MyUser()->getUserId(),
																'menutype'				=>	$menuType,
																'title'					=>	$title,
																'alias'					=>	$title,
																'link'					=>	$url,
																'type'					=>	'component',
																'language'				=>	'*',
																'note'					=>	'',
																'img'					=>	'',
																'associations'			=>	array(),
																'published'				=>	1,
																'parent_id'				=>	1,
																'level'					=>	1,
																'access'				=>	1,
																'client_id'				=>	0,
																'home'					=>	0,
																'browserNav'			=>	0,
																'template_style_id'		=>	0,
												), $data );

		// Be sure the alias, whether default or custom, is URL safe:
		$data['alias']					=	Application::Router()->stringToAlias( $data['alias'] );

		if ( ( $data['parent_id'] > 1 ) && ( $data['level']	 == 1 ) ) {
			// Parent was set, but the nesting level is missing so adjust the nesting level for submenu:
			$data['level']				=	2;
		}

		if ( ! isset( $data['component_id'] ) ) {
			// Most menu items are CB specific so lets set their component id if not already set:
			$query						=	'SELECT ' . $_CB_database->NameQuote( 'extension_id' )
										.	"\n FROM " . $_CB_database->NameQuote( '#__extensions' )
										.	"\n WHERE " . $_CB_database->NameQuote( 'type' ) . " = " . $_CB_database->Quote( 'component' )
										.	"\n AND " . $_CB_database->NameQuote( 'element' ) . " = " . $_CB_database->Quote( 'com_comprofiler' )
										.	"\n ORDER BY " . $_CB_database->NameQuote( 'extension_id' ) . " DESC";
			$_CB_database->setQuery( $query, 0, 1 );
			$data['component_id']		=	(int) $_CB_database->loadResult();
		}

		$parent							=	$data['parent_id'];

		if ( ! isset( $menuOrder[$data['parent_id']] ) ) {
			// New sub menu starting so lets be sure the first entry is always ordered first:
			$menuOrder[$parent]			=	-1;
		}

		// Check if the menu exists already so we can update it:
		$menu							=	$menuModel->getTable();

		$menu->load( array(	'menutype'		=>	$data['menutype'],
							'alias'			=>	$data['alias'],
							'parent_id'		=>	$parent,
							'component_id'	=>	$data['component_id']
						) );

		if ( $menu->id ) {
			// Remove some defaults so modified menu items retain their changes:
			unset( $data['language'] );
			unset( $data['note'] );
			unset( $data['img'] );
			unset( $data['associations'] );
			unset( $data['published'] );
			unset( $data['home'] );
			unset( $data['browserNav'] );
			unset( $data['template_style_id'] );

			if ( isset( $data['params'] ) ) {
				$menuParams				=	$menu->params;

				if ( ! is_array( $menuParams ) ) {
					$paramsRegistry		=	new Registry( $menuParams );
					$menuParams			=	$paramsRegistry->asArray();
				}

				$data['params']			=	array_merge( $menuParams, $data['params'] );
			}

			$data['id']					=	(int) $menu->id;
			// Only change the ordering of sub-menu items as we don't want to rearrange a users top level menus:
			$data['menuordering']		=	( $data['parent_id'] > 1 ? $menuOrder[$parent] : (int) $menu->id );
		}

		if ( ! $menuModel->save( $data ) ) {
			return 0;
		}

		$menuId							=	( $menu->id ? $menu->id : $menuModel->getState( 'item.id' ) );

		// Menu ordering is based off ordering after the selected menu id so lets set the last created id here so we always insert after one another:
		$menuOrder[$parent]				=	$menuId;

		return $menuId;
	}
}