<?php
/**
* Community Builder (TM)
* @version $Id: $
* @package CommunityBuilder
* @copyright (C) 2004-2019 www.joomlapolis.com / Lightning MultiCom SA - and its licensors, all rights reserved
* @license http://www.gnu.org/licenses/old-licenses/gpl-2.0.html GNU/GPL version 2
*/

use CBLib\Application\Application;
use CBLib\Language\CBTxt;
use CBLib\Registry\ParamsInterface;
use CBLib\Registry\GetterInterface;
use CBLib\Input\Get;
use CB\Database\Table\UserTable;
use CB\Database\Table\TabTable;
use CB\Plugin\Activity\ActivityInterface;
use CB\Plugin\Activity\NotificationsInterface;
use CB\Plugin\Activity\CommentsInterface;
use CB\Plugin\Activity\TagsInterface;
use CB\Plugin\Activity\FollowingInterface;
use CB\Plugin\Activity\LikesInterface;
use CB\Plugin\Activity\CBActivity;
use CB\Plugin\Activity\Table\ActivityTable;
use CB\Plugin\Activity\Table\NotificationTable;
use CB\Plugin\Activity\Table\CommentTable;
use CB\Plugin\Activity\Table\TagTable;
use CB\Plugin\Activity\Table\FollowTable;
use CB\Plugin\Activity\Table\LikeTable;
use CB\Plugin\Activity\Table\HiddenTable;
use CB\Plugin\Activity\Table\ReadTable;
use CB\Plugin\Activity\Activity;
use CB\Plugin\Activity\Notifications;
use CB\Plugin\Activity\Comments;
use CB\Plugin\Activity\Tags;
use CB\Plugin\Activity\Following;
use CB\Plugin\Activity\Likes;

if ( ! ( defined( '_VALID_CB' ) || defined( '_JEXEC' ) || defined( '_VALID_MOS' ) ) ) { die( 'Direct Access to this location is not allowed.' ); }

global $_PLUGINS;

$_PLUGINS->loadPluginGroup( 'user' );

class CBplug_cbactivity extends cbPluginHandler
{

	/**
	 * @param  TabTable   $tab       Current tab
	 * @param  UserTable  $user      Current user
	 * @param  int        $ui        1 front, 2 admin UI
	 * @param  array      $postdata  Raw unfiltred POST data
	 */
	public function getCBpluginComponent( $tab, $user, $ui, $postdata )
	{
		if ( $this->getInput()->getString( 'action' ) === 'cleanup' ) {
			$this->cleanUp();
		} else {
			$this->getStream();
		}
	}

	/**
	 * Loads in a stream directly or by URL
	 *
	 * @param null|Activity|Comments|Tags $stream
	 * @param null|string                 $view
	 * @param int                         $id
	 */
	public function getStream( $stream = null, $view = null, $id = 0 )
	{
		global $_CB_framework, $_PLUGINS;

		$viewer								=	CBuser::getMyUserDataInstance();
		$raw								=	false;
		$menu								=	null;
		$access								=	true;

		if ( $stream ) {
			if ( $stream instanceof ActivityInterface ) {
				$action						=	'activity';
			} elseif ( $stream instanceof NotificationsInterface ) {
				$action						=	'notifications';
			} elseif ( $stream instanceof CommentsInterface ) {
				$action						=	'comments';
			} elseif ( $stream instanceof TagsInterface ) {
				$action						=	'tags';
			} elseif ( $stream instanceof FollowingInterface ) {
				$action						=	'following';
			} elseif ( $stream instanceof LikesInterface ) {
				$action						=	'likes';
			} else {
				return;
			}

			$function						=	( $view ? $view : 'show' );

			if ( strpos( $function, '.' ) !== false ) {
				list( $action, $function )	=	explode( '.', $function, 2 );
			}

			$inline							=	$stream->getBool( 'inline', true );
		} else {
			$raw							=	( $this->getInput()->getString( 'format' ) === 'raw' );
			$action							=	$this->getInput()->getString( 'action' );

			if ( strpos( $action, '.' ) !== false ) {
				list( $action, $function )	=	explode( '.', $action, 2 );
			} else {
				$function					=	$this->getInput()->getString( 'func' );
			}

			$id								=	$this->getInput()->getInt( 'id', 0 );
			$streamId						=	$this->getInput()->getString( 'stream' );
			$streamAsset					=	null;

			switch ( $action ) {
				case 'recentactivity':
					$action					=	'activity';
					$function				=	'show';
					$streamAsset			=	'all';
					break;
				case 'myactivity':
					$action					=	'activity';
					$function				=	'show';
					$streamAsset			=	'user.connections,following';
					break;
				case 'hiddenactivity':
					$action					=	'activity';
					$function				=	'hidden';
					$streamAsset			=	'all';
					break;
				case 'hiddencomments':
					$action					=	'comments';
					$function				=	'hidden';
					$streamAsset			=	'all';
					break;
				case 'hiddennotifications':
					$action					=	'notifications';
					$function				=	'hidden';
					$streamAsset			=	'all';
					break;
			}

			if ( ! $streamId ) {
				if ( $id ) {
					$streamAsset			=	CBActivity::getAsset( $action, $id );
				}

				$menu						=	Application::Cms()->getActiveMenuWithParams();

				if ( $menu->getInt( 'id' )
					 && ( $menu->getString( 'query.option' ) === 'com_comprofiler' )
					 && ( $menu->getString( 'query.view' ) === 'pluginclass' )
					 && ( $menu->getString( 'query.plugin' ) === 'cbactivity' )
				) {
					if ( ! $streamAsset ) {
						$streamAsset		=	str_replace( '[page_id]', $menu->getInt( 'id' ), $menu->getString( 'params/' . $action . '_asset' ) );
					}
				} else {
					$menu					=	null;
				}
			}

			switch ( $action ) {
				case 'likes':
					$stream					=	new Likes( $streamAsset, $viewer );
					break;
				case 'following':
					$stream					=	new Following( $streamAsset, $viewer );
					break;
				case 'tags':
					$stream					=	new Tags( $streamAsset, $viewer );
					break;
				case 'comments':
					$stream					=	new Comments( $streamAsset, $viewer );
					break;
				case 'notifications':
					$stream					=	new Notifications( $streamAsset, $viewer );
					break;
				case 'activity':
				default:
					$stream					=	new Activity( $streamAsset, $viewer );
					break;
			}

			if ( $menu ) {
				$stream->set( 'menu', $menu->getInt( 'id' ) );

				$stream->parse( $menu->getNamespaceRegistry( 'params' ), $action . '_' );

				if ( $action === 'notifications' ) {
					switch ( $menu->getString( 'params/notifications_state' ) ) {
						case 'read':
							$stream->set( 'read', 'readonly' );
							break;
						case 'unread':
							$stream->set( 'read', 'unreadonly' );
							break;
						case 'all':
							$stream->set( 'read', 'status' );
							break;
					}
				}
			}

			if ( $streamId ) {
				if ( ! $stream->load( $streamId ) ) {
					if ( $id && in_array( $action, array( 'activity', 'comments' ), true ) ) {
						$streamAsset		=	CBActivity::getAsset( $action, $id );

						if ( $streamAsset ) {
							$stream->assets( $streamAsset );
						} else {
							$access			=	false;
						}
					} else {
						$access				=	false;
					}
				}
			} elseif ( $function && ( ! in_array( $function, array( 'show', 'hidden' ), true ) ) ) {
				$access						=	false;
			}

			$inline							=	$stream->getBool( 'inline', false );
		}

		if ( ! $stream->asset() ) {
			$access							=	false;
		} elseif ( $stream instanceof NotificationsInterface ) {
			if ( ( ! $viewer->getInt( 'id', 0 ) ) || ( $viewer->getInt( 'id', 0 ) !== $stream->user()->getInt( 'id', 0 ) ) ) {
				$access						=	false;
			}
		} elseif ( ( $function !== 'hidden' ) && ( preg_match( '/^profile(?:\.(\d+)(?:\.field\.(\d+))?)?/', $stream->asset(), $matches ) || $stream->getInt( 'tab', 0 ) || $stream->getInt( 'field', 0 ) ) ) {
			$profileId						=	( isset( $matches[1] ) ? (int) $matches[1] : $stream->user()->getInt( 'id', 0 ) );
			$fieldId						=	( isset( $matches[2] ) ? (int) $matches[2] : $stream->getInt( 'field', 0 ) );
			$tabId							=	$stream->getInt( 'tab', 0 );

			if ( $profileId !== $stream->user()->getInt( 'id', 0 ) ) {
				$stream->user( $profileId );
			}

			if ( $fieldId ) {
				$field						=	CBActivity::getField( $fieldId, $profileId );

				if ( ! $field ) {
					$access					=	false;
				} else {
					// ALWAYS reload field params to prevent streams going out of sync:
					$stream->set( 'field', $field->getInt( 'fieldid', 0 ) );

					switch ( $field->getString( 'type' ) ) {
						case 'activity':
							$stream->parse( $field->params, 'activity_', false );
							break;
						case 'comments':
							$stream->parse( $field->params, 'comments_', false );
							break;
						case 'follow':
							$stream->parse( $field->params, 'following_', false );
							break;
						case 'like':
							$stream->parse( $field->params, 'likes_', false );
							break;
						case 'notifications':
							$stream->parse( $field->params, 'notifications_', false );
							break;
					}
				}
			} else {
				$tab						=	CBActivity::getTab( $tabId, $profileId );

				if ( ! $tab ) {
					if ( ! in_array( 'all', $stream->assets(), true ) ) {
						$access				=	false;
					}
				} else {
					// ALWAYS reload tab params to prevent streams going out of sync:
					$stream->set( 'tab', $tab->getInt( 'tabid', 0 ) );

					$stream->parse( $tab->params, 'activity_', false );
				}
			}
		}

		$_PLUGINS->trigger( 'activity_onStreamAccess', array( &$stream, &$access ) );

		if ( ! $access ) {
			if ( $inline ) {
				return;
			}

			if ( $raw ) {
				header( 'HTTP/1.0 401 Unauthorized' );
				exit();
			}

			cbRedirect( 'index.php', CBTxt::T( 'Not authorized.' ), 'error' );
		}

		if ( ! $stream->id() ) {
			$stream->cache();
		}

		if ( ! $raw ) {
			static $once					=	0;

			if ( ! $once++ ) {
				outputCbJs();
				outputCbTemplate();
			}

			ob_start();
		}

		switch ( $action ) {
			case 'activity':
				switch ( $function ) {
					case 'edit':
						$this->showActivityEdit( $id, $viewer, $stream );
						break;
					case 'save':
						$this->saveActivity( $id, $viewer, $stream );
						break;
					case 'delete':
						$this->deleteActivity( $id, $viewer, $stream );
						break;
					case 'button':
						$this->showActivityButton( $viewer, $stream, ( $raw ? 'refresh' : null ) );
						break;
					case 'hidden':
						if ( ! $viewer->getInt( 'id', 0 ) ) {
							cbRedirect( 'index.php', CBTxt::T( 'Not authorized.' ), 'error' );
						}

						$stream->user( $viewer );
						$stream->assets( 'all' );

						$stream->set( 'hidden', true );
						$stream->set( 'filters', array() );
						$stream->set( 'create', false );
						$stream->set( 'auto_update', false );
						$stream->set( 'pinned', false );
						$stream->set( 'likes', false );
						$stream->set( 'comments', false );

						$stream->cache();

						$this->showActivity( $id, $viewer, $stream );
						break;
					case 'hide':
						$this->hideActivity( $id, $viewer, $stream );
						break;
					case 'unhide':
						$this->unhideActivity( $id, $viewer, $stream );
						break;
					case 'unfollow':
						$this->unfollowActivity( $id, $viewer, $stream );
						break;
					case 'report':
						$this->reportActivity( $id, $viewer, $stream );
						break;
					case 'pin':
						$this->pinActivity( $id, $viewer, $stream );
						break;
					case 'unpin':
						$this->unpinActivity( $id, $viewer, $stream );
						break;
					case 'reactions':
						$this->showReactions( $viewer, $stream );
						break;
					case 'modal':
					case 'toggle':
						$this->showActivity( null, $viewer, $stream, $function );
						break;
					case 'update':
					case 'load':
						$this->showActivity( $id, $viewer, $stream, $function );
						break;
					case 'show':
					default:
						$this->showActivity( $id, $viewer, $stream );
						break;
				}
				break;
			case 'notifications':
				switch ( $function ) {
					case 'delete':
						$this->deleteNotification( $id, $viewer, $stream );
						break;
					case 'button':
						$this->showNotificationsButton( $viewer, $stream, ( $raw ? 'refresh' : null ) );
						break;
					case 'hidden':
						if ( ! $viewer->getInt( 'id', 0 ) ) {
							cbRedirect( 'index.php', CBTxt::T( 'Not authorized.' ), 'error' );
						}

						$stream->user( $viewer );
						$stream->assets( 'all' );

						$stream->set( 'hidden', true );
						$stream->set( 'read', null );
						$stream->set( 'auto_update', false );
						$stream->set( 'pinned', false );

						$stream->cache();

						$this->showNotifications( $id, $viewer, $stream );
						break;
					case 'hide':
						$this->hideNotification( $id, $viewer, $stream );
						break;
					case 'unhide':
						$this->unhideNotification( $id, $viewer, $stream );
						break;
					case 'load':
					case 'modal':
					case 'toggle':
						$this->showNotifications( null, $viewer, $stream, $function );
						break;
					case 'update':
						$this->showNotifications( $id, $viewer, $stream, 'update' );
						break;
					case 'show':
					default:
						$this->showNotifications( $id, $viewer, $stream );
						break;
				}
				break;
			case 'comments':
				switch ( $function ) {
					case 'edit':
						$this->showCommentEdit( $id, $viewer, $stream );
						break;
					case 'save':
						$this->saveComment( $id, $viewer, $stream );
						break;
					case 'delete':
						$this->deleteComment( $id, $viewer, $stream );
						break;
					case 'button':
						$this->showCommentsButton( $viewer, $stream, ( $raw ? 'refresh' : null ) );
						break;
					case 'hidden':
						if ( ! $viewer->getInt( 'id', 0 ) ) {
							cbRedirect( 'index.php', CBTxt::T( 'Not authorized.' ), 'error' );
						}

						$stream->user( $viewer );
						$stream->assets( 'all' );

						$stream->set( 'hidden', true );
						$stream->set( 'create', false );
						$stream->set( 'auto_update', false );
						$stream->set( 'pinned', false );
						$stream->set( 'likes', false );
						$stream->set( 'replies', false );

						$stream->cache();

						$this->showComments( $id, $viewer, $stream );
						break;
					case 'hide':
						$this->hideComment( $id, $viewer, $stream );
						break;
					case 'unhide':
						$this->unhideComment( $id, $viewer, $stream );
						break;
					case 'report':
						$this->reportComment( $id, $viewer, $stream );
						break;
					case 'pin':
						$this->pinComment( $id, $viewer, $stream );
						break;
					case 'unpin':
						$this->unpinComment( $id, $viewer, $stream );
						break;
					case 'reactions':
						$this->showReactions( $viewer, $stream );
						break;
					case 'modal':
					case 'toggle':
						$this->showComments( null, $viewer, $stream, $function );
						break;
					case 'update':
					case 'load':
						$this->showComments( $id, $viewer, $stream, $function );
						break;
					case 'show':
					default:
						$this->showComments( $id, $viewer, $stream );
						break;
				}
				break;
			case 'tags':
				switch ( $function ) {
					case 'edit':
						$this->showTagsEdit( $viewer, $stream );
						break;
					case 'save':
						$this->saveTags( $viewer, $stream );
						break;
					case 'tagged':
						$this->showTagged( $viewer, $stream );
						break;
					case 'list':
						$this->showTagsList( $viewer, $stream );
						break;
					case 'load':
					case 'modal':
						$this->showTags( $viewer, $stream, $function );
						break;
					case 'show':
					default:
						$this->showTags( $viewer, $stream );
						break;
				}
				break;
			case 'following':
				switch ( $function ) {
					case 'follow':
						$this->followAsset( $viewer, $stream );
						break;
					case 'unfollow':
						$this->unfollowAsset( $viewer, $stream );
						break;
					case 'button':
						$this->showFollowButton( $viewer, $stream );
						break;
					case 'load':
					case 'modal':
						$this->showFollowing( $viewer, $stream, $function );
						break;
					case 'show':
					default:
						$this->showFollowing( $viewer, $stream );
						break;
				}
				break;
			case 'likes':
				switch ( $function ) {
					case 'like':
						$this->likeAsset( $viewer, $stream );
						break;
					case 'unlike':
						$this->unlikeAsset( $viewer, $stream );
						break;
					case 'button':
						$this->showLikeButton( $viewer, $stream );
						break;
					case 'load':
					case 'modal':
						$this->showLikes( $viewer, $stream, $function );
						break;
					case 'show':
					default:
						$this->showLikes( $viewer, $stream );
						break;
				}
				break;
		}

		if ( ! $raw ) {
			$html							=	ob_get_clean();

			if ( ! $html ) {
				return;
			}

			if ( $inline ) {
				echo $html;
			} else {
				$class						=	$this->params->getString( 'general_class' );

				$return						=	'<' . ( in_array( $function, array( 'button', 'tagged' ), true ) ? 'span' : 'div' ) . ' class="cbActivity' . ( $class ? ' ' . htmlspecialchars( $class ) : null ) . '">'
											.		$html
											.	'</' . ( in_array( $function, array( 'button', 'tagged' ), true ) ? 'span' : 'div' ) . '>';

				echo $return;

				if ( $menu && $menu->getInt( 'id' ) ) {
					$_CB_framework->setMenuMeta();
				}
			}
		}
	}

	/**
	 * Prunes old activity entries
	 */
	private function cleanUp()
	{
		global $_CB_database, $_CB_framework;

		if ( $this->getInput()->getString( 'token' ) !== md5( $_CB_framework->getCfg( 'secret' ) ) ) {
			header( 'HTTP/1.0 401 Unauthorized' );
			exit();
		}

		$durationActivity			=	$this->params->getString( 'cleanup_activity', '-2 YEAR' );

		if ( $durationActivity === 'custom' ) {
			$durationActivity		=	$this->params->getString( 'cleanup_activity_custom', '-10 YEAR' );
		}

		if ( $durationActivity && ( $durationActivity !== 'forever' ) ) {
			$query					=	'SELECT *'
									.	"\n FROM " . $_CB_database->NameQuote( '#__comprofiler_plugin_activity' )
									.	"\n WHERE " . $_CB_database->NameQuote( 'date' ) . " <= " . $_CB_database->Quote( Application::Date( 'now', 'UTC' )->modify( $durationActivity )->format( 'Y-m-d H:i:s' ) );
			$_CB_database->setQuery( $query );
			$activities				=	$_CB_database->loadObjectList( null, '\CB\Plugin\Activity\Table\ActivityTable', array( $_CB_database ) );

			/** @var ActivityTable[] $activities */
			foreach ( $activities as $activity ) {
				$activity->delete();
			}
		}

		$durationNotifications		=	$this->params->getString( 'cleanup_notifications', '-2 YEAR' );

		if ( $durationNotifications === 'custom' ) {
			$durationNotifications	=	$this->params->getString( 'cleanup_notifications_custom', '-10 YEAR' );
		}

		if ( $durationNotifications && ( $durationNotifications !== 'forever' ) ) {
			$query					=	'SELECT *'
									.	"\n FROM " . $_CB_database->NameQuote( '#__comprofiler_plugin_activity_notifications' )
									.	"\n WHERE " . $_CB_database->NameQuote( 'date' ) . " <= " . $_CB_database->Quote( Application::Date( 'now', 'UTC' )->modify( $durationNotifications )->format( 'Y-m-d H:i:s' ) );
			$_CB_database->setQuery( $query );
			$notifications			=	$_CB_database->loadObjectList( null, '\CB\Plugin\Activity\Table\NotificationTable', array( $_CB_database ) );

			/** @var NotificationTable[] $notifications */
			foreach ( $notifications as $notification ) {
				$notification->delete();
			}
		}

		$durationComments			=	$this->params->getString( 'cleanup_comments', '-2 YEARS' );

		if ( $durationComments === 'custom' ) {
			$durationComments		=	$this->params->getString( 'cleanup_comments_custom', '-10 YEAR' );
		}

		if ( $durationComments && ( $durationComments !== 'forever' ) ) {
			$query					=	'SELECT *'
									.	"\n FROM " . $_CB_database->NameQuote( '#__comprofiler_plugin_activity_comments' )
									.	"\n WHERE " . $_CB_database->NameQuote( 'date' ) . " <= " . $_CB_database->Quote( Application::Date( 'now', 'UTC' )->modify( $durationComments )->format( 'Y-m-d H:i:s' ) );
			$_CB_database->setQuery( $query );
			$comments				=	$_CB_database->loadObjectList( null, '\CB\Plugin\Activity\Table\CommentTable', array( $_CB_database ) );

			/** @var CommentTable[] $comments */
			foreach ( $comments as $comment ) {
				$comment->delete();
			}
		}

		$durationLikes				=	$this->params->getString( 'cleanup_likes', 'forever' );

		if ( $durationLikes === 'custom' ) {
			$durationLikes			=	$this->params->getString( 'cleanup_likes_custom', '-10 YEAR' );
		}

		if ( $durationLikes && ( $durationLikes !== 'forever' ) ) {
			$query					=	'SELECT *'
									.	"\n FROM " . $_CB_database->NameQuote( '#__comprofiler_plugin_activity_likes' )
									.	"\n WHERE " . $_CB_database->NameQuote( 'date' ) . " <= " . $_CB_database->Quote( Application::Date( 'now', 'UTC' )->modify( $durationLikes )->format( 'Y-m-d H:i:s' ) );
			$_CB_database->setQuery( $query );
			$likes					=	$_CB_database->loadObjectList( null, '\CB\Plugin\Activity\Table\LikeTable', array( $_CB_database ) );

			/** @var LikeTable[] $likes */
			foreach ( $likes as $like ) {
				$like->delete();
			}
		}

		$durationFollowing			=	$this->params->getString( 'cleanup_following', 'forever' );

		if ( $durationFollowing === 'custom' ) {
			$durationFollowing		=	$this->params->getString( 'cleanup_following_custom', '-10 YEAR' );
		}

		if ( $durationFollowing && ( $durationFollowing !== 'forever' ) ) {
			$query					=	'SELECT *'
									.	"\n FROM " . $_CB_database->NameQuote( '#__comprofiler_plugin_activity_following' )
									.	"\n WHERE " . $_CB_database->NameQuote( 'date' ) . " <= " . $_CB_database->Quote( Application::Date( 'now', 'UTC' )->modify( $durationFollowing )->format( 'Y-m-d H:i:s' ) );
			$_CB_database->setQuery( $query );
			$following				=	$_CB_database->loadObjectList( null, '\CB\Plugin\Activity\Table\FollowTable', array( $_CB_database ) );

			/** @var FollowTable[] $following */
			foreach ( $following as $follow ) {
				$follow->delete();
			}
		}

		$durationTags				=	$this->params->getString( 'cleanup_tags', 'forever' );

		if ( $durationTags === 'custom' ) {
			$durationTags			=	$this->params->getString( 'cleanup_tags_custom', '-10 YEAR' );
		}

		if ( $durationTags && ( $durationTags !== 'forever' ) ) {
			$query					=	'SELECT *'
									.	"\n FROM " . $_CB_database->NameQuote( '#__comprofiler_plugin_activity_tags' )
									.	"\n WHERE " . $_CB_database->NameQuote( 'date' ) . " <= " . $_CB_database->Quote( Application::Date( 'now', 'UTC' )->modify( $durationTags )->format( 'Y-m-d H:i:s' ) );
			$_CB_database->setQuery( $query );
			$tags					=	$_CB_database->loadObjectList( null, '\CB\Plugin\Activity\Table\TagTable', array( $_CB_database ) );

			/** @var TagTable[] $tags */
			foreach ( $tags as $tag ) {
				$tag->delete();
			}
		}

//		$durationHidden				=	$this->params->getString( 'cleanup_hidden', 'forever' );
//
//		if ( $durationHidden === 'custom' ) {
//			$durationHidden			=	$this->params->getString( 'cleanup_hidden_custom', '-10 YEAR' );
//		}
//
//		if ( $durationHidden && ( $durationHidden !== 'forever' ) ) {
//			$query					=	'SELECT *'
//									.	"\n FROM " . $_CB_database->NameQuote( '#__comprofiler_plugin_activity_hidden' )
//									.	"\n WHERE " . $_CB_database->NameQuote( 'date' ) . " <= " . $_CB_database->Quote( Application::Date( 'now', 'UTC' )->modify( $durationHidden )->format( 'Y-m-d H:i:s' ) );
//			$_CB_database->setQuery( $query );
//			$hidden					=	$_CB_database->loadObjectList( null, '\CB\Plugin\Activity\Table\HiddenTable', array( $_CB_database ) );
//
//			/** @var HiddenTable[] $hidden */
//			foreach ( $hidden as $hiddenRow ) {
//				$hiddenRow->delete();
//			}
//		}

		$durationRead				=	$this->params->getString( 'cleanup_read', 'forever' );

		if ( $durationRead === 'custom' ) {
			$durationRead			=	$this->params->getString( 'cleanup_read_custom', '-10 YEAR' );
		}

		if ( $durationRead && ( $durationRead !== 'forever' ) ) {
			$query					=	'SELECT *'
									.	"\n FROM " . $_CB_database->NameQuote( '#__comprofiler_plugin_activity_read' )
									.	"\n WHERE " . $_CB_database->NameQuote( 'date' ) . " <= " . $_CB_database->Quote( Application::Date( 'now', 'UTC' )->modify( $durationRead )->format( 'Y-m-d H:i:s' ) );
			$_CB_database->setQuery( $query );
			$read					=	$_CB_database->loadObjectList( null, '\CB\Plugin\Activity\Table\ReadTable', array( $_CB_database ) );

			/** @var ReadTable[] $read */
			foreach ( $read as $readRow ) {
				$readRow->delete();
			}
		}

		header( 'HTTP/1.0 200 OK' );
		exit();
	}

	/**
	 * Displays activity stream
	 *
	 * @param int       $id
	 * @param UserTable $viewer
	 * @param Activity  $stream
	 * @param string    $output
	 */
	private function showActivity( $id, $viewer, $stream, $output = null )
	{
		global $_CB_framework, $_PLUGINS;

		$activityPrefix				=	'activity_' . substr( $stream->id(), 0, 5 ) . '_';

		if ( $id ) {
			if ( $output === 'update' ) {
				$stream->set( 'date', array( Application::Database()->getUtcDateTime( $id ), '>' ) );
			} elseif ( $output === 'load' ) {
				$stream->set( 'date', array( Application::Database()->getUtcDateTime( $id ), '<' ) );
			} else {
				$stream->set( 'id', $id );
			}
		} elseif ( $output === 'update' ) {
			return;
		}

		$rowsHashtag				=	$this->getInput()->getString( 'hashtag' );
		$rowsFilter					=	$this->getInput()->getInt( $activityPrefix . 'filter' );

		if ( $rowsFilter ) {
			$stream->set( 'filter', $rowsFilter );
		} else {
			$rowsFilter				=	null;
		}

		if ( $rowsHashtag && preg_match( '/^(\w+)$/i', $rowsHashtag ) ) {
			$rowsSearch				=	'#' . $rowsHashtag;
		} else {
			$rowsHashtag			=	null;
			$rowsSearch				=	$this->getInput()->getString( $activityPrefix . 'search', '' );
		}

		$searching					=	false;

		if ( $rowsSearch !== '' ) {
			$searching				=	true;

			$stream->set( 'search', $rowsSearch );
		} else {
			$rowsSearch				=	null;
		}

		$stream->set( 'paging_limitstart', $this->getInput()->getInt( $activityPrefix . 'limitstart', 0 ) );

		if ( $stream->getBool( 'query', true ) ) {
			if ( $stream->has( 'query_count' ) ) {
				$rowsTotal			=	$stream->getInt( 'query_count', 0 );
			} else {
				// We'll find out the total for paging after we perform the SELECT query:
				$rowsTotal			=	null;
			}
		} else {
			$rowsTotal				=	0;
		}

		if ( $stream->getInt( 'paging_limitstart', 0 ) === 0 ) {
			$pageLimit				=	$stream->getInt( 'paging_first_limit', 15 );
		} else {
			$pageLimit				=	$stream->getInt( 'paging_limit', 15 );
		}

		$pageNav					=	new cbPageNav( $rowsTotal, $stream->getInt( 'paging_limitstart', 0 ), $pageLimit );

		$pageNav->setInputNamePrefix( $activityPrefix );
		$pageNav->setBaseURL( $_CB_framework->pluginClassUrl( $this->element, false, array( 'action' => 'activity', 'func' => 'load', 'stream' => $stream->id(), 'hashtag' => $rowsHashtag, $activityPrefix . 'filter' => $rowsFilter, $activityPrefix . 'search' => $rowsSearch ), 'raw', 0, true ) );

		$rows						=	array();

		if ( $stream->getBool( 'query', true ) && ( $rowsTotal || ( $rowsTotal === null ) ) ) {
			$rows					=	$stream->rows();

			if ( $stream->getString( 'direction', 'down' ) === 'up' ) {
				$rows				=	array_reverse( $rows, true );
			}

			if ( $rowsTotal === null ) {
				$pageNav->total		=	$stream->getInt( 'paging_total', 0 );
			}
		}

		$pageNav->limitstart		=	$stream->getInt( 'paging_limitstart', 0 );

		$integrations				=	$_PLUGINS->trigger( 'activity_onBeforeDisplayActivityStream', array( &$rows, &$pageNav, &$searching, $viewer, &$stream, $output ) );
		$canCreate					=	( ( ! $stream->getRaw( 'id' ) ) && CBActivity::canCreate( 'activity', $stream ) );

		if ( ( ! $rows ) && ( in_array( $output, array( 'load', 'update' ), true ) || ( $stream->getBool( 'inline', false ) && ( ! $canCreate ) ) ) ) {
			return;
		}

		$pinnedTooltip				=	cbTooltip( null, CBTxt::T( 'Pinned' ), null, 'auto', null, null, null, 'data-hascbtooltip="true" data-cbtooltip-position-my="bottom center" data-cbtooltip-position-at="top center" data-cbtooltip-classes="qtip-simple"' );
		$globalTooltip				=	cbTooltip( null, CBTxt::T( 'Announcement' ), null, 'auto', null, null, null, 'data-hascbtooltip="true" data-cbtooltip-position-my="bottom center" data-cbtooltip-position-at="top center" data-cbtooltip-classes="qtip-simple"' );
		$reportedTooltip			=	cbTooltip( null, CBTxt::T( 'Controversial' ), null, 'auto', null, null, null, 'data-hascbtooltip="true" data-cbtooltip-position-my="bottom center" data-cbtooltip-position-at="top center" data-cbtooltip-classes="qtip-simple"' );
		$direction					=	$stream->getString( 'direction', 'down' );

		if ( ! in_array( $output, array( 'load', 'update' ), true ) ) {
			CBActivity::bindStream();

			require CBActivity::getTemplate( $stream->getString( 'template' ), 'activity/display', ( $output ? false : true ) );
		} else {
			require CBActivity::getTemplate( $stream->getString( 'template' ), 'activity/rows', false );
		}
	}

	/**
	 * Displays activity edit
	 *
	 * @param int       $id
	 * @param UserTable $viewer
	 * @param Activity  $stream
	 * @param string    $output
	 */
	private function showActivityEdit( $id, $viewer, $stream, $output = null )
	{
		global $_CB_framework, $_PLUGINS;

		$row	=	$stream->row( $id );

		if ( ! $row->getInt( 'id', 0 ) ) {
			if ( ! CBActivity::canCreate( 'activity', $stream, $viewer ) ) {
				return;
			}
		} elseif ( ( ( $viewer->getInt( 'id', 0 ) !== $row->getInt( 'user_id', 0 ) ) && ( ! CBActivity::canModerate( $stream ) ) )
			 || ( $row->getBool( 'system', false ) && ( ! Application::MyUser()->isGlobalModerator() ) )
			 || ( $row->getBool( 'global', false ) && ( ! Application::MyUser()->isGlobalModerator() ) )
			 || ( ! CBActivity::findParamOverride( $row, 'edit', true ) )
		) {
			CBActivity::ajaxResponse( CBTxt::T( 'You do not have permission to edit this activity.' ), 'error' );
		}

		if ( $row->getInt( 'id', 0 ) ) {
			$rowId				=	md5( $stream->id() . '_edit_' . $row->getInt( 'id', 0 ) );
		} else {
			$rowId				=	md5( $stream->id() . '_new' );
		}

		$canModerate			=	CBActivity::canModerate( $stream );
		$rowOwner				=	( ( $viewer->getInt( 'id', 0 ) === $row->getInt( 'user_id', 0 ) ) || ( ! $row->getInt( 'id', 0 ) ) );
		$form					=	array();
		$buttons				=	array( 'left' => array(), 'right' => array() );

		if ( $row->getInt( 'id', 0 ) ) {
			$integrations		=	$_PLUGINS->trigger( 'activity_onDisplayStreamActivityEdit', array( &$row, &$buttons, $viewer, $stream, $output ) );
			$collapsed			=	false;
		} else {
			$integrations		=	$_PLUGINS->trigger( 'activity_onDisplayStreamActivityNew', array( &$buttons, $viewer, $stream, $output ) );
			$collapsed			=	$stream->getBool( 'collapsed', true );
		}

		$form['integrations']	=	$integrations;
		$form['collapsed']		=	$collapsed;
		$form['buttons']		=	$buttons;

		$messageLimit			=	( $canModerate ? 0 : $stream->getInt( 'message_limit', 400 ) );
		$showThemes				=	CBActivity::findParamOverride( $row, 'themes', true, $stream );
		$showActions			=	CBActivity::findParamOverride( $row, 'actions', true, $stream );
		$actionLimit			=	( $canModerate ? 0 : $stream->getInt( 'actions_message_limit', 100 ) );
		$showLocations			=	CBActivity::findParamOverride( $row, 'locations', true, $stream );
		$locationLimit			=	( $canModerate ? 0 : $stream->getInt( 'locations_address_limit', 200 ) );
		$showLinks				=	CBActivity::findParamOverride( $row, 'links', true, $stream );
		$linkLimit				=	( $canModerate ? 0 : $stream->getInt( 'links_link_limit', 5 ) );
		$showTags				=	CBActivity::findParamOverride( $row, 'tags', true, $stream );
		$showEmotes				=	$stream->getBool( 'parser_emotes', true );
		$showReactions			=	( $stream->getBool( 'parser_reactions', true ) && CBActivity::getGlobalParams()->getString( 'reactions_giphy' ) );

		$themeTooltip			=	cbTooltip( null, CBTxt::T( 'Is there a theme for your post?' ), null, 'auto', null, null, null, 'data-hascbtooltip="true" data-cbtooltip-position-my="bottom center" data-cbtooltip-position-at="top center" data-cbtooltip-classes="qtip-simple"' );
		$actionTooltip			=	cbTooltip( null, CBTxt::T( 'What are you doing or feeling?' ), null, 'auto', null, null, null, 'data-hascbtooltip="true" data-cbtooltip-position-my="bottom center" data-cbtooltip-position-at="top center" data-cbtooltip-classes="qtip-simple"' );
		$locationTooltip		=	cbTooltip( null, CBTxt::T( 'Share your location.' ), null, 'auto', null, null, null, 'data-hascbtooltip="true" data-cbtooltip-position-my="bottom center" data-cbtooltip-position-at="top center" data-cbtooltip-classes="qtip-simple"' );
		$findLocationTooltip	=	cbTooltip( null, CBTxt::T( 'Click to try and find your location.' ), null, 'auto', null, null, null, 'data-hascbtooltip="true" data-cbtooltip-position-my="bottom center" data-cbtooltip-position-at="top center" data-cbtooltip-classes="qtip-simple"' );
		$tagTooltip				=	cbTooltip( null, CBTxt::T( 'Are you with anyone?' ), null, 'auto', null, null, null, 'data-hascbtooltip="true" data-cbtooltip-position-my="bottom center" data-cbtooltip-position-at="top center" data-cbtooltip-classes="qtip-simple"' );
		$linkTooltip			=	cbTooltip( null, CBTxt::T( 'Have a link to share?' ), null, 'auto', null, null, null, 'data-hascbtooltip="true" data-cbtooltip-position-my="bottom center" data-cbtooltip-position-at="top center" data-cbtooltip-classes="qtip-simple"' );
		$systemTooltip			=	cbTooltip( null, CBTxt::T( 'Is this a post from the site?' ), null, 'auto', null, null, null, 'data-hascbtooltip="true" data-cbtooltip-position-my="bottom center" data-cbtooltip-position-at="top center" data-cbtooltip-classes="qtip-simple"' );
		$globalTooltip			=	cbTooltip( null, CBTxt::T( 'Is this a global announcement?' ), null, 'auto', null, null, null, 'data-hascbtooltip="true" data-cbtooltip-position-my="bottom center" data-cbtooltip-position-at="top center" data-cbtooltip-classes="qtip-simple"' );
		$themeOptions			=	( $showThemes ? CBActivity::loadThemeOptions( false, $row->getInt( 'user_id', $viewer->getInt( 'id', 0 ) ), $stream ) : array() );
		$actionOptions			=	( $showActions ? CBActivity::loadActionOptions( false, $row->getInt( 'user_id', $viewer->getInt( 'id', 0 ) ), $stream ) : array() );
		$locationOptions		=	( $showLocations ? CBActivity::loadLocationOptions( false, $row->getInt( 'user_id', $viewer->getInt( 'id', 0 ) ), $stream ) : array() );
		$emoteOptions			=	( $actionOptions || $showEmotes ? CBActivity::loadEmoteOptions( false, false, $row->getInt( 'user_id', $viewer->getInt( 'id', 0 ) ) ) : array() );

		$form['url']						=	$_CB_framework->pluginClassUrl( 'cbactivity', true, array( 'action' => 'activity', 'func' => 'save', 'id' => $row->getInt( 'id', 0 ), 'stream' => $stream->id() ), 'raw', 0, true );
		$form['message']					=	'<textarea name="message" rows="1" class="form-control shadow-none border-0 streamInput streamInputAutosize streamInputMessage' . ( $collapsed ? ' streamInputMessageCollapse' : null ) . '" placeholder="' . htmlspecialchars( CBTxt::T( "What's on your mind?" ) ) . '"' . ( $messageLimit ? ' data-cbactivity-input-limit="' . (int) $messageLimit . '" maxlength="' . (int) $messageLimit . '"' : null ) . '>' . htmlspecialchars( $row->getString( 'message' ) ) . '</textarea>';
		$form['message_limit']				=	$messageLimit;
		$form['reactions']					=	null;

		if ( $showReactions ) {
			$form['reactions']				=	CBActivity::renderSelectList( array(), 'insert_reaction', 'class="streamInputSelect streamInputSelectInsert streamInputMessageReactionSelect" data-cbselect-width="auto" data-cbselect-height="auto" data-cbactivity-toggle-icon="text-small streamReactionsIcon" data-cbselect-dropdown-css-class="streamReactionOptions streamReactionInsertOptions" data-cbselect-url="' . $_CB_framework->pluginClassUrl( $this->element, true, array( 'action' => 'activity', 'func' => 'reactions', 'stream' => $stream->id() ), 'raw', 0, true ) . '"', $stream );
		}

		$form['emotes']						=	null;

		if ( $showEmotes && $emoteOptions ) {
			$form['emotes']					=	CBActivity::renderSelectList( $emoteOptions, 'insert_emote', 'class="streamInputSelect streamInputSelectInsert streamInputMessageEmoteSelect" data-cbselect-width="auto" data-cbselect-height="auto" data-cbactivity-toggle-icon="fa-before fa-smile-o" data-cbselect-dropdown-css-class="streamEmoteOptions streamEmoteInsertOptions"', $stream );
		}

		$form['actions']					=	null;
		$form['actions_message']			=	null;
		$form['actions_emotes']				=	null;

		if ( $actionOptions ) {
			$action							=	$row->params()->subTree( 'action' );
			$actionId						=	$action->getInt( 'id', 0 );

			$form['actions']				=	CBActivity::renderSelectList( $actionOptions, 'actions[id]', 'class="btn btn-sm ' . ( $actionId ? 'btn-info' : 'btn-light border' ) . ' streamInputSelect streamInputSelectToggle streamInputAction" data-cbactivity-toggle-target=".streamInputActionContainer" data-cbactivity-toggle-active-classes="btn-info" data-cbactivity-toggle-inactive-classes="btn-light border" data-cbactivity-toggle-icon="fa-before fa-smile-o" data-cbselect-dropdown-css-class="streamSelectOptions streamActionOptions"' . $actionTooltip, $stream, $actionId );
			$form['actions_message']		=	'<input type="text" name="actions[message]" value="' . htmlspecialchars( $action->getString( 'message' ) ) . '" class="form-control shadow-none border-0 h-100 w-100 streamInput streamInputActionMessage streamInputSelectTogglePlaceholder streamInputAutoComplete"' . ( $actionLimit ? ' maxlength="' . (int) $actionLimit . '"' : null ) . ( ! $actionId ? ' disabled="disabled"' : null ) . ' />';

			if ( $emoteOptions ) {
				$form['actions_emotes']		=	CBActivity::renderSelectList( $emoteOptions, 'actions[emote]', 'class="streamInputSelect streamInputEmote" data-cbselect-width="auto" data-cbselect-height="auto" data-cbselect-dropdown-css-class="streamEmoteOptions"' . ( ! $actionId ? ' disabled="disabled"' : null ), $stream, $action->getInt( 'emote' ) );
			}
		}

		$form['locations']					=	null;
		$form['locations_tooltip']			=	$findLocationTooltip;
		$form['locations_place']			=	null;
		$form['locations_addr']				=	null;

		if ( $locationOptions ) {
			$location						=	$row->params()->subTree( 'location' );
			$locationId						=	$location->getInt( 'id', 0 );

			$form['locations']				=	CBActivity::renderSelectList( $locationOptions, 'location[id]', 'class="btn btn-sm ' . ( $locationId ? 'btn-info' : 'btn-light border' ) . ' streamInputSelect streamInputSelectToggle streamInputLocation" data-cbactivity-toggle-target=".streamInputLocationContainer" data-cbactivity-toggle-active-classes="btn-info" data-cbactivity-toggle-inactive-classes="btn-light border" data-cbactivity-toggle-icon="fa-before fa-map-marker" data-cbselect-dropdown-css-class="streamSelectOptions streamLocationOptions"' . $locationTooltip, $stream, $locationId );
			$form['locations_place']		=	'<input type="text" name="location[place]" value="' . htmlspecialchars( $location->getString( 'place' ) ) . '" class="form-control shadow-none border-0 w-100 streamInput streamInputLocationPlace" placeholder="' . CBTxt::T( 'Where are you?' ) . '"' . ( $locationLimit ? ' maxlength="' . (int) $locationLimit . '"' : null ) . ( ! $locationId ? ' disabled="disabled"' : null ) . ' />';
			$form['locations_addr']			=	'<input type="text" name="location[address]" value="' . htmlspecialchars( $location->getString( 'address' ) ) . '" class="form-control shadow-none border-0 w-100 streamInput streamInputLocationAddress" placeholder="' . CBTxt::T( 'Have the address to share?' ) . '"' . ( $locationLimit ? ' maxlength="' . (int) $locationLimit . '"' : null ) . ( ! $locationId ? ' disabled="disabled"' : null ) . ' />';
		}

		$tagged								=	0;
		$form['tags']						=	null;
		$form['tags_tooltip']				=	$tagTooltip;

		if ( $showTags ) {
			$tagsStream						=	$row->tags( $stream );

			if ( $row->getInt( 'id', 0 ) ) {
				if ( $row->getRaw( '_tags' ) === false ) {
					$tagged					=	0;
				} elseif ( ( $row->getRaw( '_tags' ) !== null ) && ( $row->getRaw( '_tags' ) !== true ) ) {
					$tagged					=	$row->getInt( '_tags', 0 );
				} else {
					$tagged					=	CBActivity::prefetchAssets( 'tags', array(), $tagsStream );
				}

				if ( ! $tagged ) {
					$tagsStream->set( 'query', false );
				}
			} else {
				$tagsStream->set( 'query', false );
			}

			$form['tags']					=	$tagsStream->tags( 'edit' );
		}

		$form['themes']						=	null;
		$form['themes_classes']				=	null;
		$form['themes_styles']				=	null;

		if ( $themeOptions ) {
			$theme							=	$row->theme( $stream );

			if ( $theme ) {
				$form['themes_classes']		=	' ' . trim( 'streamMessageTheme ' . htmlspecialchars( $theme['class'] ) );
				$form['themes_styles']		=	( $theme['background'] ? ' style="background-image: url(' . htmlspecialchars( $theme['background'] ) . ')" data-cbactivity-active-theme-background="' . htmlspecialchars( $theme['background'] ) . '"' : null )
											.	( $theme['class'] ? ' data-cbactivity-active-theme-class="' . htmlspecialchars( $theme['class'] ) . '"' : null );
			}

			$themeId						=	$row->params()->getInt( 'theme', 0 );

			$form['themes']					=	CBActivity::renderSelectList( $themeOptions, 'theme', 'class="btn btn-sm ' . ( $themeId ? 'btn-info' : 'btn-light border' ) . ' streamInputSelect streamInputSelectToggle streamInputTheme" data-cbactivity-toggle-active-classes="btn-info" data-cbactivity-toggle-inactive-classes="btn-light border" data-cbactivity-toggle-icon="fa-before fa-paint-brush" data-cbselect-dropdown-css-class="streamThemeOptions"' . $themeTooltip, $stream, $themeId );
		}

		$links								=	0;
		$form['links']						=	array();
		$form['links_tooltip']				=	$linkTooltip;
		$form['links_limit']				=	$linkLimit;

		if ( $showLinks ) {
			$attachments					=	$row->attachments();
			$links							=	$attachments->count();

			if ( $links ) {
				foreach ( $attachments as $i => $attachment ) {
					/** @var ParamsInterface $attachment */
					if ( $attachment->getString( 'type', 'url' ) === 'custom' ) {
						continue;
					}

					$form['links'][]		=	'<input type="text" name="links[' . $i . '][url]" value="' . htmlspecialchars( $attachment->getString( 'url' ) ) . '" class="form-control shadow-none border-0 h-100 w-100 streamInput streamInputLinkURL" placeholder="' . htmlspecialchars( CBTxt::T( "What link would you like to share?" ) ) . '" />';
				}
			} else {
				$form['links'][]			=	'<input type="text" name="links[0][url]" class="form-control shadow-none border-0 h-100 w-100 streamInput streamInputLinkURL" placeholder="' . htmlspecialchars( CBTxt::T( "What link would you like to share?" ) ) . '" disabled="disabled" />';
			}
		}

		$form['system']						=	null;
		$form['system_tooltip']				=	$systemTooltip;
		$form['global']						=	null;
		$form['global_tooltip']				=	$globalTooltip;

		if ( $rowOwner && Application::MyUser()->isGlobalModerator() ) {
			if ( $stream->getBool( 'create_system', false ) ) {
				$form['system']				=	'<input type="checkbox" name="system" value="1" class="hidden"' . ( $row->getBool( 'system', false ) ? ' checked' : null ) . ' /><span class="fa fa-user-secret"></span>';
			}

			if ( $stream->getBool( 'global', true ) ) {
				$form['global']				=	'<input type="checkbox" name="global" value="1" class="hidden"' . ( $row->getBool( 'global', false ) ? ' checked' : null ) . ' /><span class="fa fa-bullhorn"></span>';
			}
		}

		if ( $row->getInt( 'id', 0 ) ) {
			if ( $this->getInput()->getBool( 'inline', false ) ) {
				$row->params()->set( 'overrides.inline', true );
			}

			ob_start();
			require CBActivity::getTemplate( $stream->getString( 'template' ), 'activity/edit', false );
			$html				=	ob_get_clean();

			CBActivity::ajaxResponse( $html );
		} else {
			require CBActivity::getTemplate( $stream->getString( 'template' ), 'activity/new' );
		}
	}

	/**
	 * Saves activity
	 *
	 * @param int       $id
	 * @param UserTable $viewer
	 * @param Activity  $stream
	 */
	private function saveActivity( $id, $viewer, $stream )
	{
		global $_PLUGINS;

		$canModerate						=	CBActivity::canModerate( $stream );

		$row								=	$stream->row( $id );

		if ( ! $row->getInt( 'id', 0 ) ) {
			if ( ! CBActivity::canCreate( 'activity', $stream, $viewer ) ) {
				CBActivity::ajaxResponse( CBTxt::T( 'You do not have permission to create activity.' ), 'error' );
			}
		} elseif ( ( ( $viewer->getInt( 'id', 0 ) !== $row->getInt( 'user_id', 0 ) ) && ( ! CBActivity::canModerate( $stream ) ) )
				   || ( $row->getBool( 'system', false ) && ( ! Application::MyUser()->isGlobalModerator() ) )
				   || ( $row->getBool( 'global', false ) && ( ! Application::MyUser()->isGlobalModerator() ) )
				   || ( ! CBActivity::findParamOverride( $row, 'edit', true ) )
		) {
			CBActivity::ajaxResponse( CBTxt::T( 'You do not have permission to edit this activity.' ), 'error' );
		}

		$messageLimit						=	( $canModerate ? 0 : $stream->getInt( 'message_limit', 400 ) );
		$showThemes							=	CBActivity::findParamOverride( $row, 'themes', true, $stream );
		$showActions						=	CBActivity::findParamOverride( $row, 'actions', true, $stream );
		$actionLimit						=	( $canModerate ? 0 : $stream->getInt( 'actions_message_limit', 100 ) );
		$showLocations						=	CBActivity::findParamOverride( $row, 'locations', true, $stream );
		$locationLimit						=	( $canModerate ? 0 : $stream->getInt( 'locations_address_limit', 200 ) );
		$showLinks							=	CBActivity::findParamOverride( $row, 'links', true, $stream );
		$linkLimit							=	( $canModerate ? 0 : $stream->getInt( 'links_link_limit', 5 ) );
		$showTags							=	CBActivity::findParamOverride( $row, 'tags', true, $stream );

		$row->set( 'user_id', $row->getInt( 'user_id', $viewer->getInt( 'id', 0 ) ) );
		$row->set( 'asset', $row->getString( 'asset', $stream->asset() ) );

		if ( Application::MyUser()->isGlobalModerator() ) {
			if ( $stream->getBool( 'create_system', true ) ) {
				if ( $this->getInput()->getBool( 'system', false ) ) {
					$systemUser				=	$this->params->getInt( 'system_user', 0 );

					if ( $systemUser ) {
						$row->set( 'user_id', $systemUser );
					}

					$row->set( 'system', 1 );
				} else {
					$row->set( 'system', 0 );
				}
			}

			if ( $stream->getBool( 'global', true ) ) {
				if ( $this->getInput()->getBool( 'global', false ) ) {
					$globalUser				=	$this->params->getInt( 'global_user', 0 );

					if ( $globalUser ) {
						$row->set( 'user_id', $globalUser );
					}

					$row->set( 'global', 1 );

					if ( $this->params->getBool( 'global_system', false ) ) {
						$row->set( 'system', 1 );
					}
				} else {
					$row->set( 'global', 0 );

					if ( $this->params->getBool( 'global_system', false ) ) {
						$row->set( 'system', 0 );
					}
				}
			}
		}

		$message							=	trim( $this->getInput()->getString( 'message', $row->getString( 'message', '' ) ) );

		// Remove duplicate spaces:
		$message							=	preg_replace( '/ {2,}/i', ' ', $message );
		// Remove duplicate tabs:
		$message							=	preg_replace( '/\t{2,}/i', "\t", $message );
		// Remove duplicate linebreaks:
		$message							=	preg_replace( '/((?:\r\n|\r|\n){2})(?:\r\n|\r|\n)*/i', '$1', $message );

		if ( $messageLimit && ( cbutf8_strlen( $message ) > $messageLimit ) ) {
			$message						=	cbutf8_substr( $message, 0, $messageLimit );
		}

		$row->set( 'message', $message );

		$new								=	( ! $row->getInt( 'id', 0 ) );

		if ( $showThemes ) {
			$themeId						=	$this->getInput()->getInt( 'theme', $row->params()->getInt( 'theme', 0 ) );

			if ( ! array_key_exists( $themeId, CBActivity::loadThemeOptions( true, $row->getInt( 'user_id', 0 ), $stream ) ) ) {
				$themeId					=	0;
			}

			$row->params()->set( 'theme', $themeId );
		}

		if ( $showActions ) {
			$existingAction					=	$row->params()->subTree( 'action' );
			$action							=	$this->getInput()->subTree( 'actions' );
			$actionId						=	$action->getInt( 'id', $existingAction->getInt( 'id', 0 ) );

			if ( ! array_key_exists( $actionId, CBActivity::loadActionOptions( true, $row->getInt( 'user_id', 0 ), $stream ) ) ) {
				$actionId					=	0;
			}

			$actionMessage					=	( $actionId ? trim( $action->getString( 'message', $existingAction->getString( 'message', '' ) ) ) : '' );
			$actionEmote					=	( $actionId ? $action->getInt( 'emote', $existingAction->getInt( 'emote', 0 ) ) : 0 );

			if ( ! array_key_exists( $actionEmote, CBActivity::loadEmoteOptions( false, true, $row->getInt( 'user_id', 0 ) ) ) ) {
				$actionEmote				=	0;
			}

			// Remove linebreaks:
			$actionMessage					=	str_replace( array( "\n", "\r\n" ), ' ', $actionMessage );
			// Remove duplicate spaces:
			$actionMessage					=	preg_replace( '/ {2,}/i', ' ', $actionMessage );
			// Remove duplicate tabs:
			$actionMessage					=	preg_replace( '/\t{2,}/i', "\t", $actionMessage );

			if ( $actionLimit && ( cbutf8_strlen( $actionMessage ) > $actionLimit ) ) {
				$actionMessage				=	cbutf8_substr( $actionMessage, 0, $actionLimit );
			}

			$actionId						=	( $actionMessage ? $actionId : 0 );

			$newAction						=	array(	'id'		=>	$actionId,
														'message'	=>	( $actionId ? $actionMessage : '' ),
														'emote'		=>	( $actionId ? $actionEmote : 0 )
													);

			if ( $actionId ) {
				$row->params()->set( 'overrides.message', false );
			}

			$row->params()->set( 'action', $newAction );
		}

		if ( $showLocations ) {
			$existingLocation				=	$row->params()->subTree( 'location' );
			$location						=	$this->getInput()->subTree( 'location' );
			$locationId						=	$location->getInt( 'id', $existingLocation->getInt( 'id', 0 ) );

			if ( ! array_key_exists( $locationId, CBActivity::loadLocationOptions( true, $row->getInt( 'user_id', 0 ), $stream ) ) ) {
				$locationId					=	0;
			}

			$locationPlace					=	( $locationId ? trim( $location->getString( 'place', $existingLocation->getString( 'place', '' ) ) ) : '' );
			$locationAddress				=	( $locationId ? trim( $location->getString( 'address', $existingLocation->getString( 'address', '' ) ) ) : '' );

			if ( $locationLimit && ( cbutf8_strlen( $locationPlace ) > $locationLimit ) ) {
				$locationPlace				=	cbutf8_substr( $locationPlace, 0, $locationLimit );
			}

			if ( $locationLimit && ( cbutf8_strlen( $locationAddress ) > $locationLimit ) ) {
				$locationAddress			=	cbutf8_substr( $locationAddress, 0, $locationLimit );
			}

			$locationId						=	( $locationPlace ? $locationId : 0 );

			$newLocation					=	array(	'id'		=>	$locationId,
														'place'		=>	( $locationId ? $locationPlace : '' ),
														'address'	=>	( $locationId ? $locationAddress : '' )
													);

			if ( $locationId ) {
				$row->params()->set( 'overrides.message', false );
			}

			$row->params()->set( 'location', $newLocation );
		}

		if ( $showLinks ) {
			$newUrls						=	array();
			$newLinks						=	array();
			$urls							=	$stream->parser( $message )->urls();

			/** @var ParamsInterface $links */
			$links							=	$this->getInput()->subTree( 'links' );

			if ( $stream->getBool( 'links_embedded', false ) ) {
				$index						=	( $links->count() - 1 );

				foreach ( $urls as $url ) {
					foreach ( $links as $link ) {
						/** @var ParamsInterface $link */
						if ( trim( $link->getString( 'url', '' ) ) === $url ) {
							continue 2;
						}
					}

					$index++;

					$links->set( $index, array( 'url' => $url, 'embedded' => true ) );
				}
			}

			foreach ( $links as $i => $link ) {
				if ( $linkLimit && ( ( $i + 1 ) > $linkLimit ) ) {
					break;
				}

				$linkUrl					=	trim( $link->getString( 'url', '' ) );

				if ( ( ! $linkUrl ) || in_array( $linkUrl, $newUrls, true ) ) {
					continue;
				}

				$linkEmbedded				=	$link->getBool( 'embedded', false );

				if ( $linkEmbedded && ( ! in_array( $linkUrl, $urls, true ) ) ) {
					continue;
				}

				if ( $link->getBool( 'parsed', false ) ) {
					foreach ( $row->params()->subTree( 'links' ) as $existingLink ) {
						/** @var ParamsInterface $existingLink */
						if ( trim( $existingLink->getString( 'url', '' ) ) === $linkUrl ) {
							$existingLink->set( 'title', trim( $link->getString( 'title', $existingLink->getString( 'title', '' ) ) ) );
							$existingLink->set( 'description', trim( $link->getString( 'description', $existingLink->getString( 'description', '' ) ) ) );
							$existingLink->set( 'thumbnail', $link->getBool( 'thumbnail', true ) );

							if ( in_array( $existingLink->getString( 'type' ), array( 'url', 'video', 'audio' ), true ) ) {
								$selected	=	$link->getInt( 'selected', 0 );

								/** @var ParamsInterface $thumbnail */
								$thumbnail	=	$existingLink->subTree( 'thumbnails' )->subTree( $selected );

								if ( $thumbnail->getString( 'url' ) ) {
									$existingLink->set( 'media', $thumbnail->asArray() );
									$existingLink->set( 'selected', $selected );
								}
							}

							$newLinks[]		=	$existingLink->asArray();
							break;
						}
					}

					continue;
				}

				$attachment					=	$stream->parser()->attachment( $linkUrl );

				if ( ! $attachment ) {
					if ( ! $linkEmbedded ) {
						CBActivity::ajaxResponse( CBTxt::T( 'Please provide a valid link.' ), 'warning' );
					}

					continue;
				}

				$newLink					=	$this->attachmentToLink( $link, $attachment );

				if ( ! $newLink ) {
					continue;
				}

				$newLinks[]					=	$newLink;
				$newUrls[]					=	$linkUrl;
			}

			if ( ! $new ) {
				foreach ( $row->params()->subTree( 'links' ) as $link ) {
					/** @var ParamsInterface $link */
					if ( $link->getString( 'type', 'url' ) !== 'custom' ) {
						continue;
					}

					$newLinks[]				=	$link->asArray();
				}
			}

			if ( $newLinks ) {
				$row->params()->set( 'overrides.message', false );
			}

			$row->params()->set( 'links', $newLinks );
		}

		$old								=	new ActivityTable();
		$source								=	$row->source();

		if ( ! $new ) {
			$old->load( $row->getInt( 'id', 0 ) );

			if ( Application::Date( $row->getString( 'date' ), 'UTC' )->modify( '+5 MINUTES' )->getTimestamp() < Application::Date( 'now', 'UTC' )->getTimestamp() ) {
				$row->params()->set( 'modified', Application::Database()->getUtcDateTime() );
			}

			$_PLUGINS->trigger( 'activity_onBeforeUpdateStreamActivity', array( $stream, $source, &$row, $old ) );
		} else {
			$_PLUGINS->trigger( 'activity_onBeforeCreateStreamActivity', array( $stream, $source, &$row ) );
		}

		if ( ( $row->getString( 'asset' ) === $stream->asset() ) && ( $message === '' ) && CBActivity::findParamOverride( $row, 'message', true ) ) {
			CBActivity::ajaxResponse( CBTxt::T( 'Please provide a message.' ), 'warning' );
		}

		$newParams							=	clone $row->params();

		$newParams->unsetEntry( 'overrides' );

		$row->set( 'params', $newParams->asJson() );

		if ( $row->getError() || ( ! $row->check() ) ) {
			CBActivity::ajaxResponse( CBTxt::T( 'ACTIVITY_EDIT_FAILED_TO_SAVE', 'Activity failed to save! Error: [error]', array( '[error]' => $row->getError() ) ), 'error' );
		}

		if ( $row->getError() || ( ! $row->store() ) ) {
			CBActivity::ajaxResponse( CBTxt::T( 'ACTIVITY_EDIT_FAILED_TO_SAVE', 'Activity failed to save! Error: [error]', array( '[error]' => $row->getError() ) ), 'error' );
		}

		if ( $showTags ) {
			$tagsStream						=	$row->tags( $stream );

			$this->saveTags( $viewer, $tagsStream );

			$row->set( '_tags', null );
		}

		if ( ! $new ) {
			$_PLUGINS->trigger( 'activity_onAfterUpdateStreamActivity', array( $stream, $source, $row, $old ) );
		} else {
			$_PLUGINS->trigger( 'activity_onAfterCreateStreamActivity', array( $stream, $source, $row ) );
		}

		$pinnedTooltip						=	cbTooltip( null, CBTxt::T( 'Pinned' ), null, 'auto', null, null, null, 'data-hascbtooltip="true" data-cbtooltip-position-my="bottom center" data-cbtooltip-position-at="top center" data-cbtooltip-classes="qtip-simple"' );
		$globalTooltip						=	cbTooltip( null, CBTxt::T( 'Announcement' ), null, 'auto', null, null, null, 'data-hascbtooltip="true" data-cbtooltip-position-my="bottom center" data-cbtooltip-position-at="top center" data-cbtooltip-classes="qtip-simple"' );
		$reportedTooltip					=	cbTooltip( null, CBTxt::T( 'Controversial' ), null, 'auto', null, null, null, 'data-hascbtooltip="true" data-cbtooltip-position-my="bottom center" data-cbtooltip-position-at="top center" data-cbtooltip-classes="qtip-simple"' );
		$output								=	'save';

		if ( $this->getInput()->getBool( 'inline', false ) ) {
			$row->params()->set( 'overrides.inline', true );
		}

		ob_start();
		require CBActivity::getTemplate( $stream->getString( 'template' ), 'activity/container', false );
		$html								=	ob_get_clean();

		CBActivity::ajaxResponse( $html );
	}

	/**
	 * Parses an activity or comment attachment to params link
	 *
	 * @param ParamsInterface $link
	 * @param ParamsInterface $attachment
	 * @return array|null
	 */
	private function attachmentToLink( $link, $attachment )
	{
		$type					=	$attachment->getString( 'type', 'url' );

		switch ( $type ) {
			case 'custom':
				return null;
			case 'video':
				$mediaType		=	'video';
				$media			=	$attachment->subTree( 'media' )->subTree( 'video' )->subTree( 0 );
				break;
			case 'audio':
				$mediaType		=	'audio';
				$media			=	$attachment->subTree( 'media' )->subTree( 'audio' )->subTree( 0 );
				break;
			case 'file':
				$mediaType		=	'file';
				$media			=	$attachment->subTree( 'media' )->subTree( 'file' )->subTree( 0 );
				break;
			case 'image':
			case 'url':
			default:
				$mediaType		=	'image';
				$media			=	$attachment->subTree( 'media' )->subTree( 'image' )->subTree( 0 );
				break;
		}

		$thumbnails				=	array();

		if ( in_array( $type, array( 'url', 'video', 'audio' ), true ) ) {
			$images				=	array(	$attachment->subTree( 'media' )->subTree( 'image' )->subTree( 0 ),
											$attachment->subTree( 'media' )->subTree( 'image' )->subTree( 1 ),
											$attachment->subTree( 'media' )->subTree( 'image' )->subTree( 2 ),
											$attachment->subTree( 'media' )->subTree( 'image' )->subTree( 3 ),
											$attachment->subTree( 'media' )->subTree( 'image' )->subTree( 4 )
										);

			$thumbnails			=	array();

			foreach ( $images as $image ) {
				$imageUrl		=	$image->getString( 'url', '' );

				if ( ! $imageUrl ) {
					continue;
				}

				$thumbnails[]	=	array(	'type'		=>	$image->getString( 'type', $mediaType ),
											'url'		=>	$imageUrl,
											'filename'	=>	$image->getString( 'filename', '' ),
											'mimetype'	=>	$image->getString( 'mimetype', '' ),
											'extension'	=>	$image->getString( 'extension', '' ),
											'filesize'	=>	$image->getInt( 'filesize', 0 ),
											'custom'	=>	'',
											'internal'	=>	$image->getBool( 'internal', false )
										);
			}

			if ( $thumbnails <= 1 ) {
				$thumbnails		=	array();
			}
		}

		return array(	'type'			=>	$type,
						'url'			=>	trim( $link->getString( 'url', '' ) ),
						'title'			=>	trim( $link->getString( 'title', $attachment->subTree( 'title' )->getString( 0, '' ) ) ),
						'description'	=>	trim( $link->getString( 'description', $attachment->subTree( 'description' )->getString( 0, '' ) ) ),
						'media'			=>	array(	'type'		=>	$media->getString( 'type', $mediaType ),
													'url'		=>	$media->getString( 'url', '' ),
													'filename'	=>	$media->getString( 'filename', '' ),
													'mimetype'	=>	$media->getString( 'mimetype', '' ),
													'extension'	=>	$media->getString( 'extension', '' ),
													'filesize'	=>	$media->getInt( 'filesize', 0 ),
													'custom'	=>	'',
													'internal'	=>	$media->getBool( 'internal', false )
												),
						'thumbnails'	=>	$thumbnails,
						'selected'		=>	$link->getInt( 'selected', 0 ),
						'thumbnail'		=>	$link->getBool( 'thumbnail', true ),
						'internal'		=>	$attachment->getBool( 'internal', false ),
						'date'			=>	$attachment->getString( 'date' ),
						'embedded'		=>	$link->getBool( 'embedded', false )
					);
	}

	/**
	 * Deletes activity
	 *
	 * @param int       $id
	 * @param UserTable $viewer
	 * @param Activity  $stream
	 */
	private function deleteActivity( $id, $viewer, $stream )
	{
		global $_PLUGINS;

		$row		=	$stream->row( $id );

		if ( ( ! $row->getInt( 'id', 0 ) )
			 || ( ( $viewer->getInt( 'id', 0 ) !== $row->getInt( 'user_id', 0 ) ) && ( ! CBActivity::canModerate( $stream ) ) )
			 || ( $row->getBool( 'system', false ) && ( ! Application::MyUser()->isGlobalModerator() ) )
			 || ( $row->getBool( 'global', false ) && ( ! Application::MyUser()->isGlobalModerator() ) )
		) {
			CBActivity::ajaxResponse( CBTxt::T( 'You do not have permission to delete this activity.' ), 'error' );
		}

		$source		=	$row->source();

		$_PLUGINS->trigger( 'activity_onBeforeDeleteStreamActivity', array( $stream, $source, &$row ) );

		if ( $row->getError() || ( ! $row->canDelete() ) ) {
			CBActivity::ajaxResponse( CBTxt::T( 'ACTIVITY_FAILED_TO_DELETE', 'Activity failed to delete! Error: [error]', array( '[error]' => $row->getError() ) ), 'error' );
		}

		if ( $row->getError() || ( ! $row->delete() ) ) {
			CBActivity::ajaxResponse( CBTxt::T( 'ACTIVITY_FAILED_TO_DELETE', 'Activity failed to delete! Error: [error]', array( '[error]' => $row->getError() ) ), 'error' );
		}

		$_PLUGINS->trigger( 'activity_onAfterDeleteStreamActivity', array( $stream, $source, $row ) );

		CBActivity::ajaxResponse( CBTxt::T( 'This activity has been deleted.' ), 'notice' );
	}

	/**
	 * Hides activity
	 *
	 * @param int       $id
	 * @param UserTable $viewer
	 * @param Activity  $stream
	 */
	private function hideActivity( $id, $viewer, $stream )
	{
		global $_CB_framework, $_PLUGINS;

		$type				=	$this->getInput()->getString( 'type', 'activity' );
		$activity			=	$stream->row( $id );

		if ( ( ! $activity->getInt( 'id', 0 ) )
			 || ( ! $viewer->getInt( 'id', 0 ) )
			 || ( $viewer->getInt( 'id', 0 ) === $activity->getInt( 'user_id', 0 ) )
			 || ( ! in_array( $type, array( 'activity', 'asset', 'user' ), true ) ) )
		{
			CBActivity::ajaxResponse( CBTxt::T( 'You do not have permission to hide this activity.' ), 'error' );
		}

		$row				=	new HiddenTable();

		switch ( $type ) {
			case 'user':
				if ( $activity->getBool( 'system', false ) ) {
					CBActivity::ajaxResponse( CBTxt::T( 'You do not have permission to hide this activity.' ), 'error' );
				}

				$hideType	=	'activity.' . $type;
				$hideItem	=	$activity->getInt( 'user_id', 0 );
				break;
			case 'asset':
				$hideType	=	'activity.' . $type;
				$hideItem	=	$activity->getString( 'asset' );
				break;
			case 'activity':
			default:
				$hideType	=	'activity';
				$hideItem	=	$activity->getInt( 'id', 0 );
				break;
		}

		$row->load( array( 'user_id' => $viewer->getInt( 'id', 0 ), 'type' => $hideType, ( $type === 'asset' ? 'asset' : 'object' ) => $hideItem ) );

		if ( $row->getInt( 'id', 0 ) ) {
			CBActivity::ajaxResponse( CBTxt::T( 'You have already hidden this activity.' ), 'error' );
		}

		$row->set( 'user_id', $viewer->getInt( 'id', 0 ) );
		$row->set( 'type', $hideType );
		$row->set( ( $type === 'asset' ? 'asset' : 'object' ), $hideItem );

		$source				=	$activity->source();

		$_PLUGINS->trigger( 'activity_onBeforeHideStreamActivity', array( $stream, $source, &$row, $activity ) );

		if ( $row->getError() || ( ! $row->check() ) ) {
			CBActivity::ajaxResponse( CBTxt::T( 'ACTIVITY_HIDE_FAILED_TO_SAVE', 'Activity failed to hide! Error: [error]', array( '[error]' => $row->getError() ) ), 'error' );
		}

		if ( $row->getError() || ( ! $row->store() ) ) {
			CBActivity::ajaxResponse( CBTxt::T( 'ACTIVITY_HIDE_FAILED_TO_SAVE', 'Activity failed to hide! Error: [error]', array( '[error]' => $row->getError() ) ), 'error' );
		}

		$_PLUGINS->trigger( 'activity_onAfterHideStreamActivity', array( $stream, $source, $row, $activity ) );

		$unhide				=	'<a href="javascript: void(0);" data-cbactivity-action-url="' . $_CB_framework->pluginClassUrl( $this->element, true, array( 'action' => 'activity', 'func' => 'unhide', 'type' => $type, 'id' => $activity->getInt( 'id', 0 ), 'stream' => $stream->id() ), 'raw', 0, true ) . '" class="activityContainerUnhide streamItemAction streamItemActionResponsesRevert">' . CBTxt::T( 'Unhide' ) . '</a>';

		CBActivity::ajaxResponse( CBTxt::T( 'ACTIVITY_HIDDEN_UNHIDE', 'This activity has been hidden. [unhide]', array( '[unhide]' => $unhide ) ), 'notice' );
	}

	/**
	 * Deletes activity hide
	 *
	 * @param int       $id
	 * @param UserTable $viewer
	 * @param Activity  $stream
	 */
	private function unhideActivity( $id, $viewer, $stream )
	{
		global $_PLUGINS;

		$type				=	$this->getInput()->getString( 'type', 'activity' );
		$activity			=	$stream->row( $id );

		if ( ( ! $activity->getInt( 'id', 0 ) )
			 || ( ! $viewer->getInt( 'id', 0 ) )
			 || ( ! in_array( $type, array( 'activity', 'asset', 'user' ), true ) ) )
		{
			CBActivity::ajaxResponse( CBTxt::T( 'You do not have permission to unhide this activity.' ), 'error' );
		}

		$row				=	new HiddenTable();

		switch ( $type ) {
			case 'user':
				$row->load( array( 'user_id' => $viewer->getInt( 'id', 0 ), 'type' => 'activity.user', 'object' => $activity->getInt( 'user_id', 0 ) ) );
				break;
			case 'asset':
				$row->load( array( 'user_id' => $viewer->getInt( 'id', 0 ), 'type' => 'activity.asset', 'asset' => $activity->getString( 'asset' ) ) );
				break;
			case 'activity':
			default:
				$row->load( array( 'user_id' => $viewer->getInt( 'id', 0 ), 'type' => 'activity', 'object' => $activity->getInt( 'id', 0 ) ) );
				break;
		}

		if ( ! $row->getInt( 'id', 0 ) ) {
			CBActivity::ajaxResponse( CBTxt::T( 'You have not hidden this activity.' ), 'error' );
		}

		$source				=	$activity->source();

		$_PLUGINS->trigger( 'activity_onBeforeUnhideStreamActivity', array( $stream, $source, &$row, $activity ) );

		if ( $row->getError() || ( ! $row->canDelete() ) ) {
			CBActivity::ajaxResponse( CBTxt::T( 'ACTIVITY_HIDE_FAILED_TO_DELETE', 'Activity failed to unhide! Error: [error]', array( '[error]' => $row->getError() ) ), 'error' );
		}

		if ( $row->getError() || ( ! $row->delete() ) ) {
			CBActivity::ajaxResponse( CBTxt::T( 'ACTIVITY_HIDE_FAILED_TO_DELETE', 'Activity failed to unhide! Error: [error]', array( '[error]' => $row->getError() ) ), 'error' );
		}

		$_PLUGINS->trigger( 'activity_onAfterUnhideStreamActivity', array( $stream, $source, $row, $activity ) );

		if ( $stream->getBool( 'hidden', false ) ) {
			CBActivity::ajaxResponse( CBTxt::T( 'This activity has been unhidden.' ), 'notice' );
		}

		CBActivity::ajaxResponse();
	}

	/**
	 * Deletes activity follow
	 *
	 * @param int       $id
	 * @param UserTable $viewer
	 * @param Activity  $stream
	 */
	private function unfollowActivity( $id, $viewer, $stream )
	{
		global $_PLUGINS;

		$row				=	$stream->row( $id );

		if ( ( ! $row->getInt( 'id', 0 ) ) || ( ! $viewer->getInt( 'id', 0 ) ) ) {
			CBActivity::ajaxResponse( CBTxt::T( 'You do not have permission to unfollow this activity.' ), 'error' );
		}

		$follow				=	new FollowTable();

		$follow->load( array( 'user_id' => $viewer->getInt( 'id', 0 ), 'asset' => $row->getString( 'asset' ) ) );

		if ( ! $follow->getInt( 'id', 0 ) ) {
			CBActivity::ajaxResponse( CBTxt::T( 'You have not followed this activity.' ), 'error' );
		}

		$source				=	$row->source();

		$_PLUGINS->trigger( 'activity_onBeforeUnfollowStreamActivity', array( $stream, $source, &$follow, $row ) );

		if ( $follow->getError() || ( ! $follow->canDelete() ) ) {
			CBActivity::ajaxResponse( CBTxt::T( 'ACTIVITY_FOLLOW_FAILED_TO_DELETE', 'Activity failed to unfollow! Error: [error]', array( '[error]' => $follow->getError() ) ), 'error' );
		}

		if ( $follow->getError() || ( ! $follow->delete() ) ) {
			CBActivity::ajaxResponse( CBTxt::T( 'ACTIVITY_FOLLOW_FAILED_TO_DELETE', 'Activity failed to unfollow! Error: [error]', array( '[error]' => $follow->getError() ) ), 'error' );
		}

		$_PLUGINS->trigger( 'activity_onAfterUnfollowStreamActivity', array( $stream, $source, $follow, $row ) );

		$row->params()->set( 'overrides.following', false );

		$pinnedTooltip		=	cbTooltip( null, CBTxt::T( 'Pinned' ), null, 'auto', null, null, null, 'data-hascbtooltip="true" data-cbtooltip-position-my="bottom center" data-cbtooltip-position-at="top center" data-cbtooltip-classes="qtip-simple"' );
		$globalTooltip		=	cbTooltip( null, CBTxt::T( 'Announcement' ), null, 'auto', null, null, null, 'data-hascbtooltip="true" data-cbtooltip-position-my="bottom center" data-cbtooltip-position-at="top center" data-cbtooltip-classes="qtip-simple"' );
		$reportedTooltip	=	cbTooltip( null, CBTxt::T( 'Controversial' ), null, 'auto', null, null, null, 'data-hascbtooltip="true" data-cbtooltip-position-my="bottom center" data-cbtooltip-position-at="top center" data-cbtooltip-classes="qtip-simple"' );
		$output				=	'save';

		if ( $this->getInput()->getBool( 'inline', false ) ) {
			$row->params()->set( 'overrides.inline', true );
		}

		ob_start();
		require CBActivity::getTemplate( $stream->getString( 'template' ), 'activity/container', false );
		$html				=	ob_get_clean();

		CBActivity::ajaxResponse( $html, 'html', 'replace', 'container' );
	}

	/**
	 * Reports an activity entry
	 *
	 * @param int       $id
	 * @param UserTable $viewer
	 * @param Activity  $stream
	 */
	private function reportActivity( $id, $viewer, $stream )
	{
		global $_CB_framework, $_PLUGINS;

		$activity		=	$stream->row( $id );

		if ( ( ! $this->params->getBool( 'reporting', true ) )
			 || ( ! $activity->getInt( 'id', 0 ) )
			 || ( ! $viewer->getInt( 'id', 0 ) )
			 || ( $viewer->getInt( 'id', 0 ) === $activity->getInt( 'user_id', 0 ) )
			 || CBActivity::canModerate( $stream, $activity->getInt( 'user_id', 0 ) ) )
		{
			CBActivity::ajaxResponse( CBTxt::T( 'You do not have permission to report this activity.' ), 'error' );
		}

		$row			=	new HiddenTable();

		$row->load( array( 'user_id' => $viewer->getInt( 'id', 0 ), 'type' => 'activity', 'object' => $activity->getInt( 'id', 0 ) ) );

		$source			=	$activity->source();

		$activity->params()->set( 'reports', ( $activity->params()->getInt( 'reports', 0 ) + 1 ) );
		$activity->params()->set( 'reported', Application::Database()->getUtcDateTime() );

		$reportLimit	=	$this->params->getInt( 'reporting_limit', 10 );

		if ( $reportLimit && ( $activity->params()->getInt( 'reports', 0 ) >= $reportLimit ) ) {
			$activity->set( 'published', 0 );
		}

		if ( ! $row->getInt( 'id', 0 ) ) {
			$row->set( 'user_id', $viewer->getInt( 'id', 0 ) );
			$row->set( 'type', 'activity' );
			$row->set( 'object', $activity->getInt( 'id', 0 ) );

			$_PLUGINS->trigger( 'activity_onBeforeHideStreamActivity', array( $stream, $source, &$row, $activity ) );

			if ( $row->getError() || ( ! $row->check() ) ) {
				CBActivity::ajaxResponse( CBTxt::T( 'ACTIVITY_HIDE_FAILED_TO_SAVE', 'Activity failed to hide! Error: [error]', array( '[error]' => $row->getError() ) ), 'error' );
			}

			if ( $row->getError() || ( ! $row->store() ) ) {
				CBActivity::ajaxResponse( CBTxt::T( 'ACTIVITY_HIDE_FAILED_TO_SAVE', 'Activity failed to hide! Error: [error]', array( '[error]' => $row->getError() ) ), 'error' );
			}

			$_PLUGINS->trigger( 'activity_onAfterHideStreamActivity', array( $stream, $source, $row, $activity ) );
		}

		$_PLUGINS->trigger( 'activity_onBeforeReportStreamActivity', array( $stream, $source, &$activity, $row ) );

		$newParams		=	clone $activity->params();

		$newParams->unsetEntry( 'overrides' );

		$activity->set( 'params', $newParams->asJson() );

		if ( $activity->getError() || ( ! $activity->check() ) ) {
			CBActivity::ajaxResponse( CBTxt::T( 'ACTIVITY_REPORT_FAILED_TO_SAVE', 'Activity failed to report! Error: [error]', array( '[error]' => $activity->getError() ) ), 'error' );
		}

		if ( $activity->getError() || ( ! $activity->store() ) ) {
			CBActivity::ajaxResponse( CBTxt::T( 'ACTIVITY_REPORT_FAILED_TO_SAVE', 'Activity failed to report! Error: [error]', array( '[error]' => $activity->getError() ) ), 'error' );
		}

		$_PLUGINS->trigger( 'activity_onAfterReportStreamActivity', array( &$stream, $source, $activity, $row ) );

		if ( $this->params->getBool( 'reporting_notify', true ) ) {
			$cbUser				=	CBuser::getInstance( $viewer->getInt( 'id', 0 ), false );

			$extraStrings		=	array(	'activity_id'		=>	$activity->getInt( 'id', 0 ),
											'activity_title'	=>	$activity->getString( 'title' ),
											'activity_message'	=>	$activity->getString( 'message' ),
											'activity_url'		=>	$_CB_framework->pluginClassUrl( $this->element, true, array( 'action' => 'activity', 'func' => 'show', 'id' => $activity->getInt( 'id', 0 ), 'stream' => $stream->id() ) ),
											'user_url'			=>	$_CB_framework->viewUrl( 'userprofile', true, array( 'user' => $viewer->getInt( 'user_id', 0 ) ) )
										);

			$subject			=	$cbUser->replaceUserVars( CBTxt::T( 'Activity - Reported!' ), false, true, $extraStrings, false );

			if ( ! $activity->getInt( 'published', 1 ) ) {
				$message		=	$cbUser->replaceUserVars( CBTxt::T( '<a href="[user_url]">[formatname]</a> reported activity <a href="[activity_url]">[cb:if activity_message!=""][activity_message][cb:else]#[activity_id][/cb:else][/cb:if]</a> as controversial and has now been unpublished!' ), false, true, $extraStrings, false );
			} else {
				$message		=	$cbUser->replaceUserVars( CBTxt::T( '<a href="[user_url]">[formatname]</a> reported activity <a href="[activity_url]">[cb:if activity_message!=""][activity_message][cb:else]#[activity_id][/cb:else][/cb:if]</a> as controversial!' ), false, true, $extraStrings, false );
			}

			$notifications		=	new cbNotification();
			$recipients			=	$stream->getRaw( 'moderators', array() );

			if ( $recipients ) {
				cbToArrayOfInt( $recipients );

				foreach ( $recipients as $recipient ) {
					$notifications->sendFromSystem( $recipient, $subject, $message, false, 1 );
				}
			} else {
				$notifications->sendToModerators( $subject, $message, false, 1 );
			}
		}

		CBActivity::ajaxResponse( CBTxt::T( 'This activity has been reported and hidden.' ), 'notice' );
	}

	/**
	 * Pins activity to the top of a stream
	 *
	 * @param int       $id
	 * @param UserTable $viewer
	 * @param Activity  $stream
	 */
	private function pinActivity( $id, $viewer, $stream )
	{
		global $_PLUGINS;

		$row				=	$stream->row( $id );

		if ( ( ! $row->getInt( 'id', 0 ) ) || ( ! Application::MyUser()->isGlobalModerator() ) ) {
			CBActivity::ajaxResponse( CBTxt::T( 'You do not have permission to pin this activity.' ), 'error' );
		}

		if ( $row->getBool( 'pinned', false ) ) {
			CBActivity::ajaxResponse( CBTxt::T( 'You have already pinned this activity.' ), 'error' );
		}

		$row->set( 'pinned', 1 );

		$source				=	$row->source();

		$_PLUGINS->trigger( 'activity_onBeforePinStreamActivity', array( $stream, $source, &$row ) );

		if ( $row->getError() || ( ! $row->check() ) ) {
			CBActivity::ajaxResponse( CBTxt::T( 'ACTIVITY_PIN_FAILED_TO_SAVE', 'Activity failed to pin! Error: [error]', array( '[error]' => $row->getError() ) ), 'error' );
		}

		if ( $row->getError() || ( ! $row->store() ) ) {
			CBActivity::ajaxResponse( CBTxt::T( 'ACTIVITY_PIN_FAILED_TO_SAVE', 'Activity failed to pin! Error: [error]', array( '[error]' => $row->getError() ) ), 'error' );
		}

		$_PLUGINS->trigger( 'activity_onAfterPinStreamActivity', array( $stream, $source, $row ) );

		$pinnedTooltip		=	cbTooltip( null, CBTxt::T( 'Pinned' ), null, 'auto', null, null, null, 'data-hascbtooltip="true" data-cbtooltip-position-my="bottom center" data-cbtooltip-position-at="top center" data-cbtooltip-classes="qtip-simple"' );
		$globalTooltip		=	cbTooltip( null, CBTxt::T( 'Announcement' ), null, 'auto', null, null, null, 'data-hascbtooltip="true" data-cbtooltip-position-my="bottom center" data-cbtooltip-position-at="top center" data-cbtooltip-classes="qtip-simple"' );
		$reportedTooltip	=	cbTooltip( null, CBTxt::T( 'Controversial' ), null, 'auto', null, null, null, 'data-hascbtooltip="true" data-cbtooltip-position-my="bottom center" data-cbtooltip-position-at="top center" data-cbtooltip-classes="qtip-simple"' );
		$output				=	'save';

		if ( $this->getInput()->getBool( 'inline', false ) ) {
			$row->params()->set( 'overrides.inline', true );
		}

		ob_start();
		require CBActivity::getTemplate( $stream->getString( 'template' ), 'activity/container', false );
		$html				=	ob_get_clean();

		CBActivity::ajaxResponse( $html, 'html', 'replace', 'container' );
	}

	/**
	 * Unpins activity from the top of a stream
	 *
	 * @param int       $id
	 * @param UserTable $viewer
	 * @param Activity  $stream
	 */
	private function unpinActivity( $id, $viewer, $stream )
	{
		global $_PLUGINS;

		$row				=	$stream->row( $id );

		if ( ( ! $row->getInt( 'id', 0 ) ) || ( ! Application::MyUser()->isGlobalModerator() ) ) {
			CBActivity::ajaxResponse( CBTxt::T( 'You do not have permission to unpin this activity.' ), 'error' );
		}

		if ( ! $row->getBool( 'pinned', false ) ) {
			CBActivity::ajaxResponse( CBTxt::T( 'You have not pinned this activity.' ), 'error' );
		}

		$row->set( 'pinned', 0 );

		$source				=	$row->source();

		$_PLUGINS->trigger( 'activity_onBeforeUnpinStreamActivity', array( $stream, $source, &$row ) );

		if ( $row->getError() || ( ! $row->check() ) ) {
			CBActivity::ajaxResponse( CBTxt::T( 'ACTIVITY_UNPIN_FAILED_TO_SAVE', 'Activity failed to unpin! Error: [error]', array( '[error]' => $row->getError() ) ), 'error' );
		}

		if ( $row->getError() || ( ! $row->store() ) ) {
			CBActivity::ajaxResponse( CBTxt::T( 'ACTIVITY_UNPIN_FAILED_TO_SAVE', 'Activity failed to unpin! Error: [error]', array( '[error]' => $row->getError() ) ), 'error' );
		}

		$_PLUGINS->trigger( 'activity_onAfterUnpinStreamActivity', array( $stream, $source, $row ) );

		$pinnedTooltip		=	cbTooltip( null, CBTxt::T( 'Pinned' ), null, 'auto', null, null, null, 'data-hascbtooltip="true" data-cbtooltip-position-my="bottom center" data-cbtooltip-position-at="top center" data-cbtooltip-classes="qtip-simple"' );
		$globalTooltip		=	cbTooltip( null, CBTxt::T( 'Announcement' ), null, 'auto', null, null, null, 'data-hascbtooltip="true" data-cbtooltip-position-my="bottom center" data-cbtooltip-position-at="top center" data-cbtooltip-classes="qtip-simple"' );
		$reportedTooltip	=	cbTooltip( null, CBTxt::T( 'Controversial' ), null, 'auto', null, null, null, 'data-hascbtooltip="true" data-cbtooltip-position-my="bottom center" data-cbtooltip-position-at="top center" data-cbtooltip-classes="qtip-simple"' );
		$output				=	'save';

		if ( $this->getInput()->getBool( 'inline', false ) ) {
			$row->params()->set( 'overrides.inline', true );
		}

		ob_start();
		require CBActivity::getTemplate( $stream->getString( 'template' ), 'activity/container', false );
		$html				=	ob_get_clean();

		CBActivity::ajaxResponse( $html, 'html', 'replace', 'container' );
	}

	/**
	 * Displays activity as a button
	 *
	 * @param UserTable $viewer
	 * @param Activity  $stream
	 * @param string    $output
	 */
	private function showActivityButton( $viewer, $stream, $output = null )
	{
		$total				=	0;

		if ( $stream->getBool( 'query', true ) ) {
			if ( $stream->has( 'query_count' ) ) {
				$total		=	$stream->getInt( 'query_count', 0 );
			} else {
				$total		=	CBActivity::prefetchAssets( 'activity', array(), $stream );
			}
		}

		CBActivity::bindStream();

		$layout				=	$stream->getString( 'layout', 'stream' );
		$autoUpdate			=	$stream->getBool( 'auto_update', false );

		require CBActivity::getTemplate( $stream->getString( 'template' ), 'activity/button' );
	}

	/**
	 * Displays notifications stream
	 *
	 * @param int           $id
	 * @param UserTable     $viewer
	 * @param Notifications $stream
	 * @param string        $output
	 */
	private function showNotifications( $id, $viewer, $stream, $output = null )
	{
		global $_CB_framework, $_PLUGINS;

		$notificationPrefix			=	'notifications_' . substr( $stream->id(), 0, 5 ) . '_';

		if ( $id ) {
			if ( $output === 'update' ) {
				$stream->set( 'date', array( Application::Database()->getUtcDateTime( $id ), '>' ) );
			} else {
				$stream->set( 'id', $id );
			}
		} elseif ( $output === 'update' ) {
			return;
		}

		$stream->set( 'paging_limitstart', $this->getInput()->getInt( $notificationPrefix . 'limitstart', 0 ) );

		if ( $stream->getBool( 'query', true ) ) {
			if ( $stream->has( 'query_count' ) ) {
				$rowsTotal			=	$stream->getInt( 'query_count', 0 );
			} else {
				// We'll find out the total for paging after we perform the SELECT query:
				$rowsTotal			=	null;
			}
		} else {
			$rowsTotal				=	0;
		}

		if ( $stream->getInt( 'paging_limitstart', 0 ) === 0 ) {
			$pageLimit				=	$stream->getInt( 'paging_first_limit', 15 );
		} else {
			$pageLimit				=	$stream->getInt( 'paging_limit', 15 );
		}

		$pageNav					=	new cbPageNav( $rowsTotal, $stream->getInt( 'paging_limitstart', 0 ), $pageLimit );

		$pageNav->setInputNamePrefix( $notificationPrefix );
		$pageNav->setBaseURL( $_CB_framework->pluginClassUrl( $this->element, false, array( 'action' => 'notifications', 'func' => 'load', 'stream' => $stream->id() ), 'raw', 0, true ) );

		$rows						=	array();

		if ( $stream->getBool( 'query', true ) && ( $rowsTotal || ( $rowsTotal === null ) ) ) {
			$rows					=	$stream->rows();

			if ( $stream->getString( 'direction', 'down' ) === 'up' ) {
				$rows				=	array_reverse( $rows, true );
			}

			if ( $rowsTotal === null ) {
				$pageNav->total		=	$stream->getInt( 'paging_total', 0 );
			}
		}

		$pageNav->limitstart		=	$stream->getInt( 'paging_limitstart', 0 );
		$integrations				=	$_PLUGINS->trigger( 'activity_onBeforeDisplayNotificationsStream', array( &$rows, &$pageNav, $viewer, &$stream, $output ) );

		if ( ( ! $rows ) && ( in_array( $output, array( 'load', 'update' ), true ) || $stream->getBool( 'inline', false ) ) ) {
			return;
		}

		$pinnedTooltip				=	cbTooltip( null, CBTxt::T( 'Pinned' ), null, 'auto', null, null, null, 'data-hascbtooltip="true" data-cbtooltip-position-my="bottom center" data-cbtooltip-position-at="top center" data-cbtooltip-classes="qtip-simple"' );
		$globalTooltip				=	cbTooltip( null, CBTxt::T( 'Announcement' ), null, 'auto', null, null, null, 'data-hascbtooltip="true" data-cbtooltip-position-my="bottom center" data-cbtooltip-position-at="top center" data-cbtooltip-classes="qtip-simple"' );
		$reportedTooltip			=	cbTooltip( null, CBTxt::T( 'Controversial' ), null, 'auto', null, null, null, 'data-hascbtooltip="true" data-cbtooltip-position-my="bottom center" data-cbtooltip-position-at="top center" data-cbtooltip-classes="qtip-simple"' );
		$direction					=	$stream->getString( 'direction', 'down' );

		if ( ! in_array( $output, array( 'load', 'update' ), true ) ) {
			CBActivity::bindStream();

			require CBActivity::getTemplate( $stream->getString( 'template' ), 'notifications/display', ( $output ? false : true ) );
		} else {
			require CBActivity::getTemplate( $stream->getString( 'template' ), 'notifications/rows', false );
		}

		// Skip the read check if the stream isn't outputting read state:
		if ( $stream->getString( 'read' ) === null ) {
			return;
		}

		$read						=	new ReadTable();

		$read->load( array( 'user_id' => $viewer->getInt( 'id', 0 ), 'asset' => $stream->asset() ) );

		$read->set( 'user_id', $viewer->getInt( 'id', 0 ) );
		$read->set( 'asset', $stream->asset() );
		$read->set( 'date', Application::Database()->getUtcDateTime() );

		$read->store();
	}

	/**
	 * Deletes notification
	 *
	 * @param int           $id
	 * @param UserTable     $viewer
	 * @param Notifications $stream
	 */
	private function deleteNotification( $id, $viewer, $stream )
	{
		global $_CB_database, $_PLUGINS;

		if ( $id ) {
			$row				=	$stream->row( $id );

			if ( ( ! $row->getInt( 'id', 0 ) ) || ( ( $viewer->getInt( 'id', 0 ) !== $row->getInt( 'user', 0 ) ) && ( ! CBActivity::canModerate( $stream ) ) ) ) {
				CBActivity::ajaxResponse( CBTxt::T( 'You do not have permission to delete this notification.' ), 'error', 'append' );
			}

			$source				=	$row->source();

			$_PLUGINS->trigger( 'activity_onBeforeDeleteStreamNotification', array( $stream, $source, &$row ) );

			if ( $row->getError() || ( ! $row->canDelete() ) ) {
				CBActivity::ajaxResponse( CBTxt::T( 'NOTIFICATION_FAILED_TO_DELETE', 'Notification failed to delete! Error: [error]', array( '[error]' => $row->getError() ) ), 'error', 'append' );
			}

			if ( $row->getError() || ( ! $row->delete() ) ) {
				CBActivity::ajaxResponse( CBTxt::T( 'NOTIFICATION_FAILED_TO_DELETE', 'Notification failed to delete! Error: [error]', array( '[error]' => $row->getError() ) ), 'error', 'append' );
			}

			$_PLUGINS->trigger( 'activity_onAfterDeleteStreamNotification', array( $stream, $source, $row ) );

			CBActivity::ajaxResponse();
		} else {
			if ( ( $viewer->getInt( 'id', 0 ) !== $stream->user()->getInt( 'id', 0 ) ) && ( ! CBActivity::canModerate( $stream ) ) ) {
				CBActivity::ajaxResponse( CBTxt::T( 'You do not have permission to delete notifications.' ), 'error', 'append' );
			}

			$query				=	'SELECT *'
								.	"\n FROM " . $_CB_database->NameQuote( '#__comprofiler_plugin_activity_notifications' )
								.	"\n WHERE " . $_CB_database->NameQuote( 'user' ) . " = " . $stream->user()->getInt( 'id', 0 )
								.	"\n AND " . $_CB_database->NameQuote( 'global' ) . " != 1";
			$_CB_database->setQuery( $query );
			$rows				=	$_CB_database->loadObjectList( 'id', '\CB\Plugin\Activity\Table\NotificationTable', array( $_CB_database ) );

			/** @var NotificationTable[] $rows */
			foreach ( $rows as $row ) {
				$source			=	$row->source();

				$_PLUGINS->trigger( 'activity_onBeforeDeleteStreamNotification', array( $stream, $source, &$row ) );

				if ( $row->getError() || ( ! $row->canDelete() ) ) {
					continue;
				}

				if ( $row->getError() || ( ! $row->delete() ) ) {
					continue;
				}

				$_PLUGINS->trigger( 'activity_onAfterDeleteStreamNotification', array( $stream, $source, $row ) );
			}

			// Hide the remaining notifications that the user doesn't own:
			foreach ( $stream->rows( 'all' ) as $row ) {
				if ( $viewer->getInt( 'id', 0 ) === $row->getInt( 'user_id', 0 ) ) {
					continue;
				}

				$hidden			=	new HiddenTable();

				$hidden->load( array( 'user_id' => $viewer->getInt( 'id', 0 ), 'type' => 'notification', 'object' => $row->getInt( 'id', 0 ) ) );

				if ( $hidden->getInt( 'id', 0 ) ) {
					continue;
				}

				$hidden->set( 'user_id', $viewer->getInt( 'id', 0 ) );
				$hidden->set( 'type', 'notification' );
				$hidden->set( 'object', $row->getInt( 'id', 0 ) );

				$source			=	$row->source();

				$_PLUGINS->trigger( 'activity_onBeforeHideStreamNotification', array( $stream, $source, &$hidden, $row ) );

				if ( $hidden->getError() || ( ! $hidden->check() ) ) {
					continue;
				}

				if ( $hidden->getError() || ( ! $hidden->store() ) ) {
					continue;
				}

				$_PLUGINS->trigger( 'activity_onAfterHideStreamNotification', array( $stream, $source, $hidden, $row ) );
			}

			CBActivity::ajaxResponse( CBTxt::T( 'No notifications to display.' ) );
		}
	}

	/**
	 * Hides notification
	 *
	 * @param int           $id
	 * @param UserTable     $viewer
	 * @param Notifications $stream
	 */
	private function hideNotification( $id, $viewer, $stream )
	{
		global $_PLUGINS;

		$type				=	$this->getInput()->getString( 'type', 'notification' );
		$notification		=	$stream->row( $id );

		if ( ( ! $notification->getInt( 'id', 0 ) )
			 || ( ! $viewer->getInt( 'id', 0 ) )
			 || ( $viewer->getInt( 'id', 0 ) === $notification->getInt( 'user_id', 0 ) )
			 || ( ! in_array( $type, array( 'notification', 'asset', 'user' ), true ) ) )
		{
			CBActivity::ajaxResponse( CBTxt::T( 'You do not have permission to hide this notification.' ), 'error', 'append' );
		}

		$row				=	new HiddenTable();

		switch ( $type ) {
//			case 'user':
//				$hideType	=	'notification.' . $type;
//				$hideItem	=	$notification->getInt( 'user_id', 0 );
//				break;
//			case 'asset':
//				$hideType	=	'notification.' . $type;
//				$hideItem	=	$notification->getString( 'asset' );
//				break;
			case 'notification':
			default:
				$hideType	=	'notification';
				$hideItem	=	$notification->getInt( 'id', 0 );
				break;
		}

		$row->load( array( 'user_id' => $viewer->getInt( 'id', 0 ), 'type' => $hideType, ( $type === 'asset' ? 'asset' : 'object' ) => $hideItem ) );

		if ( $row->getInt( 'id', 0 ) ) {
			CBActivity::ajaxResponse( CBTxt::T( 'You have already hidden this notification.' ), 'error', 'append' );
		}

		$row->set( 'user_id', $viewer->getInt( 'id', 0 ) );
		$row->set( 'type', $hideType );
		$row->set( ( $type === 'asset' ? 'asset' : 'object' ), $hideItem );

		$source				=	$notification->source();

		$_PLUGINS->trigger( 'activity_onBeforeHideStreamNotification', array( $stream, $source, &$row, $notification ) );

		if ( $row->getError() || ( ! $row->check() ) ) {
			CBActivity::ajaxResponse( CBTxt::T( 'NOTIFICATION_HIDE_FAILED_TO_SAVE', 'Notification failed to hide! Error: [error]', array( '[error]' => $row->getError() ) ), 'error', 'append' );
		}

		if ( $row->getError() || ( ! $row->store() ) ) {
			CBActivity::ajaxResponse( CBTxt::T( 'NOTIFICATION_HIDE_FAILED_TO_SAVE', 'Notification failed to hide! Error: [error]', array( '[error]' => $row->getError() ) ), 'error', 'append' );
		}

		$_PLUGINS->trigger( 'activity_onAfterHideStreamNotification', array( $stream, $source, $row, $notification ) );

		CBActivity::ajaxResponse();
	}

	/**
	 * Deletes notification hide
	 *
	 * @param int           $id
	 * @param UserTable     $viewer
	 * @param Notifications $stream
	 */
	private function unhideNotification( $id, $viewer, $stream )
	{
		global $_PLUGINS;

		$type				=	$this->getInput()->getString( 'type', 'notification' );
		$notification		=	$stream->row( $id );

		if ( ( ! $notification->getInt( 'id', 0 ) )
			 || ( ! $viewer->getInt( 'id', 0 ) )
			 || ( ! in_array( $type, array( 'notification', 'asset', 'user' ), true ) ) )
		{
			CBActivity::ajaxResponse( CBTxt::T( 'You do not have permission to unhide this notification.' ), 'error', 'append' );
		}

		$row				=	new HiddenTable();

		switch ( $type ) {
//			case 'user':
//				$row->load( array( 'user_id' => $viewer->getInt( 'id', 0 ), 'type' => 'notification.user', 'object' => $notification->getInt( 'user_id', 0 ) ) );
//				break;
//			case 'asset':
//				$row->load( array( 'user_id' => $viewer->getInt( 'id', 0 ), 'type' => 'notification.asset', 'asset' => $notification->getString( 'asset' ) ) );
//				break;
			case 'notification':
			default:
				$row->load( array( 'user_id' => $viewer->getInt( 'id', 0 ), 'type' => 'notification', 'object' => $notification->getInt( 'id', 0 ) ) );
				break;
		}

		if ( ! $row->getInt( 'id', 0 ) ) {
			CBActivity::ajaxResponse( CBTxt::T( 'You have not hidden this notification.' ), 'error', 'append' );
		}

		$source				=	$notification->source();

		$_PLUGINS->trigger( 'activity_onBeforeUnhideStreamNotification', array( $stream, $source, &$row, $notification ) );

		if ( $row->getError() || ( ! $row->canDelete() ) ) {
			CBActivity::ajaxResponse( CBTxt::T( 'NOTIFICATION_HIDE_FAILED_TO_DELETE', 'Notification failed to unhide! Error: [error]', array( '[error]' => $row->getError() ) ), 'error', 'append' );
		}

		if ( $row->getError() || ( ! $row->delete() ) ) {
			CBActivity::ajaxResponse( CBTxt::T( 'NOTIFICATION_HIDE_FAILED_TO_DELETE', 'Notification failed to unhide! Error: [error]', array( '[error]' => $row->getError() ) ), 'error', 'append' );
		}

		$_PLUGINS->trigger( 'activity_onAfterUnhideStreamNotification', array( $stream, $source, $row, $notification ) );

		CBActivity::ajaxResponse();
	}

	/**
	 * Displays notifications as a button
	 *
	 * @param UserTable     $viewer
	 * @param Notifications $stream
	 * @param string        $output
	 */
	private function showNotificationsButton( $viewer, $stream, $output = null )
	{
		$unread						=	false;
		$total						=	0;

		if ( $stream->getBool( 'query', true ) ) {
			switch ( $stream->getString( 'read' ) ) {
				case 'read':
					$total			=	$stream->reset()->setRead( 'readonly' )->rows( 'count' );
					break;
				case 'unread':
					$total			=	$stream->reset()->setRead( 'unreadonly' )->rows( 'count' );

					if ( $total ) {
						$unread		=	true;
					}
					break;
				default:
					if ( $stream->has( 'query_count' ) ) {
						$total		=	$stream->getInt( 'query_count', 0 );
					} else {
						$total		=	CBActivity::prefetchAssets( 'notifications', array(), $stream );
					}
					break;
			}
		}

		$layout						=	$stream->getString( 'layout', 'stream' );
		$autoUpdate					=	$stream->getBool( 'auto_update', false );

		CBActivity::bindStream();

		require CBActivity::getTemplate( $stream->getString( 'template' ), 'notifications/button' );
	}

	/**
	 * Displays comments stream
	 *
	 * @param int       $id
	 * @param UserTable $viewer
	 * @param Comments  $stream
	 * @param string    $output
	 */
	private function showComments( $id, $viewer, $stream, $output = null )
	{
		global $_CB_framework, $_PLUGINS;

		$commentsPrefix				=	'comments_' . substr( $stream->id(), 0, 5 ) . '_';

		if ( $id ) {
			if ( $output === 'update' ) {
				$stream->set( 'date', array( $id, '>' ) );
			} elseif ( $output === 'load' ) {
				$stream->set( 'date', array( Application::Database()->getUtcDateTime( $id ), '<' ) );
			} else {
				$stream->set( 'id', $id );
			}
		} elseif ( $output === 'update' ) {
			return;
		}

		$stream->set( 'paging_limitstart', $this->getInput()->getInt( $commentsPrefix . 'limitstart', 0 ) );

		if ( $stream->getBool( 'query', true ) ) {
			if ( $stream->has( 'query_count' ) ) {
				$rowsTotal			=	$stream->getInt( 'query_count', 0 );
			} else {
				// We'll find out the total for paging after we perform the SELECT query:
				$rowsTotal			=	null;
			}
		} else {
			$rowsTotal				=	0;
		}

		if ( $stream->getInt( 'paging_limitstart', 0 ) === 0 ) {
			$pageLimit				=	$stream->getInt( 'paging_first_limit', 15 );
		} else {
			$pageLimit				=	$stream->getInt( 'paging_limit', 15 );
		}

		$pageNav					=	new cbPageNav( $rowsTotal, $stream->getInt( 'paging_limitstart', 0 ), $pageLimit );

		$pageNav->setInputNamePrefix( $commentsPrefix );
		$pageNav->setBaseURL( $_CB_framework->pluginClassUrl( $this->element, false, array( 'action' => 'comments', 'func' => 'load', 'stream' => $stream->id() ), 'raw', 0, true ) );

		$rows						=	array();

		if ( $stream->getBool( 'query', true ) && ( $rowsTotal || ( $rowsTotal === null ) ) ) {
			$rows					=	$stream->rows();

			if ( $stream->getString( 'direction', 'down' ) === 'up' ) {
				$rows				=	array_reverse( $rows, true );
			}

			if ( $rowsTotal === null ) {
				$pageNav->total		=	$stream->getInt( 'paging_total', 0 );
			}
		}

		$pageNav->limitstart		=	$stream->getInt( 'paging_limitstart', 0 );

		$integrations				=	$_PLUGINS->trigger( 'activity_onBeforeDisplayCommentsStream', array( &$rows, &$pageNav, $viewer, &$stream, $output ) );
		$canCreate					=	( ( ! $stream->getRaw( 'id' ) ) && CBActivity::canCreate( 'comment', $stream ) );

		if ( ( ! $rows ) && ( in_array( $output, array( 'load', 'update' ), true ) || ( $stream->getBool( 'inline', false ) && ( ! $canCreate ) ) ) ) {
			return;
		}

		$pinnedTooltip				=	cbTooltip( null, CBTxt::T( 'Pinned' ), null, 'auto', null, null, null, 'data-hascbtooltip="true" data-cbtooltip-position-my="bottom center" data-cbtooltip-position-at="top center" data-cbtooltip-classes="qtip-simple"' );
		$globalTooltip				=	cbTooltip( null, CBTxt::T( 'Announcement' ), null, 'auto', null, null, null, 'data-hascbtooltip="true" data-cbtooltip-position-my="bottom center" data-cbtooltip-position-at="top center" data-cbtooltip-classes="qtip-simple"' );
		$reportedTooltip			=	cbTooltip( null, CBTxt::T( 'Controversial' ), null, 'auto', null, null, null, 'data-hascbtooltip="true" data-cbtooltip-position-my="bottom center" data-cbtooltip-position-at="top center" data-cbtooltip-classes="qtip-simple"' );
		$isReply					=	preg_match( '/^(?:(.+)\.)?comment\.(\d+)/', $stream->asset() );
		$direction					=	$stream->getString( 'direction', 'down' );
		$count						=	0;

		if ( ! in_array( $output, array( 'load', 'update' ), true ) ) {
			CBActivity::bindStream();

			if ( $rows && ( $output !== 'modal' ) && ( ! $isReply ) && $stream->getBool( 'count', true ) && $stream->getBool( 'query', true ) ) {
				if ( $stream->has( 'query_count' ) ) {
					$count			=	$stream->getInt( 'query_count', 0 );
				} else {
					$count			=	CBActivity::prefetchAssets( 'comments', array(), $stream );
				}
			}

			if ( $count ) {
				// We got the true count so lets set it in the paging:
				$pageNav->total		=	$count;
			}

			require CBActivity::getTemplate( $stream->getString( 'template' ), 'comments/display', ( $output ? false : true ) );
		} else {
			require CBActivity::getTemplate( $stream->getString( 'template' ), 'comments/rows', false );
		}
	}

	/**
	 * Displays comment new and edit
	 *
	 * @param int       $id
	 * @param UserTable $viewer
	 * @param Comments  $stream
	 * @param string    $output
	 */
	private function showCommentEdit( $id, $viewer, $stream, $output = null )
	{
		global $_CB_framework, $_PLUGINS;

		$row					=	$stream->row( $id );

		if ( ! $row->getInt( 'id', 0 ) ) {
			if ( ! CBActivity::canCreate( 'comment', $stream, $viewer ) ) {
				return;
			}
		} else {
			if ( ( ( $viewer->getInt( 'id', 0 ) !== $row->getInt( 'user_id', 0 ) ) && ( ! CBActivity::canModerate( $stream ) ) )
				 || ( $row->getBool( 'system', false ) && ( ! Application::MyUser()->isGlobalModerator() ) )
				 || ( $row->getBool( 'global', false ) && ( ! Application::MyUser()->isGlobalModerator() ) )
				 || ( ! CBActivity::findParamOverride( $row, 'edit', true ) )
			) {
				CBActivity::ajaxResponse( CBTxt::T( 'You do not have permission to edit this comment.' ), 'error' );
			}
		}

		if ( $row->getInt( 'id', 0 ) ) {
			$rowId				=	md5( $stream->id() . '_edit_' . $row->getInt( 'id', 0 ) );
		} else {
			$rowId				=	md5( $stream->id() . '_new' );
		}

		$canModerate			=	CBActivity::canModerate( $stream );
		$rowOwner				=	( ( $viewer->getInt( 'id', 0 ) === $row->getInt( 'user_id', 0 ) ) || ( ! $row->getInt( 'id', 0 ) ) );
		$isReply				=	preg_match( '/^(?:(.+)\.)?comment\.(\d+)/', $stream->asset() );
		$form					=	array();
		$buttons				=	array( 'left' => array(), 'right' => array() );

		if ( $row->getInt( 'id', 0 ) ) {
			$integrations		=	$_PLUGINS->trigger( 'activity_onDisplayStreamCommentEdit', array( &$row, &$buttons, $viewer, $stream, $output ) );
			$collapsed			=	false;
		} else {
			$integrations		=	$_PLUGINS->trigger( 'activity_onDisplayStreamCommentNew', array( &$buttons, $viewer, $stream, $output ) );
			$collapsed			=	$stream->getBool( 'collapsed', true );
		}

		$form['integrations']	=	$integrations;
		$form['collapsed']		=	$collapsed;
		$form['buttons']		=	$buttons;

		$inline					=	$stream->getBool( 'inline', false );
		$messageLimit			=	( $canModerate ? 0 : $stream->getInt( 'message_limit', 400 ) );
		$showThemes				=	CBActivity::findParamOverride( $row, 'themes', false, $stream );
		$showActions			=	CBActivity::findParamOverride( $row, 'actions', false, $stream );
		$actionLimit			=	( $canModerate ? 0 : $stream->getInt( 'actions_message_limit', 100 ) );
		$showLocations			=	CBActivity::findParamOverride( $row, 'locations', false, $stream );
		$locationLimit			=	( $canModerate ? 0 : $stream->getInt( 'locations_address_limit', 200 ) );
		$showLinks				=	CBActivity::findParamOverride( $row, 'links', false, $stream );
		$linkLimit				=	( $canModerate ? 0 : $stream->getInt( 'links_link_limit', 5 ) );
		$showTags				=	CBActivity::findParamOverride( $row, 'tags', false, $stream );
		$showEmotes				=	$stream->getBool( 'parser_emotes', true );
		$showReactions			=	( $stream->getBool( 'parser_reactions', true ) && CBActivity::getGlobalParams()->getString( 'reactions_giphy' ) );

		$themeTooltip			=	cbTooltip( null, CBTxt::T( 'Is there a theme for your post?' ), null, 'auto', null, null, null, 'data-hascbtooltip="true" data-cbtooltip-position-my="bottom center" data-cbtooltip-position-at="top center" data-cbtooltip-classes="qtip-simple"' );
		$actionTooltip			=	cbTooltip( null, CBTxt::T( 'What are you doing or feeling?' ), null, 'auto', null, null, null, 'data-hascbtooltip="true" data-cbtooltip-position-my="bottom center" data-cbtooltip-position-at="top center" data-cbtooltip-classes="qtip-simple"' );
		$locationTooltip		=	cbTooltip( null, CBTxt::T( 'Share your location.' ), null, 'auto', null, null, null, 'data-hascbtooltip="true" data-cbtooltip-position-my="bottom center" data-cbtooltip-position-at="top center" data-cbtooltip-classes="qtip-simple"' );
		$findLocationTooltip	=	cbTooltip( null, CBTxt::T( 'Click to try and find your location.' ), null, 'auto', null, null, null, 'data-hascbtooltip="true" data-cbtooltip-position-my="bottom center" data-cbtooltip-position-at="top center" data-cbtooltip-classes="qtip-simple"' );
		$tagTooltip				=	cbTooltip( null, CBTxt::T( 'Are you with anyone?' ), null, 'auto', null, null, null, 'data-hascbtooltip="true" data-cbtooltip-position-my="bottom center" data-cbtooltip-position-at="top center" data-cbtooltip-classes="qtip-simple"' );
		$linkTooltip			=	cbTooltip( null, CBTxt::T( 'Have a link to share?' ), null, 'auto', null, null, null, 'data-hascbtooltip="true" data-cbtooltip-position-my="bottom center" data-cbtooltip-position-at="top center" data-cbtooltip-classes="qtip-simple"' );
		$systemTooltip			=	cbTooltip( null, CBTxt::T( 'Is this a comment from the site?' ), null, 'auto', null, null, null, 'data-hascbtooltip="true" data-cbtooltip-position-my="bottom center" data-cbtooltip-position-at="top center" data-cbtooltip-classes="qtip-simple"' );
		$themeOptions			=	( $showThemes ? CBActivity::loadThemeOptions( false, $row->getInt( 'user_id', $viewer->getInt( 'id', 0 ) ), $stream ) : array() );
		$actionOptions			=	( $showActions ? CBActivity::loadActionOptions( false, $row->getInt( 'user_id', $viewer->getInt( 'id', 0 ) ), $stream ) : array() );
		$locationOptions		=	( $showLocations ? CBActivity::loadLocationOptions( false, $row->getInt( 'user_id', $viewer->getInt( 'id', 0 ) ), $stream ) : array() );
		$emoteOptions			=	( $actionOptions || $showEmotes ? CBActivity::loadEmoteOptions( false, false, $row->getInt( 'user_id', $viewer->getInt( 'id', 0 ) ) ) : array() );

		$form['url']						=	$_CB_framework->pluginClassUrl( 'cbactivity', true, array( 'action' => 'comments', 'func' => 'save', 'id' => $row->getInt( 'id', 0 ), 'stream' => $stream->id() ), 'raw', 0, true );
		$form['message']					=	'<textarea name="message" rows="1" class="form-control shadow-none border rounded-0 streamInput streamInputAutosize streamInputMessage' . ( $collapsed ? ' streamInputMessageCollapse' : null ) . '" placeholder="' . htmlspecialchars( ( $isReply ? CBTxt::T( 'Write a reply...' ) : CBTxt::T( 'Write a comment...' ) ) ) . '"' . ( $messageLimit ? ' data-cbactivity-input-limit="' . (int) $messageLimit . '" maxlength="' . (int) $messageLimit . '"' : null ) . '>' . htmlspecialchars( $row->getString( 'message' ) ) . '</textarea>';
		$form['message_limit']				=	$messageLimit;
		$form['reactions']					=	null;

		if ( $showReactions ) {
			$form['reactions']				=	CBActivity::renderSelectList( array(), 'insert_reaction', 'class="streamInputSelect streamInputSelectInsert streamInputMessageReactionSelect" data-cbselect-width="auto" data-cbselect-height="auto" data-cbactivity-toggle-icon="text-small streamReactionsIcon" data-cbselect-dropdown-css-class="streamReactionOptions streamReactionInsertOptions" data-cbselect-url="' . $_CB_framework->pluginClassUrl( $this->element, true, array( 'action' => 'comments', 'func' => 'reactions', 'stream' => $stream->id() ), 'raw', 0, true ) . '"', $stream );
		}

		$form['emotes']						=	null;

		if ( $showEmotes && $emoteOptions ) {
			$form['emotes']					=	CBActivity::renderSelectList( $emoteOptions, 'insert_emote', 'class="streamInputSelect streamInputSelectInsert streamInputMessageEmoteSelect" data-cbselect-width="auto" data-cbselect-height="auto" data-cbactivity-toggle-icon="fa-before fa-smile-o" data-cbselect-dropdown-css-class="streamEmoteOptions streamEmoteInsertOptions"', $stream );
		}

		$form['actions']					=	null;
		$form['actions_message']			=	null;
		$form['actions_emotes']				=	null;

		if ( $actionOptions ) {
			$action							=	$row->params()->subTree( 'action' );
			$actionId						=	$action->getInt( 'id', 0 );

			$form['actions']				=	CBActivity::renderSelectList( $actionOptions, 'actions[id]', 'class="btn btn-sm ' . ( $actionId ? 'btn-info' : 'btn-light border' ) . ' streamInputSelect streamInputSelectToggle streamInputAction" data-cbactivity-toggle-target=".streamInputActionContainer" data-cbactivity-toggle-active-classes="btn-info" data-cbactivity-toggle-inactive-classes="btn-light border" data-cbactivity-toggle-icon="fa-before fa-smile-o" data-cbselect-dropdown-css-class="streamSelectOptions streamActionOptions"' . $actionTooltip, $stream, $actionId );
			$form['actions_message']		=	'<input type="text" name="actions[message]" value="' . htmlspecialchars( $action->getString( 'message' ) ) . '" class="form-control shadow-none border-0 h-100 w-100 streamInput streamInputActionMessage streamInputSelectTogglePlaceholder streamInputAutoComplete"' . ( $actionLimit ? ' maxlength="' . (int) $actionLimit . '"' : null ) . ( ! $actionId ? ' disabled="disabled"' : null ) . ' />';

			if ( $emoteOptions ) {
				$form['actions_emotes']		=	CBActivity::renderSelectList( $emoteOptions, 'actions[emote]', 'class="streamInputSelect streamInputEmote" data-cbselect-width="auto" data-cbselect-height="auto" data-cbselect-dropdown-css-class="streamEmoteOptions"' . ( ! $actionId ? ' disabled="disabled"' : null ), $stream, $action->getInt( 'emote' ) );
			}
		}

		$form['locations']					=	null;
		$form['locations_tooltip']			=	$findLocationTooltip;
		$form['locations_place']			=	null;
		$form['locations_addr']				=	null;

		if ( $locationOptions ) {
			$location						=	$row->params()->subTree( 'location' );
			$locationId						=	$location->getInt( 'id', 0 );

			$form['locations']				=	CBActivity::renderSelectList( $locationOptions, 'location[id]', 'class="btn btn-sm ' . ( $locationId ? 'btn-info' : 'btn-light border' ) . ' streamInputSelect streamInputSelectToggle streamInputLocation" data-cbactivity-toggle-target=".streamInputLocationContainer" data-cbactivity-toggle-active-classes="btn-info" data-cbactivity-toggle-inactive-classes="btn-light border" data-cbactivity-toggle-icon="fa-before fa-map-marker" data-cbselect-dropdown-css-class="streamSelectOptions streamLocationOptions"' . $locationTooltip, $stream, $locationId );
			$form['locations_place']		=	'<input type="text" name="location[place]" value="' . htmlspecialchars( $location->getString( 'place' ) ) . '" class="form-control shadow-none border-0 w-100 streamInput streamInputLocationPlace" placeholder="' . CBTxt::T( 'Where are you?' ) . '"' . ( $locationLimit ? ' maxlength="' . (int) $locationLimit . '"' : null ) . ( ! $locationId ? ' disabled="disabled"' : null ) . ' />';
			$form['locations_addr']			=	'<input type="text" name="location[address]" value="' . htmlspecialchars( $location->getString( 'address' ) ) . '" class="form-control shadow-none border-0 w-100 streamInput streamInputLocationAddress" placeholder="' . CBTxt::T( 'Have the address to share?' ) . '"' . ( $locationLimit ? ' maxlength="' . (int) $locationLimit . '"' : null ) . ( ! $locationId ? ' disabled="disabled"' : null ) . ' />';
		}

		$tagged								=	0;
		$form['tags']						=	null;
		$form['tags_tooltip']				=	$tagTooltip;

		if ( $showTags ) {
			$tagsStream						=	$row->tags( $stream );

			if ( $row->getInt( 'id', 0 ) ) {
				if ( $row->getRaw( '_tags' ) === false ) {
					$tagged					=	0;
				} elseif ( ( $row->getRaw( '_tags' ) !== null ) && ( $row->getRaw( '_tags' ) !== true ) ) {
					$tagged					=	$row->getInt( '_tags', 0 );
				} else {
					$tagged					=	CBActivity::prefetchAssets( 'tags', array(), $tagsStream );
				}

				if ( ! $tagged ) {
					$tagsStream->set( 'query', false );
				}
			} else {
				$tagsStream->set( 'query', false );
			}

			$form['tags']					=	$tagsStream->tags( 'edit' );
		}

		$form['themes']						=	null;
		$form['themes_classes']				=	null;
		$form['themes_styles']				=	null;

		if ( $themeOptions ) {
			$theme							=	$row->theme( $stream );

			if ( $theme ) {
				$form['themes_classes']		=	' ' . trim( 'streamMessageTheme ' . htmlspecialchars( $theme['class'] ) );
				$form['themes_styles']		=	( $theme['background'] ? ' style="background-image: url(' . htmlspecialchars( $theme['background'] ) . ')" data-cbactivity-active-theme-background="' . htmlspecialchars( $theme['background'] ) . '"' : null )
											.	( $theme['class'] ? ' data-cbactivity-active-theme-class="' . htmlspecialchars( $theme['class'] ) . '"' : null );
			}

			$themeId						=	$row->params()->getInt( 'theme', 0 );

			$form['themes']					=	CBActivity::renderSelectList( $themeOptions, 'theme', 'class="btn btn-sm ' . ( $themeId ? 'btn-info' : 'btn-light border' ) . ' streamInputSelect streamInputSelectToggle streamInputTheme" data-cbactivity-toggle-active-classes="btn-info" data-cbactivity-toggle-inactive-classes="btn-light border" data-cbactivity-toggle-icon="fa-before fa-paint-brush" data-cbselect-dropdown-css-class="streamThemeOptions"' . $themeTooltip, $stream, $themeId );
		}

		$links								=	0;
		$form['links']						=	array();
		$form['links_tooltip']				=	$linkTooltip;
		$form['links_limit']				=	$linkLimit;

		if ( $showLinks ) {
			$attachments					=	$row->attachments();
			$links							=	$attachments->count();

			if ( $links ) {
				foreach ( $attachments as $i => $attachment ) {
					/** @var ParamsInterface $attachment */
					if ( $attachment->getString( 'type', 'url' ) === 'custom' ) {
						continue;
					}

					$form['links'][]		=	'<input type="text" name="links[' . $i . '][url]" value="' . htmlspecialchars( $attachment->getString( 'url' ) ) . '" class="form-control shadow-none border-0 h-100 w-100 streamInput streamInputLinkURL" placeholder="' . htmlspecialchars( CBTxt::T( "What link would you like to share?" ) ) . '" />';
				}
			} else {
				$form['links'][]			=	'<input type="text" name="links[0][url]" class="form-control shadow-none border-0 h-100 w-100 streamInput streamInputLinkURL" placeholder="' . htmlspecialchars( CBTxt::T( "What link would you like to share?" ) ) . '" disabled="disabled" />';
			}
		}

		$form['system']						=	null;
		$form['system_tooltip']				=	$systemTooltip;

		if ( $rowOwner && Application::MyUser()->isGlobalModerator() && $stream->getBool( 'create_system', false ) ) {
			$form['system']					=	'<input type="checkbox" name="system" value="1" class="hidden"' . ( $row->getBool( 'system', false ) ? ' checked' : null ) . ' /><span class="fa fa-user-secret"></span>';
		}

		if ( $row->getInt( 'id', 0 ) ) {
			ob_start();
			require CBActivity::getTemplate( $stream->getString( 'template' ), 'comments/edit', false );
			$html				=	ob_get_clean();

			CBActivity::ajaxResponse( $html );
		} else {
			require CBActivity::getTemplate( $stream->getString( 'template' ), 'comments/new' );
		}
	}

	/**
	 * Saves comment
	 *
	 * @param int       $id
	 * @param UserTable $viewer
	 * @param Comments  $stream
	 */
	private function saveComment( $id, $viewer, $stream )
	{
		global $_PLUGINS;

		$row								=	$stream->row( $id );

		if ( ! $row->getInt( 'id', 0 ) ) {
			if ( ! CBActivity::canCreate( 'comment', $stream, $viewer ) ) {
				CBActivity::ajaxResponse( CBTxt::T( 'You do not have permission to create comments.' ), 'error', 'before', '.streamInputMessage' );
			}
		} elseif ( ( ( $viewer->getInt( 'id', 0 ) !== $row->getInt( 'user_id', 0 ) ) && ( ! CBActivity::canModerate( $stream ) ) )
				   || ( $row->getBool( 'system', false ) && ( ! Application::MyUser()->isGlobalModerator() ) )
				   || ( $row->getBool( 'global', false ) && ( ! Application::MyUser()->isGlobalModerator() ) )
				   || ( ! CBActivity::findParamOverride( $row, 'edit', true ) )
		) {
			CBActivity::ajaxResponse( CBTxt::T( 'You do not have permission to edit this comment.' ), 'error', 'before', '.streamInputMessage' );
		}

		$canModerate						=	CBActivity::canModerate( $stream );
		$messageLimit						=	( $canModerate ? 0 : $stream->getInt( 'message_limit', 400 ) );
		$showThemes							=	CBActivity::findParamOverride( $row, 'themes', true, $stream );
		$showActions						=	CBActivity::findParamOverride( $row, 'actions', false, $stream );
		$actionLimit						=	( $canModerate ? 0 : $stream->getInt( 'actions_message_limit', 100 ) );
		$showLocations						=	CBActivity::findParamOverride( $row, 'locations', false, $stream );
		$locationLimit						=	( $canModerate ? 0 : $stream->getInt( 'locations_address_limit', 200 ) );
		$showLinks							=	CBActivity::findParamOverride( $row, 'links', false, $stream );
		$linkLimit							=	( $canModerate ? 0 : $stream->getInt( 'links_link_limit', 5 ) );
		$showTags							=	CBActivity::findParamOverride( $row, 'tags', false, $stream );

		$row->set( 'user_id', $row->getInt( 'user_id', $viewer->getInt( 'id', 0 ) ) );
		$row->set( 'asset', $row->getString( 'asset', $stream->asset() ) );

		if ( Application::MyUser()->isGlobalModerator() && $stream->getBool( 'create_system', false ) ) {
			if ( $this->getInput()->getBool( 'system', false ) ) {
				$systemUser					=	$this->params->getInt( 'system_user', 0 );

				if ( $systemUser ) {
					$row->set( 'user_id', $systemUser );
				}

				$row->set( 'system', 1 );
			} else {
				$row->set( 'system', 0 );
			}
		}

		$message							=	trim( $this->getInput()->getString( 'message', $row->getString( 'message', '' ) ) );

		// Remove duplicate spaces:
		$message							=	preg_replace( '/ {2,}/i', ' ', $message );
		// Remove duplicate tabs:
		$message							=	preg_replace( '/\t{2,}/i', "\t", $message );
		// Remove duplicate linebreaks:
		$message							=	preg_replace( '/((?:\r\n|\r|\n){2})(?:\r\n|\r|\n)*/i', '$1', $message );

		if ( $messageLimit && ( cbutf8_strlen( $message ) > $messageLimit ) ) {
			$message						=	cbutf8_substr( $message, 0, $messageLimit );
		}

		$row->set( 'message', $message );

		if ( $message === '' ) {
			CBActivity::ajaxResponse( CBTxt::T( 'Please provide a message.' ), 'warning', 'before', '.streamInputMessage' );
		}

		$new								=	( ! $row->getInt( 'id', 0 ) );

		if ( $showThemes ) {
			$themeId						=	$this->getInput()->getInt( 'theme', $row->params()->getInt( 'theme', 0 ) );

			if ( ! array_key_exists( $themeId, CBActivity::loadThemeOptions( true, $row->getInt( 'user_id', 0 ), $stream ) ) ) {
				$themeId					=	0;
			}

			$row->params()->set( 'theme', $themeId );
		}

		if ( $showActions ) {
			$existingAction					=	$row->params()->subTree( 'action' );
			$action							=	$this->getInput()->subTree( 'actions' );
			$actionId						=	$action->getInt( 'id', $existingAction->getInt( 'id', 0 ) );

			if ( ! array_key_exists( $actionId, CBActivity::loadActionOptions( true, $row->getInt( 'user_id', 0 ), $stream ) ) ) {
				$actionId					=	0;
			}

			$actionMessage					=	( $actionId ? trim( $action->getString( 'message', $existingAction->getString( 'message', '' ) ) ) : '' );
			$actionEmote					=	( $actionId ? $action->getInt( 'emote', $existingAction->getInt( 'emote', 0 ) ) : 0 );

			if ( ! array_key_exists( $actionEmote, CBActivity::loadEmoteOptions( false, true, $row->getInt( 'user_id', 0 ) ) ) ) {
				$actionEmote				=	0;
			}

			// Remove linebreaks:
			$actionMessage					=	str_replace( array( "\n", "\r\n" ), ' ', $actionMessage );
			// Remove duplicate spaces:
			$actionMessage					=	preg_replace( '/ {2,}/i', ' ', $actionMessage );
			// Remove duplicate tabs:
			$actionMessage					=	preg_replace( '/\t{2,}/i', "\t", $actionMessage );

			if ( $actionLimit && ( cbutf8_strlen( $actionMessage ) > $actionLimit ) ) {
				$actionMessage				=	cbutf8_substr( $actionMessage, 0, $actionLimit );
			}

			$actionId						=	( $actionMessage ? $actionId : 0 );

			$newAction						=	array(	'id'		=>	$actionId,
														'message'	=>	( $actionId ? $actionMessage : '' ),
														'emote'		=>	( $actionId ? $actionEmote : 0 )
													);

			$row->params()->set( 'action', $newAction );
		}

		if ( $showLocations ) {
			$existingLocation				=	$row->params()->subTree( 'location' );
			$location						=	$this->getInput()->subTree( 'location' );
			$locationId						=	$location->getInt( 'id', $existingLocation->getInt( 'id', 0 ) );

			if ( ! array_key_exists( $locationId, CBActivity::loadLocationOptions( true, $row->getInt( 'user_id', 0 ), $stream ) ) ) {
				$locationId					=	0;
			}

			$locationPlace					=	( $locationId ? trim( $location->getString( 'place', $existingLocation->getString( 'place', '' ) ) ) : '' );
			$locationAddress				=	( $locationId ? trim( $location->getString( 'address', $existingLocation->getString( 'address', '' ) ) ) : '' );

			if ( $locationLimit && ( cbutf8_strlen( $locationPlace ) > $locationLimit ) ) {
				$locationPlace				=	cbutf8_substr( $locationPlace, 0, $locationLimit );
			}

			if ( $locationLimit && ( cbutf8_strlen( $locationAddress ) > $locationLimit ) ) {
				$locationAddress			=	cbutf8_substr( $locationAddress, 0, $locationLimit );
			}

			$locationId						=	( $locationPlace ? $locationId : 0 );

			$newLocation					=	array(	'id'		=>	$locationId,
														'place'		=>	( $locationId ? $locationPlace : '' ),
														'address'	=>	( $locationId ? $locationAddress : '' )
													);

			$row->params()->set( 'location', $newLocation );
		}

		if ( $showLinks ) {
			$newUrls						=	array();
			$newLinks						=	array();
			$urls							=	$stream->parser( $message )->urls();

			/** @var ParamsInterface $links */
			$links							=	$this->getInput()->subTree( 'links' );

			if ( $stream->getBool( 'links_embedded', false ) ) {
				$index						=	( $links->count() - 1 );

				foreach ( $urls as $url ) {
					foreach ( $links as $link ) {
						/** @var ParamsInterface $link */
						if ( trim( $link->getString( 'url', '' ) ) === $url ) {
							continue 2;
						}
					}

					$index++;

					$links->set( $index, array( 'url' => $url, 'embedded' => true ) );
				}
			}

			foreach ( $links as $i => $link ) {
				if ( $linkLimit && ( ( $i + 1 ) > $linkLimit ) ) {
					break;
				}

				$linkUrl					=	trim( $link->getString( 'url', '' ) );

				if ( ( ! $linkUrl ) || in_array( $linkUrl, $newUrls, true ) ) {
					continue;
				}

				$linkEmbedded				=	$link->getBool( 'embedded', false );

				if ( $linkEmbedded && ( ! in_array( $linkUrl, $urls, true ) ) ) {
					continue;
				}

				if ( $link->getBool( 'parsed', false ) ) {
					foreach ( $row->params()->subTree( 'links' ) as $existingLink ) {
						/** @var ParamsInterface $existingLink */
						if ( trim( $existingLink->getString( 'url', '' ) ) === $linkUrl ) {
							$existingLink->set( 'title', trim( $link->getString( 'title', $existingLink->getString( 'title', '' ) ) ) );
							$existingLink->set( 'description', trim( $link->getString( 'description', $existingLink->getString( 'description', '' ) ) ) );
							$existingLink->set( 'thumbnail', $link->getBool( 'thumbnail', true ) );

							if ( in_array( $existingLink->getString( 'type' ), array( 'url', 'video', 'audio' ), true ) ) {
								$selected	=	$link->getInt( 'selected', 0 );

								/** @var ParamsInterface $thumbnail */
								$thumbnail	=	$existingLink->subTree( 'thumbnails' )->subTree( $selected );

								if ( $thumbnail->getString( 'url' ) ) {
									$existingLink->set( 'media', $thumbnail->asArray() );
									$existingLink->set( 'selected', $selected );
								}
							}

							$newLinks[]		=	$existingLink->asArray();
							break;
						}
					}

					continue;
				}

				$attachment					=	$stream->parser()->attachment( $linkUrl );

				if ( ! $attachment ) {
					if ( ! $linkEmbedded ) {
						CBActivity::ajaxResponse( CBTxt::T( 'Please provide a valid link.' ), 'warning' );
					}

					continue;
				}

				$newLink					=	$this->attachmentToLink( $link, $attachment );

				if ( ! $newLink ) {
					continue;
				}

				$newLinks[]					=	$newLink;
				$newUrls[]					=	$linkUrl;
			}

			if ( ! $new ) {
				foreach ( $row->params()->subTree( 'links' ) as $link ) {
					/** @var ParamsInterface $link */
					if ( $link->getString( 'type', 'url' ) !== 'custom' ) {
						continue;
					}

					$newLinks[]				=	$link->asArray();
				}
			}

			$row->params()->set( 'links', $newLinks );
		}

		$old								=	new CommentTable();
		$source								=	$row->source();

		if ( ! $new ) {
			$old->load( $row->getInt( 'id', 0 ) );

			if ( Application::Date( $row->getString( 'date' ), 'UTC' )->modify( '+5 MINUTES' )->getTimestamp() < Application::Date( 'now', 'UTC' )->getTimestamp() ) {
				$row->params()->set( 'modified', Application::Database()->getUtcDateTime() );
			}

			$_PLUGINS->trigger( 'activity_onBeforeUpdateStreamComment', array( $stream, $source, &$row, $old ) );
		} else {
			$_PLUGINS->trigger( 'activity_onBeforeCreateStreamComment', array( $stream, $source, &$row ) );
		}

		$newParams							=	clone $row->params();

		$newParams->unsetEntry( 'overrides' );

		$row->set( 'params', $newParams->asJson() );

		if ( $row->getError() || ( ! $row->check() ) ) {
			CBActivity::ajaxResponse( CBTxt::T( 'COMMENT_EDIT_FAILED_TO_SAVE', 'Comment failed to save! Error: [error]', array( '[error]' => $row->getError() ) ), 'error', 'before', '.streamInputMessage' );
		}

		if ( $row->getError() || ( ! $row->store() ) ) {
			CBActivity::ajaxResponse( CBTxt::T( 'COMMENT_EDIT_FAILED_TO_SAVE', 'Comment failed to save! Error: [error]', array( '[error]' => $row->getError() ) ), 'error', 'before', '.streamInputMessage' );
		}

		if ( $showTags ) {
			$tagsStream						=	$row->tags( $stream );

			$this->saveTags( $viewer, $tagsStream );

			$row->set( '_tags', null );
		}

		if ( ! $new ) {
			$_PLUGINS->trigger( 'activity_onAfterUpdateStreamComment', array( $stream, $source, $row, $old ) );
		} else {
			$_PLUGINS->trigger( 'activity_onAfterCreateStreamComment', array( $stream, $source, $row ) );
		}

		$pinnedTooltip						=	cbTooltip( null, CBTxt::T( 'Pinned' ), null, 'auto', null, null, null, 'data-hascbtooltip="true" data-cbtooltip-position-my="bottom center" data-cbtooltip-position-at="top center" data-cbtooltip-classes="qtip-simple"' );
		$globalTooltip						=	cbTooltip( null, CBTxt::T( 'Announcement' ), null, 'auto', null, null, null, 'data-hascbtooltip="true" data-cbtooltip-position-my="bottom center" data-cbtooltip-position-at="top center" data-cbtooltip-classes="qtip-simple"' );
		$reportedTooltip					=	cbTooltip( null, CBTxt::T( 'Controversial' ), null, 'auto', null, null, null, 'data-hascbtooltip="true" data-cbtooltip-position-my="bottom center" data-cbtooltip-position-at="top center" data-cbtooltip-classes="qtip-simple"' );
		$output								=	'save';

		ob_start();
		require CBActivity::getTemplate( $stream->getString( 'template' ), 'comments/container', false );
		$html								=	ob_get_clean();

		CBActivity::ajaxResponse( $html );
	}

	/**
	 * Deletes comment
	 *
	 * @param int       $id
	 * @param UserTable $viewer
	 * @param Comments  $stream
	 */
	private function deleteComment( $id, $viewer, $stream )
	{
		global $_PLUGINS;

		$row		=	$stream->row( $id );

		if ( ( ! $row->getInt( 'id', 0 ) )
			 || ( ( $viewer->getInt( 'id', 0 ) !== $row->getInt( 'user_id', 0 ) ) && ( ! CBActivity::canModerate( $stream ) ) )
			 || ( $row->getBool( 'system', false ) && ( ! Application::MyUser()->isGlobalModerator() ) )
			 || ( $row->getBool( 'global', false ) && ( ! Application::MyUser()->isGlobalModerator() ) )
		) {
			CBActivity::ajaxResponse( CBTxt::T( 'You do not have permission to delete this comment.' ), 'error', 'append' );
		}

		$source		=	$row->source();

		$_PLUGINS->trigger( 'activity_onBeforeDeleteStreamComment', array( $stream, $source, &$row ) );

		if ( $row->getError() || ( ! $row->canDelete() ) ) {
			CBActivity::ajaxResponse( CBTxt::T( 'COMMENT_FAILED_TO_DELETE', 'Comment failed to delete! Error: [error]', array( '[error]' => $row->getError() ) ), 'error', 'append' );
		}

		if ( $row->getError() || ( ! $row->delete() ) ) {
			CBActivity::ajaxResponse( CBTxt::T( 'COMMENT_FAILED_TO_DELETE', 'Comment failed to delete! Error: [error]', array( '[error]' => $row->getError() ) ), 'error', 'append' );
		}

		$_PLUGINS->trigger( 'activity_onAfterDeleteStreamComment', array( $stream, $source, $row ) );

		if ( preg_match( '/^(?:(.+)\.)?comment\.(\d+)/', $stream->asset() ) ) {
			CBActivity::ajaxResponse( CBTxt::T( 'This reply has been deleted.' ), 'notice' );
		} else {
			CBActivity::ajaxResponse( CBTxt::T( 'This comment has been deleted.' ), 'notice' );
		}
	}

	/**
	 * Hides comment
	 *
	 * @param int       $id
	 * @param UserTable $viewer
	 * @param Comments  $stream
	 */
	private function hideComment( $id, $viewer, $stream )
	{
		global $_CB_framework, $_PLUGINS;

		$type				=	$this->getInput()->getString( 'type', 'comment' );
		$comment			=	$stream->row( $id );

		if ( ( ! $comment->getInt( 'id', 0 ) )
			 || ( ! $viewer->getInt( 'id', 0 ) )
			 || ( $viewer->getInt( 'id', 0 ) === $comment->getInt( 'user_id', 0 ) )
			 || ( ! in_array( $type, array( 'comment', 'asset', 'user' ), true ) ) )
		{
			CBActivity::ajaxResponse( CBTxt::T( 'You do not have permission to hide this comment.' ), 'error', 'append' );
		}

		$row				=	new HiddenTable();

		switch ( $type ) {
			case 'user':
				if ( $comment->getBool( 'system', false ) ) {
					CBActivity::ajaxResponse( CBTxt::T( 'You do not have permission to hide this comment.' ), 'error', 'append' );
				}

				$hideType	=	'comment.' . $type;
				$hideItem	=	$comment->getInt( 'user_id', 0 );
				break;
			case 'asset':
				$hideType	=	'comment.' . $type;
				$hideItem	=	$comment->getString( 'asset' );
				break;
			case 'comment':
			default:
				$hideType	=	'comment';
				$hideItem	=	$comment->getInt( 'id', 0 );
				break;
		}

		$row->load( array( 'user_id' => $viewer->getInt( 'id', 0 ), 'type' => $hideType, ( $type === 'asset' ? 'asset' : 'object' ) => $hideItem ) );

		if ( $row->getInt( 'id', 0 ) ) {
			CBActivity::ajaxResponse( CBTxt::T( 'You have already hidden this comment.' ), 'error', 'append' );
		}

		$row->set( 'user_id', $viewer->getInt( 'id', 0 ) );
		$row->set( 'type', $hideType );
		$row->set( ( $type === 'asset' ? 'asset' : 'object' ), $hideItem );

		$source				=	$comment->source();

		$_PLUGINS->trigger( 'activity_onBeforeHideStreamComment', array( $stream, $source, &$row, $comment ) );

		if ( $row->getError() || ( ! $row->check() ) ) {
			CBActivity::ajaxResponse( CBTxt::T( 'COMMENT_HIDE_FAILED_TO_SAVE', 'Comment failed to hide! Error: [error]', array( '[error]' => $row->getError() ) ), 'error', 'append' );
		}

		if ( $row->getError() || ( ! $row->store() ) ) {
			CBActivity::ajaxResponse( CBTxt::T( 'COMMENT_HIDE_FAILED_TO_SAVE', 'Comment failed to hide! Error: [error]', array( '[error]' => $row->getError() ) ), 'error', 'append' );
		}

		$_PLUGINS->trigger( 'activity_onAfterHideStreamComment', array( $stream, $source, $row, $comment ) );

		$unhide				=	'<a href="javascript: void(0);" data-cbactivity-action-url="' . $_CB_framework->pluginClassUrl( $this->element, true, array( 'action' => 'comments', 'func' => 'unhide', 'type' => $type, 'id' => $comment->getInt( 'id', 0 ), 'stream' => $stream->id() ), 'raw', 0, true ) . '" class="commentsContainerUnhide streamItemAction streamItemActionResponsesRevert">' . CBTxt::T( 'Unhide' ) . '</a>';

		if ( preg_match( '/^(?:(.+)\.)?comment\.(\d+)/', $stream->asset() ) ) {
			CBActivity::ajaxResponse( CBTxt::T( 'COMMENT_REPLY_HIDDEN_UNHIDE', 'This reply has been hidden. [unhide]', array( '[unhide]' => $unhide ) ), 'notice' );
		} else {
			CBActivity::ajaxResponse( CBTxt::T( 'COMMENT_HIDDEN_UNHIDE', 'This comment has been hidden. [unhide]', array( '[unhide]' => $unhide ) ), 'notice' );
		}
	}

	/**
	 * Deletes comment hide
	 *
	 * @param int       $id
	 * @param UserTable $viewer
	 * @param Comments  $stream
	 */
	private function unhideComment( $id, $viewer, $stream )
	{
		global $_PLUGINS;

		$type				=	$this->getInput()->getString( 'type', 'comment' );
		$comment			=	$stream->row( $id );

		if ( ( ! $comment->getInt( 'id', 0 ) ) || ( ! $viewer->getInt( 'id', 0 ) ) || ( ! in_array( $type, array( 'comment', 'asset', 'user' ), true ) ) ) {
			CBActivity::ajaxResponse( CBTxt::T( 'You do not have permission to unhide this comment.' ), 'error', 'append' );
		}

		$row				=	new HiddenTable();

		switch ( $type ) {
			case 'user':
				$row->load( array( 'user_id' => $viewer->getInt( 'id', 0 ), 'type' => 'comment.user', 'object' => $comment->getInt( 'user_id', 0 ) ) );
				break;
			case 'asset':
				$row->load( array( 'user_id' => $viewer->getInt( 'id', 0 ), 'type' => 'comment.asset', 'asset' => $comment->getString( 'asset' ) ) );
				break;
			case 'comment':
			default:
				$row->load( array( 'user_id' => $viewer->getInt( 'id', 0 ), 'type' => 'comment', 'object' => $comment->getInt( 'id', 0 ) ) );
				break;
		}

		if ( ! $row->getInt( 'id', 0 ) ) {
			CBActivity::ajaxResponse( CBTxt::T( 'You have not hidden this comment.' ), 'error', 'append' );
		}

		$source				=	$comment->source();

		$_PLUGINS->trigger( 'activity_onBeforeUnhideStreamComment', array( $stream, $source, &$row, $comment ) );

		if ( $row->getError() || ( ! $row->canDelete() ) ) {
			CBActivity::ajaxResponse( CBTxt::T( 'COMMENT_HIDE_FAILED_TO_DELETE', 'Comment failed to unhide! Error: [error]', array( '[error]' => $row->getError() ) ), 'error', 'append' );
		}

		if ( $row->getError() || ( ! $row->delete() ) ) {
			CBActivity::ajaxResponse( CBTxt::T( 'COMMENT_HIDE_FAILED_TO_DELETE', 'Comment failed to unhide! Error: [error]', array( '[error]' => $row->getError() ) ), 'error', 'append' );
		}

		$_PLUGINS->trigger( 'activity_onAfterUnhideStreamComment', array( $stream, $source, $row, $comment ) );

		if ( $stream->getBool( 'hidden', false ) ) {
			CBActivity::ajaxResponse( CBTxt::T( 'This comment has been unhidden.' ), 'notice' );
		}

		CBActivity::ajaxResponse();
	}

	/**
	 * Reports a comment entry
	 *
	 * @param int       $id
	 * @param UserTable $viewer
	 * @param Comments  $stream
	 */
	private function reportComment( $id, $viewer, $stream )
	{
		global $_CB_framework, $_PLUGINS;

		$comment				=	$stream->row( $id );

		if ( ( ! $this->params->getBool( 'reporting', true ) )
			 || ( ! $comment->getInt( 'id', 0 ) )
			 || ( ! $viewer->getInt( 'id', 0 ) )
			 || ( $viewer->getInt( 'id', 0 ) === $comment->getInt( 'user_id', 0 ) )
			 || CBActivity::canModerate( $stream, $comment->getInt( 'user_id', 0 ) ) )
		{
			CBActivity::ajaxResponse( CBTxt::T( 'You do not have permission to report this comment.' ), 'error', 'append' );
		}

		$row					=	new HiddenTable();

		$row->load( array( 'user_id' => $viewer->getInt( 'id', 0 ), 'type' => 'comment', 'object' => $comment->getInt( 'id', 0 ) ) );

		$source					=	$comment->source();

		$comment->params()->set( 'reports', ( $comment->params()->getInt( 'reports', 0 ) + 1 ) );
		$comment->params()->set( 'reported', Application::Database()->getUtcDateTime() );

		$reportLimit			=	$this->params->getInt( 'reporting_limit', 10 );

		if ( $reportLimit && ( $comment->params()->getInt( 'reports', 0 ) >= $reportLimit ) ) {
			$comment->set( 'published', 0 );
		}

		if ( ! $row->getInt( 'id', 0 ) ) {
			$row->set( 'user_id', $viewer->getInt( 'id', 0 ) );
			$row->set( 'type', 'comment' );
			$row->set( 'object', $comment->getInt( 'id', 0 ) );

			$_PLUGINS->trigger( 'activity_onBeforeHideStreamComment', array( $stream, $source, &$row, $comment ) );

			if ( $row->getError() || ( ! $row->check() ) ) {
				CBActivity::ajaxResponse( CBTxt::T( 'COMMENT_HIDE_FAILED_TO_SAVE', 'Comment failed to hide! Error: [error]', array( '[error]' => $row->getError() ) ), 'error', 'append' );
			}

			if ( $row->getError() || ( ! $row->store() ) ) {
				CBActivity::ajaxResponse( CBTxt::T( 'COMMENT_HIDE_FAILED_TO_SAVE', 'Comment failed to hide! Error: [error]', array( '[error]' => $row->getError() ) ), 'error', 'append' );
			}

			$_PLUGINS->trigger( 'activity_onAfterHideStreamComment', array( $stream, $source, $row, $comment ) );
		}

		$_PLUGINS->trigger( 'activity_onBeforeReportStreamComment', array( $stream, $source, &$comment, $row ) );

		$newParams				=	clone $comment->params();

		$newParams->unsetEntry( 'overrides' );

		$comment->set( 'params', $newParams->asJson() );

		if ( $comment->getError() || ( ! $comment->check() ) ) {
			CBActivity::ajaxResponse( CBTxt::T( 'COMMENT_REPORT_FAILED_TO_SAVE', 'Comment failed to report! Error: [error]', array( '[error]' => $comment->getError() ) ), 'error', 'append' );
		}

		if ( $comment->getError() || ( ! $comment->store() ) ) {
			CBActivity::ajaxResponse( CBTxt::T( 'COMMENT_REPORT_FAILED_TO_SAVE', 'Comment failed to report! Error: [error]', array( '[error]' => $comment->getError() ) ), 'error', 'append' );
		}

		$_PLUGINS->trigger( 'activity_onAfterReportStreamComment', array( &$stream, $source, $comment, $row ) );

		if ( $this->params->getBool( 'reporting_notify', true ) ) {
			$cbUser				=	CBuser::getInstance( $viewer->getInt( 'id', 0 ), false );

			$extraStrings		=	array(	'comment_id'		=>	$comment->getInt( 'id', 0 ),
											'comment_message'	=>	$comment->getString( 'message' ),
											'comment_url'		=>	$_CB_framework->pluginClassUrl( $this->element, true, array( 'action' => 'comments', 'func' => 'show', 'id' => $comment->getInt( 'id', 0 ), 'stream' => $stream->id() ) ),
											'user_url'			=>	$_CB_framework->viewUrl( 'userprofile', true, array( 'user' => $viewer->getInt( 'user_id', 0 ) ) )
										);

			$subject			=	$cbUser->replaceUserVars( CBTxt::T( 'Comment - Reported!' ), false, true, $extraStrings, false );

			if ( ! $comment->getInt( 'published', 1 ) ) {
				$message		=	$cbUser->replaceUserVars( CBTxt::T( '<a href="[user_url]">[formatname]</a> reported comment <a href="[comment_url]">[cb:if comment_message!=""][comment_message][cb:else]#[comment_id][/cb:else][/cb:if]</a> as controversial and has now been unpublished!' ), false, true, $extraStrings, false );
			} else {
				$message		=	$cbUser->replaceUserVars( CBTxt::T( '<a href="[user_url]">[formatname]</a> reported comment <a href="[comment_url]">[cb:if comment_message!=""][comment_message][cb:else]#[comment_id][/cb:else][/cb:if]</a> as controversial!' ), false, true, $extraStrings, false );
			}

			$notifications		=	new cbNotification();
			$recipients			=	$stream->getRaw( 'moderators', array() );

			if ( $recipients ) {
				cbToArrayOfInt( $recipients );

				foreach ( $recipients as $recipient ) {
					$notifications->sendFromSystem( $recipient, $subject, $message, false, 1 );
				}
			} else {
				$notifications->sendToModerators( $subject, $message, false, 1 );
			}
		}

		CBActivity::ajaxResponse( CBTxt::T( 'This comment has been reported and hidden.' ), 'notice' );
	}

	/**
	 * Pins comment to the top of a stream
	 *
	 * @param int       $id
	 * @param UserTable $viewer
	 * @param Comments  $stream
	 */
	private function pinComment( $id, $viewer, $stream )
	{
		global $_PLUGINS;

		$row				=	$stream->row( $id );

		if ( ( ! $row->getInt( 'id', 0 ) ) || ( ! Application::MyUser()->isGlobalModerator() ) ) {
			CBActivity::ajaxResponse( CBTxt::T( 'You do not have permission to pin this comment.' ), 'error', 'append' );
		} elseif ( $row->getBool( 'pinned', false ) ) {
			CBActivity::ajaxResponse( CBTxt::T( 'You have already pinned this comment.' ), 'error', 'append' );
		}

		$row->set( 'pinned', 1 );

		$source				=	$row->source();

		$_PLUGINS->trigger( 'activity_onBeforePinStreamComment', array( $stream, $source, &$row ) );

		if ( $row->getError() || ( ! $row->check() ) ) {
			CBActivity::ajaxResponse( CBTxt::T( 'COMMENT_PIN_FAILED_TO_SAVE', 'Comment failed to pin! Error: [error]', array( '[error]' => $row->getError() ) ), 'error', 'append' );
		}

		if ( $row->getError() || ( ! $row->store() ) ) {
			CBActivity::ajaxResponse( CBTxt::T( 'COMMENT_PIN_FAILED_TO_SAVE', 'Comment failed to pin! Error: [error]', array( '[error]' => $row->getError() ) ), 'error', 'append' );
		}

		$_PLUGINS->trigger( 'activity_onAfterPinStreamComment', array( $stream, $source, $row ) );

		$pinnedTooltip		=	cbTooltip( null, CBTxt::T( 'Pinned' ), null, 'auto', null, null, null, 'data-hascbtooltip="true" data-cbtooltip-position-my="bottom center" data-cbtooltip-position-at="top center" data-cbtooltip-classes="qtip-simple"' );
		$globalTooltip		=	cbTooltip( null, CBTxt::T( 'Announcement' ), null, 'auto', null, null, null, 'data-hascbtooltip="true" data-cbtooltip-position-my="bottom center" data-cbtooltip-position-at="top center" data-cbtooltip-classes="qtip-simple"' );
		$reportedTooltip	=	cbTooltip( null, CBTxt::T( 'Controversial' ), null, 'auto', null, null, null, 'data-hascbtooltip="true" data-cbtooltip-position-my="bottom center" data-cbtooltip-position-at="top center" data-cbtooltip-classes="qtip-simple"' );
		$output				=	'save';

		ob_start();
		require CBActivity::getTemplate( $stream->getString( 'template' ), 'comments/container', false );
		$html				=	ob_get_clean();

		CBActivity::ajaxResponse( $html, 'html', 'replace', 'container' );
	}

	/**
	 * Unpins comment from the top of a stream
	 *
	 * @param int       $id
	 * @param UserTable $viewer
	 * @param Comments  $stream
	 */
	private function unpinComment( $id, $viewer, $stream )
	{
		global $_PLUGINS;

		$row				=	$stream->row( $id );

		if ( ( ! $row->getInt( 'id', 0 ) ) || ( ! Application::MyUser()->isGlobalModerator() ) ) {
			CBActivity::ajaxResponse( CBTxt::T( 'You do not have permission to unpin this comment.' ), 'error', 'append' );
		} elseif ( ! $row->getBool( 'pinned', false ) ) {
			CBActivity::ajaxResponse( CBTxt::T( 'You have not pinned this comment.' ), 'error', 'append' );
		}

		$row->set( 'pinned', 0 );

		$source				=	$row->source();

		$_PLUGINS->trigger( 'activity_onBeforeUnpinStreamComment', array( $stream, $source, &$row ) );

		if ( $row->getError() || ( ! $row->check() ) ) {
			CBActivity::ajaxResponse( CBTxt::T( 'COMMENT_UNPIN_FAILED_TO_SAVE', 'Comment failed to unpin! Error: [error]', array( '[error]' => $row->getError() ) ), 'error', 'append' );
		}

		if ( $row->getError() || ( ! $row->store() ) ) {
			CBActivity::ajaxResponse( CBTxt::T( 'COMMENT_UNPIN_FAILED_TO_SAVE', 'Comment failed to unpin! Error: [error]', array( '[error]' => $row->getError() ) ), 'error', 'append' );
		}

		$_PLUGINS->trigger( 'activity_onAfterUnpinStreamComment', array( $stream, $source, $row ) );

		$pinnedTooltip		=	cbTooltip( null, CBTxt::T( 'Pinned' ), null, 'auto', null, null, null, 'data-hascbtooltip="true" data-cbtooltip-position-my="bottom center" data-cbtooltip-position-at="top center" data-cbtooltip-classes="qtip-simple"' );
		$globalTooltip		=	cbTooltip( null, CBTxt::T( 'Announcement' ), null, 'auto', null, null, null, 'data-hascbtooltip="true" data-cbtooltip-position-my="bottom center" data-cbtooltip-position-at="top center" data-cbtooltip-classes="qtip-simple"' );
		$reportedTooltip	=	cbTooltip( null, CBTxt::T( 'Controversial' ), null, 'auto', null, null, null, 'data-hascbtooltip="true" data-cbtooltip-position-my="bottom center" data-cbtooltip-position-at="top center" data-cbtooltip-classes="qtip-simple"' );
		$output				=	'save';

		ob_start();
		require CBActivity::getTemplate( $stream->getString( 'template' ), 'comments/container', false );
		$html				=	ob_get_clean();

		CBActivity::ajaxResponse( $html, 'html', 'replace', 'container' );
	}

	/**
	 * Displays comments as a button
	 *
	 * @param UserTable $viewer
	 * @param Comments  $stream
	 * @param string    $output
	 */
	private function showCommentsButton( $viewer, $stream, $output = null )
	{
		$total				=	0;

		if ( $stream->getBool( 'query', true ) ) {
			if ( $stream->has( 'query_count' ) ) {
				$total		=	$stream->getInt( 'query_count', 0 );
			} else {
				$total		=	CBActivity::prefetchAssets( 'comments', array(), $stream );
			}
		}

		$layout				=	$stream->getString( 'layout', 'stream' );
		$autoUpdate			=	$stream->getBool( 'auto_update', false );

		CBActivity::bindStream();

		require CBActivity::getTemplate( $stream->getString( 'template' ), 'comments/button' );
	}

	/**
	 * Caches and displays reaction options
	 *
	 * @param UserTable         $viewer
	 * @param Activity|Comments $stream
	 */
	private function showReactions( $viewer, $stream )
	{
		global $_CB_framework;

		$key					=	CBActivity::getGlobalParams()->getString( 'reactions_giphy' );
		$search					=	trim( $this->getInput()->getString( 'search' ) );
		$page					=	$this->getInput()->getInt( 'page', 1 );

		if ( ( ! $key ) || ( ! $stream->getBool( 'parser_reactions', true ) ) ) {
			CBActivity::ajaxResponse( array( 'results' => array(), 'term' => $search, 'pagination' => array( 'count' => 0, 'total' => 0, 'more' => false, 'page' => $page ) ), 'raw' );
		}

		$reactions				=	null;
		$limit					=	25;

		$cachePath				=	$_CB_framework->getCfg( 'absolute_path' ) . '/cache/activity_reactions';
		$cache					=	$cachePath . '/' . md5( $search . $page );

		if ( file_exists( $cache ) ) {
			if ( ( ( Application::Date( 'now', 'UTC' )->getTimestamp() - filemtime( $cache ) ) / 3600 ) > 24 ) {
				$request		=	true;
			} else {
				$reactions		=	trim( file_get_contents( $cache ) );
				$request		=	false;
			}
		} else {
			$request			=	true;
		}

		if ( $request ) {
			$client				=	@new \GuzzleHttp\Client();

			try {
				$query			=	array(	'api_key'	=>	$key,
											'limit'		=>	$limit,
											'offset'	=>	( ( $page - 1 ) * $limit ),
										);

				if ( $search ) {
					$query['q']	=	$search;
				}

				$result			=	$client->get( 'https://api.giphy.com/v1/gifs/' . ( $search ? 'search' : 'trending' ), array( 'query' => $query ) );

				if ( (int) $result->getStatusCode() === 200 ) {
					$reactions	=	(string) $result->getBody();
				}
			} catch( \Exception $e ) {}

			if ( ! $reactions ) {
				CBActivity::ajaxResponse( array( 'results' => array(), 'term' => $search, 'pagination' => array( 'count' => 0, 'total' => 0, 'more' => false, 'page' => $page ) ), 'raw' );
			}

			if ( ! is_dir( $cachePath ) ) {
				$oldMask		=	@umask( 0 );

				if ( @mkdir( $cachePath, 0755, true ) ) {
					@umask( $oldMask );
					@chmod( $cachePath, 0755 );
				} else {
					@umask( $oldMask );
				}
			}

			file_put_contents( $cache, $reactions );
		}

		$options				=	array();
		$total					=	0;
		$more					=	false;

		if ( $reactions ) {
			$data				=	json_decode( $reactions, true );

			if ( $data ) {
				$total			=	$data['pagination']['total_count'];
				$more			=	( ( $data['pagination']['offset'] + $data['pagination']['count'] ) < $total );

				foreach ( $data['data'] as $option ) {
					$options[]	=	array( 'id' => '(giphy:' . htmlspecialchars( $option['id'] ) . ')', 'text' => '<img src="' . htmlspecialchars( $option['images']['fixed_width']['url'] ) . '" loading="lazy" />' );
				}
			}
		}

		CBActivity::ajaxResponse( array( 'results' => $options, 'term' => $search, 'pagination' => array( 'count' => count( $options ), 'total' => $total, 'more' => $more, 'page' => $page ) ), 'raw' );
	}

	/**
	 * Displays tags stream
	 *
	 * @param UserTable $viewer
	 * @param Tags      $stream
	 * @param string    $output
	 */
	private function showTags( $viewer, $stream, $output = null )
	{
		global $_CB_framework, $_PLUGINS;

		$tagsPrefix				=	'tags_' . substr( $stream->id(), 0, 5 ) . '_';

		$stream->set( 'paging_limitstart', $this->getInput()->getInt( $tagsPrefix . 'limitstart', 0 ) );

		if ( $stream->getBool( 'query', true ) ) {
			if ( $stream->has( 'query_count' ) ) {
				$rowsTotal		=	$stream->getInt( 'query_count', 0 );
			} else {
				// We'll find out the total for paging after we perform the SELECT query:
				$rowsTotal		=	null;
			}
		} else {
			$rowsTotal			=	0;
		}

		$pageNav				=	new cbPageNav( $rowsTotal, $stream->getInt( 'paging_limitstart', 0 ), 15 );

		$pageNav->setInputNamePrefix( $tagsPrefix );
		$pageNav->setBaseURL( $_CB_framework->pluginClassUrl( $this->element, false, array( 'action' => 'tags', 'func' => 'load', 'stream' => $stream->id() ), 'raw', 0, true ) );

		$stream->set( 'paging', true );
		$stream->set( 'paging_first_limit', $pageNav->limit );
		$stream->set( 'paging_limit', $pageNav->limit );

		$rows					=	array();

		if ( $stream->getBool( 'query', true ) && ( $rowsTotal || ( $rowsTotal === null ) ) ) {
			$rows				=	$stream->rows();

			if ( $rowsTotal === null ) {
				// Calculated from the SELECT since we didn't prefetch it above:
				$pageNav->total	=	$stream->getInt( 'paging_total', 0 );
			}
		}

		$pageNav->limitstart	=	$stream->getInt( 'paging_limitstart', 0 );
		$integrations			=	$_PLUGINS->trigger( 'activity_onBeforeDisplayTagsStream', array( &$rows, &$pageNav, $viewer, &$stream, $output ) );

		if ( ! $rows ) {
			return;
		}

		$count					=	0;

		if ( ! in_array( $output, array( 'load', 'update' ), true ) ) {
			CBActivity::bindStream();

			if ( ( $output !== 'modal' ) && $stream->getBool( 'count', true ) && $stream->getBool( 'query', true ) ) {
				if ( $stream->has( 'query_count' ) ) {
					$count		=	$stream->getInt( 'query_count', 0 );
				} else {
					$count		=	CBActivity::prefetchAssets( 'tags', array(), $stream );
				}
			}

			if ( $count ) {
				// We got the true count so lets set it in the paging:
				$pageNav->total	=	$count;
			}

			require CBActivity::getTemplate( $stream->getString( 'template' ), 'tags/display', ( $output ? false : true ) );
		} else {
			require CBActivity::getTemplate( $stream->getString( 'template' ), 'tags/rows', false );
		}
	}

	/**
	 * Displays tagged stream
	 *
	 * @param UserTable $viewer
	 * @param Tags      $stream
	 * @param string    $output
	 */
	private function showTagged( $viewer, $stream, $output = null )
	{
		global $_PLUGINS;

		$stream->set( 'paging', true );
		$stream->set( 'paging_first_limit', 3 );
		$stream->set( 'paging_limit', 3 );
		$stream->set( 'paging_limitstart', 0 );

		$rows				=	array();

		if ( $stream->getBool( 'query', true ) ) {
			$rows			=	$stream->rows();
		}

		$integrations		=	$_PLUGINS->trigger( 'activity_onBeforeDisplayStreamTagged', array( &$rows, $viewer, &$stream, $output ) );

		if ( ! $rows ) {
			return;
		}

		$tags				=	array();

		foreach ( $rows as $row ) {
			if ( is_numeric( $row->getString( 'tag' ) ) ) {
				$name		=	CBuser::getInstance( $row->getInt( 'tag', 0 ), false )->getField( 'formatname', null, 'html', 'none', 'list', 0, true );
			} else {
				$name		=	htmlspecialchars( $row->getString( 'tag' ) );
			}

			if ( ! $name ) {
				continue;
			}

			$tags[]			=	$name;
		}

		if ( ! $tags ) {
			return;
		}

		$total				=	0;

		if ( count( $tags ) > 2 ) {
			if ( $stream->getBool( 'query', true ) ) {
				if ( $stream->has( 'query_count' ) ) {
					$total	=	$stream->getInt( 'query_count', 0 );
				} else {
					$total	=	CBActivity::prefetchAssets( 'tags', array(), $stream );
				}
			}
		}

		CBActivity::bindStream();

		require CBActivity::getTemplate( $stream->getString( 'template' ), 'tags/tagged' );
	}

	/**
	 * Displays users tags list
	 *
	 * @param UserTable $viewer
	 * @param Tags      $stream
	 */
	private function showTagsList( $viewer, $stream )
	{
		global $_CB_database;

		$search				=	trim( $this->getInput()->getString( 'search' ) );
		$page				=	$this->getInput()->getInt( 'page', 1 );

		if ( ! $stream->user()->getInt( 'id', 0 ) ) {
			CBActivity::ajaxResponse( array( 'results' => array(), 'term' => $search, 'pagination' => array( 'count' => 0, 'total' => 0, 'more' => false, 'page' => $page ) ), 'raw' );
		}

		$limit				=	25;

		$query				=	"SELECT j." . $_CB_database->NameQuote( 'id' )
							.	", j." . $_CB_database->NameQuote( 'name' )
							.	", j." . $_CB_database->NameQuote( 'username' )
							.	"\n FROM " . $_CB_database->NameQuote( '#__comprofiler_members' ) . " AS m"
							.	"\n LEFT JOIN " . $_CB_database->NameQuote( '#__comprofiler' ) . " AS cb"
							.	" ON cb." . $_CB_database->NameQuote( 'id' ) . " = m." . $_CB_database->NameQuote( 'memberid' )
							.	"\n LEFT JOIN " . $_CB_database->NameQuote( '#__users' ) . " AS j"
							.	" ON j." . $_CB_database->NameQuote( 'id' ) . " = m." . $_CB_database->NameQuote( 'memberid' )
							.	"\n WHERE cb." . $_CB_database->NameQuote( 'approved' ) . " = 1"
							.	"\n AND cb." . $_CB_database->NameQuote( 'confirmed' ) . " = 1"
							.	"\n AND j." . $_CB_database->NameQuote( 'block' ) . " = 0";

		if ( $search ) {
			$query			.=	"\n AND ( j." . $_CB_database->NameQuote( 'username' ) . " LIKE " . $_CB_database->Quote( '%' . $_CB_database->getEscaped( $search, true ) . '%', false )
							.	" OR j." . $_CB_database->NameQuote( 'name' ) . " LIKE " . $_CB_database->Quote( '%' . $_CB_database->getEscaped( $search, true ) . '%', false ) . " )";
		}

		$query				.=	"\n AND m." . $_CB_database->NameQuote( 'referenceid' ) . " = ". $stream->user()->getInt( 'id', 0 )
							.	"\n AND m." . $_CB_database->NameQuote( 'accepted' ) . " = 1";
		$_CB_database->setQuery( $query, ( ( $page - 1 ) * $limit ), ( $limit + 1 ) );
		$connections		=	$_CB_database->loadObjectList();

		$more				=	( count( $connections ) > $limit );
		$options			=	array();

		foreach ( $connections as $connection ) {
			$options[]		=	array( 'id' => (string) $connection->id, 'text' => getNameFormat( $connection->name, $connection->username, Application::Config()->getInt( 'name_format', 3 ) ) );
		}

		CBActivity::ajaxResponse( array( 'results' => $options, 'term' => $search, 'pagination' => array( 'count' => count( $options ), 'total' => 0, 'more' => $more, 'page' => $page ) ), 'raw' );
	}

	/**
	 * Displays tags stream
	 *
	 * @param UserTable $viewer
	 * @param Tags      $stream
	 */
	private function showTagsEdit( $viewer, $stream )
	{
		global $_CB_framework, $_PLUGINS;

		if ( ! CBActivity::canCreate( 'tag', $stream, $viewer ) ) {
			return;
		}

		$rows					=	array();

		if ( $stream->getBool( 'query', true ) ) {
			$rows				=	$stream->rows( 'all' );
		}

		$_PLUGINS->trigger( 'activity_onDisplayStreamTagsEdit', array( &$rows, $viewer, $stream ) );

		if ( $stream->getBool( 'inline', false ) ) {
			$classes			=	'form-control shadow-none border-0 w-100 tagsStreamEdit streamInputSelect streamInputTags';
		} else {
			static $loaded		=	0;

			if ( ! $loaded++ ) {
				$_CB_framework->outputCbJQuery( "$( '.tagsStreamEdit' ).cbselect();", 'cbselect' );
			}

			$classes			=	'form-control tagsStreamEdit';
		}

		$selected				=	array();
		$tagged					=	array();

		foreach ( $rows as $row ) {
			$tag				=	$row->getString( 'tag' );

			$selected[]			=	$tag;
			$tagged[]			=	moscomprofilerHTML::makeOption( $tag, ( is_numeric( $tag ) ? CBuser::getInstance( (int) $tag, false )->getField( 'formatname', null, 'html', 'none', 'profile', 0, true, array( '_allowProfileLink' => false, '_hideLayout' => 1 ) ) : $tag ) );
		}

		$placeholder			=	$stream->getString( 'placeholder' );
		$name					=	md5( 'tags_' . $stream->asset() );
		$attributes				=	null;

		if ( $placeholder ) {
			$attributes			.=	' data-cbselect-placeholder="' . htmlspecialchars( $placeholder ) . '"';
		}

		if ( Application::Config()->getBool( 'allowConnections', true ) ) {
			$attributes			.=	' data-cbselect-url="' . $_CB_framework->pluginClassUrl( $this->element, true, array( 'action' => 'tags', 'func' => 'list', 'stream' => $stream->id() ), 'raw', 0, true ) . '"';
		}

		echo CBActivity::renderSelectList( $tagged, $name . '[]', 'multiple="multiple" class="' . htmlspecialchars( $classes ) . '" data-cbselect-tags="true" data-cbselect-width="100%" data-cbselect-height="100%" data-cbselect-dropdown-css-class="streamTagsOptions"' . $attributes, $stream, $selected );
	}

	/**
	 * Save tags
	 *
	 * @param UserTable $viewer
	 * @param Tags      $stream
	 */
	private function saveTags( $viewer, $stream )
	{
		global $_PLUGINS;

		if ( ! CBActivity::canCreate( 'tag', $stream, $viewer ) ) {
			return;
		}

		$tags				=	$this->getInput()->getRaw( md5( 'tags_' . $stream->asset() ), array() );

		if ( ! $tags ) {
			// Check if for post values if this is a new entry:
			$tags			=	$this->getInput()->getRaw( md5( 'tags_' . preg_replace( '/\.(\d+)$/', '.0', $stream->asset() ) ), array() );
		}

		foreach ( $stream->reset()->rows( 'all' ) as $tag ) {
			/** @var TagTable $tag */
			if ( ! in_array( $tag->getString( 'tag' ), $tags, true ) ) {
				$source		=	$tag->source();

				$_PLUGINS->trigger( 'activity_onBeforeDeleteStreamTag', array( $stream, $source, &$tag ) );

				if ( $tag->getError() || ( ! $tag->canDelete() ) || ( ! $tag->delete() ) ) {
					continue;
				}

				$_PLUGINS->trigger( 'activity_onAfterDeleteStreamTag', array( $stream, $source, $tag ) );
			} else {
				$key		=	array_search( $tag->getString( 'tag' ), $tags, true );

				if ( $key !== false ) {
					unset( $tags[$key] );
				}
			}
		}

		foreach ( $tags as $tagUser ) {
			if ( is_numeric( $tagUser ) ) {
				$tagUser	=	(int) $tagUser;

				if ( ! in_array( $tagUser, CBActivity::getConnections( $stream->user()->getInt( 'id', 0 ) ), true ) ) {
					continue;
				}
			} else {
				$tagUser	=	Get::clean( $tagUser, GetterInterface::STRING );
			}

			$tag			=	new TagTable();

			$tag->set( 'user_id', $stream->user()->getInt( 'id', 0 ) );
			$tag->set( 'asset', $stream->asset() );
			$tag->set( 'tag', $tagUser );

			$source			=	$tag->source();

			$_PLUGINS->trigger( 'activity_onBeforeCreateStreamTag', array( $stream, $source, &$tag ) );

			if ( $tag->getError() || ( ! $tag->check() ) || ( ! $tag->store() ) ) {
				continue;
			}

			$_PLUGINS->trigger( 'activity_onAfterCreateStreamTag', array( $stream, $source, $tag ) );
		}

		$stream->clear();
	}

	/**
	 * Follow a stream asset
	 *
	 * @param UserTable $viewer
	 * @param Following $stream
	 */
	private function followAsset( $viewer, $stream )
	{
		global $_PLUGINS;

		if ( ! CBActivity::canCreate( 'follow', $stream, $viewer ) ) {
			CBActivity::ajaxResponse( CBTxt::T( 'You do not have permission to follow this stream.' ), 'error' );
		}

		$following			=	in_array( $stream->asset(), CBActivity::getFollowing( $viewer->getInt( 'id', 0 ) ), true );

		if ( ! $following ) {
			foreach ( $stream->assets( false ) as $asset ) {
				$row		=	new FollowTable();

				$row->load( array( 'user_id' => $viewer->getInt( 'id', 0 ), 'asset' => $asset ) );

				if ( $row->getInt( 'id', 0 ) ) {
					continue;
				}

				$row->set( 'user_id', $viewer->getInt( 'id', 0 ) );
				$row->set( 'asset', $asset );

				$source		=	$row->source();

				$_PLUGINS->trigger( 'activity_onBeforeFollowStream', array( $stream, $source, &$row ) );

				if ( $row->getError() || ( ! $row->check() ) ) {
					continue;
				}

				if ( $row->getError() || ( ! $row->store() ) ) {
					continue;
				}

				$_PLUGINS->trigger( 'activity_onAfterFollowStream', array( $stream, $source, $row ) );

				$following	=	true;
			}
		}

		ob_start();
		$this->showFollowButton( $viewer, $stream, ( $following ? 'follow' : 'save' ) );
		$html				=	ob_get_clean();

		CBActivity::ajaxResponse( $html, 'html', 'replace', 'container' );
	}

	/**
	 * Unfollow a stream asset
	 *
	 * @param UserTable $viewer
	 * @param Following $stream
	 */
	private function unfollowAsset( $viewer, $stream )
	{
		global $_PLUGINS;

		if ( ! $viewer->getInt( 'id', 0 ) ) {
			CBActivity::ajaxResponse( CBTxt::T( 'You do not have permission to unfollow this stream.' ), 'error' );
		}

		$following			=	in_array( $stream->asset(), CBActivity::getFollowing( $viewer->getInt( 'id', 0 ) ), true );

		if ( $following ) {
			foreach ( $stream->assets( false ) as $asset ) {
				$row		=	new FollowTable();

				$row->load( array( 'user_id' => $viewer->getInt( 'id', 0 ), 'asset' => $asset ) );

				if ( ! $row->getInt( 'id', 0 ) ) {
					continue;
				}

				$source		=	$row->source();

				$_PLUGINS->trigger( 'activity_onBeforeUnfollowStream', array( $stream, $source, &$row ) );

				if ( $row->getError() || ( ! $row->canDelete() ) ) {
					continue;
				}

				if ( $row->getError() || ( ! $row->delete() ) ) {
					continue;
				}

				$_PLUGINS->trigger( 'activity_onAfterUnfollowStream', array( $stream, $source, $row ) );

				$following	=	false;
			}
		}

		ob_start();
		$this->showFollowButton( $viewer, $stream, ( ! $following ? 'unfollow' : 'save' ) );
		$html				=	ob_get_clean();

		CBActivity::ajaxResponse( $html, 'html', 'replace', 'container' );
	}

	/**
	 * Displays following stream button
	 *
	 * @param UserTable $viewer
	 * @param Following $stream
	 * @param string    $output
	 */
	private function showFollowButton( $viewer, $stream, $output = null )
	{
		$following				=	false;

		if ( $output === 'follow' ) {
			$following			=	true;

			$output				=	'save';
		} elseif ( $output === 'unfollow' ) {
			$output				=	'save';
		} else {
			if ( $stream->getBool( 'query', true ) ) {
				$following		=	in_array( $stream->asset(), CBActivity::getFollowing( $viewer->getInt( 'id', 0 ) ), true );
			}
		}

		$total					=	0;

		if ( $stream->getBool( 'count', true ) && $stream->getBool( 'query', true ) ) {
			if ( $stream->has( 'query_count' ) ) {
				$total			=	$stream->getInt( 'query_count', 0 );
			} else {
				$total			=	CBActivity::prefetchAssets( 'following', array(), $stream );
			}
		}

		$canCreate				=	CBActivity::canCreate( 'follow', $stream, $viewer );

		if ( ( ! $total ) && ( ! $canCreate ) ) {
			return;
		}

		$layout					=	$stream->getString( 'layout', 'button' );
		$follows				=	array();

		if ( $total && ( $layout === 'extended' ) ) {
			// We need to get the first 3 follower names, but we don't want to alter the stream object forever so lets cache the values and reset later:
			$paging				=	$stream->getBool( 'paging', true );
			$pagingFirst		=	$stream->getInt( 'paging_first_limit', 15 );
			$pagingLimit		=	$stream->getInt( 'paging_limit', 15 );
			$pagingStart		=	$stream->getInt( 'paging_limitstart', 0 );

			$stream->set( 'paging', true );
			$stream->set( 'paging_first_limit', 3 );
			$stream->set( 'paging_limit', 3 );
			$stream->set( 'paging_limitstart', 0 );

			foreach ( $stream->rows() as $row ) {
				$name			=	CBuser::getInstance( $row->getInt( 'user_id', 0 ), false )->getField( 'formatname', null, 'html', 'none', 'profile', 0, true );

				if ( ! $name ) {
					continue;
				}

				$follows[]		=	$name;
			}

			$stream->set( 'paging', $paging );
			$stream->set( 'paging_first_limit', $pagingFirst );
			$stream->set( 'paging_limit', $pagingLimit );
			$stream->set( 'paging_limitstart', $pagingStart );
		}

		CBActivity::bindStream();

		require CBActivity::getTemplate( $stream->getString( 'template' ), 'following/follow', ( $output ? false : true ) );
	}

	/**
	 * Displays following stream
	 *
	 * @param UserTable $viewer
	 * @param Following $stream
	 * @param string    $output
	 */
	private function showFollowing( $viewer, $stream, $output = null )
	{
		global $_CB_framework, $_PLUGINS;

		$followingPrefix		=	'following_' . substr( $stream->id(), 0, 5 ) . '_';

		$stream->set( 'paging_limitstart', $this->getInput()->getInt( $followingPrefix . 'limitstart', 0 ) );

		if ( $stream->getBool( 'query', true ) ) {
			if ( $stream->has( 'query_count' ) ) {
				$rowsTotal		=	$stream->getInt( 'query_count', 0 );
			} else {
				// We'll find out the total for paging after we perform the SELECT query:
				$rowsTotal		=	null;
			}
		} else {
			$rowsTotal			=	0;
		}

		$pageNav				=	new cbPageNav( $rowsTotal, $stream->getInt( 'paging_limitstart', 0 ), 15 );

		$pageNav->setInputNamePrefix( $followingPrefix );
		$pageNav->setBaseURL( $_CB_framework->pluginClassUrl( $this->element, false, array( 'action' => 'following', 'func' => 'load', 'stream' => $stream->id() ), 'raw', 0, true ) );

		$stream->set( 'paging', true );
		$stream->set( 'paging_first_limit', $pageNav->limit );
		$stream->set( 'paging_limit', $pageNav->limit );

		$rows					=	array();

		if ( $stream->getBool( 'query', true ) && ( $rowsTotal || ( $rowsTotal === null ) ) ) {
			$rows				=	$stream->rows();

			if ( $rowsTotal === null ) {
				$pageNav->total	=	$stream->getInt( 'paging_total', 0 );
			}
		}

		$pageNav->limitstart	=	$stream->getInt( 'paging_limitstart', 0 );
		$integrations			=	$_PLUGINS->trigger( 'activity_onBeforeDisplayFollowingStream', array( &$rows, &$pageNav, $viewer, &$stream, $output ) );

		if ( ( ! $rows ) && ( in_array( $output, array( 'load', 'update', 'modal' ), true ) || ( $stream->getBool( 'inline', false ) && ( ! CBActivity::canCreate( 'follow', $stream, $viewer ) ) ) ) ) {
			return;
		}

		$count					=	0;

		if ( ! in_array( $output, array( 'load', 'update' ), true ) ) {
			CBActivity::bindStream();

			if ( $rows && ( $output !== 'modal' ) && $stream->getBool( 'count', true ) && $stream->getBool( 'query', true ) ) {
				if ( $stream->has( 'query_count' ) ) {
					$count		=	$stream->getInt( 'query_count', 0 );
				} else {
					$count		=	CBActivity::prefetchAssets( 'following', array(), $stream );
				}
			}

			if ( $count ) {
				// We got the true count so lets set it in the paging:
				$pageNav->total	=	$count;
			}

			require CBActivity::getTemplate( $stream->getString( 'template' ), 'following/display', ( $output ? false : true ) );
		} else {
			require CBActivity::getTemplate( $stream->getString( 'template' ), 'following/rows', false );
		}
	}

	/**
	 * Follow a stream asset
	 *
	 * @param UserTable $viewer
	 * @param Likes     $stream
	 */
	private function likeAsset( $viewer, $stream )
	{
		global $_PLUGINS;

		if ( ! CBActivity::canCreate( 'like', $stream, $viewer ) ) {
			CBActivity::ajaxResponse( CBTxt::T( 'You do not have permission to follow this stream.' ), 'error' );
		}

		$liked				=	in_array( $stream->asset(), CBActivity::getLikes( $viewer->getInt( 'id', 0 ) ), true );

		if ( ! $liked ) {
			foreach ( $stream->assets( false ) as $asset ) {
				$row		=	new LikeTable();

				$row->load( array( 'user_id' => $viewer->getInt( 'id', 0 ), 'asset' => $asset ) );

				if ( $row->getInt( 'id', 0 ) ) {
					continue;
				}

				$types		=	array_keys( CBActivity::loadLikeOptions( true, $viewer, $stream ) );
				$type		=	$this->getInput()->getInt( 'type', ( isset( $types[0] ) ? $types[0] : 0 ) );

				if ( ! in_array( $type, $types, true ) ) {
					continue;
				}

				$row->set( 'user_id', $viewer->getInt( 'id', 0 ) );
				$row->set( 'asset', $asset );
				$row->set( 'type', $type );

				$source		=	$row->source();

				$_PLUGINS->trigger( 'activity_onBeforeLikeStream', array( $stream, $source, &$row ) );

				if ( $row->getError() || ( ! $row->check() ) ) {
					continue;
				}

				if ( $row->getError() || ( ! $row->store() ) ) {
					continue;
				}

				$_PLUGINS->trigger( 'activity_onAfterLikeStream', array( $stream, $source, $row ) );

				$liked		=	true;
			}
		}

		ob_start();
		$this->showLikeButton( $viewer, $stream, ( $liked ? 'liked' : 'save' ) );
		$html				=	ob_get_clean();

		CBActivity::ajaxResponse( $html, 'html', 'replace', 'container' );
	}

	/**
	 * Unlike a stream asset
	 *
	 * @param UserTable $viewer
	 * @param Likes     $stream
	 */
	private function unlikeAsset( $viewer, $stream )
	{
		global $_PLUGINS;

		if ( ! $viewer->getInt( 'id', 0 ) ) {
			CBActivity::ajaxResponse( CBTxt::T( 'You do not have permission to unlike this stream.' ), 'error' );
		}

		$liked				=	in_array( $stream->asset(), CBActivity::getLikes( $viewer->getInt( 'id', 0 ) ), true );

		if ( $liked ) {
			foreach ( $stream->assets( false ) as $asset ) {
				$row		=	new LikeTable();

				$row->load( array( 'user_id' => $viewer->getInt( 'id', 0 ), 'asset' => $asset ) );

				if ( ! $row->getInt( 'id', 0 ) ) {
					continue;
				}

				$source		=	$row->source();

				$_PLUGINS->trigger( 'activity_onBeforeUnlikeStream', array( $stream, $source, &$row ) );

				if ( $row->getError() || ( ! $row->canDelete() ) ) {
					continue;
				}

				if ( $row->getError() || ( ! $row->delete() ) ) {
					continue;
				}

				$_PLUGINS->trigger( 'activity_onAfterUnlikeStream', array( $stream, $source, $row ) );

				$liked		=	false;
			}
		}

		ob_start();
		$this->showLikeButton( $viewer, $stream, ( ! $liked ? 'unliked' : 'save' ) );
		$html				=	ob_get_clean();

		CBActivity::ajaxResponse( $html, 'html', 'replace', 'container' );
	}

	/**
	 * Displays likes stream button
	 *
	 * @param UserTable $viewer
	 * @param Likes     $stream
	 * @param string    $output
	 */
	private function showLikeButton( $viewer, $stream, $output = null )
	{
		$liked				=	false;

		if ( $output === 'liked' ) {
			$liked			=	true;

			$output			=	'save';
		} elseif ( $output === 'unliked' ) {
			$output			=	'save';
		} elseif ( $stream->getBool( 'query', true ) ) {
			$liked			=	in_array( $stream->asset(), CBActivity::getLikes( $viewer->getInt( 'id', 0 ) ), true );
		}

		$total				=	0;

		if ( $stream->getBool( 'count', true ) && $stream->getBool( 'query', true ) ) {
			if ( $stream->has( 'query_count' ) ) {
				$total		=	$stream->getInt( 'query_count', 0 );
			} else {
				$total		=	CBActivity::prefetchAssets( 'likes', array(), $stream );
			}
		}

		$canCreate			=	CBActivity::canCreate( 'like', $stream, $viewer );

		if ( ( ! $total ) && ( ! $canCreate ) ) {
			return;
		}

		$layout				=	$stream->getString( 'layout', 'button' );
		$likes				=	array();

		if ( $total && ( $layout === 'extended' ) ) {
			// We need to get the first 3 likes names, but we don't want to alter the stream object forever so lets cache the values and reset later:
			$paging			=	$stream->getBool( 'paging', true );
			$pagingFirst	=	$stream->getInt( 'paging_first_limit', 15 );
			$pagingLimit	=	$stream->getInt( 'paging_limit', 15 );
			$pagingStart	=	$stream->getInt( 'paging_limitstart', 0 );

			$stream->set( 'paging', true );
			$stream->set( 'paging_first_limit', 3 );
			$stream->set( 'paging_limit', 3 );
			$stream->set( 'paging_limitstart', 0 );

			foreach ( $stream->rows() as $row ) {
				$name		=	CBuser::getInstance( $row->getInt( 'user_id', 0 ), false )->getField( 'formatname', null, 'html', 'none', 'profile', 0, true );

				if ( ! $name ) {
					continue;
				}

				$likes[]	=	$name;
			}

			$stream->set( 'paging', $paging );
			$stream->set( 'paging_first_limit', $pagingFirst );
			$stream->set( 'paging_limit', $pagingLimit );
			$stream->set( 'paging_limitstart', $pagingStart );
		}

		CBActivity::bindStream();

		require CBActivity::getTemplate( $stream->getString( 'template' ), 'likes/like', ( $output ? false : true ) );
	}

	/**
	 * Displays likes stream
	 *
	 * @param UserTable $viewer
	 * @param Likes     $stream
	 * @param string    $output
	 */
	private function showLikes( $viewer, $stream, $output = null )
	{
		global $_CB_framework, $_PLUGINS;

		$likesPrefix			=	'likes_' . substr( $stream->id(), 0, 5 ) . '_';

		$stream->set( 'paging_limitstart', $this->getInput()->getInt( $likesPrefix . 'limitstart', 0 ) );

		if ( $stream->getBool( 'query', true ) ) {
			if ( $stream->has( 'query_count' ) ) {
				$rowsTotal		=	$stream->getInt( 'query_count', 0 );
			} else {
				// We'll find out the total for paging after we perform the SELECT query:
				$rowsTotal		=	null;
			}
		} else {
			$rowsTotal			=	0;
		}

		$pageNav				=	new cbPageNav( $rowsTotal, $stream->getInt( 'paging_limitstart', 0 ), 15 );

		$pageNav->setInputNamePrefix( $likesPrefix );
		$pageNav->setBaseURL( $_CB_framework->pluginClassUrl( $this->element, false, array( 'action' => 'likes', 'func' => 'load', 'stream' => $stream->id() ), 'raw', 0, true ) );

		$stream->set( 'paging', true );
		$stream->set( 'paging_first_limit', $pageNav->limit );
		$stream->set( 'paging_limit', $pageNav->limit );

		$rows					=	array();

		if ( $stream->getBool( 'query', true ) && ( $rowsTotal || ( $rowsTotal === null ) ) ) {
			$rows				=	$stream->rows();

			if ( $rowsTotal === null ) {
				$pageNav->total	=	$stream->getInt( 'paging_total', 0 );
			}
		}

		$pageNav->limitstart	=	$stream->getInt( 'paging_limitstart', 0 );
		$integrations			=	$_PLUGINS->trigger( 'activity_onBeforeDisplayLikesStream', array( &$rows, &$pageNav, $viewer, &$stream, $output ) );

		if ( ( ! $rows ) && ( in_array( $output, array( 'load', 'update', 'modal' ), true ) || ( $stream->getBool( 'inline', false ) && ( ! CBActivity::canCreate( 'like', $stream, $viewer ) ) ) ) ) {
			return;
		}

		$count					=	0;

		if ( ! in_array( $output, array( 'load', 'update' ), true ) ) {
			CBActivity::bindStream();

			if ( $rows && ( $output !== 'modal' ) && $stream->getBool( 'count', true ) && $stream->getBool( 'query', true ) ) {
				if ( $stream->has( 'query_count' ) ) {
					$count		=	$stream->getInt( 'query_count', 0 );
				} else {
					$count		=	CBActivity::prefetchAssets( 'likes', array(), $stream );
				}
			}

			if ( $count ) {
				// We got the true count so lets set it in the paging:
				$pageNav->total	=	$count;
			}

			require CBActivity::getTemplate( $stream->getString( 'template' ), 'likes/display', ( $output ? false : true ) );
		} else {
			require CBActivity::getTemplate( $stream->getString( 'template' ), 'likes/rows', false );
		}
	}
}