<?php
/**
* Community Builder (TM)
* @version $Id: $
* @package CommunityBuilder
* @copyright (C) 2004-2021 www.joomlapolis.com / Lightning MultiCom SA - and its licensors, all rights reserved
* @license http://www.gnu.org/licenses/old-licenses/gpl-2.0.html GNU/GPL version 2
*/

namespace CB\Plugin\GroupJive\Field;

use CB\Database\Table\UserTable;
use CB\Database\Table\FieldTable;
use CB\Plugin\GroupJive\CBGroupJive;
use CBLib\Application\Application;
use CBLib\Input\Get;
use CBLib\Language\CBTxt;
use CBLib\Registry\GetterInterface;

defined('CBLIB') or die();

class AutoJoinField extends \cbFieldHandler
{

	/**
	 * Accessor:
	 * Returns a field in specified format
	 *
	 * @param  FieldTable  $field
	 * @param  UserTable   $user
	 * @param  string      $output               'html', 'xml', 'json', 'php', 'csvheader', 'csv', 'rss', 'fieldslist', 'htmledit'
	 * @param  string      $reason               'profile' for user profile view, 'edit' for profile edit, 'register' for registration, 'search' for searches
	 * @param  int         $list_compare_types   IF reason == 'search' : 0 : simple 'is' search, 1 : advanced search with modes, 2 : simple 'any' search
	 * @return mixed
	 */
	public function getField( &$field, &$user, $output, $reason, $list_compare_types )
	{
		global $_CB_framework, $_CB_database;

		if ( $output === 'htmledit' ) {
			if ( $reason === 'search' ) {
				return null;
			}

			$value						=	null;

			switch ( $field->getString( 'type' ) ) {
				case 'groupmultiautojoin':
					$fieldType			=	'multiselect';
					break;
				case 'groupautojoin':
				default:
					$fieldType			=	'select';
					break;
			}

			$classes					=	array();
			$attributes					=	null;

			if ( CBGroupJive::checkTooManyGroups() ) {
				static $LOADED			=	0;

				if ( ! $LOADED++ ) {
					$js					=	"$( '.cbGroupJiveSelect' ).cbselect({"
										.		"height: 'auto',"
										.		"paginationMore: true,"
										.		"minimumInputLength: 3,"
										.		"maximumInputLength: 40,"
										.		"minimumResultsForSearch: 0,"
										.		"escapeMarkup: function( message ) {"
										.			"return message;"
										.		"},"
										.		"language: {"
										.			"errorLoading: function () {"
										.				"return null;"
										.			"},"
										.			"inputTooLong: function () {"
										.				"return null;"
										.			"},"
										.			"inputTooShort: function () {"
										.				"return null;"
										.			"},"
										.			"loadingMore: function () {"
										.				"return '<div class=\"cbGroupJiveSelectMore text-center\"><div class=\"spinner-border spinner-border-sm\"></div></div>';"
										.			"},"
										.			"noResults: function () {"
										.				"return null;"
										.			"},"
										.			"searching: function () {"
										.				"return null;"
										.			"}"
										.		"}"
										.	"});";

					$_CB_framework->outputCbJQuery( $js, 'cbselect' );
				}

				if ( Application::Application()->isClient( 'administrator' ) ) {
					$optionsUrl			=	$_CB_framework->backendViewUrl( 'fieldclass', false, array( 'field' => $field->getString( 'name' ), 'function' => 'gjautojoin', 'user' => $user->getInt( 'id', 0 ), 'reason' => $reason ), 'raw' );
				} else {
					$optionsUrl			=	$_CB_framework->viewUrl( 'fieldclass', false, array( 'field' => $field->getString( 'name' ), 'function' => 'gjautojoin', 'user' => $user->getInt( 'id', 0 ), 'reason' => $reason ), 'raw' );
				}

				$classes[]				=	'cbGroupJiveSelect';
				$attributes				=	' data-cbselect-url="' . htmlspecialchars( $optionsUrl ) . '"';
				$options				=	$this->getGroups( $field, $user, false, false, 0, 100 );
			} else {
				$options				=	$this->getGroups( $field, $user );
			}

			if ( ! $options ) {
				return null;
			}

			return $this->_fieldEditToHtml( $field, $user, $reason, 'input', $fieldType, $value, $attributes, $options, true, $classes, false );
		}

		static $cache					=	array();

		$userId							=	$user->getInt( 'id', 0 );
		$viewerId						=	Application::MyUser()->getUserId();

		if ( ! isset( $cache[$userId][$viewerId] ) ) {
			$query						=	'SELECT COUNT(*)'
										.	"\n FROM " . $_CB_database->NameQuote( '#__groupjive_groups' ) . " AS g";

			if ( ! CBGroupJive::isModerator( $viewerId ) ) {
				$query					.=	"\n LEFT JOIN " . $_CB_database->NameQuote( '#__groupjive_categories' ) . " AS c"
										.	' ON c.' . $_CB_database->NameQuote( 'id' ) . ' = g.' . $_CB_database->NameQuote( 'category' );

				if ( $viewerId && ( $viewerId !== $userId ) ) {
					$query				.=	"\n LEFT JOIN " . $_CB_database->NameQuote( '#__groupjive_users' ) . " AS mu"
										.	' ON mu.' . $_CB_database->NameQuote( 'user_id' ) . ' = ' . $viewerId
										.	' AND mu.' . $_CB_database->NameQuote( 'group' ) . ' = g.' . $_CB_database->NameQuote( 'id' )
										.	' AND mu.' . $_CB_database->NameQuote( 'status' ) . ' > 0';
				}
			}

			$query						.=	"\n LEFT JOIN " . $_CB_database->NameQuote( '#__groupjive_users' ) . " AS u"
										.	' ON u.' . $_CB_database->NameQuote( 'user_id' ) . ' = ' . $userId
										.	' AND u.' . $_CB_database->NameQuote( 'group' ) . ' = g.' . $_CB_database->NameQuote( 'id' )
										.	' AND u.' . $_CB_database->NameQuote( 'status' ) . ' > 0'
										.	"\n WHERE u." . $_CB_database->NameQuote( 'id' ) . " IS NOT NULL";

			if ( ! CBGroupJive::isModerator( $viewerId ) ) {
				$query					.=	"\n AND g." . $_CB_database->NameQuote( 'published' ) . " = 1"
										.	"\n AND ( ( c." . $_CB_database->NameQuote( 'published' ) . " = 1"
										.		' AND c.' . $_CB_database->NameQuote( 'access' ) . ' IN ' . $_CB_database->safeArrayOfIntegers( CBGroupJive::getAccess( $userId ) )
										.		' AND c.' . $_CB_database->NameQuote( 'access' ) . ' IN ' . $_CB_database->safeArrayOfIntegers( CBGroupJive::getAccess( $viewerId ) ) . ' )'
										.		( CBGroupJive::getGlobalParams()->getBool( 'groups_uncategorized', true ) ? ' OR g.' . $_CB_database->NameQuote( 'category' ) . ' = 0 )' : ' )' );
			}

			if ( ( $viewerId !== $userId ) && ( ! CBGroupJive::isModerator( $viewerId ) ) ) {
				if ( $viewerId ) {
					$query				.=	"\n AND ( g." . $_CB_database->NameQuote( 'user_id' ) . " = " . $viewerId
										.		' OR g.' . $_CB_database->NameQuote( 'type' ) . ' != 3'
										.		' OR mu.' . $_CB_database->NameQuote( 'id' ) . ' IS NOT NULL )';
				} else {
					$query				.=	"\n AND g." . $_CB_database->NameQuote( 'type' ) . " != 3";
				}
			}

			$_CB_database->setQuery( $query );
			$cache[$userId][$viewerId]	=	(int) $_CB_database->loadResult();
		}

		$return							=	$this->_formatFieldOutput( $field->getString( 'name' ), $cache[$userId][$viewerId], $output, false );

		if ( $output === 'html' ) {
			$return						=	$this->formatFieldValueLayout( $return, $reason, $field, $user );
		}

		return $return;
	}

	/**
	 * Direct access to field for custom operations, like for Ajax
	 *
	 * WARNING: direct unchecked access, except if $user is set, then check well for the $reason ...
	 *
	 * @param  FieldTable  $field
	 * @param  UserTable    $user
	 * @param  array                 $postdata
	 * @param  string                $reason     'profile' for user profile view, 'edit' for profile edit, 'register' for registration, 'search' for searches
	 * @return string                            Expected output.
	 */
	public function fieldClass( &$field, &$user, &$postdata, $reason )
	{
		if ( $this->getInput()->getString( 'function' ) === 'gjautojoin' ) {
			$search			=	trim( $this->getInput()->getString( 'search' ) );
			$page			=	$this->getInput()->getInt( 'page', 1 );
			$limit			=	100;
			$groups			=	$this->getGroups( $field, $user, true, false, ( ( $page - 1 ) * $limit ), ( $limit + 1 ), $search );
			$more			=	( count( $groups ) > $limit );
			$categories		=	array();

			foreach ( $groups as $group ) {
				$category								=	(int) $group->category;

				if ( ! isset( $categories[$category] ) ) {
					$categories[$category]				=	array( 'text' => ( $category ? CBTxt::T( $group->category_name ) : CBTxt::T( 'Uncategorized' ) ), 'children' => array() );
				}

				$categories[$category]['children'][]	=	array( 'id' => (int) $group->value, 'text' => CBTxt::T( $group->text ) );
			}

			$options		=	array_values( $categories );

			if ( ( $page === 1 ) && ( ! $this->_isRequired( $field, $user, $reason ) ) ) {
				array_unshift( $options, array( 'id' => '', 'text' => '&nbsp;' ) );
			}

			return json_encode( array( 'results' => $options, 'term' => $search, 'pagination' => array( 'count' => count( $groups ), 'total' => 0, 'more' => $more, 'page' => $page ) ) );
		}

		parent::fieldClass( $field, $user, $postdata, $reason );

		return null;
	}

	/**
	 * Mutator:
	 * Prepares field data for saving to database (safe transfer from $postdata to $user)
	 * Override
	 *
	 * @param  FieldTable  $field
	 * @param  UserTable   $user      RETURNED populated: touch only variables related to saving this field (also when not validating for showing re-edit)
	 * @param  array       $postdata  Typically $_POST (but not necessarily), filtering required.
	 * @param  string      $reason    'edit' for save user edit, 'register' for save registration
	 */
	public function prepareFieldDataSave( &$field, &$user, &$postdata, $reason )
	{
		$this->_prepareFieldMetaSave( $field, $user, $postdata, $reason );

		$value		=	$this->getValue( $field, $user, $postdata );

		if ( $this->validate( $field, $user, $field->getString( 'name' ), $value, $postdata, $reason ) ) {
			$this->_logFieldUpdate( $field, $user, $reason, null, $value );
		}
	}

	/**
	 * Mutator:
	 * Prepares field data commit
	 * Override
	 *
	 * @param  FieldTable  $field
	 * @param  UserTable   $user      RETURNED populated: touch only variables related to saving this field (also when not validating for showing re-edit)
	 * @param  array       $postdata  Typically $_POST (but not necessarily), filtering required.
	 * @param  string      $reason    'edit' for save user edit, 'register' for save registration
	 */
	public function commitFieldDataSave( &$field, &$user, &$postdata, $reason )
	{
		$value				=	$this->getValue( $field, $user, $postdata );

		if ( $value ) {
			$groups			=	explode( '|*|', $value );

			cbArrayToInts( $groups );

			foreach ( $groups as $groupId ) {
				$row		=	new \CB\Plugin\GroupJive\Table\UserTable();

				$row->load( array( 'user_id' => $user->getInt( 'id', 0 ), 'group' => (int) $groupId ) );

				if ( $row->getInt( 'id', 0 ) ) {
					continue;
				}

				$row->set( 'user_id', $user->getInt( 'id', 0 ) );
				$row->set( 'group', (int) $groupId );

				if ( ( ! CBGroupJive::isModerator() ) && ( $row->group()->getInt( 'type', 1 ) === 2 ) ) {
					$row->set( 'status', 0 );
				} else {
					$row->set( 'status', 1 );
				}

				if ( $row->getError() || ( ! $row->check() ) ) {
					$this->_setValidationError( $field, $user, $reason, $row->getError() );
					break;
				}

				if ( $row->getError() || ( ! $row->store() ) ) {
					$this->_setValidationError( $field, $user, $reason, $row->getError() );
					break;
				}
			}
		}
	}

	/**
	 * Mutator:
	 * Prepares field data rollback
	 * Override
	 *
	 * @param  FieldTable  $field
	 * @param  UserTable   $user      RETURNED populated: touch only variables related to saving this field (also when not validating for showing re-edit)
	 * @param  array       $postdata  Typically $_POST (but not necessarily), filtering required.
	 * @param  string      $reason    'edit' for save user edit, 'register' for save registration
	 */
	public function rollbackFieldDataSave( &$field, &$user, &$postdata, $reason )
	{
		$value			=	$this->getValue( $field, $user, $postdata, true );

		if ( $value ) {
			$groups		=	explode( '|*|', $value );

			cbArrayToInts( $groups );

			foreach ( $groups as $groupId ) {
				$row	=	new \CB\Plugin\GroupJive\Table\UserTable();

				$row->load( array( 'user_id' => $user->getInt( 'id', 0 ), 'group' => (int) $groupId ) );

				if ( ! $row->getInt( 'id', 0 ) ) {
					continue;
				}

				if ( ! $row->canDelete() ) {
					$this->_setValidationError( $field, $user, $reason, $row->getError() );
					break;
				}

				if ( ! $row->delete() ) {
					$this->_setValidationError( $field, $user, $reason, $row->getError() );
					break;
				}
			}
		}
	}

	/**
	 * @param FieldTable        $field
	 * @param UserTable         $user
	 * @param bool              $raw
	 * @param bool              $joined
	 * @param int               $offset
	 * @param int               $limit
	 * @param string|array|null $search
	 * @return \stdClass[]
	 */
	private function getGroups( $field, $user, $raw = false, $joined = false, $offset = 0, $limit = 0, $search = null )
	{
		$excludeCategories		=	explode( '|*|', $field->params->getString( 'autojoin_exclude_categories' ) );
		$excludeGroups			=	explode( '|*|', $field->params->getString( 'autojoin_exclude_groups' ) );

		return CBGroupJive::getGroupOptions( $user, $raw, $excludeCategories, $excludeGroups, ( ! $joined ), $offset, $limit, $search );
	}

	/**
	 * @param FieldTable $field
	 * @param UserTable  $user
	 * @param array      $postdata
	 * @param bool       $joined
	 * @return null|string
	 */
	private function getValue( $field, $user, $postdata, $joined = false )
	{
		$value				=	Get::get( $postdata, $field->getString( 'name' ), null, GetterInterface::RAW );

		if ( ( $value === null ) || ( $value === '' ) || ( is_array( $value ) && ( count( $value ) <= 0 ) ) ) {
			$value			=	'';
		} else {
			if ( ! is_array( $value ) ) {
				$value		=	array( $value );
			}

			$options		=	$this->getGroups( $field, $user, true, $joined, 0, 0, $value );
			$value			=	array();

			foreach ( $options as $option ) {
				$value[]	=	(int) $option->value;
			}

			$value			=	$this->_implodeCBvalues( $value );
		}

		return $value;
	}
}