<?php
/**
* Community Builder (TM)
* @version $Id: $
* @package CommunityBuilder
* @copyright (C) 2004-2022 www.joomlapolis.com / Lightning MultiCom SA - and its licensors, all rights reserved
* @license http://www.gnu.org/licenses/old-licenses/gpl-2.0.html GNU/GPL version 2
*/

namespace CB\Plugin\Conditional\Trigger;

use CB\Database\Table\FieldTable;
use CB\Database\Table\UserTable;
use CBLib\Language\CBTxt;
use CBLib\Registry\Registry;
use CBLib\Registry\ParamsInterface;
use CBLib\Application\Application;
use CB\Plugin\Conditional\CBConditional;

defined('CBLIB') or die();

class FieldTrigger extends \cbFieldHandler
{

	/**
	 * @param FieldTable $field
	 * @param UserTable  $user
	 * @param string     $reason
	 * @return string
	 */
	private function fieldOverride( $field, $user, $reason )
	{
		if ( ! ( $field->getRaw( 'params' ) instanceof ParamsInterface ) ) {
			$field->set( 'params', new Registry( $field->getRaw( 'params', '' ) ) );
		}

		if ( ! in_array( $reason, array( 'profile', 'list' ), true ) ) {
			return null;
		}

		$override			=	CBTxt::T( $field->params->getRaw( 'cbconditional_display_override', '' ) );
		$override			=	trim( preg_replace( '/\[cb:(userdata +field|userfield +field)="' . preg_quote( $field->getString( 'name' ), '/' ) . '"[^]]+\]/i', '', $override ) );

		if ( ! $override ) {
			return null;
		}

		if ( $field->params->getBool( 'cbconditional_display_prepare', false ) ) {
			$override		=	Application::Cms()->prepareHtmlContentPlugins( $override, 'condition.field' );
		}

		return \CBuser::getInstance( $user->getInt( 'id', 0 ) )->replaceUserVars( $override, false, false, array( 'reason' => $reason ), false );
	}

	/**
	 * @param FieldTable[] $fields
	 * @param UserTable    $user
	 * @param string       $reason
	 * @param int          $tabid
	 * @param int|string   $fieldIdOrName
	 * @param bool         $fullAccess
	 */
	public function fieldsFetch( &$fields, &$user, $reason, $tabid, $fieldIdOrName, $fullAccess )
	{
		if ( ! $user instanceof UserTable ) {
			// fieldClass can result in a NULL value for $user so lets replace it with an actual user object like it should be:
			$user						=	new UserTable();
		}

		if ( $fieldIdOrName
			 || ( ! $fields )
			 || $user->getError()
			 || $user->getBool( '_isFieldGroup', false )
		) {
			// getFields usage provides this and in this case $user is the viewing user and not the profile owner so skip this check:
			// also skip if we're trying to get a list of available fields inside of a field group as we've a different trigger for that:
			return;
		}

		$post							=	$this->getInput()->getNamespaceRegistry( 'post' );
		$view							=	$this->getInput()->getString( 'view' );

		if ( ( ! Application::Application()->isClient( 'administrator' ) ) && ( ! $fullAccess ) ) {
			$isSave						=	( in_array( $reason, array( 'register', 'edit' ), true ) && $post->count() && in_array( $view, array( 'saveregisters', 'saveuseredit' ), true ) );
			$isProfile					=	( $reason === 'profile' );
		} elseif ( Application::Application()->isClient( 'administrator' ) && CBConditional::getGlobalParams()->getBool( 'conditions_backend', false ) && ( ! $fullAccess ) ) {
			$isSave						=	( in_array( $reason, array( 'register', 'edit' ), true ) && $post->count() && in_array( $view, array( 'apply', 'save' ), true ) );
			$isProfile					=	( $reason === 'profile' );
		} else {
			$isSave						=	false;
			$isProfile					=	false;
		}

		if ( $isSave || $isProfile ) {
			$hasDefaultOverride			=	array( 'avatar', 'canvas', 'formatname' );

			foreach ( $fields as $k => $field ) {
				if ( $field->getBool( '_noCondition', false ) ) {
					continue;
				}

				$conditioned			=	CBConditional::getTabConditional( $field->getInt( 'tabid', 0 ), $reason, $user->getInt( 'id', 0 ) );

				if ( ! $conditioned ) {
					$conditioned		=	CBConditional::getFieldConditional( $field, $reason, $user->getInt( 'id', 0 ) );
				}

				if ( ! $conditioned ) {
					continue;
				}

				if ( ! ( $field->getRaw( 'params' ) instanceof ParamsInterface ) ) {
					$field->set( 'params', new Registry( $field->getRaw( 'params' ) ) );
				}

				if ( $isSave && CBConditional::getReset( $field ) ) {
					CBConditional::resetFieldValue( $user, $field );
				}

				if ( $this->fieldOverride( $field, $user, $reason )
					 || in_array( $field->getString( 'name' ), $hasDefaultOverride, true )
				) {
					continue;
				}

				unset( $fields[$k] );
			}
		}
	}

	/**
	 * @param FieldTable $groupField
	 * @param array      $fields
	 * @param UserTable  $user
	 * @param string     $reason
	 * @param bool       $fullAccess
	 */
	public function fieldGroupsFieldsFetch( $groupField, &$fields, &$user, $reason, $fullAccess )
	{
		if ( ( ! $fields )
			 || $user->getError()
		) {
			return;
		}

		$post								=	$this->getInput()->getNamespaceRegistry( 'post' );
		$view								=	$this->getInput()->getString( 'view' );

		if ( ( ! Application::Application()->isClient( 'administrator' ) ) && ( ! $fullAccess ) ) {
			$isSave							=	( in_array( $reason, array( 'register', 'edit' ), true ) && $post->count() && in_array( $view, array( 'saveregisters', 'saveuseredit' ), true ) );
			$isProfile						=	( $reason === 'profile' );
		} elseif ( Application::Application()->isClient( 'administrator' ) && CBConditional::getGlobalParams()->getBool( 'conditions_backend', false )  && ( ! $fullAccess )) {
			$isSave							=	( in_array( $reason, array( 'register', 'edit' ), true ) && $post->count() && in_array( $view, array( 'apply', 'save' ), true ) );
			$isProfile						=	( $reason === 'profile' );
		} else {
			$isSave							=	false;
			$isProfile						=	false;
		}

		if ( $isSave || $isProfile ) {
			$hasDefaultOverride				=	array( 'avatar', 'canvas', 'formatname' );

			foreach ( $fields as $i => $rowFields ) {
				foreach ( $rowFields as $k => $field ) {
					/** @var FieldTable $field */
					if ( $field->getBool( '_noCondition', false ) ) {
						continue;
					}

					$conditioned			=	CBConditional::getTabConditional( $field->getInt( 'tabid', 0 ), $reason, $user->getInt( 'id', 0 ) );

					if ( ! $conditioned ) {
						$conditioned		=	CBConditional::getFieldConditional( $field, $reason, $user->getInt( 'id', 0 ) );
					}

					if ( ! $conditioned ) {
						continue;
					}

					if ( ! ( $field->getRaw( 'params' ) instanceof ParamsInterface ) ) {
						$field->set( 'params', new Registry( $field->getRaw( 'params' ) ) );
					}

					if ( $isSave && CBConditional::getReset( $field ) ) {
						CBConditional::resetFieldValue( $user, $field );
					}

					if ( $this->fieldOverride( $field, $user, $reason )
						 || in_array( $field->getString( 'name' ), $hasDefaultOverride, true )
					) {
						continue;
					}

					unset( $fields[$i][$k] );
				}
			}
		}
	}

	/**
	 * @param FieldTable $field
	 * @param UserTable  $user
	 * @param string     $output
	 * @param string     $formatting
	 * @param string     $reason
	 * @param int        $list_compare_types
	 * @return mixed|null|string
	 */
	public function fieldDisplay( &$field, &$user, $output, $formatting, $reason, $list_compare_types )
	{
		$return								=	null;

		if ( ( ! $field->getBool( '_noCondition', false ) ) && ( ( ! Application::Application()->isClient( 'administrator' ) ) || CBConditional::getGlobalParams()->getBool( 'conditions_backend', false ) ) ) {
			$field->set( '_noCondition', true );

			if ( $output === 'html' ) {
				$conditioned				=	CBConditional::getTabConditional( $field->getInt( 'tabid', 0 ), $reason, $user->getInt( 'id', 0 ) );

				if ( ! $conditioned ) {
					$conditioned			=	CBConditional::getFieldConditional( $field, $reason, $user->getInt( 'id', 0 ) );
				}

				if ( $conditioned ) {
					$return					=	$this->fieldOverride( $field, $user, $reason );

					if ( $return ) {
						$return				=	$this->renderFieldHtml( $field, $user, $this->_formatFieldOutput( $field->getString( 'name' ), $return, $output, false ), $output, $formatting, $reason, array() );
					} else {
						switch ( $field->getString( 'name' ) ) {
							case 'avatar':
								$return		=	\CBuser::getInstance( 0, false )->getField( 'avatar', null, $output, $formatting, $reason, $list_compare_types, true, array( '_allowProfileLink' => false ) );
								break;
							case 'canvas':
								$return		=	\CBuser::getInstance( 0, false )->getField( 'canvas', null, $output, $formatting, $reason, $list_compare_types, true );
								break;
							case 'formatname':
								$return		=	$this->renderFieldHtml( $field, $user, $this->_formatFieldOutput( $field->getString( 'name' ), $user->getFormattedName(), $output, true ), $output, $formatting, $reason, array() );
								break;
						}
					}

					if ( ! $return ) {
						$return				=	' ';
					}
				}
			} elseif ( $output === 'htmledit' ) {
				$conditioned				=	CBConditional::getTabConditional( $field->getInt( 'tabid', 0 ), $reason, $user->getInt( 'id', 0 ), ( $formatting !== 'none' ) );
				$display					=	true;

				if ( $conditioned === 2 ) {
					$display				=	false;
				} elseif ( ( $formatting === 'none' ) && $conditioned ) {
					$display				=	false;
				}

				if ( $display ) {
					$conditioned			=	CBConditional::getFieldConditional( $field, $reason, $user->getInt( 'id', 0 ), ( $formatting !== 'none' ) );

					if ( $conditioned === 2 ) {
						$display			=	false;
					} elseif ( ( $formatting === 'none' ) && $conditioned ) {
						$display			=	false;
					}
				}

				if ( ! $display ) {
					$return					=	' ';
				}
			}

			$field->set( '_noCondition', false );
		}

		return $return;
	}
}