<?php
/**
* Community Builder (TM)
* @version $Id: $
* @package CommunityBuilder
* @copyright (C) 2004-2021 www.joomlapolis.com / Lightning MultiCom SA - and its licensors, all rights reserved
* @license http://www.gnu.org/licenses/old-licenses/gpl-2.0.html GNU/GPL version 2
*/

namespace CB\Plugin\ReconfirmEmail\Trigger;

use CBLib\Application\Application;
use CBLib\Registry\GetterInterface;
use CB\Database\Table\UserTable;
use CBLib\Language\CBTxt;
use CBLib\Input\Get;
use CB\Plugin\ReconfirmEmail\Helper;
use CB\Plugin\ReconfirmEmail\Table\ReconfirmEmailTable;
use CB\Database\Table\PluginTable;
use CB\Database\Table\FieldTable;

defined('CBLIB') or die();

class UserTrigger extends \cbPluginHandler
{
	/**
	 * Deletes email changes when the user is deleted
	 *
	 * @param  UserTable $user
	 * @param  int       $status
	 */
	public function onAfterDeleteUser( $user, $status )
	{
		global $_CB_database;

		if ( $this->params->get( 'general_delete', true, GetterInterface::BOOLEAN ) ) {
			$query		=	'DELETE FROM ' . $_CB_database->NameQuote( '#__comprofiler_plugin_emails' )
						.	"\n WHERE " . $_CB_database->NameQuote( 'user_id' ) . " = " . $user->get( 'id', 0, GetterInterface::INT );
			$_CB_database->setQuery( $query );
			$_CB_database->query();
		}
	}

	/**
	 * @param string      $action
	 * @param string      $object
	 * @param string      $type
	 * @param UserTable   $user
	 * @param PluginTable $plugin
	 * @param FieldTable  $field
	 * @param mixed       $oldValues
	 * @param mixed       $newValues
	 * @param string      $reason
	 */
	public function onLogChange( $action, $object, $type, &$user, &$plugin, &$field, $oldValues, $newValues, $reason )
	{
		if ( ( $reason != 'edit' ) || ( $field->get( 'name', null, GetterInterface::STRING ) != 'email' ) ) {
			// Skip if this isn't profile edit and isn't the email field:
			return;
		}

		if ( Application::MyUser()->isGlobalModerator() || Application::Cms()->getClientId() || ( $reason != 'edit' ) ) {
			// Moderators and backend do not require reconfirmation:
			return;
		}

		$userId		=	$user->get( 'id', 0, GetterInterface::INT );

		if ( Helper::getEmailChanged( $userId ) || Helper::getEmailChanging( $userId ) ) {
			// Skip since email address has already changed or it's currently being changed:
			return;
		}

		if ( ( $oldValues == $newValues ) || ( ! $newValues ) ) {
			// Nothing to change so don't bother with reconfirmation:
			return;
		}

		Helper::setEmailChanging( $userId, $newValues, $oldValues );
	}

	/**
	 * @param FieldTable  $field
	 * @param UserTable   $user
	 * @param PluginTable $postdata
	 * @param string      $reason
	 */
	public function onAfterPrepareFieldDataSave( &$field, &$user, &$postdata, $reason )
	{
		if ( ( $reason != 'edit' ) || ( $field->get( 'name', null, GetterInterface::STRING ) != 'email' ) ) {
			// Skip if this isn't profile edit and isn't the email field:
			return;
		}

		if ( Application::MyUser()->isGlobalModerator() || Application::Cms()->getClientId() || ( $reason != 'edit' ) ) {
			// Moderators and backend do not require reconfirmation:
			return;
		}

		$userId		=	$user->get( 'id', 0, GetterInterface::INT );

		if ( ! Helper::getEmailChanging( $userId ) ) {
			// Skip since email address didn't change:
			return;
		}

		// Reset the email address:
		$user->set( 'email', Helper::getOldEmail( $userId ) );
	}

	/**
	 * @param UserTable $user
	 * @param UserTable $userDuplicate
	 * @param UserTable $oldUser
	 */
	public function onAfterUserUpdate( &$user, &$userDuplicate, $oldUser )
	{
		global $_CB_framework, $_CB_database;

		if ( Application::MyUser()->isGlobalModerator() || Application::Cms()->getClientId() ) {
			// Moderators and backend do not require reconfirmation:
			return;
		}

		$userId						=	$user->get( 'id', 0, GetterInterface::INT );

		if ( Helper::getEmailChanged( $userId ) ) {
			// Skip since email address has already been changed:
			return;
		}

		if ( ! Helper::getEmailChanging( $userId ) ) {
			$newEmail				=	$user->get( 'email', null, GetterInterface::STRING );
			$oldEmail				=	$oldUser->get( 'email', null, GetterInterface::STRING );

			if ( $newEmail != $oldEmail ) {
				// Our earlier behavior was interrupted so lets note the change now and reset below:
				Helper::setEmailChanging( $userId, $newEmail, $oldEmail );
			} else {
				// Skip since email address isn't being changed:
				return;
			}
		}

		$newEmail					=	Helper::getNewEmail( $userId );
		$oldEmail					=	Helper::getOldEmail( $userId );

		if ( ( $newEmail == $oldEmail ) || ( ! $newEmail ) || ( ! $oldEmail ) ) {
			// Nothing to change so don't bother with reconfirmation:
			Helper::setEmailChanging( $userId, null );

			return;
		} elseif ( $newEmail == $user->get( 'email', null, GetterInterface::STRING ) ) {
			// Something has added the new email back into the user object so we need to force it to reset here:
			$user->storeDatabaseValue( 'email', $oldEmail, false );
			$user->set( 'email', $oldEmail );
		}

		$row						=	new ReconfirmEmailTable();

		$row->load( array( 'user_id' => $userId, 'to' => $newEmail, 'status' => 'P' ) );

		$message					=	CBTxt::T( $this->params->get( 'reconfirm_changed', 'Your email address has changed and requires reconfirmation. Please check your new email address for your confirmation email.', GetterInterface::HTML ) );

		if ( $row->get( 'id', 0, GetterInterface::INT ) ) {
			// Already have a pending reconfirm for this email address so just stop here:
			if ( $message ) {
				$_CB_framework->enqueueMessage( $message, 'message' );
			}

			Helper::setEmailChanging( $userId, null );

			return;
		}

		$cbUser						=	\CBuser::getInstance( $userId, false );

		// Cancel all other pending change requests as we only want 1 active change request:
		$query						=	'UPDATE ' . $_CB_database->NameQuote( '#__comprofiler_plugin_emails' )
									.	"\n SET " . $_CB_database->NameQuote( 'status' ) . " = " . $_CB_database->Quote( 'X' )
									.	"\n WHERE " . $_CB_database->NameQuote( 'user_id' ) . " = " . $userId
									.	"\n AND " . $_CB_database->NameQuote( 'status' ) . " = " . $_CB_database->Quote( 'P' );
		$_CB_database->setQuery( $query );
		$_CB_database->query();

		$row->set( 'user_id', $userId );
		$row->set( 'from', $oldEmail );
		$row->set( 'to', $newEmail );

		if ( $row->getError() || ( ! $row->check() ) ) {
			$_CB_framework->enqueueMessage( CBTxt::T( 'RECONFIRM_EMAIL_FAILED_TO_SAVE', 'Email address failed to save! Error: [error]', array( '[error]' => $row->getError() ) ), 'error' );

			return;
		}

		if ( $row->getError() || ( ! $row->store() ) ) {
			$_CB_framework->enqueueMessage( CBTxt::T( 'RECONFIRM_EMAIL_FAILED_TO_SAVE', 'Email address failed to save! Error: [error]', array( '[error]' => $row->getError() ) ), 'error' );

			return;
		}

		$reconfirm					=	$_CB_framework->pluginClassUrl( $this->element, true, array( 'confirmcode' => $row->get( 'code', null, GetterInterface::STRING ) ) );
		$cancel						=	$_CB_framework->pluginClassUrl( $this->element, true, array( 'confirmcode' => $row->get( 'code', null, GetterInterface::STRING ), 'action' => 'cancel' ) );

		$extra						=	array(	'reconfirm'	=>	$reconfirm,
												'cancel'	=>	$cancel,
												'new_email'	=>	$newEmail,
												'old_email'	=>	$oldEmail
											);

		$savedLanguage				=	CBTxt::setLanguage( $user->getUserLanguage() );

		$mailFromName				=	Get::clean( $cbUser->replaceUserVars( $this->params->get( 'reconfirm_from_name', null, GetterInterface::RAW ), true, false, $extra ), GetterInterface::STRING );
		$mailFromAddr				=	Get::clean( $cbUser->replaceUserVars( $this->params->get( 'reconfirm_from_address', null, GetterInterface::RAW ), true, false, $extra ), GetterInterface::STRING );
		$mailSubject				=	Get::clean( $cbUser->replaceUserVars( $this->params->get( 'reconfirm_subject', 'Your email address requires confirmation', GetterInterface::RAW ), true, false, $extra ), GetterInterface::STRING );
		$mailBody					=	Get::clean( $cbUser->replaceUserVars( $this->params->get( 'reconfirm_body', 'The email address attached to your account [username] has changed to [new_email] and requires confirmation.<br><br>You can confirm your email address by clicking on the following link:<br><a href="[reconfirm]">[reconfirm]</a><br><br>If this was done in error please contact administration or cancel by <a href="[cancel]">clicking here</a>.', GetterInterface::RAW ), false, false, $extra ), GetterInterface::HTML );
		$mailCC						=	Get::clean( $cbUser->replaceUserVars( $this->params->get( 'reconfirm_cc', null, GetterInterface::RAW ), true, false, $extra ), GetterInterface::STRING );
		$mailBCC					=	Get::clean( $cbUser->replaceUserVars( $this->params->get( 'reconfirm_bcc', null, GetterInterface::RAW ), true, false, $extra ), GetterInterface::STRING );
		$mailAttachments			=	Get::clean( $cbUser->replaceUserVars( $this->params->get( 'reconfirm_attachments', null, GetterInterface::RAW ), true, false, $extra ), GetterInterface::STRING );

		CBTxt::setLanguage( $savedLanguage );

		if ( $mailCC ) {
			$mailCC					=	preg_split( '/ *, */', $mailCC );
		}

		if ( $mailBCC ) {
			$mailBCC				=	preg_split( '/ *, */', $mailBCC );
		}

		if ( $mailAttachments ) {
			$mailAttachments		=	preg_split( '/ *, */', $mailAttachments );
		}

		$cbNotification				=	new \cbNotification();

		$cbNotification->sendFromSystem( $newEmail, $mailSubject, $mailBody, false, 1, $mailCC, $mailBCC, $mailAttachments, $extra, true, $mailFromName, $mailFromAddr );

		if ( $this->params->get( 'reconfirm_changed_notify', false, GetterInterface::BOOLEAN ) ) {
			$mailSubject			=	Get::clean( $cbUser->replaceUserVars( $this->params->get( 'reconfirm_changed_subject', 'Your email address has changed', GetterInterface::RAW ), true, false, $extra ), GetterInterface::STRING );
			$mailBody				=	Get::clean( $cbUser->replaceUserVars( $this->params->get( 'reconfirm_changed_body', 'The email address attached to your account [username] has changed to [new_email] and requires confirmation.<br><br>Please check the new email address for a link to confirm this change. If this was done in error please contact administration or cancel by <a href="[cancel]">clicking here</a>.', GetterInterface::RAW ), false, false, $extra ), GetterInterface::HTML );
			$mailCC					=	Get::clean( $cbUser->replaceUserVars( $this->params->get( 'reconfirm_changed_cc', null, GetterInterface::RAW ), true, false, $extra ), GetterInterface::STRING );
			$mailBCC				=	Get::clean( $cbUser->replaceUserVars( $this->params->get( 'reconfirm_changed_bcc', null, GetterInterface::RAW ), true, false, $extra ), GetterInterface::STRING );
			$mailAttachments		=	Get::clean( $cbUser->replaceUserVars( $this->params->get( 'reconfirm_changed_attachments', null, GetterInterface::RAW ), true, false, $extra ), GetterInterface::STRING );

			$cbNotification			=	new \cbNotification();

			$cbNotification->sendFromSystem( $oldEmail, $mailSubject, $mailBody, false, 1, $mailCC, $mailBCC, $mailAttachments, $extra, true, $mailFromName, $mailFromAddr );
		}

		if ( $message ) {
			$_CB_framework->enqueueMessage( $message, 'message' );
		}

		Helper::setEmailChanging( $userId, null );
	}
}