<?php
/**
* Community Builder (TM)
* @version $Id: $
* @package CommunityBuilder
* @copyright (C) 2004-2021 www.joomlapolis.com / Lightning MultiCom SA - and its licensors, all rights reserved
* @license http://www.gnu.org/licenses/old-licenses/gpl-2.0.html GNU/GPL version 2
*/

namespace CB\Plugin\ReconfirmEmail\Table;

use CB\Plugin\ReconfirmEmail\Helper;
use CBLib\Application\Application;
use CBLib\Database\Table\Table;
use CBLib\Language\CBTxt;
use CBLib\Registry\GetterInterface;

defined('CBLIB') or die();

class ReconfirmEmailTable extends Table
{
	/** @var int  */
	public $id				=	null;
	/** @var int  */
	public $user_id			=	null;
	/** @var string  */
	public $from			=	null;
	/** @var string  */
	public $to				=	null;
	/** @var string  */
	public $code			=	null;
	/** @var string  */
	public $date			=	null;
	/** @var string  */
	public $status			=	null;

	/**
	 * Table name in database
	 *
	 * @var string
	 */
	protected $_tbl			=	'#__comprofiler_plugin_emails';

	/**
	 * Primary key(s) of table
	 *
	 * @var string
	 */
	protected $_tbl_key		=	'id';

	/**
	 * @return bool
	 */
	public function check()
	{
		if ( $this->get( 'from', null, GetterInterface::STRING ) == '' ) {
			$this->setError( CBTxt::T( 'Old email address not specified!' ) );

			return false;
		} elseif ( $this->get( 'to', null, GetterInterface::STRING ) == '' ) {
			$this->setError( CBTxt::T( 'New email address not specified!' ) );

			return false;
		} elseif ( ! $this->get( 'user_id', 0, GetterInterface::INT ) ) {
			$this->setError( CBTxt::T( 'User not specified!' ) );

			return false;
		}

		return true;
	}

	/**
	 * @param bool $updateNulls
	 * @return bool
	 */
	public function store( $updateNulls = false )
	{
		global $_CB_framework, $_PLUGINS;

		$new	=	( $this->get( 'id', 0, GetterInterface::INT ) ? false : true );
		$old	=	new self();

		$this->set( 'date', $this->get( 'date', Application::Database()->getUtcDateTime(), GetterInterface::STRING ) );

		if ( version_compare( phpversion(), '7.0.0', '>=' ) ) {
			$this->set( 'code', $this->get( 'code', bin2hex( random_bytes( 32 ) ), GetterInterface::STRING ) );
		} else {
			$this->set( 'code', $this->get( 'code', hash( 'sha256', $this->get( 'id', 0, GetterInterface::INT ) . $this->get( 'from', null, GetterInterface::STRING ) . $this->get( 'to', null, GetterInterface::STRING ) . $_CB_framework->getCfg( 'secret' ) ), GetterInterface::STRING ) );
		}

		$this->set( 'status', $this->get( 'status', 'P', GetterInterface::STRING ) );

		if ( ! $new ) {
			$old->load( $this->get( 'id', 0, GetterInterface::INT ) );

			$_PLUGINS->trigger( 'reconfirm_onBeforeUpdateReconfirmEmail', array( &$this, $old ) );
		} else {
			$_PLUGINS->trigger( 'reconfirm_onBeforeCreateReconfirmEmail', array( &$this ) );
		}

		if ( ! parent::store( $updateNulls ) ) {
			return false;
		}

		if ( ! $new ) {
			$_PLUGINS->trigger( 'reconfirm_onAfterUpdateReconfirmEmail', array( $this, $old ) );
		} else {
			$_PLUGINS->trigger( 'reconfirm_onAfterCreateReconfirmEmail', array( $this ) );
		}

		return true;
	}

	/**
	 * @param null|int $id
	 * @return bool
	 */
	public function delete( $id = null )
	{
		global $_PLUGINS;

		$_PLUGINS->trigger( 'reconfirm_onBeforeDeleteReconfirmEmail', array( &$this ) );

		if ( ! parent::delete( $id ) ) {
			return false;
		}

		$_PLUGINS->trigger( 'reconfirm_onAfterDeleteReconfirmEmail', array( $this ) );

		return true;
	}

	/**
	 * Confirms the email change
	 *
	 * @return bool
	 */
	public function confirm()
	{
		if ( $this->get( 'status', null, GetterInterface::STRING ) == 'A' ) {
			return true;
		}

		$user	=	\CBuser::getUserDataInstance( $this->get( 'user_id', 0, GetterInterface::INT ) );

		if ( ! $user->storeDatabaseValue( 'email', $this->get( 'to', null, GetterInterface::STRING ) ) ) {
			$this->setError( $user->getError() );

			return false;
		}

		$this->set( 'status', 'A' );

		if ( $this->getError() || ( ! $this->check() ) ) {
			return false;
		}

		if ( $this->getError() || ( ! $this->store() ) ) {
			return false;
		}

		return true;
	}

	/**
	 * Cancels the email change
	 *
	 * @return bool
	 */
	public function cancel()
	{
		if ( $this->get( 'status', null, GetterInterface::STRING ) == 'X' ) {
			return true;
		}

		$this->set( 'status', 'X' );

		if ( $this->getError() || ( ! $this->check() ) ) {
			return false;
		}

		if ( $this->getError() || ( ! $this->store() ) ) {
			return false;
		}

		return true;
	}

	/**
	 * Checks if email change has expired and sets expired status if it has
	 *
	 * @return bool
	 */
	public function expired()
	{
		static $cache		=	array();

		if ( $this->get( 'status', null, GetterInterface::STRING ) == 'E' ) {
			return true;
		}

		$timeframe			=	Helper::getGlobalParams()->get( 'reconfirm_expired_timeframe', '+1 WEEK', GetterInterface::STRING );

		if ( ! $timeframe ) {
			return false;
		}

		$id					=	$this->get( 'id', 0, GetterInterface::INT );

		if ( ! isset( $cache[$id] ) ) {
			$expired		=	( Application::Date( 'now', 'UTC' )->getTimestamp() >= Application::Date( $this->get( 'date', null, GetterInterface::STRING ), 'UTC' )->modify( strtoupper( $timeframe ) )->getTimestamp() );

			if ( $expired ) {
				$this->set( 'status', 'E' );

				$this->store();
			}

			$cache[$id]		=	$expired;
		}

		return $cache[$id];
	}
}