<?php
/**
* Community Builder (TM)
* @version $Id: $
* @package CommunityBuilder
* @copyright (C) 2004-2021 www.joomlapolis.com / Lightning MultiCom SA - and its licensors, all rights reserved
* @license http://www.gnu.org/licenses/old-licenses/gpl-2.0.html GNU/GPL version 2
*/

namespace CB\Plugin\Privacy\Trigger;

use CB\Database\Table\FieldTable;
use CB\Database\Table\UserTable;
use CB\Plugin\PMS\Table\MessageTable;
use CBLib\Application\Application;
use CBLib\Language\CBTxt;
use CBLib\Registry\GetterInterface;
use CB\Plugin\Privacy\CBPrivacy;
use CB\Plugin\Privacy\Table\PrivacyTable;
use CB\Plugin\Privacy\Table\ClosedTable;
use CBLib\Registry\ParamsInterface;
use CB\Plugin\Privacy\Privacy;
use CBLib\Registry\Registry;
use CB\Database\Table\TabTable;
use CB\Plugin\Privacy\Table\BlockedTable;

defined('CBLIB') or die();

class UserTrigger extends \cbPluginHandler
{

	/**
	 * @param int    $uid
	 * @param string $msg
	 */
	public function getProfile( $uid, &$msg )
	{
		if ( ( ! Application::Cms()->getClientId() ) && ( ! Application::MyUser()->isGlobalModerator() ) ) {
			$user				=	loadComprofilerUser( $uid );

			if ( $user && ( Application::MyUser()->getUserId() != $user->get( 'id', 0, GetterInterface::INT ) ) ) {
				if ( ( ( ! CBPrivacy::checkProfileDisplayAccess( $user ) ) && ( ! $this->params->get( 'profile_direct_access', false, GetterInterface::BOOLEAN ) ) ) || in_array( Application::MyUser()->getUserId(), CBPrivacy::getBlocked( $user ) ) ) {
					$url		=	$this->params->getString( 'profile_direct_url', '' );

					if ( $url ) {
						$url	=	\CBuser::getInstance( $user->get( 'id', 0, GetterInterface::INT ) )->replaceUserVars( $url );
					}

					$msg		=	$this->params->getString( 'profile_direct_message', '' );

					if ( $msg ) {
						$msg	=	\CBuser::getInstance( $user->get( 'id', 0, GetterInterface::INT ) )->replaceUserVars( $msg );
					}

					if ( $url ) {
						cbRedirect( $url, $msg, $this->params->getString( 'profile_direct_type', 'message' ) );
					}

					if ( ! $msg ) {
						$msg	=	CBTxt::Th( 'UE_NOT_AUTHORIZED', 'You are not authorized to view this page!' );
					}
				}
			}
		}
	}

	/**
	 * @param UserTable $rowFrom
	 * @param UserTable $rowTo
	 * @param int       $html
	 * @param int       $allowPublic
	 * @param string    $name
	 * @param string    $email
	 * @param string    $subject
	 * @param string    $message
	 */
	public function getEmailForm( &$rowFrom, &$rowTo, $html, &$allowPublic, &$name, &$email, &$subject, &$message )
	{
		global $_CB_framework;

		$blocked				=	false;

		if ( ( ! Application::Cms()->getClientId() )
			 && ( ! Application::MyUser()->isGlobalModerator() )
			 && ( Application::MyUser()->getUserId() != $rowTo->get( 'id', 0, GetterInterface::INT ) )
		) {
			if ( ( ! CBPrivacy::checkFieldDisplayAccess( 'email', $rowTo ) ) || ( $blocked = in_array( Application::MyUser()->getUserId(), CBPrivacy::getBlocked( $rowTo->get( 'id', 0, GetterInterface::INT ) ) ) ) ) {
				$redirect		=	'index.php';

				if ( $this->params->get( 'profile_direct_access', false, GetterInterface::BOOLEAN ) && ( ! $blocked ) ) {
					$redirect	=	$_CB_framework->userProfileUrl( $rowTo->get( 'id', 0, GetterInterface::INT ), false );
				}

				cbRedirect( $redirect, CBTxt::T( 'You do not have permission to email this user.' ), 'error' );
			} elseif ( in_array( $rowTo->get( 'id', 0, GetterInterface::INT ), CBPrivacy::getBlocked() ) ) {
				cbRedirect( $_CB_framework->userProfileUrl( $rowTo->get( 'id', 0, GetterInterface::INT ), false ), CBTxt::T( 'You can not email a user you have blocked.' ), 'error' );
			}
		}
	}

	/**
	 * @param MessageTable $row
	 * @param array        $input
	 * @param UserTable    $user
	 */
	public function getPMForm( &$row, &$input, $user )
	{
		global $_CB_framework;

		if ( ( ! Application::Cms()->getClientId() )
			 && ( ! Application::MyUser()->isGlobalModerator() )
			 && ( ! $row->get( 'id', 0, GetterInterface::INT ) )
		) {
			$to						=	$this->input( 'get/to', null, GetterInterface::STRING );
			$userId					=	0;

			if ( $to ) {
				if ( is_numeric( $to ) ) {
					$userId			=	(int) $to;
				} elseif ( is_string( $to ) ) {
					$toUser			=	new UserTable();

					$toUser->loadByUsername( trim( $to ) );

					$userId			=	$toUser->get( 'id', 0, GetterInterface::INT );
				}
			}

			$blocked				=	false;

			if ( $userId && ( Application::MyUser()->getUserId() != $userId ) ) {
				if ( ( ! CBPrivacy::checkFieldDisplayAccess( 'pm', $userId ) ) || ( $blocked = in_array( Application::MyUser()->getUserId(), CBPrivacy::getBlocked( $userId ) ) ) ) {
					$redirect		=	'index.php';

					if ( $this->params->get( 'profile_direct_access', false, GetterInterface::BOOLEAN ) && ( ! $blocked ) ) {
						$redirect	=	$_CB_framework->userProfileUrl( $userId, false );
					}

					cbRedirect( $redirect, CBTxt::T( 'You do not have permission to message this user.' ), 'error' );
				} elseif ( in_array( $userId, CBPrivacy::getBlocked() ) ) {
					cbRedirect( $_CB_framework->userProfileUrl( $userId, false ), CBTxt::T( 'You can not message a user you have blocked.' ), 'error' );
				}
			}
		}
	}

	/**
	 * @param MessageTable $row
	 * @return bool
	 */
	public function sendPM( &$row )
	{
		if ( ( ! Application::Cms()->getClientId() )
			 && ( ! Application::MyUser()->isGlobalModerator() )
			 && $row->get( 'to_user', 0, GetterInterface::INT )
			 && ( Application::MyUser()->getUserId() != $row->get( 'to_user', 0, GetterInterface::INT ) )
			 && ( ! $row->get( 'from_system', 0, GetterInterface::INT ) )
		) {
			if ( ( ! CBPrivacy::checkFieldDisplayAccess( 'pm', $row->get( 'to_user', 0, GetterInterface::INT ) ) ) || in_array( Application::MyUser()->getUserId(), CBPrivacy::getBlocked( $row->get( 'to_user', 0, GetterInterface::INT ) ) ) ) {
				$row->setError( CBTxt::T( 'You do not have permission to message this user.' ) );

				return false;
			} elseif ( in_array( $row->get( 'to_user', 0, GetterInterface::INT ), CBPrivacy::getBlocked() ) ) {
				$row->setError( CBTxt::T( 'You can not message a user you have blocked.' ) );

				return false;
			}
		}

		return true;
	}

	/**
	 * @param MessageTable $row
	 * @param string       $avatar
	 * @param string       $name
	 * @param array        $menu
	 * @param UserTable    $user
	 */
	public function readPM( &$row, &$avatar, &$name, &$menu, $user )
	{
		global $_CB_framework;

		if ( ( ! Application::Cms()->getClientId() )
			 && $row->get( 'from_user', 0, GetterInterface::INT )
			 && ( ! Application::User( $row->get( 'from_user', 0, GetterInterface::INT ) )->isGlobalModerator() )
			 && ( Application::MyUser()->getUserId() != $row->get( 'from_user', 0, GetterInterface::INT ) )
			 && ( ! $row->get( 'from_system', 0, GetterInterface::INT ) )
		) {
			if ( Application::Input()->getString( 'func', '' ) != 'modal' ) {
				$returnUrl	=	CBPrivacy::getReturn();
			} else {
				$returnUrl	=	base64_encode( $_CB_framework->userProfileUrl( $user->get( 'id', 0, GetterInterface::INT ), false ) );
			}

			if ( in_array( $row->get( 'from_user', 0, GetterInterface::INT ), CBPrivacy::getBlocked() ) ) {
				$menu[]		=	'<li class="pmMessagesMenuItem" role="presentation"><a href="' . $_CB_framework->pluginClassUrl( $this->element, true, array( 'action' => 'blocked', 'func' => 'unblock', 'id' => $row->get( 'from_user', 0, GetterInterface::INT ), 'return' => $returnUrl ) ) . '" class="dropdown-item" role="menuitem"><span class="fa fa-check-circle-o"></span> ' . CBTxt::T( 'UNBLOCK_MESSAGES_FROM_USER', 'Unblock Messages from <span class="text-info">[user]</span>', array( '[user]' => \CBuser::getInstance( $row->get( 'from_user', 0, GetterInterface::INT ), false )->getField( 'formatname', null, 'html', 'none', 'profile', 0, true ) ) ) . '</a></li>';
				$menu[]		=	'<li class="pmMessagesMenuItem"><div class="dropdown-divider"></div></li>';
			} else {
				$menu[]		=	'<li class="pmMessagesMenuItem" role="presentation"><a href="javascript: void(0);" onclick="cbjQuery.cbconfirm( \'' . addslashes( CBTxt::T( 'Are you sure you want to block this user from accessing your profile and communicating with you?' ) ) . '\' ).done( function() { window.location.href = \'' . addslashes( $_CB_framework->pluginClassUrl( $this->element, true, array( 'action' => 'blocked', 'func' => 'block', 'id' => $row->get( 'from_user', 0, GetterInterface::INT ), 'return' => $returnUrl ) ) ) . '\'; })" class="dropdown-item" role="menuitem"><span class="fa fa-ban"></span> ' . CBTxt::T( 'BLOCK_MESSAGES_FROM_USER', 'Block Messages from <span class="text-info">[user]</span>', array( '[user]' => \CBuser::getInstance( $row->get( 'from_user', 0, GetterInterface::INT ), false )->getField( 'formatname', null, 'html', 'none', 'profile', 0, true ) ) ) . '</a></li>';
				$menu[]		=	'<li class="pmMessagesMenuItem"><div class="dropdown-divider"></div></li>';
			}
		}
	}

	/**
	 * @param int    $userid
	 * @param int    $connectionid
	 * @param string $msg
	 * @param string $umsg
	 * @param string $act
	 */
	public function requestConnection( $userid, $connectionid, &$msg, &$umsg, $act )
	{
		if ( ( ! Application::Cms()->getClientId() )
			 && ( ! Application::MyUser()->isGlobalModerator() )
			 && $connectionid
			 && ( Application::MyUser()->getUserId() != $connectionid )
		) {
			if ( ( ! CBPrivacy::checkProfileDisplayAccess( $connectionid ) ) || in_array( Application::MyUser()->getUserId(), CBPrivacy::getBlocked( $connectionid ) ) ) {
				$msg	=	CBTxt::T( 'You do not have permission to connect with this user.' );
			} elseif ( in_array( $connectionid, CBPrivacy::getBlocked() ) ) {
				$msg	=	CBTxt::T( 'You can not connect with a user you have blocked.' );
			}
		}
	}

	/**
	 * @param UserTable $user
	 */
	public function deletePrivacy( $user )
	{
		global $_CB_database;

		$params				=	CBPrivacy::getGlobalParams();

		if ( $params->get( 'general_delete', true, GetterInterface::BOOLEAN ) ) {
			$query			=	'SELECT *'
							.	"\n FROM " . $_CB_database->NameQuote( '#__comprofiler_plugin_privacy' )
							.	"\n WHERE " . $_CB_database->NameQuote( 'user_id' ) . " = " . $user->get( 'id', 0, GetterInterface::INT );
			$_CB_database->setQuery( $query );
			$privacy		=	$_CB_database->loadObjectList( null, '\CB\Plugin\Privacy\Table\PrivacyTable', array( $_CB_database ) );

			/** @var PrivacyTable[] $privacy */
			foreach ( $privacy as $rule ) {
				$rule->delete();
			}

			$query			=	'SELECT *'
							.	"\n FROM " . $_CB_database->NameQuote( '#__comprofiler_plugin_privacy_blocked' )
							.	"\n WHERE ( " . $_CB_database->NameQuote( 'user_id' ) . " = " . $user->get( 'id', 0, GetterInterface::INT )
							.	' OR ' . $_CB_database->NameQuote( 'block' ) . ' = ' . $user->get( 'id', 0, GetterInterface::INT ) . ' )';
			$_CB_database->setQuery( $query );
			$blocked		=	$_CB_database->loadObjectList( null, '\CB\Plugin\Privacy\Table\BlockedTable', array( $_CB_database ) );

			/** @var BlockedTable[] $blocked */
			foreach ( $blocked as $block ) {
				$block->delete();
			}

			if ( $params->get( 'general_delete_closed', true, GetterInterface::BOOLEAN ) ) {
				$query		=	'SELECT *'
							.	"\n FROM " . $_CB_database->NameQuote( '#__comprofiler_plugin_privacy_closed' )
							.	"\n WHERE " . $_CB_database->NameQuote( 'user_id' ) . " = " . $user->get( 'id', 0, GetterInterface::INT );
				$_CB_database->setQuery( $query );
				$closed		=	$_CB_database->loadObjectList( null, '\CB\Plugin\Privacy\Table\ClosedTable', array( $_CB_database ) );

				/** @var ClosedTable[] $closed */
				foreach ( $closed as $close ) {
					$close->delete();
				}
			}
		}
	}

	/**
	 * @param UserTable $user
	 * @param int       $ui
	 * @param string    $return
	 * @return null|array
	 */
	public function checkDisabled( &$user, $ui, &$return )
	{
		global $_CB_framework, $_PLUGINS;

		if ( $_PLUGINS->is_errors() || ( ! $user->get( 'block', 0, GetterInterface::INT ) ) || ( ! $user->get( 'approved', 1, GetterInterface::INT ) ) || ( ! $user->get( 'confirmed', 1, GetterInterface::INT ) ) ) {
			return null;
		}

		$closed							=	new ClosedTable();

		$closed->load( array( 'user_id' => $user->get( 'id', 0, GetterInterface::INT ), 'type' => 'disable' ) );

		if ( ! $closed->get( 'id', 0, GetterInterface::INT ) ) {
			if ( ! CBPrivacy::getGlobalParams()->get( 'disabled_enable', true, GetterInterface::BOOLEAN ) ) {
				// Re-enable isn't allowed so don't bother checking for pending re-enable:
				return null;
			}

			$closed->load( array( 'user_id' => $user->get( 'id', 0, GetterInterface::INT ), 'type' => 'pending' ) );

			if ( $closed->get( 'id', 0, GetterInterface::INT ) ) {
				$user->set( 'block', 0 );

				if ( $user->storeBlock() ) {
					$closed->delete();

					$notification		=	new \cbNotification();

					$extra				=	array(	'ip_address'	=>	Application::Input()->getRequestIP(),
													'reason'		=>	$closed->getString( 'reason', '' ),
													'date'			=>	cbFormatDate( $closed->getString( 'date', '' ) )
												);

					if ( $this->params->get( 'enable_notify', true, GetterInterface::BOOLEAN ) ) {
						$cbUser			=	\CBuser::getInstance( $user->get( 'id', 0, GetterInterface::INT ), false );

						$subject		=	$cbUser->replaceUserVars( CBTxt::T( 'User Account Re-Enabled' ), true, false, $extra, false );
						$body			=	$cbUser->replaceUserVars( CBTxt::T( 'Name: [name]<br />Username: [username]<br />Email: [email]<br />IP Address: [ip_address]<br />Date: [date]<br /><br />' ), false, false, $extra, false );

						if ( $subject && $body ) {
							$notification->sendToModerators( $subject, $body, false, 1 );
						}
					}

					$savedLanguage		=	CBTxt::setLanguage( $user->getUserLanguage() );

					$subject			=	CBTxt::T( 'Your Account has been Re-Enabled' );
					$body				=	CBTxt::T( 'This is a notice that your account [username] on [siteurl] has been re-enabled.' );

					CBTxt::setLanguage( $savedLanguage );

					if ( $subject && $body ) {
						$notification->sendFromSystem( $user, $subject, $body, true, 1, null, null, null, $extra );
					}
				}
			}

			return null;
		}

		$reason							=	$closed->getString( 'reason', '' );

		if ( ! CBPrivacy::getGlobalParams()->get( 'disabled_enable', true, GetterInterface::BOOLEAN ) ) {
			if ( $reason ) {
				$message				=	CBTxt::T( 'PRIVACY_ACCOUNT_PERMANENT_DISABLED_REASON', '<div class="alert alert-warning">Your account is disabled. Reason: [reason].</div>', array( '[reason]' => $reason ) );
			} else {
				$message				=	CBTxt::T( 'PRIVACY_ACCOUNT_PERMANENT_DISABLED', '<div class="alert alert-warning">Your account is disabled.</div>' );
			}
		} else {
			$enableUrl					=	$_CB_framework->pluginClassUrl( $this->element, true, array( 'action' => 'disable', 'func' => 'enable', 'id' => $user->get( 'id', 0, GetterInterface::INT ) ) );

			if ( $reason ) {
				$message				=	CBTxt::T( 'PRIVACY_ACCOUNT_DISABLED_REASON', '<div class="alert alert-warning">Your account is disabled. Reason: [reason]. <a href="[enable_url]">Please click here if you would like to re-enable your account.</a></div>', array( '[reason]' => $reason, '[enable_url]' => $enableUrl ) );
			} else {
				$message				=	CBTxt::T( 'PRIVACY_ACCOUNT_DISABLED', '<div class="alert alert-warning">Your account is disabled. <a href="[enable_url]">Please click here if you would like to re-enable your account.</a></div>', array( '[enable_url]' => $enableUrl ) );
			}
		}

		return array( 'messagesToUser' => $message );
	}

	/**
	 * Saves privacy after successful registration
	 *
	 * @param UserTable $user
	 */
	public function saveRegistrationPrivacy( $user )
	{
		$this->savePrivacy( $user, 'register' );
	}

	/**
	 * Saves privacy after successful profile edit
	 *
	 * @param UserTable $user
	 */
	public function saveEditPrivacy( $user )
	{
		$this->savePrivacy( $user, 'edit' );
	}

	/**
	 * Handles saving tab and field privacy after a successful registration or profile edit
	 *
	 * @param UserTable $user
	 * @param string    $reason
	 */
	private function savePrivacy( $user, $reason )
	{
		if ( ! $user->get( 'id', 0, GetterInterface::INT ) ) {
			return;
		}

		$cbTabs		=	new \cbTabs( 0, Application::Cms()->getClientId(), null, false );

		foreach ( $cbTabs->_getTabsDb( $user, $reason ) as $tab ) {
			if ( ! $tab instanceof TabTable ) {
				continue;
			}

			if ( ! $tab->params instanceof ParamsInterface ) {
				$tab->params		=	new Registry( $tab->params );
			}

			$display				=	$tab->params->get( 'cbprivacy_display', 0, GetterInterface::INT );

			if ( ( $reason == 'register' ) && ( ! $tab->params->get( 'cbprivacy_display_reg', false, GetterInterface::BOOLEAN ) ) ) {
				$display			=	0;
			}

			if ( ( $display == 1 ) || ( ( $display == 2 ) && Application::MyUser()->isGlobalModerator() ) ) {
				$asset				=	str_replace( '[tab_id]', $tab->get( 'tabid', 0, GetterInterface::INT ), $tab->params->getString( 'cbprivacy_asset', '' ) );

				if ( ! $asset ) {
					$asset			=	'profile.tab.' . $tab->get( 'tabid', 0, GetterInterface::INT );
				}

				$privacy			=	new Privacy( $asset, $user );

				if ( $display == 2 ) {
					$privacy->set( 'options_moderator', true );
				}

				if ( $reason == 'register' ) {
					$privacy->set( 'guest', true );
				}

				$privacy->parse( $tab->params, 'privacy_' );

				if ( in_array( $reason, array( 'edit', 'register' ) ) ) {
					$privacy->set( 'ajax', false );
				}

				$privacy->set( 'tab', $tab->get( 'tabid', 0, GetterInterface::INT ) );

				if ( $privacy->getString( 'layout', 'button' ) == 'hidden' ) {
					continue;
				}

				$privacy->privacy( 'save' );
			}
		}

		foreach ( $cbTabs->_getTabFieldsDb( null, $user, $reason, null, false ) as $field ) {
			if ( ( ! $field instanceof FieldTable )
				 || $field->get( '_noPrivacy', false, GetterInterface::BOOLEAN )
				 || ( ! $field->get( 'profile', 1, GetterInterface::INT ) )
				 || ( $field->getString( 'name', '' ) == 'privacy_profile' )
				 || $field->get( '_isGrouped', false, GetterInterface::BOOLEAN )
			) {
				continue;
			}

			$this->saveFieldPrivacy( $field, $user, $reason );
		}
	}

	/**
	 * @param FieldTable $fieldGroup
	 * @param array      $fields
	 * @param array      $oldValues
	 * @param array      $newValues
	 * @param UserTable  $user
	 * @param array      $postdata
	 * @param string     $reason
	 */
	public function saveFieldGroupPrivacy( $fieldGroup, &$fields, $oldValues, &$newValues, &$user, &$postdata, $reason )
	{
		global $_CB_database;

		if ( ! $user->get( 'id', 0, GetterInterface::INT ) ) {
			return;
		}

		$privacyFields	=	array();
		$deletedRows	=	array_diff_key( $oldValues, $newValues );

		foreach ( $fields as $i => $rowFields ) {
			foreach ( $rowFields as $k => $field ) {
				/** @var FieldTable $field */
				if ( $field->get( '_noPrivacy', false, GetterInterface::BOOLEAN )
					 || ( ! $field->get( 'profile', 1, GetterInterface::INT ) )
					 || ( $field->getString( 'name', '' ) == 'privacy_profile' )
				) {
					continue;
				}

				if ( $this->saveFieldPrivacy( $field, $user, $reason ) && ( $i == 0 ) ) {
					$privacyFields[]	=	$field;
				}
			}
		}

		if ( $deletedRows ) {
			// We've old privacy rows we need to delete:
			$assets				=	array();

			foreach ( $privacyFields as $privacyField ) {
				$privacyFieldId	=	$privacyField->get( 'fieldid', 0, GetterInterface::INT );
				$asset			=	str_replace( '[field_id]', $privacyFieldId, $privacyField->params->getString( 'cbprivacy_asset', '' ) );

				if ( $asset ) {
					// Don't delete custom asset usages:
					continue;
				}

				foreach ( $deletedRows as $i => $rowFields ) {
					$asset		=	'profile.field.' . $privacyFieldId . '.group.' . $privacyField->get( '_fieldGroupId', 0, GetterInterface::INT ) . '.row.' . $i;

					if ( in_array( $asset, $assets ) ) {
						continue;
					}

					$assets[]	=	$asset;
				}
			}

			if ( $assets ) {
				$query			=	'SELECT *'
								.	"\n FROM " . $_CB_database->NameQuote( '#__comprofiler_plugin_privacy' )
								.	"\n WHERE " . $_CB_database->NameQuote( 'user_id' ) . " = " . $user->get( 'id', 0, GetterInterface::INT )
								.	"\n AND " . $_CB_database->NameQuote( 'asset' ) . " IN " . $_CB_database->safeArrayOfStrings( $assets );
				$_CB_database->setQuery( $query );
				$privacy		=	$_CB_database->loadObjectList( null, '\CB\Plugin\Privacy\Table\PrivacyTable', array( $_CB_database ) );

				/** @var PrivacyTable[] $privacy */
				foreach ( $privacy as $rule ) {
					$rule->delete();
				}
			}
		}
	}

	/**
	 * @param FieldTable $field
	 * @param UserTable  $user
	 * @param string     $reason
	 * @return bool
	 */
	private function saveFieldPrivacy( $field, $user, $reason )
	{
		$fieldId				=	$field->get( 'fieldid', 0, GetterInterface::INT );

		if ( ! $field->params instanceof ParamsInterface ) {
			$field->params		=	new Registry( $field->params );
		}

		$display				=	$field->params->get( 'cbprivacy_display', 0, GetterInterface::INT );

		if ( ( $reason == 'register' ) && ( ! $field->params->get( 'cbprivacy_display_reg', false, GetterInterface::BOOLEAN ) ) ) {
			$display			=	0;
		}

		if ( ( $display == 1 ) || ( ( $display == 2 ) && Application::MyUser()->isGlobalModerator() ) ) {
			$asset				=	str_replace( '[field_id]', $fieldId, $field->params->getString( 'cbprivacy_asset', '' ) );

			if ( ! $asset ) {
				$asset			=	'profile.field.' . $fieldId;

				if ( $field->get( '_isGrouped', false, GetterInterface::BOOLEAN ) ) {
					$asset		.=	'.group.' . $field->get( '_fieldGroupId', 0, GetterInterface::INT ) . '.row.' . $field->get( '_fieldGroupIndex', 0, GetterInterface::INT );
				}
			}

			$privacy			=	new Privacy( $asset, $user );

			if ( $display == 2 ) {
				$privacy->set( 'options_moderator', true );
			}

			if ( $reason == 'register' ) {
				$privacy->set( 'guest', true );
			}

			$privacy->parse( $field->params, 'privacy_' );

			if ( in_array( $reason, array( 'edit', 'register' ) ) ) {
				$privacy->set( 'ajax', false );
			}

			$privacy->name( $field->getString( 'name', '' ) . '__privacy' );
			$privacy->set( 'field', $fieldId );

			if ( $privacy->getString( 'layout', 'button' ) == 'hidden' ) {
				return false;
			}

			$privacy->privacy( 'save' );

			return true;
		}

		return false;
	}

	/**
	 * Allows access to CB Privacy if privacy consent is required
	 *
	 * @param string $plugin
	 * @return bool
	 */
	public function verifyConsent( $plugin )
	{
		return ( $plugin == 'cbprivacy' );
	}
}
