<?php
/**
* Community Builder (TM)
* @version $Id: $
* @package CommunityBuilder
* @copyright (C) 2004-2021 www.joomlapolis.com / Lightning MultiCom SA - and its licensors, all rights reserved
* @license http://www.gnu.org/licenses/old-licenses/gpl-2.0.html GNU/GPL version 2
*/

namespace CB\Plugin\Privacy;

use CBLib\Application\Application;
use CBLib\Registry\GetterInterface;
use CBLib\Language\CBTxt;
use CBLib\Registry\Registry;
use CB\Database\Table\UserTable;
use CB\Database\Table\TabTable;
use CB\Database\Table\FieldTable;
use CBLib\Registry\ParamsInterface;
use CB\Plugin\Privacy\Table\PrivacyTable;

defined('CBLIB') or die();

class CBPrivacy
{

	/**
	 * @return Registry
	 */
	static public function getGlobalParams()
	{
		global $_PLUGINS;

		static $params	=	null;

		if ( ! $params ) {
			$plugin		=	$_PLUGINS->getLoadedPlugin( 'user', 'cbprivacy' );
			$params		=	new Registry();

			if ( $plugin ) {
				$params->load( $plugin->params );
			}
		}

		return $params;
	}

	/**
	 * @param null|string $template
	 * @param null|string $file
	 * @param bool|array  $headers
	 * @return null|string
	 */
	static public function getTemplate( $template = null, $file = null, $headers = array( 'template', 'override' ) )
	{
		global $_CB_framework, $_PLUGINS;

		$plugin							=	$_PLUGINS->getLoadedPlugin( 'user', 'cbprivacy' );

		if ( ! $plugin ) {
			return null;
		}

		static $defaultTemplate			=	null;

		if ( $defaultTemplate === null ) {
			$defaultTemplate			=	self::getGlobalParams()->getString( 'general_template', 'default' );
		}

		if ( ( $template === '' ) || ( $template === null ) || ( $template === '-1' ) ) {
			$template					=	$defaultTemplate;
		}

		if ( ! $template ) {
			$template					=	'default';
		}

		$livePath						=	$_PLUGINS->getPluginLivePath( $plugin );
		$absPath						=	$_PLUGINS->getPluginPath( $plugin );

		$file							=	preg_replace( '/[^-a-zA-Z0-9_]/', '', $file );
		$return							=	null;

		if ( $file ) {
			if ( $headers !== false ) {
				$headers[]				=	$file;
			}

			$php						=	$absPath . '/templates/' . $template . '/' . $file . '.php';

			if ( ! file_exists( $php ) ) {
				$php					=	$absPath . '/templates/default/' . $file . '.php';
			}

			if ( file_exists( $php ) ) {
				$return					=	$php;
			}
		}

		if ( $headers !== false ) {
			static $loaded				=	array();

			$loaded[$template]			=	array();

			// Global CSS File:
			if ( in_array( 'template', $headers ) && ( ! in_array( 'template', $loaded[$template] ) ) ) {
				$global					=	'/templates/' . $template . '/template.css';

				if ( ! file_exists( $absPath . $global ) ) {
					$global				=	'/templates/default/template.css';
				}

				if ( file_exists( $absPath . $global ) ) {
					$_CB_framework->document->addHeadStyleSheet( $livePath . $global );
				}

				$loaded[$template][]	=	'template';
			}

			// File or Custom CSS/JS Headers:
			foreach ( $headers as $header ) {
				if ( in_array( $header, $loaded[$template] ) || in_array( $header, array( 'template', 'override' ) ) ) {
					continue;
				}

				$header					=	preg_replace( '/[^-a-zA-Z0-9_]/', '', $header );

				if ( ! $header ) {
					continue;
				}

				$css					=	'/templates/' . $template . '/' . $header . '.css';
				$js						=	'/templates/' . $template . '/' . $header . '.js';

				if ( ! file_exists( $absPath . $css ) ) {
					$css				=	'/templates/default/' . $header . '.css';
				}

				if ( file_exists( $absPath . $css ) ) {
					$_CB_framework->document->addHeadStyleSheet( $livePath . $css );
				}

				if ( ! file_exists( $absPath . $js ) ) {
					$js					=	'/templates/default/' . $header . '.js';
				}

				if ( file_exists( $absPath . $js ) ) {
					$_CB_framework->document->addHeadScriptUrl( $livePath . $js );
				}

				$loaded[$template][]	=	$header;
			}

			// Override CSS File:
			if ( in_array( 'override', $headers ) && ( ! in_array( 'override', $loaded[$template] ) ) ) {
				$override				=	'/templates/' . $template . '/override.css';

				if ( file_exists( $absPath . $override ) ) {
					$_CB_framework->document->addHeadStyleSheet( $livePath . $override );
				}

				$loaded[$template][]	=	'override';
			}
		}

		return $return;
	}

	/**
	 * Returns the current return url or generates one from current page
	 *
	 * @param bool|false $current
	 * @param bool|false $raw
	 * @return null|string
	 */
	static public function getReturn( $current = false, $raw = false )
	{
		static $cache				=	array();

		if ( ! isset( $cache[$current] ) ) {
			$url					=	null;

			if ( $current ) {
				$returnUrl			=	Application::Input()->get( 'get/return', '', GetterInterface::BASE64 );

				if ( $returnUrl ) {
					$returnUrl		=	base64_decode( $returnUrl );

					if ( \JUri::isInternal( $returnUrl ) || ( $returnUrl[0] == '/' ) ) {
						$url		=	$returnUrl;
					}
				}
			} else {
				$isHttps			=	( isset( $_SERVER['HTTPS'] ) && ( ! empty( $_SERVER['HTTPS'] ) ) && ( $_SERVER['HTTPS'] != 'off' ) );
				$returnUrl			=	'http' . ( $isHttps ? 's' : '' ) . '://' . $_SERVER['HTTP_HOST'];

				if ( ( ! empty( $_SERVER['PHP_SELF'] ) ) && ( ! empty( $_SERVER['REQUEST_URI'] ) ) ) {
					$returnUrl		.=	$_SERVER['REQUEST_URI'];
				} else {
					$returnUrl		.=	$_SERVER['SCRIPT_NAME'];

					if ( isset( $_SERVER['QUERY_STRING'] ) && ( ! empty( $_SERVER['QUERY_STRING'] ) ) ) {
						$returnUrl	.=	'?' . $_SERVER['QUERY_STRING'];
					}
				}

				$url				=	cbUnHtmlspecialchars( preg_replace( '/[\\\"\\\'][\\s]*javascript:(.*)[\\\"\\\']/', '""', preg_replace( '/eval\((.*)\)/', '', htmlspecialchars( urldecode( $returnUrl ) ) ) ) );
			}

			$cache[$current]		=	$url;
		}

		$return						=	$cache[$current];

		if ( ( ! $raw ) && $return ) {
			$return					=	base64_encode( $return );
		}

		return $return;
	}

	/**
	 * Redirects to the return url if available otherwise to the url specified
	 *
	 * @param string      $url
	 * @param null|string $message
	 * @param string      $messageType
	 */
	static public function returnRedirect( $url, $message = null, $messageType = 'message' )
	{
		$returnUrl		=	self::getReturn( true, true );

		cbRedirect( ( $returnUrl ? $returnUrl : $url ), $message, $messageType );
	}

	/**
	 * Returns an array of users connections as array( userId => type )
	 *
	 * @param int  $profileId
	 * @return array
	 */
	static public function getConnections( $profileId )
	{
		global $_CB_database;

		if ( ( ! $profileId ) || ( ! Application::Config()->get( 'allowConnections', true, GetterInterface::BOOLEAN ) ) ) {
			return array();
		}

		static $cache				=	array();

		if ( ! isset( $cache[$profileId] ) ) {
			$query					=	"SELECT m." . $_CB_database->NameQuote( 'memberid' )
									.	", IF ( m." . $_CB_database->NameQuote( 'type' ) . " IS NULL, '', m." . $_CB_database->NameQuote( 'type' ) . " ) AS membertype"
									.	"\n FROM " . $_CB_database->NameQuote( '#__comprofiler_members' ) . " AS m"
									.	"\n WHERE m." . $_CB_database->NameQuote( 'referenceid' ) . " = " . (int) $profileId
									.	"\n AND m." . $_CB_database->NameQuote( 'accepted' ) . " = 1";
			$_CB_database->setQuery( $query );
			$cache[$profileId]		=	$_CB_database->loadAssocList( 'memberid', 'membertype' );
		}

		return $cache[$profileId];
	}

	/**
	 * Checks if two users are actively connected to one another
	 *
	 * @param int $fromUserId
	 * @param int $toUserId
	 * @return bool
	 */
	static public function getConnectionOfConnection( $fromUserId, $toUserId )
	{
		if ( ( ! $fromUserId ) || ( ! $toUserId ) || ( ! Application::Config()->get( 'allowConnections', true, GetterInterface::BOOLEAN ) ) ) {
			return false;
		}

		static $connected							=	array();

		if ( ! isset( $connected[$fromUserId][$toUserId] ) ) {
			$cbConnection							=	new \cbConnection( $fromUserId );

			$connected[$fromUserId][$toUserId]		=	( $cbConnection->getDegreeOfSepPathArray( $fromUserId, $toUserId, 1, 2 ) ? true : false );
		}

		return $connected[$fromUserId][$toUserId];
	}

	/**
	 * Returns an options array of available privacy values
	 *
	 * @param null|Privacy $privacy
	 * @param bool         $raw
	 * @return array
	 */
	static public function getPrivacyOptions( $privacy = null, $raw = false )
	{
		global $_CB_database, $_PLUGINS, $ueConfig;

		static $cache							=	null;

		if ( Application::Cms()->getClientId() ) {
			$privacy							=	null;
			$raw								=	false;
		}

		if ( $privacy ) {
			$privacyId							=	$privacy->id();
		} else {
			$privacyId							=	0;
		}

		if ( ! isset( $cache[$privacyId] ) ) {
			if ( $privacy ) {
				$userId							=	$privacy->user()->get( 'id', 0, GetterInterface::INT );
				$params							=	$privacy;
				$prefix							=	null;
			} else {
				$userId							=	Application::MyUser()->getUserId();
				$params							=	self::getGlobalParams();
				$prefix							=	'privacy_';
			}

			$options							=	array();

			$_PLUGINS->trigger( 'privacy_onBeforePrivacyOptions', array( &$options, $privacy ) );

			if ( $params->get( $prefix . 'options_visible', true, GetterInterface::BOOLEAN ) && in_array( $params->get( $prefix . 'options_visible_access', 1, GetterInterface::INT ), Application::MyUser()->getAuthorisedViewLevels() ) ) {
				$options[]						=	\moscomprofilerHTML::makeOption( '0', CBTxt::T( 'PRIVACY_PUBLIC', 'Public' ), 'value', 'text', null, null, 'data-cbprivacy-option-icon="' . htmlspecialchars( '<span class="cbPrivacySelectOptionIconPublic fa fa-globe"></span>' ) . '"' );
			}

			if ( ( ( $ueConfig['profile_viewaccesslevel'] == 1 ) && $params->get( $prefix . 'options_users', true, GetterInterface::BOOLEAN ) ) && in_array( $params->get( $prefix . 'options_users_access', 1, GetterInterface::INT ), Application::MyUser()->getAuthorisedViewLevels() ) ) {
				$options[]						=	\moscomprofilerHTML::makeOption( '1', CBTxt::T( 'PRIVACY_USERS', 'Users' ), 'value', 'text', null, null, 'data-cbprivacy-option-icon="' . htmlspecialchars( '<span class="cbPrivacySelectOptionIconUsers fa fa-user"></span>' ) . '"' );
			}

			if ( $params->get( $prefix . 'options_invisible', true, GetterInterface::BOOLEAN ) && in_array( $params->get( $prefix . 'options_invisible_access', 1, GetterInterface::INT ), Application::MyUser()->getAuthorisedViewLevels() ) ) {
				$options[]						=	\moscomprofilerHTML::makeOption( '99', CBTxt::T( 'PRIVACY_PRIVATE', 'Private' ), 'value', 'text', null, null, 'data-cbprivacy-option-icon="' . htmlspecialchars( '<span class="cbPrivacySelectOptionIconPrivate fa fa-lock"></span>' ) . '"' );
			}

			if ( $params->get( $prefix . 'options_moderator', false, GetterInterface::BOOLEAN ) ) {
				$options[]						=	\moscomprofilerHTML::makeOption( '999', CBTxt::T( 'PRIVACY_MODERATORS', 'Moderators' ), 'value', 'text', null, null, 'data-cbprivacy-option-icon="' . htmlspecialchars( '<span class="cbPrivacySelectOptionIconModerators fa fa-user-secret"></span>' ) . '"' );
			}

			if ( $ueConfig['allowConnections'] ) {
				if ( $params->get( $prefix . 'options_conn', true, GetterInterface::BOOLEAN ) && in_array( $params->get( $prefix . 'options_conn_access', 1, GetterInterface::INT ), Application::MyUser()->getAuthorisedViewLevels() ) ) {
					$options[]					=	\moscomprofilerHTML::makeOption( '2', CBTxt::T( 'PRIVACY_CONNECTIONS', 'Connections' ), 'value', 'text', null, null, 'data-cbprivacy-option-icon="' . htmlspecialchars( '<span class="cbPrivacySelectOptionIconConnections fa fa-users"></span>' ) . '"' );
				}

				if ( $params->get( $prefix . 'options_connofconn', true, GetterInterface::BOOLEAN ) && in_array( $params->get( $prefix . 'options_connofconn_access', 1, GetterInterface::INT ), Application::MyUser()->getAuthorisedViewLevels() ) ) {
					$options[]					=	\moscomprofilerHTML::makeOption( '3', CBTxt::T( 'PRIVACY_CONNECTIONS_OF_CONNECTIONS', 'Connections of Connections' ), 'value', 'text', null, null, 'data-cbprivacy-option-icon="' . htmlspecialchars( '<span class="cbPrivacySelectOptionIconConnectionsOfConnections fa fa-users"></span>' ) . '"' );
				}

				if ( $ueConfig['connection_categories'] && $params->get( $prefix . 'options_conntype', true, GetterInterface::BOOLEAN ) && in_array( $params->get( $prefix . 'options_conntype_access', 1, GetterInterface::INT ), Application::MyUser()->getAuthorisedViewLevels() ) ) {
					$types						=	self::getConnectionTypes();

					if ( $types ) {
						$connTypes				=	explode( '|*|', $params->getString( $prefix . 'options_conntypes', '0' ) );
						$connOpt				=	false;

						foreach ( $types as $type ) {
							if ( in_array( '0', $connTypes ) || in_array( $type->value, $connTypes ) ) {
								if ( ! $connOpt ) {
									$options[]	=	\moscomprofilerHTML::makeOptGroup( CBTxt::T( 'PRIVACY_CONNECTION_TYPES', 'Connection Types' ) );

									$connOpt	=	true;
								}

								$options[]		=	\moscomprofilerHTML::makeOption( 'CONN-' . (string) $type->value, $type->text, 'value', 'text', null, null, 'data-cbprivacy-option-icon="' . htmlspecialchars( '<span class="cbPrivacySelectOptionIconConnectionType fa fa-cog"></span>' ) . '"' );
							}
						}

						$options[]				=	\moscomprofilerHTML::makeOptGroup( null );
					}
				}
			}

			if ( $params->get( $prefix . 'options_specificusers', false, GetterInterface::BOOLEAN ) && in_array( $params->get( $prefix . 'options_specificusers_access', 1, GetterInterface::INT ), Application::MyUser()->getAuthorisedViewLevels() ) ) {
				$options[]						=	\moscomprofilerHTML::makeOptGroup( CBTxt::T( 'PRIVACY_SPECIFIC_USERS', 'Specific Users' ) );

				$addUser						=	'<input type="text" class="w-100 form-control form-control-sm cbPrivacySelectAddUser" placeholder="' . htmlspecialchars( CBTxt::T( 'PRIVACY_ADD_SPECIFIC_USER', 'Add User...' ) ) . '" />';

				$options[]						=	\moscomprofilerHTML::makeOption( 'USER-ADD', CBTxt::T( 'PRIVACY_ADD_SPECIFIC_USER', 'Add User...' ), 'value', 'text', null, null, 'data-cbprivacy-option-icon="' . htmlspecialchars( '<span class="cbPrivacySelectOptionIconAddUser fa fa-plus"></span>' ) . '" data-cbprivacy-option-text="' . htmlspecialchars( $addUser ) . '"' );

				if ( $privacy ) {
					// Add the custom existing privacy rules to the list:
					$userRules					=	array();

					foreach ( $privacy->reset()->rows( 'all' ) as $rule ) {
						if ( ! preg_match( '/USER-(\d+)/', $rule->getString( 'rule', '' ), $userMatch ) ) {
							continue;
						}

						$userRuleID				=	(int) $userMatch[1];

						if ( ! in_array( $userRuleID, $userRules ) ) {
							$userRules[]		=	$userRuleID;
						}
					}

					if ( $userRules ) {
						$query					=	"SELECT *"
												.	"\n FROM " . $_CB_database->NameQuote( '#__comprofiler' ) . " AS c"
												.	"\n INNER JOIN " . $_CB_database->NameQuote( '#__users' ) . ' AS u'
												.	' ON u.' . $_CB_database->NameQuote( 'id' ) . ' = c.' . $_CB_database->NameQuote( 'id' )
												.	"\n WHERE c." . $_CB_database->NameQuote( 'id' ) . " IN " . $_CB_database->safeArrayOfIntegers( $userRules );
						$_CB_database->setQuery( $query );
						$privacyUsers			=	$_CB_database->loadObjectList( null, '\CB\Database\Table\UserTable', array( $_CB_database ) );

						/** @var UserTable[] $privacyUsers */
						foreach ( $privacyUsers as $privacyUser ) {
							$options[]			=	\moscomprofilerHTML::makeOption( 'USER-' . $privacyUser->get( 'id', 0, GetterInterface::INT ), $privacyUser->getFormattedName(), 'value', 'text', null, null, 'data-cbprivacy-option-icon="' . htmlspecialchars( '<span class="cbPrivacySelectOptionIconUser fa fa-user"></span>' ) . '"' );
						}
					}
				}

				$options[]						=	\moscomprofilerHTML::makeOptGroup( null );
			}

			if ( $params->get( $prefix . 'options_viewaccesslevel', false, GetterInterface::BOOLEAN ) && in_array( $params->get( $prefix . 'options_viewaccesslevel_access', 1, GetterInterface::INT ), Application::MyUser()->getAuthorisedViewLevels() ) ) {
				$visibility						=	$params->get( $prefix . 'options_viewaccesslevel_visibility', false, GetterInterface::BOOLEAN );
				$accessLevels					=	Application::CmsPermissions()->getAllViewAccessLevels( true, ( $visibility ? Application::User( $userId ) : null ) );

				if ( $accessLevels ) {
					$viewAccessLevels			=	explode( '|*|', $params->getString( $prefix . 'options_viewaccesslevels', '0' ) );
					$accessOpt					=	false;

					foreach ( $accessLevels as $accessLevel ) {
						if ( in_array( '0', $viewAccessLevels ) || in_array( $accessLevel->value, $viewAccessLevels ) ) {
							if ( ! $accessOpt ) {
								$options[]		=	\moscomprofilerHTML::makeOptGroup( CBTxt::T( 'PRIVACY_VIEWACCESSLEVELS', 'View Access Levels' ) );

								$accessOpt		=	true;
							}

							$options[]			=	\moscomprofilerHTML::makeOption( 'ACCESS-' . (string) $accessLevel->value, CBTxt::T( $accessLevel->text ), 'value', 'text', null, null, 'data-cbprivacy-option-icon="' . htmlspecialchars( '<span class="cbPrivacySelectOptionIconViewAccessLevel fa fa-cog"></span>' ) . '"' );
						}
					}

					$options[]					=	\moscomprofilerHTML::makeOptGroup( null );
				}
			}

			if ( $params->get( $prefix . 'options_usergroup', false, GetterInterface::BOOLEAN ) && in_array( $params->get( $prefix . 'options_usergroup_access', 1, GetterInterface::INT ), Application::MyUser()->getAuthorisedViewLevels() ) ) {
				$visibility						=	$params->get( $prefix . 'options_usergroup_visibility', false, GetterInterface::BOOLEAN );

				if ( Application::MyUser()->isSuperAdmin() ) {
					// Mimic view access level exception behavior in getAllViewAccessLevels:
					$visibility					=	false;
				}

				$groups							=	Application::CmsPermissions()->getAllGroups( true, '' );
				$myGroups						=	array();

				if ( $visibility ) {
					$myGroups					=	Application::User( $userId )->getAuthorisedGroups();
				}

				if ( $groups ) {
					$userGroups					=	explode( '|*|', $params->getString( $prefix . 'options_usergroups', '0' ) );
					$groupsOpt					=	false;

					foreach ( $groups as $group ) {
						if ( in_array( '0', $userGroups ) || in_array( $group->value, $userGroups ) ) {
							if ( $visibility && ( ! in_array( (int) $group->value, $myGroups ) ) ) {
								continue;
							}

							if ( ! $groupsOpt ) {
								$options[]		=	\moscomprofilerHTML::makeOptGroup( CBTxt::T( 'PRIVACY_USERGROUPS', 'Usergroups' ) );

								$groupsOpt		=	true;
							}

							$options[]			=	\moscomprofilerHTML::makeOption( 'GROUP-' . (string) $group->value, CBTxt::T( $group->text ), 'value', 'text', null, null, 'data-cbprivacy-option-icon="' . htmlspecialchars( '<span class="cbPrivacySelectOptionIconUsergroup fa fa-cog"></span>' ) . '"' );
						}
					}

					$options[]					=	\moscomprofilerHTML::makeOptGroup( null );
				}
			}

			$_PLUGINS->trigger( 'privacy_onAfterPrivacyOptions', array( &$options, $privacy ) );

			$cache[$privacyId]					=	$options;
		}

		if ( $raw ) {
			$opts								=	array();

			foreach ( $cache[$privacyId] as $opt ) {
				if ( is_array( $opt->value ) ) {
					continue;
				}

				$opts[$opt->value]				=	$opt->text;
			}

			return $opts;
		}

		return $cache[$privacyId];
	}

	/**
	 * Returns an options array of connection types
	 *
	 * @return array
	 */
	static public function getConnectionTypes()
	{
		static $options			=	null;

		if ( $options === null ) {
			$options			=	array();

			if ( Application::Config()->getString( 'connection_categories' ) ) {
				$types			=	explode( "\n", Application::Config()->getString( 'connection_categories', '' ) );

				foreach ( $types as $type ) {
					if ( trim( $type ) == '' ) {
						continue;
					}

					$options[]	=	\moscomprofilerHTML::makeOption( htmlspecialchars( trim( $type ) ), CBTxt::T( trim( $type ) ) );
				}
			}
		}

		return $options;
	}

	/**
	 * Returns a cached tab object from tab id or existing object
	 *
	 * @param TabTable|int|string $tab TabTable: caches the object, int: loads by tab id, string: loads by tab pluginclass
	 * @return TabTable|null
	 */
	static public function getTab( $tab )
	{
		static $tabs					=	array();
		static $classes					=	array();

		if ( is_numeric( $tab ) ) {
			$tab						=	(int) $tab;

			if ( ! isset( $tabs[$tab] ) ) {
				$loadedTab				=	new TabTable();

				$loadedTab->load( $tab );

				$class					=	$loadedTab->getString( 'pluginclass', '' );

				$tabs[$tab]				=	$loadedTab;

				if ( ! isset( $classes[$class] ) ) {
					$classes[$class]	=	$tab;
				}

				return $loadedTab;
			}

			return $tabs[$tab];
		} elseif ( is_string( $tab ) ) {
			if ( ! isset( $classes[$tab] ) ) {
				$loadedTab				=	new TabTable();

				$loadedTab->load( array( 'pluginclass' => $tab ) );

				$id						=	$loadedTab->get( 'tabid', 0, GetterInterface::INT );

				$classes[$tab]			=	$id;

				if ( ! isset( $tabs[$id] ) ) {
					$tabs[$id]			=	$loadedTab;
				}

				return $loadedTab;
			}

			return $tabs[$classes[$tab]];
		} elseif ( $tab instanceof TabTable ) {
			$id							=	$tab->get( 'tabid', 0, GetterInterface::INT );
			$class						=	$tab->getString( 'pluginclass', '' );

			if ( ! isset( $tabs[$id] ) ) {
				$tabs[$id]				=	$tab;
			}

			if ( ! isset( $classes[$class] ) ) {
				$classes[$class]		=	$id;
			}

			return $tab;
		}

		return null;
	}

	/**
	 * Returns a cached field object from field id or existing object
	 *
	 * @param FieldTable|int|string $field FieldTable: caches the object, int: loads by field id, string: loads by field name
	 * @return FieldTable|null
	 */
	static public function getField( $field )
	{
		static $fields				=	array();
		static $names				=	array();

		if ( is_numeric( $field ) ) {
			$field					=	(int) $field;

			if ( ! isset( $fields[$field] ) ) {
				$loadedField		=	new FieldTable();

				$loadedField->load( $field );

				$name				=	$loadedField->getString( 'name', '' );

				$fields[$field]		=	$loadedField;

				if ( ! isset( $names[$name] ) ) {
					$names[$name]	=	$field;
				}

				return $loadedField;
			}

			return $fields[$field];
		} elseif ( is_string( $field ) ) {
			if ( ! isset( $names[$field] ) ) {
				$loadedField		=	new FieldTable();

				$loadedField->load( array( 'name' => $field ) );

				$id					=	$loadedField->get( 'fieldid', 0, GetterInterface::INT );

				$names[$field]		=	$id;

				if ( ! isset( $fields[$id] ) ) {
					$fields[$id]	=	$loadedField;
				}

				return $loadedField;
			}

			return $fields[$names[$field]];
		} elseif ( $field instanceof FieldTable ) {
			$id						=	$field->get( 'fieldid', 0, GetterInterface::INT );
			$name					=	$field->getString( 'name', '' );

			if ( ! isset( $fields[$id] ) ) {
				$fields[$id]		=	$field;
			}

			if ( ! isset( $names[$name] ) ) {
				$names[$name]		=	$id;
			}

			return $field;
		}

		return null;
	}

	/**
	 * Returns an array of users privacy rows
	 *
	 * @param null|int|UserTable|array $profileId array: prefetches privacy rules for supplied ids; else: loads and returns the privacy rules for supplied user
	 * @return array
	 */
	static public function getPrivacy( $profileId = null )
	{
		global $_CB_database;

		if ( $profileId === null ) {
			$profileId								=	Application::MyUser()->getUserId();
		} elseif ( $profileId instanceof UserTable ) {
			$profileId								=	$profileId->get( 'id', 0, GetterInterface::INT );
		}

		if ( ! $profileId ) {
			return array();
		}

		static $cache								=	array();

		if ( is_array( $profileId ) ) {
			// Prefetch and cache multiple profile privacy rules at once to save on performance:
			$ids									=	array();

			foreach ( $profileId as $id ) {
				if ( isset( $cache[$id] ) ) {
					continue;
				}

				if ( in_array( $id, $ids ) ) {
					continue;
				}

				$ids[]								=	$id;
			}

			if ( $ids ) {
				$query								=	"SELECT *"
													.	"\n FROM " . $_CB_database->NameQuote( '#__comprofiler_plugin_privacy' )
													.	"\n WHERE " . $_CB_database->NameQuote( 'user_id' ) . " IN " . $_CB_database->safeArrayOfIntegers( $ids );
				$_CB_database->setQuery( $query );
				$rules								=	$_CB_database->loadObjectList( 'id', '\CB\Plugin\Privacy\Table\PrivacyTable', array( $_CB_database ) );

				/** @var PrivacyTable[] $rules */
				foreach ( $rules as $id => $rule ) {
					$profileId						=	$rule->get( 'user_id', 0, GetterInterface::INT );
					$asset							=	$rule->getString( 'asset', '' );

					$cache[$profileId][$asset][$id]	=	$rule;
				}

				foreach ( $ids as $id ) {
					if ( isset( $cache[$id] ) ) {
						continue;
					}

					// We requested this users privacy but they have none so lets make sure we cache that:
					$cache[$id]						=	array();
				}
			}

			return array();
		} elseif ( ! isset( $cache[$profileId] ) ) {
			$query									=	"SELECT *"
													.	"\n FROM " . $_CB_database->NameQuote( '#__comprofiler_plugin_privacy' )
													.	"\n WHERE " . $_CB_database->NameQuote( 'user_id' ) . " = " . (int) $profileId;
			$_CB_database->setQuery( $query );
			$rules									=	$_CB_database->loadObjectList( 'id', '\CB\Plugin\Privacy\Table\PrivacyTable', array( $_CB_database ) );

			$privacy								=	array();

			/** @var PrivacyTable[] $rules */
			foreach ( $rules as $id => $rule ) {
				$asset								=	$rule->getString( 'asset', '' );

				$privacy[$asset][$id]				=	$rule;
			}

			$cache[$profileId]						=	$privacy;
		}

		return $cache[$profileId];
	}

	/**
	 * Caches tab and field privacy controls to be parsed into a simplified privacy tab
	 *
	 * @param TabTable|FieldTable $object
	 * @param Privacy             $privacy
	 * @return array
	 */
	static public function getGroupedPrivacy( $object = null, $privacy = null )
	{
		static $cache									=	array();

		if ( ( ! $object ) && ( ! $privacy ) ) {
			return $cache;
		}

		$group											=	$object->params->get( 'cbprivacy_simple_group', null, GetterInterface::HTML );

		if ( ! $group ) {
			$group										=	null;
		}

		if ( $object instanceof TabTable ) {
			$tabId										=	$object->get( 'tabid', 0, GetterInterface::INT );

			$cache[$group][$tabId]						=	array( 'tab' => $object, 'privacy' => $privacy );
		} elseif ( $object instanceof FieldTable ) {
			// We always need to cache with tabid in mind so the privacy controls are grouped by tab:
			$tabId										=	$object->get( 'tabid', 0, GetterInterface::INT );
			$fieldId									=	$object->get( 'fieldid', 0, GetterInterface::INT );

			$cache[$group][$tabId]['fields'][$fieldId]	=	array( 'field' => $object, 'privacy' => $privacy );
		}

		return $cache;
	}

	/**
	 * Returns a list of blocked ids blocked by $userId
	 *
	 * @param UserTable|int|null $userId
	 * @return array
	 */
	static public function getBlocked( $userId = null )
	{
		global $_CB_database;

		if ( ! self::getGlobalParams()->get( 'profile_blocking', true, GetterInterface::BOOLEAN ) ) {
			return array();
		}

		if ( ! $userId ) {
			$userId				=	\CBuser::getMyUserDataInstance()->get( 'id', 0, GetterInterface::INT );
		} elseif ( $userId instanceof UserTable ) {
			$userId				=	$userId->get( 'id', 0, GetterInterface::INT );
		}

		static $cache			=	array();

		if ( ! isset( $cache[$userId] ) ) {
			$query				=	"SELECT " . $_CB_database->NameQuote( 'block' )
								.	"\n FROM " . $_CB_database->NameQuote( '#__comprofiler_plugin_privacy_blocked' )
								.	"\n WHERE " . $_CB_database->NameQuote( 'user_id' ) . " = " . (int) $userId;
			$_CB_database->setQuery( $query );

			$cache[$userId]		=	$_CB_database->loadResultArray();
		}

		return $cache[$userId];
	}

	/**
	 * @param UserTable|int|null $user
	 * @param null|int           $tabId
	 * @param null|int           $fieldId
	 * @return bool
	 */
	static public function checkProfileDisplayAccess( $user = null, $tabId = null, $fieldId = null )
	{
		if ( ! $user ) {
			$user						=	\CBuser::getMyUserDataInstance();
		} elseif ( is_int( $user ) ) {
			$user						=	\CBuser::getInstance( $user, false )->getUserData();
		}

		if ( Application::Cms()->getClientId() || Application::MyUser()->isGlobalModerator() || ( $user->get( 'id', 0, GetterInterface::INT ) == Application::MyUser()->getUserId() ) ) {
			return true;
		}

		$myId							=	Application::MyUser()->getUserId();

		static $cache					=	array();

		static $field					=	null;
		static $hideTabs				=	array();
		static $hideFields				=	array();

		$userId							=	$user->get( 'id', 0, GetterInterface::INT );

		if ( ! isset( $cache[$userId][$myId] ) ) {
			$authorized					=	true;

			if ( ! $field ) {
				$field					=	new FieldTable();

				$field->load( array( 'name' => 'privacy_profile', 'published' => 1 ) );

				if ( ! $field->params instanceof ParamsInterface ) {
					$field->params		=	new Registry( $field->params );
				}

				$hideTabs				=	cbToArrayOfInt( explode( '|*|', $field->params->getString( 'cbprivacy_profile_tabs', '' ) ) );
				$hideFields				=	cbToArrayOfInt( explode( '|*|', $field->params->getString( 'cbprivacy_profile_fields', '' ) ) );
			}

			if ( $field->get( 'fieldid', 0, GetterInterface::INT ) && ( $field->get( 'edit', 1, GetterInterface::INT ) || $field->get( 'registration', 1, GetterInterface::INT ) ) ) {
				$privacy				=	new Privacy( 'profile', $user );

				$privacy->parse( $field->params, 'privacy_' );

				$authorized				=	$privacy->authorized( $myId );
			}

			$cache[$userId][$myId]		=	$authorized;
		}

		$access							=	$cache[$userId][$myId];

		if ( ! $access ) {
			if ( self::getGlobalParams()->get( 'profile_direct_access', false, GetterInterface::BOOLEAN ) ) {
				if ( $fieldId ) {
					if ( ! in_array( $fieldId, $hideFields ) ) {
						$access			=	true;
					}
				} elseif ( $tabId ) {
					if ( ! in_array( $tabId, $hideTabs ) ) {
						$access			=	true;
					}
				}
			}
		}

		return $access;
	}

	/**
	 * @param TabTable|int|string $tab
	 * @param UserTable|int       $user
	 * @return bool
	 */
	static public function checkTabDisplayAccess( $tab, $user = null )
	{
		if ( ! $user ) {
			$user								=	\CBuser::getMyUserDataInstance();
		} elseif ( is_int( $user ) ) {
			$user								=	\CBuser::getInstance( $user, false )->getUserData();
		}

		if ( Application::Cms()->getClientId() || Application::MyUser()->isGlobalModerator() ) {
			return true;
		}

		$tab									=	self::getTab( $tab );

		if ( ! $tab ) {
			return true;
		}

		$myId									=	Application::MyUser()->getUserId();
		$tabId									=	$tab->get( 'tabid', 0, GetterInterface::INT );

		// Lets first check if this users profile can even be accessed:
		if ( ! self::checkProfileDisplayAccess( $user, $tabId ) ) {
			return false;
		}

		static $cache							=	array();

		$userId									=	$user->get( 'id', 0, GetterInterface::INT );

		// Can access the profile so lets now check if this tab can be accessed:
		if ( ! isset( $cache[$tabId][$userId][$myId] ) ) {
			$authorized							=	true;

			if ( ! $tab->params instanceof ParamsInterface ) {
				$tab->params					=	new Registry( $tab->params );
			}

			$display							=	$tab->params->get( 'cbprivacy_display', 0, GetterInterface::INT );

			if ( $display == 4 ) {
				if ( ! Application::User( $myId )->isGlobalModerator() ) {
					$authorized					=	false;
				}
			} elseif ( $user->get( 'id', 0, GetterInterface::INT ) == Application::MyUser()->getUserId() ) {
				$authorized						=	true;
			} elseif ( $display ) {
				$asset							=	str_replace( '[tab_id]', $tabId, $tab->params->getString( 'cbprivacy_asset', '' ) );

				if ( ! $asset ) {
					$asset						=	'profile.tab.' . $tabId;
					$forced						=	( $display == 3 ? true : false );
				} else {
					$forced						=	false;
				}

				$privacy						=	new Privacy( $asset, $user );

				$privacy->parse( $tab->params, 'privacy_' );

				$privacy->set( 'tab', $tabId );

				$authorized						=	$privacy->authorized( $myId, $forced );
			}

			$cache[$tabId][$userId][$myId]		=	$authorized;
		}

		return $cache[$tabId][$userId][$myId];
	}

	/**
	 * @param TabTable|int|string $tab
	 * @return bool
	 */
	static public function checkTabEditAccess( $tab )
	{
		if ( Application::Cms()->getClientId() ) {
			return true;
		}

		$tab						=	self::getTab( $tab );

		if ( ! $tab ) {
			return true;
		}

		static $cache				=	array();

		$tabId						=	$tab->get( 'tabid', 0, GetterInterface::INT );
		$myId						=	Application::MyUser()->getUserId();

		if ( ! isset( $cache[$tabId][$myId] ) ) {
			$authorized				=	true;

			if ( ! $tab->params instanceof ParamsInterface ) {
				$tab->params		=	new Registry( $tab->params );
			}

			$display				=	$tab->params->get( 'cbprivacy_edit', 0, GetterInterface::INT );

			if ( ( $display == 1 )
				 || ( ( $display == 2 ) && ( ! Application::MyUser()->isGlobalModerator() ) )
				 || ( ( $display == 3 ) && ( ! Application::MyUser()->canViewAccessLevel( $tab->params->get( 'cbprivacy_edit_access', 1, GetterInterface::INT ) ) ) && ( ! Application::MyUser()->isGlobalModerator() ) )
				 || ( ( $display == 4 ) && ( ! in_array( $tab->params->get( 'cbprivacy_edit_group', 2, GetterInterface::INT ), Application::MyUser()->getAuthorisedGroups( true ) ) ) && ( ! Application::MyUser()->isGlobalModerator() ) ) ) {
				$authorized			=	false;
			}

			$cache[$tabId][$myId]	=	$authorized;
		}

		return $cache[$tabId][$myId];
	}

	/**
	 * @param FieldTable|int|string $field
	 * @param UserTable|int         $user
	 * @return bool
	 */
	static public function checkFieldDisplayAccess( $field, $user = null )
	{
		if ( ! $user ) {
			$user									=	\CBuser::getMyUserDataInstance();
		} elseif ( is_int( $user ) ) {
			$user									=	\CBuser::getInstance( $user, false )->getUserData();
		}

		if ( Application::Cms()->getClientId() || Application::MyUser()->isGlobalModerator() ) {
			return true;
		}

		$field										=	self::getField( $field );

		if ( ! $field  ) {
			return true;
		}

		$myId										=	Application::MyUser()->getUserId();
		$fieldId									=	$field->get( 'fieldid', 0, GetterInterface::INT );
		$fieldName									=	$field->getString( 'name', '' );
		$tabId										=	$field->get( 'tabid', 0, GetterInterface::INT );

		// Lets first check and see if this users profile can even be accessed:
		if ( ! self::checkProfileDisplayAccess( $user, $tabId, $fieldId ) ) {
			return false;
		}

		// Profile can be accessed so lets now see if the tab for this field can even be accessed:
		if ( ! self::checkTabDisplayAccess( $tabId, $user ) ) {
			return false;
		}

		if ( ! $field->get( 'profile', 1, GetterInterface::INT ) ) {
			// Field isn't being displayed so don't bother checking field access:
			return true;
		}

		static $cache								=	array();

		$userId										=	$user->get( 'id', 0, GetterInterface::INT );

		// Profile and tab access is allowed so lets now check this fields access:
		if ( ! isset( $cache[$fieldName][$userId][$myId] ) ) {
			$authorized								=	true;

			if ( ! $field->params instanceof ParamsInterface ) {
				$field->params						=	new Registry( $field->params );
			}

			$display								=	$field->params->get( 'cbprivacy_display', 0, GetterInterface::INT );

			if ( $display == 4 ) {
				if ( ! Application::User( $myId )->isGlobalModerator() ) {
					$authorized						=	false;
				}
			} elseif ( $user->get( 'id', 0, GetterInterface::INT ) == Application::MyUser()->getUserId() ) {
				$authorized							=	true;
			} elseif ( $display ) {
				$asset								=	str_replace( '[field_id]', $fieldId, $field->params->getString( 'cbprivacy_asset', '' ) );

				if ( ! $asset ) {
					$asset							=	'profile.field.' . $fieldId;

					if ( $field->get( '_isGrouped', false, GetterInterface::BOOLEAN ) ) {
						$asset						.=	'.group.' . $field->get( '_fieldGroupId', 0, GetterInterface::INT ) . '.row.' . $field->get( '_fieldGroupIndex', 0, GetterInterface::INT );
					}

					$forced							=	( $display == 3 ? true : false );
				} else {
					$forced							=	false;
				}

				$privacy							=	new Privacy( $asset, $user );

				$privacy->parse( $field->params, 'privacy_' );

				$privacy->name( $fieldName . '__privacy' );
				$privacy->set( 'field', $fieldId );

				$authorized							=	$privacy->authorized( $myId, $forced );
			}

			$cache[$fieldName][$userId][$myId]		=	$authorized;
		}

		return $cache[$fieldName][$userId][$myId];
	}

	/**
	 * @param FieldTable|int|string $field
	 * @return bool
	 */
	static public function checkFieldEditAccess( $field )
	{
		if ( Application::Cms()->getClientId() ) {
			return true;
		}

		$field							=	self::getField( $field );

		if ( ! $field ) {
			return true;
		}

		$tabId							=	$field->get( 'tabid', 0, GetterInterface::INT );

		// Check if the tab edit access is even allowed:
		if ( ! self::checkTabEditAccess( $tabId ) ) {
			return false;
		}

		static $cache					=	array();

		$fieldId						=	$field->get( 'fieldid', 0, GetterInterface::INT );
		$fieldName						=	$field->getString( 'name', '' );
		$myId							=	Application::MyUser()->getUserId();

		// Tab edit access is allowed so lets see if we can edit this field specifically:
		if ( ! isset( $cache[$fieldName][$myId] ) ) {
			$authorized					=	true;

			if ( ! $field->params instanceof ParamsInterface ) {
				$field->params			=	new Registry( $field->params );
			}

			$display					=	$field->params->get( 'cbprivacy_edit', 0, GetterInterface::INT );

			if ( ( $display == 1 ) // This is for stored B/C (not edit display at all)
				 || ( ( $display == 2 ) && ( ! Application::MyUser()->isGlobalModerator() ) )
				 || ( ( $display == 3 ) && ( ! Application::MyUser()->canViewAccessLevel( $field->params->get( 'cbprivacy_edit_access', 1, GetterInterface::INT ) ) ) )
				 || ( ( $display == 4 ) && ( ! in_array( $field->params->get( 'cbprivacy_edit_group', 2, GetterInterface::INT ), Application::MyUser()->getAuthorisedGroups( true ) ) ) ) ) {
				$authorized				=	false;
			}

			$cache[$fieldName][$myId]	=	$authorized;
		}

		return $cache[$fieldName][$myId];
	}

	/**
	 * XML helper function to check if simple mode is enabled or not
	 *
	 * @return bool
	 */
	static public function checkSimpleMode()
	{
		if ( ( ! self::getGlobalParams()->get( 'profile_simple_edit', false, GetterInterface::BOOLEAN ) ) && ( ! self::getGlobalParams()->get( 'profile_simple_registration', false, GetterInterface::BOOLEAN ) ) ) {
			return false;
		}

		return true;
	}
}
