<?php
/**
* Community Builder (TM)
* @version $Id: $
* @package CommunityBuilder
* @copyright (C) 2004-2019 www.joomlapolis.com / Lightning MultiCom SA - and its licensors, all rights reserved
* @license http://www.gnu.org/licenses/old-licenses/gpl-2.0.html GNU/GPL version 2
*/

namespace CB\Plugin\Connect\Provider;

use CBLib\Application\Application;
use CB\Plugin\Connect\Provider;
use CB\Plugin\Connect\Profile;
use CBLib\Registry\GetterInterface;
use CBLib\Registry\Registry;
use CBLib\Registry\ParamsInterface;
use CBLib\Xml\SimpleXMLElement;
use GuzzleHttp\Client;
use CBLib\Language\CBTxt;
use GuzzleHttp\Exception\ClientException;
use Exception;

defined('CBLIB') or die();

class LinkedInProvider extends Provider
{
	/**
	 * https://docs.microsoft.com/en-us/linkedin/shared/integrations/people/profile-api?context=linkedin/consumer/context#permissions
	 *
	 * @var array
	 */
	protected $scope			=	array( 'r_liteprofile', 'r_emailaddress' );
	/** @var array  */
	protected $fields			=	array( 'id', 'localizedFirstName', 'localizedLastName', 'profilePicture(displayImage~:playableStreams)' );
	/** @var array  */
	protected $urls				=	array(	'base'		=>	'https://api.linkedin.com/v2',
											'authorize'	=>	'https://www.linkedin.com/oauth/v2/authorization',
											'access'	=>	'https://www.linkedin.com/oauth/v2/accessToken'
										);

	/**
	 * Authenticates a LinkedIn user (redirect and token exchange)
	 * https://docs.microsoft.com/en-us/linkedin/shared/authentication/authentication?context=linkedin/context
	 *
	 * @throws Exception
	 */
	public function authenticate()
	{
		$code					=	Application::Input()->get( 'code', null, GetterInterface::STRING );

		if ( ( ! $this->session()->get( 'linkedin.state', null, GetterInterface::STRING ) ) || ( $this->session()->get( 'linkedin.state', null, GetterInterface::STRING ) != Application::Input()->get( 'state', null, GetterInterface::STRING ) ) ) {
			$code				=	null;
		}

		if ( $code ) {
			$this->session()->set( 'linkedin.code', $code );

			$client				=	new Client();

			if ( cbGuzzleVersion() >= 6 ) {
				$key			=	'form_params';
			} else {
				$key			=	'body';
			}

			$options			=	array(	$key	=>	array(	'grant_type'	=>	'authorization_code',
																'code'			=>	$code,
																'redirect_uri'	=>	$this->callback,
																'client_id'		=>	$this->clientId,
																'client_secret'	=>	$this->clientSecret
															)
									);

			try {
				$result			=	$client->post( $this->urls['access'], $options );
			} catch( ClientException $e ) {
				$response		=	$this->response( $e->getResponse() );

				if ( ( $response instanceof ParamsInterface ) && $response->get( 'error.message', null, GetterInterface::STRING ) ) {
					$error		=	CBTxt::T( 'FAILED_EXCHANGE_CODE_ERROR', 'Failed to exchange code. Error: [error]', array( '[error]' => $response->get( 'error.message', null, GetterInterface::STRING ) ) );
				} else {
					$error		=	$e->getMessage();
				}

				$this->debug( $e );

				throw new Exception( $error );
			}

			$response			=	$this->response( $result );

			$this->debug( $result, $response );

			if ( ( $response instanceof ParamsInterface ) && $response->get( 'access_token', null, GetterInterface::STRING ) ) {
				$this->session()->set( 'linkedin.access_token', $response->get( 'access_token', null, GetterInterface::STRING ) );
				$this->session()->set( 'linkedin.expires', Application::Date( 'now', 'UTC' )->add( $response->get( 'expires_in', 0, GetterInterface::INT ) . ' SECONDS' )->getTimestamp() );
			} else {
				throw new Exception( CBTxt::T( 'Failed to retrieve access token.' ) );
			}
		} elseif ( ! $this->authorized() ) {
			$state				=	uniqid();

			$this->session()->set( 'linkedin.state', $state );

			$url				=	$this->urls['authorize']
								.	'?response_type=code'
								.	'&client_id=' . urlencode( $this->clientId )
								.	'&redirect_uri=' . urlencode( $this->callback )
								.	'&state=' . urlencode( $state )
								.	( $this->scope ? '&scope=' . urlencode( implode( ' ', $this->scope ) ) : null );

			cbRedirect( $url );
		}
	}

	/**
	 * Checks if access token exists and ensures it's not expired
	 *
	 * @return bool
	 */
	public function authorized()
	{
		$expired			=	true;

		if ( $this->session()->get( 'linkedin.access_token', null, GetterInterface::STRING ) ) {
			$expires		=	$this->session()->get( 'linkedin.expires', 0, GetterInterface::INT );

			if ( $expires ) {
				$expired	=	( Application::Date( 'now', 'UTC' )->getDateTime() > Application::Date( $expires, 'UTC' )->getDateTime() );
			}
		}

		return ( ! $expired );
	}

	/**
	 * Request current users LinkedIn profile
	 * https://docs.microsoft.com/en-us/linkedin/shared/integrations/people/profile-api?context=linkedin/consumer/context
	 *
	 * @param array $fields
	 * @return Profile
	 * @throws Exception
	 */
	public function profile( $fields = array() )
	{
		$profile				=	new Profile();

		if ( ! $fields ) {
			$fields				=	$this->fields;
		}

		$response				=	$this->api( '/me' . ( $fields ? '?projection=(' . implode( ',', $fields ) . ')' : null ) );

		if ( $response instanceof ParamsInterface ) {
			$fieldMap			=	array(	'id'			=>	'id',
											'firstname'		=>	'localizedFirstName',
											'lastname'		=>	'localizedLastName',
											'avatar'		=>	'profilePicture.displayImage~.elements.0.identifiers.0.identifier'
										);

			foreach ( $fieldMap as $cbField => $pField ) {
				$profile->set( $cbField, $response->get( $pField, null, GetterInterface::STRING ) );
			}

			if ( $profile->get( 'id', null, GetterInterface::STRING ) ) {
				$this->session()->set( 'linkedin.id', $profile->get( 'id', null, GetterInterface::STRING ) );
			}

			$emails				=	$this->api( '/emailAddress?q=members&projection=(elements*(handle~))' );

			if ( $emails instanceof ParamsInterface ) {
				$profile->set( 'email', $emails->get( 'elements.0.handle~.emailAddress', null, GetterInterface::STRING ) );
			}

			$profile->set( 'profile', $response );
		}

		return $profile;
	}

	/**
	 * Make a custom LinkedIn API request
	 * https://docs.microsoft.com/en-us/linkedin/shared/api-guide/concepts?context=linkedin/consumer/context
	 *
	 * @param string $api
	 * @param string $type
	 * @param array  $params
	 * @param array  $headers
	 * @return string|Registry|SimpleXMLElement
	 * @throws Exception
	 */
	public function api( $api, $type = 'GET', $params = array(), $headers = array() )
	{
		$client							=	new Client();

		if ( $this->session()->get( 'linkedin.access_token', null, GetterInterface::STRING ) ) {
			$headers['Authorization']	=	'Bearer ' . $this->session()->get( 'linkedin.access_token', null, GetterInterface::STRING );
		}

		$options						=	array();

		if ( $headers ) {
			$options['headers']			=	$headers;
		}

		if ( $params ) {
			if ( $type == 'POST' ) {
				if ( cbGuzzleVersion() >= 6 ) {
					$options['form_params']		=	$params;
				} else {
					$options['body']			=	$params;
				}
			} else {
				$options['query']		=	$params;
			}
		}

		try {
			if ( $type == 'POST' ) {
				$result					=	$client->post( $this->urls['base'] . $api, $options );
			} else {
				$result					=	$client->get( $this->urls['base'] . $api, $options );
			}
		} catch( ClientException $e ) {
			$response					=	$this->response( $e->getResponse() );

			if ( ( $response instanceof ParamsInterface ) && $response->get( 'message', null, GetterInterface::STRING ) ) {
				$error					=	CBTxt::T( 'FAILED_API_REQUEST_ERROR', 'Failed API request [api]. Error: [error]', array( '[api]' => $api, '[error]' => $response->get( 'message', null, GetterInterface::STRING ) ) );
			} else {
				$error					=	$e->getMessage();
			}

			$this->debug( $e );

			throw new Exception( $error );
		}

		$response						=	$this->response( $result );

		$this->debug( $result, $response );

		return $response;
	}
}
