<?php
/**
* Community Builder (TM)
* @version $Id: $
* @package CommunityBuilder
* @copyright (C) 2004-2019 www.joomlapolis.com / Lightning MultiCom SA - and its licensors, all rights reserved
* @license http://www.gnu.org/licenses/old-licenses/gpl-2.0.html GNU/GPL version 2
*/

namespace CB\Plugin\Connect;

use CBLib\Application\Application;
use CBLib\Registry\GetterInterface;
use CBLib\Language\CBTxt;
use CB\Database\Table\FieldTable;
use CBLib\Registry\Registry;

defined('CBLIB') or die();

class CBConnect
{

	/**
	 * Returns the current return url or generates one from current page
	 *
	 * @param bool|false $current
	 * @param bool|false $raw
	 * @return null|string
	 */
	static public function getReturn( $current = false, $raw = false )
	{
		static $cache				=	array();

		if ( ! isset( $cache[$current] ) ) {
			$url					=	null;

			if ( $current ) {
				$returnUrl			=	Application::Input()->get( 'get/return', '', GetterInterface::BASE64 );

				if ( $returnUrl ) {
					$returnUrl		=	base64_decode( $returnUrl );

					if ( \JUri::isInternal( $returnUrl ) ) {
						$url		=	$returnUrl;
					}
				}
			} else {
				$isHttps			=	( isset( $_SERVER['HTTPS'] ) && ( ! empty( $_SERVER['HTTPS'] ) ) && ( $_SERVER['HTTPS'] != 'off' ) );
				$returnUrl			=	'http' . ( $isHttps ? 's' : '' ) . '://' . $_SERVER['HTTP_HOST'];

				if ( ( ! empty( $_SERVER['PHP_SELF'] ) ) && ( ! empty( $_SERVER['REQUEST_URI'] ) ) ) {
					$returnUrl		.=	$_SERVER['REQUEST_URI'];
				} else {
					$returnUrl		.=	$_SERVER['SCRIPT_NAME'];

					if ( isset( $_SERVER['QUERY_STRING'] ) && ( ! empty( $_SERVER['QUERY_STRING'] ) ) ) {
						$returnUrl	.=	'?' . $_SERVER['QUERY_STRING'];
					}
				}

				$url				=	cbUnHtmlspecialchars( preg_replace( '/[\\\"\\\'][\\s]*javascript:(.*)[\\\"\\\']/', '""', preg_replace( '/eval\((.*)\)/', '', htmlspecialchars( urldecode( $returnUrl ) ) ) ) );
			}

			if ( preg_match( '/index\.php\?option=com_comprofiler&view=(registers|saveregisters|confirm|lostpassword|login|logout)/', $returnUrl ) ) {
				$url				=	'index.php';
			}

			$cache[$current]		=	$url;
		}

		$return						=	$cache[$current];

		if ( ( ! $raw ) && $return ) {
			$return					=	base64_encode( $return );
		}

		return $return;
	}

	/**
	 * Redirects to the return url if available otherwise to the url specified
	 *
	 * @param string      $provider
	 * @param string      $url
	 * @param null|string $message
	 * @param string      $messageType
	 */
	static public function returnRedirect( $provider, $url, $message = null, $messageType = 'message' )
	{
		$returnUrl			=	null;

		if ( $provider ) {
			$session		=	Application::Session();

			if ( $session->get( $provider . '.return', null, GetterInterface::STRING ) ) {
				$returnUrl	=	base64_decode( $session->get( $provider . '.return', null, GetterInterface::STRING ) );
			}
		}

		if ( ! $returnUrl ) {
			$returnUrl		=	self::getReturn( true, true );
		}

		if ( ! $returnUrl ) {
			$returnUrl		=	$url;
		}

		cbRedirect( $returnUrl, $message, $messageType );
	}

	/**
	 * Returns the minimum size needed for an image field
	 * This is used to tell APIs what size image we need
	 *
	 * @param string $fieldName
	 * @return int
	 * @throws \Exception
	 */
	static public function getMinImgLimit( $fieldName )
	{
		static $cache			=	array();

		if ( ! isset( $cache[$fieldName] ) ) {
			$field				=	new FieldTable();

			$field->load( array( 'name' => $fieldName ) );

			$field->set( 'params', new Registry( $field->get( 'params', null, GetterInterface::RAW ) ) );

			$width				=	$field->params->get( 'avatarWidth', ( $fieldName == 'canvas' ? 1280 : '' ), GetterInterface::STRING );

			if ( $width == '' ) {
				if ( isset( $ueConfig['avatarWidth'] ) ) {
					$width		=	$ueConfig['avatarWidth'];
				} else {
					$width		=	200;
				}
			}

			$height				=	$field->params->get( 'avatarHeight', ( $fieldName == 'canvas' ? 640 : '' ), GetterInterface::STRING );

			if ( $height == '' ) {
				if ( isset( $ueConfig['avatarHeight'] ) ) {
					$height		=	$ueConfig['avatarHeight'];
				} else {
					$height		=	500;
				}
			}

			$cache[$fieldName]	=	( $width > $height ? $width : $height );
		}

		return $cache[$fieldName];
	}

	/**
	 * Returns an array of available CB Connect providers
	 *
	 * @return array
	 */
	static public function getProviders()
	{
		return array(	'facebook'		=>	array(	'name'			=>	CBTxt::T( 'Facebook' ),
													'field'			=>	'fb_userid',
													'icon'			=>	'facebook',
													'button'		=>	'primary',
													'profile'		=>	null,
													'permissions'	=>	array(	'user_age_range', 'user_birthday', 'user_friends', 'user_gender', 'user_hometown', 'user_likes', 'user_link', 'user_location', 'user_photos',
																				'user_posts', 'user_videos'
																			),
													'fields'		=>	array(	'id', 'age_range.min', 'age_range.max', 'birthday', 'email', 'favorite_athletes', 'favorite_athletes.0.name', 'favorite_teams', 'favorite_teams.0.name',
																				'first_name', 'gender', 'hometown', 'hometown.name', 'inspirational_people', 'inspirational_people.0.name', 'languages', 'languages.0.name', 'last_name',
																				'link', 'location', 'location.name', 'name', 'name_format', 'quotes', 'significant_other.name', 'sports', 'sports.0.name'
																			),
													'ssl'			=>	false,
													'callback'		=>	'normal'
												),
						'twitter'		=>	array(	'name'			=>	CBTxt::T( 'Twitter' ),
													'field'			=>	'twitter_userid',
													'icon'			=>	'twitter text-info',
													'button'		=>	'light border',
													'profile'		=>	'https://twitter.com/intent/user?user_id=[id]',
													'permissions'	=>	array(),
													'fields'		=>	array(	'id', 'name', 'screen_name', 'location', 'description', 'url', 'protected', 'followers_count', 'friends_count', 'listed_count',
																				'created_at', 'favourites_count', 'utc_offset', 'time_zone', 'geo_enabled', 'verified', 'statuses_count', 'lang', 'contributors_enabled',
																				'is_translator', 'is_translation_enabled', 'profile_background_color', 'profile_background_image_url', 'profile_background_image_url_https',
																				'profile_background_tile', 'profile_image_url', 'profile_image_url_https', 'profile_banner_url', 'profile_link_color', 'profile_sidebar_border_color',
																				'profile_sidebar_fill_color', 'profile_text_color', 'profile_use_background_image', 'has_extended_profile', 'default_profile', 'default_profile_image',
																				'following', 'follow_request_sent', 'notifications'
																			),
													'ssl'			=>	false,
													'callback'		=>	'normal'
												),
						'google'		=>	array(	'name'			=>	CBTxt::T( 'Google' ),
													'field'			=>	'google_userid',
													'icon'			=>	'google',
													'button'		=>	'light border',
													'profile'		=>	null,
													'permissions'	=>	array(	'https://www.googleapis.com/auth/activity', 'https://www.googleapis.com/auth/adexchange.buyer', 'https://www.googleapis.com/auth/admin.datatransfer', 'https://www.googleapis.com/auth/admin.datatransfer.readonly',
																				'https://www.googleapis.com/auth/admin.directory.customer', 'https://www.googleapis.com/auth/admin.directory.customer.readonly', 'https://www.googleapis.com/auth/admin.directory.device.chromeos', 'https://www.googleapis.com/auth/admin.directory.device.chromeos.readonly',
																				'https://www.googleapis.com/auth/admin.directory.device.mobile', 'https://www.googleapis.com/auth/admin.directory.device.mobile.action', 'https://www.googleapis.com/auth/admin.directory.device.mobile.readonly', 'https://www.googleapis.com/auth/admin.directory.domain',
																				'https://www.googleapis.com/auth/admin.directory.domain.readonly', 'https://www.googleapis.com/auth/admin.directory.group', 'https://www.googleapis.com/auth/admin.directory.group.member', 'https://www.googleapis.com/auth/admin.directory.group.member.readonly',
																				'https://www.googleapis.com/auth/admin.directory.group.readonly', 'https://www.googleapis.com/auth/admin.directory.notifications', 'https://www.googleapis.com/auth/admin.directory.orgunit', 'https://www.googleapis.com/auth/admin.directory.orgunit.readonly',
																				'https://www.googleapis.com/auth/admin.directory.resource.calendar', 'https://www.googleapis.com/auth/admin.directory.resource.calendar.readonly', 'https://www.googleapis.com/auth/admin.directory.rolemanagement', 'https://www.googleapis.com/auth/admin.directory.rolemanagement.readonly',
																				'https://www.googleapis.com/auth/admin.directory.user', 'https://www.googleapis.com/auth/admin.directory.user.alias', 'https://www.googleapis.com/auth/admin.directory.user.alias.readonly', 'https://www.googleapis.com/auth/admin.directory.user.readonly',
																				'https://www.googleapis.com/auth/admin.directory.user.security', 'https://www.googleapis.com/auth/admin.directory.userschema', 'https://www.googleapis.com/auth/admin.directory.userschema.readonly', 'https://www.googleapis.com/auth/admin.reports.audit.readonly',
																				'https://www.googleapis.com/auth/admin.reports.usage.readonly', 'https://www.googleapis.com/auth/adsense', 'https://www.googleapis.com/auth/adsense.readonly', 'https://www.googleapis.com/auth/adsensehost',
																				'https://www.googleapis.com/auth/analytics', 'https://www.googleapis.com/auth/analytics.edit', 'https://www.googleapis.com/auth/analytics.manage.users', 'https://www.googleapis.com/auth/analytics.manage.users.readonly',
																				'https://www.googleapis.com/auth/analytics.provision', 'https://www.googleapis.com/auth/analytics.readonly', 'https://www.googleapis.com/auth/analytics.user.deletion', 'https://www.googleapis.com/auth/androidenterprise',
																				'https://www.googleapis.com/auth/androidmanagement', 'https://www.googleapis.com/auth/androidpublisher', 'https://www.googleapis.com/auth/appengine.admin', 'https://www.googleapis.com/auth/apps.groups.migration',
																				'https://www.googleapis.com/auth/apps.groups.settings', 'https://www.googleapis.com/auth/apps.licensing', 'https://www.googleapis.com/auth/apps.order', 'https://www.googleapis.com/auth/apps.order.readonly',
																				'https://www.googleapis.com/auth/appstate', 'https://www.googleapis.com/auth/bigquery', 'https://www.googleapis.com/auth/bigquery.insertdata', 'https://www.googleapis.com/auth/blogger',
																				'https://www.googleapis.com/auth/blogger.readonly', 'https://www.googleapis.com/auth/books', 'https://www.googleapis.com/auth/calendar', 'https://www.googleapis.com/auth/calendar.events',
																				'https://www.googleapis.com/auth/calendar.events.readonly', 'https://www.googleapis.com/auth/calendar.readonly', 'https://www.googleapis.com/auth/calendar.settings.readonly', 'https://www.googleapis.com/auth/classroom.announcements',
																				'https://www.googleapis.com/auth/classroom.announcements.readonly', 'https://www.googleapis.com/auth/classroom.courses', 'https://www.googleapis.com/auth/classroom.courses.readonly', 'https://www.googleapis.com/auth/classroom.coursework.me',
																				'https://www.googleapis.com/auth/classroom.coursework.me.readonly', 'https://www.googleapis.com/auth/classroom.coursework.students', 'https://www.googleapis.com/auth/classroom.coursework.students.readonly', 'https://www.googleapis.com/auth/classroom.guardianlinks.me.readonly',
																				'https://www.googleapis.com/auth/classroom.guardianlinks.students', 'https://www.googleapis.com/auth/classroom.guardianlinks.students.readonly', 'https://www.googleapis.com/auth/classroom.profile.emails', 'https://www.googleapis.com/auth/classroom.profile.photos',
																				'https://www.googleapis.com/auth/classroom.push-notifications', 'https://www.googleapis.com/auth/classroom.rosters', 'https://www.googleapis.com/auth/classroom.rosters.readonly', 'https://www.googleapis.com/auth/classroom.student-submissions.me.readonly',
																				'https://www.googleapis.com/auth/classroom.student-submissions.students.readonly', 'https://www.googleapis.com/auth/cloud-language', 'https://www.googleapis.com/auth/cloud-platform', 'https://www.googleapis.com/auth/cloud-platform.read-only',
																				'https://www.googleapis.com/auth/cloud-translation', 'https://www.googleapis.com/auth/cloud-vision', 'https://www.googleapis.com/auth/cloud_debugger', 'https://www.googleapis.com/auth/cloudiot',
																				'https://www.googleapis.com/auth/cloudruntimeconfig', 'https://www.googleapis.com/auth/compute', 'https://www.googleapis.com/auth/compute.readonly', 'https://www.googleapis.com/auth/contacts',
																				'https://www.googleapis.com/auth/contacts.readonly', 'https://www.googleapis.com/auth/content', 'https://www.googleapis.com/auth/datastore', 'https://www.googleapis.com/auth/ddmconversions',
																				'https://www.googleapis.com/auth/devstorage.full_control', 'https://www.googleapis.com/auth/devstorage.read_only', 'https://www.googleapis.com/auth/devstorage.read_write', 'https://www.googleapis.com/auth/dfareporting',
																				'https://www.googleapis.com/auth/dfatrafficking', 'https://www.googleapis.com/auth/dialogflow', 'https://www.googleapis.com/auth/documents', 'https://www.googleapis.com/auth/doubleclickbidmanager',
																				'https://www.googleapis.com/auth/doubleclicksearch', 'https://www.googleapis.com/auth/drive', 'https://www.googleapis.com/auth/drive.appdata', 'https://www.googleapis.com/auth/drive.file',
																				'https://www.googleapis.com/auth/drive.metadata', 'https://www.googleapis.com/auth/drive.metadata.readonly', 'https://www.googleapis.com/auth/drive.photos.readonly', 'https://www.googleapis.com/auth/drive.readonly',
																				'https://www.googleapis.com/auth/drive.scripts', 'https://www.googleapis.com/auth/ediscovery', 'https://www.googleapis.com/auth/ediscovery.readonly', 'https://www.googleapis.com/auth/firebase',
																				'https://www.googleapis.com/auth/firebase.readonly', 'https://www.googleapis.com/auth/fitness.activity.read', 'https://www.googleapis.com/auth/fitness.activity.write', 'https://www.googleapis.com/auth/fitness.blood_glucose.read',
																				'https://www.googleapis.com/auth/fitness.blood_glucose.write', 'https://www.googleapis.com/auth/fitness.blood_pressure.read', 'https://www.googleapis.com/auth/fitness.blood_pressure.write', 'https://www.googleapis.com/auth/fitness.body.read',
																				'https://www.googleapis.com/auth/fitness.body.write', 'https://www.googleapis.com/auth/fitness.body_temperature.read', 'https://www.googleapis.com/auth/fitness.body_temperature.write', 'https://www.googleapis.com/auth/fitness.location.read',
																				'https://www.googleapis.com/auth/fitness.location.write', 'https://www.googleapis.com/auth/fitness.nutrition.read', 'https://www.googleapis.com/auth/fitness.nutrition.write', 'https://www.googleapis.com/auth/fitness.oxygen_saturation.read',
																				'https://www.googleapis.com/auth/fitness.oxygen_saturation.write', 'https://www.googleapis.com/auth/fitness.reproductive_health.read', 'https://www.googleapis.com/auth/fitness.reproductive_health.write', 'https://www.googleapis.com/auth/forms',
																				'https://www.googleapis.com/auth/forms.currentonly', 'https://www.googleapis.com/auth/fusiontables', 'https://www.googleapis.com/auth/fusiontables.readonly', 'https://www.googleapis.com/auth/games',
																				'https://www.googleapis.com/auth/genomics', 'https://www.googleapis.com/auth/genomics.readonly', 'https://www.googleapis.com/auth/glass.location', 'https://www.googleapis.com/auth/glass.timeline',
																				'https://www.googleapis.com/auth/gmail.compose', 'https://www.googleapis.com/auth/gmail.insert', 'https://www.googleapis.com/auth/gmail.labels', 'https://www.googleapis.com/auth/gmail.metadata',
																				'https://www.googleapis.com/auth/gmail.modify', 'https://www.googleapis.com/auth/gmail.readonly', 'https://www.googleapis.com/auth/gmail.send', 'https://www.googleapis.com/auth/gmail.settings.basic',
																				'https://www.googleapis.com/auth/gmail.settings.sharing', 'https://www.googleapis.com/auth/groups', 'https://www.googleapis.com/auth/jobs', 'https://www.googleapis.com/auth/logging.admin',
																				'https://www.googleapis.com/auth/logging.read', 'https://www.googleapis.com/auth/logging.write', 'https://www.googleapis.com/auth/manufacturercenter', 'https://www.googleapis.com/auth/monitoring',
																				'https://www.googleapis.com/auth/monitoring.read', 'https://www.googleapis.com/auth/monitoring.write', 'https://www.googleapis.com/auth/ndev.clouddns.readonly', 'https://www.googleapis.com/auth/ndev.clouddns.readwrite',
																				'https://www.googleapis.com/auth/ndev.cloudman', 'https://www.googleapis.com/auth/ndev.cloudman.readonly', 'https://www.googleapis.com/auth/presentations', 'https://www.googleapis.com/auth/presentations.readonly',
																				'https://www.googleapis.com/auth/pubsub', 'https://www.googleapis.com/auth/replicapool', 'https://www.googleapis.com/auth/replicapool.readonly', 'https://www.googleapis.com/auth/script.deployments',
																				'https://www.googleapis.com/auth/script.deployments.readonly', 'https://www.googleapis.com/auth/script.metrics', 'https://www.googleapis.com/auth/script.processes', 'https://www.googleapis.com/auth/script.projects',
																				'https://www.googleapis.com/auth/script.projects.readonly', 'https://www.googleapis.com/auth/service.management', 'https://www.googleapis.com/auth/service.management.readonly', 'https://www.googleapis.com/auth/servicecontrol',
																				'https://www.googleapis.com/auth/siteverification', 'https://www.googleapis.com/auth/siteverification.verify_only', 'https://www.googleapis.com/auth/source.full_control', 'https://www.googleapis.com/auth/source.read_only',
																				'https://www.googleapis.com/auth/source.read_write', 'https://www.googleapis.com/auth/spanner.admin', 'https://www.googleapis.com/auth/spanner.data', 'https://www.googleapis.com/auth/spreadsheets',
																				'https://www.googleapis.com/auth/spreadsheets.readonly', 'https://www.googleapis.com/auth/sqlservice.admin', 'https://www.googleapis.com/auth/streetviewpublish', 'https://www.googleapis.com/auth/tagmanager.delete.containers',
																				'https://www.googleapis.com/auth/tagmanager.edit.containers', 'https://www.googleapis.com/auth/tagmanager.edit.containerversions', 'https://www.googleapis.com/auth/tagmanager.manage.accounts', 'https://www.googleapis.com/auth/tagmanager.manage.users',
																				'https://www.googleapis.com/auth/tagmanager.publish', 'https://www.googleapis.com/auth/tagmanager.readonly', 'https://www.googleapis.com/auth/tasks', 'https://www.googleapis.com/auth/tasks.readonly',
																				'https://www.googleapis.com/auth/trace.append', 'https://www.googleapis.com/auth/urlshortener', 'https://www.googleapis.com/auth/user.addresses.read', 'https://www.googleapis.com/auth/user.birthday.read',
																				'https://www.googleapis.com/auth/user.emails.read', 'https://www.googleapis.com/auth/user.phonenumbers.read', 'https://www.googleapis.com/auth/userinfo.email', 'https://www.googleapis.com/auth/userinfo.profile',
																				'https://www.googleapis.com/auth/userlocation.beacon.registry', 'https://www.googleapis.com/auth/webmasters', 'https://www.googleapis.com/auth/webmasters.readonly', 'https://www.googleapis.com/auth/youtube',
																				'https://www.googleapis.com/auth/youtube.force-ssl', 'https://www.googleapis.com/auth/youtube.readonly', 'https://www.googleapis.com/auth/youtube.upload', 'https://www.googleapis.com/auth/youtubepartner',
																				'https://www.googleapis.com/auth/youtubepartner-channel-audit', 'https://www.googleapis.com/auth/yt-analytics-monetary.readonly', 'https://www.googleapis.com/auth/yt-analytics.readonly'
																			),
													'fields'		=>	array(	'resourceName', 'names.0.displayName', 'names.0.givenName', 'names.0.middleName', 'names.0.familyName', 'nicknames.0.value', 'emailAddresses.0.value', 'emailAddresses.1.value', 'emailAddresses.2.value', 'photos.0.url',
																				'coverPhotos.0.url', 'locales.0.value', 'locales.1.value', 'locales.2.value', 'genders.0.value', 'ageRanges.0.ageRange', 'birthdays.0.date', 'birthdays.0.text', 'events.0.type', 'events.0.date', 'events.1.type', 'events.1.date',
																				'events.2.type', 'events.2.date', 'addresses.0.type', 'addresses.0.poBox', 'addresses.0.streetAddress', 'addresses.0.extendedAddress', 'addresses.0.city', 'addresses.0.region', 'addresses.0.postalCode', 'addresses.0.country',
																				'addresses.0.countryCode', 'residences.0.value', 'phoneNumbers.0.type', 'phoneNumbers.0.value', 'phoneNumbers.1.type', 'phoneNumbers.1.value', 'phoneNumbers.2.type', 'phoneNumbers.2.value', 'imClients.0.type', 'imClients.0.protocol', 'imClients.0.username',
																				'imClients.1.type', 'imClients.1.protocol', 'imClients.1.username', 'imClients.2.type', 'imClients.2.protocol', 'imClients.2.username', 'interests.0.value', 'interests.1.value', 'interests.2.value', 'occupations.0.value', 'occupations.1.value', 'occupations.2.value',
																				'organizations.0.type', 'organizations.0.name', 'organizations.0.title', 'organizations.1.type', 'organizations.1.name', 'organizations.1.title', 'organizations.2.type', 'organizations.2.name', 'organizations.2.title', 'relations.0.type', 'relations.0.person',
																				'relations.1.type', 'relations.1.person', 'relations.2.type', 'relations.2.person', 'relationshipInterests.0.value', 'relationshipInterests.1.value', 'relationshipInterests.2.value', 'relationshipStatuses.0.value', 'relationshipStatuses.1.value', 'relationshipStatuses.2.value',
																				'skills.0.value', 'skills.1.value', 'skills.2.value', 'taglines.0.value', 'taglines.1.value', 'taglines.2.value', 'urls.0.type', 'urls.0.value', 'urls.1.type', 'urls.1.value', 'urls.2.type', 'urls.2.value'
																			),
													'ssl'			=>	false,
													'callback'		=>	'normal'
												),
						'linkedin'		=>	array(	'name'			=>	CBTxt::T( 'LinkedIn' ),
													'field'			=>	'linkedin_userid',
													'icon'			=>	'linkedin',
													'button'		=>	'info',
													'profile'		=>	null,
													'permissions'	=>	array(	'r_basicprofile', 'r_fullprofile' ),
													'fields'		=>	array(	'id', 'localizedFirstName', 'localizedLastName', 'localizedMaidenName', 'localizedHeadline', 'vanityName' ),
													'ssl'			=>	false,
													'callback'		=>	'normal'
												),
						'windowslive'	=>	array(	'name'			=>	CBTxt::T( 'Windows Live' ),
													'field'			=>	'windowslive_userid',
													'icon'			=>	'windows',
													'button'		=>	'light border',
													'profile'		=>	'https://profile.live.com/[id]',
													'permissions'	=>	array(	'wl.offline_access', 'wl.birthday', 'wl.calendars', 'wl.calendars_update', 'wl.contacts_birthday', 'wl.contacts_create', 'wl.contacts_calendars', 'wl.contacts_photos',
																				'wl.contacts_skydrive', 'wl.events_create', 'wl.imap', 'wl.phone_numbers', 'wl.photos', 'wl.postal_addresses', 'wl.skydrive', 'wl.skydrive_update', 'wl.work_profile', 'office.onenote_create'
																			),
													'fields'		=>	array(	'id', 'name', 'first_name', 'last_name', 'link', 'birth_day', 'birth_month', 'birth_year', 'emails.preferred', 'emails.account', 'emails.personal', 'emails.business', 'emails.other',
																				'addresses.personal.street', 'addresses.personal.street_2', 'addresses.personal.city', 'addresses.personal.state', 'addresses.personal.postal_code', 'addresses.personal.region',
																				'addresses.business.street', 'addresses.business.street_2', 'addresses.business.city', 'addresses.business.state', 'addresses.business.postal_code', 'addresses.business.region',
																				'phones.personal', 'phones.business', 'phones.mobile', 'locale', 'updated_time'
																			),
													'ssl'			=>	false,
													'callback'		=>	'normal'
												),
						'instagram'		=>	array(	'name'			=>	CBTxt::T( 'Instagram' ),
													'field'			=>	'instagram_userid',
													'icon'			=>	'instagram',
													'button'		=>	'warning',
													'profile'		=>	'http://instagram.com/[id]',
													'permissions'	=>	array(	'public_content', 'follower_list', 'comments', 'relationships', 'likes' ),
													'fields'		=>	array(	'id', 'username', 'full_name', 'profile_picture', 'bio', 'website', 'counts.media', 'counts.followers', 'counts.followed_by' ),
													'ssl'			=>	false,
													'callback'		=>	'normal'
												),
						'foursquare'	=>	array(	'name'			=>	CBTxt::T( 'Foursquare' ),
													'field'			=>	'foursquare_userid',
													'icon'			=>	'foursquare',
													'button'		=>	'info',
													'profile'		=>	'https://foursquare.com/user/[id]',
													'permissions'	=>	array(),
													'fields'		=>	array(	'id', 'firstName', 'lastName', 'gender', 'relationship', 'canonicalUrl', 'friends.count', 'birthday', 'tips.count', 'homeCity', 'bio', 'contact.phone', 'contact.verifiedPhone',
																				'contact.email', 'contact.twitter', 'contact.facebook', 'photos.count', 'checkinPings', 'pings', 'type', 'mayorships.count', 'checkins.count', 'requests.count', 'lists.count',
																				'blockedStatus', 'createdAt', 'referralId', 'venue', 'scores.recent', 'scores.max', 'scores.checkinsCount', 'scores.goal'
																			),
													'ssl'			=>	false,
													'callback'		=>	'normal'
												),
						'github'		=>	array(	'name'			=>	CBTxt::T( 'GitHub' ),
													'field'			=>	'github_userid',
													'icon'			=>	'github',
													'button'		=>	'light border',
													'profile'		=>	null,
													'permissions'	=>	array(	'user:follow', 'repo', 'repo:status', 'repo_deployment', 'public_repo', 'repo:invite', 'security_events', 'admin:repo_hook', 'write:repo_hook', 'read:repo_hook', 'admin:org', 'write:org', 'read:org',
																				'admin:public_key', 'write:public_key', 'read:public_key', 'admin:org_hook', 'gist', 'notifications', 'delete_repo', 'write:discussion', 'read:discussion', 'write:packages', 'read:packages',
																				'delete:packages', 'admin:gpg_key', 'write:gpg_key', 'read:gpg_key', 'workflow'
																			),
													'fields'		=>	array(	'login', 'id', 'node_id', 'avatar_url', 'gravatar_id', 'url', 'html_url', 'followers_url', 'following_url', 'gists_url', 'starred_url', 'subscriptions_url', 'organizations_url', 'repos_url',
																				'events_url', 'received_events_url', 'type', 'site_admin', 'name', 'company', 'blog', 'location', 'email', 'hireable', 'bio', 'twitter_username', 'public_repos', 'public_gists', 'followers',
																				'following', 'created_at', 'updated_at', 'private_gists', 'total_private_repos', 'owned_private_repos', 'disk_usage', 'collaborators', 'two_factor_authentication', 'plan.name', 'plan.space',
																				'plan.private_repos', 'plan.collaborators'
																			),
													'ssl'			=>	false,
													'callback'		=>	'normal'
												),
						'vkontakte'		=>	array(	'name'			=>	CBTxt::T( 'VKontakte' ),
													'field'			=>	'vkontakte_userid',
													'icon'			=>	'vk',
													'button'		=>	'primary',
													'profile'		=>	'http://vk.com/id[id]',
													'permissions'	=>	array(	'notify', 'friends', 'photos', 'audio', 'video', 'docs', 'notes', 'pages', 'status', 'offers', 'questions', 'groups', 'notifications', 'stats', 'ads', 'offline' ),
													'fields'		=>	array(	'uid', 'first_name', 'last_name', 'deactivated', 'verified', 'blacklisted', 'sex', 'bdate', 'city.title', 'country.title', 'home_town', 'photo_50', 'photo_100', 'photo_200_orig', 'photo_200',
																				'photo_400_orig', 'photo_max', 'photo_max_orig', 'online', 'domain', 'has_mobile', 'contacts.mobile_phone', 'contacts.home_phone', 'site', 'status', 'last_seen.time', 'followers_count',
																				'common_count', 'counters.albums', 'counters.videos', 'counters.audios', 'counters.notes', 'counters.friends', 'counters.groups', 'counters.online_friends', 'counters.mutual_friends',
																				'counters.user_videos', 'counters.followers', 'counters.user_photos', 'counters.subscriptions', 'occupation.name', 'nickname', 'relation', 'personal.political', 'personal.langs',
																				'personal.religion', 'personal.inspired_by', 'personal.people_main', 'personal.life_main', 'personal.smoking', 'personal.alcohol', 'wall_comments', 'activities', 'interests', 'music',
																				'movies', 'tv', 'books', 'games', 'about', 'quotes', 'can_post', 'can_see_all_posts', 'can_see_audio', 'can_write_private_message', 'timezone', 'screen_name'
																			),
													'ssl'			=>	false,
													'callback'		=>	'normal'
												),
						'steam'			=>	array(	'name'			=>	CBTxt::T( 'Steam' ),
													'field'			=>	'steam_userid',
													'icon'			=>	'steam',
													'button'		=>	'success',
													'profile'		=>	'http://steamcommunity.com/profiles/[id]',
													'permissions'	=>	array(),
													'fields'		=>	array(	'steamid', 'personaname', 'profileurl', 'avatar', 'avatarmedium', 'avatarfull', 'personastate', 'communityvisibilitystate', 'profilestate', 'lastlogoff', 'commentpermission', 'realname',
																				'primaryclanid', 'timecreated', 'gameid', 'gameserverip', 'gameextrainfo', 'cityid', 'loccountrycode', 'locstatecode', 'loccityid'
																			),
													'ssl'			=>	false,
													'callback'		=>	'normal'
												),
						'reddit'		=>	array(	'name'			=>	CBTxt::T( 'Reddit' ),
													'field'			=>	'reddit_userid',
													'icon'			=>	'reddit-alien',
													'button'		=>	'light border',
													'profile'		=>	null,
													'permissions'	=>	array(	'edit', 'flair', 'history', 'modconfig', 'modflair', 'modlog', 'modposts', 'modwiki', 'mysubreddits', 'privatemessages', 'read', 'report', 'save', 'submit', 'subscribe', 'vote', 'wikiedit', 'wikiread' ),
													'fields'		=>	array(	'id', 'name', 'created', 'hide_from_robots', 'is_suspended', 'created_utc', 'link_karma', 'comment_karma', 'over_18', 'is_gold', 'is_mod', 'gold_expiration', 'inbox_count', 'has_verified_email',
																				'gold_creddits', 'suspension_expiration_utc'
																			),
													'ssl'			=>	false,
													'callback'		=>	'normal'
												),
						'twitch'		=>	array(	'name'			=>	CBTxt::T( 'Twitch' ),
													'field'			=>	'twitch_userid',
													'icon'			=>	'twitch',
													'button'		=>	'light border',
													'profile'		=>	null,
													'permissions'	=>	array(	'user_blocks_edit', 'user_blocks_read', 'user_follows_edit', 'channel_read', 'channel_editor', 'channel_commercial', 'channel_stream', 'channel_subscriptions', 'user_subscriptions',
																				'channel_check_subscription', 'chat_login'
																			),
													'fields'		=>	array(	'_id', 'type', 'name', 'created_at', 'updated_at', '_links.self', 'logo', 'display_name', 'email', 'partnered', 'bio', 'notifications.push', 'notifications.email' ),
													'ssl'			=>	false,
													'callback'		=>	'normal'
												),
						'stackexchange'	=>	array(	'name'			=>	CBTxt::T( 'Stack Exchange' ),
													'field'			=>	'stackexchange_userid',
													'icon'			=>	'stack-exchange',
													'button'		=>	'info',
													'profile'		=>	'http://stackoverflow.com/users/[id]',
													'permissions'	=>	array(	'read_inbox', 'no_expiry', 'write_access', 'private_info' ),
													'fields'		=>	array(	'user_id', 'display_name', 'accept_rate', 'account_id', 'age', 'badge_counts.bronze', 'badge_counts.silver', 'badge_counts.gold', 'creation_date', 'is_employee', 'last_access_date', 'last_modified_date',
																				'link', 'location', 'profile_image', 'reputation', 'reputation_change_day', 'reputation_change_month', 'reputation_change_quarter', 'reputation_change_week', 'reputation_change_year', 'timed_penalty_date',
																				'user_type', 'website_url'
																			),
													'ssl'			=>	false,
													'callback'		=>	'normal'
												),
						'pinterest'		=>	array(	'name'			=>	CBTxt::T( 'Pinterest' ),
													'field'			=>	'pinterest_userid',
													'icon'			=>	'pinterest text-danger',
													'button'		=>	'light border',
													'profile'		=>	null,
													'permissions'	=>	array(	'write_public', 'read_relationships', 'write_relationships' ),
													'fields'		=>	array(	'id', 'username', 'first_name', 'last_name', 'bio', 'created_at', 'counts.pins', 'counts.following', 'counts.followers', 'counts.boards', 'counts.likes' ),
													'ssl'			=>	true,
													'callback'		=>	'normal'
												),
						'amazon'		=>	array(	'name'			=>	CBTxt::T( 'Amazon' ),
													'field'			=>	'amazon_userid',
													'icon'			=>	'amazon',
													'button'		=>	'warning',
													'profile'		=>	null,
													'permissions'	=>	array(	'postal_code' ),
													'fields'		=>	array(	'user_id', 'name', 'email', 'postal_code' ),
													'ssl'			=>	false,
													'callback'		=>	'normal'
												),
						'yahoo'			=>	array(	'name'			=>	CBTxt::T( 'Yahoo' ),
													'field'			=>	'yahoo_userid',
													'icon'			=>	'yahoo',
													'button'		=>	'light border',
													'profile'		=>	null,
													'permissions'	=>	array(),
													'fields'		=>	array(	'guid', 'ageCategory', 'birthYear', 'birthdate', 'created', 'displayAge', 'familyName', 'gender', 'givenName', 'image.height', 'image.width', 'image.size', 'image.imageUrl', 'lang', 'location', 'memberSince',
																				'nickname', 'profileUrl', 'timeZone', 'isConnected'
																			),
													'ssl'			=>	false,
													'callback'		=>	'normal'
												),
						'paypal'		=>	array(	'name'			=>	CBTxt::T( 'PayPal' ),
													'field'			=>	'paypal_userid',
													'icon'			=>	'paypal',
													'button'		=>	'primary',
													'profile'		=>	null,
													'permissions'	=>	array(	'address', 'phone', 'https://uri.paypal.com/services/paypalattributes' ),
													'fields'		=>	array(	'user_id', 'sub', 'name', 'given_name', 'family_name', 'middle_name', 'picture', 'email', 'email_verified', 'gender', 'birthdate', 'zoneinfo', 'locale', 'phone_number', 'address.street_address', 'address.locality',
																				'address.region', 'address.postal_code', 'address.country', 'verified_account', 'account_type', 'age_range', 'payer_id'
																			),
													'ssl'			=>	false,
													'callback'		=>	'normal'
												),
						'disqus'		=>	array(	'name'			=>	CBTxt::T( 'Disqus' ),
													'field'			=>	'disqus_userid',
													'icon'			=>	'disqus',
													'button'		=>	'info',
													'profile'		=>	null,
													'permissions'	=>	array(	'write', 'admin' ),
													'fields'		=>	array(	'id', 'username', 'name', 'email', 'isFollowing', 'disable3rdPartyTrackers', 'isPowerContributor', 'isFollowedBy', 'isPrimary', 'numFollowers', 'rep', 'numFollowing', 'numPosts', 'location', 'isPrivate', 'joinedAt',
																				'numLikesReceived', 'about', 'url', 'numForumsFollowing', 'profileUrl', 'reputation', 'avatar.permalink', 'signedUrl', 'isAnonymous'
																			),
													'ssl'			=>	false,
													'callback'		=>	'normal'
												),
						'wordpress'		=>	array(	'name'			=>	CBTxt::T( 'WordPress' ),
													'field'			=>	'wordpress_userid',
													'icon'			=>	'wordpress text-muted',
													'button'		=>	'light border',
													'profile'		=>	'https://en.gravatar.com/[id]',
													'permissions'	=>	array(),
													'fields'		=>	array(	'ID', 'display_name', 'username', 'email', 'primary_blog', 'primary_blog_url', 'language', 'token_site_id', 'token_scope', 'avatar_URL', 'profile_URL', 'verified', 'email_verified', 'date', 'site_count', 'visible_site_count',
																				'has_unseen_notes', 'newest_note_type', 'phone_account', 'is_valid_google_apps_country', 'logout_URL'
																			),
													'ssl'			=>	false,
													'callback'		=>	'normal'
												),
						'meetup'		=>	array(	'name'			=>	CBTxt::T( 'Meetup' ),
													'field'			=>	'meetup_userid',
													'icon'			=>	'meetup text-danger',
													'button'		=>	'light border',
													'profile'		=>	'http://www.meetup.com/members/[id]',
													'permissions'	=>	array(	'ageless', 'event_management', 'group_edit', 'group_content_edit', 'group_join_edit', 'messaging', 'profile_edit', 'reporting', 'rsvp' ),
													'fields'		=>	array(	'id', 'name', 'bio', 'birthday.day', 'birthday.month', 'birthday.year', 'country', 'city', 'state', 'gender', 'hometown', 'joined', 'lang', 'lat', 'lon', 'link', 'membership_count', 'messagable', 'messaging_pref', 'photo.highres_link',
																				'photo.photo_id', 'photo.photo_link', 'photo.thumb_link', 'reachable', 'visited'
																			),
													'ssl'			=>	false,
													'callback'		=>	'normal'
												),
						'flickr'		=>	array(	'name'			=>	CBTxt::T( 'Flickr' ),
													'field'			=>	'flickr_userid',
													'icon'			=>	'flickr text-info',
													'button'		=>	'light border',
													'profile'		=>	'https://www.flickr.com/people/[id]/',
													'permissions'	=>	array(),
													'fields'		=>	array(	'profile.id', 'profile.join_date', 'profile.occupation', 'profile.hometown', 'profile.first_name', 'profile.last_name', 'profile.email', 'profile.profile_description',
																				'profile.city', 'profile.country', 'profile.facebook', 'profile.twitter', 'profile.tumblr', 'profile.instagram', 'profile.pinterest', 'person.ispro', 'person.can_buy_pro',
																				'person.iconserver', 'person.iconfarm', 'person.username._content', 'person.realname._content', 'person.location._content', 'person.description._content', 'person.photosurl._content',
																				'person.profileurl._content', 'person.mobileurl._content'
																			),
													'ssl'			=>	false,
													'callback'		=>	'normal'
												),
						'vimeo'			=>	array(	'name'			=>	CBTxt::T( 'Vimeo' ),
													'field'			=>	'vimeo_userid',
													'icon'			=>	'vimeo text-inverse',
													'button'		=>	'info',
													'profile'		=>	'https://vimeo.com/user[id]',
													'permissions'	=>	array(	'private', 'purchased', 'purchase', 'create', 'edit', 'delete', 'interact', 'upload', 'promo_codes', 'video_files' ),
													'fields'		=>	array(	'uri', 'name', 'link', 'location', 'bio', 'created_time', 'account', 'pictures.0.link', 'pictures.sizes.0.link' ),
													'ssl'			=>	false,
													'callback'		=>	'normal'
												),
						'line'			=>	array(	'name'			=>	CBTxt::T( 'LINE' ),
													'field'			=>	'line_userid',
													'icon'			=>	'line text-inverse',
													'button'		=>	'success',
													'profile'		=>	null,
													'permissions'	=>	array(	'openid' ),
													'fields'		=>	array(	'userId', 'displayName', 'pictureUrl', 'statusMessage' ),
													'ssl'			=>	false,
													'callback'		=>	'state'
												),
						'spotify'		=>	array(	'name'			=>	CBTxt::T( 'Spotify' ),
													'field'			=>	'spotify_userid',
													'icon'			=>	'spotify text-success',
													'button'		=>	'success',
													'profile'		=>	'https://open.spotify.com/user/[id]',
													'permissions'	=>	array(	'user-read-private', 'playlist-read-private', 'playlist-read-collaborative', 'playlist-modify-public', 'playlist-modify-private', 'streaming', 'ugc-image-upload', 'user-follow-modify', 'user-follow-read',
																				'user-library-read', 'user-library-modify', 'user-read-birthdate', 'user-top-read', 'user-read-playback-state', 'user-modify-playback-state', 'user-read-currently-playing', 'user-read-recently-played'
																			),
													'fields'		=>	array(	'id', 'display_name', 'email', 'birthdate', 'country', 'product', 'type', 'uri', 'href', 'images.0.url' ),
													'ssl'			=>	false,
													'callback'		=>	'normal'
												),
						'soundcloud'	=>	array(	'name'			=>	CBTxt::T( 'SoundCloud' ),
													'field'			=>	'soundcloud_userid',
													'icon'			=>	'soundcloud text-inverse',
													'button'		=>	'warning',
													'profile'		=>	null,
													'permissions'	=>	array(),
													'fields'		=>	array(	'id', 'permalink', 'username', 'uri', 'permalink_url', 'avatar_url', 'country', 'full_name', 'city', 'description', 'discogs-name', 'myspace-name', 'website', 'website-title', 'online', 'track_count',
																				'playlist_count', 'followers_count', 'followings_count', 'public_favorites_count', 'plan', 'private_tracks_count', 'private_playlists_count', 'primary_email_confirmed'
																			),
													'ssl'			=>	false,
													'callback'		=>	'normal'
												)
					);
	}

	/**
	 * Returns the provider callback url
	 *
	 * @deprecated Backend XML Only!
	 *
	 * @param string $name
	 * @return string
	 */
	public function getProviderCallback( $name )
	{
		global $_CB_framework;

		if ( ( ! $name ) || ( ! preg_match( '/^([a-zA-Z0-9]+)_/', $name, $matches ) ) ) {
			return null;
		}

		$providers		=	self::getProviders();
		$providerId		=	$matches[1];

		if ( ! isset( $providers[$providerId] ) ) {
			return null;
		}

		$liveSite		=	$_CB_framework->getCfg( 'live_site' );

		if ( $providers[$providerId]['ssl'] ) {
			$liveSite	=	str_replace( 'http://', 'https://', $liveSite );
		}

		$callback		=	$liveSite . '/components/com_comprofiler/plugin/user/plug_cbconnect/endpoint.php';

		if ( ( $providers[$providerId]['callback'] != 'state' ) && ( $providerId != 'twitter' ) ) {
			$callback	.=	'?provider=' . urlencode( trim( strtolower( $providerId ) ) );
		}

		return $callback;
	}

	/**
	 * Returns an options array of provider permissions
	 *
	 * @deprecated Backend XML Only!
	 *
	 * @param string $name
	 * @return array
	 */
	public function getProviderPermissions( $name )
	{
		$options		=	array();

		if ( ( ! $name ) || ( ! preg_match( '/^([a-zA-Z0-9]+)_/', $name, $matches ) ) ) {
			return $options;
		}

		$providers		=	self::getProviders();
		$providerId		=	$matches[1];

		if ( ! isset( $providers[$providerId] ) ) {
			return $options;
		}

		$permissions	=	$providers[$providerId]['permissions'];

		foreach ( $permissions as $permission ) {
			$options[]	=	\moscomprofilerHTML::makeOption( $permission, $permission );
		}

		return $options;
	}

	/**
	 * Returns an options array of provider fields
	 *
	 * @deprecated Backend XML Only!
	 *
	 * @param string $name
	 * @param string $value
	 * @param string $controlName
	 * @return array
	 */
	public function getProviderFields( $name, $value, $controlName )
	{
		$options		=	array();

		if ( ( ! $controlName ) || ( ! preg_match( '/^params\[([a-zA-Z0-9]+)_/', $controlName, $matches ) ) ) {
			return $options;
		}

		$providers		=	self::getProviders();
		$providerId		=	$matches[1];

		if ( ! isset( $providers[$providerId] ) ) {
			return $options;
		}

		$fields			=	$providers[$providerId]['fields'];

		foreach ( $fields as $field ) {
			$label		=	preg_replace_callback( '/^(\w+)\.(\d+)\.(.+)$/i', function( array $matches ) {
								$matches[2]++;

								return $matches[1] . ' ' . $matches[2] . ' (' . trim( ucwords( $matches[3] ) ) . ')';
							}, $field );

			$options[]	=	\moscomprofilerHTML::makeOption( $field, trim( preg_replace( '/([a-z])([A-Z])/', '$1 $2', ucwords( str_replace( array( '-', '_', '.' ), ' ', $label ) ) ) ) );
		}

		return $options;
	}

	/**
	 * Returns a list of provider substitutions for username formatting
	 *
	 * @deprecated Backend XML Only!
	 *
	 * @param string $name
	 * @return string
	 */
	public function getProviderSubstitutions( $name )
	{
		$substitutions			=	array( '[provider]', '[profile_id]', '[profile_username]', '[profile_name]', '[profile_firstname]', '[profile_middlename]', '[profile_lastname]', '[profile_email]' );

		if ( ( ! $name ) || ( ! preg_match( '/^([a-zA-Z0-9]+)_/', $name, $matches ) ) ) {
			return implode( ', ', $substitutions );
		}

		$providers				=	self::getProviders();
		$providerId				=	$matches[1];

		if ( ! isset( $providers[$providerId] ) ) {
			return implode( ', ', $substitutions );
		}

		$fields					=	$providers[$providerId]['fields'];

		foreach ( $fields as $field ) {
			$substitutions[]	=	'[' . trim( strtolower( $providerId ) ) . '_' . trim( strtolower( str_replace( array( '.', '-' ), '_', $field ) ) ) . ']';
		}

		return implode( ', ', $substitutions );
	}
}
