<?php
/**
* Community Builder (TM)
* @version $Id: $
* @package CommunityBuilder
* @copyright (C) 2004-2021 www.joomlapolis.com / Lightning MultiCom SA - and its licensors, all rights reserved
* @license http://www.gnu.org/licenses/old-licenses/gpl-2.0.html GNU/GPL version 2
*/

namespace CB\Plugin\GroupJive\Trigger;

use CB\Plugin\Activity\Notifications;
use CB\Plugin\Activity\NotificationsInterface;
use CB\Plugin\Activity\StreamInterface;
use CB\Plugin\GroupJive\Table\NotificationTable;
use CB\Plugin\GroupJive\CBGroupJive;
use CB\Plugin\Activity\Table\ActivityTable;
use CB\Plugin\Activity\Activity;
use CB\Plugin\Activity\Comments;
use CB\Database\Table\UserTable;

defined('CBLIB') or die();

class ActivityTrigger extends \cbPluginHandler
{
	/** @var array */
	public static $activitySubTypes	=	array(	'join', 'user_join', 'user_cancel', 'user_approve', 'user_accepted',
												'user_rejected', 'leave', 'user_leave', 'create', 'group_approved',
												'group_pending', 'invite', 'user_invite', 'invite_reject', 'invite_accept',
												'message', 'group_message', 'activity', 'gallery', 'custom'
											);

	/**
	 * @return bool
	 */
	private function isCompatible()
	{
		global $_PLUGINS;

		static $compatible		=	null;

		if ( $compatible === null ) {
			$plugin				=	$_PLUGINS->getLoadedPlugin( 'user', 'cbactivity' );

			if ( $plugin ) {
				$pluginVersion	=	str_replace( '+build.', '+', $_PLUGINS->getPluginVersion( $plugin, true ) );

				$compatible		=	( version_compare( $pluginVersion, '4.0.0', '>=' ) && version_compare( $pluginVersion, '6.0.0', '<' ) );
			}
		}

		return $compatible;
	}

	/**
	 * @param null|string|array      $assets
	 * @param null|int|UserTable     $user
	 * @param array                  $defaults
	 * @param Activity|Notifications $stream
	 */
	public function activityBuild( &$assets, &$user, &$defaults, &$stream )
	{
		if ( ( ! $assets ) || ( ! $this->isCompatible() ) ) {
			return;
		}

		$asset			=	( is_array( $assets ) ? $assets[0] : $assets );

		if ( strpos( $asset, 'groupjive.group' ) !== 0 ) {
			return;
		}

		if ( ( $this->getInput()->getString( 'plugin' ) === 'cbgroupjive' ) && ( $this->getInput()->getString( 'action' ) === 'groups' ) ) {
			$group		=	$this->getInput()->getInt( 'group', 0 );

			if ( ! $group ) {
				$group	=	$this->getInput()->getInt( 'id', 0 );
			}

			$stream->set( 'groupjive.ingroup', $group );
		}
	}

	/**
	 * @param bool              $access
	 * @param string            $type
	 * @param UserTable         $user
	 * @param Activity|Comments $stream
	 */
	public function createAccess( &$access, $type, $user, $stream )
	{
		if ( ! $this->isCompatible() ) {
			return;
		}

		$groupId		=	0;

		if ( preg_match( '/^groupjive\.group\.(\d+)(?:\.([a-zA-Z_]+))?/', $stream->asset(), $matches ) ) {
			$groupId	=	(int) $matches[1];
		} elseif ( preg_match( '/^groupjive\.group\.(\d+)(?:\.([a-zA-Z_]+))?/', $stream->getString( 'parent' ), $matches ) ) {
			$groupId	=	(int) $matches[1];
		}

		if ( ! $groupId ) {
			return;
		}

		$group			=	CBGroupJive::getGroup( (int) $matches[1] );

		if ( ! $group->getInt( 'id', 0 ) ) {
			return;
		}

		if ( $group->getInt( 'published', 0 ) === -1 ) {
			$access		=	false;
		} elseif ( $group->getInt( 'type', 0 ) === 4 ) {
			$access		=	true;
		} else {
			$access		=	( ( $group->getInt( 'user_id', 0 ) === $user->getInt( 'id', 0 ) ) || ( CBGroupJive::getGroupStatus( $user, $group ) >= 1 ) );
		}
	}

	/**
	 * @param bool              $access
	 * @param UserTable         $user
	 * @param Activity|Comments $stream
	 */
	public function moderateAccess( &$access, $user, $stream )
	{
		if ( ! $this->isCompatible() ) {
			return;
		}

		$groupId		=	0;

		if ( preg_match( '/^groupjive\.group\.(\d+)(?:\.([a-zA-Z_]+))?/', $stream->asset(), $matches ) ) {
			$groupId	=	(int) $matches[1];
		} elseif ( preg_match( '/^groupjive\.group\.(\d+)(?:\.([a-zA-Z_]+))?/', $stream->getString( 'parent' ), $matches ) ) {
			$groupId	=	(int) $matches[1];
		}

		if ( ! $groupId ) {
			return;
		}

		$group			=	CBGroupJive::getGroup( (int) $matches[1] );

		if ( ! $group->getInt( 'id', 0 ) ) {
			return;
		}

		if ( $group->getInt( 'published', 0 ) === -1 ) {
			$access		=	false;
		} else {
			$access		=	( ( $group->getInt( 'user_id', 0 ) === $user->getInt( 'id', 0 ) ) || ( CBGroupJive::getGroupStatus( $user, $group ) >= 2 ) );
		}
	}

	/**
	 * @param string                 $output
	 * @param array                  $select
	 * @param array                  $join
	 * @param array                  $where
	 * @param Activity|Notifications $stream
	 */
	public function activityQuery( $output, &$select, &$join, &$where, &$stream )
	{
		global $_CB_database;

		if ( ! $this->isCompatible() ) {
			return;
		}

		$user					=	\CBuser::getMyUserDataInstance();
		$notifications			=	( $stream instanceof NotificationsInterface );
		$filter					=	$stream->getString( 'groupjive.filter' );

		$group					=	"SELECT COUNT(*)"
								.	" FROM " . $_CB_database->NameQuote( '#__groupjive_groups' ) . " AS gj_g";

		if ( $notifications || ( ! CBGroupJive::isModerator() ) ) {
			$group				.=	" LEFT JOIN " . $_CB_database->NameQuote( '#__groupjive_categories' ) . " AS gj_c"
								.	" ON gj_c." . $_CB_database->NameQuote( 'id' ) . " = gj_g." . $_CB_database->NameQuote( 'category' );

			if ( $notifications || $filter || $user->getInt( 'id', 0 ) ) {
				$group			.=	" LEFT JOIN " . $_CB_database->NameQuote( '#__groupjive_users' ) . " AS gj_u"
								.	" ON gj_u." . $_CB_database->NameQuote( 'user_id' ) . " = " . $user->getInt( 'id', 0 )
								.	" AND gj_u." . $_CB_database->NameQuote( 'group' ) . " = gj_g." . $_CB_database->NameQuote( 'id' );

				if ( $notifications ) {
					$group		.=	" LEFT JOIN " . $_CB_database->NameQuote( '#__groupjive_invites' ) . " AS gj_i"
								.	" ON gj_i." . $_CB_database->NameQuote( 'group' ) . " = gj_g." . $_CB_database->NameQuote( 'id' )
								.	" AND gj_i." . $_CB_database->NameQuote( 'accepted' ) . " IS NULL"
								.	" AND ( ( gj_i." . $_CB_database->NameQuote( 'email' ) . " = " . $_CB_database->Quote( $user->getString( 'email' ) )
								.	" AND gj_i." . $_CB_database->NameQuote( 'email' ) . " != '' )"
								.	" OR ( gj_i." . $_CB_database->NameQuote( 'user' ) . " = " . $user->getInt( 'id', 0 )
								.	" AND gj_i." . $_CB_database->NameQuote( 'user' ) . " > 0 ) )";
				}

				$group			.=	" WHERE gj_g." . $_CB_database->NameQuote( 'id' ) . " = SUBSTRING_INDEX( SUBSTRING_INDEX( a." . $_CB_database->NameQuote( 'asset' ) . ", '.', 3 ) , '.' , -1 )"
								.	" AND ( gj_g." . $_CB_database->NameQuote( 'user_id' ) . " = " . $user->getInt( 'id', 0 )
								.	" OR ( gj_g." . $_CB_database->NameQuote( 'published' ) . " = 1";

				if ( $notifications ) {
					$group		.=	" AND ( gj_u." . $_CB_database->NameQuote( 'status' ) . " > 0"
								.	" OR ( a." . $_CB_database->NameQuote( 'asset' ) . " LIKE " . $_CB_database->Quote( 'groupjive.group.%.user_invite' )
								.	" AND gj_i." . $_CB_database->NameQuote( 'id' ) . " IS NOT NULL )"
								.	" OR a." . $_CB_database->NameQuote( 'asset' ) . " LIKE " . $_CB_database->Quote( 'groupjive.group.%.user_rejected' ) . " )";
				} elseif ( $filter === 'my' ) {
					$group		.=	" AND ( gj_g." . $_CB_database->NameQuote( 'user_id' ) . " = " . $user->getInt( 'id', 0 )
								.	" OR gj_u." . $_CB_database->NameQuote( 'status' ) . " BETWEEN 0 AND 3 )";
				} elseif ( $filter === 'joined' ) {
					$group		.=	" AND gj_u." . $_CB_database->NameQuote( 'status' ) . " BETWEEN 0 AND 3";
				} elseif ( $filter === 'owned' ) {
					$group		.=	" AND gj_g." . $_CB_database->NameQuote( 'user_id' ) . " = " . $user->getInt( 'id', 0 );
				} else {
					$group		.=	" AND ( gj_g." . $_CB_database->NameQuote( 'type' ) . " != 3"
								.	" OR gj_u." . $_CB_database->NameQuote( 'status' ) . " BETWEEN 0 AND 3 )";
				}

				$group			.=	" ) )";
			} else {
				$group			.=	" WHERE gj_g." . $_CB_database->NameQuote( 'id' ) . " = SUBSTRING_INDEX( SUBSTRING_INDEX( a." . $_CB_database->NameQuote( 'asset' ) . ", '.', 3 ) , '.' , -1 )"
								.	" AND gj_g." . $_CB_database->NameQuote( 'published' ) . " = 1"
								.	" AND gj_g." . $_CB_database->NameQuote( 'type' ) . " != 3";
			}

			$group				.=	" AND ( ( gj_c." . $_CB_database->NameQuote( 'published' ) . " = 1"
								.	" AND gj_c." . $_CB_database->NameQuote( 'access' ) . " IN " . $_CB_database->safeArrayOfIntegers( CBGroupJive::getAccess( $user->getInt( 'id', 0 ) ) ) . " )"
								.	( $this->params->getBool( 'groups_uncategorized', true ) ? " OR gj_g." . $_CB_database->NameQuote( 'category' ) . " = 0" : null ) . " )";
		} else {
			if ( ( $filter === 'my' ) || ( $filter === 'joined' ) ) {
				$group			.=	" LEFT JOIN " . $_CB_database->NameQuote( '#__groupjive_users' ) . " AS gj_u"
								.	" ON gj_u." . $_CB_database->NameQuote( 'user_id' ) . " = " . $user->getInt( 'id', 0 )
								.	" AND gj_u." . $_CB_database->NameQuote( 'group' ) . " = gj_g." . $_CB_database->NameQuote( 'id' );
			}

			$group				.=	" WHERE gj_g." . $_CB_database->NameQuote( 'id' ) . " = SUBSTRING_INDEX( SUBSTRING_INDEX( a." . $_CB_database->NameQuote( 'asset' ) . ", '.', 3 ) , '.' , -1 )";

			if ( $filter === 'my' ) {
				$group			.=	" AND ( gj_g." . $_CB_database->NameQuote( 'user_id' ) . " = " . $user->getInt( 'id', 0 )
								.	" OR gj_u." . $_CB_database->NameQuote( 'status' ) . " BETWEEN 0 AND 3 )";
			} elseif ( $filter === 'joined' ) {
				$group			.=	" AND gj_u." . $_CB_database->NameQuote( 'status' ) . " BETWEEN 0 AND 3";
			} elseif ( $filter === 'owned' ) {
				$group			.=	" AND gj_g." . $_CB_database->NameQuote( 'user_id' ) . " = " . $user->getInt( 'id', 0 );
			}
		}

		$where[]				=	"( ( a." . $_CB_database->NameQuote( 'asset' ) . " LIKE " . $_CB_database->Quote( 'groupjive.group.%' )
								.	" AND ( " . $group . " ) > 0 )"
								.	" OR ( a." . $_CB_database->NameQuote( 'asset' ) . " NOT LIKE " . $_CB_database->Quote( 'groupjive.group.%' ) . " ) )";
	}

	/**
	 * @param ActivityTable[]|NotificationTable[] $rows
	 * @param Activity|Notifications              $stream
	 */
	public function activityPrefetch( &$rows, $stream )
	{
		global $_CB_database;

		if ( ! $this->isCompatible() ) {
			return;
		}

		$groupIds				=	array();

		foreach ( $rows as $k => $row ) {
			if ( ! preg_match( '/^groupjive\.group\.(\d+)(?:\.([a-zA-Z_]+))?/', $row->getString( 'asset' ), $matches ) ) {
				continue;
			}

			$groupId			=	(int) $matches[1];

			if ( $groupId ) {
				$groupIds[$k]	=	$groupId;
			}
		}

		if ( ! $groupIds ) {
			return;
		}

		$user					=	\CBuser::getMyUserDataInstance();

		$users					=	'SELECT COUNT(*)'
								.	"\n FROM " . $_CB_database->NameQuote( '#__groupjive_users' ) . " AS uc"
								.	"\n LEFT JOIN " . $_CB_database->NameQuote( '#__comprofiler' ) . " AS uccb"
								.	' ON uccb.' . $_CB_database->NameQuote( 'id' ) . ' = uc.' . $_CB_database->NameQuote( 'user_id' )
								.	"\n LEFT JOIN " . $_CB_database->NameQuote( '#__users' ) . " AS ucj"
								.	' ON ucj.' . $_CB_database->NameQuote( 'id' ) . ' = uc.' . $_CB_database->NameQuote( 'user_id' )
								.	"\n WHERE uc." . $_CB_database->NameQuote( 'group' ) . " = g." . $_CB_database->NameQuote( 'id' )
								.	"\n AND uccb." . $_CB_database->NameQuote( 'approved' ) . " = 1"
								.	"\n AND uccb." . $_CB_database->NameQuote( 'confirmed' ) . " = 1"
								.	"\n AND ucj." . $_CB_database->NameQuote( 'block' ) . " = 0";

		if ( ! $this->params->getBool( 'groups_users_owner', true ) ) {
			$users				.=	"\n AND uc." . $_CB_database->NameQuote( 'status' ) . " != 4";
		}

		$query					=	'SELECT g.*'
								.	', c.' . $_CB_database->NameQuote( 'name' ) . ' AS _category_name';

		if ( $user->getInt( 'id', 0 ) ) {
			$query				.=	', u.' . $_CB_database->NameQuote( 'status' ) . ' AS _user_status'
								.	', i.' . $_CB_database->NameQuote( 'id' ) . ' AS _invite_id';
		} else {
			$query				.=	', NULL AS _user_status'
								.	', NULL AS _invite_id';
		}

		$query					.=	', ( ' . $users . ' ) AS _users'
								.	"\n FROM " . $_CB_database->NameQuote( '#__groupjive_groups' ) . " AS g"
								.	"\n LEFT JOIN " . $_CB_database->NameQuote( '#__groupjive_categories' ) . " AS c"
								.	' ON c.' . $_CB_database->NameQuote( 'id' ) . ' = g.' . $_CB_database->NameQuote( 'category' );

		if ( $user->getInt( 'id', 0 ) ) {
			$query				.=	"\n LEFT JOIN " . $_CB_database->NameQuote( '#__groupjive_users' ) . " AS u"
								.	' ON u.' . $_CB_database->NameQuote( 'user_id' ) . ' = ' . $user->getInt( 'id', 0 )
								.	' AND u.' . $_CB_database->NameQuote( 'group' ) . ' = g.' . $_CB_database->NameQuote( 'id' )
								.	"\n LEFT JOIN " . $_CB_database->NameQuote( '#__groupjive_invites' ) . " AS i"
								.	' ON i.' . $_CB_database->NameQuote( 'group' ) . ' = g.' . $_CB_database->NameQuote( 'id' )
								.	' AND i.' . $_CB_database->NameQuote( 'accepted' ) . ' IS NULL'
								.	' AND ( ( i.' . $_CB_database->NameQuote( 'email' ) . ' = ' . $_CB_database->Quote( $user->getString( 'email' ) )
								.	' AND i.' . $_CB_database->NameQuote( 'email' ) . ' != "" )'
								.	' OR ( i.' . $_CB_database->NameQuote( 'user' ) . ' = ' . $user->getInt( 'id', 0 )
								.	' AND i.' . $_CB_database->NameQuote( 'user' ) . ' > 0 ) )';
		}

		$query					.=	"\n WHERE g." . $_CB_database->NameQuote( 'id' ) . " IN " . $_CB_database->safeArrayOfIntegers( array_unique( $groupIds ) );
		$_CB_database->setQuery( $query );
		$groups					=	$_CB_database->loadObjectList( null, '\CB\Plugin\GroupJive\Table\GroupTable', array( $_CB_database ) );

		if ( ! $groups ) {
			foreach ( $groupIds as $k => $groupId ) {
				unset( $rows[$k] );
			}

			return;
		}

		CBGroupJive::prefetchGroups( $groups );
		CBGroupJive::prefetchUsers( $groups );

		foreach ( $groupIds as $k => $groupId ) {
			$group				=	CBGroupJive::getGroup( (int) $groupId );

			if ( ( ! $group->getInt( 'id', 0 ) ) || ( ! CBGroupJive::canAccessGroup( $group, $user ) ) ) {
				unset( $rows[$k] );
			}
		}
	}

	/**
	 * @param ActivityTable|NotificationTable $row
	 * @param null|string                     $title
	 * @param null|string                     $date
	 * @param null|string                     $message
	 * @param null|string                     $insert
	 * @param null|string                     $footer
	 * @param array                           $menu
	 * @param Activity|Notifications          $stream
	 * @param string                          $output
	 */
	public function activityDisplay( &$row, &$title, &$date, &$message, &$insert, &$footer, &$menu, $stream, $output )
	{
		if ( ( ! $this->isCompatible() ) || ( ! preg_match( '/^groupjive\.group\.(\d+)(?:\.([a-zA-Z_]+))?/', $row->getString( 'asset' ), $matches ) ) ) {
			return;
		}

		if ( isset( $matches[2] ) && ( ! in_array( $matches[2], self::$activitySubTypes, true ) ) ) {
			return;
		}

		$group		=	CBGroupJive::getGroup( (int) $matches[1] );

		if ( ! $group->getInt( 'id', 0 ) ) {
			return;
		}

		CBGroupJive::getTemplate( 'activity' );

		if ( ( ! $stream instanceof NotificationsInterface  && ( isset( $matches[2] ) ? $matches[2] : '' ) === 'leave' ) ) {
			$row->params()->set( 'overrides.edit', false );
		}

		\HTML_groupjiveActivity::showActivity( $row, $title, $date, $message, $insert, $footer, $menu, $stream, $matches, $group, $this, $output );
	}

	/**
	 * @param string          $asset
	 * @param UserTable       $primaryUser
	 * @param UserTable       $assetsUser
	 * @param UserTable       $filtered
	 * @param StreamInterface $stream
	 */
	public function assetPrepare( &$asset, &$primaryUser, &$assetsUser, $filtered, &$stream )
	{
		if ( preg_match( '/^groupjive\.(my|joined|owned)$/', $asset, $matches ) ) {
			$stream->set( 'groupjive.filter', $matches[1] );

			if ( ! $filtered ) {
				$stream->set( 'user_id', \CBuser::getMyUserDataInstance()->getInt( 'id' ) );
			}

			$asset			=	'groupjive.%';
		}
	}

	/**
	 * @param string $asset
	 * @param mixed  $source
	 */
	public function assetSource( $asset, &$source )
	{
		if ( ! preg_match( '/^groupjive\.group\.(\d+)(?:\.([a-zA-Z_]+))?/', $asset, $matches ) ) {
			return;
		}

		if ( isset( $matches[2] ) && ( ! in_array( $matches[2], self::$activitySubTypes, true ) ) ) {
			return;
		}

		if ( preg_match( '/^(?:.+\.)?gallery\./', $asset ) ) {
			// CB Gallery is prefixed with GJ path so be sure to skip that part here:
			return;
		}

		$group		=	CBGroupJive::getGroup( (int) $matches[1] );

		if ( ! $group->getInt( 'id', 0 ) ) {
			return;
		}

		$source		=	$group;
	}
}