<?php
/**
* Community Builder (TM)
* @version $Id: $
* @package CommunityBuilder
* @copyright (C) 2004-2019 www.joomlapolis.com / Lightning MultiCom SA - and its licensors, all rights reserved
* @license http://www.gnu.org/licenses/old-licenses/gpl-2.0.html GNU/GPL version 2
*/

namespace CB\Plugin\Activity;

use CBLib\Application\Application;
use CB\Database\Table\UserTable;
use CB\Plugin\Activity\Table\NotificationTable;

defined('CBLIB') or die();

/**
 * @method string getAsset()
 * @method Notifications setAsset( $asset )
 * @method array getAssets()
 * @method Notifications setAssets( $assets )
 * @method string getParentAsset()
 * @method Notifications setParentAsset( $asset )
 * @method UserTable getUser()
 * @method Notifications setUser( $user )
 * @method int|array getId()
 * @method Notifications setId( $id )
 * @method array getModerators()
 * @method Notifications setModerators( $moderators )
 * @method int|array getUserId()
 * @method Notifications setUserId( $userId )
 * @method string|array getDate()
 * @method Notifications setDate( $datetime )
 * @method string getSearch()
 * @method Notifications setSearch( $search )
 * @method string getTitle()
 * @method Notifications setTitle( $title )
 * @method string getMessage()
 * @method Notifications setMessage( $message )
 * @method string getRead()
 * @method Notifications setRead( $read )
 * @method string getHidden()
 * @method Notifications setHidden( $hidden )
 * @method bool getInline()
 * @method Notifications setInline( $inline )
 * @method int getPublished()
 * @method Notifications setPublished( $published )
 */
class Notifications extends Stream implements NotificationsInterface
{
	/** @var array $defaults */
	protected $defaults					=	array(	'template'					=>	null,
													'layout'					=>	'button',
													'direction'					=>	'down',
													'auto_update'				=>	false,
													'auto_load'					=>	false,
													'global'					=>	true,
													'pinned'					=>	false,
													'paging'					=>	true,
													'paging_first_limit'		=>	15,
													'paging_limit'				=>	15,
													'parser_substitutions'		=>	true,
													'parser_emotes'				=>	true,
													'parser_reactions'			=>	true,
													'parser_hashtags'			=>	true,
													'parser_profiles'			=>	true,
													'parser_links'				=>	true,
													'parser_prepare'			=>	false,
													'parser_bbcode'				=>	false
												);

	/** @var NotificationTable[] $loadedRows */
	protected static $loadedRows		=	array();

	/**
	 * Constructor for notifications object
	 *
	 * @param null|string|array  $assets
	 * @param null|int|UserTable $user 
	 * @param null|string        $namespace
	 */
	public function __construct( $assets = null, $user = null, $namespace = null )
	{
		if ( ! $assets ) {
			$assets		=	array( 'all' );
		}

		$this->set( 'published', 1 );

		parent::__construct( $assets, $user, $namespace );
	}

	/**
	 * @param string $name
	 * @param array  $arguments
	 * @return self|string|int|array|null
	 */
	public function __call( $name, $arguments )
	{
		$method									=	substr( $name, 0, 3 );

		if ( in_array( $method, array( 'get', 'set' ), true ) ) {
			$variables							=	array( 'asset', 'assets', 'user', 'id', 'parent', 'moderators', 'user_id', 'date', 'search', 'title', 'message', 'read', 'hidden', 'inline', 'published' );
			$variable							=	strtolower( substr( $name, 3 ) );

			switch ( $variable ) {
				case 'userid':
					$variable					=	'user_id';
					break;
				case 'parentasset':
					$variable					=	'parent';
					break;
			}

			if ( in_array( $variable, $variables, true ) ) {
				switch ( $method ) {
					case 'get':
						switch ( $variable ) {
							case 'asset':
								return $this->asset();
							case 'assets':
								return $this->assets();
							case 'user':
								return $this->user();
							case 'id':
							case 'user_id':
								if ( is_array( $this->getRaw( $variable ) ) ) {
									return $this->getRaw( $variable, array() );
								}

								return $this->getInt( $variable, 0 );
							case 'moderators':
								return $this->getRaw( $variable, array() );
							case 'inline':
							case 'hidden':
								return $this->getBool( $variable, false );
							case 'published':
								return $this->getInt( $variable, 0 );
							default:
								if ( is_array( $this->getRaw( $variable ) ) ) {
									return $this->getRaw( $variable, array() );
								}

								return $this->getString( $variable );
						}
					case 'set':
						switch ( $variable ) {
							case 'asset':
							case 'assets':
								$this->assets( ( $arguments ? $arguments[0] : null ) );
								break;
							case 'user':
								$this->user( ( $arguments ? $arguments[0] : null ) );
								break;
							default:
								$this->set( $variable, ( $arguments ? $arguments[0] : null ) );
								break;
						}

						return $this;
				}
			}
		}

		trigger_error( 'Call to undefined method ' . __CLASS__ . '::' . $name . '()', E_USER_ERROR );
	}

	/**
	 * Gets the primary notifications asset
	 *
	 * @return string
	 */
	public function asset()
	{
		return 'notification.' . $this->user()->getInt( 'id', 0 );
	}

	/**
	 * Retrieves notifications rows or row count
	 *
	 * @param string $output
	 * @return NotificationTable[]|int
	 */
	public function rows( $output = null )
	{
		global $_CB_database, $_PLUGINS;

		static $cache						=	array();

		$myId								=	Application::MyUser()->getUserId();
		$id									=	$this->getRaw( 'id' );
		$hasId								=	( ( ( $id !== null ) && ( $id !== '' ) ) || ( is_array( $id ) && $id ) );

		if ( $this->getInt( 'paging_limitstart', 0 ) === 0 ) {
			$pageLimit						=	$this->getInt( 'paging_first_limit', 15 );
		} else {
			$pageLimit						=	$this->getInt( 'paging_limit', 15 );
		}

		$paging								=	( ( ! $hasId ) && $pageLimit && ( $output !== 'all' ) );
		$select								=	array();
		$join								=	array();
		$where								=	array();

		if ( $output === 'count' ) {
			$select[]						=	'COUNT(*)';
		} else {
			$select[]						=	'a.*';
		}

		if ( $hasId ) {
			if ( is_array( $this->getRaw( 'id' ) ) ) {
				$where[]					=	"a." . $_CB_database->NameQuote( 'id' ) . " IN " . $_CB_database->safeArrayOfIntegers( $id );
			} else {
				$where[]					=	"a." . $_CB_database->NameQuote( 'id' ) . " = " . (int) $id;
			}
		}

		$userId								=	$this->getRaw( 'user_id' );

		if ( ( ( $userId !== null ) && ( $userId !== '' ) ) || ( is_array( $userId ) && $userId ) ) {
			if ( is_array( $userId ) ) {
				$where[]					=	"a." . $_CB_database->NameQuote( 'user_id' ) . " IN " . $_CB_database->safeArrayOfIntegers( $userId );
			} else {
				$where[]					=	"a." . $_CB_database->NameQuote( 'user_id' ) . " = " . (int) $userId;
			}
		}

		if ( $this->getBool( 'global', true ) ) {
			$where[]						=	"( a." . $_CB_database->NameQuote( 'user' ) . " = " . $this->user()->getInt( 'id', 0 )
											.	" OR a." . $_CB_database->NameQuote( 'global' ) . " = 1 )";
		} else {
			$where[]						=	"a." . $_CB_database->NameQuote( 'user' ) . " = " . $this->user()->getInt( 'id', 0 );
		}

		if ( $this->assets() && ( ! in_array( 'all', $this->assets(), true ) ) ) {
			$queryAssets					=	$this->queryAssets();

			if ( $queryAssets['assets'] || $queryAssets['wildcards'] || $queryAssets['exists'] ) {
				$assetsWhere				=	array();

				if ( $queryAssets['assets'] ) {
					$assetsWhere[]			=	"a." . $_CB_database->NameQuote( 'asset' ) . ( count( $queryAssets['assets'] ) > 1 ? " IN " . $_CB_database->safeArrayOfStrings( $queryAssets['assets'] ) : " = " . $_CB_database->Quote( $queryAssets['assets'][0] ) );
				}

				if ( $queryAssets['wildcards'] ) {
					foreach ( $queryAssets['wildcards'] as $wildcard ) {
						$assetsWhere[]		=	"a." . $_CB_database->NameQuote( 'asset' ) . " LIKE " . $_CB_database->Quote( $wildcard );
					}
				}

				if ( $queryAssets['exists'] ) {
					foreach ( $queryAssets['exists'] as $exist ) {
						$assetsWhere[]		=	"EXISTS ( " . $exist . " )";
					}
				}

				$where[]					=	( count( $assetsWhere ) > 1 ? "( " . implode( " OR ", $assetsWhere ) . " )" : $assetsWhere[0] );
			} elseif ( $output === 'count' ) {
				return 0;
			} else {
				return array();
			}
		}

		if ( ! $hasId ) {
			$date							=	$this->getRaw( 'date' );

			if ( is_array( $date ) ) {
				if ( count( $date ) > 1 ) {
					if ( ! in_array( $date[1], array( '=', '<>', '<', '>', '<=', '>=', 'REGEXP', 'NOT REGEXP', 'LIKE', 'NOT LIKE' ), true ) ) {
						$date[1]			=	'>';
					}

					$where[]				=	"a." . $_CB_database->NameQuote( 'date' ) . " " . $date[1] . " " . $_CB_database->Quote( ( is_int( $date[0] ) ? Application::Database()->getUtcDateTime( $date[0] ) : $date[0] ) );
				} else {
					$where[]				=	"a." . $_CB_database->NameQuote( 'date' ) . " > " . $_CB_database->Quote( ( is_int( $date[0] ) ? Application::Database()->getUtcDateTime( $date[0] ) : $date[0] ) );
				}
			} elseif ( $date ) {
				$where[]					=	"a." . $_CB_database->NameQuote( 'date' ) . " = " . $_CB_database->Quote( ( is_int( $date ) ? Application::Database()->getUtcDateTime( $date ) : $date ) );
			}

			if ( $this->getString( 'title', '' ) !== '' ) {
				if ( strpos( $this->getString( 'title' ), '%' ) !== false ) {
					$where[]				=	"a." . $_CB_database->NameQuote( 'title' ) . " LIKE " . $_CB_database->Quote( $this->getString( 'title' ) );
				} else {
					$where[]				=	"a." . $_CB_database->NameQuote( 'title' ) . " = " . $_CB_database->Quote( $this->getString( 'title' ) );
				}
			}

			if ( $this->getString( 'message', '' ) !== '' ) {
				if ( strpos( $this->getString( 'message' ), '%' ) !== false ) {
					$where[]				=	"a." . $_CB_database->NameQuote( 'message' ) . " LIKE " . $_CB_database->Quote( $this->getString( 'message' ) );
				} else {
					$where[]				=	"a." . $_CB_database->NameQuote( 'message' ) . " = " . $_CB_database->Quote( $this->getString( 'message' ) );
				}
			}

			if ( $this->getString( 'search', '' ) !== '' ) {
				$where[]					=	"( a." . $_CB_database->NameQuote( 'title' ) . " LIKE " . $_CB_database->Quote( '%' . $_CB_database->getEscaped( $this->getString( 'search' ), true ) . '%', false )
											.	" OR a." . $_CB_database->NameQuote( 'message' ) . " LIKE " . $_CB_database->Quote( '%' . $_CB_database->getEscaped( $this->getString( 'search' ), true ) . '%', false )
											.	" OR a." . $_CB_database->NameQuote( 'date' ) . " LIKE " . $_CB_database->Quote( '%' . $_CB_database->getEscaped( $this->getString( 'search' ), true ) . '%', false ) . " )";
			}

			if ( $myId ) {
				if ( $this->getBool( 'hidden', false ) ) {
					$select[]				=	"h1." . $_CB_database->NameQuote( 'id' ) . " AS _hidden_id";

					$join[]					=	"LEFT JOIN " . $_CB_database->NameQuote( '#__comprofiler_plugin_activity_hidden' ) . " AS h1"
											.	" ON h1." . $_CB_database->NameQuote( 'user_id' ) . " = " . $myId
											.	" AND h1." . $_CB_database->NameQuote( 'type' ) . " = " . $_CB_database->Quote( 'notification' )
											.	" AND h1." . $_CB_database->NameQuote( 'object' ) . " = a." . $_CB_database->NameQuote( 'id' );

					$where[]				=	"h1." . $_CB_database->NameQuote( 'id' ) . " IS NOT NULL";
				} else {
					$where[]				=	"NOT EXISTS ("
											.	" SELECT 1 FROM " . $_CB_database->NameQuote( '#__comprofiler_plugin_activity_hidden' ) . " AS h1"
											.	" WHERE h1." . $_CB_database->NameQuote( 'user_id' ) . " = " . $myId
											.	" AND h1." . $_CB_database->NameQuote( 'type' ) . " = " . $_CB_database->Quote( 'notification' )
											.	" AND h1." . $_CB_database->NameQuote( 'object' ) . " = a." . $_CB_database->NameQuote( 'id' )
											.	" )";
				}

				$read						=	$this->getString( 'read' );

				if ( in_array( $read, array( 'status', 'read', 'unread', 'readonly', 'unreadonly' ), true ) ) {
					$lastRead				=	CBActivity::getRead( $this->asset(), $myId );

					if ( in_array( $read, array( 'readonly', 'unreadonly' ), true ) ) {
						if ( $output !== 'count' ) {
							$select[]		=	( $read === 'unreadonly' ? "0" : "1" ) . " AS _read";
						}

						$where[]			=	"a." . $_CB_database->NameQuote( 'date' ) . ( $read === 'unreadonly' ? " > " : " <= " ) . $_CB_database->Quote( $lastRead );
					} elseif ( $output !== 'count' ) {
						$select[]			=	"IF( a." . $_CB_database->NameQuote( 'date' ) . " <= " . $_CB_database->Quote( $lastRead ) . ", 1, 0 ) AS _read";
					}
				}
			}
		}

		if ( ( $this->getRaw( 'published' ) !== null ) && ( $this->getRaw( 'published' ) !== '' ) ) {
			$where[]						=	"a." . $_CB_database->NameQuote( 'published' ) . " = " . $this->getInt( 'published' );
		}

		$_PLUGINS->trigger( 'activity_onQueryNotificationsStream', array( $output, &$select, &$join, &$where, &$this ) );

		$query								=	"SELECT " . implode( ", ", $select )
											.	"\n FROM " . $_CB_database->NameQuote( '#__comprofiler_plugin_activity_notifications' ) . " AS a"
											.	"\n LEFT JOIN " . $_CB_database->NameQuote( '#__comprofiler' ) . " AS cb"
											.	" ON cb." . $_CB_database->NameQuote( 'id' ) . " = a." . $_CB_database->NameQuote( 'user_id' )
											.	"\n LEFT JOIN " . $_CB_database->NameQuote( '#__users' ) . " AS j"
											.	" ON j." . $_CB_database->NameQuote( 'id' ) . " = a." . $_CB_database->NameQuote( 'user_id' )
											.	( $join ? "\n " . implode( "\n ", $join ) : null )
											.	"\n WHERE cb." . $_CB_database->NameQuote( 'approved' ) . " = 1"
											.	"\n AND cb." . $_CB_database->NameQuote( 'confirmed' ) . " = 1"
											.	"\n AND j." . $_CB_database->NameQuote( 'block' ) . " = 0"
											.	( $where ? "\n AND " . implode( "\n AND ", $where ) : null );

		if ( ( ! $hasId ) && ( $output !== 'count' ) ) {
			$query							.=	"\n ORDER BY ";

			if ( $this->getBool( 'pinned', false ) ) {
				$query						.=	"a." . $_CB_database->NameQuote( 'pinned' ) . " DESC, ";
			}

			$query							.=	"a." . $_CB_database->NameQuote( 'date' ) . " DESC";
		}

		$cacheId							=	md5( $query . ( $output === 'count' ? $output : ( $output ? $output : null ) . ( $paging ? $this->getInt( 'paging_limitstart', 0 ) . $pageLimit : null ) ) );

		if ( ( ! isset( $cache[$cacheId] ) ) || ( ( ( $output === 'count' ) && $this->clearRowCount ) || $this->clearRowSelect ) ) {
			if ( $output === 'count' ) {
				$this->clearRowCount		=	false;

				$_CB_database->setQuery( $query );

				$cache[$cacheId]			=	(int) $_CB_database->loadResult();

				$this->set( 'query_count', $cache[$cacheId] );
			} else {
				$this->clearRowSelect		=	false;

				if ( $paging ) {
					$_CB_database->setQuery( $query, $this->getInt( 'paging_limitstart', 0 ), ( $pageLimit + 1 ) );
				} else {
					$_CB_database->setQuery( $query );
				}

				$this->set( 'paging_limitstart', ( $this->getInt( 'paging_limitstart', 0 ) + $pageLimit ) );

				$rows						=	$_CB_database->loadObjectList( 'id', '\CB\Plugin\Activity\Table\NotificationTable', array( $_CB_database ) );
				$rowsCount					=	count( $rows );

				if ( $paging ) {
					$this->set( 'paging_total', $rowsCount );

					$rows					=	array_slice( $rows, 0, $pageLimit, true );
					$rowsCount				=	count( $rows );
				}

				$userIds					=	array( $this->user()->getInt( 'id', 0 ) );

				/** @var NotificationTable[] $rows */
				foreach ( $rows as $row ) {
					if ( preg_match( '/^profile\.(\d+)/', $row->getString( 'asset' ), $matches ) ) {
						$userIds[]			=	(int) $matches[1];
					}

					$userIds[]				=	$row->getInt( 'user_id', 0 );
				}

				if ( $userIds ) {
					\CBuser::advanceNoticeOfUsersNeeded( $userIds );
				}

				$_PLUGINS->trigger( 'activity_onLoadNotificationsStream', array( &$rows, $this ) );

				if ( $rows ) {
					static $group			=	array();
					static $parents			=	array();

					$streamId				=	$this->id();

					// Check if any rows have been told to group up by asset:
					foreach ( $rows as $k => $row ) {
						$asset								=	$row->params()->getString( 'overrides.group' );

						if ( $asset === 'asset' ) {
							$asset							=	$row->getString( 'asset' );
						} elseif ( ! $asset ) {
							$asset							=	$row->params()->getString( 'asset' );
						}

						if ( ! $asset ) {
							if ( $parents ) {
								// Check if this entries asset is a parent of a group:
								$asset						=	$row->getString( 'asset' );

								if ( isset( $parents[$asset] ) ) {
									unset( $rows[$k] );
								}

								// Check if this entries activity is a parent of a group:
								$asset						=	'activity.' . $row->getInt( 'id', 0 );

								if ( isset( $parents[$asset] ) ) {
									unset( $rows[$k] );
								}
							}

							continue;
						}

						$parent								=	$row->params()->getString( 'overrides.group_parent' );

						if ( $parent ) {
							$parents[$parent]				=	$parent;
						}

						$days								=	$row->params()->getInt( 'overrides.group_days' );

						if ( ! $days ) {
							$days							=	10;
						}

						if ( ! isset( $group[$streamId][$asset] ) ) {
							$group[$streamId][$asset]		=	$row;
						} else {
							$dateDiff						=	Application::Date( $group[$streamId][$asset]->getString( 'date' ), 'UTC' )->diff( $row->getString( 'date' ) );

							if ( $dateDiff->days <= $days ) {
								$names						=	$group[$streamId][$asset]->params()->getRaw( 'overrides.names', array() );

								if ( $group[$streamId][$asset]->getInt( 'user_id', 0 ) !== $row->getInt( 'user_id', 0 ) ) {
									$names[]				=	$row->getInt( 'user_id', 0 );
								}

								$group[$streamId][$asset]->params()->set( 'overrides.names', array_unique( $names ) );

								unset( $rows[$k] );
							} else {
								$group[$streamId][$asset]	=	$row;
							}
						}
					}

					if ( $rows ) {
						self::$loadedRows	+=	$rows;
					}
				}

				$newCount					=	count( $rows );

				if ( $paging && $rowsCount && ( $newCount < round( $rowsCount / 1.25 ) ) ) {
					$pagingTotal			=	$this->getInt( 'paging_total', 0 );
					$nextLimit				=	( $pageLimit - $newCount );

					if ( $nextLimit <= 0 ) {
						$nextLimit			=	1;
					}

					$this->set( 'paging_limit', $nextLimit );

					$cache[$cacheId]		=	( $rows + $this->rows( $output ) );

					$this->set( 'paging_total', $pagingTotal );
					$this->set( 'paging_limit', $pageLimit );
				} else {
					$cache[$cacheId]		=	$rows;
				}
			}
		} elseif ( $output !== 'count' ) {
			$this->set( 'paging_limitstart', ( $this->getInt( 'paging_limitstart', 0 ) + count( $cache[$cacheId] ) ) );
		}

		return $cache[$cacheId];
	}

	/**
	 * Retrieves notifications row
	 *
	 * @param int $id
	 * @return NotificationTable
	 */
	public function row( $id )
	{
		if ( ! $id ) {
			return new NotificationTable();
		}

		if ( isset( self::$loadedRows[$id] ) ) {
			return self::$loadedRows[$id];
		}

		static $cache		=	array();

		if ( ! isset( $cache[$id] ) ) {
			$rows			=	$this->reset()->setId( $id )->rows();

			if ( isset( $rows[$id] ) ) {
				$row		=	$rows[$id];
			} else {
				$row		=	new NotificationTable();
			}

			$cache[$id]		=	$row;
		}

		return $cache[$id];
	}

	/**
	 * Outputs notifications HTML
	 *
	 * @param null|string $view
	 * @param int         $id
	 * @param array       $params
	 * @return string
	 */
	public function notifications( $view = null, $id = 0, $params = array() )
	{
		return $this->display( $view, $id, $params );
	}
}