<?php
/**
* Community Builder (TM)
* @version $Id: $
* @package CommunityBuilder
* @copyright (C) 2004-2019 www.joomlapolis.com / Lightning MultiCom SA - and its licensors, all rights reserved
* @license http://www.gnu.org/licenses/old-licenses/gpl-2.0.html GNU/GPL version 2
*/

namespace CB\Plugin\Activity;

use CB\Plugin\Activity\Table\LikeTypeTable;
use CB\Database\Table\UserTable;
use CB\Plugin\Activity\Table\LikeTable;

defined('CBLIB') or die();

/**
 * @method string getAsset()
 * @method Likes setAsset( $asset )
 * @method array getAssets()
 * @method Likes setAssets( $assets )
 * @method string getParentAsset()
 * @method Likes setParentAsset( $asset )
 * @method UserTable getUser()
 * @method Likes setUser( $user )
 * @method int|array getId()
 * @method Likes setId( $id )
 * @method string getParent()
 * @method Likes setParent( $asset )
 * @method array getModerators()
 * @method Likes setModerators( $moderators )
 * @method int|array getType()
 * @method Likes setType( $type )
 * @method int|array getUserId()
 * @method Likes setUserId( $userId )
 * @method bool getInline()
 * @method Likes setInline( $inline )
 */
class Likes extends Stream implements LikesInterface
{
	/** @var array $defaults */
	protected $defaults					=	array(	'template'				=>	null,
													'layout'				=>	'button',
													'count'					=>	true,
													'include'				=>	0,
													'exclude'				=>	0,
													'paging'				=>	true,
													'paging_first_limit'	=>	15,
													'paging_limit'			=>	15
												);

	/** @var LikeTable[] $loadedRows */
	protected static $loadedRows		=	array();

	/**
	 * @param string $name
	 * @param array  $arguments
	 * @return self|string|int|array|null
	 */
	public function __call( $name, $arguments )
	{
		$method									=	substr( $name, 0, 3 );

		if ( in_array( $method, array( 'get', 'set' ), true ) ) {
			$variables							=	array( 'asset', 'assets', 'user', 'id', 'parent', 'moderators', 'user_id', 'type', 'inline' );
			$variable							=	strtolower( substr( $name, 3 ) );

			switch ( $variable ) {
				case 'userid':
					$variable					=	'user_id';
					break;
				case 'parentasset':
					$variable					=	'parent';
					break;
			}

			if ( in_array( $variable, $variables, true ) ) {
				switch ( $method ) {
					case 'get':
						switch ( $variable ) {
							case 'asset':
								return $this->asset();
							case 'assets':
								return $this->assets();
							case 'user':
								return $this->user();
							case 'id':
							case 'user_id':
							case 'type':
								if ( is_array( $this->getRaw( $variable ) ) ) {
									return $this->getRaw( $variable, array() );
								}

								return $this->getInt( $variable, 0 );
							case 'moderators':
								return $this->getRaw( $variable, array() );
							case 'inline':
								return $this->getBool( $variable, false );
							default:
								if ( is_array( $this->getRaw( $variable ) ) ) {
									return $this->getRaw( $variable, array() );
								}

								return $this->getString( $variable );
						}
					case 'set':
						switch ( $variable ) {
							case 'asset':
							case 'assets':
								$this->assets( ( $arguments ? $arguments[0] : null ) );
								break;
							case 'user':
								$this->user( ( $arguments ? $arguments[0] : null ) );
								break;
							default:
								$this->set( $variable, ( $arguments ? $arguments[0] : null ) );
								break;
						}

						return $this;
				}
			}
		}

		trigger_error( 'Call to undefined method ' . __CLASS__ . '::' . $name . '()', E_USER_ERROR );
	}

	/**
	 * Returns an array of types for likes
	 *
	 * @return LikeTypeTable[]
	 */
	public function types()
	{
		return CBActivity::loadLikeOptions( true, null, $this );
	}

	/**
	 * Retrieves likes rows or row count
	 *
	 * @param string $output
	 * @return LikeTable[]|int
	 */
	public function rows( $output = null )
	{
		global $_CB_database, $_PLUGINS;

		static $cache						=	array();

		$id									=	$this->getRaw( 'id' );
		$hasId								=	( ( ( $id !== null ) && ( $id !== '' ) ) || ( is_array( $id ) && $id ) );

		if ( $this->getInt( 'paging_limitstart', 0 ) === 0 ) {
			$pageLimit						=	$this->getInt( 'paging_first_limit', 15 );
		} else {
			$pageLimit						=	$this->getInt( 'paging_limit', 15 );
		}

		$paging								=	( ( ! $hasId ) && $pageLimit && ( $output !== 'all' ) );
		$select								=	array();
		$join								=	array();
		$where								=	array();

		if ( $output === 'count' ) {
			$select[]						=	'COUNT(*)';
		} else {
			$select[]						=	'a.*';
		}

		if ( $hasId ) {
			if ( is_array( $this->getRaw( 'id' ) ) ) {
				$where[]					=	"a." . $_CB_database->NameQuote( 'id' ) . " IN " . $_CB_database->safeArrayOfIntegers( $id );
			} else {
				$where[]					=	"a." . $_CB_database->NameQuote( 'id' ) . " = " . (int) $id;
			}
		}

		$userId								=	$this->getRaw( 'user_id' );

		if ( ( ( $userId !== null ) && ( $userId !== '' ) ) || ( is_array( $userId ) && $userId ) ) {
			if ( is_array( $userId ) ) {
				$where[]					=	"a." . $_CB_database->NameQuote( 'user_id' ) . " IN " . $_CB_database->safeArrayOfIntegers( $userId );
			} else {
				$where[]					=	"a." . $_CB_database->NameQuote( 'user_id' ) . " = " . (int) $userId;
			}
		}

		if ( $this->assets() && ( ! in_array( 'all', $this->assets(), true ) ) )  {
			$queryAssets					=	$this->queryAssets();

			if ( $queryAssets['assets'] || $queryAssets['wildcards'] || $queryAssets['exists'] ) {
				$assetsWhere				=	array();

				if ( $queryAssets['assets'] ) {
					$assetsWhere[]			=	"a." . $_CB_database->NameQuote( 'asset' ) . ( count( $queryAssets['assets'] ) > 1 ? " IN " . $_CB_database->safeArrayOfStrings( $queryAssets['assets'] ) : " = " . $_CB_database->Quote( $queryAssets['assets'][0] ) );
				}

				if ( $queryAssets['wildcards'] ) {
					foreach ( $queryAssets['wildcards'] as $wildcard ) {
						$assetsWhere[]		=	"a." . $_CB_database->NameQuote( 'asset' ) . " LIKE " . $_CB_database->Quote( $wildcard );
					}
				}

				if ( $queryAssets['exists'] ) {
					foreach ( $queryAssets['exists'] as $exist ) {
						$assetsWhere[]		=	"EXISTS ( " . $exist . " )";
					}
				}

				$where[]					=	( count( $assetsWhere ) > 1 ? "( " . implode( " OR ", $assetsWhere ) . " )" : $assetsWhere[0] );
			} elseif ( $output === 'count' ) {
				return 0;
			} else {
				return array();
			}
		}

		if ( ! $hasId ) {
			$type							=	$this->getRaw( 'type' );

			if ( ( ( $type !== null ) && ( $type !== '' ) ) || ( is_array( $type ) && $type ) ) {
				if ( is_array( $type ) ) {
					$where[]				=	"a." . $_CB_database->NameQuote( 'type' ) . " IN " . $_CB_database->safeArrayOfIntegers( $type );
				} else {
					$where[]				=	"a." . $_CB_database->NameQuote( 'type' ) . " = " . (int) $type;
				}
			}
		}

		$_PLUGINS->trigger( 'activity_onQueryLikesStream', array( $output, &$select, &$join, &$where, &$this ) );

		$query								=	"SELECT " . implode( ", ", $select )
											.	"\n FROM " . $_CB_database->NameQuote( '#__comprofiler_plugin_activity_likes' ) . " AS a"
											.	"\n LEFT JOIN " . $_CB_database->NameQuote( '#__comprofiler' ) . " AS cb"
											.	" ON cb." . $_CB_database->NameQuote( 'id' ) . " = a." . $_CB_database->NameQuote( 'user_id' )
											.	"\n LEFT JOIN " . $_CB_database->NameQuote( '#__users' ) . " AS j"
											.	" ON j." . $_CB_database->NameQuote( 'id' ) . " = cb." . $_CB_database->NameQuote( 'id' )
											.	( $join ? "\n " . implode( "\n ", $join ) : null )
											.	"\n WHERE cb." . $_CB_database->NameQuote( 'approved' ) . " = 1"
											.	"\n AND cb." . $_CB_database->NameQuote( 'confirmed' ) . " = 1"
											.	"\n AND j." . $_CB_database->NameQuote( 'block' ) . " = 0"
											.	( $where ? "\n AND " . implode( "\n AND ", $where ) : null );

		if ( ( ! $hasId ) && ( $output !== 'count' ) ) {
			$query							.=	"\n ORDER BY a." . $_CB_database->NameQuote( 'date' ) . " DESC";
		}

		$cacheId							=	md5( $query . ( $output === 'count' ? $output : ( $output ? $output : null ) . ( $paging ? $this->getInt( 'paging_limitstart', 0 ) . $pageLimit : null ) ) );

		if ( ( ! isset( $cache[$cacheId] ) ) || ( ( ( $output === 'count' ) && $this->clearRowCount ) || $this->clearRowSelect ) ) {
			if ( $output === 'count' ) {
				$this->clearRowCount		=	false;

				$_CB_database->setQuery( $query );

				$cache[$cacheId]			=	(int) $_CB_database->loadResult();

				$this->set( 'query_count', $cache[$cacheId] );
			} else {
				$this->clearRowSelect		=	false;

				if ( $paging ) {
					$_CB_database->setQuery( $query, $this->getInt( 'paging_limitstart', 0 ), ( $pageLimit + 1 ) );
				} else {
					$_CB_database->setQuery( $query );
				}

				$this->set( 'paging_limitstart', ( $this->getInt( 'paging_limitstart', 0 ) + $pageLimit ) );

				$rows						=	$_CB_database->loadObjectList( 'id', '\CB\Plugin\Activity\Table\LikeTable', array( $_CB_database ) );
				$rowsCount					=	count( $rows );

				if ( $paging ) {
					$this->set( 'paging_total', $rowsCount );

					$rows					=	array_slice( $rows, 0, $pageLimit, true );
					$rowsCount				=	count( $rows );
				}

				$userIds					=	array();

				/** @var LikeTable[] $rows */
				foreach ( $rows as $row ) {
					if ( preg_match( '/^profile\.(\d+)/', $row->getString( 'asset' ), $matches ) ) {
						$userIds[]			=	(int) $matches[1];
					}

					$userIds[]				=	$row->getInt( 'user_id', 0 );
				}

				if ( $userIds ) {
					\CBuser::advanceNoticeOfUsersNeeded( $userIds );
				}

				$_PLUGINS->trigger( 'activity_onLoadLikesStream', array( &$rows, $this ) );

				if ( $rows ) {
					self::$loadedRows		+=	$rows;
				}

				$newCount					=	count( $rows );

				if ( $paging && $rowsCount && ( $newCount < round( $rowsCount / 1.25 ) ) ) {
					$pagingTotal			=	$this->getInt( 'paging_total', 0 );
					$nextLimit				=	( $pageLimit - $newCount );

					if ( $nextLimit <= 0 ) {
						$nextLimit			=	1;
					}

					$this->set( 'paging_limit', $nextLimit );

					$cache[$cacheId]		=	( $rows + $this->rows( $output ) );

					$this->set( 'paging_total', $pagingTotal );
					$this->set( 'paging_limit', $pageLimit );
				} else {
					$cache[$cacheId]		=	$rows;
				}
			}
		} elseif ( $output !== 'count' ) {
			$this->set( 'paging_limitstart', ( $this->getInt( 'paging_limitstart', 0 ) + count( $cache[$cacheId] ) ) );
		}

		return $cache[$cacheId];
	}

	/**
	 * Retrieves likes row
	 *
	 * @param int $id
	 * @return LikeTable
	 */
	public function row( $id )
	{
		if ( ! $id ) {
			return new LikeTable();
		}

		if ( isset( self::$loadedRows[$id] ) ) {
			return self::$loadedRows[$id];
		}

		static $cache		=	array();

		if ( ! isset( $cache[$id] ) ) {
			$rows			=	$this->reset()->setId( $id )->rows();

			if ( isset( $rows[$id] ) ) {
				$row		=	$rows[$id];
			} else {
				$row		=	new LikeTable();
			}

			$cache[$id]		=	$row;
		}

		return $cache[$id];
	}

	/**
	 * Outputs likes HTML
	 *
	 * @param null|string $view
	 * @param int         $id
	 * @param array       $params
	 * @return string
	 */
	public function likes( $view = null, $id = 0, $params = array() )
	{
		return $this->display( $view, $id, $params );
	}
}