<?php
/**
* Community Builder (TM)
* @version $Id: $
* @package CommunityBuilder
* @copyright (C) 2004-2021 www.joomlapolis.com / Lightning MultiCom SA - and its licensors, all rights reserved
* @license http://www.gnu.org/licenses/old-licenses/gpl-2.0.html GNU/GPL version 2
*/

namespace CB\Plugin\Invites\Field;

use CB\Database\Table\UserTable;
use CB\Database\Table\FieldTable;
use CB\Plugin\Invites\Table\InviteTable;
use CBLib\Application\Application;
use CBLib\Input\Get;
use CBLib\Language\CBTxt;
use CBLib\Registry\GetterInterface;

defined('CBLIB') or die();

class InviteField extends \cbFieldHandler
{

	/**
	 * formats variable array into data attribute string
	 *
	 * @param  FieldTable $field
	 * @param  UserTable  $user
	 * @param  string     $output
	 * @param  string     $reason
	 * @param  array      $attributeArray
	 * @return null|string
	 */
	protected function getDataAttributes( $field, $user, $output, $reason, $attributeArray = array() )
	{
		if ( $field->params->getBool( 'hiddenInviteCode', false ) ) {
			return null;
		}

		if ( $field->params->getBool( 'validateInviteCode', true ) ) {
			$attributeArray[]	=	\cbValidator::getRuleHtmlAttributes( 'cbfield', array( 'user' => $user->getInt( 'id', 0 ), 'field' => $field->getString( 'name' ), 'reason' => $reason ) );
		}

		return parent::getDataAttributes( $field, $user, $output, $reason, $attributeArray );
	}

	/**
	 * @param FieldTable $field
	 * @param UserTable  $user
	 * @param string     $output
	 * @param string     $reason
	 * @param int        $list_compare_types
	 * @return mixed
	 */
	public function getField( &$field, &$user, $output, $reason, $list_compare_types )
	{
		if ( ( $reason === 'register' ) && ( $output === 'htmledit' ) ) {
			$code	=	$this->getInput()->getString( 'invite_code', ( ! Application::Application()->isClient( 'administrator' ) ? Application::Session()->getString( 'invite_code' ) : null ) );

			if ( $code && ( strlen( $code ) === 32 ) ) {
				$user->set( 'invite_code', $code );
			}
		}

		if ( $field->params->getBool( 'hiddenInviteCode', false ) ) {
			$field->set( 'type', 'hidden' );
			$field->set( 'cssclass', 'hidden' );
		} else {
			$field->set( 'type', 'text' );
		}

		return parent::getField( $field, $user, $output, $reason, $list_compare_types );
	}

	/**
	 * @param FieldTable $field
	 * @param UserTable  $user
	 * @param array      $postdata
	 * @param string     $reason
	 * @return null|string
	 */
	public function fieldClass( &$field, &$user, &$postdata, $reason )
	{
		parent::fieldClass( $field, $user, $postdata, $reason );

		if ( $this->getInput()->getString( 'function' ) === 'checkvalue' ) {
			$valid				=	true;
			$message			=	null;
			$value				=	Get::get( $postdata, 'value', GetterInterface::STRING );

			if ( $value ) {
				$invite			=	new InviteTable();

				$invite->load( array( 'code' => $value ) );

				if ( ! $invite->getInt( 'id', 0 ) ) {
					$valid		=	false;
					$message	=	CBTxt::T( 'Invite code not valid.' );
				} elseif ( $invite->isAccepted() ) {
					$valid		=	false;
					$message	=	CBTxt::T( 'Invite code already used.' );
				} elseif ( $invite->isExpired() ) {
					$valid		=	false;
					$message	=	CBTxt::T( 'Invite code has expired.' );
				} else {
					$message	=	CBTxt::T( 'Invite code is valid.' );
				}
			}

			return json_encode( array( 'valid' => $valid, 'message' => $message ) );
		}

		return null;
	}

	/**
	 * @param FieldTable $field
	 * @param UserTable  $user
	 * @param string     $columnName
	 * @param string     $value
	 * @param array      $postdata
	 * @param string     $reason
	 * @return bool
	 */
	public function validate( &$field, &$user, $columnName, &$value, &$postdata, $reason )
	{
		$validated				=	parent::validate( $field, $user, $columnName, $value, $postdata, $reason );

		if ( $validated && $value && ( $user->getString( $columnName ) !== $value ) ) {
			$invite				=	new InviteTable();

			$invite->load( array( 'code' => $value ) );

			if ( ! $invite->getInt( 'id', 0 ) ) {
				$this->_setValidationError( $field, $user, $reason, CBTxt::T( 'Invite code not valid.' ) );

				$validated		=	false;
			} elseif ( $user->getInt( 'id', 0 ) !== $invite->getInt( 'user', 0 ) ) {
				if ( $invite->isAccepted() ) {
					$this->_setValidationError( $field, $user, $reason, CBTxt::T( 'Invite code already used.' ) );

					$validated	=	false;
				} elseif ( $invite->isExpired() ) {
					$this->_setValidationError( $field, $user, $reason, CBTxt::T( 'Invite code has expired.' ) );

					$validated	=	false;
				}
			}
		}

		return $validated;
	}
}
