<?php
/**
* Community Builder (TM)
* @version $Id: $
* @package CommunityBuilder
* @copyright (C) 2004-2021 www.joomlapolis.com / Lightning MultiCom SA - and its licensors, all rights reserved
* @license http://www.gnu.org/licenses/old-licenses/gpl-2.0.html GNU/GPL version 2
*/

namespace CB\Plugin\Privacy\Trigger;

use CBLib\Application\Application;
use CBLib\Registry\GetterInterface;
use CB\Database\Table\UserTable;
use CB\Plugin\Privacy\CBPrivacy;
use CB\Plugin\Privacy\Privacy;
use CB\Database\Table\FieldTable;
use CBLib\Registry\ParamsInterface;
use CBLib\Registry\Registry;
use CBLib\Language\CBTxt;

defined('CBLIB') or die();

class FieldTrigger extends \cbFieldHandler
{

	/**
	 * @param null|FieldTable $field
	 * @param UserTable       $user
	 * @param string          $reason
	 * @return string
	 */
	private function fieldOverride( $field, $user, $reason )
	{
		if ( ! $field->params instanceof ParamsInterface ) {
			$field->params	=	new Registry( $field->params );
		}

		switch( $reason ) {
			case 'register':
			case 'edit':
				$override	=	CBTxt::T( $field->params->get( 'cbprivacy_edit_override', null, GetterInterface::RAW ) );
				break;
			case 'list':
			case 'profile':
			default:
				$override	=	CBTxt::T( $field->params->get( 'cbprivacy_display_override', null, GetterInterface::RAW ) );
				break;
		}

		$override			=	trim( preg_replace( '/\[cb:(userdata +field|userfield +field)="' . preg_quote( $field->getString( 'name', '' ), '/' ) . '"[^]]+\]/i', '', $override ) );

		if ( $override ) {
			$override		=	\CBuser::getInstance( $user->get( 'id', 0, GetterInterface::INT ) )->replaceUserVars( $override, false, false, array(), false );
		}

		return $override;
	}

	/**
	 * @param FieldTable[] $fields
	 * @param UserTable    $user
	 * @param string       $reason
	 * @param int          $tabid
	 * @param int|string   $fieldIdOrName
	 * @param bool         $fullAccess
	 */
	public function fieldsFetch( &$fields, &$user, $reason, $tabid, $fieldIdOrName, $fullAccess )
	{
		if ( $fieldIdOrName
			 || ( ! $fields )
			 || ( ! $user instanceof UserTable )
			 || $user->getError()
			 || Application::Cms()->getClientId()
			 || Application::MyUser()->isGlobalModerator()
			 || $fullAccess
			 || $user->get( '_isFieldGroup', false, GetterInterface::BOOLEAN )
		) {
			return;
		}

		if ( $reason == 'profile' ) {
			$hasDefaultOverride		=	array( 'avatar', 'canvas', 'formatname' );

			foreach ( $fields as $fieldId => $field ) {
				if ( isset( $fields[$fieldId] )
					 && ( ! $field->get( '_noPrivacy', false, GetterInterface::BOOLEAN ) )
					 && ( ! CBPrivacy::checkFieldDisplayAccess( $field, $user ) )
					 && ( ! $this->fieldOverride( $field, $user, $reason ) )
					 && ( ! in_array( $field->getString( 'name', '' ), $hasDefaultOverride ) )
				) {
					unset( $fields[$fieldId] );
				}
			}
		} elseif ( ( $reason == 'edit' ) && $user->get( 'id', 0, GetterInterface::INT ) ) {
			foreach ( $fields as $fieldId => $field ) {
				if ( isset( $fields[$fieldId] ) && ( ! $field->get( '_noPrivacy', false, GetterInterface::BOOLEAN ) ) && ( ! CBPrivacy::checkFieldEditAccess( $field ) ) && ( ! $this->fieldOverride( $field, $user, $reason ) ) ) {
					unset( $fields[$fieldId] );
				}
			}
		}
	}

	/**
	 * @param FieldTable $groupField
	 * @param array      $fields
	 * @param UserTable  $user
	 * @param string     $reason
	 * @param bool       $fullAccess
	 */
	public function fieldGroupsFieldsFetch( $groupField, &$fields, &$user, $reason, $fullAccess )
	{
		if ( ( ! $fields )
			 || ( ! $user instanceof UserTable )
			 || $user->getError()
			 || Application::Cms()->getClientId()
			 || Application::MyUser()->isGlobalModerator()
			 || $fullAccess
		) {
			return;
		}

		if ( $reason == 'profile' ) {
			$hasDefaultOverride		=	array( 'avatar', 'canvas', 'formatname' );

			foreach ( $fields as $i => $rowFields ) {
				foreach ( $rowFields as $k => $field ) {
					/** @var FieldTable $field */
					if ( ( ! $field->get( '_noPrivacy', false, GetterInterface::BOOLEAN ) )
						 && ( ! CBPrivacy::checkFieldDisplayAccess( $field, $user ) )
						 && ( ! $this->fieldOverride( $field, $user, $reason ) )
						 && ( ! in_array( $field->getString( 'name', '' ), $hasDefaultOverride ) )
					) {
						unset( $fields[$i][$k] );
					}
				}
			}
		} elseif ( ( $reason == 'edit' ) && $user->get( 'id', 0, GetterInterface::INT ) ) {
			foreach ( $fields as $i => $rowFields ) {
				foreach ( $rowFields as $k => $field ) {
					/** @var FieldTable $field */
					if ( ( ! $field->get( '_noPrivacy', false, GetterInterface::BOOLEAN ) )
						 && ( ! CBPrivacy::checkFieldEditAccess( $field ) )
						 && ( ! $this->fieldOverride( $field, $user, $reason ) )
					) {
						unset( $fields[$i][$k] );
					}
				}
			}
		}
	}

	/**
	 * @param FieldTable $field
	 * @param UserTable  $user
	 * @param string     $output
	 * @param string     $formatting
	 * @param string     $reason
	 * @param int        $list_compare_types
	 * @return mixed|null|string
	 */
	public function beforeFieldDisplay( &$field, &$user, $output, $formatting, $reason, $list_compare_types )
	{
		if ( $field->get( '_noPrivacy', false, GetterInterface::BOOLEAN ) || Application::Cms()->getClientId() || Application::MyUser()->isGlobalModerator() ) {
			return null;
		}

		$return						=	null;

		$field->set( '_noPrivacy', true );

		if ( ( $output == 'html' ) && ( $reason != 'search' ) ) {
			if ( ! CBPrivacy::checkFieldDisplayAccess( $field, $user ) ) {
				$return				=	$this->fieldOverride( $field, $user, $reason );

				if ( $return ) {
					$return			=	$this->renderFieldHtml( $field, $user, $this->_formatFieldOutput( $field->getString( 'name', '' ), $return, $output, false ), $output, $formatting, $reason, array() );
				} else {
					switch ( $field->getString( 'name', '' ) ) {
						case 'avatar':
							$return	=	\CBuser::getInstance( 0, false )->getField( 'avatar', null, $output, $formatting, $reason, $list_compare_types, true, array( '_allowProfileLink' => false ) );
							break;
						case 'canvas':
							$return	=	\CBuser::getInstance( 0, false )->getField( 'canvas', null, $output, $formatting, $reason, $list_compare_types, true );
							break;
						case 'formatname':
							$return	=	$this->renderFieldHtml( $field, $user, $this->_formatFieldOutput( $field->getString( 'name', '' ), $user->getFormattedName(), $output, true ), $output, $formatting, $reason, array() );
							break;
					}
				}

				if ( ! $return ) {
					$return			=	' ';
				}
			}
		} elseif ( ( $output == 'htmledit' ) && ( $reason != 'search' ) && $user->get( 'id', 0, GetterInterface::INT ) ) {
			if ( ! CBPrivacy::checkFieldEditAccess( $field ) ) {
				$return				=	$this->fieldOverride( $field, $user, $reason );

				if ( $return ) {
					$return			=	$this->renderFieldHtml( $field, $user, $this->_formatFieldOutput( $field->getString( 'name', '' ), $return, $output, false ), $output, $formatting, $reason, array() );
				}

				if ( ! $return ) {
					$return			=	' ';
				}
			}
		}

		$field->set( '_noPrivacy', false );

		return $return;
	}

	/**
	 * @param FieldTable $field
	 * @param UserTable  $user
	 * @param string     $output
	 * @param string     $formatting
	 * @param string     $reason
	 * @param int        $list_compare_types
	 * @param mixed      $return
	 */
	public function afterFieldDisplay( &$field, &$user, $output, $formatting, $reason, $list_compare_types, &$return )
	{
		if ( $field->get( '_noPrivacy', false, GetterInterface::BOOLEAN )
			 || ( $output != 'htmledit' )
			 || ( ! in_array( $reason, array( 'edit', 'register' ) ) )
			 || ( ! $field instanceof FieldTable )
			 || ( ! $field->get( 'profile', 1, GetterInterface::INT ) )
		) {
			return;
		}

		if ( ! $return ) {
			$privacy		=	$this->getPrivacyControls( $field, $user, $reason );

			if ( $privacy ) {
				$return		=	$this->renderFieldHtml( $field, $user, $this->_formatFieldOutput( $field->getString( 'name', '' ), $privacy, $output, false ), $output, $formatting, $reason, array() );
			}
		}
	}

	/**
	 * @param \cbFieldHandler $fieldHandler
	 * @param FieldTable      $field
	 * @param UserTable       $user
	 * @param string          $output
	 * @param string          $reason
	 * @param string          $tag
	 * @param string          $type
	 * @param string          $value
	 * @param string          $additional
	 * @param string          $allValues
	 * @param bool            $displayFieldIcons
	 * @param bool            $required
	 * @return null|string
	 */
	public function fieldIcons( &$fieldHandler, &$field, &$user, $output, $reason, $tag, $type, $value, $additional, $allValues, $displayFieldIcons, $required )
	{
		return $this->getPrivacyControls( $field, $user, $reason );
	}

	/**
	 * @param FieldTable      $field
	 * @param UserTable       $user
	 * @param string          $reason
	 * @return null|string
	 */
	private function getPrivacyControls( &$field, &$user, $reason )
	{
		if ( $field->get( '_noPrivacy', false, GetterInterface::BOOLEAN )
			 || ( ! in_array( $reason, array( 'edit', 'register' ) ) )
			 || ( ! $field instanceof FieldTable )
			 || ( ! $field->get( 'profile', 1, GetterInterface::INT ) )
		) {
			return null;
		}

		$fieldId				=	$field->get( 'fieldid', 0, GetterInterface::INT );

		if ( ! $field->params instanceof ParamsInterface ) {
			$field->params		=	new Registry( $field->params );
		}

		$display				=	$field->params->get( 'cbprivacy_display', 0, GetterInterface::INT );

		if ( ( $reason == 'register' ) && ( ! $field->params->get( 'cbprivacy_display_reg', false, GetterInterface::BOOLEAN ) ) ) {
			$display			=	0;
		}

		$return					=	null;

		if ( ( $display == 1 ) || ( ( $display == 2 ) && Application::MyUser()->isGlobalModerator() ) ) {
			$asset				=	str_replace( '[field_id]', $fieldId, $field->params->getString( 'cbprivacy_asset', '' ) );

			if ( ! $asset ) {
				$asset			=	'profile.field.' . $fieldId;

				if ( $field->get( '_isGrouped', false, GetterInterface::BOOLEAN ) ) {
					$asset		.=	'.group.' . $field->get( '_fieldGroupId', 0, GetterInterface::INT ) . '.row.' . $field->get( '_fieldGroupIndex', 0, GetterInterface::INT );
				}
			}

			$privacy			=	new Privacy( $asset, $user );

			if ( $display == 2 ) {
				$privacy->set( 'options_moderator', true );
			}

			$privacy->parse( $field->params, 'privacy_' );

			if ( in_array( $reason, array( 'edit', 'register' ) ) ) {
				$privacy->set( 'ajax', false );
			}

			$privacy->name( $field->getString( 'name', '' ) . '__privacy' );
			$privacy->set( 'field', $fieldId );

			if ( ( ( ( $reason == 'edit' ) && CBPrivacy::getGlobalParams()->get( 'profile_simple_edit', false, GetterInterface::BOOLEAN ) )
				   || ( ( $reason == 'register' ) && CBPrivacy::getGlobalParams()->get( 'profile_simple_registration', false, GetterInterface::BOOLEAN ) )
				 )
				 && ( $this->input( 'function', null, GetterInterface::STRING ) != 'ajax_edit' )
				 && ( ! in_array( $field->get( 'fieldid', 0, GetterInterface::INT ), explode( '|*|', CBPrivacy::getGlobalParams()->getString( 'profile_simple_exclude_fields', '' ) ) ) )
				 && ( ! $field->get( '_isGrouped', false, GetterInterface::BOOLEAN ) )
			) {
				CBPrivacy::getGroupedPrivacy( $field, $privacy );
			} else {
				$return			=	$privacy->privacy( 'edit' );
			}
		}

		return $return;
	}

	/**
	 * @param FieldTable $field
	 * @param UserTable  $user
	 * @param array      $postdata
	 * @param string     $reason
	 * @return null|string
	 */
	public function fieldPrepareSave( &$field, &$user, &$postdata, $reason )
	{
		if ( $field->get( '_noPrivacy', false, GetterInterface::BOOLEAN ) ) {
			return null;
		}

		$return			=	null;

		if ( ( $reason != 'search' ) && $user->get( 'id', 0, GetterInterface::INT ) && ( ! CBPrivacy::checkFieldEditAccess( $field ) ) ) {
			$return		=	' ';
		}

		return $return;
	}
}
