<?php
/**
* Community Builder (TM)
* @version $Id: $
* @package CommunityBuilder
* @copyright (C) 2004-2021 www.joomlapolis.com / Lightning MultiCom SA - and its licensors, all rights reserved
* @license http://www.gnu.org/licenses/old-licenses/gpl-2.0.html GNU/GPL version 2
*/

namespace CB\Plugin\ProfileUpdateLogger;

use CB\Database\Table\FieldTable;
use CBLib\Application\Application;
use CBLib\Registry\ParamsInterface;
use CBLib\Registry\Registry;

defined('CBLIB') or die();

class CBProfileUpdateLogger
{

	/**
	 * @return Registry
	 */
	static public function getGlobalParams()
	{
		global $_PLUGINS;

		static $params	=	null;

		if ( ! $params ) {
			$plugin		=	$_PLUGINS->getLoadedPlugin( 'user', 'cb.pulog' );
			$params		=	new Registry();

			if ( $plugin ) {
				$params->load( $plugin->params );
			}
		}

		return $params;
	}

	/**
	 * @param null|string $template
	 * @param null|string $file
	 * @param bool|array  $headers
	 * @return null|string
	 */
	static public function getTemplate( $template = null, $file = null, $headers = array( 'template', 'override' ) )
	{
		global $_CB_framework, $_PLUGINS;

		$plugin							=	$_PLUGINS->getLoadedPlugin( 'user', 'cb.pulog' );

		if ( ! $plugin ) {
			return null;
		}

		static $defaultTemplate			=	null;

		if ( $defaultTemplate === null ) {
			$defaultTemplate			=	self::getGlobalParams()->getString( 'general_template', 'default' );
		}

		if ( ( $template === '' ) || ( $template === null ) || ( $template === '-1' ) ) {
			$template					=	$defaultTemplate;
		}

		if ( ! $template ) {
			$template					=	'default';
		}

		$livePath						=	$_PLUGINS->getPluginLivePath( $plugin );
		$absPath						=	$_PLUGINS->getPluginPath( $plugin );

		$file							=	preg_replace( '/[^-a-zA-Z0-9_]/', '', $file );
		$return							=	null;

		if ( $file ) {
			if ( $headers !== false ) {
				$headers[]				=	$file;
			}

			$php						=	$absPath . '/templates/' . $template . '/' . $file . '.php';

			if ( ! file_exists( $php ) ) {
				$php					=	$absPath . '/templates/default/' . $file . '.php';
			}

			if ( file_exists( $php ) ) {
				$return					=	$php;
			}
		}

		if ( $headers !== false ) {
			static $loaded				=	array();

			$loaded[$template]			=	array();

			// Global CSS File:
			if ( in_array( 'template', $headers ) && ( ! in_array( 'template', $loaded[$template] ) ) ) {
				$global					=	'/templates/' . $template . '/template.css';

				if ( ! file_exists( $absPath . $global ) ) {
					$global				=	'/templates/default/template.css';
				}

				if ( file_exists( $absPath . $global ) ) {
					$_CB_framework->document->addHeadStyleSheet( $livePath . $global );
				}

				$loaded[$template][]	=	'template';
			}

			// File or Custom CSS/JS Headers:
			foreach ( $headers as $header ) {
				if ( in_array( $header, $loaded[$template] ) || in_array( $header, array( 'template', 'override' ) ) ) {
					continue;
				}

				$header					=	preg_replace( '/[^-a-zA-Z0-9_]/', '', $header );

				if ( ! $header ) {
					continue;
				}

				$css					=	'/templates/' . $template . '/' . $header . '.css';
				$js						=	'/templates/' . $template . '/' . $header . '.js';

				if ( ! file_exists( $absPath . $css ) ) {
					$css				=	'/templates/default/' . $header . '.css';
				}

				if ( file_exists( $absPath . $css ) ) {
					$_CB_framework->document->addHeadStyleSheet( $livePath . $css );
				}

				if ( ! file_exists( $absPath . $js ) ) {
					$js					=	'/templates/default/' . $header . '.js';
				}

				if ( file_exists( $absPath . $js ) ) {
					$_CB_framework->document->addHeadScriptUrl( $livePath . $js );
				}

				$loaded[$template][]	=	$header;
			}

			// Override CSS File:
			if ( in_array( 'override', $headers ) && ( ! in_array( 'override', $loaded[$template] ) ) ) {
				$override				=	'/templates/' . $template . '/override.css';

				if ( file_exists( $absPath . $override ) ) {
					$_CB_framework->document->addHeadStyleSheet( $livePath . $override );
				}

				$loaded[$template][]	=	'override';
			}
		}

		return $return;
	}

	/**
	 * @param string $name
	 * @param int    $profileId
	 * @return null|FieldTable
	 */
	public static function getField( $name, $profileId = null )
	{
		if ( ! $name ) {
			return null;
		}

		$userId				=	Application::MyUser()->getUserId();

		if ( $profileId === null ) {
			$profileId		=	$userId;
		}

		static $fields		=	array();

		if ( ! isset( $fields[$profileId][$userId] ) ) {
			$profileUser	=	\CBuser::getInstance( $profileId, false );
			$profileFields	=	$profileUser->_getCbTabs( false )->_getTabFieldsDb( null, $profileUser->getUserData(), 'adminfulllist', null, true, true );

			foreach ( $profileFields as $profileField ) {
				$fields[$profileId][$userId][$profileField->getString( 'name' )]	=	clone $profileField;
			}
		}

		if ( ! isset( $fields[$profileId][$userId][$name] ) ) {
			return null;
		}

		$field				=	$fields[$profileId][$userId][$name];

		if ( ! ( $field->getRaw( 'params' ) instanceof ParamsInterface ) ) {
			$field->set( 'params', new Registry( $field->getRaw( 'params' ) ) );
		}

		return $field;
	}

	/**
	 * @param string $name
	 * @param int    $userId
	 * @return null|string
	 */
	public static function getFieldTitle( $name, $userId = null )
	{
		global $_PLUGINS;

		if ( ! $name ) {
			return null;
		}

		static $cache					=	array();

		if ( ! isset( $cache[$userId][$name] ) ) {
			$field						=	self::getField( $name, $userId );

			if ( ! $field ) {
				$cache[$userId][$name]	=	null;

				return null;
			}

			$user						=	clone \CBuser::getUserDataInstance( $userId );

			$cache[$userId][$name]		=	$_PLUGINS->callField( $field->getString( 'type' ), 'getFieldTitle', array( &$field, &$user, 'html', 'profile' ), $field );
		}

		return $cache[$userId][$name];
	}

	/**
	 * @param string $name
	 * @param mixed  $value
	 * @param int    $userId
	 * @return null|string
	 */
	public static function getFieldValue( $name, $value, $userId = null )
	{
		global $_PLUGINS;

		if ( ! $name ) {
			return null;
		}

		static $cache								=	array();

		$cacheId									=	md5( $value ); // value could be really long so just use a hash of it

		if ( ! isset( $cache[$userId][$name][$cacheId] ) ) {
			$field									=	self::getField( $name, $userId );

			if ( ! $field ) {
				$cache[$userId][$name][$cacheId]	=	null;

				return null;
			}

			$user									=	clone \CBuser::getUserDataInstance( $userId );

			$user->set( $name, $value );

			$field->set( 'registration', 0 );
			$field->set( 'edit', 0 );
			$field->set( 'readonly', 1 );

			$field->set( '_noPrivacy', true );
			$field->set( '_noCondition', true );
			$field->set( '_noAjax', true );

			$cache[$userId][$name][$cacheId]		=	$_PLUGINS->callField( $field->getString( 'type' ), 'getFieldRow', array( &$field, &$user, 'html', 'none', 'profile', 0 ), $field );
		}

		return $cache[$userId][$name][$cacheId];
	}

	/**
	 * @param array $from
	 * @param array $to
	 * @return bool
	 */
	public static function isArraysEqual( $from, $to )
	{
		if ( $from === $to ) {
			return true;
		}

		if ( ! is_array( $from ) ) {
			return false;
		}

		if ( ! is_array( $to ) ) {
			return false;
		}

		// Check if old values changed to new values:
		foreach ( $from as $k => $v ) {
			if ( ! array_key_exists( $k, $to ) ) {
				return false;
			}

			if ( $v === $to[$k] ) {
				continue;
			}

			if ( is_array( $v ) ) {
				if ( ! self::isArraysEqual( $v, $to[$k] ) ) {
					return false;
				}
			} else {
				return false;
			}
		}

		// Check if new values were added entirely:
		foreach ( $to as $k => $v ) {
			if ( ! array_key_exists( $k, $from ) ) {
				return false;
			}
		}

		return true;
	}
}
